import { withScope, captureException, showReportDialog } from '@sentry/browser';
import { logger } from '@sentry/utils';
import hoistNonReactStatics from 'hoist-non-react-statics';
import * as React from 'react';
import { IS_DEBUG_BUILD } from './flags.js';

var _jsxFileName = "/home/runner/work/sentry-javascript/sentry-javascript/packages/react/src/errorboundary.tsx";
function isAtLeastReact17(version) {
  var major = version.match(/^([^.]+)/);
  return major !== null && parseInt(major[0]) >= 17;
}

var UNKNOWN_COMPONENT = 'unknown';

var INITIAL_STATE = {
  componentStack: null,
  error: null,
  eventId: null,
};

/**
 * A ErrorBoundary component that logs errors to Sentry. Requires React >= 16.
 * NOTE: If you are a Sentry user, and you are seeing this stack frame, it means the
 * Sentry React SDK ErrorBoundary caught an error invoking your application code. This
 * is expected behavior and NOT indicative of a bug with the Sentry React SDK.
 */
class ErrorBoundary extends React.Component {constructor(...args) { super(...args); ErrorBoundary.prototype.__init.call(this);ErrorBoundary.prototype.__init2.call(this); }
   __init() {this.state = INITIAL_STATE;}

   componentDidCatch(error, { componentStack }) {
    const { beforeCapture, onError, showDialog, dialogOptions } = this.props;

    withScope(scope => {
      // If on React version >= 17, create stack trace from componentStack param and links
      // to to the original error using `error.cause` otherwise relies on error param for stacktrace.
      // Linking errors requires the `LinkedErrors` integration be enabled.
      if (isAtLeastReact17(React.version)) {
        var errorBoundaryError = new Error(error.message);
        errorBoundaryError.name = `React ErrorBoundary ${errorBoundaryError.name}`;
        errorBoundaryError.stack = componentStack;

        // Using the `LinkedErrors` integration to link the errors together.
        error.cause = errorBoundaryError;
      }

      if (beforeCapture) {
        beforeCapture(scope, error, componentStack);
      }
      var eventId = captureException(error, { contexts: { react: { componentStack } } });
      if (onError) {
        onError(error, componentStack, eventId);
      }
      if (showDialog) {
        showReportDialog({ ...dialogOptions, eventId });
      }

      // componentDidCatch is used over getDerivedStateFromError
      // so that componentStack is accessible through state.
      this.setState({ error, componentStack, eventId });
    });
  }

   componentDidMount() {
    const { onMount } = this.props;
    if (onMount) {
      onMount();
    }
  }

   componentWillUnmount() {
    const { error, componentStack, eventId } = this.state;
    const { onUnmount } = this.props;
    if (onUnmount) {
      onUnmount(error, componentStack, eventId);
    }
  }

   __init2() {this.resetErrorBoundary = () => {
    const { onReset } = this.props;
    const { error, componentStack, eventId } = this.state;
    if (onReset) {
      onReset(error, componentStack, eventId);
    }
    this.setState(INITIAL_STATE);
  };}

   render() {
    const { fallback, children } = this.props;
    const { error, componentStack, eventId } = this.state;

    if (error) {
      let element = undefined;
      if (typeof fallback === 'function') {
        element = fallback({ error, componentStack, resetError: this.resetErrorBoundary, eventId });
      } else {
        element = fallback;
      }

      if (React.isValidElement(element)) {
        return element;
      }

      if (fallback) {
        IS_DEBUG_BUILD && logger.warn('fallback did not produce a valid ReactElement');
      }

      // Fail gracefully if no fallback provided or is not valid
      return null;
    }

    if (typeof children === 'function') {
      return (children )();
    }
    return children;
  }
}

function withErrorBoundary(
  WrappedComponent,
  errorBoundaryOptions,
) {
    var componentDisplayName = WrappedComponent.displayName || WrappedComponent.name || UNKNOWN_COMPONENT;

  var Wrapped = (props) => (
    React.createElement(ErrorBoundary, { ...errorBoundaryOptions, __self: this, __source: {fileName: _jsxFileName, lineNumber: 170}}
      , React.createElement(WrappedComponent, { ...props, __self: this, __source: {fileName: _jsxFileName, lineNumber: 171}} )
    )
  );

    Wrapped.displayName = `errorBoundary(${componentDisplayName})`;

  // Copy over static methods from Wrapped component to Profiler HOC
  // See: https://reactjs.org/docs/higher-order-components.html#static-methods-must-be-copied-over
  hoistNonReactStatics(Wrapped, WrappedComponent);
  return Wrapped;
}

export { ErrorBoundary, UNKNOWN_COMPONENT, isAtLeastReact17, withErrorBoundary };
//# sourceMappingURL=errorboundary.js.map
