import type { ReplayRecordingMode } from '@sentry/types';
import type { AddUpdateCallback, AllPerformanceEntry, EventBuffer, InternalEventContext, RecordingOptions, ReplayContainer as ReplayContainerInterface, ReplayPluginOptions, Session } from './types';
/**
 * The main replay container class, which holds all the state and methods for recording and sending replays.
 */
export declare class ReplayContainer implements ReplayContainerInterface {
    eventBuffer: EventBuffer | null;
    /**
     * List of PerformanceEntry from PerformanceObserver
     */
    performanceEvents: AllPerformanceEntry[];
    session: Session | undefined;
    /**
     * Recording can happen in one of two modes:
     * * session: Record the whole session, sending it continuously
     * * error: Always keep the last 60s of recording, and when an error occurs, send it immediately
     */
    recordingMode: ReplayRecordingMode;
    /**
     * Options to pass to `rrweb.record()`
     */
    private readonly _recordingOptions;
    private readonly _options;
    private _performanceObserver;
    private _debouncedFlush;
    private _flushLock;
    /**
     * Timestamp of the last user activity. This lives across sessions.
     */
    private _lastActivity;
    /**
     * Is the integration currently active?
     */
    private _isEnabled;
    /**
     * Paused is a state where:
     * - DOM Recording is not listening at all
     * - Nothing will be added to event buffer (e.g. core SDK events)
     */
    private _isPaused;
    /**
     * Have we attached listeners to the core SDK?
     * Note we have to track this as there is no way to remove instrumentation handlers.
     */
    private _hasInitializedCoreListeners;
    /**
     * Function to stop recording
     */
    private _stopRecording;
    private _context;
    constructor({ options, recordingOptions, }: {
        options: ReplayPluginOptions;
        recordingOptions: RecordingOptions;
    });
    /** Get the event context. */
    getContext(): InternalEventContext;
    /** If recording is currently enabled. */
    isEnabled(): boolean;
    /** If recording is currently paused. */
    isPaused(): boolean;
    /** Get the replay integration options. */
    getOptions(): ReplayPluginOptions;
    /**
     * Initializes the plugin.
     *
     * Creates or loads a session, attaches listeners to varying events (DOM,
     * _performanceObserver, Recording, Sentry SDK, etc)
     */
    start(): void;
    /**
     * Start recording.
     *
     * Note that this will cause a new DOM checkout
     */
    startRecording(): void;
    /**
     * Stops the recording, if it was running.
     * Returns true if it was stopped, else false.
     */
    stopRecording(): boolean;
    /**
     * Currently, this needs to be manually called (e.g. for tests). Sentry SDK
     * does not support a teardown
     */
    stop(reason?: string): void;
    /**
     * Pause some replay functionality. See comments for `_isPaused`.
     * This differs from stop as this only stops DOM recording, it is
     * not as thorough of a shutdown as `stop()`.
     */
    pause(): void;
    /**
     * Resumes recording, see notes for `pause().
     *
     * Note that calling `startRecording()` here will cause a
     * new DOM checkout.`
     */
    resume(): void;
    /**
     * We want to batch uploads of replay events. Save events only if
     * `<flushMinDelay>` milliseconds have elapsed since the last event
     * *OR* if `<flushMaxDelay>` milliseconds have elapsed.
     *
     * Accepts a callback to perform side-effects and returns true to stop batch
     * processing and hand back control to caller.
     */
    addUpdate(cb: AddUpdateCallback): void;
    /**
     * Updates the user activity timestamp and resumes recording. This should be
     * called in an event handler for a user action that we consider as the user
     * being "active" (e.g. a mouse click).
     */
    triggerUserActivity(): void;
    /**
     *
     * Always flush via `_debouncedFlush` so that we do not have flushes triggered
     * from calling both `flush` and `_debouncedFlush`. Otherwise, there could be
     * cases of mulitple flushes happening closely together.
     */
    flushImmediate(): Promise<void>;
    /** Get the current sesion (=replay) ID */
    getSessionId(): string | undefined;
    /**
     * Checks if recording should be stopped due to user inactivity. Otherwise
     * check if session is expired and create a new session if so. Triggers a new
     * full snapshot on new session.
     *
     * Returns true if session is not expired, false otherwise.
     * @hidden
     */
    checkAndHandleExpiredSession(): boolean | void;
    /** A wrapper to conditionally capture exceptions. */
    private _handleException;
    /**
     * Loads (or refreshes) the current session.
     * Returns false if session is not recorded.
     */
    private _loadAndCheckSession;
    /**
     * Capture some initial state that can change throughout the lifespan of the
     * replay. This is required because otherwise they would be captured at the
     * first flush.
     */
    private _setInitialState;
    /**
     * Adds listeners to record events for the replay
     */
    private _addListeners;
    /**
     * Cleans up listeners that were created in `_addListeners`
     */
    private _removeListeners;
    /**
     * Handler for recording events.
     *
     * Adds to event buffer, and has varying flushing behaviors if the event was a checkout.
     */
    private _handleRecordingEmit;
    /**
     * Handle when visibility of the page content changes. Opening a new tab will
     * cause the state to change to hidden because of content of current page will
     * be hidden. Likewise, moving a different window to cover the contents of the
     * page will also trigger a change to a hidden state.
     */
    private _handleVisibilityChange;
    /**
     * Handle when page is blurred
     */
    private _handleWindowBlur;
    /**
     * Handle when page is focused
     */
    private _handleWindowFocus;
    /**
     * Tasks to run when we consider a page to be hidden (via blurring and/or visibility)
     */
    private _doChangeToBackgroundTasks;
    /**
     * Tasks to run when we consider a page to be visible (via focus and/or visibility)
     */
    private _doChangeToForegroundTasks;
    /**
     * Trigger rrweb to take a full snapshot which will cause this plugin to
     * create a new Replay event.
     */
    private _triggerFullSnapshot;
    /**
     * Update user activity (across session lifespans)
     */
    private _updateUserActivity;
    /**
     * Updates the session's last activity timestamp
     */
    private _updateSessionActivity;
    /**
     * Helper to create (and buffer) a replay breadcrumb from a core SDK breadcrumb
     */
    private _createCustomBreadcrumb;
    /**
     * Observed performance events are added to `this.performanceEvents`. These
     * are included in the replay event before it is finished and sent to Sentry.
     */
    private _addPerformanceEntries;
    /**
     * Only flush if `this.recordingMode === 'session'`
     */
    private _conditionalFlush;
    /**
     * Clear _context
     */
    private _clearContext;
    /**
     * Return and clear _context
     */
    private _popEventContext;
    /**
     * Flushes replay event buffer to Sentry.
     *
     * Performance events are only added right before flushing - this is
     * due to the buffered performance observer events.
     *
     * Should never be called directly, only by `flush`
     */
    private _runFlush;
    /**
     * Flush recording data to Sentry. Creates a lock so that only a single flush
     * can be active at a time. Do not call this directly.
     */
    private _flush;
    /** Save the session, if it is sticky */
    private _maybeSaveSession;
}
//# sourceMappingURL=replay.d.ts.map