import { Component, Element, Event, Method, Prop, State, Watch, h } from '@stencil/core';
import { getLibrary, watchIcon, unwatchIcon } from '../icon-library/icon-library-registry';
import { requestIcon } from './request';
const parser = new DOMParser();
/**
 * @since 2.0
 * @status stable
 *
 * @part base - The component's base wrapper.
 */
export class Icon {
  constructor() {
    /** The name of a registered custom icon library. */
    this.library = 'default';
  }
  handleChange() {
    this.setIcon();
  }
  connectedCallback() {
    watchIcon(this.host);
  }
  componentDidLoad() {
    this.setIcon();
  }
  disconnectedCallback() {
    unwatchIcon(this.host);
  }
  /** @internal Fetches the icon and redraws it. Used to handle library registrations. */
  async redraw() {
    this.setIcon();
  }
  getLabel() {
    let label = '';
    if (this.label) {
      label = this.label;
    }
    else if (this.name) {
      label = this.name.replace(/-/g, ' ');
    }
    else if (this.src) {
      label = this.src.replace(/.*\//, '').replace(/-/g, ' ').replace(/\.svg/i, '');
    }
    return label;
  }
  async setIcon() {
    const library = getLibrary(this.library);
    let url = this.src;
    if (this.name && library) {
      url = library.resolver(this.name);
    }
    if (url) {
      try {
        const file = await requestIcon(url);
        if (file.ok) {
          const doc = parser.parseFromString(file.svg, 'text/html');
          const svg = doc.body.querySelector('svg');
          if (svg) {
            if (library && library.mutator) {
              library.mutator(svg);
            }
            this.svg = svg.outerHTML;
            this.slLoad.emit();
          }
          else {
            this.svg = '';
            this.slError.emit({ status: file.status });
          }
        }
      }
      catch (_a) {
        this.slError.emit();
      }
    }
  }
  render() {
    return h("div", { part: "base", class: "icon", role: "img", "aria-label": this.getLabel(), innerHTML: this.svg });
  }
  static get is() { return "sl-icon"; }
  static get encapsulation() { return "shadow"; }
  static get originalStyleUrls() { return {
    "$": ["icon.scss"]
  }; }
  static get styleUrls() { return {
    "$": ["icon.css"]
  }; }
  static get assetsDirs() { return ["icons"]; }
  static get properties() { return {
    "name": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The name of the icon to draw."
      },
      "attribute": "name",
      "reflect": false
    },
    "src": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "An external URL of an SVG file."
      },
      "attribute": "src",
      "reflect": false
    },
    "label": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "An alternative description to use for accessibility. If omitted, the name or src will be used to generate it."
      },
      "attribute": "label",
      "reflect": false
    },
    "library": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The name of a registered custom icon library."
      },
      "attribute": "library",
      "reflect": false,
      "defaultValue": "'default'"
    }
  }; }
  static get states() { return {
    "svg": {}
  }; }
  static get events() { return [{
      "method": "slLoad",
      "name": "sl-load",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted when the icon has loaded."
      },
      "complexType": {
        "original": "any",
        "resolved": "any",
        "references": {}
      }
    }, {
      "method": "slError",
      "name": "sl-error",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted when the icon failed to load."
      },
      "complexType": {
        "original": "{ status?: number }",
        "resolved": "{ status?: number; }",
        "references": {}
      }
    }]; }
  static get methods() { return {
    "redraw": {
      "complexType": {
        "signature": "() => Promise<void>",
        "parameters": [],
        "references": {
          "Promise": {
            "location": "global"
          }
        },
        "return": "Promise<void>"
      },
      "docs": {
        "text": "",
        "tags": [{
            "name": "internal",
            "text": "Fetches the icon and redraws it. Used to handle library registrations."
          }]
      }
    }
  }; }
  static get elementRef() { return "host"; }
  static get watchers() { return [{
      "propName": "name",
      "methodName": "handleChange"
    }, {
      "propName": "src",
      "methodName": "handleChange"
    }, {
      "propName": "library",
      "methodName": "handleChange"
    }]; }
}
