import { Component, Element, Event, Method, Prop, State, Watch, h } from '@stencil/core';
const toastStack = Object.assign(document.createElement('div'), { className: 'sl-toast-stack' });
/**
 * @since 2.0
 * @status stable
 *
 * @slot - The alert's content.
 * @slot icon - An icon to show in the alert.
 *
 * @part base - The component's base wrapper.
 * @part icon - The container that wraps the alert icon.
 * @part message - The alert message.
 * @part close-button - The close button.
 */
export class Alert {
  constructor() {
    this.isVisible = false;
    /** Indicates whether or not the alert is open. You can use this in lieu of the show/hide methods. */
    this.open = false;
    /** Set to true to make the alert closable. */
    this.closable = false;
    /** The type of alert. */
    this.type = 'primary';
    /**
     * The length of time, in milliseconds, the alert will show before closing itself. If the user interacts with the
     * alert before it closes (e.g. moves the mouse over it), the timer will restart.
     */
    this.duration = Infinity;
  }
  handleOpenChange() {
    this.open ? this.show() : this.hide();
  }
  handleDurationChange() {
    this.restartAutoHide();
  }
  connectedCallback() {
    this.handleCloseClick = this.handleCloseClick.bind(this);
    this.handleMouseMove = this.handleMouseMove.bind(this);
    this.handleTransitionEnd = this.handleTransitionEnd.bind(this);
  }
  componentWillLoad() {
    // Show on init if open
    if (this.open) {
      this.show();
    }
  }
  /** Shows the alert. */
  async show() {
    // Prevent subsequent calls to the method, whether manually or triggered by the `open` watcher
    if (this.isVisible) {
      return;
    }
    const slShow = this.slShow.emit();
    if (slShow.defaultPrevented) {
      this.open = false;
      return;
    }
    this.isVisible = true;
    this.open = true;
    if (this.duration < Infinity) {
      this.autoHideTimeout = setTimeout(() => this.hide(), this.duration);
    }
  }
  /** Hides the alert */
  async hide() {
    // Prevent subsequent calls to the method, whether manually or triggered by the `open` watcher
    if (!this.isVisible) {
      return;
    }
    const slHide = this.slHide.emit();
    if (slHide.defaultPrevented) {
      this.open = true;
      return;
    }
    clearTimeout(this.autoHideTimeout);
    this.isVisible = false;
    this.open = false;
  }
  /**
   * Displays the alert as a toast notification. This will move the alert out of its position in the DOM and, when
   * dismissed, it will be removed from the DOM completely. By storing a reference to the alert, you can reuse it by
   * calling this method again. The returned promise will resolve after the alert is hidden.
   */
  async toast() {
    return new Promise(resolve => {
      if (!toastStack.parentElement) {
        document.body.append(toastStack);
      }
      toastStack.append(this.host);
      requestAnimationFrame(() => this.show());
      this.host.addEventListener('sl-after-hide', () => {
        this.host.remove();
        resolve();
        // Remove the toast stack from the DOM when there are no more alerts
        if (toastStack.querySelector('sl-alert') === null) {
          toastStack.remove();
        }
      }, { once: true });
    });
  }
  handleCloseClick() {
    this.hide();
  }
  handleMouseMove() {
    this.restartAutoHide();
  }
  handleTransitionEnd(event) {
    const target = event.target;
    // Ensure we only emit one event when the target element is no longer visible
    if (event.propertyName === 'opacity' && target.classList.contains('alert')) {
      this.isVisible = this.open;
      this.open ? this.slAfterShow.emit() : this.slAfterHide.emit();
    }
  }
  restartAutoHide() {
    clearTimeout(this.autoHideTimeout);
    if (this.open && this.duration < Infinity) {
      this.autoHideTimeout = setTimeout(() => this.hide(), this.duration);
    }
  }
  render() {
    return (h("div", { ref: el => (this.alert = el), part: "base", class: {
        alert: true,
        'alert--open': this.open,
        'alert--visible': this.isVisible,
        'alert--closable': this.closable,
        'alert--primary': this.type === 'primary',
        'alert--success': this.type === 'success',
        'alert--info': this.type === 'info',
        'alert--warning': this.type === 'warning',
        'alert--danger': this.type === 'danger'
      }, role: "alert", "aria-live": "assertive", "aria-atomic": "true", "aria-hidden": this.open ? 'false' : 'true', onMouseMove: this.handleMouseMove, onTransitionEnd: this.handleTransitionEnd },
      h("span", { part: "icon", class: "alert__icon" },
        h("slot", { name: "icon" })),
      h("span", { part: "message", class: "alert__message" },
        h("slot", null)),
      this.closable && (h("span", { class: "alert__close" },
        h("sl-icon-button", { exportparts: "base:close-button", name: "x", onClick: this.handleCloseClick })))));
  }
  static get is() { return "sl-alert"; }
  static get encapsulation() { return "shadow"; }
  static get originalStyleUrls() { return {
    "$": ["alert.scss"]
  }; }
  static get styleUrls() { return {
    "$": ["alert.css"]
  }; }
  static get properties() { return {
    "open": {
      "type": "boolean",
      "mutable": true,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "Indicates whether or not the alert is open. You can use this in lieu of the show/hide methods."
      },
      "attribute": "open",
      "reflect": true,
      "defaultValue": "false"
    },
    "closable": {
      "type": "boolean",
      "mutable": false,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "Set to true to make the alert closable."
      },
      "attribute": "closable",
      "reflect": true,
      "defaultValue": "false"
    },
    "type": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "'primary' | 'success' | 'info' | 'warning' | 'danger'",
        "resolved": "\"danger\" | \"info\" | \"primary\" | \"success\" | \"warning\"",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The type of alert."
      },
      "attribute": "type",
      "reflect": true,
      "defaultValue": "'primary'"
    },
    "duration": {
      "type": "number",
      "mutable": false,
      "complexType": {
        "original": "number",
        "resolved": "number",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The length of time, in milliseconds, the alert will show before closing itself. If the user interacts with the\nalert before it closes (e.g. moves the mouse over it), the timer will restart."
      },
      "attribute": "duration",
      "reflect": false,
      "defaultValue": "Infinity"
    }
  }; }
  static get states() { return {
    "isVisible": {}
  }; }
  static get events() { return [{
      "method": "slShow",
      "name": "sl-show",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted when the alert opens. Calling `event.preventDefault()` will prevent it from being opened."
      },
      "complexType": {
        "original": "any",
        "resolved": "any",
        "references": {}
      }
    }, {
      "method": "slAfterShow",
      "name": "sl-after-show",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted after the alert opens and all transitions are complete."
      },
      "complexType": {
        "original": "any",
        "resolved": "any",
        "references": {}
      }
    }, {
      "method": "slHide",
      "name": "sl-hide",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted when the alert closes. Calling `event.preventDefault()` will prevent it from being closed."
      },
      "complexType": {
        "original": "any",
        "resolved": "any",
        "references": {}
      }
    }, {
      "method": "slAfterHide",
      "name": "sl-after-hide",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted after the alert closes and all transitions are complete."
      },
      "complexType": {
        "original": "any",
        "resolved": "any",
        "references": {}
      }
    }]; }
  static get methods() { return {
    "show": {
      "complexType": {
        "signature": "() => Promise<void>",
        "parameters": [],
        "references": {
          "Promise": {
            "location": "global"
          }
        },
        "return": "Promise<void>"
      },
      "docs": {
        "text": "Shows the alert.",
        "tags": []
      }
    },
    "hide": {
      "complexType": {
        "signature": "() => Promise<void>",
        "parameters": [],
        "references": {
          "Promise": {
            "location": "global"
          }
        },
        "return": "Promise<void>"
      },
      "docs": {
        "text": "Hides the alert",
        "tags": []
      }
    },
    "toast": {
      "complexType": {
        "signature": "() => Promise<void>",
        "parameters": [],
        "references": {
          "Promise": {
            "location": "global"
          }
        },
        "return": "Promise<void>"
      },
      "docs": {
        "text": "Displays the alert as a toast notification. This will move the alert out of its position in the DOM and, when\ndismissed, it will be removed from the DOM completely. By storing a reference to the alert, you can reuse it by\ncalling this method again. The returned promise will resolve after the alert is hidden.",
        "tags": []
      }
    }
  }; }
  static get elementRef() { return "host"; }
  static get watchers() { return [{
      "propName": "open",
      "methodName": "handleOpenChange"
    }, {
      "propName": "duration",
      "methodName": "handleDurationChange"
    }]; }
}
