import { Component, Element, Event, Prop, State, Watch, h } from '@stencil/core';
import { requestInclude } from './request';
/**
 * @since 2.0
 * @status stable
 */
export class Include {
  constructor() {
    this.html = '';
    /** The fetch mode to use. */
    this.mode = 'cors';
    /**
     * Allows included scripts to be executed. You must ensure the content you're including is trusted, otherwise this
     * option can lead to XSS vulnerabilities in your app!
     */
    this.allowScripts = false;
  }
  handleSrcChange() {
    this.loadSource();
  }
  componentWillLoad() {
    this.loadSource();
  }
  executeScript(script) {
    // Create a copy of the script and swap it out so the browser executes it
    const newScript = document.createElement('script');
    [...script.attributes].map(attr => newScript.setAttribute(attr.name, attr.value));
    newScript.textContent = script.textContent;
    script.parentNode.replaceChild(newScript, script);
  }
  async loadSource() {
    try {
      const src = this.src;
      const file = await requestInclude(src, this.mode);
      // If the src changed since the request started do nothing, otherwise we risk overwriting a subsequent response
      if (src !== this.src) {
        return;
      }
      if (!file.ok) {
        this.slError.emit({ status: file.status });
        return;
      }
      this.host.innerHTML = file.html;
      if (this.allowScripts) {
        [...this.host.querySelectorAll('script')].map(script => this.executeScript(script));
      }
      this.slLoad.emit();
    }
    catch (_a) {
      this.slError.emit();
    }
  }
  render() {
    return h("slot", null);
  }
  static get is() { return "sl-include"; }
  static get encapsulation() { return "shadow"; }
  static get originalStyleUrls() { return {
    "$": ["include.scss"]
  }; }
  static get styleUrls() { return {
    "$": ["include.css"]
  }; }
  static get properties() { return {
    "src": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The location of the HTML file to include."
      },
      "attribute": "src",
      "reflect": false
    },
    "mode": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "'cors' | 'no-cors' | 'same-origin'",
        "resolved": "\"cors\" | \"no-cors\" | \"same-origin\"",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The fetch mode to use."
      },
      "attribute": "mode",
      "reflect": false,
      "defaultValue": "'cors'"
    },
    "allowScripts": {
      "type": "boolean",
      "mutable": false,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "Allows included scripts to be executed. You must ensure the content you're including is trusted, otherwise this\noption can lead to XSS vulnerabilities in your app!"
      },
      "attribute": "allow-scripts",
      "reflect": false,
      "defaultValue": "false"
    }
  }; }
  static get states() { return {
    "html": {}
  }; }
  static get events() { return [{
      "method": "slLoad",
      "name": "sl-load",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted when the included file is loaded."
      },
      "complexType": {
        "original": "any",
        "resolved": "any",
        "references": {}
      }
    }, {
      "method": "slError",
      "name": "sl-error",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted when the included file fails to load due to an error."
      },
      "complexType": {
        "original": "{ status?: number }",
        "resolved": "{ status?: number; }",
        "references": {}
      }
    }]; }
  static get elementRef() { return "host"; }
  static get watchers() { return [{
      "propName": "src",
      "methodName": "handleSrcChange"
    }]; }
}
