import { Component, Element, Event, Method, Prop, State, Watch, h } from '@stencil/core';
import FormControl from '../../functional-components/form-control/form-control';
import { getTextContent } from '../../utilities/slot';
import { hasSlot } from '../../utilities/slot';
let id = 0;
/**
 * @since 2.0
 * @status stable
 *
 * @slot - The select's options in the form of menu items.
 * @slot label - The select's label. Alternatively, you can use the label prop.
 * @slot help-text - Help text that describes how to use the select.
 *
 * @part base - The component's base wrapper.
 * @part clear-button - The input's clear button, exported from <sl-input>.
 * @part form-control - The form control that wraps the label, input, and help text.
 * @part help-text - The select's help text.
 * @part icon - The select's icon.
 * @part label - The select's label.
 * @part menu - The select menu, a <sl-menu> element.
 * @part tag - The multiselect option, a <sl-tag> element.
 * @part tags - The container in which multiselect options are rendered.
 */
export class Select {
  constructor() {
    this.inputId = `select-${++id}`;
    this.labelId = `select-label-${id}`;
    this.helpTextId = `select-help-text-${id}`;
    this.hasFocus = false;
    this.hasHelpTextSlot = false;
    this.hasLabelSlot = false;
    this.isOpen = false;
    this.items = [];
    this.displayLabel = '';
    this.displayTags = [];
    /** Set to true to enable multiselect. */
    this.multiple = false;
    /**
     * The maximum number of tags to show when `multiple` is true. After the maximum, "+n" will be shown to indicate the
     * number of additional items that are selected. Set to -1 to remove the limit.
     */
    this.maxTagsVisible = 3;
    /** Set to true to disable the select control. */
    this.disabled = false;
    /** The select's name. */
    this.name = '';
    /** The select's placeholder text. */
    this.placeholder = '';
    /** The select's size. */
    this.size = 'medium';
    /**
     * Enable this option to prevent the panel from being clipped when the component is placed inside a container with
     * `overflow: auto|scroll`.
     */
    this.hoist = false;
    /** The value of the control. This will be a string or an array depending on `multiple`. */
    this.value = '';
    /** Set to true to draw a pill-style select with rounded edges. */
    this.pill = false;
    /** The select's label. Alternatively, you can use the label slot. */
    this.label = '';
    /** The select's help text. Alternatively, you can use the help-text slot. */
    this.helpText = '';
    /** The select's required attribute. */
    this.required = false;
    /** Set to true to add a clear button when the select is populated. */
    this.clearable = false;
    /** This will be true when the control is in an invalid state. Validity is determined by the `required` prop. */
    this.invalid = false;
  }
  handleDisabledChange() {
    if (this.disabled && this.isOpen) {
      this.dropdown.hide();
    }
  }
  handleLabelChange() {
    this.handleSlotChange();
  }
  handleMultipleChange() {
    // Cast to array | string based on `this.multiple`
    const value = this.getValueAsArray();
    this.value = this.multiple ? value : value[0] || '';
    this.syncItemsFromValue();
  }
  handleValueChange() {
    this.syncItemsFromValue();
    this.slChange.emit();
  }
  connectedCallback() {
    this.handleBlur = this.handleBlur.bind(this);
    this.handleFocus = this.handleFocus.bind(this);
    this.handleClearClick = this.handleClearClick.bind(this);
    this.handleKeyDown = this.handleKeyDown.bind(this);
    this.handleLabelClick = this.handleLabelClick.bind(this);
    this.handleMenuHide = this.handleMenuHide.bind(this);
    this.handleMenuShow = this.handleMenuShow.bind(this);
    this.handleMenuSelect = this.handleMenuSelect.bind(this);
    this.handleSlotChange = this.handleSlotChange.bind(this);
    this.handleTagInteraction = this.handleTagInteraction.bind(this);
    this.host.shadowRoot.addEventListener('slotchange', this.handleSlotChange);
  }
  componentWillLoad() {
    this.handleSlotChange();
  }
  componentDidLoad() {
    this.resizeObserver = new ResizeObserver(() => this.resizeMenu());
    this.reportDuplicateItemValues();
    // We need to do an initial sync after the component has rendered, so this will suppress the re-render warning
    requestAnimationFrame(() => this.syncItemsFromValue());
  }
  disconnectedCallback() {
    this.host.shadowRoot.removeEventListener('slotchange', this.handleSlotChange);
  }
  /** Checks for validity and shows the browser's validation message if the control is invalid. */
  async reportValidity() {
    return this.input.reportValidity();
  }
  /** Sets a custom validation message. If `message` is not empty, the field will be considered invalid. */
  async setCustomValidity(message) {
    this.input.setCustomValidity(message);
    this.invalid = !this.input.checkValidity();
  }
  getItemLabel(item) {
    const slot = item.shadowRoot.querySelector('slot:not([name])');
    return getTextContent(slot);
  }
  getItems() {
    return [...this.host.querySelectorAll('sl-menu-item')];
  }
  getValueAsArray() {
    return Array.isArray(this.value) ? this.value : [this.value];
  }
  handleBlur() {
    this.hasFocus = false;
    this.slBlur.emit();
  }
  handleFocus() {
    this.hasFocus = true;
    this.slFocus.emit();
  }
  handleClearClick(event) {
    event.stopPropagation();
    this.value = this.multiple ? [] : '';
    this.syncItemsFromValue();
  }
  handleKeyDown(event) {
    const target = event.target;
    const items = this.getItems();
    const firstItem = items[0];
    const lastItem = items[items.length - 1];
    // Ignore key presses on tags
    if (target.tagName.toLowerCase() === 'sl-tag') {
      return;
    }
    // Tabbing out of the control closes it
    if (event.key === 'Tab') {
      if (this.isOpen) {
        this.dropdown.hide();
      }
      return;
    }
    // Up/down opens the menu
    if (['ArrowDown', 'ArrowUp'].includes(event.key)) {
      event.preventDefault();
      // Show the menu if it's not already open
      if (!this.isOpen) {
        this.dropdown.show();
      }
      // Focus on a menu item
      if (event.key === 'ArrowDown' && firstItem) {
        firstItem.setFocus();
        return;
      }
      if (event.key === 'ArrowUp' && lastItem) {
        lastItem.setFocus();
        return;
      }
    }
    // All other keys open the menu and initiate type to select
    if (!this.isOpen) {
      event.stopPropagation();
      event.preventDefault();
      this.dropdown.show();
      this.menu.typeToSelect(event.key);
    }
  }
  handleLabelClick() {
    this.box.focus();
  }
  handleMenuSelect(event) {
    const item = event.detail.item;
    if (this.multiple) {
      this.value = this.value.includes(item.value)
        ? this.value.filter(v => v !== item.value)
        : [...this.value, item.value];
    }
    else {
      this.value = item.value;
    }
    this.syncItemsFromValue();
  }
  handleMenuShow(event) {
    if (this.disabled) {
      event.preventDefault();
      return;
    }
    this.resizeMenu();
    this.resizeObserver.observe(this.host);
    this.isOpen = true;
  }
  handleMenuHide() {
    this.resizeObserver.unobserve(this.host);
    this.isOpen = false;
  }
  handleSlotChange() {
    this.hasHelpTextSlot = hasSlot(this.host, 'help-text');
    this.hasLabelSlot = hasSlot(this.host, 'label');
    this.syncItemsFromValue();
    this.reportDuplicateItemValues();
  }
  handleTagInteraction(event) {
    // Don't toggle the menu when a tag's clear button is activated
    const path = event.composedPath();
    const clearButton = path.find(el => {
      if (el instanceof HTMLElement) {
        const element = el;
        return element.classList.contains('tag__clear');
      }
    });
    if (clearButton) {
      event.stopPropagation();
    }
  }
  reportDuplicateItemValues() {
    const items = this.getItems();
    // Report duplicate values since they can break selection logic
    const duplicateValues = items.map(item => item.value).filter((e, i, a) => a.indexOf(e) !== i);
    if (duplicateValues.length) {
      throw new Error('Duplicate value found on <sl-menu-item> in <sl-select>: "' + duplicateValues.join('", "') + '"');
    }
  }
  resizeMenu() {
    this.menu.style.width = `${this.box.clientWidth}px`;
  }
  syncItemsFromValue() {
    const items = this.getItems();
    const value = this.getValueAsArray();
    // Sync checked states
    items.map(item => (item.checked = value.includes(item.value)));
    // Sync display label
    if (this.multiple) {
      const checkedItems = [];
      value.map(val => items.map(item => (item.value === val ? checkedItems.push(item) : null)));
      this.displayTags = checkedItems.map(item => {
        return (h("sl-tag", { exportparts: "base:tag", type: "info", size: this.size, pill: this.pill, clearable: true, onClick: this.handleTagInteraction, onKeyDown: this.handleTagInteraction, "onSl-clear": event => {
            event.stopPropagation();
            if (!this.disabled) {
              item.checked = false;
              this.syncValueFromItems();
            }
          } }, this.getItemLabel(item)));
      });
      if (this.maxTagsVisible > 0 && this.displayTags.length > this.maxTagsVisible) {
        const total = this.displayTags.length;
        this.displayLabel = '';
        this.displayTags = this.displayTags.slice(0, this.maxTagsVisible);
        this.displayTags.push(h("sl-tag", { exportparts: "base:tag", type: "info", size: this.size },
          "+",
          total - this.maxTagsVisible));
      }
    }
    else {
      const checkedItem = items.filter(item => item.value === value[0])[0];
      this.displayLabel = checkedItem ? this.getItemLabel(checkedItem) : '';
      this.displayTags = [];
    }
  }
  syncValueFromItems() {
    const items = this.getItems();
    const checkedItems = items.filter(item => item.checked);
    const checkedValues = checkedItems.map(item => item.value);
    if (this.multiple) {
      this.value = this.value.filter(val => checkedValues.includes(val));
    }
    else {
      this.value = checkedValues.length > 0 ? checkedValues[0] : '';
    }
  }
  render() {
    var _a;
    const hasSelection = this.multiple ? this.value.length > 0 : this.value !== '';
    return (h(FormControl, { inputId: this.inputId, label: this.label, labelId: this.labelId, hasLabelSlot: this.hasLabelSlot, helpTextId: this.helpTextId, helpText: this.helpText, hasHelpTextSlot: this.hasHelpTextSlot, size: this.size, onLabelClick: this.handleLabelClick },
      h("sl-dropdown", { part: "base", ref: el => (this.dropdown = el), hoist: this.hoist, closeOnSelect: !this.multiple, containingElement: this.host, class: {
          select: true,
          'select--open': this.isOpen,
          'select--empty': ((_a = this.value) === null || _a === void 0 ? void 0 : _a.length) === 0,
          'select--focused': this.hasFocus,
          'select--clearable': this.clearable,
          'select--disabled': this.disabled,
          'select--multiple': this.multiple,
          'select--has-tags': this.multiple && hasSelection,
          'select--placeholder-visible': this.displayLabel === '',
          'select--small': this.size === 'small',
          'select--medium': this.size === 'medium',
          'select--large': this.size === 'large',
          'select--pill': this.pill,
          'select--invalid': this.invalid
        }, "onSl-show": this.handleMenuShow, "onSl-hide": this.handleMenuHide },
        h("div", { slot: "trigger", ref: el => (this.box = el), id: this.inputId, class: "select__box", role: "combobox", "aria-labelledby": this.labelId, "aria-describedby": this.helpTextId, "aria-haspopup": "true", "aria-expanded": this.isOpen ? 'true' : 'false', tabIndex: this.disabled ? -1 : 0, onBlur: this.handleBlur, onFocus: this.handleFocus, onKeyDown: this.handleKeyDown },
          h("div", { class: "select__label" }, this.displayTags.length ? (h("span", { part: "tags", class: "select__tags" }, this.displayTags)) : (this.displayLabel || this.placeholder)),
          this.clearable && hasSelection && (h("sl-icon-button", { exportparts: "base:clear-button", class: "select__clear", name: "x-circle", onClick: this.handleClearClick, tabindex: "-1" })),
          h("span", { part: "icon", class: "select__icon" },
            h("sl-icon", { name: "chevron-down" })),
          h("input", { ref: el => (this.input = el), class: "select__hidden-select", "aria-hidden": "true", required: this.required, value: hasSelection ? '1' : '', tabIndex: -1 })),
        h("sl-menu", { ref: el => (this.menu = el), part: "menu", class: "select__menu", "onSl-select": this.handleMenuSelect },
          h("slot", { onSlotchange: this.handleSlotChange })))));
  }
  static get is() { return "sl-select"; }
  static get encapsulation() { return "shadow"; }
  static get originalStyleUrls() { return {
    "$": ["select.scss"]
  }; }
  static get styleUrls() { return {
    "$": ["select.css"]
  }; }
  static get properties() { return {
    "multiple": {
      "type": "boolean",
      "mutable": false,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "Set to true to enable multiselect."
      },
      "attribute": "multiple",
      "reflect": false,
      "defaultValue": "false"
    },
    "maxTagsVisible": {
      "type": "number",
      "mutable": false,
      "complexType": {
        "original": "number",
        "resolved": "number",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The maximum number of tags to show when `multiple` is true. After the maximum, \"+n\" will be shown to indicate the\nnumber of additional items that are selected. Set to -1 to remove the limit."
      },
      "attribute": "max-tags-visible",
      "reflect": false,
      "defaultValue": "3"
    },
    "disabled": {
      "type": "boolean",
      "mutable": false,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "Set to true to disable the select control."
      },
      "attribute": "disabled",
      "reflect": false,
      "defaultValue": "false"
    },
    "name": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The select's name."
      },
      "attribute": "name",
      "reflect": false,
      "defaultValue": "''"
    },
    "placeholder": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The select's placeholder text."
      },
      "attribute": "placeholder",
      "reflect": false,
      "defaultValue": "''"
    },
    "size": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "'small' | 'medium' | 'large'",
        "resolved": "\"large\" | \"medium\" | \"small\"",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The select's size."
      },
      "attribute": "size",
      "reflect": false,
      "defaultValue": "'medium'"
    },
    "hoist": {
      "type": "boolean",
      "mutable": false,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "Enable this option to prevent the panel from being clipped when the component is placed inside a container with\n`overflow: auto|scroll`."
      },
      "attribute": "hoist",
      "reflect": false,
      "defaultValue": "false"
    },
    "value": {
      "type": "string",
      "mutable": true,
      "complexType": {
        "original": "string | Array<string>",
        "resolved": "string | string[]",
        "references": {
          "Array": {
            "location": "global"
          }
        }
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The value of the control. This will be a string or an array depending on `multiple`."
      },
      "attribute": "value",
      "reflect": false,
      "defaultValue": "''"
    },
    "pill": {
      "type": "boolean",
      "mutable": false,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "Set to true to draw a pill-style select with rounded edges."
      },
      "attribute": "pill",
      "reflect": false,
      "defaultValue": "false"
    },
    "label": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The select's label. Alternatively, you can use the label slot."
      },
      "attribute": "label",
      "reflect": false,
      "defaultValue": "''"
    },
    "helpText": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The select's help text. Alternatively, you can use the help-text slot."
      },
      "attribute": "help-text",
      "reflect": false,
      "defaultValue": "''"
    },
    "required": {
      "type": "boolean",
      "mutable": false,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "The select's required attribute."
      },
      "attribute": "required",
      "reflect": false,
      "defaultValue": "false"
    },
    "clearable": {
      "type": "boolean",
      "mutable": false,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "Set to true to add a clear button when the select is populated."
      },
      "attribute": "clearable",
      "reflect": false,
      "defaultValue": "false"
    },
    "invalid": {
      "type": "boolean",
      "mutable": true,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean",
        "references": {}
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "This will be true when the control is in an invalid state. Validity is determined by the `required` prop."
      },
      "attribute": "invalid",
      "reflect": false,
      "defaultValue": "false"
    }
  }; }
  static get states() { return {
    "hasFocus": {},
    "hasHelpTextSlot": {},
    "hasLabelSlot": {},
    "isOpen": {},
    "items": {},
    "displayLabel": {},
    "displayTags": {}
  }; }
  static get events() { return [{
      "method": "slChange",
      "name": "sl-change",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted when the control's value changes."
      },
      "complexType": {
        "original": "any",
        "resolved": "any",
        "references": {}
      }
    }, {
      "method": "slFocus",
      "name": "sl-focus",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted when the control gains focus."
      },
      "complexType": {
        "original": "any",
        "resolved": "any",
        "references": {}
      }
    }, {
      "method": "slBlur",
      "name": "sl-blur",
      "bubbles": true,
      "cancelable": true,
      "composed": true,
      "docs": {
        "tags": [],
        "text": "Emitted when the control loses focus."
      },
      "complexType": {
        "original": "any",
        "resolved": "any",
        "references": {}
      }
    }]; }
  static get methods() { return {
    "reportValidity": {
      "complexType": {
        "signature": "() => Promise<boolean>",
        "parameters": [],
        "references": {
          "Promise": {
            "location": "global"
          }
        },
        "return": "Promise<boolean>"
      },
      "docs": {
        "text": "Checks for validity and shows the browser's validation message if the control is invalid.",
        "tags": []
      }
    },
    "setCustomValidity": {
      "complexType": {
        "signature": "(message: string) => Promise<void>",
        "parameters": [{
            "tags": [],
            "text": ""
          }],
        "references": {
          "Promise": {
            "location": "global"
          }
        },
        "return": "Promise<void>"
      },
      "docs": {
        "text": "Sets a custom validation message. If `message` is not empty, the field will be considered invalid.",
        "tags": []
      }
    }
  }; }
  static get elementRef() { return "host"; }
  static get watchers() { return [{
      "propName": "disabled",
      "methodName": "handleDisabledChange"
    }, {
      "propName": "helpText",
      "methodName": "handleLabelChange"
    }, {
      "propName": "label",
      "methodName": "handleLabelChange"
    }, {
      "propName": "multiple",
      "methodName": "handleMultipleChange"
    }, {
      "propName": "value",
      "methodName": "handleValueChange"
    }]; }
}
