import { r as registerInstance, c as createEvent, h, g as getElement } from './index-9f8c8536.js';
import { s as scrollIntoView } from './scroll-63c302dc.js';
import { P as Popover } from './popover-81b3bf0e.js';

function isTabbable(el) {
  const tabIndex = el.tabIndex;
  return tabIndex > -1;
}
function getNearestTabbableElement(el) {
  // Check the element
  if (isTabbable(el)) {
    return el;
  }
  // Check the element's shadow root
  if (el.shadowRoot) {
    const tabbableShadowChild = [...el.shadowRoot.children].find(isTabbable);
    if (tabbableShadowChild) {
      return tabbableShadowChild;
    }
  }
  // Check the element's children
  if (el.children) {
    return [...el.children].map(getNearestTabbableElement)[0];
  }
  return null;
}

const dropdownCss = ":host{position:relative;box-sizing:border-box}:host *,:host *:before,:host *:after{box-sizing:inherit}:host{display:inline-block}.dropdown{position:relative}.dropdown__trigger{display:block}.dropdown__positioner{position:absolute;z-index:var(--sl-z-index-dropdown)}.dropdown__panel{max-height:50vh;font-family:var(--sl-font-sans);font-size:var(--sl-font-size-medium);font-weight:var(--sl-font-weight-normal);color:var(--color);background-color:var(--sl-panel-background-color);border:solid 1px var(--sl-panel-border-color);border-radius:var(--sl-border-radius-medium);box-shadow:var(--sl-shadow-large);opacity:0;overflow:auto;overscroll-behavior:none;pointer-events:none;transform:scale(0.9);transition:var(--sl-transition-fast) opacity, var(--sl-transition-fast) transform}.dropdown__positioner[data-popper-placement^=top] .dropdown__panel{transform-origin:bottom}.dropdown__positioner[data-popper-placement^=bottom] .dropdown__panel{transform-origin:top}.dropdown__positioner[data-popper-placement^=left] .dropdown__panel{transform-origin:right}.dropdown__positioner[data-popper-placement^=right] .dropdown__panel{transform-origin:left}.dropdown__positioner.popover-visible .dropdown__panel{opacity:1;transform:none;pointer-events:all}";

let id = 0;
const Dropdown = class {
  constructor(hostRef) {
    registerInstance(this, hostRef);
    this.slShow = createEvent(this, "sl-show", 7);
    this.slAfterShow = createEvent(this, "sl-after-show", 7);
    this.slHide = createEvent(this, "sl-hide", 7);
    this.slAfterHide = createEvent(this, "sl-after-hide", 7);
    this.componentId = `dropdown-${++id}`;
    this.isVisible = false;
    /** Indicates whether or not the dropdown is open. You can use this in lieu of the show/hide methods. */
    this.open = false;
    /**
     * The preferred placement of the dropdown panel. Note that the actual placement may vary as needed to keep the panel
     * inside of the viewport.
     */
    this.placement = 'bottom-start';
    /** Determines whether the dropdown should hide when a menu item is selected. */
    this.closeOnSelect = true;
    /** The distance in pixels from which to offset the panel away from its trigger. */
    this.distance = 2;
    /** The distance in pixels from which to offset the panel along its trigger. */
    this.skidding = 0;
    /**
     * Enable this option to prevent the panel from being clipped when the component is placed inside a container with
     * `overflow: auto|scroll`.
     */
    this.hoist = false;
  }
  handleOpenChange() {
    this.open ? this.show() : this.hide();
    this.updateAccessibleTrigger();
  }
  handlePopoverOptionsChange() {
    this.popover.setOptions({
      strategy: this.hoist ? 'fixed' : 'absolute',
      placement: this.placement,
      distance: this.distance,
      skidding: this.skidding
    });
  }
  connectedCallback() {
    if (!this.containingElement) {
      this.containingElement = this.host;
    }
    this.handleDocumentKeyDown = this.handleDocumentKeyDown.bind(this);
    this.handleDocumentMouseDown = this.handleDocumentMouseDown.bind(this);
    this.handleMenuItemActivate = this.handleMenuItemActivate.bind(this);
    this.handlePanelSelect = this.handlePanelSelect.bind(this);
    this.handleTriggerClick = this.handleTriggerClick.bind(this);
    this.handleTriggerKeyDown = this.handleTriggerKeyDown.bind(this);
    this.handleTriggerKeyUp = this.handleTriggerKeyUp.bind(this);
    this.handleTriggerSlotChange = this.handleTriggerSlotChange.bind(this);
  }
  componentDidLoad() {
    this.popover = new Popover(this.trigger, this.positioner, {
      strategy: this.hoist ? 'fixed' : 'absolute',
      placement: this.placement,
      distance: this.distance,
      skidding: this.skidding,
      transitionElement: this.panel,
      onAfterHide: () => this.slAfterHide.emit(),
      onAfterShow: () => this.slAfterShow.emit(),
      onTransitionEnd: () => {
        if (!this.open) {
          this.panel.scrollTop = 0;
        }
      }
    });
    // Show on init if open
    if (this.open) {
      this.show();
    }
  }
  disconnectedCallback() {
    this.hide();
    this.popover.destroy();
  }
  /** Shows the dropdown panel */
  async show() {
    // Prevent subsequent calls to the method, whether manually or triggered by the `open` watcher
    if (this.isVisible) {
      return;
    }
    const slShow = this.slShow.emit();
    if (slShow.defaultPrevented) {
      this.open = false;
      return;
    }
    this.panel.addEventListener('sl-activate', this.handleMenuItemActivate);
    this.panel.addEventListener('sl-select', this.handlePanelSelect);
    document.addEventListener('keydown', this.handleDocumentKeyDown);
    document.addEventListener('mousedown', this.handleDocumentMouseDown);
    this.isVisible = true;
    this.open = true;
    this.popover.show();
  }
  /** Hides the dropdown panel */
  async hide() {
    // Prevent subsequent calls to the method, whether manually or triggered by the `open` watcher
    if (!this.isVisible) {
      return;
    }
    const slHide = this.slHide.emit();
    if (slHide.defaultPrevented) {
      this.open = true;
      return;
    }
    this.panel.removeEventListener('sl-activate', this.handleMenuItemActivate);
    this.panel.removeEventListener('sl-select', this.handlePanelSelect);
    document.addEventListener('keydown', this.handleDocumentKeyDown);
    document.removeEventListener('mousedown', this.handleDocumentMouseDown);
    this.isVisible = false;
    this.open = false;
    this.popover.hide();
  }
  focusOnTrigger() {
    const slot = this.trigger.querySelector('slot');
    const trigger = slot.assignedElements({ flatten: true })[0];
    if (trigger) {
      if (typeof trigger.setFocus === 'function') {
        trigger.setFocus();
      }
      else if (typeof trigger.focus === 'function') {
        trigger.focus();
      }
    }
  }
  getMenu() {
    return this.panel
      .querySelector('slot')
      .assignedElements({ flatten: true })
      .filter(el => el.tagName.toLowerCase() === 'sl-menu')[0];
  }
  handleDocumentKeyDown(event) {
    var _a;
    // Close when escape is pressed
    if (event.key === 'Escape') {
      this.hide();
      this.focusOnTrigger();
      return;
    }
    // Handle tabbing
    if (event.key === 'Tab') {
      // Tabbing within an open menu should close the dropdown and refocus the trigger
      if (this.open && ((_a = document.activeElement) === null || _a === void 0 ? void 0 : _a.tagName.toLowerCase()) === 'sl-menu-item') {
        event.preventDefault();
        this.hide();
        this.focusOnTrigger();
        return;
      }
      // Tabbing outside of the containing element closes the panel
      //
      // If the dropdown is used within a shadow DOM, we need to obtain the activeElement within that shadowRoot,
      // otherwise `document.activeElement` will only return the name of the parent shadow DOM element.
      setTimeout(() => {
        var _a;
        const activeElement = this.containingElement.getRootNode() instanceof ShadowRoot
          ? (_a = document.activeElement.shadowRoot) === null || _a === void 0 ? void 0 : _a.activeElement : document.activeElement;
        if ((activeElement === null || activeElement === void 0 ? void 0 : activeElement.closest(this.containingElement.tagName.toLowerCase())) !== this.containingElement) {
          this.hide();
          return;
        }
      });
    }
  }
  handleDocumentMouseDown(event) {
    // Close when clicking outside of the containing element
    const path = event.composedPath();
    if (!path.includes(this.containingElement)) {
      this.hide();
      return;
    }
  }
  handleMenuItemActivate(event) {
    const item = event.target;
    scrollIntoView(item, this.panel);
  }
  handlePanelSelect(event) {
    const target = event.target;
    // Hide the dropdown when a menu item is selected
    if (this.closeOnSelect && target.tagName.toLowerCase() === 'sl-menu') {
      this.hide();
      this.focusOnTrigger();
    }
  }
  handleTriggerClick() {
    this.open ? this.hide() : this.show();
  }
  handleTriggerKeyDown(event) {
    const menu = this.getMenu();
    const menuItems = menu ? [...menu.querySelectorAll('sl-menu-item')] : null;
    const firstMenuItem = menuItems[0];
    const lastMenuItem = menuItems[menuItems.length - 1];
    // Close when escape or tab is pressed
    if (event.key === 'Escape') {
      this.focusOnTrigger();
      this.hide();
      return;
    }
    // When spacebar/enter is pressed, show the panel but don't focus on the menu. This let's the user press the same
    // key again to hide the menu in case they don't want to make a selection.
    if ([' ', 'Enter'].includes(event.key)) {
      event.preventDefault();
      this.open ? this.hide() : this.show();
      return;
    }
    // When up/down is pressed, we make the assumption that the user is familiar with the menu and plans to make a
    // selection. Rather than toggle the panel, we focus on the menu (if one exists) and activate the first item for
    // faster navigation.
    if (['ArrowDown', 'ArrowUp'].includes(event.key)) {
      event.preventDefault();
      // Show the menu if it's not already open
      if (!this.open) {
        this.show();
      }
      // Focus on a menu item
      if (event.key === 'ArrowDown' && firstMenuItem) {
        firstMenuItem.setFocus();
        return;
      }
      if (event.key === 'ArrowUp' && lastMenuItem) {
        lastMenuItem.setFocus();
        return;
      }
    }
    // Other keys bring focus to the menu and initiate type-to-select behavior
    const ignoredKeys = ['Tab', 'Shift', 'Meta', 'Ctrl', 'Alt'];
    if (this.open && menu && !ignoredKeys.includes(event.key)) {
      menu.typeToSelect(event.key);
      return;
    }
  }
  handleTriggerKeyUp(event) {
    // Prevent space from triggering a click event in Firefox
    if (event.key === ' ') {
      event.preventDefault();
    }
  }
  handleTriggerSlotChange() {
    this.updateAccessibleTrigger();
  }
  //
  // Slotted triggers can be arbitrary content, but we need to link them to the dropdown panel with `aria-haspopup` and
  // `aria-expanded`. These must be applied to the "accessible trigger" (the tabbable portion of the trigger element
  // that gets slotted in) so screen readers will understand them. The accessible trigger could be the slotted element,
  // a child of the slotted element, or an element in the slotted element's shadow root.
  //
  // For example, the accessible trigger of an <sl-button> is a <button> located inside its shadow root.
  //
  // To determine this, we assume the first tabbable element in the trigger slot is the "accessible trigger."
  //
  updateAccessibleTrigger() {
    const slot = this.trigger.querySelector('slot');
    const assignedElements = slot.assignedElements({ flatten: true });
    const accessibleTrigger = assignedElements.map(getNearestTabbableElement)[0];
    if (accessibleTrigger) {
      accessibleTrigger.setAttribute('aria-haspopup', 'true');
      accessibleTrigger.setAttribute('aria-expanded', this.open ? 'true' : 'false');
    }
  }
  render() {
    return (h("div", { part: "base", id: this.componentId, class: {
        dropdown: true,
        'dropdown--open': this.open
      } }, h("span", { part: "trigger", class: "dropdown__trigger", ref: el => (this.trigger = el), onClick: this.handleTriggerClick, onKeyDown: this.handleTriggerKeyDown, onKeyUp: this.handleTriggerKeyUp }, h("slot", { name: "trigger", onSlotchange: this.handleTriggerSlotChange })), h("div", { ref: el => (this.positioner = el), class: "dropdown__positioner" }, h("div", { ref: el => (this.panel = el), part: "panel", class: "dropdown__panel", role: "menu", "aria-hidden": this.open ? 'false' : 'true', "aria-labelledby": this.componentId }, h("slot", null)))));
  }
  get host() { return getElement(this); }
  static get watchers() { return {
    "open": ["handleOpenChange"],
    "distance": ["handlePopoverOptionsChange"],
    "hoist": ["handlePopoverOptionsChange"],
    "placement": ["handlePopoverOptionsChange"],
    "skidding": ["handlePopoverOptionsChange"]
  }; }
};
Dropdown.style = dropdownCss;

export { Dropdown as sl_dropdown };
