import {
  __spreadValues
} from "./chunk.ICGTMF5Z.js";

// src/internal/formdata-event-polyfill.ts
var FormDataEventPolyfill = class extends Event {
  constructor(formData) {
    super("formdata");
    this.formData = formData;
  }
};
var FormDataPolyfill = class extends FormData {
  constructor(form) {
    super(form);
    this.form = form;
    form.dispatchEvent(new FormDataEventPolyfill(this));
  }
  append(name, value) {
    let input = this.form.elements[name];
    if (!input) {
      input = document.createElement("input");
      input.type = "hidden";
      input.name = name;
      this.form.appendChild(input);
    }
    if (this.has(name)) {
      const entries = this.getAll(name);
      const index = entries.indexOf(input.value);
      if (index !== -1) {
        entries.splice(index, 1);
      }
      entries.push(value);
      this.set(name, entries);
    } else {
      super.append(name, value);
    }
    input.value = value;
  }
};
function supportsFormDataEvent() {
  const form = document.createElement("form");
  let isSupported = false;
  document.body.append(form);
  form.addEventListener("submit", (event) => {
    new FormData(event.target);
    event.preventDefault();
  });
  form.addEventListener("formdata", () => isSupported = true);
  form.dispatchEvent(new Event("submit", { cancelable: true }));
  form.remove();
  return isSupported;
}
function polyfillFormData() {
  if (!window.FormData || supportsFormDataEvent()) {
    return;
  }
  window.FormData = FormDataPolyfill;
  window.addEventListener("submit", (event) => {
    if (!event.defaultPrevented) {
      new FormData(event.target);
    }
  });
}
if (document.readyState === "complete") {
  polyfillFormData();
} else {
  window.addEventListener("DOMContentLoaded", () => polyfillFormData());
}

// src/internal/form.ts
var FormSubmitController = class {
  constructor(host, options) {
    (this.host = host).addController(this);
    this.options = __spreadValues({
      form: (input) => input.closest("form"),
      name: (input) => input.name,
      value: (input) => input.value,
      disabled: (input) => input.disabled,
      reportValidity: (input) => {
        return typeof input.reportValidity === "function" ? input.reportValidity() : true;
      }
    }, options);
    this.handleFormData = this.handleFormData.bind(this);
    this.handleFormSubmit = this.handleFormSubmit.bind(this);
  }
  hostConnected() {
    this.form = this.options.form(this.host);
    if (this.form) {
      this.form.addEventListener("formdata", this.handleFormData);
      this.form.addEventListener("submit", this.handleFormSubmit);
    }
  }
  hostDisconnected() {
    if (this.form) {
      this.form.removeEventListener("formdata", this.handleFormData);
      this.form.removeEventListener("submit", this.handleFormSubmit);
      this.form = void 0;
    }
  }
  handleFormData(event) {
    const disabled = this.options.disabled(this.host);
    const name = this.options.name(this.host);
    const value = this.options.value(this.host);
    if (!disabled && typeof name === "string" && typeof value !== "undefined") {
      if (Array.isArray(value)) {
        value.forEach((val) => {
          event.formData.append(name, val.toString());
        });
      } else {
        event.formData.append(name, value.toString());
      }
    }
  }
  handleFormSubmit(event) {
    const disabled = this.options.disabled(this.host);
    const reportValidity = this.options.reportValidity;
    if (this.form && !this.form.noValidate && !disabled && !reportValidity(this.host)) {
      event.preventDefault();
      event.stopImmediatePropagation();
    }
  }
  submit(submitter) {
    if (this.form) {
      const button = document.createElement("button");
      button.type = "submit";
      button.style.position = "absolute";
      button.style.width = "0";
      button.style.height = "0";
      button.style.clip = "rect(0 0 0 0)";
      button.style.clipPath = "inset(50%)";
      button.style.overflow = "hidden";
      button.style.whiteSpace = "nowrap";
      if (submitter) {
        ["formaction", "formmethod", "formnovalidate", "formtarget"].forEach((attr) => {
          if (submitter.hasAttribute(attr)) {
            button.setAttribute(attr, submitter.getAttribute(attr));
          }
        });
      }
      this.form.append(button);
      button.click();
      button.remove();
    }
  }
};

export {
  FormSubmitController
};
