'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _listingsSerializer = require('./serializers/listings-serializer');

var _listingsSerializer2 = _interopRequireDefault(_listingsSerializer);

var _listingsAdapter = require('./adapters/listings-adapter');

var _listingsAdapter2 = _interopRequireDefault(_listingsAdapter);

var _cartSerializer = require('./serializers/cart-serializer');

var _cartSerializer2 = _interopRequireDefault(_cartSerializer);

var _referenceSerializer = require('./serializers/reference-serializer');

var _referenceSerializer2 = _interopRequireDefault(_referenceSerializer);

var _localStorageAdapter = require('./adapters/local-storage-adapter');

var _localStorageAdapter2 = _interopRequireDefault(_localStorageAdapter);

var _coreObject = require('./metal/core-object');

var _coreObject2 = _interopRequireDefault(_coreObject);

var _assign = require('./metal/assign');

var _assign2 = _interopRequireDefault(_assign);

var _setGuidFor = require('./metal/set-guid-for');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @module shopify-buy
 * @submodule shop-client
 */

function fetchFactory(fetchType, type) {
  var func = void 0;

  switch (fetchType) {
    case 'all':
      func = function func() {
        return this.fetchAll(type);
      };
      break;
    case 'one':
      func = function func() {
        return this.fetch.apply(this, [type].concat(Array.prototype.slice.call(arguments)));
      };
      break;
    case 'query':
      func = function func() {
        return this.fetchQuery.apply(this, [type].concat(Array.prototype.slice.call(arguments)));
      };
      break;
  }

  return func;
}

var ShopClient = _coreObject2.default.extend({
  /**
   * @class ShopClient
   * @constructor
   * @param {Config} [config] Config data to be used throughout all API
   * interaction
   */

  constructor: function constructor(config) {
    this.config = config;

    this.serializers = {
      products: _listingsSerializer2.default,
      collections: _listingsSerializer2.default,
      carts: _cartSerializer2.default,
      references: _referenceSerializer2.default
    };

    this.adapters = {
      products: _listingsAdapter2.default,
      collections: _listingsAdapter2.default,
      carts: _localStorageAdapter2.default,
      references: _localStorageAdapter2.default
    };
  },


  config: null,

  /**
   * @attribute
   * @default {
   *  products: ListingsAdapter,
   *  collections: ListingsAdapter,
   *  carts: CartAdapter
   * }
   * @type Object
   * @protected
   */
  // Prevent leaky state
  get serializers() {
    return (0, _assign2.default)({}, this.shadowedSerializers);
  },

  set serializers(values) {
    this.shadowedSerializers = (0, _assign2.default)({}, values);
  },

  get adapters() {
    return (0, _assign2.default)({}, this.shadowedAdapters);
  },

  set adapters(values) {
    this.shadowedAdapters = (0, _assign2.default)({}, values);
  },

  /**
   * Fetch all of a `type`, returning a promise.
   *
   * ```javascript
   * client.fetchAll('products').then(products => {
   *   // do things with products
   * });
   * ```
   *
   * @method fetchAll
   * @private
   * @param {String} type The pluralized name of the type, in lower case.
   * @return {Promise|Array} a promise resolving with an array of `type`
   */
  fetchAll: function fetchAll(type) {
    var _this = this;

    var adapter = new this.adapters[type](this.config);

    return adapter.fetchMultiple(type).then(function (payload) {
      return _this.deserialize(type, payload, adapter, null, { multiple: true });
    });
  },


  /**
   * Fetch one of a `type`, returning a promise.
   *
   * ```javascript
   * client.fetch('products', 123).then(product => {
   *   // do things with the product
   * });
   * ```
   *
   * @method fetch
   * @private
   * @param {String} type The pluralized name of the type, in lower case.
   * @param {String|Number} id a unique identifier
   * @return {Promise|BaseModel} a promise resolving with a single instance of
   * `type` expressed as a `BaseModel`.
   */
  fetch: function fetch(type, id) {
    var _this2 = this;

    var adapter = new this.adapters[type](this.config);

    return adapter.fetchSingle(type, id).then(function (payload) {
      return _this2.deserialize(type, payload, adapter, null, { single: true });
    });
  },


  /**
   * Fetch many of a `type`, that match `query`
   *
   * ```javascript
   * client.fetchQuery('products', { collection_id: 456 }).then(products => {
   *   // do things with the products
   * });
   * ```
   *
   * @method fetchQuery
   * @private
   * @param {String} type The pluralized name of the type, in lower case.
   * @param {Object} query a query sent to the api server.
   * @return {Promise|Array} a promise resolving with an array of `type`.
   */
  fetchQuery: function fetchQuery(type, query) {
    var _this3 = this;

    var adapter = new this.adapters[type](this.config);

    return adapter.fetchMultiple(type, query).then(function (payload) {
      return _this3.deserialize(type, payload, adapter, null, { multiple: true });
    });
  },


  /**
   * Create an instance of `type`, optionally including `modelAttrs`.
   *
   * ```javascript
   * client.create('carts', { line_items: [ ... ] }).then(cart => {
   *   // do things with the cart.
   * });
   * ```
   *
   * @method create
   * @private
   * @param {String} type The pluralized name of the type, in lower case.
   * @param {Object} [modelAttrs={}] attributes representing the internal state
   * of the model to be persisted.
   * @return {Promise|CartModel} a promise resolving with a single instance of
   * `type`
   */
  create: function create(type) {
    var _this4 = this;

    var modelAttrs = arguments.length <= 1 || arguments[1] === undefined ? {} : arguments[1];

    var adapter = new this.adapters[type](this.config);
    var serializer = new this.serializers[type](this.config);
    var Model = serializer.modelForType(type);
    var model = new Model(modelAttrs, { shopClient: this });
    var attrs = serializer.serialize(type, model);

    return adapter.create(type, attrs).then(function (payload) {
      return _this4.deserialize(type, payload, adapter, serializer, { single: true });
    });
  },


  /**
   * Create an instance of `type`, optionally including `attrs`.
   *
   * ```javascript
   * client.create('carts', { line_items: [ ... ] }).then(cart => {
   *   // do things with the cart.
   * });
   * ```
   *
   * @method update
   * @private
   * @param {String} type The pluralized name of the type, in lower case.
   * @param {BaseModel} updatedModel The model that represents new state to
   * to persist.
   * @return {Promise|CartModel} a promise resolving with a single instance of
   * `type`
   */
  update: function update(type, updatedModel) {
    var _this5 = this;

    var adapter = updatedModel.adapter;
    var serializer = updatedModel.serializer;
    var serializedModel = serializer.serialize(type, updatedModel);
    var id = updatedModel.attrs[adapter.idKeyForType(type)];

    return adapter.update(type, id, serializedModel).then(function (payload) {
      return _this5.deserialize(type, payload, adapter, serializer, { single: true });
    });
  },


  /**
   * Proxy to serializer's deserialize.
   *
   * @method deserialize
   * @private
   * @param {String} type The pluralized name of the type, in lower case.
   * @param {Object} payload The raw payload returned by the adapter.
   * @param {BaseAdapter} adapter The adapter that yielded the payload.
   * @param {BaseSerializer} existingSerializer The serializer to attach. If
   * none is passed, then `this.deserialize` will create one for the type.
   * @param {Object} opts Options that determine which deserialization method to
   * use.
   * @param {Boolean} opts.multiple true when the payload represents multiple
   * models
   * @param {Boolean} opts.single true when the payload represents one model.
   * @return {BaseModel} an instance of `type` reified into a model.
   */
  deserialize: function deserialize(type, payload, adapter, existingSerializer) {
    var opts = arguments.length <= 4 || arguments[4] === undefined ? {} : arguments[4];

    var serializer = existingSerializer || new this.serializers[type](this.config);
    var meta = { shopClient: this, adapter: adapter, serializer: serializer, type: type };
    var serializedPayload = void 0;

    if (opts.multiple) {
      serializedPayload = serializer.deserializeMultiple(type, payload, meta);
    } else {
      serializedPayload = serializer.deserializeSingle(type, payload, meta);
    }

    return serializedPayload;
  },


  /**
    * Creates a {{#crossLink "CartModel"}}CartModel{{/crossLink}} instance, optionally including `attrs`.
    *
    * ```javascript
    * client.createCart().then(cart => {
    *   // do something with cart
    * });
    * ```
    *
    * @param {Object}[attrs={}] attributes representing the internal state of the cart to be persisted to localStorage.
    * @method createCart
    * @public
    * @return {Promise|CartModel} - new cart instance.
  */
  createCart: function createCart() {
    var userAttrs = arguments.length <= 0 || arguments[0] === undefined ? {} : arguments[0];

    var baseAttrs = {
      line_items: []
    };
    var attrs = {};

    (0, _assign2.default)(attrs, baseAttrs);
    (0, _assign2.default)(attrs, userAttrs);

    return this.create('carts', attrs);
  },


  /**
    * Updates an existing {{#crossLink "CartModel"}}CartModel{{/crossLink}} instance and persists it to localStorage.
    *
    * ```javascript
    * client.createCart().then(cart => {
    *   cart.lineItems = [
    *     // ...
    *   ];
    *   client.updateCart(cart);
    * });
    * ```
    *
    * @param {Object}[attrs={}] attributes representing the internal state of the cart to be persisted to localStorage.
    * @method updateCart
    * @public
    * @return {Promise|CartModel} - updated cart instance.
  */
  updateCart: function updateCart(updatedCart) {
    return this.update('carts', updatedCart);
  },


  /**
   * Retrieve a previously created cart by its key.
   *
   * ```javascript
   * client.fetchCart('shopify-buy.1459804699118.2').then(cart => {
   *   console.log(cart); // The retrieved cart
   * });
   *
   * @method fetchCart
   * @public
   * @param {String} id The cart's unique identifier
   * @return {Promise|CartModel} The cart model.
   *
   */
  fetchCart: fetchFactory('one', 'carts'),

  /**
   * Convenience wrapper for {{#crossLink "ShopClient/fetchAll:method"}}
   * {{/crossLink}}. Equivalent to:
   *
   * ```javascript
   * client.fetchAll('products');
   * ```
   *
   * @method fetchAllProducts
   * @private
   * @return {Promise|Array} The product models.
   */
  fetchAllProducts: fetchFactory('all', 'products'),

  /**
   * Convenience wrapper for {{#crossLink "ShopClient/fetchAll:method"}}
   * {{/crossLink}}. Equivalent to:
   *
   * ```javascript
   * client.fetchAll('collections');
   * ```
   *
   * @method fetchAllCollections
   * @private
   * @return {Promise|Array} The collection models.
   */
  fetchAllCollections: fetchFactory('all', 'collections'),

  /**
   * Fetch one product by its ID.
   *
   * ```javascript
   * client.fetchProduct(123).then(product => {
   *   console.log(product); // The product with an ID of 123
   * });
   * ```
   *
   * @method fetchProduct
   * @public
   * @param {String|Number} id a unique identifier
   * @return {Promise|BaseModel} The product model with the specified ID.
   */
  fetchProduct: fetchFactory('one', 'products'),

  /**
   * Fetch one collection by its ID.
   *
   * ```javascript
   * client.fetchCollection(123).then(collection => {
   *   console.log(collection); // The collection with an ID of 123
   * });
   * ```
   *
   * @method fetchCollection
   * @public
   * @param {String|Number} id a unique identifier
   * @return {Promise|BaseModel} The collection model with the specified ID.
   */
  fetchCollection: fetchFactory('one', 'collections'),

  /**
   * Fetches a list of products matching a specified query.
   *
   * ```javascript
   * client.fetchQueryProducts({ collection_id: 123}).then(products => {
   *   console.log(products); // An array of products in collection 123
   * });
   * ```
   * @method fetchQueryProducts
   * @public
   * @param {Object} query a query sent to the api server containing one or more of:
   *   @param {String|Number} [query.collection_id] the ID of a collection to retrieve products from
   *   @param {Array} [query.product_ids] a list of product IDs to retrieve
   *   @param {String|Number} [query.page=1] the page offset number of the current lookup (based on the `limit`)
   *   @param {String|Number} [query.limit=50] the number of products to retrieve per page
   * @return {Promise|Array} The product models.
   */
  fetchQueryProducts: fetchFactory('query', 'products'),

  /**
   * Fetches a list of collections matching a specified query.
   *
   * ```javascript
   * client.fetchQueryCollections({page: 2, limit: 20}).then(collections => {
   *   console.log(collections); // An array of collection resources
   * });
   * ```
   *
   * @method fetchQueryCollections
   * @public
   * @param {Object} query a query sent to the api server.
   *   @param {String|Number} [query.page=1] the page offset number of the current lookup (based on the `limit`)
   *   @param {String|Number} [query.limit=50] the number of collections to retrieve per page
   * @return {Promise|Array} The collection models.
   */
  fetchQueryCollections: fetchFactory('query', 'collections'),

  /**
   * This method looks up a reference in localStorage to the most recent cart.
   * If one is not found, creates one. If the cart the reference points to
   * doesn't exist, create one and store the new reference.
   *
   * ```javascript
   * client.fetchRecentCart().then(cart => {
   *  // do stuff with the cart
   * });
   * ```
   *
   * @method fetchRecentCart
   * @public
   * @return {Promise|CartModel} The cart.
   */
  fetchRecentCart: function fetchRecentCart() {
    var _this6 = this;

    return this.fetch('references', this.config.myShopifyDomain + '.recent-cart').then(function (reference) {
      var cartId = reference.referenceId;

      return _this6.fetchCart(cartId);
    }).catch(function () {
      return _this6.createCart().then(function (cart) {
        var refAttrs = {
          referenceId: cart.id
        };

        refAttrs[_setGuidFor.GUID_KEY] = _this6.config.myShopifyDomain + '.recent-cart';

        _this6.create('references', refAttrs);

        return cart;
      });
    });
  }
});

exports.default = ShopClient;