"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var dates_1 = require("../dates");
describe('abbreviationForWeekday()', function () {
    it('abbreviates the word correctly', function () {
        expect(dates_1.abbreviationForWeekday(dates_1.Weekdays.Sunday)).toBe('Su');
        expect(dates_1.abbreviationForWeekday(dates_1.Weekdays.Monday)).toBe('Mo');
        expect(dates_1.abbreviationForWeekday(dates_1.Weekdays.Tuesday)).toBe('Tu');
        expect(dates_1.abbreviationForWeekday(dates_1.Weekdays.Wednesday)).toBe('We');
        expect(dates_1.abbreviationForWeekday(dates_1.Weekdays.Thursday)).toBe('Th');
        expect(dates_1.abbreviationForWeekday(dates_1.Weekdays.Friday)).toBe('Fr');
        expect(dates_1.abbreviationForWeekday(dates_1.Weekdays.Saturday)).toBe('Sa');
    });
});
describe('getDateDiff()', function () {
    var now = new Date();
    var testDate = new Date(now.getTime());
    testDate.setFullYear(now.getFullYear() - 1);
    it('returns expected diff in seconds', function () {
        var diff = dates_1.getDateDiff(dates_1.TimeUnit.Second, testDate, now);
        expect(diff).toEqual(dates_1.TimeUnit.Year / dates_1.TimeUnit.Second);
    });
    it('returns expected diff in minutes', function () {
        var diff = dates_1.getDateDiff(dates_1.TimeUnit.Minute, testDate, now);
        expect(diff).toEqual(dates_1.TimeUnit.Year / dates_1.TimeUnit.Minute);
    });
    it('returns expected diff in hours', function () {
        var diff = dates_1.getDateDiff(dates_1.TimeUnit.Hour, testDate, now);
        expect(diff).toEqual(dates_1.TimeUnit.Year / dates_1.TimeUnit.Hour);
    });
    it('returns expected diff in days', function () {
        var diff = dates_1.getDateDiff(dates_1.TimeUnit.Day, testDate, now);
        expect(diff).toEqual(dates_1.TimeUnit.Year / dates_1.TimeUnit.Day);
    });
    it('returns expected diff in weeks', function () {
        var diff = dates_1.getDateDiff(dates_1.TimeUnit.Week, testDate, now);
        expect(diff).toEqual(Math.floor(dates_1.TimeUnit.Year / dates_1.TimeUnit.Week));
    });
    it('returns expected diff in years', function () {
        var diff = dates_1.getDateDiff(dates_1.TimeUnit.Year, testDate, now);
        expect(diff).toEqual(dates_1.TimeUnit.Year / dates_1.TimeUnit.Year);
    });
    describe('second date defaults to today', function () {
        it('returns expected diff in seconds', function () {
            var diff = dates_1.getDateDiff(dates_1.TimeUnit.Second, testDate);
            expect(diff).toEqual(dates_1.TimeUnit.Year / dates_1.TimeUnit.Second);
        });
        it('returns expected diff in minutes', function () {
            var diff = dates_1.getDateDiff(dates_1.TimeUnit.Minute, testDate);
            expect(diff).toEqual(dates_1.TimeUnit.Year / dates_1.TimeUnit.Minute);
        });
        it('returns expected diff in hours', function () {
            var diff = dates_1.getDateDiff(dates_1.TimeUnit.Hour, testDate);
            expect(diff).toEqual(dates_1.TimeUnit.Year / dates_1.TimeUnit.Hour);
        });
        it('returns expected diff in days', function () {
            var diff = dates_1.getDateDiff(dates_1.TimeUnit.Day, testDate);
            expect(diff).toEqual(dates_1.TimeUnit.Year / dates_1.TimeUnit.Day);
        });
        it('returns expected diff in weeks', function () {
            var diff = dates_1.getDateDiff(dates_1.TimeUnit.Week, testDate);
            expect(diff).toEqual(Math.floor(dates_1.TimeUnit.Year / dates_1.TimeUnit.Week));
        });
        it('returns expected diff in years', function () {
            var diff = dates_1.getDateDiff(dates_1.TimeUnit.Year, testDate);
            expect(diff).toEqual(dates_1.TimeUnit.Year / dates_1.TimeUnit.Year);
        });
    });
});
describe('getWeeksForMonth()', function () {
    it('starts the week on Sunday by default', function () {
        var weeks = dates_1.getWeeksForMonth(1, 2018);
        weeks.map(function (week) {
            var startDay = week[0];
            if (startDay !== null) {
                expect(startDay.getDay()).toBe(dates_1.Weekdays.Sunday);
            }
        });
    });
    it('always has 7 value for each weeks', function () {
        var weeks = dates_1.getWeeksForMonth(6, 2018);
        weeks.map(function (week) {
            expect(week).toHaveLength(7);
        });
    });
    it('first day of the week is the one passed as argument', function () {
        [
            dates_1.Weekdays.Sunday,
            dates_1.Weekdays.Monday,
            dates_1.Weekdays.Tuesday,
            dates_1.Weekdays.Wednesday,
            dates_1.Weekdays.Thursday,
            dates_1.Weekdays.Friday,
            dates_1.Weekdays.Saturday,
        ].map(function (weekDay) {
            var weeks = dates_1.getWeeksForMonth(8, 2018, weekDay);
            weeks.map(function (week) {
                var startDay = week[0];
                if (startDay !== null) {
                    expect(startDay.getDay()).toBe(weekDay);
                }
                else {
                    expect(startDay).toBeNull();
                }
            });
        });
    });
    it('sets values to null before first day of month', function () {
        var weeks = dates_1.getWeeksForMonth(6, 2018, dates_1.Weekdays.Monday);
        expect(weeks[0][0]).toBeNull();
        expect(weeks[0][1]).toBeNull();
        expect(weeks[0][2]).toBeNull();
        expect(weeks[0][3]).toBeNull();
        expect(weeks[0][4]).toBeNull();
        expect(weeks[0][5]).toBeNull();
        expect(weeks[0][6]).not.toBeNull();
    });
});
describe('getNewRange()', function () {
    it('returns a new range with end date in the future', function () {
        // startDate and endDate are the same date but have different references for
        // test purposes
        var startDate = new Date('01 Jan 2018 00:00:00 GMT');
        var endDate = new Date('01 Jan 2018 00:00:00 GMT');
        var futureDate = new Date('05 Jan 2018 00:00:00 GMT');
        var range = {
            start: startDate,
            end: endDate,
        };
        expect(dates_1.getNewRange(range, futureDate)).toEqual({
            start: startDate,
            end: futureDate,
        });
    });
});
describe('isLessThanOneMinuteAgo', function () {
    it('returns false for dates more than one minute apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setSeconds(now.getSeconds() - 61);
        expect(dates_1.isLessThanOneMinuteAgo(other, now)).toBe(false);
    });
    it('returns false for dates exactly one minute apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setSeconds(now.getSeconds() - 60);
        expect(dates_1.isLessThanOneMinuteAgo(other, now)).toBe(false);
    });
    it('returns true for dates less than one minute apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setSeconds(now.getSeconds() - 59);
        expect(dates_1.isLessThanOneMinuteAgo(other, now)).toBe(true);
    });
});
describe('isLessThanOneHourAgo', function () {
    it('returns false for dates more than one hour apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setMinutes(now.getMinutes() - 61);
        expect(dates_1.isLessThanOneHourAgo(other, now)).toBe(false);
    });
    it('returns false for dates exactly one hour apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setMinutes(now.getMinutes() - 60);
        expect(dates_1.isLessThanOneHourAgo(other, now)).toBe(false);
    });
    it('returns true for dates less than one hour apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setMinutes(now.getMinutes() - 59);
        expect(dates_1.isLessThanOneHourAgo(other, now)).toBe(true);
    });
});
describe('isLessThanOneDayAgo', function () {
    it('returns false for dates more than one day apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setHours(now.getHours() - 25);
        expect(dates_1.isLessThanOneDayAgo(other, now)).toBe(false);
    });
    it('returns false for dates exactly one day apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setHours(now.getHours() - 24);
        expect(dates_1.isLessThanOneDayAgo(other, now)).toBe(false);
    });
    it('returns true for dates less than one day apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setHours(now.getHours() - 23);
        expect(dates_1.isLessThanOneDayAgo(other, now)).toBe(true);
    });
});
describe('isLessThanOneWeekAgo', function () {
    it('returns false for dates more than one week apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setDate(now.getDate() - 8);
        expect(dates_1.isLessThanOneWeekAgo(other, now)).toBe(false);
    });
    it('returns false for dates exactly one week apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setDate(now.getDate() - 7);
        expect(dates_1.isLessThanOneWeekAgo(other, now)).toBe(false);
    });
    it('returns true for dates less than one week apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setDate(now.getDate() - 6);
        expect(dates_1.isLessThanOneWeekAgo(other, now)).toBe(true);
    });
});
describe('isLessThanOneYearAgo', function () {
    it('returns false for dates more than one year apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setDate(now.getDate() - 366);
        expect(dates_1.isLessThanOneYearAgo(other, now)).toBe(false);
    });
    it('returns false for dates exactly one year apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setDate(now.getDate() - 365);
        expect(dates_1.isLessThanOneYearAgo(other, now)).toBe(false);
    });
    it('returns true for dates less than one year apart', function () {
        var now = new Date();
        var other = new Date(now.getTime());
        other.setDate(now.getDate() - 364);
        expect(dates_1.isLessThanOneYearAgo(other, now)).toBe(true);
    });
});
describe('isSameMonthAndYear', function () {
    it('returns true for dates with same month and year', function () {
        var date1 = new Date('01 Jan 2018 00:00:00 GMT');
        var date2 = new Date(date1.getTime());
        expect(dates_1.isSameMonthAndYear(date1, date2)).toBe(true);
        date2.setDate(2);
        expect(dates_1.isSameMonthAndYear(date1, date2)).toBe(true);
    });
    it('returns false for dates with same month and different year', function () {
        var date1 = new Date('01 Jan 2018 00:00:00 GMT');
        var date2 = new Date(date1.getTime());
        date2.setFullYear(date1.getFullYear() + 1);
        expect(dates_1.isSameMonthAndYear(date1, date2)).toBe(false);
    });
    it('returns false for dates with different month and same year', function () {
        var date1 = new Date('01 Jan 2018 00:00:00 GMT');
        var date2 = new Date(date1.getTime());
        date2.setMonth(date1.getMonth() + 1);
        expect(dates_1.isSameMonthAndYear(date1, date2)).toBe(false);
    });
});
describe('isSameDate', function () {
    it('returns true for dates with same day, month, and year', function () {
        var date1 = new Date('01 Jan 2018 00:00:00 GMT');
        var date2 = new Date(date1.getTime());
        expect(dates_1.isSameDate(date1, date2)).toBe(true);
        date2.setHours(2);
        expect(dates_1.isSameDate(date1, date2)).toBe(true);
    });
    it('returns false for dates with same day and month, but different year', function () {
        var date1 = new Date('01 Jan 2018 00:00:00 GMT');
        var date2 = new Date(date1.getTime());
        date2.setFullYear(date1.getFullYear() + 1);
        expect(dates_1.isSameDate(date1, date2)).toBe(false);
    });
    it('returns false for dates with same day and year, but different month', function () {
        var date1 = new Date('01 Jan 2018 00:00:00 GMT');
        var date2 = new Date(date1.getTime());
        date2.setMonth(date1.getMonth() + 1);
        expect(dates_1.isSameDate(date1, date2)).toBe(false);
    });
    it('returns false for dates with same month and year, but different day', function () {
        var date1 = new Date('01 Jan 2018 00:00:00 GMT');
        var date2 = new Date(date1.getTime());
        date2.setDate(date1.getDate() + 1);
        expect(dates_1.isSameDate(date1, date2)).toBe(false);
    });
});
describe('isToday', function () {
    it('returns true for dates with same day, month, and year as today', function () {
        var today = new Date();
        expect(dates_1.isToday(today)).toBe(true);
        // Time is irrelevant
        var differentMinutes = new Date(today.getTime());
        differentMinutes.setMinutes(today.getMinutes() + 1);
        expect(dates_1.isToday(differentMinutes)).toBe(true);
        var differentHours = new Date(today.getTime());
        differentHours.setHours(today.getHours() + 1);
        expect(dates_1.isToday(differentHours)).toBe(true);
    });
    it('returns false for dates with different day, month, or year from today', function () {
        var today = new Date();
        var differentDay = new Date(today.getTime());
        differentDay.setDate(today.getDate() + 1);
        expect(dates_1.isToday(differentDay)).toBe(false);
        var differentMonth = new Date(today.getTime());
        differentMonth.setMonth(today.getMonth() + 1);
        expect(dates_1.isToday(differentMonth)).toBe(false);
        var differentYear = new Date(today.getTime());
        differentYear.setFullYear(today.getFullYear() + 1);
        expect(dates_1.isToday(differentYear)).toBe(false);
    });
});
describe('isYesterday', function () {
    it('returns true for dates with same day, month, and year as the day before today', function () {
        var today = new Date();
        var yesterday = new Date(today.getTime());
        yesterday.setDate(today.getDate() - 1);
        expect(dates_1.isYesterday(yesterday)).toBe(true);
        // Time is irrelevant
        var differentMinutes = new Date(yesterday.getTime());
        differentMinutes.setMinutes(yesterday.getMinutes() + 1);
        expect(dates_1.isYesterday(differentMinutes)).toBe(true);
        var differentHours = new Date(yesterday.getTime());
        differentHours.setHours(yesterday.getHours() + 1);
        expect(dates_1.isYesterday(differentHours)).toBe(true);
    });
    it('returns false for dates with different day, month, or year from today', function () {
        var today = new Date();
        var yesterday = new Date(today.getTime());
        yesterday.setDate(today.getDate() - 1);
        var differentDay = new Date(yesterday.getTime());
        differentDay.setDate(yesterday.getDate() + 1);
        expect(dates_1.isYesterday(differentDay)).toBe(false);
        var differentMonth = new Date(yesterday.getTime());
        differentMonth.setMonth(yesterday.getMonth() + 1);
        expect(dates_1.isYesterday(differentMonth)).toBe(false);
        var differentYear = new Date(yesterday.getTime());
        differentYear.setFullYear(yesterday.getFullYear() + 1);
        expect(dates_1.isYesterday(differentYear)).toBe(false);
    });
});
