import * as React from 'react';
import { createUniqueIDFactory } from '@shopify/javascript-utilities/other';
import OptionList from '../../../OptionList';
import ActionList from '../../../ActionList';
import Popover from '../../../Popover';
import { Key } from '../../../../types';
import { contextTypes } from '../types';
import KeypressListener from '../../../KeypressListener';
import { TextField } from './components';
import styles from './ComboBox.scss';
const getUniqueId = createUniqueIDFactory('ComboBox');
export default class ComboBox extends React.PureComponent {
    constructor() {
        super(...arguments);
        this.state = {
            comboBoxId: this.getComboBoxId(),
            selectedOption: undefined,
            selectedIndex: -1,
            selectedOptions: this.props.selected,
            navigableOptions: [],
            popoverActive: false,
            popoverWasActive: false,
        };
        this.subscriptions = [];
        this.subscribe = (callback) => {
            this.subscriptions.push(callback);
        };
        this.unsubscribe = (callback) => {
            this.subscriptions = this.subscriptions.filter((subscription) => subscription !== callback);
        };
        this.handleDownArrow = () => {
            this.selectNextOption();
            this.handlePopoverOpen;
        };
        this.handleUpArrow = () => {
            this.selectPreviousOption();
            this.handlePopoverOpen;
        };
        this.handleEnter = () => {
            const { selectedOption } = this.state;
            if (this.state.popoverActive && selectedOption) {
                if (isOption(selectedOption)) {
                    this.handleSelection(selectedOption.value);
                }
                else {
                    selectedOption.onAction && selectedOption.onAction();
                }
            }
            this.handlePopoverOpen;
        };
        this.handleFocus = () => {
            this.setState({ popoverActive: true, popoverWasActive: true });
        };
        this.handleBlur = () => {
            this.setState({ popoverActive: false, popoverWasActive: false }, () => {
                this.resetVisuallySelectedOptions();
            });
        };
        this.handleClick = () => {
            !this.state.popoverActive && this.setState({ popoverActive: true });
        };
        this.handleSelection = (newSelected) => {
            const { selected, allowMultiple } = this.props;
            let newlySelectedOptions = selected;
            if (selected.includes(newSelected)) {
                newlySelectedOptions.splice(newlySelectedOptions.indexOf(newSelected), 1);
            }
            else if (allowMultiple) {
                newlySelectedOptions.push(newSelected);
            }
            else {
                newlySelectedOptions = [newSelected];
            }
            this.selectOptions(newlySelectedOptions);
        };
        this.selectOptions = (selected) => {
            const { onSelect, allowMultiple } = this.props;
            selected && onSelect(selected);
            if (!allowMultiple) {
                this.resetVisuallySelectedOptions();
                this.setState({ popoverActive: false, popoverWasActive: false });
            }
        };
        this.updateIndexOfSelectedOption = (newOptions) => {
            const { selectedIndex, selectedOption } = this.state;
            if (selectedOption && newOptions.includes(selectedOption)) {
                this.selectOptionAtIndex(newOptions.indexOf(selectedOption));
            }
            else if (selectedIndex > newOptions.length - 1) {
                this.resetVisuallySelectedOptions();
            }
            else {
                this.selectOptionAtIndex(selectedIndex);
            }
        };
        this.resetVisuallySelectedOptions = () => {
            const { navigableOptions } = this.state;
            this.setState({
                selectedOption: undefined,
                selectedIndex: -1,
            });
            navigableOptions &&
                navigableOptions.forEach((option) => {
                    option.active = false;
                });
        };
        this.handlePopoverClose = () => {
            this.setState({ popoverActive: false, popoverWasActive: false });
        };
        this.handlePopoverOpen = () => {
            const { popoverActive, navigableOptions } = this.state;
            !popoverActive &&
                navigableOptions &&
                navigableOptions.length > 0 &&
                this.setState({ popoverActive: true, popoverWasActive: true });
        };
        this.selectNextOption = () => {
            const { selectedIndex, navigableOptions } = this.state;
            if (!navigableOptions || navigableOptions.length === 0) {
                return;
            }
            let newIndex = selectedIndex;
            if (selectedIndex + 1 >= navigableOptions.length) {
                newIndex = 0;
            }
            else {
                newIndex++;
            }
            this.selectOptionAtIndex(newIndex);
        };
        this.selectPreviousOption = () => {
            const { selectedIndex, navigableOptions } = this.state;
            if (!navigableOptions || navigableOptions.length === 0) {
                return;
            }
            let newIndex = selectedIndex;
            if (selectedIndex <= 0) {
                newIndex = navigableOptions.length - 1;
            }
            else {
                newIndex--;
            }
            this.selectOptionAtIndex(newIndex);
        };
        this.selectOptionAtIndex = (newOptionIndex) => {
            const { navigableOptions, selectedOption: oldSelectedOption } = this.state;
            if (!navigableOptions || navigableOptions.length === 0) {
                return;
            }
            const newSelectedOption = navigableOptions[newOptionIndex];
            this.setState({
                selectedOption: newSelectedOption,
                selectedIndex: newOptionIndex,
            }, () => {
                this.visuallyUpdateSelectedOption(newSelectedOption, oldSelectedOption);
            });
        };
        this.visuallyUpdateSelectedOption = (newOption, oldOption) => {
            if (oldOption) {
                oldOption.active = false;
            }
            if (newOption) {
                newOption.active = true;
            }
        };
    }
    static getDerivedStateFromProps({ options: nextOptions, selected: nextSelected, actionsBefore: nextActionsBefore, actionsAfter: nextActionsAfter, }, { navigableOptions, selectedOptions, comboBoxId }) {
        const optionsChanged = filterForOptions(navigableOptions) &&
            nextOptions &&
            !optionsAreEqual(navigableOptions, nextOptions);
        let newNavigableOptions = [];
        if (nextActionsBefore) {
            newNavigableOptions = newNavigableOptions.concat(nextActionsBefore);
        }
        if (optionsChanged || nextActionsBefore) {
            newNavigableOptions = newNavigableOptions.concat(nextOptions);
        }
        if (nextActionsAfter) {
            newNavigableOptions = newNavigableOptions.concat(nextActionsAfter);
        }
        newNavigableOptions = assignOptionIds(newNavigableOptions, comboBoxId);
        if (optionsChanged && selectedOptions !== nextSelected) {
            return {
                navigableOptions: newNavigableOptions,
                selectedOptions: nextSelected,
            };
        }
        else if (optionsChanged) {
            return {
                navigableOptions: newNavigableOptions,
            };
        }
        else if (selectedOptions !== nextSelected) {
            return { selectedOptions: nextSelected };
        }
        return null;
    }
    getChildContext() {
        return {
            comboBoxId: this.state.comboBoxId,
            selectedOptionId: this.selectedOptionId,
            subscribe: this.subscribe,
            unsubscribe: this.unsubscribe,
        };
    }
    componentDidMount() {
        const { options, actionsBefore, actionsAfter } = this.props;
        const comboBoxId = this.getComboBoxId();
        let navigableOptions = [];
        if (actionsBefore) {
            navigableOptions = navigableOptions.concat(actionsBefore);
        }
        if (options) {
            navigableOptions = navigableOptions.concat(options);
        }
        if (actionsAfter) {
            navigableOptions = navigableOptions.concat(actionsAfter);
        }
        navigableOptions = assignOptionIds(navigableOptions, comboBoxId);
        this.setState({
            navigableOptions,
        });
    }
    componentDidUpdate(_, prevState) {
        const { contentBefore, contentAfter, emptyState } = this.props;
        const { navigableOptions, popoverWasActive } = this.state;
        this.subscriptions.forEach((subscriberCallback) => subscriberCallback());
        const optionsChanged = navigableOptions &&
            prevState.navigableOptions &&
            !optionsAreEqual(navigableOptions, prevState.navigableOptions);
        if (optionsChanged) {
            this.updateIndexOfSelectedOption(navigableOptions);
        }
        if (navigableOptions &&
            navigableOptions.length === 0 &&
            !contentBefore &&
            !contentAfter &&
            !emptyState) {
            // eslint-disable-next-line react/no-did-update-set-state
            this.setState({ popoverActive: false });
        }
        else if (popoverWasActive &&
            navigableOptions &&
            navigableOptions.length !== 0) {
            // eslint-disable-next-line react/no-did-update-set-state
            this.setState({ popoverActive: true });
        }
    }
    getComboBoxId() {
        if (this.state && this.state.comboBoxId) {
            return this.state.comboBoxId;
        }
        return this.props.id || getUniqueId();
    }
    render() {
        const { options, textField, listTitle, allowMultiple, preferredPosition, actionsBefore, actionsAfter, contentBefore, contentAfter, onEndReached, emptyState, } = this.props;
        const actionsBeforeMarkup = actionsBefore &&
            actionsBefore.length > 0 && (<ActionList actionRole="option" items={actionsBefore}/>);
        const actionsAfterMarkup = actionsAfter &&
            actionsAfter.length > 0 && (<ActionList actionRole="option" items={actionsAfter}/>);
        const optionsMarkup = options.length > 0 && (<OptionList role="presentation" optionRole="option" options={filterForOptions(this.state.navigableOptions)} onChange={this.selectOptions} selected={this.state.selectedOptions} title={listTitle} allowMultiple={allowMultiple}/>);
        const emptyStateMarkup = !actionsAfter &&
            !actionsBefore &&
            !contentAfter &&
            !contentBefore &&
            options.length === 0 &&
            emptyState && <div className={styles.EmptyState}>{emptyState}</div>;
        return (<div onClick={this.handleClick} role="combobox" aria-expanded={this.state.popoverActive} aria-owns={this.state.comboBoxId} aria-controls={this.state.comboBoxId} aria-haspopup onFocus={this.handleFocus} onBlur={this.handleBlur} tabIndex={0}>
        <KeypressListener keyCode={Key.DownArrow} handler={this.handleDownArrow}/>
        <KeypressListener keyCode={Key.UpArrow} handler={this.handleUpArrow}/>
        <KeypressListener keyCode={Key.Enter} handler={this.handleEnter}/>
        <KeypressListener keyCode={Key.Escape} handler={this.handlePopoverClose}/>
        <Popover activator={textField} active={this.state.popoverActive} onClose={this.handlePopoverClose} preferredPosition={preferredPosition} fullWidth preventAutofocus>
          <Popover.Pane onScrolledToBottom={onEndReached}>
            <div id={this.state.comboBoxId} role="listbox" aria-multiselectable={allowMultiple}>
              {contentBefore}
              {actionsBeforeMarkup}
              {optionsMarkup}
              {actionsAfterMarkup}
              {contentAfter}
              {emptyStateMarkup}
            </div>
          </Popover.Pane>
        </Popover>
      </div>);
    }
    get selectedOptionId() {
        const { selectedOption, selectedIndex, comboBoxId } = this.state;
        return selectedOption ? `${comboBoxId}-${selectedIndex}` : undefined;
    }
}
ComboBox.TextField = TextField;
ComboBox.OptionList = OptionList;
ComboBox.childContextTypes = contextTypes;
function assignOptionIds(options, comboBoxId) {
    options.map((option, optionIndex) => {
        option.id = `${comboBoxId}-${optionIndex}`;
    });
    return options;
}
function optionsAreEqual(firstOptions, secondOptions) {
    if (firstOptions.length !== secondOptions.length) {
        return false;
    }
    return firstOptions.every((firstItem, index) => {
        const secondItem = secondOptions[index];
        if (isOption(firstItem)) {
            if (isOption(secondItem)) {
                return firstItem.value === secondItem.value;
            }
            return false;
        }
        else {
            if (!isOption(secondItem)) {
                return firstItem.content === secondItem.content;
            }
            return false;
        }
    });
}
function isOption(navigableOption) {
    return navigableOption.value !== undefined;
}
function filterForOptions(mixedArray) {
    return mixedArray.filter((item) => isOption(item));
}
