import * as React from 'react';
import * as PropTypes from 'prop-types';
import { classNames } from '@shopify/react-utilities/styles';
import { addEventListener, removeEventListener, } from '@shopify/javascript-utilities/events';
import { read } from '@shopify/javascript-utilities/fastdom';
import { withAppProvider } from '../AppProvider';
import styles from './Collapsible.scss';
const CONTEXT_TYPES = {
    parentCollapsibleExpanding: PropTypes.bool,
};
export class Collapsible extends React.Component {
    constructor() {
        super(...arguments);
        this.state = {
            height: null,
            animationState: 'idle',
        };
        this.node = null;
        this.heightNode = null;
        this.bindNode = (node) => {
            this.node = node;
        };
        this.bindHeightNode = (node) => {
            this.heightNode = node;
        };
        this.handleTransitionEnd = (event) => {
            const { target } = event;
            if (target === this.node) {
                this.setState({ animationState: 'idle', height: null });
            }
        };
    }
    getChildContext() {
        const { open } = this.props;
        const { animationState } = this.state;
        const { parentCollapsibleExpanding } = this.context;
        return {
            parentCollapsibleExpanding: parentCollapsibleExpanding || (open && animationState !== 'idle'),
        };
    }
    componentWillReceiveProps({ open: willOpen }) {
        const { open } = this.props;
        if (open !== willOpen) {
            this.setState({ animationState: 'measuring' });
        }
    }
    componentDidUpdate({ open: wasOpen }) {
        const { animationState } = this.state;
        const { parentCollapsibleExpanding } = this.context;
        if (parentCollapsibleExpanding && animationState !== 'idle') {
            // eslint-disable-next-line react/no-did-update-set-state
            this.setState({
                animationState: 'idle',
            });
            return;
        }
        read(() => {
            switch (animationState) {
                case 'idle':
                    break;
                case 'measuring':
                    this.setState({
                        animationState: wasOpen ? 'closingStart' : 'openingStart',
                        height: wasOpen && this.heightNode ? this.heightNode.scrollHeight : 0,
                    });
                    break;
                case 'closingStart':
                    this.setState({
                        animationState: 'closing',
                        height: 0,
                    });
                    break;
                case 'openingStart':
                    this.setState({
                        animationState: 'opening',
                        height: this.heightNode ? this.heightNode.scrollHeight : 0,
                    });
            }
        });
    }
    componentDidMount() {
        if (this.node == null) {
            return;
        }
        addEventListener(this.node, 'transitionend', this.handleTransitionEnd);
    }
    componentWillUnmount() {
        if (this.node == null) {
            return;
        }
        removeEventListener(this.node, 'transitionend', this.handleTransitionEnd);
    }
    render() {
        const { id, open, children } = this.props;
        const { animationState, height } = this.state;
        const animating = animationState !== 'idle';
        const wrapperClassName = classNames(styles.Collapsible, open && styles.open, animating && styles.animating, !animating && open && styles.fullyOpen);
        const displayHeight = collapsibleHeight(open, animationState, height);
        const content = animating || open ? children : null;
        return (<div id={id} aria-hidden={!open} style={{ height: displayHeight }} className={wrapperClassName} ref={this.bindNode}>
        <div ref={this.bindHeightNode}>{content}</div>
      </div>);
    }
}
Collapsible.contextTypes = CONTEXT_TYPES;
Collapsible.childContextTypes = CONTEXT_TYPES;
function collapsibleHeight(open, animationState, height) {
    if (animationState === 'idle' && open) {
        return open ? 'auto' : undefined;
    }
    if (animationState === 'measuring') {
        return open ? undefined : 'auto';
    }
    return `${height || 0}px`;
}
export default withAppProvider()(Collapsible);
