import * as React from 'react';
import { classNames } from '@shopify/react-utilities/styles';
import isEqual from 'lodash/isEqual';
import debounce from 'lodash/debounce';
import { headerCell } from '../shared';
import { withAppProvider } from '../AppProvider';
import EventListener from '../EventListener';
import { Cell, Navigation } from './components';
import { measureColumn, getPrevAndCurrentColumns } from './utilities';
import styles from './DataTable.scss';
export class DataTable extends React.PureComponent {
    constructor(props) {
        super(props);
        this.state = {
            collapsed: false,
            columnVisibilityData: [],
            heights: [],
            preservedScrollPosition: {},
            isScrolledFarthestLeft: true,
            isScrolledFarthestRight: false,
        };
        this.dataTable = React.createRef();
        this.scrollContainer = React.createRef();
        this.table = React.createRef();
        this.handleResize = debounce(() => {
            const { footerContent, truncate } = this.props;
            const { table: { current: table }, scrollContainer: { current: scrollContainer }, } = this;
            let collapsed = false;
            if (table && scrollContainer) {
                collapsed = table.scrollWidth > scrollContainer.clientWidth;
                scrollContainer.scrollLeft = 0;
            }
            this.setState(Object.assign({ collapsed, heights: [] }, this.calculateColumnVisibilityData(collapsed)), () => {
                if (footerContent || !truncate) {
                    this.setHeightsAndScrollPosition();
                }
            });
        });
        this.tallestCellHeights = () => {
            const { footerContent, truncate } = this.props;
            const { table: { current: table }, } = this;
            let { heights } = this.state;
            if (table) {
                const rows = Array.from(table.getElementsByTagName('tr'));
                if (!truncate) {
                    return (heights = rows.map((row) => {
                        const fixedCell = row.childNodes[0];
                        return Math.max(row.clientHeight, fixedCell.clientHeight);
                    }));
                }
                if (footerContent) {
                    const footerCellHeight = rows[rows.length - 1]
                        .childNodes[0].clientHeight;
                    heights = [footerCellHeight];
                }
            }
            return heights;
        };
        this.resetScrollPosition = () => {
            const { scrollContainer: { current: scrollContainer }, } = this;
            if (scrollContainer) {
                const { preservedScrollPosition: { left, top }, } = this.state;
                if (left) {
                    scrollContainer.scrollLeft = left;
                }
                if (top) {
                    window.scrollTo(0, top);
                }
            }
        };
        this.setHeightsAndScrollPosition = () => {
            this.setState({ heights: this.tallestCellHeights() }, this.resetScrollPosition);
        };
        this.calculateColumnVisibilityData = (collapsed) => {
            const { table: { current: table }, scrollContainer: { current: scrollContainer }, dataTable: { current: dataTable }, } = this;
            if (collapsed && table && scrollContainer && dataTable) {
                const headerCells = table.querySelectorAll(headerCell.selector);
                const collapsedHeaderCells = Array.from(headerCells).slice(1);
                const fixedColumnWidth = headerCells[0].offsetWidth;
                const firstVisibleColumnIndex = collapsedHeaderCells.length - 1;
                const tableLeftVisibleEdge = scrollContainer.scrollLeft + fixedColumnWidth;
                const tableRightVisibleEdge = scrollContainer.scrollLeft + dataTable.offsetWidth;
                const tableData = {
                    fixedColumnWidth,
                    firstVisibleColumnIndex,
                    tableLeftVisibleEdge,
                    tableRightVisibleEdge,
                };
                const columnVisibilityData = collapsedHeaderCells.map(measureColumn(tableData));
                const lastColumn = columnVisibilityData[columnVisibilityData.length - 1];
                return Object.assign({ fixedColumnWidth,
                    columnVisibilityData }, getPrevAndCurrentColumns(tableData, columnVisibilityData), { isScrolledFarthestLeft: tableLeftVisibleEdge === fixedColumnWidth, isScrolledFarthestRight: lastColumn.rightEdge <= tableRightVisibleEdge });
            }
            return {
                columnVisibilityData: [],
                previousColumn: undefined,
                currentColumn: undefined,
            };
        };
        this.scrollListener = () => {
            this.setState((prevState) => (Object.assign({}, this.calculateColumnVisibilityData(prevState.collapsed))));
        };
        this.navigateTable = (direction) => {
            const { currentColumn, previousColumn, fixedColumnWidth } = this.state;
            const { scrollContainer: { current: scrollContainer }, } = this;
            const handleScroll = () => {
                if (!currentColumn || !previousColumn || !fixedColumnWidth) {
                    return;
                }
                if (scrollContainer) {
                    scrollContainer.scrollLeft =
                        direction === 'right'
                            ? currentColumn.rightEdge - fixedColumnWidth
                            : previousColumn.leftEdge - fixedColumnWidth;
                    requestAnimationFrame(() => {
                        this.setState((prevState) => (Object.assign({}, this.calculateColumnVisibilityData(prevState.collapsed))));
                    });
                }
            };
            return handleScroll;
        };
        this.renderTotals = (total, index) => {
            const id = `totals-cell-${index}`;
            const { heights } = this.state;
            const { truncate = false } = this.props;
            let content;
            let contentType;
            if (index === 0) {
                content = this.totalsRowHeading;
            }
            if (total !== '' && index > 0) {
                contentType = 'numeric';
                content = total;
            }
            return (<Cell total fixed={index === 0} testID={id} key={id} height={heights[1]} content={content} contentType={contentType} truncate={truncate}/>);
        };
        this.defaultRenderRow = (row, index) => {
            const className = classNames(styles.TableRow);
            const { columnContentTypes, totals, footerContent, truncate = false, } = this.props;
            const { heights } = this.state;
            const bodyCellHeights = totals ? heights.slice(2) : heights.slice(1);
            if (footerContent) {
                bodyCellHeights.pop();
            }
            return (<tr key={`row-${index}`} className={className}>
        {row.map((content, cellIndex) => {
                const id = `cell-${cellIndex}-row-${index}`;
                return (<Cell key={id} testID={id} height={bodyCellHeights[index]} content={content} contentType={columnContentTypes[cellIndex]} fixed={cellIndex === 0} truncate={truncate}/>);
            })}
      </tr>);
        };
        this.renderFooter = () => {
            const { heights } = this.state;
            const footerCellHeight = heights[heights.length - 1];
            return (<Cell footer testID="footer-cell" height={footerCellHeight} content={this.props.footerContent} truncate={this.props.truncate}/>);
        };
        this.defaultOnSort = (headingIndex) => {
            const { onSort, truncate, defaultSortDirection = 'ascending', initialSortColumnIndex, } = this.props;
            const { sortDirection = defaultSortDirection, sortedColumnIndex = initialSortColumnIndex, } = this.state;
            let newSortDirection = defaultSortDirection;
            if (sortedColumnIndex === headingIndex) {
                newSortDirection =
                    sortDirection === 'ascending' ? 'descending' : 'ascending';
            }
            const handleSort = () => {
                this.setState({
                    sortDirection: newSortDirection,
                    sortedColumnIndex: headingIndex,
                }, () => {
                    if (onSort) {
                        onSort(headingIndex, newSortDirection);
                        if (!truncate && this.scrollContainer.current) {
                            const preservedScrollPosition = {
                                left: this.scrollContainer.current.scrollLeft,
                                top: window.scrollY,
                            };
                            this.setState({ preservedScrollPosition });
                            this.handleResize();
                        }
                    }
                });
            };
            return handleSort;
        };
        const { translate } = props.polaris.intl;
        this.totalsRowHeading = translate('Polaris.DataTable.totalsRowHeading');
    }
    componentDidMount() {
        // We need to defer the calculation in development so the styles have time to be injected.
        if (process.env.NODE_ENV === 'development') {
            setTimeout(() => {
                this.handleResize();
            }, 10);
        }
        else {
            this.handleResize();
        }
    }
    componentDidUpdate(prevProps) {
        if (isEqual(prevProps, this.props)) {
            return;
        }
        this.handleResize();
    }
    render() {
        const { columnContentTypes, headings, totals, rows, truncate, footerContent, sortable, defaultSortDirection = 'ascending', initialSortColumnIndex = 0, } = this.props;
        const { collapsed, columnVisibilityData, heights, sortedColumnIndex = initialSortColumnIndex, sortDirection = defaultSortDirection, isScrolledFarthestLeft, isScrolledFarthestRight, } = this.state;
        const className = classNames(styles.DataTable, collapsed && styles.collapsed, footerContent && styles.hasFooter);
        const wrapperClassName = classNames(styles.TableWrapper, collapsed && styles.collapsed);
        const footerClassName = classNames(footerContent && styles.TableFoot);
        const footerMarkup = footerContent ? (<tfoot className={footerClassName}>
        <tr>{this.renderFooter()}</tr>
      </tfoot>) : null;
        const totalsMarkup = totals ? (<tr>{totals.map(this.renderTotals)}</tr>) : null;
        const headingMarkup = (<tr>
        {headings.map((heading, headingIndex) => {
            let sortableHeadingProps;
            const id = `heading-cell-${headingIndex}`;
            if (sortable) {
                const isSortable = sortable[headingIndex];
                const isSorted = sortedColumnIndex === headingIndex;
                const direction = isSorted ? sortDirection : 'none';
                sortableHeadingProps = {
                    defaultSortDirection,
                    sorted: isSorted,
                    sortable: isSortable,
                    sortDirection: direction,
                    onSort: this.defaultOnSort(headingIndex),
                };
            }
            const height = !truncate ? heights[0] : undefined;
            return (<Cell header key={id} testID={id} height={height} content={heading} contentType={columnContentTypes[headingIndex]} fixed={headingIndex === 0} truncate={truncate} {...sortableHeadingProps}/>);
        })}
      </tr>);
        const bodyMarkup = rows.map(this.defaultRenderRow);
        const style = footerContent
            ? { marginBottom: `${heights[heights.length - 1]}px` }
            : undefined;
        return (<div className={wrapperClassName}>
        <Navigation columnVisibilityData={columnVisibilityData} isScrolledFarthestLeft={isScrolledFarthestLeft} isScrolledFarthestRight={isScrolledFarthestRight} navigateTableLeft={this.navigateTable('left')} navigateTableRight={this.navigateTable('right')}/>
        <div className={className} ref={this.dataTable}>
          <div className={styles.ScrollContainer} ref={this.scrollContainer} style={style}>
            <EventListener event="resize" handler={this.handleResize}/>
            <EventListener capture event="scroll" handler={this.scrollListener}/>
            <table className={styles.Table} ref={this.table}>
              <thead>
                {headingMarkup}
                {totalsMarkup}
              </thead>
              <tbody>{bodyMarkup}</tbody>
              {footerMarkup}
            </table>
          </div>
        </div>
      </div>);
    }
}
export default withAppProvider()(DataTable);
