import * as React from 'react';
import { MobileCancelMajorMonotone } from '@shopify/polaris-icons';
import { classNames } from '@shopify/react-utilities/styles';
import { durationSlow } from '@shopify/polaris-tokens';
import { CSSTransition } from 'react-transition-group';
import { navigationBarCollapsed } from '../../utilities/breakpoints';
import Button from '../Button';
import Icon from '../Icon';
import EventListener from '../EventListener';
import { withAppProvider } from '../AppProvider';
import Backdrop from '../Backdrop';
import TrapFocus from '../TrapFocus';
import { UserMenuProvider } from '../TopBar';
import { dataPolarisTopBar, layer } from '../shared';
import { setRootProperty } from '../../utilities/setRootProperty';
import { frameContextTypes, } from './types';
import { ToastManager, Loading, ContextualSaveBar } from './components';
import styles from './Frame.scss';
export const GLOBAL_RIBBON_CUSTOM_PROPERTY = '--global-ribbon-height';
export const APP_FRAME_MAIN = 'AppFrameMain';
const APP_FRAME_NAV = 'AppFrameNav';
const APP_FRAME_TOP_BAR = 'AppFrameTopBar';
const APP_FRAME_LOADING_BAR = 'AppFrameLoadingBar';
export class Frame extends React.PureComponent {
    constructor() {
        super(...arguments);
        this.state = {
            skipFocused: false,
            globalRibbonHeight: 0,
            loadingStack: 0,
            toastMessages: [],
            mobileView: isMobileView(),
            showContextualSaveBar: false,
        };
        this.globalRibbonContainer = null;
        this.setGlobalRibbonHeight = () => {
            const { globalRibbonContainer } = this;
            if (globalRibbonContainer) {
                this.setState({
                    globalRibbonHeight: globalRibbonContainer.offsetHeight,
                }, this.setGlobalRibbonRootProperty);
            }
        };
        this.setGlobalRibbonRootProperty = () => {
            const { globalRibbonHeight } = this.state;
            setRootProperty(GLOBAL_RIBBON_CUSTOM_PROPERTY, `${globalRibbonHeight}px`, null);
        };
        this.showToast = (toast) => {
            this.setState(({ toastMessages }) => {
                const hasToastById = toastMessages.find(({ id }) => id === toast.id) != null;
                return {
                    toastMessages: hasToastById ? toastMessages : [...toastMessages, toast],
                };
            });
        };
        this.hideToast = ({ id }) => {
            this.setState(({ toastMessages }) => {
                return {
                    toastMessages: toastMessages.filter(({ id: toastId }) => id !== toastId),
                };
            });
        };
        this.setContextualSaveBar = (props) => {
            const { showContextualSaveBar } = this.state;
            this.contextualSaveBar = Object.assign({}, props);
            if (showContextualSaveBar === true) {
                this.forceUpdate();
            }
            else {
                this.setState({ showContextualSaveBar: true });
            }
        };
        this.removeContextualSaveBar = () => {
            this.contextualSaveBar = null;
            this.setState({ showContextualSaveBar: false });
        };
        this.startLoading = () => {
            this.setState(({ loadingStack }) => ({
                loadingStack: loadingStack + 1,
            }));
        };
        this.stopLoading = () => {
            this.setState(({ loadingStack }) => ({
                loadingStack: Math.max(0, loadingStack - 1),
            }));
        };
        this.handleResize = () => {
            const { mobileView } = this.state;
            if (isMobileView() && !mobileView) {
                this.setState({ mobileView: true });
            }
            else if (!isMobileView() && mobileView) {
                this.setState({ mobileView: false });
            }
            if (this.props.globalRibbon) {
                this.setGlobalRibbonHeight();
            }
        };
        this.handleClick = () => {
            focusAppFrameMain();
        };
        this.handleFocus = () => {
            this.setState({ skipFocused: true });
        };
        this.handleBlur = () => {
            this.setState({ skipFocused: false });
        };
        this.handleNavigationDismiss = () => {
            const { onNavigationDismiss } = this.props;
            if (onNavigationDismiss != null) {
                onNavigationDismiss();
            }
        };
        this.setGlobalRibbonContainer = (node) => {
            this.globalRibbonContainer = node;
        };
        this.handleNavKeydown = (event) => {
            const { key } = event;
            if (key === 'Escape') {
                this.handleNavigationDismiss();
            }
        };
    }
    getChildContext() {
        return {
            frame: {
                showToast: this.showToast,
                hideToast: this.hideToast,
                startLoading: this.startLoading,
                stopLoading: this.stopLoading,
                setContextualSaveBar: this.setContextualSaveBar,
                removeContextualSaveBar: this.removeContextualSaveBar,
            },
        };
    }
    componentDidMount() {
        this.handleResize();
        if (this.props.globalRibbon) {
            return;
        }
        this.setGlobalRibbonRootProperty();
    }
    componentDidUpdate(prevProps) {
        if (this.props.globalRibbon !== prevProps.globalRibbon) {
            this.setGlobalRibbonHeight();
        }
    }
    render() {
        const { skipFocused, loadingStack, toastMessages, showContextualSaveBar, mobileView, } = this.state;
        const { children, navigation, topBar, globalRibbon, showMobileNavigation = false, polaris: { intl }, } = this.props;
        const navClassName = classNames(styles.Navigation, showMobileNavigation && styles['Navigation-visible']);
        const mobileNavHidden = mobileView && !showMobileNavigation;
        const mobileNavShowing = mobileView && showMobileNavigation;
        const tabIndex = mobileNavShowing ? 0 : -1;
        const navigationMarkup = navigation ? (<TrapFocus trapping={mobileNavShowing}>
        <CSSTransition appear={mobileView} exit={mobileView} in={showMobileNavigation} timeout={durationSlow} classNames={navTransitionClasses}>
          <div className={navClassName} onKeyDown={this.handleNavKeydown} id={APP_FRAME_NAV} key="NavContent" hidden={mobileNavHidden}>
            {navigation}
            <button type="button" className={styles.NavigationDismiss} onClick={this.handleNavigationDismiss} aria-hidden={mobileNavHidden || (!mobileView && !showMobileNavigation)} aria-label={intl.translate('Polaris.Frame.Navigation.closeMobileNavigationLabel')} tabIndex={tabIndex}>
              <Icon source={MobileCancelMajorMonotone} color="white"/>
            </button>
          </div>
        </CSSTransition>
      </TrapFocus>) : null;
        const loadingMarkup = loadingStack > 0 ? (<div className={styles.LoadingBar} id={APP_FRAME_LOADING_BAR}>
          <Loading />
        </div>) : null;
        const contextualSaveBarMarkup = (<CSSTransition appear exit in={showContextualSaveBar} timeout={300} classNames={contextualSaveBarTransitionClasses} mountOnEnter unmountOnExit>
        <div className={styles.ContextualSaveBar}>
          <ContextualSaveBar {...this.contextualSaveBar}/>
        </div>
      </CSSTransition>);
        const topBarMarkup = topBar ? (<div className={styles.TopBar} {...layer.props} {...dataPolarisTopBar.props} id={APP_FRAME_TOP_BAR}>
        {topBar}
      </div>) : null;
        const globalRibbonMarkup = globalRibbon ? (<div className={styles.GlobalRibbonContainer} ref={this.setGlobalRibbonContainer}>
        {globalRibbon}
      </div>) : null;
        const skipClassName = classNames(styles.Skip, skipFocused && styles.focused);
        const skipMarkup = (<div className={skipClassName}>
        <Button onClick={this.handleClick} onFocus={this.handleFocus} onBlur={this.handleBlur}>
          {intl.translate('Polaris.Frame.skipToContent')}
        </Button>
      </div>);
        const navigationAttributes = navigation
            ? {
                'data-has-navigation': true,
            }
            : {};
        const frameClassName = classNames(styles.Frame, navigation && styles.hasNav, topBar && styles.hasTopBar);
        const navigationOverlayMarkup = showMobileNavigation && mobileView ? (<Backdrop belowNavigation onClick={this.handleNavigationDismiss} onTouchStart={this.handleNavigationDismiss}/>) : null;
        return (<div className={frameClassName} {...layer.props} {...navigationAttributes}>
        {skipMarkup}
        <UserMenuProvider mobileView={mobileView || false}>
          {topBarMarkup}
          {navigationMarkup}
        </UserMenuProvider>
        {contextualSaveBarMarkup}
        {loadingMarkup}
        {navigationOverlayMarkup}
        <main className={styles.Main} id={APP_FRAME_MAIN} data-has-global-ribbon={Boolean(globalRibbon)}>
          <div className={styles.Content}>{children}</div>
        </main>
        <ToastManager toastMessages={toastMessages}/>
        {globalRibbonMarkup}
        <EventListener event="resize" handler={this.handleResize}/>
      </div>);
    }
}
Frame.childContextTypes = frameContextTypes;
const navTransitionClasses = {
    enter: classNames(styles['Navigation-enter']),
    enterActive: classNames(styles['Navigation-enterActive']),
    enterDone: classNames(styles['Navigation-enterActive']),
    exit: classNames(styles['Navigation-exit']),
    exitActive: classNames(styles['Navigation-exitActive']),
};
const contextualSaveBarTransitionClasses = {
    enter: classNames(styles['ContextualSaveBar-enter']),
    enterActive: classNames(styles['ContextualSaveBar-enterActive']),
    enterDone: classNames(styles['ContextualSaveBar-enterActive']),
    exit: classNames(styles['ContextualSaveBar-exit']),
    exitActive: classNames(styles['ContextualSaveBar-exitActive']),
};
function focusAppFrameMain() {
    window.location.assign(`${window.location.pathname}#${APP_FRAME_MAIN}`);
}
function isMobileView() {
    return navigationBarCollapsed().matches;
}
export default withAppProvider()(Frame);
