import * as React from 'react';
import { createUniqueIDFactory } from '@shopify/javascript-utilities/other';
import { arraysAreEqual } from '../../utilities/arrays';
import { withAppProvider } from '../AppProvider';
import { Option } from './components';
import styles from './OptionList.scss';
const getUniqueId = createUniqueIDFactory('OptionList');
export class OptionList extends React.Component {
    constructor() {
        super(...arguments);
        this.state = {
            normalizedOptions: createNormalizedOptions(this.props.options, this.props.sections, this.props.title),
        };
        this.id = this.props.id || getUniqueId();
        this.handleClick = (sectionIndex, optionIndex) => {
            const { selected, onChange, allowMultiple } = this.props;
            const selectedValue = this.state.normalizedOptions[sectionIndex].options[optionIndex].value;
            const foundIndex = selected.indexOf(selectedValue);
            if (allowMultiple) {
                const newSelection = foundIndex === -1
                    ? [selectedValue, ...selected]
                    : [
                        ...selected.slice(0, foundIndex),
                        ...selected.slice(foundIndex + 1, selected.length),
                    ];
                onChange(newSelection);
                return;
            }
            onChange([selectedValue]);
        };
    }
    componentWillReceiveProps({ options: nextOptions = [], sections: nextSections = [], id: nextID, title: nextTitle, }) {
        const { options = [], sections = [], id, title } = this.props;
        if (id !== nextID) {
            this.id = nextID || this.id;
        }
        const optionsChanged = !arraysAreEqual(nextOptions, options);
        const sectionsChanged = !arraysAreEqual(nextSections, sections, testSectionsPropEquality);
        const titleChanged = title !== nextTitle;
        if (optionsChanged || sectionsChanged || titleChanged) {
            this.setState({
                normalizedOptions: createNormalizedOptions(nextOptions, nextSections, nextTitle),
            });
        }
    }
    render() {
        const { normalizedOptions } = this.state;
        const { selected, allowMultiple, role, optionRole } = this.props;
        const optionsExist = normalizedOptions.length > 0;
        const optionsMarkup = optionsExist
            ? normalizedOptions.map(({ title, options }, sectionIndex) => {
                const titleMarkup = title ? (<p className={styles.Title} role={role}>
              {title}
            </p>) : null;
                const optionsMarkup = options &&
                    options.map((option, optionIndex) => {
                        const isSelected = selected.includes(option.value);
                        const id = option.id || `${this.id}-${sectionIndex}-${optionIndex}`;
                        return (<Option {...option} key={id} id={id} section={sectionIndex} index={optionIndex} onClick={this.handleClick} select={isSelected} allowMultiple={allowMultiple} role={optionRole}/>);
                    });
                return (<li key={title || `noTitle-${sectionIndex}`}>
              {titleMarkup}
              <ul className={styles.Options} id={this.id} role={role} aria-multiselectable={allowMultiple}>
                {optionsMarkup}
              </ul>
            </li>);
            })
            : null;
        return (<ul className={styles.OptionList} role={role}>
        {optionsMarkup}
      </ul>);
    }
}
function createNormalizedOptions(options, sections, title) {
    if (options == null) {
        const section = { options: [], title };
        return sections == null ? [] : [section, ...sections];
    }
    if (sections == null) {
        return [
            {
                title,
                options,
            },
        ];
    }
    return [
        {
            title,
            options,
        },
        ...sections,
    ];
}
function testSectionsPropEquality(previousSection, currentSection) {
    const { options: previousOptions } = previousSection;
    const { options: currentOptions } = currentSection;
    const optionsAreEqual = arraysAreEqual(previousOptions, currentOptions);
    const titlesAreEqual = previousSection.title === currentSection.title;
    return optionsAreEqual && titlesAreEqual;
}
export default withAppProvider()(OptionList);
