import * as React from 'react';
import { nodeContainsDescendant } from '@shopify/javascript-utilities/dom';
import { write } from '@shopify/javascript-utilities/fastdom';
import { classNames } from '@shopify/react-utilities/styles';
import { isElementOfType, wrapWithComponent } from '@shopify/react-utilities';
import { durationBase } from '@shopify/polaris-tokens';
import { Transition } from 'react-transition-group';
import { Key } from '../../../../types';
import { overlay } from '../../../shared';
import EventListener from '../../../EventListener';
import KeypressListener from '../../../KeypressListener';
import PositionedOverlay from '../../../PositionedOverlay';
import Pane from '../Pane';
import styles from '../../Popover.scss';
export var CloseSource;
(function (CloseSource) {
    CloseSource[CloseSource["Click"] = 0] = "Click";
    CloseSource[CloseSource["EscapeKeypress"] = 1] = "EscapeKeypress";
    CloseSource[CloseSource["FocusOut"] = 2] = "FocusOut";
    CloseSource[CloseSource["ScrollOut"] = 3] = "ScrollOut";
})(CloseSource || (CloseSource = {}));
export default class PopoverOverlay extends React.PureComponent {
    constructor() {
        super(...arguments);
        this.contentNode = React.createRef();
        this.transitionStatus = null;
        this.renderOverlay = (transitionStatus) => {
            const { active, activator, fullWidth, preferredPosition = 'below', preferredAlignment = 'center', fixed, } = this.props;
            return (<PositionedOverlay testID="positionedOverlay" fullWidth={fullWidth} active={active} activator={activator} preferredPosition={preferredPosition} preferredAlignment={preferredAlignment} 
            // eslint-disable-next-line react/jsx-no-bind
            render={this.renderPopover.bind(this, transitionStatus)} fixed={fixed} onScrollOut={this.handleScrollOut}/>);
        };
        this.renderPopover = (transitionStatus, overlayDetails) => {
            const { measuring, desiredHeight, positioning } = overlayDetails;
            const { id, children, sectioned, fullWidth, fullHeight } = this.props;
            const className = classNames(styles.Popover, transitionStatus && animationVariations(transitionStatus), positioning === 'above' && styles.positionedAbove, fullWidth && styles.fullWidth, measuring && styles.measuring);
            this.transitionStatus = transitionStatus;
            const contentStyles = measuring ? undefined : { height: desiredHeight };
            const contentClassNames = classNames(styles.Content, fullHeight && styles['Content-fullHeight']);
            const content = (<div id={id} tabIndex={-1} className={contentClassNames} style={contentStyles} ref={this.contentNode}>
        {renderPopoverContent(children, { sectioned })}
      </div>);
            return (<div className={className} {...overlay.props}>
        <EventListener event="click" handler={this.handleClick}/>
        <EventListener event="touchstart" handler={this.handleClick}/>
        <KeypressListener keyCode={Key.Escape} handler={this.handleEscape}/>
        <div className={styles.FocusTracker} 
            // eslint-disable-next-line jsx-a11y/no-noninteractive-tabindex
            tabIndex={0} onFocus={this.handleFocusFirstItem}/>
        <div className={styles.Wrapper}>{content}</div>
        <div className={styles.FocusTracker} 
            // eslint-disable-next-line jsx-a11y/no-noninteractive-tabindex
            tabIndex={0} onFocus={this.handleFocusLastItem}/>
      </div>);
        };
        this.handleClick = (event) => {
            const target = event.target;
            const { contentNode, props: { activator, onClose }, } = this;
            const isDescendant = contentNode.current != null &&
                nodeContainsDescendant(contentNode.current, target);
            const isActivatorDescendant = nodeContainsDescendant(activator, target);
            if (isDescendant ||
                isActivatorDescendant ||
                this.transitionStatus !== 'entered') {
                return;
            }
            onClose(CloseSource.Click);
        };
        this.handleScrollOut = () => {
            this.props.onClose(CloseSource.ScrollOut);
        };
        this.handleEscape = () => {
            this.props.onClose(CloseSource.EscapeKeypress);
        };
        this.handleFocusFirstItem = () => {
            this.props.onClose(CloseSource.FocusOut);
        };
        this.handleFocusLastItem = () => {
            this.props.onClose(CloseSource.FocusOut);
        };
    }
    componentDidMount() {
        if (this.props.active) {
            this.focusContent();
        }
    }
    componentDidUpdate(oldProps) {
        if (this.props.active && !oldProps.active) {
            this.focusContent();
        }
    }
    render() {
        const { active } = this.props;
        return (<Transition in={active} timeout={durationBase} mountOnEnter unmountOnExit>
        {this.renderOverlay}
      </Transition>);
    }
    focusContent() {
        if (this.props.preventAutofocus) {
            return;
        }
        if (this.contentNode == null) {
            return;
        }
        write(() => {
            if (this.contentNode.current == null) {
                return;
            }
            this.contentNode.current.focus();
        });
    }
}
function renderPopoverContent(children, props) {
    const childrenArray = React.Children.toArray(children);
    if (isElementOfType(childrenArray[0], Pane)) {
        return childrenArray;
    }
    return wrapWithComponent(childrenArray, Pane, props);
}
function animationVariations(status) {
    switch (status) {
        case 'exiting':
            return styles.exiting;
        default:
            return null;
    }
}
