import * as React from 'react';
import { createUniqueIDFactory } from '@shopify/javascript-utilities/other';
import { classNames } from '@shopify/react-utilities/styles';
import { withAppProvider } from '../AppProvider';
import Labelled, { helpTextID } from '../Labelled';
import styles from './RangeSlider.scss';
const getUniqueID = createUniqueIDFactory('RangeSlider');
const cssVarPrefix = '--Polaris-RangeSlider-';
export class RangeSlider extends React.PureComponent {
    constructor(props) {
        super(props);
        this.handleChange = (event) => {
            const { onChange } = this.props;
            if (onChange == null) {
                return;
            }
            onChange(parseFloat(event.currentTarget.value), this.state.id);
        };
        this.state = {
            id: props.id || getUniqueID(),
        };
    }
    static getDerivedStateFromProps(nextProps, prevState) {
        return nextProps.id != null && nextProps.id !== prevState.id
            ? {
                id: nextProps.id || prevState.id,
            }
            : null;
    }
    render() {
        const { id } = this.state;
        const { min = 0, max = 100 } = this.props;
        const { label, labelAction, labelHidden, step, value, output, helpText, error, disabled, prefix, suffix, onFocus, onBlur, } = this.props;
        const describedBy = [];
        if (error) {
            describedBy.push(`${id}Error`);
        }
        if (helpText) {
            describedBy.push(helpTextID(id));
        }
        const ariaDescribedBy = describedBy.length
            ? describedBy.join(' ')
            : undefined;
        const sliderProgress = ((value - min) * 100) / (max - min);
        const cssVars = {
            [`${cssVarPrefix}min`]: min,
            [`${cssVarPrefix}max`]: max,
            [`${cssVarPrefix}current`]: value,
            [`${cssVarPrefix}progress`]: `${sliderProgress}%`,
            [`${cssVarPrefix}output-factor`]: invertNumber((sliderProgress - 50) / 100),
        };
        const outputMarkup = !disabled &&
            output && (<output htmlFor={id} className={styles.Output}>
          <div className={styles.OutputBubble}>
            <span className={styles.OutputText}>{value}</span>
          </div>
        </output>);
        const prefixMarkup = prefix && (<div className={styles.Prefix}>{prefix}</div>);
        const suffixMarkup = suffix && (<div className={styles.Suffix}>{suffix}</div>);
        const className = classNames(styles.RangeSlider, error && styles.error, disabled && styles.disabled);
        return (<Labelled id={id} label={label} error={error} action={labelAction} labelHidden={labelHidden} helpText={helpText}>
        <div className={className} style={cssVars}>
          {prefixMarkup}
          <div className={styles.InputWrapper}>
            <input type="range" className={styles.Input} id={id} name={id} min={min} max={max} step={step} value={value} disabled={disabled} onChange={this.handleChange} onFocus={onFocus} onBlur={onBlur} aria-valuemin={min} aria-valuemax={max} aria-valuenow={value} aria-invalid={Boolean(error)} aria-describedby={ariaDescribedBy}/>
            {outputMarkup}
          </div>
          {suffixMarkup}
        </div>
      </Labelled>);
    }
}
export function invertNumber(number) {
    if (Math.sign(number) === 1) {
        return -Math.abs(number);
    }
    else if (Math.sign(number) === -1) {
        return Math.abs(number);
    }
    else {
        return 0;
    }
}
export default withAppProvider()(RangeSlider);
