import * as React from 'react';
import { HorizontalDotsMinor } from '@shopify/polaris-icons';
import { classNames } from '@shopify/react-utilities/styles';
import { createUniqueIDFactory, noop } from '@shopify/javascript-utilities/other';
import compose from '@shopify/react-compose';
import ActionList from '../../../ActionList';
import Popover from '../../../Popover';
import UnstyledLink from '../../../UnstyledLink';
import ButtonGroup from '../../../ButtonGroup';
import Checkbox from '../../../Checkbox';
import Button, { buttonsFrom } from '../../../Button';
import { withAppProvider } from '../../../AppProvider';
import { SELECT_ALL_ITEMS } from '../../types';
import withContext from '../../../WithContext';
import { Consumer } from '../Context';
import styles from './Item.scss';
const getUniqueCheckboxID = createUniqueIDFactory('ResourceListItemCheckbox');
export class Item extends React.PureComponent {
    constructor() {
        super(...arguments);
        this.state = {
            actionsMenuVisible: false,
            focused: false,
            focusedInner: false,
        };
        this.node = null;
        this.checkboxId = getUniqueCheckboxID();
        this.setNode = (node) => {
            this.node = node;
        };
        this.handleAnchorFocus = () => {
            this.setState({ focused: true, focusedInner: false });
        };
        this.handleFocusedBlur = () => {
            this.setState({ focused: true, focusedInner: true });
        };
        this.handleFocus = () => {
            this.setState({ focused: true });
        };
        this.handleBlur = (event) => {
            const isInside = this.compareEventNode(event);
            if (this.node == null ||
                !this.node.contains(event.relatedTarget)) {
                this.setState({ focused: false });
            }
            else if (isInside) {
                this.setState({ focusedInner: true });
            }
        };
        this.handleMouseDown = () => {
            this.setState({ focusedInner: true });
        };
        this.handleLargerSelectionArea = (event) => {
            stopPropagation(event);
            this.handleSelection(!this.isSelected());
        };
        this.handleSelection = (value) => {
            const { id, context: { onSelectionChange }, } = this.props;
            if (id == null || onSelectionChange == null) {
                return;
            }
            this.setState({ focused: true, focusedInner: true });
            onSelectionChange(value, id);
        };
        this.handleClick = (event) => {
            const { id, onClick, url, context: { selectMode }, } = this.props;
            const { ctrlKey, metaKey } = event.nativeEvent;
            const anchor = this.node && this.node.querySelector('a');
            if (selectMode) {
                this.handleLargerSelectionArea(event);
                return;
            }
            if (anchor === event.target) {
                return;
            }
            if (onClick) {
                onClick(id);
            }
            if (url && (ctrlKey || metaKey)) {
                window.open(url, '_blank');
                return;
            }
            if (url && anchor) {
                anchor.click();
            }
        };
        this.handleKeypress = (event) => {
            const { onClick = noop, context: { selectMode }, } = this.props;
            const { key } = event;
            if (key === 'Enter' && !selectMode) {
                onClick();
            }
        };
        this.handleActionsClick = () => {
            this.setState(({ actionsMenuVisible }) => ({
                actionsMenuVisible: !actionsMenuVisible,
            }));
        };
        this.handleCloseRequest = () => {
            this.setState({ actionsMenuVisible: false });
        };
    }
    render() {
        const { children, url, media, shortcutActions, ariaControls, ariaExpanded, persistActions = false, polaris: { intl }, accessibilityLabel, context: { selectable, selectMode, loading }, } = this.props;
        const { actionsMenuVisible, focused, focusedInner } = this.state;
        const selected = this.isSelected();
        let ownedMarkup = null;
        let handleMarkup = null;
        const mediaMarkup = media ? (<div className={styles.Media} testID="Media">
        {media}
      </div>) : null;
        const checkboxAccessibilityLabel = accessibilityLabel || intl.translate('Polaris.Common.checkbox');
        if (selectable) {
            const label = selected
                ? intl.translate('Polaris.ResourceList.Item.deselectItem', {
                    accessibilityLabel: checkboxAccessibilityLabel,
                })
                : intl.translate('Polaris.ResourceList.Item.selectItem', {
                    accessibilityLabel: checkboxAccessibilityLabel,
                });
            handleMarkup = (<div className={styles.Handle} onClick={this.handleLargerSelectionArea} testID="LargerSelectionArea">
          <div onClick={stopPropagation} className={styles.CheckboxWrapper}>
            <Checkbox testID="Checkbox" id={this.checkboxId} label={label} labelHidden onChange={this.handleSelection} checked={selected} disabled={loading}/>
          </div>
        </div>);
        }
        if (media || selectable) {
            ownedMarkup = (<div className={styles.Owned}>
          {handleMarkup}
          {mediaMarkup}
        </div>);
        }
        const className = classNames(styles.Item, focused && styles.focused, selectable && styles.selectable, selected && styles.selected, selectMode && styles.selectMode, persistActions && styles.persistActions, focusedInner && styles.focusedInner);
        let actionsMarkup = null;
        let disclosureMarkup = null;
        if (shortcutActions && !loading) {
            if (persistActions) {
                actionsMarkup = (<div className={styles.Actions} onClick={stopPropagation}>
            <ButtonGroup>
              {buttonsFrom(shortcutActions, {
                    size: 'slim',
                    plain: true,
                })}
            </ButtonGroup>
          </div>);
                disclosureMarkup = (<div className={styles.Disclosure} onClick={stopPropagation}>
            <Popover activator={<Button aria-label={intl.translate('Polaris.ResourceList.Item.actionsDropdown')} onClick={this.handleActionsClick} plain icon={HorizontalDotsMinor}/>} onClose={this.handleCloseRequest} active={actionsMenuVisible}>
              <ActionList items={shortcutActions}/>
            </Popover>
          </div>);
            }
            else {
                actionsMarkup = (<div className={styles.Actions} onClick={stopPropagation}>
            <ButtonGroup segmented testID="ShortcutActions">
              {buttonsFrom(shortcutActions, {
                    size: 'slim',
                })}
            </ButtonGroup>
          </div>);
            }
        }
        const content = children ? (<div className={styles.Content}>{children}</div>) : null;
        const containerMarkup = (<div testID="Item-Content" className={styles.Container} id={this.props.id}>
        {ownedMarkup}
        {content}
        {actionsMarkup}
        {disclosureMarkup}
      </div>);
        const tabIndex = loading ? -1 : 0;
        const accessibleMarkup = url ? (<UnstyledLink aria-describedby={this.props.id} aria-label={accessibilityLabel} className={styles.Link} url={url} onFocus={this.handleAnchorFocus} onBlur={this.handleFocusedBlur} tabIndex={tabIndex}/>) : (<button className={styles.Button} aria-label={accessibilityLabel} aria-controls={ariaControls} aria-expanded={ariaExpanded} onClick={this.handleClick} onFocus={this.handleAnchorFocus} onBlur={this.handleFocusedBlur} tabIndex={tabIndex}/>);
        return (<div ref={this.setNode} className={className} onClick={this.handleClick} onFocus={this.handleFocus} onBlur={this.handleBlur} onMouseDown={this.handleMouseDown} onKeyUp={this.handleKeypress} testID="Item-Wrapper" data-href={url}>
        {accessibleMarkup}
        {containerMarkup}
      </div>);
    }
    isSelected() {
        const { id, context: { selectedItems }, } = this.props;
        return (selectedItems &&
            ((Array.isArray(selectedItems) && selectedItems.includes(id)) ||
                selectedItems === SELECT_ALL_ITEMS));
    }
    compareEventNode(event) {
        return this.props.onClick
            ? event.target === this.node
            : event.target.tagName.toLowerCase() === 'a';
    }
}
function stopPropagation(event) {
    event.stopPropagation();
}
export default compose(withContext(Consumer), withAppProvider())(Item);
