import * as React from 'react';
import EventListener from '../../../EventListener';
import styles from '../../TextField.scss';
export default class Resizer extends React.PureComponent {
    constructor() {
        super(...arguments);
        this.contentNode = React.createRef();
        this.minimumLinesNode = React.createRef();
        this.animationFrame = null;
        this.handleHeightCheck = () => {
            if (this.animationFrame) {
                cancelAnimationFrame(this.animationFrame);
            }
            this.animationFrame = requestAnimationFrame(() => {
                const contentNode = this.contentNode.current;
                const minimumLinesNode = this.minimumLinesNode.current;
                if (!contentNode || !minimumLinesNode) {
                    return;
                }
                const newHeight = Math.max(contentNode.offsetHeight, minimumLinesNode.offsetHeight);
                const { currentHeight, onHeightChange } = this.props;
                if (newHeight !== currentHeight) {
                    onHeightChange(newHeight);
                }
            });
        };
    }
    componentDidMount() {
        this.handleHeightCheck();
    }
    componentDidUpdate() {
        this.handleHeightCheck();
    }
    componentWillUnmount() {
        if (this.animationFrame) {
            cancelAnimationFrame(this.animationFrame);
        }
    }
    render() {
        const { contents, minimumLines } = this.props;
        const minimumLinesMarkup = minimumLines ? (<div testID="MinimumLines" ref={this.minimumLinesNode} className={styles.DummyInput} dangerouslySetInnerHTML={{
            __html: getContentsForMinimumLines(minimumLines),
        }}/>) : null;
        return (<div testID="ResizerWrapper" aria-hidden className={styles.Resizer}>
        <EventListener event="resize" handler={this.handleHeightCheck}/>
        <div testID="ContentsNode" ref={this.contentNode} className={styles.DummyInput} dangerouslySetInnerHTML={{ __html: getFinalContents(contents) }}/>
        {minimumLinesMarkup}
      </div>);
    }
}
const ENTITIES_TO_REPLACE = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '\n': '<br>',
    '\r': '',
};
const REPLACE_REGEX = new RegExp(`[${Object.keys(ENTITIES_TO_REPLACE).join()}]`, 'g');
function replaceEntity(entity) {
    return ENTITIES_TO_REPLACE[entity];
}
function getContentsForMinimumLines(minimumLines) {
    let content = '';
    for (let line = 0; line < minimumLines; line++) {
        content += '<br>';
    }
    return content;
}
function getFinalContents(contents) {
    return contents
        ? `${contents.replace(REPLACE_REGEX, replaceEntity)}<br>`
        : '<br>';
}
