import tokens from '@shopify/polaris-tokens';
import { noop } from '@shopify/javascript-utilities/other';
import { needsVariantList } from '../config';
import { colorToHsla, hslToString, hslToRgb, } from '../../../utilities/color-transformers';
import { isLight } from '../../../utilities/color-validation';
import { constructColorName } from '../../../utilities/color-names';
import { createLightColor } from '../../../utilities/color-manipulation';
import { compose } from '../../../utilities/compose';
export function setColors(theme) {
    let colorPairs;
    if (theme && theme.colors) {
        Object.entries(theme.colors).forEach(([colorKey, pairs]) => {
            const colorKeys = Object.keys(pairs);
            if (colorKey === 'topBar' && colorKeys.length > 1) {
                colorPairs = colorKeys.map((key) => {
                    const colors = theme.colors.topBar;
                    return [constructColorName(colorKey, key), colors[key]];
                });
            }
            else {
                colorPairs = parseColors([colorKey, pairs]);
            }
        });
    }
    return colorPairs;
}
export function needsVariant(name) {
    return needsVariantList.indexOf(name) !== -1;
}
const lightenToString = compose(hslToString, createLightColor);
export function setTextColor(name, variant = 'dark') {
    if (variant === 'light') {
        return [name, tokens.colorInk];
    }
    return [name, tokens.colorWhite];
}
export function setTheme(color, baseName, key, variant) {
    const colorPairs = [];
    switch (variant) {
        case 'light':
            colorPairs.push(setTextColor(constructColorName(baseName, null, 'color'), 'light'));
            colorPairs.push([
                constructColorName(baseName, key, 'lighter'),
                lightenToString(color, 7, -10),
            ]);
            break;
        case 'dark':
            colorPairs.push(setTextColor(constructColorName(baseName, null, 'color'), 'dark'));
            colorPairs.push([
                constructColorName(baseName, key, 'lighter'),
                lightenToString(color, 15, 15),
            ]);
            break;
        default:
    }
    return colorPairs;
}
function parseColors([baseName, colors]) {
    const keys = Object.keys(colors);
    const colorPairs = [];
    for (let i = 0; i < keys.length; i++) {
        colorPairs.push([constructColorName(baseName, keys[i]), colors[keys[i]]]);
        if (needsVariant(baseName)) {
            const hslColor = colorToHsla(colors[keys[i]]);
            if (typeof hslColor === 'string') {
                return colorPairs;
            }
            const rgbColor = hslToRgb(hslColor);
            if (isLight(rgbColor)) {
                colorPairs.push(...setTheme(hslColor, baseName, keys[i], 'light'));
            }
            else {
                colorPairs.push(...setTheme(hslColor, baseName, keys[i], 'dark'));
            }
        }
    }
    return colorPairs;
}
export function createThemeContext(theme) {
    if (!theme) {
        return { polarisTheme: { logo: null, subscribe: noop, unsubscribe: noop } };
    }
    const { logo = null, subscribe = noop, unsubscribe = noop } = theme;
    return { polarisTheme: { logo, subscribe, unsubscribe } };
}
