import React from 'react';
import { createUniqueIDFactory } from '@shopify/javascript-utilities/other';
import { OptionList } from '../../../OptionList';
import { ActionList } from '../../../ActionList';
import { Popover } from '../../../Popover';
import { Key } from '../../../../types';
import { KeypressListener } from '../../../KeypressListener';
import { EventListener } from '../../../EventListener';
import { ComboBoxContext } from './context';
import styles from './ComboBox.scss';
const getUniqueId = createUniqueIDFactory('ComboBox');
export class ComboBox extends React.PureComponent {
    constructor() {
        super(...arguments);
        this.state = {
            comboBoxId: this.getComboBoxId(),
            selectedOption: undefined,
            selectedIndex: -1,
            selectedOptions: this.props.selected,
            navigableOptions: [],
            popoverActive: false,
            popoverWasActive: false,
        };
        this.handleDownArrow = () => {
            this.selectNextOption();
            this.handlePopoverOpen;
        };
        this.handleUpArrow = () => {
            this.selectPreviousOption();
            this.handlePopoverOpen;
        };
        this.handleEnter = (event) => {
            if (event.keyCode !== Key.Enter) {
                return;
            }
            const { selectedOption } = this.state;
            if (this.state.popoverActive && selectedOption) {
                if (isOption(selectedOption)) {
                    event.preventDefault();
                    this.handleSelection(selectedOption.value);
                }
                else {
                    selectedOption.onAction && selectedOption.onAction();
                }
            }
            this.handlePopoverOpen;
        };
        this.handleFocus = () => {
            this.setState({ popoverActive: true, popoverWasActive: true });
        };
        this.handleBlur = () => {
            this.setState({ popoverActive: false, popoverWasActive: false }, () => {
                this.resetVisuallySelectedOptions();
            });
        };
        this.handleClick = () => {
            !this.state.popoverActive && this.setState({ popoverActive: true });
        };
        this.handleSelection = (newSelected) => {
            const { selected, allowMultiple } = this.props;
            let newlySelectedOptions = selected;
            if (selected.includes(newSelected)) {
                newlySelectedOptions.splice(newlySelectedOptions.indexOf(newSelected), 1);
            }
            else if (allowMultiple) {
                newlySelectedOptions.push(newSelected);
            }
            else {
                newlySelectedOptions = [newSelected];
            }
            this.selectOptions(newlySelectedOptions);
        };
        this.selectOptions = (selected) => {
            const { onSelect, allowMultiple } = this.props;
            selected && onSelect(selected);
            if (!allowMultiple) {
                this.resetVisuallySelectedOptions();
                this.setState({ popoverActive: false, popoverWasActive: false });
            }
        };
        this.updateIndexOfSelectedOption = (newOptions) => {
            const { selectedIndex, selectedOption } = this.state;
            if (selectedOption && newOptions.includes(selectedOption)) {
                this.selectOptionAtIndex(newOptions.indexOf(selectedOption));
            }
            else if (selectedIndex > newOptions.length - 1) {
                this.resetVisuallySelectedOptions();
            }
            else {
                this.selectOptionAtIndex(selectedIndex);
            }
        };
        this.resetVisuallySelectedOptions = () => {
            const { navigableOptions } = this.state;
            this.setState({
                selectedOption: undefined,
                selectedIndex: -1,
            });
            navigableOptions &&
                navigableOptions.forEach((option) => {
                    option.active = false;
                });
        };
        this.handlePopoverClose = () => {
            this.setState({ popoverActive: false, popoverWasActive: false });
        };
        this.handlePopoverOpen = () => {
            const { popoverActive, navigableOptions } = this.state;
            !popoverActive &&
                navigableOptions &&
                navigableOptions.length > 0 &&
                this.setState({ popoverActive: true, popoverWasActive: true });
        };
        this.selectNextOption = () => {
            const { selectedIndex, navigableOptions } = this.state;
            if (!navigableOptions || navigableOptions.length === 0) {
                return;
            }
            let newIndex = selectedIndex;
            if (selectedIndex + 1 >= navigableOptions.length) {
                newIndex = 0;
            }
            else {
                newIndex++;
            }
            this.selectOptionAtIndex(newIndex);
        };
        this.selectPreviousOption = () => {
            const { selectedIndex, navigableOptions } = this.state;
            if (!navigableOptions || navigableOptions.length === 0) {
                return;
            }
            let newIndex = selectedIndex;
            if (selectedIndex <= 0) {
                newIndex = navigableOptions.length - 1;
            }
            else {
                newIndex--;
            }
            this.selectOptionAtIndex(newIndex);
        };
        this.selectOptionAtIndex = (newOptionIndex) => {
            this.setState((prevState) => {
                if (!prevState.navigableOptions ||
                    prevState.navigableOptions.length === 0) {
                    return prevState;
                }
                const newSelectedOption = prevState.navigableOptions[newOptionIndex];
                this.visuallyUpdateSelectedOption(newSelectedOption, prevState.selectedOption);
                return Object.assign(Object.assign({}, prevState), { selectedOption: newSelectedOption, selectedIndex: newOptionIndex });
            });
        };
        this.visuallyUpdateSelectedOption = (newOption, oldOption) => {
            if (oldOption) {
                oldOption.active = false;
            }
            if (newOption) {
                newOption.active = true;
            }
        };
    }
    static getDerivedStateFromProps({ options: nextOptions, selected: nextSelected, actionsBefore: nextActionsBefore, actionsAfter: nextActionsAfter, }, { navigableOptions, selectedOptions, comboBoxId }) {
        const optionsChanged = filterForOptions(navigableOptions) &&
            nextOptions &&
            !optionsAreEqual(navigableOptions, nextOptions);
        let newNavigableOptions = [];
        if (nextActionsBefore) {
            newNavigableOptions = newNavigableOptions.concat(nextActionsBefore);
        }
        if (optionsChanged || nextActionsBefore) {
            newNavigableOptions = newNavigableOptions.concat(nextOptions);
        }
        if (nextActionsAfter) {
            newNavigableOptions = newNavigableOptions.concat(nextActionsAfter);
        }
        newNavigableOptions = assignOptionIds(newNavigableOptions, comboBoxId);
        if (optionsChanged && selectedOptions !== nextSelected) {
            return {
                navigableOptions: newNavigableOptions,
                selectedOptions: nextSelected,
            };
        }
        else if (optionsChanged) {
            return {
                navigableOptions: newNavigableOptions,
            };
        }
        else if (selectedOptions !== nextSelected) {
            return { selectedOptions: nextSelected };
        }
        return null;
    }
    componentDidMount() {
        const { options, actionsBefore, actionsAfter } = this.props;
        const comboBoxId = this.getComboBoxId();
        let navigableOptions = [];
        if (actionsBefore) {
            navigableOptions = navigableOptions.concat(actionsBefore);
        }
        if (options) {
            navigableOptions = navigableOptions.concat(options);
        }
        if (actionsAfter) {
            navigableOptions = navigableOptions.concat(actionsAfter);
        }
        navigableOptions = assignOptionIds(navigableOptions, comboBoxId);
        this.setState({
            navigableOptions,
        });
    }
    componentDidUpdate(_, prevState) {
        const { contentBefore, contentAfter, emptyState } = this.props;
        const { navigableOptions, popoverWasActive } = this.state;
        const optionsChanged = navigableOptions &&
            prevState.navigableOptions &&
            !optionsAreEqual(navigableOptions, prevState.navigableOptions);
        if (optionsChanged) {
            this.updateIndexOfSelectedOption(navigableOptions);
        }
        if (navigableOptions &&
            navigableOptions.length === 0 &&
            !contentBefore &&
            !contentAfter &&
            !emptyState) {
            // eslint-disable-next-line react/no-did-update-set-state
            this.setState({ popoverActive: false });
        }
        else if (popoverWasActive &&
            navigableOptions &&
            navigableOptions.length !== 0) {
            // eslint-disable-next-line react/no-did-update-set-state
            this.setState({ popoverActive: true });
        }
    }
    getComboBoxId() {
        if (this.state && this.state.comboBoxId) {
            return this.state.comboBoxId;
        }
        return this.props.id || getUniqueId();
    }
    render() {
        const { options, textField, listTitle, allowMultiple, preferredPosition, actionsBefore, actionsAfter, contentBefore, contentAfter, onEndReached, emptyState, } = this.props;
        const { comboBoxId, navigableOptions, selectedOptions } = this.state;
        const actionsBeforeMarkup = actionsBefore && actionsBefore.length > 0 && (<ActionList actionRole="option" items={actionsBefore}/>);
        const actionsAfterMarkup = actionsAfter && actionsAfter.length > 0 && (<ActionList actionRole="option" items={actionsAfter}/>);
        const optionsMarkup = options.length > 0 && (<OptionList role="presentation" optionRole="option" options={filterForOptions(navigableOptions)} onChange={this.selectOptions} selected={selectedOptions} title={listTitle} allowMultiple={allowMultiple}/>);
        const emptyStateMarkup = !actionsAfter &&
            !actionsBefore &&
            !contentAfter &&
            !contentBefore &&
            options.length === 0 &&
            emptyState && <div className={styles.EmptyState}>{emptyState}</div>;
        const context = {
            comboBoxId,
            selectedOptionId: this.selectedOptionId(),
        };
        return (<ComboBoxContext.Provider value={context}>
        <div onClick={this.handleClick} role="combobox" aria-expanded={this.state.popoverActive} aria-owns={this.state.comboBoxId} aria-controls={this.state.comboBoxId} aria-haspopup onFocus={this.handleFocus} onBlur={this.handleBlur} tabIndex={0}>
          <KeypressListener keyCode={Key.DownArrow} handler={this.handleDownArrow}/>
          <KeypressListener keyCode={Key.UpArrow} handler={this.handleUpArrow}/>
          <EventListener event="keydown" handler={this.handleEnter}/>
          <KeypressListener keyCode={Key.Escape} handler={this.handlePopoverClose}/>
          <Popover activator={textField} active={this.state.popoverActive} onClose={this.handlePopoverClose} preferredPosition={preferredPosition} fullWidth preventAutofocus>
            <Popover.Pane onScrolledToBottom={onEndReached}>
              <div id={this.state.comboBoxId} role="listbox" aria-multiselectable={allowMultiple}>
                {contentBefore}
                {actionsBeforeMarkup}
                {optionsMarkup}
                {actionsAfterMarkup}
                {contentAfter}
                {emptyStateMarkup}
              </div>
            </Popover.Pane>
          </Popover>
        </div>
      </ComboBoxContext.Provider>);
    }
    selectedOptionId() {
        const { selectedOption, selectedIndex, comboBoxId } = this.state;
        return selectedOption ? `${comboBoxId}-${selectedIndex}` : undefined;
    }
}
function assignOptionIds(options, comboBoxId) {
    options.map((option, optionIndex) => {
        option.id = `${comboBoxId}-${optionIndex}`;
    });
    return options;
}
function optionsAreEqual(firstOptions, secondOptions) {
    if (firstOptions.length !== secondOptions.length) {
        return false;
    }
    return firstOptions.every((firstItem, index) => {
        const secondItem = secondOptions[index];
        if (isOption(firstItem)) {
            if (isOption(secondItem)) {
                return firstItem.value === secondItem.value;
            }
            return false;
        }
        else {
            if (!isOption(secondItem)) {
                return firstItem.content === secondItem.content;
            }
            return false;
        }
    });
}
function isOption(navigableOption) {
    return navigableOption.value !== undefined;
}
function filterForOptions(mixedArray) {
    return mixedArray.filter((item) => isOption(item));
}
