function toCamelCase(value) {
  value = value.replace(/[\(\)\[\]\{\}\=\?\!\.\:,\-_\+\\\"#~\/]/g, ' ');
  let returnValue = '';
  let makeNextUppercase = true;
  value = value.toLowerCase();
  for (let i = 0; value.length > i; i++) {
    let c = value.charAt(i);
    if (c.match(/^\s+$/g) || c.match(/[\(\)\[\]\{\}\\\/]/g)) {
      makeNextUppercase = true;
    }
    else if (makeNextUppercase) {
      c = c.toUpperCase();
      makeNextUppercase = false;
    }
    returnValue += c;
  }
  const normalized = returnValue.replace(/\s+/g, '');
  return normalized.charAt(0).toUpperCase() + normalized.slice(1);
}
export const isSvgDataUrl = (url) => {
  if (!url) {
    return false;
  }
  if (typeof url !== 'string') {
    return false;
  }
  return url.startsWith('data:image/svg+xml');
};
let parser = null;
export function parseSVGDataContent(content) {
  if (typeof window['DOMParser'] === 'undefined') {
    console.warn('DOMParser not supported by your browser.');
    return;
  }
  if (parser === null) {
    parser = new window['DOMParser']();
  }
  const svgDocument = parser.parseFromString(content, 'text/html');
  const svgElement = svgDocument.querySelector('svg');
  if (!svgElement) {
    throw Error('No valid svg data provided');
  }
  return svgElement.outerHTML;
}
async function fetchSVG(url) {
  const response = await fetch(url);
  const responseText = await response.text();
  if (!response.ok) {
    console.error(responseText);
    throw Error(responseText);
  }
  return parseSVGDataContent(responseText);
}
const urlRegex = /^(?:(?:https?|ftp):\/\/)?(?:\S+(?::\S*)?@)?(?:www\.)?(?:\S+\.\S+)(?:\S*)$/i;
function isValidUrl(url) {
  return urlRegex.test(url);
}
async function getESMIcon(name) {
  const esmIcon = await import('./icons');
  let iconName = toCamelCase(name);
  iconName = `icon${iconName}`;
  return parseSVGDataContent(esmIcon[iconName]);
}
export async function resolveIcon(icon) {
  const { name } = icon;
  if (!name) {
    throw Error('no icon name provided');
  }
  if (isSvgDataUrl(name)) {
    return parseSVGDataContent(name);
  }
  if (isValidUrl(name)) {
    try {
      return await fetchSVG(name);
    }
    catch (error) {
      throw error;
    }
  }
  return getESMIcon(name);
}
//# sourceMappingURL=resolveIcon.js.map
