/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { h, Host, Build } from "@stencil/core";
import { parseSVGDataContent, resolveIcon } from "./resolveIcon";
const iconMissingSymbol = "data:image/svg+xml;utf8,<svg xmlns='http://www.w3.org/2000/svg' width='512' height='512' viewBox='0 0 512 512'><path fill-rule='evenodd' d='M384,0 L384,384 L0,384 L0,0 L384,0 Z M192,207.085 L57.751,341.333 L326.248,341.333 L192,207.085 Z M42.666,57.751 L42.666,326.248 L176.915,192 L42.666,57.751 Z M341.333,57.751 L207.085,192 L341.333,326.248 L341.333,57.751 Z M326.248,42.666 L57.751,42.666 L192,176.915 L326.248,42.666 Z' transform='translate(64 64)'/></svg>";
export class Icon {
    constructor() {
        this.size = undefined;
        this.color = undefined;
        this.name = undefined;
        this.lazyLoading = false;
        this.svgContent = undefined;
        this.isVisible = false;
    }
    connectedCallback() {
        this.waitForRendering(() => {
            this.isVisible = true;
            this.loadIconContent();
        });
    }
    async loadIconContent() {
        try {
            this.svgContent = await resolveIcon(this.name);
        }
        catch (error) {
            this.svgContent = parseSVGDataContent(iconMissingSymbol);
        }
    }
    waitForRendering(onRender) {
        if (Build.isBrowser && this.lazyLoading && typeof window !== 'undefined' && window.IntersectionObserver) {
            const observer = new IntersectionObserver(entries => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        onRender();
                        observer.disconnect();
                    }
                });
            }, {
                rootMargin: '25px',
            });
            observer.observe(this.hostElement);
        }
        else {
            onRender();
        }
    }
    render() {
        const style = {};
        if (this.color) {
            style['color'] = `var(--theme-${this.color})`;
        }
        return (h(Host, { key: '8dc18bf3b1ab1e2a521bd0fee33fa91b856dc52b', style: style, class: {
                ['size-12']: this.size === '12',
                ['size-16']: this.size === '16',
                ['size-24']: this.size === '24',
                ['size-32']: this.size === '32',
            } }, h("div", { key: '48c3e8ae8e0d36be5fa112dbd408756f56c19535', class: 'svg-container', innerHTML: this.svgContent })));
    }
    static get is() { return "ix-icon"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["icon.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["icon.css"]
        };
    }
    static get assetsDirs() { return ["svg"]; }
    static get properties() {
        return {
            "size": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "'12' | '16' | '24' | '32'",
                    "resolved": "\"12\" | \"16\" | \"24\" | \"32\"",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Size of the icon"
                },
                "attribute": "size",
                "reflect": false
            },
            "color": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Color of the icon"
                },
                "attribute": "color",
                "reflect": false
            },
            "name": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Use one of our defined icon names e.g. `copy`\n\nhttps://ix.siemens.io/docs/icon-library/icons\n\nor the import variant\n\n```\nimport { rocket } from '@siemens/ix-icons/icons';\n\n<ix-icon name={rocket}></ix-icon>\n```"
                },
                "attribute": "name",
                "reflect": false
            },
            "lazyLoading": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": ""
                },
                "attribute": "lazy-loading",
                "reflect": false,
                "defaultValue": "false"
            }
        };
    }
    static get states() {
        return {
            "svgContent": {},
            "isVisible": {}
        };
    }
    static get elementRef() { return "hostElement"; }
    static get watchers() {
        return [{
                "propName": "name",
                "methodName": "loadIconContent"
            }];
    }
}
//# sourceMappingURL=icon.js.map
