import { getAssetPath, proxyCustomElement, HTMLElement, Build, h, Host } from '@stencil/core/internal/client';

/*
 * COPYRIGHT (c) Siemens AG 2018-2024 ALL RIGHTS RESERVED.
 */
function getV3PreviewMetaElement() {
    return document.querySelector("meta[name='ix-icons:v3-preview']");
}
function getV3PreviewMetaContent() {
    const v3PreviewMetaElement = getV3PreviewMetaElement();
    if (v3PreviewMetaElement) {
        return v3PreviewMetaElement.getAttribute('content').split(';');
    }
    return null;
}
/**
 * Provide custom SVG path for icons
 *
 * <meta name="ix-icons:path" content="/build/svg" />
 */
function getCustomAssetUrl() {
    const assetPath = document.querySelector("meta[name='ix-icons:path']");
    if (assetPath) {
        const path = assetPath.getAttribute('content');
        return path;
    }
    return false;
}
/**
 * Enable v3 preview features
 *
 * <meta name="ix-icons:v3-preview" content="svg-path-loading;" />
 */
function isV3PreviewEnabled() {
    const features = getV3PreviewMetaContent();
    if (!features) {
        return false;
    }
    if (features.includes('svg-path-loading')) {
        return true;
    }
    return false;
}

/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
let fetchCache;
const requests = new Map();
let parser = null;
function toCamelCase(value) {
    value = value.replace(/[\(\)\[\]\{\}\=\?\!\.\:,\-_\+\\\"#~\/]/g, ' ');
    let returnValue = '';
    let makeNextUppercase = true;
    value = value.toLowerCase();
    for (let i = 0; value.length > i; i++) {
        let c = value.charAt(i);
        if (c.match(/^\s+$/g) || c.match(/[\(\)\[\]\{\}\\\/]/g)) {
            makeNextUppercase = true;
        }
        else if (makeNextUppercase) {
            c = c.toUpperCase();
            makeNextUppercase = false;
        }
        returnValue += c;
    }
    const normalized = returnValue.replace(/\s+/g, '');
    return normalized.charAt(0).toUpperCase() + normalized.slice(1);
}
const getIconCacheMap = () => {
    if (typeof window === 'undefined') {
        return new Map();
    }
    if (!fetchCache) {
        window.IxIcons = window.IxIcons || {};
        fetchCache = window.IxIcons.map = window.IxIcons.map || new Map();
    }
    return fetchCache;
};
const isSvgDataUrl = (url) => {
    if (!url) {
        return false;
    }
    if (typeof url !== 'string') {
        return false;
    }
    return url.startsWith('data:image/svg+xml');
};
function parseSVGDataContent(content) {
    if (typeof window['DOMParser'] === 'undefined') {
        console.warn('DOMParser not supported by your browser.');
        return;
    }
    if (parser === null) {
        parser = new window['DOMParser']();
    }
    const svgDocument = parser.parseFromString(content, 'text/html');
    const svgElement = svgDocument.querySelector('svg');
    if (!svgElement) {
        throw Error('No valid svg data provided');
    }
    return svgElement.outerHTML;
}
async function fetchSVG(url) {
    const cache = getIconCacheMap();
    if (cache.has(url)) {
        return cache.get(url);
    }
    if (requests.has(url)) {
        return requests.get(url);
    }
    const fetching = fetch(url).then(async (response) => {
        const responseText = await response.text();
        if (!response.ok) {
            console.error(responseText);
            throw Error(responseText);
        }
        const svgContent = parseSVGDataContent(responseText);
        cache.set(url, svgContent);
        return svgContent;
    });
    requests.set(url, fetching);
    return fetching;
}
const urlRegex = /^(?:(?:https?|ftp):\/\/)?(?:\S+(?::\S*)?@)?(?:www\.)?(?:\S+\.\S+)(?:\S*)$/i;
function isValidUrl(url) {
    return urlRegex.test(url);
}
function getAssetUrl(name) {
    const customAssetUrl = getCustomAssetUrl();
    if (customAssetUrl) {
        return `${customAssetUrl}/${name}.svg`;
    }
    return getAssetPath(`svg/${name}.svg`);
}
async function getESMIcon(name) {
    const esmIcon = await import('./icons.js');
    let iconName = toCamelCase(name);
    iconName = `icon${iconName}`;
    return parseSVGDataContent(esmIcon[iconName]);
}
async function resolveIcon(iconName) {
    if (!iconName) {
        throw Error('No icon name provided');
    }
    if (isSvgDataUrl(iconName)) {
        return parseSVGDataContent(iconName);
    }
    if (isValidUrl(iconName)) {
        try {
            return fetchSVG(iconName);
        }
        catch (error) {
            throw error;
        }
    }
    if (isV3PreviewEnabled()) {
        console.warn('Using V3 preview feature for loading icons.');
        try {
            return fetchSVG(getAssetUrl(iconName));
        }
        catch (error) {
            throw Error('Cannot resolve any icon');
        }
    }
    return getESMIcon(iconName);
}

const iconCss = ":host{display:inline-flex;height:1.5rem;width:1.5rem;min-height:1.5rem;min-width:1.5rem;color:inherit}:host .svg-container{display:block;position:relative;width:100%;height:100%}:host .svg-container svg{display:block;position:relative;height:100%;width:100%}:host .svg-container svg,:host .svg-container svg[fill],:host .svg-container svg [fill]{fill:currentColor !important}:host(.size-12){height:0.75rem;width:0.75rem;min-height:0.75rem;min-width:0.75rem}:host(.size-16){height:1rem;width:1rem;min-height:1rem;min-width:1rem}:host(.size-32){height:2rem;width:2rem;min-height:2rem;min-width:2rem}";
const IxIconStyle0 = iconCss;

const iconMissingSymbol = "data:image/svg+xml;utf8,<svg xmlns='http://www.w3.org/2000/svg' width='512' height='512' viewBox='0 0 512 512'><path fill-rule='evenodd' d='M384,0 L384,384 L0,384 L0,0 L384,0 Z M192,207.085 L57.751,341.333 L326.248,341.333 L192,207.085 Z M42.666,57.751 L42.666,326.248 L176.915,192 L42.666,57.751 Z M341.333,57.751 L207.085,192 L341.333,326.248 L341.333,57.751 Z M326.248,42.666 L57.751,42.666 L192,176.915 L326.248,42.666 Z' transform='translate(64 64)'/></svg>";
const Icon = /*@__PURE__*/ proxyCustomElement(class Icon extends HTMLElement {
    constructor() {
        super();
        this.__registerHost();
        this.__attachShadow();
        this.size = undefined;
        this.color = undefined;
        this.name = undefined;
        this.lazyLoading = false;
        this.svgContent = undefined;
        this.isVisible = false;
    }
    connectedCallback() {
        this.waitForRendering(() => {
            this.isVisible = true;
            this.loadIconContent();
        });
    }
    async loadIconContent() {
        try {
            this.svgContent = await resolveIcon(this.name);
        }
        catch (error) {
            this.svgContent = parseSVGDataContent(iconMissingSymbol);
        }
    }
    waitForRendering(onRender) {
        if (Build.isBrowser && this.lazyLoading && typeof window !== 'undefined' && window.IntersectionObserver) {
            const observer = new IntersectionObserver(entries => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        onRender();
                        observer.disconnect();
                    }
                });
            }, {
                rootMargin: '25px',
            });
            observer.observe(this.hostElement);
        }
        else {
            onRender();
        }
    }
    render() {
        const style = {};
        if (this.color) {
            style['color'] = `var(--theme-${this.color})`;
        }
        return (h(Host, { key: '8dc18bf3b1ab1e2a521bd0fee33fa91b856dc52b', style: style, class: {
                ['size-12']: this.size === '12',
                ['size-16']: this.size === '16',
                ['size-24']: this.size === '24',
                ['size-32']: this.size === '32',
            } }, h("div", { key: '48c3e8ae8e0d36be5fa112dbd408756f56c19535', class: 'svg-container', innerHTML: this.svgContent })));
    }
    static get assetsDirs() { return ["svg"]; }
    get hostElement() { return this; }
    static get watchers() { return {
        "name": ["loadIconContent"]
    }; }
    static get style() { return IxIconStyle0; }
}, [1, "ix-icon", {
        "size": [1],
        "color": [1],
        "name": [1],
        "lazyLoading": [4, "lazy-loading"],
        "svgContent": [32],
        "isVisible": [32]
    }, undefined, {
        "name": ["loadIconContent"]
    }]);
function defineCustomElement() {
    if (typeof customElements === "undefined") {
        return;
    }
    const components = ["ix-icon"];
    components.forEach(tagName => { switch (tagName) {
        case "ix-icon":
            if (!customElements.get(tagName)) {
                customElements.define(tagName, Icon);
            }
            break;
    } });
}

export { Icon as I, defineCustomElement as d };

//# sourceMappingURL=icon.js.map