/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { h, Host, } from '@stencil/core';
import { DateTime } from 'luxon';
export class TimePicker {
  constructor() {
    this._time = DateTime.now();
    this.format = 'TT';
    this.corners = 'rounded';
    this.individual = true;
    this.showHour = false;
    this.showMinutes = false;
    this.showSeconds = false;
    this.time = DateTime.now().toFormat(this.format);
    this.showTimeReference = undefined;
    this.timeReference = DateTime.fromFormat(this.time, this.format).toFormat('a');
    this.textSelectTime = 'Done';
    this.hourInputRef = undefined;
    this.minuteInputRef = undefined;
    this.secondInputRef = undefined;
    this.referenceInputRef = undefined;
  }
  get hour() {
    return this._time.hour;
  }
  get minutes() {
    return this._time.minute;
  }
  get seconds() {
    return this._time.second;
  }
  setInputValue(inputElement, value, max) {
    if (value > max) {
      inputElement.value = max.toString();
    }
    else if (value < 0) {
      inputElement.value = '0';
    }
    else {
      inputElement.value = value.toString();
    }
    this.updateAndEmitTime();
  }
  updateAndEmitTime() {
    this._time = this._time.set({
      hour: Number(this.hourInputRef.value),
      minute: Number(this.minuteInputRef.value),
      second: Number(this.secondInputRef.value),
    });
    this.setHourAccordingToReference();
    this.emitTimeChange();
  }
  toggleInputValue(inputElement, currentValue, step, max) {
    if (step === 'up') {
      if (currentValue === max) {
        inputElement.value = '0';
      }
      else {
        inputElement.stepUp();
      }
    }
    else {
      if (currentValue === 0) {
        inputElement.value = max.toString();
      }
      else {
        inputElement.stepDown();
      }
    }
  }
  toggleHourInputWithRef(inputElement, currentValue, step) {
    if (step === 'up') {
      if (currentValue === this.getMaxHour()) {
        inputElement.value = '0';
      }
      else {
        inputElement.stepUp();
      }
    }
    else {
      if ((this.showTimeReference &&
        this.timeReference === 'PM' &&
        currentValue === 12) ||
        currentValue === 0) {
        inputElement.value = this.getMaxDisplayedHour().toString();
      }
      else {
        inputElement.stepDown();
      }
    }
  }
  updateInput(step, { hours = undefined, minutes = undefined, seconds = undefined }) {
    if (hours) {
      if (this.showTimeReference) {
        this.toggleHourInputWithRef(this.hourInputRef, this.hour, step);
      }
      else {
        this.toggleInputValue(this.hourInputRef, this.hour, step, 23);
      }
    }
    if (minutes) {
      this.toggleInputValue(this.minuteInputRef, this.minutes, step, 59);
    }
    if (seconds) {
      this.toggleInputValue(this.secondInputRef, this.seconds, step, 59);
    }
    this.updateAndEmitTime();
  }
  changeReference() {
    if (this.timeReference === 'AM') {
      this.timeReference = 'PM';
    }
    else {
      this.timeReference = 'AM';
    }
    this.setHourAccordingToReference();
    this.emitTimeChange();
  }
  setHourAccordingToReference() {
    if (!this.showTimeReference) {
      return;
    }
    let hour = Number(this.hourInputRef.value);
    if (this.timeReference === 'PM')
      hour += 12;
    this._time = this._time.set({ hour });
  }
  emitTimeChange() {
    const time = this._time.toFormat(this.format);
    this.timeChange.emit(time);
  }
  componentWillLoad() {
    this._time = DateTime.fromFormat(this.time, this.format);
    if (this.showTimeReference === undefined) {
      const matchedKeys = Object.keys(DateTime.fromFormatExplain(this.time, this.format).matches);
      this.showTimeReference = matchedKeys.includes('a');
    }
  }
  /**
   * Get current time
   */
  async getCurrentTime() {
    return this._time.toFormat(this.format);
  }
  getDisplayedHour() {
    if (this.showTimeReference && this.timeReference === 'PM') {
      return this._time.hour - 12;
    }
    return this._time.hour;
  }
  getMaxDisplayedHour() {
    if (this.showTimeReference) {
      return 11;
    }
    return 23;
  }
  getMaxHour() {
    if (this.showTimeReference && this.timeReference === 'AM') {
      return 11;
    }
    return 23;
  }
  render() {
    let hideHour = !this.showHour;
    let hideMinutes = !this.showMinutes;
    let hideSeconds = !this.showSeconds;
    const hideTimeReference = !this.showTimeReference;
    const hideIndividual = !this.individual;
    if (hideHour && hideMinutes && hideSeconds) {
      hideHour = false;
      hideMinutes = false;
      hideSeconds = false;
    }
    const hideHourSeperator = hideMinutes || hideHour;
    const hideMinutesSeperator = hideSeconds || hideMinutes;
    return (h(Host, null, h("ix-date-time-card", { individual: this.individual, corners: this.corners }, h("div", { class: "header", slot: "header" }, h("div", { class: "title" }, "Time")), h("div", { class: "clock" }, h("div", { class: { columns: true, hidden: hideHour } }, h("ix-icon-button", { size: "16", onClick: () => this.updateInput('up', { hours: true }), ghost: true, icon: "chevron-up", variant: "Primary", class: "arrows" }), h("input", { class: "form-control", name: "hours", type: "number", placeholder: "HH", min: "0", max: this.getMaxDisplayedHour(), value: this.getDisplayedHour(), ref: (ref) => (this.hourInputRef = ref), onChange: () => this.setInputValue(this.hourInputRef, Number(this.hourInputRef.value), this.getMaxDisplayedHour()) }), h("ix-icon-button", { size: "16", onClick: () => this.updateInput('down', { hours: true }), ghost: true, icon: "chevron-down", variant: "Primary", class: "arrows" })), h("div", { class: { 'column-seperator': true, hidden: hideHourSeperator } }, ":"), h("div", { class: { columns: true, hidden: hideMinutes } }, h("ix-icon-button", { size: "16", onClick: () => this.updateInput('up', { minutes: true }), ghost: true, icon: "chevron-up", variant: "Primary", class: "arrows" }), h("input", { class: "form-control", name: "minutes", type: "number", placeholder: "MM", min: "0", max: "59", value: this.minutes, ref: (ref) => (this.minuteInputRef = ref), onChange: () => this.setInputValue(this.minuteInputRef, Number(this.minuteInputRef.value), 59) }), h("ix-icon-button", { size: "16", onClick: () => this.updateInput('down', { minutes: true }), ghost: true, icon: "chevron-down", variant: "Primary", class: "arrows" })), h("div", { class: { 'column-seperator': true, hidden: hideMinutesSeperator } }, ":"), h("div", { class: { columns: true, hidden: hideSeconds } }, h("ix-icon-button", { size: "16", onClick: () => this.updateInput('up', { seconds: true }), ghost: true, icon: "chevron-up", variant: "Primary", class: "arrows" }), h("input", { class: "form-control", name: "seconds", type: "number", placeholder: "SS", min: "0", max: "59", value: this.seconds, ref: (ref) => (this.secondInputRef = ref), onChange: () => this.setInputValue(this.secondInputRef, Number(this.secondInputRef.value), 59) }), h("ix-icon-button", { size: "16", onClick: () => this.updateInput('down', { seconds: true }), ghost: true, icon: "chevron-down", variant: "Primary", class: "arrows" })), h("div", { class: {
        columns: true,
        'default-space': true,
        hidden: hideTimeReference,
      } }, h("ix-icon-button", { size: "16", onClick: () => this.changeReference(), ghost: true, icon: "chevron-up", variant: "Primary", class: "arrows" }), h("div", { class: "time-reference" }, this.timeReference), h("ix-icon-button", { size: "16", onClick: () => this.changeReference(), ghost: true, icon: "chevron-down", variant: "Primary", class: "arrows" }))), h("div", { class: { button: true, hidden: hideIndividual } }, h("ix-button", { onClick: () => this.done.emit(this._time.toFormat(this.format)) }, this.textSelectTime)))));
  }
  static get is() { return "ix-time-picker"; }
  static get encapsulation() { return "scoped"; }
  static get originalStyleUrls() {
    return {
      "$": ["time-picker.scss"]
    };
  }
  static get styleUrls() {
    return {
      "$": ["time-picker.css"]
    };
  }
  static get properties() {
    return {
      "format": {
        "type": "string",
        "mutable": false,
        "complexType": {
          "original": "string",
          "resolved": "string",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [{
              "name": "since",
              "text": "1.1.0"
            }],
          "text": "Format of time string"
        },
        "attribute": "format",
        "reflect": false,
        "defaultValue": "'TT'"
      },
      "corners": {
        "type": "string",
        "mutable": false,
        "complexType": {
          "original": "TimePickerCorners",
          "resolved": "\"left\" | \"right\" | \"rounded\"",
          "references": {
            "TimePickerCorners": {
              "location": "local",
              "path": "/__w/ix/ix/packages/core/src/components/time-picker/time-picker.tsx"
            }
          }
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "Corner style"
        },
        "attribute": "corners",
        "reflect": false,
        "defaultValue": "'rounded'"
      },
      "individual": {
        "type": "boolean",
        "mutable": false,
        "complexType": {
          "original": "boolean",
          "resolved": "boolean",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [{
              "name": "deprecated",
              "text": "Will be removed in 2.0.0"
            }],
          "text": ""
        },
        "attribute": "individual",
        "reflect": false,
        "defaultValue": "true"
      },
      "showHour": {
        "type": "boolean",
        "mutable": false,
        "complexType": {
          "original": "boolean",
          "resolved": "boolean",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "Show hour input"
        },
        "attribute": "show-hour",
        "reflect": false,
        "defaultValue": "false"
      },
      "showMinutes": {
        "type": "boolean",
        "mutable": false,
        "complexType": {
          "original": "boolean",
          "resolved": "boolean",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "Show minutes input"
        },
        "attribute": "show-minutes",
        "reflect": false,
        "defaultValue": "false"
      },
      "showSeconds": {
        "type": "boolean",
        "mutable": false,
        "complexType": {
          "original": "boolean",
          "resolved": "boolean",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "Show seconds input"
        },
        "attribute": "show-seconds",
        "reflect": false,
        "defaultValue": "false"
      },
      "time": {
        "type": "string",
        "mutable": false,
        "complexType": {
          "original": "string",
          "resolved": "string",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [{
              "name": "since",
              "text": "1.1.0"
            }],
          "text": "Select time with format string"
        },
        "attribute": "time",
        "reflect": false,
        "defaultValue": "DateTime.now().toFormat(this.format)"
      },
      "showTimeReference": {
        "type": "any",
        "mutable": true,
        "complexType": {
          "original": "any",
          "resolved": "any",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [{
              "name": "since",
              "text": "1.1.0 time reference is default aligned with formt tt"
            }],
          "text": "Show time reference input"
        },
        "attribute": "show-time-reference",
        "reflect": false,
        "defaultValue": "undefined"
      },
      "timeReference": {
        "type": "string",
        "mutable": true,
        "complexType": {
          "original": "'AM' | 'PM'",
          "resolved": "\"AM\" | \"PM\"",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "Set time reference"
        },
        "attribute": "time-reference",
        "reflect": false,
        "defaultValue": "DateTime.fromFormat(\n    this.time,\n    this.format\n  ).toFormat('a') as 'PM' | 'AM'"
      },
      "textSelectTime": {
        "type": "string",
        "mutable": false,
        "complexType": {
          "original": "string",
          "resolved": "string",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [{
              "name": "since",
              "text": "1.1.0"
            }],
          "text": "Text of date select button"
        },
        "attribute": "text-select-time",
        "reflect": false,
        "defaultValue": "'Done'"
      }
    };
  }
  static get states() {
    return {
      "hourInputRef": {},
      "minuteInputRef": {},
      "secondInputRef": {},
      "referenceInputRef": {}
    };
  }
  static get events() {
    return [{
        "method": "done",
        "name": "done",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Time event"
        },
        "complexType": {
          "original": "string",
          "resolved": "string",
          "references": {}
        }
      }, {
        "method": "timeChange",
        "name": "timeChange",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Time change event"
        },
        "complexType": {
          "original": "string",
          "resolved": "string",
          "references": {}
        }
      }];
  }
  static get methods() {
    return {
      "getCurrentTime": {
        "complexType": {
          "signature": "() => Promise<string>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<string>"
        },
        "docs": {
          "text": "Get current time",
          "tags": []
        }
      }
    };
  }
  static get elementRef() { return "hostElement"; }
}
//# sourceMappingURL=time-picker.js.map
