/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { arrow, autoUpdate, computePosition, flip, offset, shift, } from '@floating-ui/dom';
import { h, Host } from '@stencil/core';
const numberToPixel = (value) => (value != null ? `${value}px` : '');
/**
 * @slot title-icon - Icon of tooltip title
 * @slot title-content - Content of tooltip title
 *
 * @since 1.4.0
 */
export class Tooltip {
  constructor() {
    this.onMouseEnterBind = this.showTooltip.bind(this);
    this.onMouseLeaveBind = this.hideTooltip.bind(this);
    this.tooltipCloseTimeInMS = 50;
    this.for = undefined;
    this.titleContent = undefined;
    this.interactive = false;
    this.placement = 'top';
    this.visible = false;
  }
  get arrowElement() {
    return this.hostElement.shadowRoot.querySelector('.arrow');
  }
  destroyAutoUpdate() {
    if (this.disposeAutoUpdate !== undefined) {
      this.disposeAutoUpdate();
    }
  }
  showTooltip(e) {
    clearTimeout(this.hideTooltipTimeout);
    this.visible = true;
    this.computeTooltipPosition(e.target);
  }
  hideTooltip() {
    this.hideTooltipTimeout = setTimeout(() => {
      this.visible = false;
    }, this.tooltipCloseTimeInMS);
    this.destroyAutoUpdate();
  }
  computeArrowPosition({ placement, middlewareData, }) {
    let { x, y } = middlewareData.arrow;
    if (placement.startsWith('top')) {
      return {
        left: numberToPixel(x),
        top: numberToPixel(y),
      };
    }
    if (placement.startsWith('right')) {
      return {
        left: numberToPixel(-4),
        top: numberToPixel(y),
      };
    }
    if (placement.startsWith('bottom')) {
      return {
        left: numberToPixel(x),
        top: numberToPixel(-4),
      };
    }
    if (placement.startsWith('left')) {
      return {
        right: numberToPixel(-4),
        top: numberToPixel(y),
      };
    }
  }
  async computeTooltipPosition(target) {
    this.disposeAutoUpdate = autoUpdate(target, this.hostElement, async () => {
      setTimeout(async () => {
        const computeResponse = await computePosition(target, this.hostElement, {
          strategy: 'fixed',
          placement: this.placement,
          middleware: [
            shift(),
            offset(8),
            arrow({
              element: this.arrowElement,
            }),
            flip({
              fallbackStrategy: 'initialPlacement',
              padding: 10,
            }),
          ],
        });
        if (computeResponse.middlewareData.arrow) {
          const arrowPosition = this.computeArrowPosition(computeResponse);
          Object.assign(this.arrowElement.style, arrowPosition);
        }
        const { x, y } = computeResponse;
        Object.assign(this.hostElement.style, {
          left: x !== null ? `${x}px` : '',
          top: y !== null ? `${y}px` : '',
        });
      });
    }, {
      ancestorResize: true,
      ancestorScroll: true,
      elementResize: true,
    });
  }
  queryAnchorElements() {
    return Array.from(document.querySelectorAll(this.for));
  }
  registerTriggerListener() {
    const elements = this.queryAnchorElements();
    elements.forEach((e) => {
      e.addEventListener('mouseenter', this.onMouseEnterBind);
      e.addEventListener('mouseleave', this.onMouseLeaveBind);
    });
  }
  registerTooltipListener() {
    this.hostElement.addEventListener('mouseenter', () => {
      if (this.interactive) {
        clearTimeout(this.hideTooltipTimeout);
      }
    });
    this.hostElement.addEventListener('mouseleave', () => {
      this.hideTooltip();
    });
  }
  componentDidLoad() {
    if (this.interactive) {
      this.tooltipCloseTimeInMS = 150;
    }
    this.observer = new MutationObserver(() => {
      this.registerTriggerListener();
    });
    this.observer.observe(document.body, {
      attributes: true,
      attributeFilter: ['data-ix-tooltip'],
      childList: true,
      subtree: true,
    });
    this.registerTriggerListener();
    this.registerTooltipListener();
  }
  disconnectedCallback() {
    this.observer.disconnect();
    this.destroyAutoUpdate();
  }
  render() {
    const tooltipContentClass = {
      'tooltip-content': true,
    };
    return (h(Host, { class: {
        visible: this.visible,
      } }, h("div", { class: 'tooltip-title' }, h("slot", { name: "title-icon" }), h("ix-typography", { variant: "default-title" }, this.titleContent, h("slot", { name: "title-content" }))), h("div", { class: tooltipContentClass }, h("slot", null)), h("div", { class: "arrow" })));
  }
  static get is() { return "ix-tooltip"; }
  static get encapsulation() { return "shadow"; }
  static get originalStyleUrls() {
    return {
      "$": ["tooltip.scss"]
    };
  }
  static get styleUrls() {
    return {
      "$": ["tooltip.css"]
    };
  }
  static get properties() {
    return {
      "for": {
        "type": "string",
        "mutable": false,
        "complexType": {
          "original": "string",
          "resolved": "string",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "CSS selector for hover trigger element e.g. `for=\"[data-my-custom-select]\"`"
        },
        "attribute": "for",
        "reflect": false
      },
      "titleContent": {
        "type": "string",
        "mutable": false,
        "complexType": {
          "original": "string",
          "resolved": "string",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "Title of the tooltip"
        },
        "attribute": "title-content",
        "reflect": false
      },
      "interactive": {
        "type": "boolean",
        "mutable": false,
        "complexType": {
          "original": "boolean",
          "resolved": "boolean",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "Define if the user can access the tooltip via mouse."
        },
        "attribute": "interactive",
        "reflect": false,
        "defaultValue": "false"
      },
      "placement": {
        "type": "string",
        "mutable": false,
        "complexType": {
          "original": "'top' | 'right' | 'bottom' | 'left'",
          "resolved": "\"bottom\" | \"left\" | \"right\" | \"top\"",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [{
              "name": "since",
              "text": "1.5.0"
            }],
          "text": "Initial placement of the tooltip. If the placement don\"t have enough space,\nthe tooltip will placed on another location."
        },
        "attribute": "placement",
        "reflect": false,
        "defaultValue": "'top'"
      }
    };
  }
  static get states() {
    return {
      "visible": {}
    };
  }
  static get elementRef() { return "hostElement"; }
}
//# sourceMappingURL=tooltip.js.map
