/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { h, Host, } from "@stencil/core";
import anime from "animejs";
import { ApplicationLayoutContext } from "../utils/application-layout/context";
import { useContextProvider } from "../utils/context";
export class MapNavigation {
    constructor() {
        this.applicationName = undefined;
        this.navigationTitle = undefined;
        this.hideContextMenu = true;
        this.isSidebarOpen = true;
        this.hasContentHeader = false;
    }
    get menu() {
        return this.hostElement.querySelector('ix-menu');
    }
    get menuOverlay() {
        return this.hostElement.querySelector('ix-menu-overlay');
    }
    get mapNavMenu() {
        return this.hostElement.shadowRoot.querySelector('.map-nav-menu');
    }
    get sidebar() {
        return this.hostElement.shadowRoot.querySelector('.map-nav-sidebar');
    }
    get overlay() {
        return this.hostElement.shadowRoot.querySelector('#overlay');
    }
    componentDidRender() {
        this.appendMenu();
        this.closeOverlay();
    }
    componentWillLoad() {
        useContextProvider(this.hostElement, ApplicationLayoutContext, {
            hideHeader: false,
            host: 'map-navigation',
        });
    }
    appendMenu() {
        this.menu.addEventListener('mapExpandChange', (event) => {
            const state = !event.detail;
            this.toggleSidebar(state);
        });
        this.menu.enableMapExpand = true;
    }
    /**
     * Change the visibility of the sidebar
     *
     * @param show new visibility state
     * @since 1.6.0
     */
    async toggleSidebar(show) {
        if (show !== undefined) {
            this.isSidebarOpen = show;
        }
        else {
            this.isSidebarOpen = !this.isSidebarOpen;
        }
        if (this.isSidebarOpen) {
            this.openSidebar();
        }
        else {
            this.closeSidebar();
        }
        this.navigationToggled.emit(this.isSidebarOpen);
        this.menu.toggleMapExpand(this.isSidebarOpen);
    }
    closeSidebar() {
        anime({
            targets: this.sidebar,
            duration: MapNavigation.defaultTime,
            marginLeft: [0, '-29.75rem'],
            opacity: [1, 0],
            easing: 'easeInSine',
            complete: () => {
                this.sidebar.classList.add('d-none');
            },
        });
    }
    openSidebar() {
        anime({
            targets: this.sidebar,
            duration: MapNavigation.defaultTime,
            marginLeft: ['-29.75rem', 0],
            opacity: [0, 1],
            easing: 'easeOutSine',
            begin: () => {
                this.sidebar.classList.remove('d-none');
            },
        });
    }
    /**
     * Open a overlay inside content area
     * @deprecated Will be removed in 2.0.0. Use slot based approach
     *
     * @param name
     * @param component
     * @param icon
     * @param color
     */
    async openOverlay(name, component, icon, color) {
        anime({
            targets: this.overlay,
            duration: MapNavigation.slowTime,
            backdropFilter: [0, 'blur(1rem)'],
            translateX: ['-4rem', 0],
            opacity: [0, 1],
            easing: 'easeOutSine',
            begin: () => {
                this.overlay.classList.remove('d-none');
            },
        });
        const overlayInstance = document.createElement('ix-map-navigation-overlay');
        overlayInstance.setAttribute('color', color);
        overlayInstance.setAttribute('name', name);
        overlayInstance.setAttribute('icon', icon);
        overlayInstance.setAttribute('slot', 'overlay');
        overlayInstance.addEventListener('closeClick', () => this.closeOverlay());
        overlayInstance.appendChild(component);
        this.hostElement.appendChild(overlayInstance);
    }
    /**
     * Close current shown overlay
     * @deprecated Will be removed in 2.0.0. Use slot based approach
     */
    async closeOverlay() {
        anime({
            targets: this.overlay,
            duration: MapNavigation.slowTime,
            backdropFilter: ['blur(1rem)', 0],
            translateX: [0, '-4rem'],
            opacity: [1, 0],
            easing: 'easeInSine',
            complete: () => {
                var _a;
                if (!this.overlay) {
                    return;
                }
                (_a = this.overlay.firstChild) === null || _a === void 0 ? void 0 : _a.remove();
                this.overlay.classList.add('d-none');
            },
        });
    }
    checkHasContentHeader(e) {
        const nodes = e.currentTarget.assignedNodes({
            flatten: true,
        });
        this.hasContentHeader = (nodes === null || nodes === void 0 ? void 0 : nodes.length) !== 0;
    }
    render() {
        return (h(Host, { key: '65d980effdeeac98d92a0c8d150cf96d9b9beb3d' }, h("slot", { key: '822f13c097836e57452c453d9b45e4774d7394ed', name: "menu" }), h("div", { key: '8f58c729f86f0e3b6d6083b5f0f0fda28e74c9eb', class: "map-nav" }, h("div", { key: '0f979e624122ce789aa094bf428dc39414d66207', class: "map-nav-sidebar" }, h("div", { key: 'a8cf78610ff79460f00716f52968f18f0fc83859', class: "map-nav-header" }, h("ix-application-header", { key: '98c2b5e98691c1505879a3ed2d1f3cdba4c94a62', name: this.applicationName, class: "map-nav-header-brand" }, h("slot", { key: 'b9036cd5a8873b9562d58783264286bb74cf08f7', slot: "logo", name: "logo" }))), h("div", { key: '42a13b7c8ee93cf69b06c12a3fb600f740555d1b', class: "map-nav-sidebar-content" }, h("div", { key: 'd4e0218e43cabf84bd7faa2c38a7e17074982ba2', class: "map-nav-sidebar-static-content" }, h("div", { key: '0b1add8805f5b3d37a546d20fa6e99a1d6bf35ac', class: "map-nav-title" }, this.navigationTitle), this.hideContextMenu ? ('') : (h("ix-icon-button", { icon: 'context-menu', ghost: true, size: "24", variant: "secondary", onClick: (_) => this.contextMenuClick.emit() }))), h("div", { key: 'b2e15f627226c8d01e7f1e73f14d40ebb58e151d', class: "map-nav-sidebar-user-content" }, h("slot", { key: 'cb26bf448b02c54e46398d65b0361e453115c20d', name: "sidebar-content" })))), h("div", { key: '8fe42e8570256b4f09874100e2d34d323e22ac0d', class: "content" }, h("div", { key: '17a816d735d80ebca9e6cc7a7e9c6671e2e9a059', class: {
                'map-nav-header-content': true,
                'bg-2': true,
                empty: !this.hasContentHeader,
            } }, h("slot", { key: '734a74f600419d597f306638cbe97c5504a2e15d', name: "content-header", onSlotchange: (e) => this.checkHasContentHeader(e) })), h("main", { key: '76d831c62b3a1339e18ab4f4b77c2dd73b915118' }, h("slot", { key: '5ec164d85791363dc232b3861458446c2a9787dc' }), h("slot", { key: 'cc823b370d2b9f0f7151388ebedcd4958277422c', name: "overlay" }))))));
    }
    static get is() { return "ix-map-navigation"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["map-navigation.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["map-navigation.css"]
        };
    }
    static get properties() {
        return {
            "applicationName": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Application name"
                },
                "attribute": "application-name",
                "reflect": false
            },
            "navigationTitle": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Navigation title"
                },
                "attribute": "navigation-title",
                "reflect": false
            },
            "hideContextMenu": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Hide the sidebar context menu button when set to true"
                },
                "attribute": "hide-context-menu",
                "reflect": false,
                "defaultValue": "true"
            }
        };
    }
    static get states() {
        return {
            "isSidebarOpen": {},
            "hasContentHeader": {}
        };
    }
    static get events() {
        return [{
                "method": "navigationToggled",
                "name": "navigationToggled",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Navigation toggled"
                },
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                }
            }, {
                "method": "contextMenuClick",
                "name": "contextMenuClick",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Context menu clicked"
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }];
    }
    static get methods() {
        return {
            "toggleSidebar": {
                "complexType": {
                    "signature": "(show?: boolean) => Promise<void>",
                    "parameters": [{
                            "name": "show",
                            "type": "boolean",
                            "docs": "new visibility state"
                        }],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Change the visibility of the sidebar",
                    "tags": [{
                            "name": "param",
                            "text": "show new visibility state"
                        }, {
                            "name": "since",
                            "text": "1.6.0"
                        }]
                }
            },
            "openOverlay": {
                "complexType": {
                    "signature": "(name: string, component: HTMLElement, icon?: string, color?: string) => Promise<void>",
                    "parameters": [{
                            "name": "name",
                            "type": "string",
                            "docs": ""
                        }, {
                            "name": "component",
                            "type": "HTMLElement",
                            "docs": ""
                        }, {
                            "name": "icon",
                            "type": "string",
                            "docs": ""
                        }, {
                            "name": "color",
                            "type": "string",
                            "docs": ""
                        }],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        },
                        "HTMLElement": {
                            "location": "global",
                            "id": "global::HTMLElement"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Open a overlay inside content area",
                    "tags": [{
                            "name": "deprecated",
                            "text": "Will be removed in 2.0.0. Use slot based approach"
                        }, {
                            "name": "param",
                            "text": "name"
                        }, {
                            "name": "param",
                            "text": "component"
                        }, {
                            "name": "param",
                            "text": "icon"
                        }, {
                            "name": "param",
                            "text": "color"
                        }]
                }
            },
            "closeOverlay": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Close current shown overlay",
                    "tags": [{
                            "name": "deprecated",
                            "text": "Will be removed in 2.0.0. Use slot based approach"
                        }]
                }
            }
        };
    }
    static get elementRef() { return "hostElement"; }
}
MapNavigation.defaultTime = 150;
MapNavigation.slowTime = 500;
//# sourceMappingURL=map-navigation.js.map
