/*
 * SPDX-FileCopyrightText: 2024 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { h, Host, } from "@stencil/core";
import { createMutationObserver } from "../utils/mutation-observer";
import { hasSlottedElements } from "../utils/shadow-dom";
export class Group {
    constructor() {
        this.observer = null;
        this.suppressHeaderSelection = false;
        this.header = undefined;
        this.subHeader = undefined;
        this.collapsed = true;
        this.selected = false;
        this.index = undefined;
        this.expandOnHeaderClick = false;
        this.itemSelected = false;
        this.slotSize = this.groupItems.length;
        this.footerVisible = false;
        this.showExpandCollapsedIcon = false;
    }
    selectedChanged(newSelected) {
        if (newSelected === false) {
            this.changeItemIndex();
        }
    }
    get dropdownItems() {
        return Array.from(this.hostElement.querySelectorAll('ix-group-dropdown-item'));
    }
    get groupItems() {
        return Array.from(this.hostElement.querySelectorAll('ix-group-item:not(.footer)'));
    }
    get groupContent() {
        var _a;
        return (_a = this.hostElement.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector('.group-content');
    }
    onExpandClick(event) {
        const oldCollapsed = this.collapsed;
        this.collapsed = !this.collapsed;
        const { defaultPrevented } = this.collapsedChanged.emit(this.collapsed);
        event.stopPropagation();
        if (defaultPrevented) {
            this.collapsed = oldCollapsed;
        }
    }
    onHeaderClick(event) {
        if (this.suppressHeaderSelection) {
            this.onExpandClick(event);
            return;
        }
        this.changeHeaderSelection(!this.selected);
        this.changeItemIndex();
    }
    changeHeaderSelection(newSelection) {
        const oldIsHeaderSelected = this.selected;
        const newIsHeaderSelected = newSelection;
        this.selected = newIsHeaderSelected;
        const { defaultPrevented } = this.selectGroup.emit(newIsHeaderSelected);
        if (defaultPrevented) {
            this.selected = oldIsHeaderSelected;
            return;
        }
    }
    changeItemIndex(index) {
        const oldIndex = this.index;
        const newIndex = index === this.index ? undefined : index;
        if (this.index === newIndex) {
            return;
        }
        this.index = newIndex;
        const { defaultPrevented } = this.selectItem.emit(newIndex);
        if (defaultPrevented) {
            this.index = oldIndex;
            return;
        }
        const items = this.groupItems;
        items.forEach((item, i) => {
            item.selected = i === this.index;
        });
        this.itemSelected = items.some((item) => item.selected);
    }
    onSlotChange() {
        var _a;
        const slot = (_a = this.hostElement.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector('slot[name="footer"]');
        if (slot) {
            this.footerVisible = hasSlottedElements(slot);
        }
    }
    componentWillRender() {
        this.groupItems.forEach((item, index) => {
            item.selected = index === this.index;
            item.index = index;
        });
    }
    componentDidLoad() {
        this.observer = createMutationObserver(() => {
            this.slotSize = this.groupItems.length;
        });
        if (!this.groupContent) {
            return;
        }
        this.observer.observe(this.groupContent, {
            childList: true,
        });
    }
    disconnectedCallback() {
        if (this.observer) {
            this.observer.disconnect();
        }
    }
    onItemClicked(event) {
        if (event.target instanceof HTMLElement) {
            const item = event.target;
            const index = this.groupItems.indexOf(item);
            this.changeItemIndex(index);
        }
    }
    render() {
        return (h(Host, { key: '812601d494909804a300f73d14e2e423e2c5cce9' }, h("div", { key: '2725ce086aee61abdfbf9e2db6bb1c1c1f936cc9', class: {
                'group-header': true,
                expand: !this.collapsed,
                selected: this.selected,
            }, tabindex: "0" }, h("div", { key: '359308aacda0079e12bb33273b93aad571b2568e', class: "group-header-clickable", onClick: (e) => this.onHeaderClick(e) }, h("div", { key: 'e770dd6b563036aef00730a5d497cc8b777bf1f9', class: {
                'group-header-selection-indicator': true,
                'group-header-selection-indicator-item-selected': this.itemSelected,
            } }), h("div", { key: '52d7d439f1f2af260bfa707a418bdfe5979ade02', class: "btn-expand-header" }, h("ix-icon", { key: 'ab70e0357f2cd002412f950b6a770825f73e481d', "data-testid": "expand-collapsed-icon", class: {
                hidden: !this.showExpandCollapsedIcon,
            }, name: this.collapsed ? 'chevron-right-small' : 'chevron-down-small', onClick: (e) => this.onExpandClick(e) })), h("div", { key: 'ec04f3094c5949cd96dc1bcbf00434dff3012be4', class: "group-header-content" }, this.header ? (h("div", { class: "group-header-props-container" }, h("div", { class: "group-header-title" }, h("span", { title: this.header }, this.header)), h("div", { class: "group-subheader", title: this.subHeader }, this.subHeader))) : null, h("slot", { key: '3a056192f8217060b9ef9f90cba385028b26c04e', name: "header" }))), h("ix-group-context-menu", { key: 'd3d14e900f4e6f6cbbbf27d76eab427f5149f900' }, h("slot", { key: 'c538f3b3fd67f4d486a94c304f9cd938f0337af2', name: "dropdown" }))), h("div", { key: 'ee27945481e3994204885fd6e0e6bd6f1601aaf5', class: {
                'group-content': true,
            } }, h("div", { key: '5cd463be44651ec638b54ea469aa3dc077e3b04f', style: {
                display: this.collapsed ? 'none' : 'contents',
            } }, h("slot", { key: '1565d5c06d317e41f4d4851c69cd38af88e4918c', onSlotchange: () => {
                var _a;
                const slot = (_a = this.hostElement.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector('slot:not([name])');
                this.showExpandCollapsedIcon = hasSlottedElements(slot);
            } }), h("ix-group-item", { key: '5d5632cba350db31eb5fa1f64fe3d33e6e0e2748', suppressSelection: true, focusable: false, class: {
                footer: true,
                'footer-visible': this.footerVisible,
            } }, h("slot", { key: '13741bc640b4bed51ffaa9e414cfc0c9efe06444', name: "footer", onSlotchange: () => this.onSlotChange() }))))));
    }
    static get is() { return "ix-group"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["group.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["group.css"]
        };
    }
    static get properties() {
        return {
            "suppressHeaderSelection": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Prevent header from being selectable"
                },
                "attribute": "suppress-header-selection",
                "reflect": false,
                "defaultValue": "false"
            },
            "header": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Group header"
                },
                "attribute": "header",
                "reflect": false
            },
            "subHeader": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Group header subtitle"
                },
                "attribute": "sub-header",
                "reflect": false
            },
            "collapsed": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Whether the group is collapsed or expanded. Defaults to true."
                },
                "attribute": "collapsed",
                "reflect": true,
                "defaultValue": "true"
            },
            "selected": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Whether the group is selected."
                },
                "attribute": "selected",
                "reflect": true,
                "defaultValue": "false"
            },
            "index": {
                "type": "number",
                "mutable": true,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The index of the selected group entry.\nIf undefined no group item is selected."
                },
                "attribute": "index",
                "reflect": true
            },
            "expandOnHeaderClick": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Expand the group if the header is clicked"
                },
                "attribute": "expand-on-header-click",
                "reflect": false,
                "defaultValue": "false"
            }
        };
    }
    static get states() {
        return {
            "itemSelected": {},
            "slotSize": {},
            "footerVisible": {},
            "showExpandCollapsedIcon": {}
        };
    }
    static get events() {
        return [{
                "method": "selectGroup",
                "name": "selectGroup",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emits when whole group gets selected."
                },
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                }
            }, {
                "method": "selectItem",
                "name": "selectItem",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emits when group item gets selected."
                },
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                }
            }, {
                "method": "collapsedChanged",
                "name": "collapsedChanged",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Group collapsed"
                },
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                }
            }];
    }
    static get elementRef() { return "hostElement"; }
    static get watchers() {
        return [{
                "propName": "selected",
                "methodName": "selectedChanged"
            }];
    }
    static get listeners() {
        return [{
                "name": "selectedChanged",
                "method": "onItemClicked",
                "target": undefined,
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=group.js.map
