/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { h, Host, } from "@stencil/core";
import anime from "animejs";
import { ApplicationLayoutContext } from "../utils/application-layout/context";
import { useContextProvider } from "../utils/context";
export class MapNavigation {
    constructor() {
        this.applicationName = undefined;
        this.navigationTitle = undefined;
        this.hideContextMenu = true;
        this.isSidebarOpen = true;
        this.hasContentHeader = false;
    }
    get menu() {
        return this.hostElement.querySelector('ix-menu');
    }
    get menuOverlay() {
        return this.hostElement.querySelector('ix-menu-overlay');
    }
    get mapNavMenu() {
        return this.hostElement.shadowRoot.querySelector('.map-nav-menu');
    }
    get sidebar() {
        return this.hostElement.shadowRoot.querySelector('.map-nav-sidebar');
    }
    get overlay() {
        return this.hostElement.shadowRoot.querySelector('#overlay');
    }
    componentDidRender() {
        this.appendMenu();
        this.closeOverlay();
    }
    componentWillLoad() {
        useContextProvider(this.hostElement, ApplicationLayoutContext, {
            hideHeader: false,
            host: 'map-navigation',
        });
    }
    appendMenu() {
        this.menu.addEventListener('mapExpandChange', (event) => {
            const state = !event.detail;
            this.toggleSidebar(state);
        });
        this.menu.enableMapExpand = true;
    }
    /**
     * Change the visibility of the sidebar
     *
     * @param show new visibility state
     * @since 1.6.0
     */
    async toggleSidebar(show) {
        if (show !== undefined) {
            this.isSidebarOpen = show;
        }
        else {
            this.isSidebarOpen = !this.isSidebarOpen;
        }
        if (this.isSidebarOpen) {
            this.openSidebar();
        }
        else {
            this.closeSidebar();
        }
        this.navigationToggled.emit(this.isSidebarOpen);
        this.menu.toggleMapExpand(this.isSidebarOpen);
    }
    closeSidebar() {
        anime({
            targets: this.sidebar,
            duration: MapNavigation.defaultTime,
            marginLeft: [0, '-29.75rem'],
            opacity: [1, 0],
            easing: 'easeInSine',
            complete: () => {
                this.sidebar.classList.add('d-none');
            },
        });
    }
    openSidebar() {
        anime({
            targets: this.sidebar,
            duration: MapNavigation.defaultTime,
            marginLeft: ['-29.75rem', 0],
            opacity: [0, 1],
            easing: 'easeOutSine',
            begin: () => {
                this.sidebar.classList.remove('d-none');
            },
        });
    }
    /**
     * Open a overlay inside content area
     * @deprecated Will be removed in 2.0.0. Use slot based approach
     *
     * @param name
     * @param component
     * @param icon
     * @param color
     */
    async openOverlay(name, component, icon, color) {
        anime({
            targets: this.overlay,
            duration: MapNavigation.slowTime,
            backdropFilter: [0, 'blur(1rem)'],
            translateX: ['-4rem', 0],
            opacity: [0, 1],
            easing: 'easeOutSine',
            begin: () => {
                this.overlay.classList.remove('d-none');
            },
        });
        const overlayInstance = document.createElement('ix-map-navigation-overlay');
        overlayInstance.setAttribute('color', color);
        overlayInstance.setAttribute('name', name);
        overlayInstance.setAttribute('icon', icon);
        overlayInstance.setAttribute('slot', 'overlay');
        overlayInstance.addEventListener('closeClick', () => this.closeOverlay());
        overlayInstance.appendChild(component);
        this.hostElement.appendChild(overlayInstance);
    }
    /**
     * Close current shown overlay
     * @deprecated Will be removed in 2.0.0. Use slot based approach
     */
    async closeOverlay() {
        anime({
            targets: this.overlay,
            duration: MapNavigation.slowTime,
            backdropFilter: ['blur(1rem)', 0],
            translateX: [0, '-4rem'],
            opacity: [1, 0],
            easing: 'easeInSine',
            complete: () => {
                var _a;
                if (!this.overlay) {
                    return;
                }
                (_a = this.overlay.firstChild) === null || _a === void 0 ? void 0 : _a.remove();
                this.overlay.classList.add('d-none');
            },
        });
    }
    checkHasContentHeader(e) {
        const nodes = e.currentTarget.assignedNodes({
            flatten: true,
        });
        this.hasContentHeader = (nodes === null || nodes === void 0 ? void 0 : nodes.length) !== 0;
    }
    render() {
        return (h(Host, { key: 'cb5c6f39555fbd806f1eca16919c277843a23351' }, h("slot", { key: '9da04acd9ef3e5b8f322669406f9ad4b812c9c79', name: "menu" }), h("div", { key: 'dced8e859776aa7d9923156a91fb149e9409af31', class: "map-nav" }, h("div", { key: '97fb4b6ff84005e96873680cfe32c7a2b33b7c0e', class: "map-nav-sidebar" }, h("div", { key: '1f069b0b89c0993df6ad24fba1335a5cb2cd2083', class: "map-nav-header" }, h("ix-application-header", { key: 'bbb0e01ae9194aed17711514b70124704831f7ac', name: this.applicationName, class: "map-nav-header-brand" }, h("slot", { key: '78b9d282b03368708a5a4a6197fccfcd586ae16f', slot: "logo", name: "logo" }))), h("div", { key: '061ae2f057117ac51f9baa296940cddc66c89d23', class: "map-nav-sidebar-content" }, h("div", { key: 'f5d5adb2daf4ba5212e7bf66ddc860431e453f1b', class: "map-nav-sidebar-static-content" }, h("div", { key: '07f6bebd3988744a4e03b61a5f30ecce310bba85', class: "map-nav-title" }, this.navigationTitle), this.hideContextMenu ? ('') : (h("ix-icon-button", { icon: 'context-menu', ghost: true, size: "24", variant: "secondary", onClick: (_) => this.contextMenuClick.emit() }))), h("div", { key: '2ac8e086d83fc82409f5bfd1ecea2ba66eb06606', class: "map-nav-sidebar-user-content" }, h("slot", { key: 'f447fb6a5995ddb0ea0c08e3101c2d7ab29805f4', name: "sidebar-content" })))), h("div", { key: 'be19906feccbba5e0fd4736df63afa4584b2ab1f', class: "content" }, h("div", { key: 'bf1127e632cc4b7ee9f5afb8cc51aabc8abe5600', class: {
                'map-nav-header-content': true,
                'bg-2': true,
                empty: !this.hasContentHeader,
            } }, h("slot", { key: '803807a9158ad13f04e6e66c5472c527dde93a4a', name: "content-header", onSlotchange: (e) => this.checkHasContentHeader(e) })), h("main", { key: 'ae117dd7c985f85cede7303f459c9a00b1981ba0' }, h("slot", { key: 'b3d43c7ab67a52d5bfe07c63da2197fa0db1b9f4' }), h("slot", { key: 'eac80e880d505ca915b96c01aedddd2685c41ad7', name: "overlay" }))))));
    }
    static get is() { return "ix-map-navigation"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["map-navigation.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["map-navigation.css"]
        };
    }
    static get properties() {
        return {
            "applicationName": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Application name"
                },
                "attribute": "application-name",
                "reflect": false
            },
            "navigationTitle": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Navigation title"
                },
                "attribute": "navigation-title",
                "reflect": false
            },
            "hideContextMenu": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Hide the sidebar context menu button when set to true"
                },
                "attribute": "hide-context-menu",
                "reflect": false,
                "defaultValue": "true"
            }
        };
    }
    static get states() {
        return {
            "isSidebarOpen": {},
            "hasContentHeader": {}
        };
    }
    static get events() {
        return [{
                "method": "navigationToggled",
                "name": "navigationToggled",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Navigation toggled"
                },
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                }
            }, {
                "method": "contextMenuClick",
                "name": "contextMenuClick",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Context menu clicked"
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }];
    }
    static get methods() {
        return {
            "toggleSidebar": {
                "complexType": {
                    "signature": "(show?: boolean) => Promise<void>",
                    "parameters": [{
                            "name": "show",
                            "type": "boolean",
                            "docs": "new visibility state"
                        }],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Change the visibility of the sidebar",
                    "tags": [{
                            "name": "param",
                            "text": "show new visibility state"
                        }, {
                            "name": "since",
                            "text": "1.6.0"
                        }]
                }
            },
            "openOverlay": {
                "complexType": {
                    "signature": "(name: string, component: HTMLElement, icon?: string, color?: string) => Promise<void>",
                    "parameters": [{
                            "name": "name",
                            "type": "string",
                            "docs": ""
                        }, {
                            "name": "component",
                            "type": "HTMLElement",
                            "docs": ""
                        }, {
                            "name": "icon",
                            "type": "string",
                            "docs": ""
                        }, {
                            "name": "color",
                            "type": "string",
                            "docs": ""
                        }],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        },
                        "HTMLElement": {
                            "location": "global",
                            "id": "global::HTMLElement"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Open a overlay inside content area",
                    "tags": [{
                            "name": "deprecated",
                            "text": "Will be removed in 2.0.0. Use slot based approach"
                        }, {
                            "name": "param",
                            "text": "name"
                        }, {
                            "name": "param",
                            "text": "component"
                        }, {
                            "name": "param",
                            "text": "icon"
                        }, {
                            "name": "param",
                            "text": "color"
                        }]
                }
            },
            "closeOverlay": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Close current shown overlay",
                    "tags": [{
                            "name": "deprecated",
                            "text": "Will be removed in 2.0.0. Use slot based approach"
                        }]
                }
            }
        };
    }
    static get elementRef() { return "hostElement"; }
}
MapNavigation.defaultTime = 150;
MapNavigation.slowTime = 500;
//# sourceMappingURL=map-navigation.js.map
