import { proxyCustomElement, HTMLElement, createEvent, h, Host } from '@stencil/core/internal/client';
import { f as flip, i as inline, s as shift, o as offset, a as autoUpdate, c as computePosition } from './floating-ui.dom.esm.js';
import { d as dropdownController, h as hasDropdownItemWrapperImplemented } from './dropdown-controller.js';
import { f as findElement } from './find-element.js';

/*
 * SPDX-FileCopyrightText: 2024 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
const addDisposableEventListener = (element, eventType, callback) => {
    element.addEventListener(eventType, callback);
    return () => {
        element.removeEventListener(eventType, callback);
    };
};

/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
class ArrowFocusController {
    constructor(items, container, callback) {
        this.wrap = false;
        this.items = items;
        this.container = container;
        this.callback = callback;
        this.keyListener = addDisposableEventListener(container, 'keydown', (e) => this.onKeyDown(e));
    }
    getActiveIndex() {
        if (!document.activeElement) {
            return -1;
        }
        return this.items.indexOf(document.activeElement);
    }
    onKeyDown(e) {
        var _a, _b, _c, _d;
        const activeIndex = this.getActiveIndex();
        if (activeIndex < 0) {
            return;
        }
        switch (e.key) {
            case 'ArrowDown':
                if (activeIndex < this.items.length - 1) {
                    e.preventDefault();
                    (_a = this.callback) === null || _a === void 0 ? void 0 : _a.call(this, activeIndex + 1);
                }
                else if (this.wrap) {
                    e.preventDefault();
                    (_b = this.callback) === null || _b === void 0 ? void 0 : _b.call(this, 0);
                }
                break;
            case 'ArrowUp':
                {
                    if (activeIndex > 0) {
                        e.preventDefault();
                        (_c = this.callback) === null || _c === void 0 ? void 0 : _c.call(this, activeIndex - 1);
                    }
                    else if (this.wrap && activeIndex === 0) {
                        e.preventDefault();
                        (_d = this.callback) === null || _d === void 0 ? void 0 : _d.call(this, this.items.length - 1);
                    }
                }
                break;
        }
    }
    disconnect() {
        if (this.keyListener) {
            this.keyListener();
        }
        this.container = undefined;
        this.callback = undefined;
    }
}

const dropdownCss = ":host{background-color:var(--theme-color-2);border-radius:var(--theme-default-border-radius);min-width:0px;z-index:var(--theme-z-index-dropdown);box-shadow:var(--theme-shadow-4);padding:0.25rem 0px}:host *,:host *::after,:host *::before{box-sizing:border-box}:host ::-webkit-scrollbar-button{display:none}@-moz-document url-prefix(){:host *{scrollbar-color:var(--theme-scrollbar-thumb--background) var(--theme-scrollbar-track--background);scrollbar-width:thin}}:host ::-webkit-scrollbar{width:0.5rem;height:0.5rem}:host ::-webkit-scrollbar-track{border-radius:5px;background:var(--theme-scrollbar-track--background)}:host ::-webkit-scrollbar-track:hover{background:var(--theme-scrollbar-track--background--hover)}:host ::-webkit-scrollbar-thumb{border-radius:5px;background:var(--theme-scrollbar-thumb--background)}:host ::-webkit-scrollbar-thumb:hover{background:var(--theme-scrollbar-thumb--background--hover)}:host ::-webkit-scrollbar-corner{display:none}:host .dropdown-header{display:flex;align-items:center;height:2.5rem;color:var(--theme-menu-header--color);padding:0 1rem}:host(.overflow){max-height:50vh;overflow-y:auto}:host(:not(.show)){display:none}";
const IxDropdownStyle0 = dropdownCss;

let sequenceId = 0;
const Dropdown = /*@__PURE__*/ proxyCustomElement(class Dropdown extends HTMLElement {
    constructor() {
        super();
        this.__registerHost();
        this.__attachShadow();
        this.showChanged = createEvent(this, "showChanged", 7);
        this.localUId = `dropdown-${sequenceId++}`;
        this.assignedSubmenu = [];
        this.itemObserver = new MutationObserver(() => {
            if (this.arrowFocusController) {
                this.arrowFocusController.items = this.dropdownItems;
            }
        });
        this.suppressAutomaticPlacement = false;
        this.show = false;
        this.trigger = undefined;
        this.anchor = undefined;
        this.closeBehavior = 'both';
        this.placement = 'bottom-start';
        this.positioningStrategy = 'fixed';
        this.header = undefined;
        this.offset = undefined;
        this.overwriteDropdownStyle = undefined;
        this.discoverAllSubmenus = false;
        this.ignoreRelatedSubmenu = false;
        this.suppressOverflowBehavior = false;
    }
    connectedCallback() {
        dropdownController.connected(this);
        if (this.trigger != undefined) {
            this.registerListener(this.trigger);
        }
    }
    cacheSubmenuId(event) {
        event.stopImmediatePropagation();
        event.preventDefault();
        const { detail } = event;
        if (this.assignedSubmenu.indexOf(detail) === -1) {
            this.assignedSubmenu.push(detail);
        }
    }
    disconnectedCallback() {
        var _a;
        dropdownController.dismiss(this);
        dropdownController.disconnected(this);
        if (this.arrowFocusController) {
            (_a = this.arrowFocusController) === null || _a === void 0 ? void 0 : _a.disconnect();
            this.arrowFocusController = undefined;
        }
        if (this.itemObserver) {
            this.itemObserver.disconnect();
            this.itemObserver = undefined;
        }
        if (this.disposeClickListener) {
            this.disposeClickListener();
            this.disposeClickListener = undefined;
        }
        if (this.disposeKeyListener) {
            this.disposeKeyListener();
            this.disposeKeyListener = undefined;
        }
        if (this.autoUpdateCleanup) {
            this.autoUpdateCleanup();
            this.autoUpdateCleanup = undefined;
        }
    }
    getAssignedSubmenuIds() {
        return this.assignedSubmenu;
    }
    isPresent() {
        return this.show;
    }
    present() {
        this.show = true;
    }
    dismiss() {
        this.show = false;
    }
    getId() {
        return this.localUId;
    }
    willDismiss() {
        const { defaultPrevented } = this.showChanged.emit(false);
        return !defaultPrevented;
    }
    willPresent() {
        const { defaultPrevented } = this.showChanged.emit(true);
        return !defaultPrevented;
    }
    get dropdownItems() {
        return Array.from(this.hostElement.querySelectorAll('ix-dropdown-item'));
    }
    get slotElement() {
        return this.hostElement.shadowRoot.querySelector('slot');
    }
    addEventListenersFor() {
        var _a, _b, _c;
        (_a = this.disposeClickListener) === null || _a === void 0 ? void 0 : _a.call(this);
        (_b = this.disposeKeyListener) === null || _b === void 0 ? void 0 : _b.call(this);
        const toggleController = () => {
            if (!this.isPresent()) {
                dropdownController.present(this);
            }
            else {
                dropdownController.dismiss(this);
            }
            dropdownController.dismissOthers(this.getId());
        };
        if (!this.triggerElement) {
            return;
        }
        this.disposeClickListener = addDisposableEventListener(this.triggerElement, 'click', (event) => {
            if (!event.defaultPrevented) {
                toggleController();
            }
        });
        (_c = this.triggerElement) === null || _c === void 0 ? void 0 : _c.setAttribute('data-ix-dropdown-trigger', this.localUId);
    }
    /** @internal */
    async discoverSubmenu() {
        var _a;
        (_a = this.triggerElement) === null || _a === void 0 ? void 0 : _a.dispatchEvent(new CustomEvent('ix-assign-sub-menu', {
            bubbles: true,
            composed: true,
            cancelable: true,
            detail: this.localUId,
        }));
    }
    registerKeyListener() {
        if (!this.triggerElement) {
            return;
        }
        this.disposeKeyListener = addDisposableEventListener(this.triggerElement, 'keydown', ((event) => {
            if (event.key !== 'ArrowDown') {
                return;
            }
            if (document.activeElement !== this.triggerElement) {
                return;
            }
            dropdownController.present(this);
            setTimeout(() => {
                this.focusDropdownItem(0);
            });
        }));
    }
    async registerListener(element) {
        this.triggerElement = await this.resolveElement(element);
        if (this.triggerElement) {
            this.addEventListenersFor();
            this.discoverSubmenu();
        }
    }
    async resolveElement(element) {
        const el = await findElement(element);
        return this.checkForSubmenuAnchor(el);
    }
    async checkForSubmenuAnchor(element) {
        if (!element) {
            return undefined;
        }
        if (hasDropdownItemWrapperImplemented(element)) {
            const dropdownItem = await element.getDropdownItemElement();
            dropdownItem.isSubMenu = true;
            this.hostElement.style.zIndex = `var(--theme-z-index-dropdown)`;
        }
        if (element.tagName === 'IX-DROPDOWN-ITEM') {
            element.isSubMenu = true;
            this.hostElement.style.zIndex = `var(--theme-z-index-dropdown)`;
        }
        return element;
    }
    async resolveAnchorElement() {
        if (this.anchor) {
            this.anchorElement = await this.resolveElement(this.anchor);
        }
        else if (this.trigger) {
            this.anchorElement = await this.resolveElement(this.trigger);
        }
    }
    async changedShow(newShow) {
        var _a, _b, _c, _d;
        if (newShow) {
            await this.resolveAnchorElement();
            if (this.anchorElement) {
                this.applyDropdownPosition();
            }
            this.arrowFocusController = new ArrowFocusController(this.dropdownItems, this.hostElement, (index) => this.focusDropdownItem(index));
            (_a = this.itemObserver) === null || _a === void 0 ? void 0 : _a.observe(this.hostElement, {
                childList: true,
                subtree: true,
            });
            this.registerKeyListener();
        }
        else {
            this.destroyAutoUpdate();
            (_b = this.arrowFocusController) === null || _b === void 0 ? void 0 : _b.disconnect();
            (_c = this.itemObserver) === null || _c === void 0 ? void 0 : _c.disconnect();
            (_d = this.disposeKeyListener) === null || _d === void 0 ? void 0 : _d.call(this);
        }
    }
    changedTrigger(newTriggerValue) {
        this.registerListener(newTriggerValue);
    }
    destroyAutoUpdate() {
        if (this.autoUpdateCleanup) {
            this.autoUpdateCleanup();
            this.autoUpdateCleanup = undefined;
        }
    }
    isAnchorSubmenu() {
        var _a;
        if (!hasDropdownItemWrapperImplemented(this.anchorElement)) {
            // Is no official dropdown-item, but check if any dropdown-item
            // is placed somewhere up the DOM
            return !!((_a = this.anchorElement) === null || _a === void 0 ? void 0 : _a.closest('ix-dropdown-item'));
        }
        return true;
    }
    async applyDropdownPosition() {
        var _a, _b;
        if (!this.show) {
            return;
        }
        if (!this.anchorElement) {
            return;
        }
        const isSubmenu = this.isAnchorSubmenu();
        let positionConfig = {
            strategy: this.positioningStrategy,
            middleware: [],
        };
        if (!this.suppressAutomaticPlacement) {
            (_a = positionConfig.middleware) === null || _a === void 0 ? void 0 : _a.push(flip({ fallbackStrategy: 'initialPlacement' }));
        }
        positionConfig.placement = isSubmenu ? 'right-start' : this.placement;
        positionConfig.middleware = [
            ...(((_b = positionConfig.middleware) === null || _b === void 0 ? void 0 : _b.filter(Boolean)) || []),
            inline(),
            shift(),
        ];
        if (this.offset) {
            positionConfig.middleware.push(offset(this.offset));
        }
        this.destroyAutoUpdate();
        if (!this.anchorElement) {
            return;
        }
        this.autoUpdateCleanup = autoUpdate(this.anchorElement, this.hostElement, async () => {
            if (this.anchorElement) {
                const computeResponse = await computePosition(this.anchorElement, this.hostElement, positionConfig);
                Object.assign(this.hostElement.style, {
                    top: '0',
                    left: '0',
                    transform: `translate(${Math.round(computeResponse.x)}px,${Math.round(computeResponse.y)}px)`,
                });
            }
            if (this.overwriteDropdownStyle) {
                const overwriteStyle = await this.overwriteDropdownStyle({
                    dropdownRef: this.hostElement,
                    triggerRef: this.triggerElement,
                });
                Object.assign(this.hostElement.style, overwriteStyle);
            }
        }, {
            ancestorResize: true,
            ancestorScroll: true,
            elementResize: true,
        });
    }
    focusDropdownItem(index) {
        requestAnimationFrame(() => {
            var _a, _b;
            const button = (_b = (_a = this.dropdownItems[index]) === null || _a === void 0 ? void 0 : _a.shadowRoot) === null || _b === void 0 ? void 0 : _b.querySelector('button');
            if (button) {
                button.focus();
            }
        });
    }
    async componentDidLoad() {
        if (!this.trigger) {
            return;
        }
        this.changedTrigger(this.trigger);
    }
    async componentDidRender() {
        await this.applyDropdownPosition();
        await this.resolveAnchorElement();
    }
    isTriggerElement(element) {
        const trigger = !!element.hasAttribute('data-ix-dropdown-trigger');
        return trigger;
    }
    onDropdownClick(event) {
        const target = dropdownController.pathIncludesTrigger(event.composedPath());
        if (target) {
            if (target !== this.triggerElement) {
                event.preventDefault();
            }
            if (this.isTriggerElement(target)) {
                if (this.closeBehavior === 'outside') {
                    event.preventDefault();
                }
                return;
            }
        }
        if (!event.defaultPrevented &&
            (this.closeBehavior === 'inside' || this.closeBehavior === 'both')) {
            dropdownController.dismissAll([this.getId()], this.ignoreRelatedSubmenu);
            return;
        }
        dropdownController.dismissOthers(this.getId());
    }
    /**
     * Update position of dropdown
     */
    async updatePosition() {
        this.applyDropdownPosition();
    }
    render() {
        return (h(Host, { key: 'f0cb7a478c8b8c066a7c163e83cbc101e6f3a5ec', "data-ix-dropdown": this.localUId, class: {
                'dropdown-menu': true,
                show: this.show,
                overflow: !this.suppressOverflowBehavior,
            }, style: {
                margin: '0',
                minWidth: '0px',
                position: this.positioningStrategy,
            }, role: "list", onClick: (event) => this.onDropdownClick(event) }, h("div", { key: '5431d09d3eb6b08f357ef7202617223ab3b347bf', style: { display: 'contents' } }, this.header && h("div", { key: 'b771273fe7292d3f253690b17955f13bb95ef56a', class: "dropdown-header" }, this.header), this.show && h("slot", { key: '0150080d095cce23717962df423ea6b8a137841e' }))));
    }
    get hostElement() { return this; }
    static get watchers() { return {
        "show": ["changedShow"],
        "trigger": ["changedTrigger"]
    }; }
    static get style() { return IxDropdownStyle0; }
}, [1, "ix-dropdown", {
        "suppressAutomaticPlacement": [4, "suppress-automatic-placement"],
        "show": [1540],
        "trigger": [1],
        "anchor": [1],
        "closeBehavior": [8, "close-behavior"],
        "placement": [1],
        "positioningStrategy": [1, "positioning-strategy"],
        "header": [1],
        "offset": [16],
        "overwriteDropdownStyle": [16],
        "discoverAllSubmenus": [4, "discover-all-submenus"],
        "ignoreRelatedSubmenu": [4, "ignore-related-submenu"],
        "suppressOverflowBehavior": [4, "suppress-overflow-behavior"],
        "discoverSubmenu": [64],
        "updatePosition": [64]
    }, [[0, "ix-assign-sub-menu", "cacheSubmenuId"]], {
        "show": ["changedShow"],
        "trigger": ["changedTrigger"]
    }]);
function defineCustomElement() {
    if (typeof customElements === "undefined") {
        return;
    }
    const components = ["ix-dropdown"];
    components.forEach(tagName => { switch (tagName) {
        case "ix-dropdown":
            if (!customElements.get(tagName)) {
                customElements.define(tagName, Dropdown);
            }
            break;
    } });
}

export { ArrowFocusController as A, Dropdown as D, addDisposableEventListener as a, defineCustomElement as d };

//# sourceMappingURL=dropdown.js.map