/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { Fragment, h, Host, readTask, } from "@stencil/core";
import { showAppSwitch } from "../utils/app-switch";
import { applicationLayoutService } from "../utils/application-layout";
import { ApplicationLayoutContext, } from "../utils/application-layout/context";
import { useContextConsumer } from "../utils/context";
import { menuController } from "../utils/menu-service/menu-service";
import { hasSlottedElements } from "../utils/shadow-dom";
/**
 * @slot logo - Location of the logo
 */
export class ApplicationHeader {
    constructor() {
        this.name = undefined;
        this.showMenu = false;
        this.breakpoint = 'lg';
        this.menuExpanded = false;
        this.suppressResponsive = false;
        this.hasSlottedElements = false;
        this.applicationLayoutContext = undefined;
    }
    get contentBackground() {
        return this.hostElement.shadowRoot.querySelector('.dropdown-content');
    }
    componentWillLoad() {
        useContextConsumer(this.hostElement, ApplicationLayoutContext, (ctx) => {
            if ((ctx === null || ctx === void 0 ? void 0 : ctx.host) === 'map-navigation') {
                this.suppressResponsive = true;
                this.breakpoint = 'md';
                return;
            }
            this.breakpoint = applicationLayoutService.breakpoint;
            this.applicationLayoutContext = ctx;
            this.tryUpdateAppSwitch();
        }, true);
        this.menuDisposable = menuController.expandChange.on((show) => {
            this.menuExpanded = show;
        });
        this.modeDisposable = applicationLayoutService.onChange.on((mode) => {
            if (this.suppressResponsive) {
                return;
            }
            this.breakpoint = mode;
        });
        this.updateIsSlottedContent();
    }
    componentDidLoad() {
        this.attachSiemensLogoIfLoaded();
    }
    disconnectedCallback() {
        var _a, _b;
        (_a = this.menuDisposable) === null || _a === void 0 ? void 0 : _a.dispose();
        (_b = this.modeDisposable) === null || _b === void 0 ? void 0 : _b.dispose();
    }
    watchApplicationLayoutContext() {
        if (this.applicationLayoutContext) {
            this.showMenu = false;
        }
    }
    watchSuppressResponsive() {
        this.breakpoint = 'md';
    }
    isLogoSlotted() {
        const slotElement = this.hostElement.shadowRoot.querySelector('slot[name="logo"]');
        const nodes = slotElement.assignedNodes({
            flatten: true,
        });
        return nodes.length !== 0;
    }
    async attachSiemensLogoIfLoaded() {
        await window.customElements.whenDefined('ix-siemens-logo');
        const logoElement = document.createElement('ix-siemens-logo');
        if (!this.isLogoSlotted()) {
            this.hostElement.shadowRoot
                .querySelector('.logo')
                .appendChild(logoElement);
        }
    }
    async onMenuClick() {
        if (this.applicationLayoutContext) {
            menuController.toggle();
        }
        else {
            this.menuExpanded = !this.menuExpanded;
        }
        this.menuToggle.emit(this.menuExpanded);
    }
    resolveContextMenuButton() {
        return new Promise((resolve) => readTask(() => resolve(this.hostElement.shadowRoot.querySelector('[data-context-menu]'))));
    }
    tryUpdateAppSwitch() {
        var _a;
        if (!this.callbackUpdateAppSwitchModal) {
            return;
        }
        this.callbackUpdateAppSwitchModal((_a = this.applicationLayoutContext) === null || _a === void 0 ? void 0 : _a.appSwitchConfig);
    }
    async showAppSwitch() {
        var _a;
        this.callbackUpdateAppSwitchModal = await showAppSwitch((_a = this.applicationLayoutContext) === null || _a === void 0 ? void 0 : _a.appSwitchConfig);
    }
    updateIsSlottedContent() {
        const slotElement = this.hostElement.shadowRoot.querySelector('.content slot');
        this.hasSlottedElements = hasSlottedElements(slotElement);
    }
    onContentBgClick(e) {
        if (e.target === this.contentBackground) {
            e.preventDefault();
        }
    }
    render() {
        var _a;
        const hasApplicationContextAvailable = !!this.applicationLayoutContext;
        const showMenuByApplicationFrame = this.breakpoint === 'sm' &&
            this.suppressResponsive === false &&
            hasApplicationContextAvailable;
        const showApplicationSwitch = ((_a = this.applicationLayoutContext) === null || _a === void 0 ? void 0 : _a.appSwitchConfig) &&
            this.breakpoint !== 'sm' &&
            this.suppressResponsive === false;
        return (h(Host, { key: '1f987dd8ac407207a23ac0821945ba9ad3d47e87', class: {
                [`breakpoint-${this.breakpoint}`]: true,
            }, slot: "application-header" }, (this.showMenu || showMenuByApplicationFrame) && (h("ix-menu-expand-icon", { key: '60c7aa44d2b294fff9cb51ecaf291d70a33bafbd', onClick: () => this.onMenuClick(), expanded: this.menuExpanded })), showApplicationSwitch && (h("ix-icon-button", { key: 'e500c3a98790345840633b8dd0f056fac198f802', onClick: () => this.showAppSwitch(), icon: "apps", ghost: true, class: "app-switch" })), h("div", { key: 'adba1c9a08be32a07f31a60d42159f996d26a87b', class: { logo: true } }, h("slot", { key: '3848321805f42653d8a9c066c8f237c8ec3497bb', name: "logo" })), h("ix-typography", { key: '9c312426ee3c2554a9a66aff6b158d36969cefea', format: "body-lg", class: "name" }, this.name), h("div", { key: '319969ceeb812f9b18b5ad3f374e16f454c6daf2', class: "content" }, this.breakpoint === 'sm' ? (h(Fragment, null, h("ix-icon-button", { class: {
                ['context-menu']: true,
                ['context-menu-visible']: this.hasSlottedElements,
            }, "data-context-menu": true, icon: "more-menu", ghost: true }), h("ix-dropdown", { "data-overflow-dropdown": true, class: "dropdown", discoverAllSubmenus: true, trigger: this.resolveContextMenuButton() }, h("div", { class: "dropdown-content", onClick: (e) => this.onContentBgClick(e) }, h("slot", { onSlotchange: () => this.updateIsSlottedContent() }))))) : (h("slot", { onSlotchange: () => this.updateIsSlottedContent() })), h("slot", { key: 'dd03dd49866605c06d6daf09d2ee72692666ee68', name: "ix-application-header-avatar" }))));
    }
    static get is() { return "ix-application-header"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["application-header.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["application-header.css"]
        };
    }
    static get properties() {
        return {
            "name": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Application name"
                },
                "attribute": "name",
                "reflect": false
            },
            "showMenu": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "since",
                            "text": "2.5.0"
                        }],
                    "text": "Controls the visibility of the menu toggle button based on the context of the application header.\n\nWhen the application header is utilized outside the application frame, the menu toggle button is displayed.\nConversely, if the header is within the application frame, this property is ineffective."
                },
                "attribute": "show-menu",
                "reflect": false,
                "defaultValue": "false"
            }
        };
    }
    static get states() {
        return {
            "breakpoint": {},
            "menuExpanded": {},
            "suppressResponsive": {},
            "hasSlottedElements": {},
            "applicationLayoutContext": {}
        };
    }
    static get events() {
        return [{
                "method": "menuToggle",
                "name": "menuToggle",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "since",
                            "text": "2.5.0"
                        }],
                    "text": "Event emitted when the menu toggle button is clicked"
                },
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                }
            }];
    }
    static get elementRef() { return "hostElement"; }
    static get watchers() {
        return [{
                "propName": "applicationLayoutContext",
                "methodName": "watchApplicationLayoutContext"
            }, {
                "propName": "suppressResponsive",
                "methodName": "watchSuppressResponsive"
            }];
    }
}
//# sourceMappingURL=application-header.js.map
