/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { h, Host, } from "@stencil/core";
import { BaseButton } from "../button/base-button";
import { InputState } from "./input-state";
import { LogicalFilterOperator } from "./logical-filter-operator";
import { makeRef } from "../utils/make-ref";
import { addDisposableEventListener, } from "../utils/disposable-event-listener";
import { a11yHostAttributes } from "../utils/a11y";
export class CategoryFilter {
    constructor() {
        this.ID_CUSTOM_FILTER_VALUE = 'CW_CUSTOM_FILTER_VALUE';
        this.textInput = makeRef();
        this.showDropdown = false;
        this.hasFocus = false;
        this.categoryLogicalOperator = LogicalFilterOperator.EQUAL;
        this.inputValue = '';
        this.category = '';
        this.filterTokens = [];
        this.disabled = false;
        this.readonly = false;
        this.filterState = undefined;
        this.placeholder = undefined;
        this.categories = undefined;
        this.nonSelectableCategories = {};
        this.suggestions = undefined;
        this.icon = 'search';
        this.hideIcon = false;
        this.staticOperator = undefined;
        this.repeatCategories = true;
        this.tmpDisableScrollIntoView = true;
        this.labelCategories = 'Categories';
        this.i18nPlainText = 'Filter by text';
    }
    get dropdown() {
        return this.hostElement.shadowRoot.querySelector('ix-dropdown');
    }
    watchFilterState(newValue) {
        this.setFilterState(newValue);
    }
    preventDefault(e) {
        e.preventDefault();
    }
    onFocusIn() {
        this.hasFocus = true;
    }
    onFocusOut() {
        this.hasFocus = false;
    }
    onInput() {
        var _a, _b, _c;
        this.inputValue = ((_b = (_a = this.textInput) === null || _a === void 0 ? void 0 : _a.current) === null || _b === void 0 ? void 0 : _b.value) || '';
        const inputState = new InputState(this.inputValue, this.category);
        this.inputChanged.emit(inputState);
        if (!((_c = this.dropdown) === null || _c === void 0 ? void 0 : _c.show)) {
            this.openDropdown();
        }
    }
    componentWillLoad() {
        this.a11yAttributes = a11yHostAttributes(this.hostElement);
    }
    componentDidLoad() {
        var _a;
        setTimeout(() => {
            if (this.filterState !== undefined) {
                this.setFilterState(this.filterState);
            }
        });
        if (this.formElement !== undefined) {
            this.formKeyDownListener = addDisposableEventListener(this.formElement, 'keydown', ((e) => this.handleFormElementKeyDown(e)));
            this.preventDefaultListener = addDisposableEventListener(this.formElement, 'submit', this.preventDefault);
        }
        if (((_a = this.textInput) === null || _a === void 0 ? void 0 : _a.current) == null) {
            console.warn('ix-category-filter - unable to add event listeners to native input element');
            return;
        }
        this.inputKeyDownListener = addDisposableEventListener(this.textInput.current, 'keydown', ((e) => this.handleInputElementKeyDown(e)));
        this.focusInListener = addDisposableEventListener(this.textInput.current, 'focusin', () => this.onFocusIn());
        this.focusOutListener = addDisposableEventListener(this.textInput.current, 'focusout', () => this.onFocusOut());
        this.inputListener = addDisposableEventListener(this.textInput.current, 'input', () => this.onInput());
    }
    setFilterState(state) {
        this.filterTokens = [];
        for (const token of state.tokens) {
            this.addToken(token, this.ID_CUSTOM_FILTER_VALUE, this.categoryLogicalOperator, false);
        }
        for (const category of state.categories) {
            this.addToken(category.value, category.id, category.operator, false);
        }
        this.emitFilterEvent();
    }
    closeDropdown() {
        if (this.disabled || this.readonly) {
            return;
        }
        if (this.dropdown) {
            this.dropdown.show = false;
        }
    }
    openDropdown() {
        if (this.disabled || this.readonly) {
            return;
        }
        if (this.dropdown) {
            this.dropdown.show = true;
        }
    }
    handleFormElementKeyDown(e) {
        var _a;
        switch (e.code) {
            case 'Enter':
            case 'NumpadEnter':
                if (!((_a = document.activeElement) === null || _a === void 0 ? void 0 : _a.classList.contains('dropdown-item'))) {
                    return;
                }
                const token = document.activeElement.getAttribute('data-id');
                if (token === null) {
                    break;
                }
                if (this.hasCategorySelection()) {
                    if (this.category !== '') {
                        this.addToken(token, this.category);
                    }
                    else if (document.activeElement.classList.contains('category-item-id')) {
                        this.selectCategory(token);
                    }
                }
                else {
                    this.addToken(token);
                }
                e.preventDefault();
                break;
            case 'ArrowUp':
                this.focusPreviousItem();
                e.preventDefault();
                break;
            case 'ArrowDown':
                this.showDropdown = true;
                this.focusNextItem();
                e.preventDefault();
                break;
            case 'Escape':
                this.closeDropdown();
                break;
        }
    }
    focusPreviousItem() {
        var _a;
        const sibling = (_a = document.activeElement) === null || _a === void 0 ? void 0 : _a.previousSibling;
        if (sibling instanceof HTMLElement) {
            sibling.focus();
        }
    }
    focusNextItem() {
        var _a;
        const sibling = (_a = document.activeElement) === null || _a === void 0 ? void 0 : _a.nextSibling;
        if (sibling instanceof HTMLElement) {
            sibling.focus();
        }
    }
    focusElement(selector) {
        const item = this.hostElement.shadowRoot.querySelector(selector);
        if (item instanceof HTMLElement) {
            item.focus();
            return true;
        }
        return false;
    }
    onArrowDown(e) {
        var _a;
        const baseSelector = `.category-item-${this.category !== '' ? 'value' : 'id'}`;
        const fallbackSelector = '.category-item';
        if (this.focusElement(baseSelector)) {
            e.stopPropagation();
            return;
        }
        if (((_a = this.suggestions) === null || _a === void 0 ? void 0 : _a.length) && this.focusElement(fallbackSelector)) {
            e.stopPropagation();
        }
    }
    handleInputElementKeyDown(e) {
        var _a, _b;
        switch (e.code) {
            case 'ArrowDown': {
                this.onArrowDown(e);
                break;
            }
            case 'Backspace':
                if (((_b = (_a = this.textInput) === null || _a === void 0 ? void 0 : _a.current) === null || _b === void 0 ? void 0 : _b.value) !== '') {
                    return;
                }
                if (this.category !== '') {
                    this.category = '';
                    return;
                }
                const tokenCount = this.filterTokens.length;
                if (tokenCount > 0) {
                    this.removeToken(tokenCount - 1);
                }
                break;
            case 'Enter':
            case 'NumpadEnter':
                this.addToken(this.inputValue, this.category || this.ID_CUSTOM_FILTER_VALUE);
                e.preventDefault();
                break;
        }
    }
    emitFilterEvent() {
        const tokens = this.filterTokens
            .filter((item) => item.id === this.ID_CUSTOM_FILTER_VALUE)
            .map((item) => item.value);
        const categories = this.filterTokens.filter((item) => item.id !== this.ID_CUSTOM_FILTER_VALUE);
        const filterState = {
            tokens,
            categories,
        };
        this.filterChanged.emit(filterState);
    }
    addToken(token, category = this.ID_CUSTOM_FILTER_VALUE, operator = this.categoryLogicalOperator, emitEvent = true) {
        var _a, _b, _c;
        if (token === undefined || token === null) {
            return;
        }
        const newToken = token.trim();
        if (newToken === '') {
            return;
        }
        if (this.hasToken(newToken)) {
            return;
        }
        const pair = { id: category, value: newToken, operator };
        this.filterTokens = [...this.filterTokens, pair];
        if ((_a = this.textInput) === null || _a === void 0 ? void 0 : _a.current) {
            this.textInput.current.value = '';
        }
        this.inputValue = '';
        this.categoryLogicalOperator = LogicalFilterOperator.EQUAL;
        if (this.category !== '') {
            this.category = '';
        }
        this.isScrollStateDirty = true;
        (_c = (_b = this.textInput) === null || _b === void 0 ? void 0 : _b.current) === null || _c === void 0 ? void 0 : _c.focus();
        if (emitEvent) {
            this.emitFilterEvent();
        }
    }
    removeToken(index) {
        this.filterTokens = this.filterTokens.filter((_, i) => i !== index);
        this.emitFilterEvent();
    }
    getCategoryIds() {
        const ids = [];
        for (const id in this.categories) {
            if (Object.prototype.hasOwnProperty.call(this.categories, id)) {
                ids.push(id);
            }
        }
        return ids;
    }
    selectCategory(category) {
        var _a, _b, _c;
        this.category = category;
        if ((_a = this.textInput) === null || _a === void 0 ? void 0 : _a.current) {
            this.textInput.current.value = '';
        }
        this.inputValue = '';
        (_c = (_b = this.textInput) === null || _b === void 0 ? void 0 : _b.current) === null || _c === void 0 ? void 0 : _c.focus();
        this.categoryChanged.emit(category);
    }
    resetFilter(e) {
        const { defaultPrevented } = this.filterCleared.emit();
        if (defaultPrevented) {
            return;
        }
        e.stopPropagation();
        this.closeDropdown();
        this.filterTokens = [];
        if (this.category) {
            this.category = '';
            this.categoryChanged.emit(undefined);
        }
        this.emitFilterEvent();
    }
    filterMultiples(value) {
        if (this.repeatCategories) {
            return true;
        }
        const isCategoryAlreadySet = this.filterTokens.find((token) => token.id === value);
        return !isCategoryAlreadySet;
    }
    hasToken(token) {
        return this.filterTokens.some((filterToken) => {
            const hasSameValue = filterToken.value === token;
            if (!hasSameValue) {
                return false;
            }
            if (this.category !== '') {
                return this.category === filterToken.id;
            }
            if (filterToken.id) {
                return filterToken.id === this.ID_CUSTOM_FILTER_VALUE;
            }
            return hasSameValue;
        });
    }
    filterDuplicateTokens(value) {
        return !this.hasToken(value);
    }
    filterByInput(value) {
        if (this.inputValue === undefined || this.inputValue === '') {
            return true;
        }
        return value.toLowerCase().indexOf(this.inputValue.toLowerCase()) !== -1;
    }
    toggleCategoryOperator() {
        switch (this.categoryLogicalOperator) {
            case LogicalFilterOperator.EQUAL:
                this.categoryLogicalOperator = LogicalFilterOperator.NOT_EQUAL;
                break;
            case LogicalFilterOperator.NOT_EQUAL:
                this.categoryLogicalOperator = LogicalFilterOperator.EQUAL;
                break;
        }
    }
    getFilterChipLabel(value) {
        var _a, _b, _c, _d, _e;
        if (value.id === this.ID_CUSTOM_FILTER_VALUE) {
            return value.value;
        }
        const operatorString = value.operator === LogicalFilterOperator.EQUAL ? '=' : '!=';
        const label = (_e = (_c = (_b = (_a = this.categories) === null || _a === void 0 ? void 0 : _a[value.id]) === null || _b === void 0 ? void 0 : _b.label) !== null && _c !== void 0 ? _c : (_d = this.nonSelectableCategories) === null || _d === void 0 ? void 0 : _d[value.id]) !== null && _e !== void 0 ? _e : value.id;
        return `${label} ${operatorString} ${value.value}`;
    }
    getFilteredSuggestions() {
        var _a, _b;
        if (!((_a = this.suggestions) === null || _a === void 0 ? void 0 : _a.length)) {
            return [];
        }
        return (_b = this.suggestions) === null || _b === void 0 ? void 0 : _b.filter((value) => this.filterByInput(value)).filter((value) => this.filterDuplicateTokens(value));
    }
    hasCategorySelection() {
        return this.categories !== undefined;
    }
    renderPlainSuggestions() {
        return (h("div", { class: "dropdown-item-container" }, this.getFilteredSuggestions().map((suggestion) => (h("button", { class: "dropdown-item", "data-id": suggestion, onClick: () => this.addToken(suggestion), key: suggestion, title: suggestion }, suggestion)))));
    }
    renderOperatorButton() {
        if (this.staticOperator) {
            return '';
        }
        const params = {
            type: 'button',
            variant: 'secondary',
            outline: false,
            ghost: true,
            iconOnly: true,
            iconOval: false,
            selected: false,
            disabled: this.disabled || this.staticOperator !== undefined,
            loading: false,
            icon: '',
            onClick: (e) => {
                e.stopPropagation();
                this.toggleCategoryOperator();
            },
            extraClasses: {
                'btn-icon-32': true,
                'btn-toggle-operator': true,
            },
        };
        return (h(BaseButton, Object.assign({}, params), this.categoryLogicalOperator === LogicalFilterOperator.NOT_EQUAL
            ? '='
            : '!='));
    }
    getFilterOperatorString() {
        let operator;
        if (this.staticOperator !== undefined) {
            operator = this.staticOperator;
        }
        else {
            operator = this.categoryLogicalOperator;
        }
        return `${operator === LogicalFilterOperator.EQUAL ? '=' : '!='} `;
    }
    renderCategoryValues() {
        var _a, _b;
        if (this.categories === undefined) {
            return;
        }
        return (h("div", { class: "dropdown-item-container" }, this.renderOperatorButton(), h("div", { class: "dropdown-header" }, (_a = this.categories[this.category]) === null || _a === void 0 ? void 0 : _a.label), (_b = this.categories[this.category]) === null || _b === void 0 ? void 0 :
            _b.options.filter((value) => this.filterByInput(value)).filter((value) => this.filterDuplicateTokens(value)).map((id) => (h("button", { class: "dropdown-item category-item-value", "data-id": id, title: id, key: id, onClick: (e) => {
                    e.preventDefault();
                    this.addToken(id, this.category);
                } }, `${this.getFilterOperatorString()} ${id}`)))));
    }
    renderDropdownContent() {
        if (this.hasCategorySelection()) {
            if (this.category !== '') {
                return this.renderCategoryValues();
            }
            else {
                return this.renderCategorySelection();
            }
        }
        else
            return this.renderPlainSuggestions();
    }
    renderCategorySelection() {
        var _a;
        return (h("div", { class: "dropdown-item-container" }, (_a = this.getCategoryIds()) === null || _a === void 0 ? void 0 : _a.filter((id) => this.categories && this.filterByInput(this.categories[id].label)).filter((id) => this.filterMultiples(id)).map((id) => {
            var _a, _b, _c, _d;
            return (h("button", { class: "dropdown-item category-item category-item-id", "data-id": id, title: (_b = (_a = this.categories) === null || _a === void 0 ? void 0 : _a[id]) === null || _b === void 0 ? void 0 : _b.label, key: id, onClick: (e) => {
                    e.preventDefault();
                    this.selectCategory(id);
                }, tabindex: "0" }, (_d = (_c = this.categories) === null || _c === void 0 ? void 0 : _c[id]) === null || _d === void 0 ? void 0 : _d.label));
        })));
    }
    getDropdownHeader() {
        if (this.categories !== undefined) {
            if (this.category !== '') {
                return undefined;
            }
            else {
                return this.labelCategories;
            }
        }
        return this.i18nPlainText;
    }
    componentDidRender() {
        var _a, _b;
        if (this.isScrollStateDirty) {
            if (!this.tmpDisableScrollIntoView) {
                (_b = (_a = this.textInput) === null || _a === void 0 ? void 0 : _a.current) === null || _b === void 0 ? void 0 : _b.scrollIntoView();
            }
            this.isScrollStateDirty = false;
        }
    }
    disconnectedCallback() {
        var _a;
        if (this.preventDefaultListener) {
            this.preventDefaultListener();
        }
        if (this.formKeyDownListener) {
            this.formKeyDownListener();
        }
        if (this.inputKeyDownListener) {
            (_a = this.inputKeyDownListener) === null || _a === void 0 ? void 0 : _a.call(this);
        }
        if (this.focusInListener) {
            this.focusInListener();
        }
        if (this.focusOutListener) {
            this.focusOutListener();
        }
        if (this.inputListener) {
            this.inputListener();
        }
    }
    getResetButton() {
        return (h("ix-icon-button", { onClick: (e) => this.resetFilter(e), class: {
                'reset-button': true,
                'hide-reset-button': !this.filterTokens.length && this.category === '',
            }, ghost: true, oval: true, icon: 'clear', size: "16" }));
    }
    getIconColor() {
        if (this.disabled) {
            return 'color-componentn-1';
        }
        if (this.readonly) {
            return 'color-std-txt';
        }
        return 'color-primary';
    }
    render() {
        var _a, _b;
        return (h(Host, { key: 'b93cf0144501eec0bf4d09c7e9218f9b79c92412' }, h("form", { key: '9f54025b49b7e371f8b39ee189ef219cdddb10c1', ref: (el) => (this.formElement = el) }, h("div", { key: '068d7ad268dcd96e56a63c2793907caefe4d6072', "read-only": this.readonly, class: {
                'input-container': true,
                disabled: this.disabled,
                focus: this.hasFocus,
                readonly: this.readonly,
                'no-icon': this.hideIcon,
            } }, h("ix-icon", { key: '4f8e7941360a304b980429496103ebf0e1229f7c', color: this.getIconColor(), class: { 'd-none': this.hideIcon }, name: this.icon, size: "16" }), h("div", { key: 'd263342567b510f7cd22bca4baa92ba49e8cfc9f', class: "token-container" }, h("div", { key: 'a7233e969fd11966d6a658d778589a234ad629fa', class: "list-unstyled" }, this.filterTokens.map((value, index) => (h("span", { key: value.toString(), class: {
                animate__animated: true,
                animate__fadein: true,
            } }, h("ix-filter-chip", { disabled: this.disabled, readonly: this.readonly, onClick: (e) => e.stopPropagation(), onCloseClick: () => this.removeToken(index) }, this.getFilterChipLabel(value))))), this.categories === undefined ? ('') : (h("span", { class: {
                'category-preview': true,
                'd-none': this.category === '',
            } }, (_a = this.categories[this.category]) === null || _a === void 0 ? void 0 : _a.label)), h("input", Object.assign({ key: '049176b6ce4a85860039f23258cea7b2aacab9a9', class: {
                'text-input': true,
                'hide-placeholder': this.readonly || this.disabled || this.category !== '',
            }, autocomplete: "off", name: "category-filter-input", disabled: this.disabled, readonly: this.readonly, ref: this.textInput, type: "text", placeholder: this.placeholder }, this.a11yAttributes)))), !this.readonly && !this.disabled && this.getResetButton())), this.disabled || this.readonly ? ('') : (h("ix-dropdown", { show: this.showDropdown, closeBehavior: "outside", offset: { mainAxis: 2 }, anchor: (_b = this.textInput) === null || _b === void 0 ? void 0 : _b.waitForCurrent(), trigger: this.hostElement, header: this.getDropdownHeader() }, this.renderDropdownContent()))));
    }
    static get is() { return "ix-category-filter"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["category-filter.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["category-filter.css"]
        };
    }
    static get properties() {
        return {
            "disabled": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If true the filter will be in disabled state"
                },
                "attribute": "disabled",
                "reflect": false,
                "defaultValue": "false"
            },
            "readonly": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If true the filter will be in readonly mode"
                },
                "attribute": "readonly",
                "reflect": false,
                "defaultValue": "false"
            },
            "filterState": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "FilterState",
                    "resolved": "FilterState",
                    "references": {
                        "FilterState": {
                            "location": "import",
                            "path": "./filter-state",
                            "id": "src/components/category-filter/filter-state.ts::FilterState"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "A set of search criteria to populate the component with."
                }
            },
            "placeholder": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Placeholder text to be displayed in an empty input field."
                },
                "attribute": "placeholder",
                "reflect": false
            },
            "categories": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "{\n    [id: string]: {\n      label: string;\n      options: string[];\n    };\n  }",
                    "resolved": "{ [id: string]: { label: string; options: string[]; }; }",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Configuration object hash used to populate the dropdown menu for type-ahead and quick selection functionality.\nEach ID maps to an object with a label and an array of options to select from."
                }
            },
            "nonSelectableCategories": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "{\n    [id: string]: string;\n  }",
                    "resolved": "{ [id: string]: string; }",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "In certain use cases some categories may not be available for selection anymore.\nTo allow proper display of set filters with these categories this ID to label mapping can be populated.\n\nConfiguration object hash used to supply labels to the filter chips in the input field.\nEach ID maps to a string representing the label to display."
                },
                "defaultValue": "{}"
            },
            "suggestions": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "string[]",
                    "resolved": "string[]",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "A list of strings that will be supplied as type-ahead suggestions not tied to any categories."
                }
            },
            "icon": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The icon next to the actual text input\nDefaults to 'search'"
                },
                "attribute": "icon",
                "reflect": false,
                "defaultValue": "'search'"
            },
            "hideIcon": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Allows to hide the icon inside the text input.\nDefaults to false"
                },
                "attribute": "hide-icon",
                "reflect": false,
                "defaultValue": "false"
            },
            "staticOperator": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "LogicalFilterOperator",
                    "resolved": "LogicalFilterOperator.EQUAL | LogicalFilterOperator.NOT_EQUAL",
                    "references": {
                        "LogicalFilterOperator": {
                            "location": "import",
                            "path": "./logical-filter-operator",
                            "id": "src/components/category-filter/logical-filter-operator.ts::LogicalFilterOperator"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "since",
                            "text": "2.2.0"
                        }],
                    "text": "If set categories will always be filtered via the respective logical operator.\nToggling of the operator will not be available to the user."
                },
                "attribute": "static-operator",
                "reflect": false
            },
            "repeatCategories": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If set to true, allows that a single category can be set more than once.\nAn already set category will not appear in the category dropdown if set to false.\n\nDefaults to true"
                },
                "attribute": "repeat-categories",
                "reflect": false,
                "defaultValue": "true"
            },
            "tmpDisableScrollIntoView": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "For debugging purposes only!"
                        }],
                    "text": ""
                },
                "attribute": "tmp-disable-scroll-into-view",
                "reflect": false,
                "defaultValue": "true"
            },
            "labelCategories": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "i18n"
                },
                "attribute": "label-categories",
                "reflect": false,
                "defaultValue": "'Categories'"
            },
            "i18nPlainText": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "i18n"
                },
                "attribute": "i-1-8n-plain-text",
                "reflect": false,
                "defaultValue": "'Filter by text'"
            }
        };
    }
    static get states() {
        return {
            "showDropdown": {},
            "hasFocus": {},
            "categoryLogicalOperator": {},
            "inputValue": {},
            "category": {},
            "filterTokens": {}
        };
    }
    static get events() {
        return [{
                "method": "categoryChanged",
                "name": "categoryChanged",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Event dispatched whenever a category gets selected in the dropdown"
                },
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                }
            }, {
                "method": "inputChanged",
                "name": "inputChanged",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Event dispatched whenever the text input changes."
                },
                "complexType": {
                    "original": "InputState",
                    "resolved": "InputState",
                    "references": {
                        "InputState": {
                            "location": "import",
                            "path": "./input-state",
                            "id": "src/components/category-filter/input-state.ts::InputState"
                        }
                    }
                }
            }, {
                "method": "filterChanged",
                "name": "filterChanged",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Event dispatched whenever the filter state changes."
                },
                "complexType": {
                    "original": "FilterState",
                    "resolved": "FilterState",
                    "references": {
                        "FilterState": {
                            "location": "import",
                            "path": "./filter-state",
                            "id": "src/components/category-filter/filter-state.ts::FilterState"
                        }
                    }
                }
            }, {
                "method": "filterCleared",
                "name": "filterCleared",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Event dispatched whenever the filter gets cleared."
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }];
    }
    static get elementRef() { return "hostElement"; }
    static get watchers() {
        return [{
                "propName": "filterState",
                "methodName": "watchFilterState"
            }];
    }
}
//# sourceMappingURL=category-filter.js.map
