/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { autoUpdate, computePosition, flip, inline, offset, shift, } from "@floating-ui/dom";
import { h, Host, } from "@stencil/core";
import { ArrowFocusController } from "../utils/focus";
import { dropdownController, hasDropdownItemWrapperImplemented, } from "./dropdown-controller";
import { findElement } from "../utils/find-element";
import { addDisposableEventListener, } from "../utils/disposable-event-listener";
let sequenceId = 0;
export class Dropdown {
    constructor() {
        this.localUId = `dropdown-${sequenceId++}`;
        this.assignedSubmenu = [];
        this.itemObserver = new MutationObserver(() => {
            if (this.arrowFocusController) {
                this.arrowFocusController.items = this.dropdownItems;
            }
        });
        this.suppressAutomaticPlacement = false;
        this.show = false;
        this.trigger = undefined;
        this.anchor = undefined;
        this.closeBehavior = 'both';
        this.placement = 'bottom-start';
        this.positioningStrategy = 'fixed';
        this.header = undefined;
        this.offset = undefined;
        this.overwriteDropdownStyle = undefined;
        this.discoverAllSubmenus = false;
        this.ignoreRelatedSubmenu = false;
        this.suppressOverflowBehavior = false;
    }
    connectedCallback() {
        dropdownController.connected(this);
        if (this.trigger != undefined) {
            this.registerListener(this.trigger);
        }
    }
    cacheSubmenuId(event) {
        event.stopImmediatePropagation();
        event.preventDefault();
        const { detail } = event;
        if (this.assignedSubmenu.indexOf(detail) === -1) {
            this.assignedSubmenu.push(detail);
        }
    }
    disconnectedCallback() {
        var _a;
        dropdownController.dismiss(this);
        dropdownController.disconnected(this);
        if (this.arrowFocusController) {
            (_a = this.arrowFocusController) === null || _a === void 0 ? void 0 : _a.disconnect();
            this.arrowFocusController = undefined;
        }
        if (this.itemObserver) {
            this.itemObserver.disconnect();
            this.itemObserver = undefined;
        }
        if (this.disposeClickListener) {
            this.disposeClickListener();
            this.disposeClickListener = undefined;
        }
        if (this.disposeKeyListener) {
            this.disposeKeyListener();
            this.disposeKeyListener = undefined;
        }
        if (this.autoUpdateCleanup) {
            this.autoUpdateCleanup();
            this.autoUpdateCleanup = undefined;
        }
    }
    getAssignedSubmenuIds() {
        return this.assignedSubmenu;
    }
    isPresent() {
        return this.show;
    }
    present() {
        this.show = true;
    }
    dismiss() {
        this.show = false;
    }
    getId() {
        return this.localUId;
    }
    willDismiss() {
        const { defaultPrevented } = this.showChanged.emit(false);
        return !defaultPrevented;
    }
    willPresent() {
        const { defaultPrevented } = this.showChanged.emit(true);
        return !defaultPrevented;
    }
    get dropdownItems() {
        return Array.from(this.hostElement.querySelectorAll('ix-dropdown-item'));
    }
    get slotElement() {
        return this.hostElement.shadowRoot.querySelector('slot');
    }
    addEventListenersFor() {
        var _a, _b, _c;
        (_a = this.disposeClickListener) === null || _a === void 0 ? void 0 : _a.call(this);
        (_b = this.disposeKeyListener) === null || _b === void 0 ? void 0 : _b.call(this);
        const toggleController = () => {
            if (!this.isPresent()) {
                dropdownController.present(this);
            }
            else {
                dropdownController.dismiss(this);
            }
            dropdownController.dismissOthers(this.getId());
        };
        if (!this.triggerElement) {
            return;
        }
        this.disposeClickListener = addDisposableEventListener(this.triggerElement, 'click', (event) => {
            if (!event.defaultPrevented) {
                toggleController();
            }
        });
        (_c = this.triggerElement) === null || _c === void 0 ? void 0 : _c.setAttribute('data-ix-dropdown-trigger', this.localUId);
    }
    /** @internal */
    async discoverSubmenu() {
        var _a;
        (_a = this.triggerElement) === null || _a === void 0 ? void 0 : _a.dispatchEvent(new CustomEvent('ix-assign-sub-menu', {
            bubbles: true,
            composed: true,
            cancelable: true,
            detail: this.localUId,
        }));
    }
    registerKeyListener() {
        if (!this.triggerElement) {
            return;
        }
        this.disposeKeyListener = addDisposableEventListener(this.triggerElement, 'keydown', ((event) => {
            if (event.key !== 'ArrowDown') {
                return;
            }
            if (document.activeElement !== this.triggerElement) {
                return;
            }
            dropdownController.present(this);
            setTimeout(() => {
                this.focusDropdownItem(0);
            });
        }));
    }
    async registerListener(element) {
        this.triggerElement = await this.resolveElement(element);
        if (this.triggerElement) {
            this.addEventListenersFor();
            this.discoverSubmenu();
        }
    }
    async resolveElement(element) {
        const el = await findElement(element);
        return this.checkForSubmenuAnchor(el);
    }
    async checkForSubmenuAnchor(element) {
        if (!element) {
            return undefined;
        }
        if (hasDropdownItemWrapperImplemented(element)) {
            const dropdownItem = await element.getDropdownItemElement();
            dropdownItem.isSubMenu = true;
            this.hostElement.style.zIndex = `var(--theme-z-index-dropdown)`;
        }
        if (element.tagName === 'IX-DROPDOWN-ITEM') {
            element.isSubMenu = true;
            this.hostElement.style.zIndex = `var(--theme-z-index-dropdown)`;
        }
        return element;
    }
    async resolveAnchorElement() {
        if (this.anchor) {
            this.anchorElement = await this.resolveElement(this.anchor);
        }
        else if (this.trigger) {
            this.anchorElement = await this.resolveElement(this.trigger);
        }
    }
    async changedShow(newShow) {
        var _a, _b, _c, _d;
        if (newShow) {
            await this.resolveAnchorElement();
            if (this.anchorElement) {
                this.applyDropdownPosition();
            }
            this.arrowFocusController = new ArrowFocusController(this.dropdownItems, this.hostElement, (index) => this.focusDropdownItem(index));
            (_a = this.itemObserver) === null || _a === void 0 ? void 0 : _a.observe(this.hostElement, {
                childList: true,
                subtree: true,
            });
            this.registerKeyListener();
        }
        else {
            this.destroyAutoUpdate();
            (_b = this.arrowFocusController) === null || _b === void 0 ? void 0 : _b.disconnect();
            (_c = this.itemObserver) === null || _c === void 0 ? void 0 : _c.disconnect();
            (_d = this.disposeKeyListener) === null || _d === void 0 ? void 0 : _d.call(this);
        }
    }
    changedTrigger(newTriggerValue) {
        this.registerListener(newTriggerValue);
    }
    destroyAutoUpdate() {
        if (this.autoUpdateCleanup) {
            this.autoUpdateCleanup();
            this.autoUpdateCleanup = undefined;
        }
    }
    isAnchorSubmenu() {
        var _a;
        if (!hasDropdownItemWrapperImplemented(this.anchorElement)) {
            // Is no official dropdown-item, but check if any dropdown-item
            // is placed somewhere up the DOM
            return !!((_a = this.anchorElement) === null || _a === void 0 ? void 0 : _a.closest('ix-dropdown-item'));
        }
        return true;
    }
    async applyDropdownPosition() {
        var _a, _b;
        if (!this.show) {
            return;
        }
        if (!this.anchorElement) {
            return;
        }
        const isSubmenu = this.isAnchorSubmenu();
        let positionConfig = {
            strategy: this.positioningStrategy,
            middleware: [],
        };
        if (!this.suppressAutomaticPlacement) {
            (_a = positionConfig.middleware) === null || _a === void 0 ? void 0 : _a.push(flip({ fallbackStrategy: 'initialPlacement' }));
        }
        positionConfig.placement = isSubmenu ? 'right-start' : this.placement;
        positionConfig.middleware = [
            ...(((_b = positionConfig.middleware) === null || _b === void 0 ? void 0 : _b.filter(Boolean)) || []),
            inline(),
            shift(),
        ];
        if (this.offset) {
            positionConfig.middleware.push(offset(this.offset));
        }
        this.destroyAutoUpdate();
        if (!this.anchorElement) {
            return;
        }
        this.autoUpdateCleanup = autoUpdate(this.anchorElement, this.hostElement, async () => {
            if (this.anchorElement) {
                const computeResponse = await computePosition(this.anchorElement, this.hostElement, positionConfig);
                Object.assign(this.hostElement.style, {
                    top: '0',
                    left: '0',
                    transform: `translate(${Math.round(computeResponse.x)}px,${Math.round(computeResponse.y)}px)`,
                });
            }
            if (this.overwriteDropdownStyle) {
                const overwriteStyle = await this.overwriteDropdownStyle({
                    dropdownRef: this.hostElement,
                    triggerRef: this.triggerElement,
                });
                Object.assign(this.hostElement.style, overwriteStyle);
            }
        }, {
            ancestorResize: true,
            ancestorScroll: true,
            elementResize: true,
        });
    }
    focusDropdownItem(index) {
        requestAnimationFrame(() => {
            var _a, _b;
            const button = (_b = (_a = this.dropdownItems[index]) === null || _a === void 0 ? void 0 : _a.shadowRoot) === null || _b === void 0 ? void 0 : _b.querySelector('button');
            if (button) {
                button.focus();
            }
        });
    }
    async componentDidLoad() {
        if (!this.trigger) {
            return;
        }
        this.changedTrigger(this.trigger);
    }
    async componentDidRender() {
        await this.applyDropdownPosition();
        await this.resolveAnchorElement();
    }
    isTriggerElement(element) {
        const trigger = !!element.hasAttribute('data-ix-dropdown-trigger');
        return trigger;
    }
    onDropdownClick(event) {
        const target = dropdownController.pathIncludesTrigger(event.composedPath());
        if (target) {
            if (target !== this.triggerElement) {
                event.preventDefault();
            }
            if (this.isTriggerElement(target)) {
                if (this.closeBehavior === 'outside') {
                    event.preventDefault();
                }
                return;
            }
        }
        if (!event.defaultPrevented &&
            (this.closeBehavior === 'inside' || this.closeBehavior === 'both')) {
            dropdownController.dismissAll([this.getId()], this.ignoreRelatedSubmenu);
            return;
        }
        dropdownController.dismissOthers(this.getId());
    }
    /**
     * Update position of dropdown
     */
    async updatePosition() {
        this.applyDropdownPosition();
    }
    render() {
        return (h(Host, { key: 'f0cb7a478c8b8c066a7c163e83cbc101e6f3a5ec', "data-ix-dropdown": this.localUId, class: {
                'dropdown-menu': true,
                show: this.show,
                overflow: !this.suppressOverflowBehavior,
            }, style: {
                margin: '0',
                minWidth: '0px',
                position: this.positioningStrategy,
            }, role: "list", onClick: (event) => this.onDropdownClick(event) }, h("div", { key: '5431d09d3eb6b08f357ef7202617223ab3b347bf', style: { display: 'contents' } }, this.header && h("div", { key: 'b771273fe7292d3f253690b17955f13bb95ef56a', class: "dropdown-header" }, this.header), this.show && h("slot", { key: '0150080d095cce23717962df423ea6b8a137841e' }))));
    }
    static get is() { return "ix-dropdown"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["dropdown.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["dropdown.css"]
        };
    }
    static get properties() {
        return {
            "suppressAutomaticPlacement": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [{
                            "name": "since",
                            "text": "2.0.0"
                        }],
                    "text": "Suppress the automatic placement of the dropdown."
                },
                "attribute": "suppress-automatic-placement",
                "reflect": false,
                "defaultValue": "false"
            },
            "show": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Show dropdown"
                },
                "attribute": "show",
                "reflect": true,
                "defaultValue": "false"
            },
            "trigger": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "ElementReference",
                    "resolved": "HTMLElement | Promise<HTMLElement> | string",
                    "references": {
                        "ElementReference": {
                            "location": "import",
                            "path": "../utils/element-reference",
                            "id": "src/components/utils/element-reference.ts::ElementReference"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Define an element that triggers the dropdown.\nA trigger can either be a string that will be interpreted as id attribute or a DOM element."
                },
                "attribute": "trigger",
                "reflect": false
            },
            "anchor": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "ElementReference",
                    "resolved": "HTMLElement | Promise<HTMLElement> | string",
                    "references": {
                        "ElementReference": {
                            "location": "import",
                            "path": "../utils/element-reference",
                            "id": "src/components/utils/element-reference.ts::ElementReference"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Define an anchor element"
                },
                "attribute": "anchor",
                "reflect": false
            },
            "closeBehavior": {
                "type": "any",
                "mutable": false,
                "complexType": {
                    "original": "CloseBehavior",
                    "resolved": "\"both\" | \"inside\" | \"outside\" | boolean",
                    "references": {
                        "CloseBehavior": {
                            "location": "import",
                            "path": "./dropdown-controller",
                            "id": "src/components/dropdown/dropdown-controller.ts::CloseBehavior"
                        }
                    }
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Controls if the dropdown will be closed in response to a click event depending on the position of the event relative to the dropdown.\nIf the dropdown is a child of another one, it will be closed with the parent, regardless of its own close behavior."
                },
                "attribute": "close-behavior",
                "reflect": false,
                "defaultValue": "'both'"
            },
            "placement": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "AlignedPlacement",
                    "resolved": "\"bottom-end\" | \"bottom-start\" | \"left-end\" | \"left-start\" | \"right-end\" | \"right-start\" | \"top-end\" | \"top-start\"",
                    "references": {
                        "AlignedPlacement": {
                            "location": "import",
                            "path": "./placement",
                            "id": "src/components/dropdown/placement.ts::AlignedPlacement"
                        }
                    }
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Placement of the dropdown"
                },
                "attribute": "placement",
                "reflect": false,
                "defaultValue": "'bottom-start'"
            },
            "positioningStrategy": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "'absolute' | 'fixed'",
                    "resolved": "\"absolute\" | \"fixed\"",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Position strategy"
                },
                "attribute": "positioning-strategy",
                "reflect": false,
                "defaultValue": "'fixed'"
            },
            "header": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "An optional header shown at the top of the dropdown"
                },
                "attribute": "header",
                "reflect": false
            },
            "offset": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "{\n    mainAxis?: number;\n    crossAxis?: number;\n    alignmentAxis?: number;\n  }",
                    "resolved": "{ mainAxis?: number; crossAxis?: number; alignmentAxis?: number; }",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }],
                    "text": "Move dropdown along main axis of alignment"
                }
            },
            "overwriteDropdownStyle": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "(delegate: {\n    dropdownRef: HTMLElement;\n    triggerRef?: HTMLElement;\n  }) => Promise<Partial<CSSStyleDeclaration>>",
                    "resolved": "(delegate: { dropdownRef: HTMLElement; triggerRef?: HTMLElement; }) => Promise<Partial<CSSStyleDeclaration>>",
                    "references": {
                        "HTMLElement": {
                            "location": "global",
                            "id": "global::HTMLElement"
                        },
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        },
                        "Partial": {
                            "location": "global",
                            "id": "global::Partial"
                        },
                        "CSSStyleDeclaration": {
                            "location": "global",
                            "id": "global::CSSStyleDeclaration"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }],
                    "text": ""
                }
            },
            "discoverAllSubmenus": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "If initialization of this dropdown is expected to be deferred submenu discovery will have to be re-run globally by the controller.\nThis property indicates the need for that to the controller."
                        }],
                    "text": ""
                },
                "attribute": "discover-all-submenus",
                "reflect": false,
                "defaultValue": "false"
            },
            "ignoreRelatedSubmenu": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }],
                    "text": ""
                },
                "attribute": "ignore-related-submenu",
                "reflect": false,
                "defaultValue": "false"
            },
            "suppressOverflowBehavior": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }],
                    "text": ""
                },
                "attribute": "suppress-overflow-behavior",
                "reflect": false,
                "defaultValue": "false"
            }
        };
    }
    static get events() {
        return [{
                "method": "showChanged",
                "name": "showChanged",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Fire event after visibility of dropdown has changed"
                },
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                }
            }];
    }
    static get methods() {
        return {
            "discoverSubmenu": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "",
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }]
                }
            },
            "updatePosition": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Update position of dropdown",
                    "tags": []
                }
            }
        };
    }
    static get elementRef() { return "hostElement"; }
    static get watchers() {
        return [{
                "propName": "show",
                "methodName": "changedShow"
            }, {
                "propName": "trigger",
                "methodName": "changedTrigger"
            }];
    }
    static get listeners() {
        return [{
                "name": "ix-assign-sub-menu",
                "method": "cacheSubmenuId",
                "target": undefined,
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=dropdown.js.map
