/*
 * SPDX-FileCopyrightText: 2024 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { Host, h } from "@stencil/core";
import { createClassMutationObserver, isIxInputFieldComponent, } from "../utils/input";
import { a11yHostAttributes } from "../utils/a11y";
import { makeRef } from "../utils/make-ref";
export class FormFieldLabel {
    constructor() {
        this.htmlForObserver = new MutationObserver(() => this.checkForInternalState());
        this.a11yAttributes = {};
        this.labelRef = makeRef();
        this.required = undefined;
        this.htmlFor = undefined;
        this.controlRef = undefined;
        this.isInvalid = false;
    }
    connectedCallback() {
        this.registerHtmlForObserver();
        this.registerControlRefObserver();
    }
    disconnectedCallback() {
        if (this.htmlForObserver) {
            this.htmlForObserver.disconnect();
        }
        if (this.htmlForClassObserver) {
            this.htmlForClassObserver.destroy();
        }
        if (this.controlRefClassObserver) {
            this.controlRefClassObserver.destroy();
        }
    }
    componentWillRender() {
        this.checkForInternalState();
    }
    componentWillLoad() {
        this.a11yAttributes = a11yHostAttributes(this.hostElement);
    }
    registerHtmlForObserver() {
        if (typeof window === 'undefined') {
            return;
        }
        if (this.htmlForObserver) {
            this.htmlForObserver.disconnect();
        }
        if (this.htmlFor) {
            this.htmlForObserver.observe(window.document, {
                childList: true,
                subtree: true,
            });
        }
    }
    async registerControlRefObserver() {
        if (typeof window === 'undefined') {
            return;
        }
        if (this.controlRefClassObserver) {
            this.controlRefClassObserver.destroy();
        }
        if (this.controlRef) {
            const input = await this.controlRef.waitForCurrent();
            this.controlRefClassObserver = createClassMutationObserver(input, () => this.checkForInvalidState(input));
        }
    }
    registerHtmlForClassObserver(forElement) {
        if (this.htmlForClassObserver) {
            this.htmlForClassObserver.destroy();
        }
        this.htmlForClassObserver = createClassMutationObserver(forElement, () => this.checkForInvalidState(forElement));
    }
    checkForInvalidState(elementToCheck) {
        this.isInvalid =
            elementToCheck.classList.contains('is-invalid') ||
                elementToCheck.classList.contains('ix-invalid');
    }
    async checkForInternalState() {
        if (this.htmlFor) {
            const forElement = document.getElementById(this.htmlFor);
            if (forElement) {
                if (typeof forElement.required === 'boolean') {
                    this.required = forElement.required;
                }
                this.registerHtmlForClassObserver(forElement);
                this.checkForInvalidState(forElement);
            }
        }
        if (this.controlRef) {
            const input = await this.controlRef.waitForCurrent();
            this.required = input.required;
        }
    }
    async focusOnClick() {
        if (this.htmlFor) {
            const target = document.getElementById(this.htmlFor);
            if (target) {
                let input = null;
                if (isIxInputFieldComponent(target)) {
                    input = await target.getNativeInputElement();
                }
                else {
                    input = target;
                }
                if (typeof input.focus === 'function') {
                    input.focus();
                }
            }
        }
        if (this.controlRef) {
            (await this.controlRef.waitForCurrent()).focus();
        }
    }
    render() {
        return (h(Host, { key: '3a62067a0af7847c578459a35bcbd56c54f59bf1', onClick: () => this.focusOnClick() }, h("label", Object.assign({ key: '05aea8c0c5783f8cffb9d3968f1621564203c2ac', htmlFor: this.htmlFor }, this.a11yAttributes, { ref: this.labelRef }), h("ix-typography", { key: '6285d1624065c691d9beb3a1c4c31268e0a6bde9', color: this.isInvalid ? 'alarm' : 'soft', format: "label" }, h("slot", { key: 'cc8185cb40bf055c6f2e29ee09265acc10719657' }), this.required && h("span", { key: 'b8c6191b6a0fd1314afd21d555268417292b1201' }, "\u00A0*")))));
    }
    static get is() { return "ix-field-label"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["field-label.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["field-label.css"]
        };
    }
    static get properties() {
        return {
            "required": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "A value is required or must be checked for the form to be submittable"
                },
                "attribute": "required",
                "reflect": true
            },
            "htmlFor": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The id of the form element that the label is associated with"
                },
                "attribute": "html-for",
                "reflect": true
            },
            "controlRef": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "MakeRef<HTMLElement>",
                    "resolved": "{ (ref: HTMLElement): void; current: HTMLElement; waitForCurrent(): Promise<HTMLElement>; }",
                    "references": {
                        "MakeRef": {
                            "location": "import",
                            "path": "../utils/make-ref",
                            "id": "src/components/utils/make-ref.ts::MakeRef"
                        },
                        "HTMLElement": {
                            "location": "global",
                            "id": "global::HTMLElement"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }],
                    "text": ""
                }
            },
            "isInvalid": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }],
                    "text": ""
                },
                "attribute": "is-invalid",
                "reflect": false,
                "defaultValue": "false"
            }
        };
    }
    static get elementRef() { return "hostElement"; }
    static get watchers() {
        return [{
                "propName": "htmlFor",
                "methodName": "registerHtmlForObserver"
            }, {
                "propName": "controlRef",
                "methodName": "registerControlRefObserver"
            }];
    }
}
//# sourceMappingURL=field-label.js.map
