/*
 * SPDX-FileCopyrightText: 2024 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { h, Host, } from "@stencil/core";
import { createMutationObserver } from "../utils/mutation-observer";
import { hasSlottedElements } from "../utils/shadow-dom";
export class Group {
    constructor() {
        this.observer = null;
        this.suppressHeaderSelection = false;
        this.header = undefined;
        this.subHeader = undefined;
        this.collapsed = true;
        this.selected = false;
        this.index = undefined;
        this.expandOnHeaderClick = false;
        this.itemSelected = false;
        this.slotSize = this.groupItems.length;
        this.footerVisible = false;
        this.showExpandCollapsedIcon = false;
    }
    selectedChanged(newSelected) {
        if (newSelected === false) {
            this.changeItemIndex();
        }
    }
    get dropdownItems() {
        return Array.from(this.hostElement.querySelectorAll('ix-group-dropdown-item'));
    }
    get groupItems() {
        return Array.from(this.hostElement.querySelectorAll('ix-group-item:not(.footer)'));
    }
    get groupContent() {
        var _a;
        return (_a = this.hostElement.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector('.group-content');
    }
    onExpandClick(event) {
        const oldCollapsed = this.collapsed;
        this.collapsed = !this.collapsed;
        const { defaultPrevented } = this.collapsedChanged.emit(this.collapsed);
        event.stopPropagation();
        if (defaultPrevented) {
            this.collapsed = oldCollapsed;
        }
    }
    onHeaderClick(event) {
        if (this.suppressHeaderSelection) {
            this.onExpandClick(event);
            return;
        }
        this.changeHeaderSelection(!this.selected);
        this.changeItemIndex();
    }
    changeHeaderSelection(newSelection) {
        const oldIsHeaderSelected = this.selected;
        const newIsHeaderSelected = newSelection;
        this.selected = newIsHeaderSelected;
        const { defaultPrevented } = this.selectGroup.emit(newIsHeaderSelected);
        if (defaultPrevented) {
            this.selected = oldIsHeaderSelected;
            return;
        }
    }
    changeItemIndex(index) {
        const oldIndex = this.index;
        const newIndex = index === this.index ? undefined : index;
        if (this.index === newIndex) {
            return;
        }
        this.index = newIndex;
        const { defaultPrevented } = this.selectItem.emit(newIndex);
        if (defaultPrevented) {
            this.index = oldIndex;
            return;
        }
        const items = this.groupItems;
        items.forEach((item, i) => {
            item.selected = i === this.index;
        });
        this.itemSelected = items.some((item) => item.selected);
    }
    onSlotChange() {
        var _a;
        const slot = (_a = this.hostElement.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector('slot[name="footer"]');
        if (slot) {
            this.footerVisible = hasSlottedElements(slot);
        }
    }
    componentWillRender() {
        this.groupItems.forEach((item, index) => {
            item.selected = index === this.index;
            item.index = index;
        });
    }
    componentDidLoad() {
        this.observer = createMutationObserver(() => {
            this.slotSize = this.groupItems.length;
        });
        if (!this.groupContent) {
            return;
        }
        this.observer.observe(this.groupContent, {
            childList: true,
        });
    }
    disconnectedCallback() {
        if (this.observer) {
            this.observer.disconnect();
        }
    }
    onItemClicked(event) {
        if (event.target instanceof HTMLElement) {
            const item = event.target;
            const index = this.groupItems.indexOf(item);
            this.changeItemIndex(index);
        }
    }
    render() {
        return (h(Host, { key: 'e3dc08bdd081992f3e104d2694289cffdd00e2d5' }, h("div", { key: 'dd9ae49a4772bca8c59b7dd30ae51bbfcf28e67a', class: {
                'group-header': true,
                expand: !this.collapsed,
                selected: this.selected,
            }, tabindex: "0" }, h("div", { key: 'bdd3e57474c382203ac92130db6aac7e5a091088', class: "group-header-clickable", onClick: (e) => this.onHeaderClick(e) }, h("div", { key: 'c53eeccfc402a6376b6ecd1130355428850af59b', class: {
                'group-header-selection-indicator': true,
                'group-header-selection-indicator-item-selected': this.itemSelected,
            } }), h("div", { key: 'f45b5cb472ac2842093e117eddebaa9f31cf4105', class: "btn-expand-header" }, h("ix-icon", { key: '50132cef721b918beacb9c94d6eb3536d8a35f56', "data-testid": "expand-collapsed-icon", class: {
                hidden: !this.showExpandCollapsedIcon,
            }, name: this.collapsed ? 'chevron-right-small' : 'chevron-down-small', onClick: (e) => this.onExpandClick(e) })), h("div", { key: '274809a5b36d4a53aac997179490e61611b615e7', class: "group-header-content" }, this.header ? (h("div", { class: "group-header-props-container" }, h("div", { class: "group-header-title" }, h("span", { title: this.header }, this.header)), h("div", { class: "group-subheader", title: this.subHeader }, this.subHeader))) : null, h("slot", { key: '3d0437fc04c0153e5df963cd90372f18e2b26446', name: "header" }))), h("ix-group-context-menu", { key: '2de378c24367eeca11a58946021fad53f5b8dc81' }, h("slot", { key: '4a46b76d48333986d6ee3b76278fc7b4bb1c3242', name: "dropdown" }))), h("div", { key: '5fc1aa340e95af877901c53c2546fe094017da89', class: {
                'group-content': true,
            } }, h("div", { key: '7ba0a9e6189940dc954ce8d2a2727b1afcf73b92', style: {
                display: this.collapsed ? 'none' : 'contents',
            } }, h("slot", { key: '980f2fd4380a6921e1ce6a5a601b255bd816ef7c', onSlotchange: () => {
                var _a;
                const slot = (_a = this.hostElement.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector('slot:not([name])');
                this.showExpandCollapsedIcon = hasSlottedElements(slot);
            } }), h("ix-group-item", { key: '0d9ecd669ba5f14d219578289853b91133c4dc43', suppressSelection: true, focusable: false, class: {
                footer: true,
                'footer-visible': this.footerVisible,
            } }, h("slot", { key: '29d5b1440855ad87320fa79c60657f2b4c6e6a0c', name: "footer", onSlotchange: () => this.onSlotChange() }))))));
    }
    static get is() { return "ix-group"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["group.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["group.css"]
        };
    }
    static get properties() {
        return {
            "suppressHeaderSelection": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Prevent header from being selectable"
                },
                "attribute": "suppress-header-selection",
                "reflect": false,
                "defaultValue": "false"
            },
            "header": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Group header"
                },
                "attribute": "header",
                "reflect": false
            },
            "subHeader": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Group header subtitle"
                },
                "attribute": "sub-header",
                "reflect": false
            },
            "collapsed": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Whether the group is collapsed or expanded. Defaults to true."
                },
                "attribute": "collapsed",
                "reflect": true,
                "defaultValue": "true"
            },
            "selected": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Whether the group is selected."
                },
                "attribute": "selected",
                "reflect": true,
                "defaultValue": "false"
            },
            "index": {
                "type": "number",
                "mutable": true,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The index of the selected group entry.\nIf undefined no group item is selected."
                },
                "attribute": "index",
                "reflect": true
            },
            "expandOnHeaderClick": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Expand the group if the header is clicked"
                },
                "attribute": "expand-on-header-click",
                "reflect": false,
                "defaultValue": "false"
            }
        };
    }
    static get states() {
        return {
            "itemSelected": {},
            "slotSize": {},
            "footerVisible": {},
            "showExpandCollapsedIcon": {}
        };
    }
    static get events() {
        return [{
                "method": "selectGroup",
                "name": "selectGroup",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emits when whole group gets selected."
                },
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                }
            }, {
                "method": "selectItem",
                "name": "selectItem",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emits when group item gets selected."
                },
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                }
            }, {
                "method": "collapsedChanged",
                "name": "collapsedChanged",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Group collapsed"
                },
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                }
            }];
    }
    static get elementRef() { return "hostElement"; }
    static get watchers() {
        return [{
                "propName": "selected",
                "methodName": "selectedChanged"
            }];
    }
    static get listeners() {
        return [{
                "name": "selectedChanged",
                "method": "onItemClicked",
                "target": undefined,
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=group.js.map
