/*
 * SPDX-FileCopyrightText: 2024 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { a11yBoolean } from "../utils/a11y";
import { shouldSuppressInternalValidation, } from "../utils/input";
import { convertToRemString } from "../utils/rwd.util";
import { generateUUID } from "../utils/uuid";
import { shakeInput } from "./input.animation";
export function createIdIfNotExists(element, idPrefix = 'input') {
    return element.hasAttribute('id')
        ? element.getAttribute('id')
        : `${idPrefix}-${generateUUID()}`;
}
export function mapValidationResult(ref, result) {
    ref.isInvalid = result.isInvalid || result.isInvalidByRequired;
    ref.isValid = result.isValid;
    ref.isInfo = result.isInfo;
    ref.isWarning = result.isWarning;
}
export function checkAllowedKeys(comp, event) {
    if (comp.allowedCharactersPattern) {
        const regex = new RegExp(comp.allowedCharactersPattern);
        if (!regex.test(event.key)) {
            event.preventDefault();
            shakeInput(comp.inputRef.current);
        }
    }
}
export async function checkInternalValidity(comp, input) {
    const validityState = input.validity;
    const eventResult = comp.validityStateChange.emit(validityState);
    if (eventResult.defaultPrevented) {
        return;
    }
    if (!comp.value) {
        return;
    }
    const skipValidation = await shouldSuppressInternalValidation(comp);
    if (skipValidation) {
        return;
    }
    const { valid } = validityState;
    comp.hostElement.classList.toggle('ix-invalid', !valid);
}
export function onInputBlur(comp, input) {
    comp.ixBlur.emit();
    if (!input) {
        throw new Error('Input element is not available');
    }
    checkInternalValidity(comp, input);
}
export function applyPaddingEnd(inputElement, width, options) {
    var _a;
    if (!inputElement) {
        return;
    }
    const remInPixels = 16;
    const padding = convertToRemString(width + remInPixels / 2);
    if (options.slotEnd) {
        inputElement.style.paddingRight = `calc(${padding} + ${(_a = options.additionalPaddingRight) !== null && _a !== void 0 ? _a : '0rem'})`;
    }
    else {
        inputElement.style.paddingLeft = padding;
    }
}
export function adjustPaddingForStartAndEnd(startElement, endElement, inputElement) {
    requestAnimationFrame(() => {
        requestAnimationFrame(() => {
            if (startElement) {
                const startBoundingRect = startElement.getBoundingClientRect();
                if (startBoundingRect) {
                    applyPaddingEnd(inputElement, startBoundingRect.width, {
                        slotEnd: false,
                    });
                }
            }
            if (endElement) {
                const endBoundingRect = endElement.getBoundingClientRect();
                if (endBoundingRect) {
                    applyPaddingEnd(inputElement, endBoundingRect.width, {
                        slotEnd: true,
                    });
                }
            }
        });
    });
}
export function getAriaAttributesForInput(component) {
    const inputAria = {
        'aria-invalid': `${a11yBoolean(component.isInvalid)}`,
        'aria-required': `${a11yBoolean(component.required)}`,
    };
    if (component.isInvalid && component.invalidText) {
        inputAria['aria-errormessage'] = component.invalidText;
    }
    return inputAria;
}
//# sourceMappingURL=input.util.js.map
