import { Host, h } from "@stencil/core";
/**
 * @since 2.6.0
 */
export class LayoutForm {
    constructor() {
        this.mediaQueryList = [];
        this.observer = new MutationObserver(() => this.calculateGridTemplateColumns());
        this.resizeObserver = new ResizeObserver(() => {
            this.calculateGridTemplateColumns();
        });
        this.layout = [
            { minWidth: '0', columns: 1 },
            { minWidth: '48em', columns: 2 },
        ];
    }
    connectedCallback() {
        this.observer.observe(this.hostElement, {
            childList: true,
            subtree: true,
            attributes: true,
            attributeFilter: ['data-colspan'],
        });
        this.resizeObserver.observe(this.hostElement);
        this.calculateGridTemplateColumns();
    }
    componentWillLoad() {
        this.calculateGridTemplateColumns();
    }
    disconnectedCallback() {
        if (this.observer) {
            this.observer.disconnect();
        }
        if (this.resizeObserver) {
            this.resizeObserver.disconnect();
        }
    }
    updateMediaQueryList() {
        this.mediaQueryList = [];
        this.layout.forEach((layout) => {
            const mediaQuery = window.matchMedia(`(min-width: ${layout.minWidth})`);
            this.mediaQueryList.push({
                mediaQuery,
                layout,
            });
        });
    }
    parseNumber(number) {
        if (!number) {
            return 1;
        }
        const result = parseInt(number);
        if (isNaN(result)) {
            return 1;
        }
        return result;
    }
    calculateGridTemplateColumns() {
        this.updateMediaQueryList();
        let layoutColumns = 1;
        let columnSpacing = 'var(--ix-layout-grid-gap)';
        this.mediaQueryList.forEach((mediaQuery) => {
            if (mediaQuery.mediaQuery.matches) {
                layoutColumns = mediaQuery.layout.columns;
            }
        });
        let column = 0;
        Array.from(this.hostElement.children).forEach((child) => {
            let colspan = this.parseNumber(child.getAttribute('data-colspan'));
            colspan = Math.min(colspan, layoutColumns);
            const childRatio = colspan / layoutColumns;
            child.style.width = `calc(${childRatio * 99.9}% - ${1 - childRatio} * ${columnSpacing})`;
            if (column + colspan > layoutColumns) {
                column = 0;
            }
            column = (column + colspan) % layoutColumns;
        });
    }
    render() {
        return (h(Host, { key: '0dd50d7088ab95ce631e5d3888055691b0499e81' }, h("div", { key: '4fd6fb7902a4fbe3232178bf260dd7fe2babdf4b', class: "container" }, h("slot", { key: '4548597f8071865d8203100a0ecbdc5fcab91293' }))));
    }
    static get is() { return "ix-layout-auto"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["layout-auto.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["layout-auto.css"]
        };
    }
    static get properties() {
        return {
            "layout": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "{\n    minWidth: string;\n    columns: number;\n  }[]",
                    "resolved": "{ minWidth: string; columns: number; }[]",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Defines the layout of the form."
                },
                "defaultValue": "[\n    { minWidth: '0', columns: 1 },\n    { minWidth: '48em', columns: 2 },\n  ]"
            }
        };
    }
    static get elementRef() { return "hostElement"; }
    static get watchers() {
        return [{
                "propName": "layout",
                "methodName": "updateMediaQueryList"
            }];
    }
}
//# sourceMappingURL=layout-auto.js.map
