/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { h, Host, } from "@stencil/core";
import anime from "animejs";
import { ApplicationLayoutContext } from "../utils/application-layout/context";
import { useContextProvider } from "../utils/context";
export class MapNavigation {
    constructor() {
        this.applicationName = undefined;
        this.navigationTitle = undefined;
        this.hideContextMenu = true;
        this.isSidebarOpen = true;
        this.hasContentHeader = false;
    }
    get menu() {
        return this.hostElement.querySelector('ix-menu');
    }
    get menuOverlay() {
        return this.hostElement.querySelector('ix-menu-overlay');
    }
    get mapNavMenu() {
        return this.hostElement.shadowRoot.querySelector('.map-nav-menu');
    }
    get sidebar() {
        return this.hostElement.shadowRoot.querySelector('.map-nav-sidebar');
    }
    get overlay() {
        return this.hostElement.shadowRoot.querySelector('#overlay');
    }
    componentDidRender() {
        this.appendMenu();
        this.closeOverlay();
    }
    componentWillLoad() {
        useContextProvider(this.hostElement, ApplicationLayoutContext, {
            hideHeader: false,
            host: 'map-navigation',
        });
    }
    appendMenu() {
        this.menu.addEventListener('mapExpandChange', (event) => {
            const state = !event.detail;
            this.toggleSidebar(state);
        });
        this.menu.enableMapExpand = true;
    }
    /**
     * Change the visibility of the sidebar
     *
     * @param show new visibility state
     * @since 1.6.0
     */
    async toggleSidebar(show) {
        if (show !== undefined) {
            this.isSidebarOpen = show;
        }
        else {
            this.isSidebarOpen = !this.isSidebarOpen;
        }
        if (this.isSidebarOpen) {
            this.openSidebar();
        }
        else {
            this.closeSidebar();
        }
        this.navigationToggled.emit(this.isSidebarOpen);
        this.menu.toggleMapExpand(this.isSidebarOpen);
    }
    closeSidebar() {
        anime({
            targets: this.sidebar,
            duration: MapNavigation.defaultTime,
            marginLeft: [0, '-29.75rem'],
            opacity: [1, 0],
            easing: 'easeInSine',
            complete: () => {
                this.sidebar.classList.add('d-none');
            },
        });
    }
    openSidebar() {
        anime({
            targets: this.sidebar,
            duration: MapNavigation.defaultTime,
            marginLeft: ['-29.75rem', 0],
            opacity: [0, 1],
            easing: 'easeOutSine',
            begin: () => {
                this.sidebar.classList.remove('d-none');
            },
        });
    }
    /**
     * Open a overlay inside content area
     * @deprecated Will be removed in 2.0.0. Use slot based approach
     *
     * @param name
     * @param component
     * @param icon
     * @param color
     */
    async openOverlay(name, component, icon, color) {
        anime({
            targets: this.overlay,
            duration: MapNavigation.slowTime,
            backdropFilter: [0, 'blur(1rem)'],
            translateX: ['-4rem', 0],
            opacity: [0, 1],
            easing: 'easeOutSine',
            begin: () => {
                this.overlay.classList.remove('d-none');
            },
        });
        const overlayInstance = document.createElement('ix-map-navigation-overlay');
        overlayInstance.setAttribute('color', color);
        overlayInstance.setAttribute('name', name);
        overlayInstance.setAttribute('icon', icon);
        overlayInstance.setAttribute('slot', 'overlay');
        overlayInstance.addEventListener('closeClick', () => this.closeOverlay());
        overlayInstance.appendChild(component);
        this.hostElement.appendChild(overlayInstance);
    }
    /**
     * Close current shown overlay
     * @deprecated Will be removed in 2.0.0. Use slot based approach
     */
    async closeOverlay() {
        anime({
            targets: this.overlay,
            duration: MapNavigation.slowTime,
            backdropFilter: ['blur(1rem)', 0],
            translateX: [0, '-4rem'],
            opacity: [1, 0],
            easing: 'easeInSine',
            complete: () => {
                var _a;
                if (!this.overlay) {
                    return;
                }
                (_a = this.overlay.firstChild) === null || _a === void 0 ? void 0 : _a.remove();
                this.overlay.classList.add('d-none');
            },
        });
    }
    checkHasContentHeader(e) {
        const nodes = e.currentTarget.assignedNodes({
            flatten: true,
        });
        this.hasContentHeader = (nodes === null || nodes === void 0 ? void 0 : nodes.length) !== 0;
    }
    render() {
        return (h(Host, { key: '9b4c4bb24f19945428477be04c10c6f80868ded0' }, h("slot", { key: '6f0db803a6755ed73737fe783e8d082a2df54f5b', name: "menu" }), h("div", { key: 'e8c578da79eed996626ed15e959dfc7c18675bfb', class: "map-nav" }, h("div", { key: 'fb834017ce2d372bc8ec93d16dd8c1975f8b6939', class: "map-nav-sidebar" }, h("div", { key: 'e0378fb0501f8e08b4c60d4051c8f0d1a72b7104', class: "map-nav-header" }, h("ix-application-header", { key: 'c03535cc7a7739e1a072a1687ab263f8a1c77d69', name: this.applicationName, class: "map-nav-header-brand" }, h("slot", { key: '9c4a8847d90dbdaf42bb9591b55680e547b1d52c', slot: "logo", name: "logo" }))), h("div", { key: '98b3542ea500a543322a84a640b839e2959cc0fe', class: "map-nav-sidebar-content" }, h("div", { key: 'a6d2f51e083a9e9b9d9839ea0b55090ee1354b42', class: "map-nav-sidebar-static-content" }, h("div", { key: '08a2f1f9c3c3730be3b5af9f6637cf4480461cda', class: "map-nav-title" }, this.navigationTitle), this.hideContextMenu ? ('') : (h("ix-icon-button", { icon: 'context-menu', ghost: true, size: "24", variant: "secondary", onClick: (_) => this.contextMenuClick.emit() }))), h("div", { key: 'b9f9fec616e9d53a5144d651c8f33cb65bd43d6b', class: "map-nav-sidebar-user-content" }, h("slot", { key: 'affb80bf56de7fab01f7c959d37f041cf8100fd2', name: "sidebar-content" })))), h("div", { key: 'ae33912f9c102c78d08c7e0a1e473ebbf4905a9c', class: "content" }, h("div", { key: '2385a7c87ded8cd7a1db225b3e45c16120b43aef', class: {
                'map-nav-header-content': true,
                'bg-2': true,
                empty: !this.hasContentHeader,
            } }, h("slot", { key: 'e230b473f2729038bc75d6b97f7a1bb5dc7b8b8e', name: "content-header", onSlotchange: (e) => this.checkHasContentHeader(e) })), h("main", { key: '062163caaa4a2b61b48c2124d23d047041688ce2' }, h("slot", { key: '444718852cea947e7bf581321cae89f113c474b5' }), h("slot", { key: 'c2160f945e134d90db2820c5ba4505d85845e5d6', name: "overlay" }))))));
    }
    static get is() { return "ix-map-navigation"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["map-navigation.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["map-navigation.css"]
        };
    }
    static get properties() {
        return {
            "applicationName": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Application name"
                },
                "attribute": "application-name",
                "reflect": false
            },
            "navigationTitle": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Navigation title"
                },
                "attribute": "navigation-title",
                "reflect": false
            },
            "hideContextMenu": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Hide the sidebar context menu button when set to true"
                },
                "attribute": "hide-context-menu",
                "reflect": false,
                "defaultValue": "true"
            }
        };
    }
    static get states() {
        return {
            "isSidebarOpen": {},
            "hasContentHeader": {}
        };
    }
    static get events() {
        return [{
                "method": "navigationToggled",
                "name": "navigationToggled",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Navigation toggled"
                },
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                }
            }, {
                "method": "contextMenuClick",
                "name": "contextMenuClick",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Context menu clicked"
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }];
    }
    static get methods() {
        return {
            "toggleSidebar": {
                "complexType": {
                    "signature": "(show?: boolean) => Promise<void>",
                    "parameters": [{
                            "name": "show",
                            "type": "boolean",
                            "docs": "new visibility state"
                        }],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Change the visibility of the sidebar",
                    "tags": [{
                            "name": "param",
                            "text": "show new visibility state"
                        }, {
                            "name": "since",
                            "text": "1.6.0"
                        }]
                }
            },
            "openOverlay": {
                "complexType": {
                    "signature": "(name: string, component: HTMLElement, icon?: string, color?: string) => Promise<void>",
                    "parameters": [{
                            "name": "name",
                            "type": "string",
                            "docs": ""
                        }, {
                            "name": "component",
                            "type": "HTMLElement",
                            "docs": ""
                        }, {
                            "name": "icon",
                            "type": "string",
                            "docs": ""
                        }, {
                            "name": "color",
                            "type": "string",
                            "docs": ""
                        }],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        },
                        "HTMLElement": {
                            "location": "global",
                            "id": "global::HTMLElement"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Open a overlay inside content area",
                    "tags": [{
                            "name": "deprecated",
                            "text": "Will be removed in 2.0.0. Use slot based approach"
                        }, {
                            "name": "param",
                            "text": "name"
                        }, {
                            "name": "param",
                            "text": "component"
                        }, {
                            "name": "param",
                            "text": "icon"
                        }, {
                            "name": "param",
                            "text": "color"
                        }]
                }
            },
            "closeOverlay": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Close current shown overlay",
                    "tags": [{
                            "name": "deprecated",
                            "text": "Will be removed in 2.0.0. Use slot based approach"
                        }]
                }
            }
        };
    }
    static get elementRef() { return "hostElement"; }
}
MapNavigation.defaultTime = 150;
MapNavigation.slowTime = 500;
//# sourceMappingURL=map-navigation.js.map
