/*
 * SPDX-FileCopyrightText: 2023 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { h, Host, } from "@stencil/core";
import anime from "animejs";
import { closestIxMenu } from "../utils/application-layout/context";
import { createMutationObserver } from "../utils/mutation-observer";
import { createEnterLeaveDebounce } from "./enter-leave";
const DefaultIxMenuItemHeight = 40;
const DefaultAnimationTimeout = 150;
/**
 * @since 2.0.0
 */
export class MenuCategory {
    constructor() {
        this.enterLeaveDebounce = createEnterLeaveDebounce(() => {
            this.onPointerEnter();
        }, () => {
            this.onPointerLeave();
        });
        this.label = undefined;
        this.icon = undefined;
        this.notifications = undefined;
        this.menuExpand = false;
        this.showItems = false;
        this.showDropdown = false;
        this.nestedItems = [];
    }
    isNestedItemActive() {
        return this.getNestedItems().some((item) => item.active);
    }
    getNestedItems() {
        return Array.from(this.hostElement.querySelectorAll(':scope ix-menu-item'));
    }
    getNestedItemsHeight() {
        const items = this.getNestedItems();
        return items.length * DefaultIxMenuItemHeight;
    }
    onExpandCategory(showItems) {
        if (showItems) {
            this.animateFadeIn();
        }
        else {
            this.animateFadeOut();
        }
    }
    animateFadeOut() {
        const slotHideThresholdMs = 25;
        anime({
            targets: this.menuItemsContainer,
            duration: DefaultAnimationTimeout,
            easing: 'easeInSine',
            opacity: [1, 0],
            maxHeight: [this.getNestedItemsHeight() + DefaultIxMenuItemHeight, 0],
            complete: () => {
                setTimeout(() => {
                    this.showItems = false;
                    this.showDropdown = false;
                }, DefaultAnimationTimeout + slotHideThresholdMs);
            },
        });
    }
    animateFadeIn() {
        anime({
            targets: this.menuItemsContainer,
            duration: DefaultAnimationTimeout,
            easing: 'easeInSine',
            opacity: [0, 1],
            maxHeight: [0, this.getNestedItemsHeight() + DefaultIxMenuItemHeight],
            begin: () => {
                this.showItems = true;
                this.showDropdown = false;
            },
        });
    }
    onPointerEnter() {
        if (this.ixMenu.expand) {
            return;
        }
        this.closeOtherCategories.emit();
        this.showDropdown = true;
    }
    onPointerLeave() {
        this.showDropdown = false;
    }
    onCategoryClick(e) {
        e.stopPropagation();
        if (this.ixMenu.expand) {
            e === null || e === void 0 ? void 0 : e.stopPropagation();
            this.onExpandCategory(!this.showItems);
            return;
        }
    }
    onNestedItemsChanged() {
        this.nestedItems = this.getNestedItems();
    }
    isCategoryItemListVisible() {
        return this.menuExpand && (this.showItems || this.isNestedItemActive());
    }
    componentWillLoad() {
        const closestMenu = closestIxMenu(this.hostElement);
        if (!closestMenu) {
            throw Error('ix-menu-category can only be used as a child of ix-menu');
        }
        this.ixMenu = closestMenu;
        this.menuExpand = this.ixMenu.expand;
        this.showItems = this.isCategoryItemListVisible();
    }
    componentDidLoad() {
        this.observer = createMutationObserver(() => this.onNestedItemsChanged());
        this.observer.observe(this.hostElement, {
            attributes: true,
            childList: true,
            subtree: true,
        });
        requestAnimationFrame(() => {
            this.onNestedItemsChanged();
        });
        this.ixMenu.addEventListener('expandChange', ({ detail: menuExpand }) => {
            this.menuExpand = menuExpand;
            if (!menuExpand) {
                this.clearMenuItemStyles();
            }
            this.showItems = this.isCategoryItemListVisible();
        });
    }
    clearMenuItemStyles() {
        this.menuItemsContainer.style.removeProperty('max-height');
        this.menuItemsContainer.style.removeProperty('opacity');
    }
    disconnectedCallback() {
        if (this.observer) {
            this.observer.disconnect();
        }
    }
    render() {
        return (h(Host, { key: 'f9f4730936ba8f8ee2a46e0ce306117f947de177', class: {
                expanded: this.showItems,
            }, onPointerEnter: () => {
                this.enterLeaveDebounce.onEnter();
            }, onPointerLeave: (event) => {
                if (event.pointerType === 'touch') {
                    return;
                }
                this.enterLeaveDebounce.onLeave();
            } }, h("ix-menu-item", { key: 'c8c152e9c345945b50803bd72998a09b72a70d74', class: 'category-parent', active: this.isNestedItemActive(), notifications: this.notifications, icon: this.icon, onClick: (e) => this.onCategoryClick(e), onFocus: () => this.onPointerEnter(), isCategory: true }, h("div", { key: '3b584077fbdbf1ce09421f33fddc52ba4da5d168', class: "category" }, h("div", { key: '73388008e380b2d6f341eef162727fd59aec88ee', class: "category-text" }, this.label), h("ix-icon", { key: 'ab5cf9fd56692b29bcc7501cb096411416a661a5', name: 'chevron-down-small', class: {
                'category-chevron': true,
                'category-chevron--open': this.showItems,
            } }))), h("div", { key: 'faa34c42bcf4250389ca2f46566b3992b235db27', ref: (ref) => (this.menuItemsContainer = ref), class: {
                'menu-items': true,
                'menu-items--expanded': this.showItems,
                'menu-items--collapsed': !this.showItems,
            } }, this.showItems ? h("slot", null) : null), h("ix-dropdown", { key: 'd366bdd594350a8f67ee80b0d7d7b98d8d57cb28', closeBehavior: 'both', show: this.showDropdown, onShowChanged: ({ detail: dropdownShown }) => {
                this.showDropdown = dropdownShown;
            }, class: 'category-dropdown', anchor: this.hostElement, placement: "right-start", offset: {
                mainAxis: 3,
            }, onClick: (e) => {
                if (e.target instanceof HTMLElement) {
                    if (e.target.tagName === 'IX-MENU-ITEM') {
                        this.showDropdown = false;
                    }
                    else {
                        e.preventDefault();
                    }
                }
            } }, h("ix-dropdown-item", { key: '66918af3f1d25eaa574e47eda975b1ce23e954d4', class: 'category-dropdown-header' }, h("ix-typography", { key: '7479a92697b5e704aa75a9281d75aad54384328e', format: "label", bold: true, color: "std" }, this.label)), h("ix-divider", { key: 'ac0f5ce34503cfe9e77071185b46789eed2c3eef' }), h("slot", { key: '78dc27c30debcc86f4040e359a8659b55d8259b1' }))));
    }
    static get is() { return "ix-menu-category"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["menu-category.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["menu-category.css"]
        };
    }
    static get properties() {
        return {
            "label": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Display name of the category"
                },
                "attribute": "label",
                "reflect": false
            },
            "icon": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Icon of the category"
                },
                "attribute": "icon",
                "reflect": false
            },
            "notifications": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Show notification count on the category"
                },
                "attribute": "notifications",
                "reflect": false
            }
        };
    }
    static get states() {
        return {
            "menuExpand": {},
            "showItems": {},
            "showDropdown": {},
            "nestedItems": {}
        };
    }
    static get events() {
        return [{
                "method": "closeOtherCategories",
                "name": "closeOtherCategories",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }],
                    "text": ""
                },
                "complexType": {
                    "original": "any",
                    "resolved": "any",
                    "references": {}
                }
            }];
    }
    static get elementRef() { return "hostElement"; }
    static get listeners() {
        return [{
                "name": "closeOtherCategories",
                "method": "onPointerLeave",
                "target": "window",
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=menu-category.js.map
