/*
 * SPDX-FileCopyrightText: 2024 Siemens AG
 *
 * SPDX-License-Identifier: MIT
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { h, Host, } from "@stencil/core";
import { requestAnimationFrameNoNgZone } from "../utils/requestAnimationFrame";
export class Tabs {
    constructor() {
        this.windowStartSize = window.innerWidth;
        this.clickAction = {
            timeout: null,
            isClick: true,
        };
        this.small = false;
        this.rounded = false;
        this.selected = 0;
        this.layout = 'auto';
        this.placement = 'bottom';
        this.totalItems = 0;
        this.currentScrollAmount = 0;
        this.scrollAmount = 100;
        this.scrollActionAmount = 0;
        this.showArrowPrevious = false;
        this.showArrowNext = false;
    }
    onWindowResize() {
        this.totalItems = 0;
        this.totalItems = this.getTabs().length;
        if (this.windowStartSize === 0)
            return (this.windowStartSize = window.innerWidth);
        this.move(this.windowStartSize - window.innerWidth);
        this.windowStartSize = window.innerWidth;
    }
    getTabs() {
        return Array.from(this.hostElement.querySelectorAll('ix-tab-item'));
    }
    getTab(tabIndex) {
        return this.getTabs()[tabIndex];
    }
    getTabsWrapper() {
        var _a;
        return (_a = this.hostElement.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector('.items-content');
    }
    initResizeObserver() {
        const parentElement = this.hostElement.parentElement;
        if (!parentElement)
            return;
        this.resizeObserver = new ResizeObserver(() => {
            this.renderArrows();
        });
        this.resizeObserver.observe(parentElement);
    }
    showArrows() {
        try {
            const tabWrapper = this.getTabsWrapper();
            return (tabWrapper &&
                tabWrapper.scrollWidth >
                    Math.ceil(tabWrapper.getBoundingClientRect().width) &&
                this.layout === 'auto');
        }
        catch (error) {
            return false;
        }
    }
    showPreviousArrow() {
        try {
            return this.showArrows() === true && this.scrollActionAmount < 0;
        }
        catch (error) {
            return false;
        }
    }
    showNextArrow() {
        try {
            const tabWrapper = this.getTabsWrapper();
            if (!tabWrapper) {
                return false;
            }
            const tabWrapperRect = tabWrapper.getBoundingClientRect();
            return (this.showArrows() === true &&
                this.scrollActionAmount >
                    (tabWrapper.scrollWidth - tabWrapperRect.width) * -1);
        }
        catch (error) {
            return false;
        }
    }
    move(amount, click = false) {
        const tabsWrapper = this.getTabsWrapper();
        if (!tabsWrapper) {
            return;
        }
        const tabsWrapperVisibleWidth = tabsWrapper.getBoundingClientRect().width;
        const maxScrollWidth = -this.currentScrollAmount +
            tabsWrapperVisibleWidth -
            tabsWrapper.scrollWidth;
        amount = amount < maxScrollWidth ? maxScrollWidth : amount;
        amount += this.currentScrollAmount;
        amount = Math.min(amount, 0);
        const styles = [
            `transform: translateX(${amount}px);`,
            click ? 'transition: all ease-in-out 400ms;' : '',
        ].join('');
        tabsWrapper.setAttribute('style', styles);
        if (click)
            this.currentScrollAmount = this.scrollActionAmount = amount;
        else
            this.scrollActionAmount = amount;
    }
    onSelectedChange(newValue) {
        var _a;
        if (!this.showArrows())
            return;
        const tabRect = this.getTab(newValue).getBoundingClientRect();
        const wrapperWidth = (_a = this.getTabsWrapper()) === null || _a === void 0 ? void 0 : _a.clientWidth;
        const arrowWidth = 32;
        if (tabRect.left < arrowWidth) {
            this.move(-tabRect.left + arrowWidth, true);
        }
        else if (wrapperWidth && tabRect.right > wrapperWidth - arrowWidth) {
            this.move(wrapperWidth - tabRect.right - arrowWidth, true);
        }
    }
    setSelected(index) {
        this.selected = index;
    }
    clickTab(index) {
        if (!this.clickAction.isClick || this.dragStop()) {
            return;
        }
        const { defaultPrevented } = this.selectedChange.emit(index);
        if (defaultPrevented) {
            return;
        }
        this.setSelected(index);
    }
    dragStart(element, event) {
        if (!this.showArrows())
            return;
        if (event.button > 0)
            return;
        this.clickAction.timeout =
            this.clickAction.timeout === null
                ? setTimeout(() => (this.clickAction.isClick = false), 300)
                : null;
        const tabPositionX = parseFloat(window.getComputedStyle(element).left);
        const mousedownPositionX = event.clientX;
        const move = (event) => this.dragMove(event, tabPositionX, mousedownPositionX);
        const windowClick = () => {
            window.removeEventListener('mousemove', move, false);
            window.removeEventListener('click', windowClick, false);
            this.dragStop();
        };
        window.addEventListener('click', windowClick);
        window.addEventListener('mousemove', move, false);
    }
    dragMove(event, tabX, mousedownX) {
        this.move(event.clientX + tabX - mousedownX);
    }
    dragStop() {
        if (this.clickAction.timeout) {
            clearTimeout(this.clickAction.timeout);
            this.clickAction.timeout = null;
        }
        if (this.clickAction.isClick)
            return false;
        this.currentScrollAmount = this.scrollActionAmount;
        this.clickAction.isClick = true;
        return true;
    }
    componentWillLoad() {
        const tabs = this.getTabs();
        tabs.map((element, index) => {
            if (this.small)
                element.setAttribute('small', 'true');
            if (this.rounded)
                element.setAttribute('rounded', 'true');
            element.setAttribute('layout', this.layout);
            element.setAttribute('selected', index === this.selected ? 'true' : 'false');
            element.setAttribute('placement', this.placement);
        });
        this.initResizeObserver();
    }
    componentDidRender() {
        const tabs = this.getTabs();
        this.totalItems = tabs.length;
        tabs.map((element, index) => {
            element.setAttribute('selected', index === this.selected ? 'true' : 'false');
        });
    }
    componentWillRender() {
        this.renderArrows();
    }
    renderArrows() {
        requestAnimationFrameNoNgZone(() => {
            this.showArrowNext = this.showNextArrow();
            this.showArrowPrevious = this.showPreviousArrow();
        });
    }
    componentDidLoad() {
        const tabs = this.getTabs();
        tabs.forEach((element) => {
            element.addEventListener('mousedown', (event) => this.dragStart(element, event));
        });
    }
    disconnectedCallback() {
        var _a;
        (_a = this.resizeObserver) === null || _a === void 0 ? void 0 : _a.disconnect();
    }
    onTabClick(event) {
        if (event.defaultPrevented) {
            return;
        }
        const target = event.target;
        const tabs = this.getTabs();
        tabs.forEach((tab, index) => {
            if (!tab.disabled && tab === target) {
                this.clickTab(index);
            }
        });
    }
    render() {
        return (h(Host, { key: '40618db9c5179c15841a181ebaff044a994fa41b' }, this.showArrowPrevious && (h("div", { key: '6470740adb92447a2c15da33d3e3bca217c4c5d2', class: "arrow", onClick: () => this.move(this.scrollAmount, true) }, h("ix-icon", { key: '4e0775a52a8c7da00f9b07b51accc14ca9ba81bb', name: 'chevron-left-small' }))), h("div", { key: '5acd5d6f79f78ac07b10538c5ad4f1574fc4345c', class: {
                'tab-items': true,
                'overflow-shadow': true,
                'shadow-left': this.showArrowPrevious,
                'shadow-right': this.showArrowNext,
                'shadow-both': this.showArrowNext && this.showArrowPrevious,
            } }, h("div", { key: '5b86235a18ff74dfe29f7aaec0894a10ce94ffd5', class: "items-content" }, h("slot", { key: '39c0c5f320952d5da92e63407c2a2a792c72e44e' }))), this.showArrowNext && (h("div", { key: '82f02f4cccb7edd793a69bf1eb4087c3fab6c75f', class: "arrow right", onClick: () => this.move(-this.scrollAmount, true) }, h("ix-icon", { key: 'c8ef765fbeff3ad9dfac79a4c0188ff67a6c1b76', name: 'chevron-right-small' })))));
    }
    static get is() { return "ix-tabs"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["tabs.scss"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["tabs.css"]
        };
    }
    static get properties() {
        return {
            "small": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Set tab items to small size"
                },
                "attribute": "small",
                "reflect": false,
                "defaultValue": "false"
            },
            "rounded": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Set rounded tabs"
                },
                "attribute": "rounded",
                "reflect": false,
                "defaultValue": "false"
            },
            "selected": {
                "type": "number",
                "mutable": true,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Set default selected tab by index"
                },
                "attribute": "selected",
                "reflect": false,
                "defaultValue": "0"
            },
            "layout": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "'auto' | 'stretched'",
                    "resolved": "\"auto\" | \"stretched\"",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Set layout width style"
                },
                "attribute": "layout",
                "reflect": false,
                "defaultValue": "'auto'"
            },
            "placement": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "'bottom' | 'top'",
                    "resolved": "\"bottom\" | \"top\"",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Set placement style"
                },
                "attribute": "placement",
                "reflect": false,
                "defaultValue": "'bottom'"
            }
        };
    }
    static get states() {
        return {
            "totalItems": {},
            "currentScrollAmount": {},
            "scrollAmount": {},
            "scrollActionAmount": {},
            "showArrowPrevious": {},
            "showArrowNext": {}
        };
    }
    static get events() {
        return [{
                "method": "selectedChange",
                "name": "selectedChange",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "since",
                            "text": "2.0.0"
                        }],
                    "text": "`selected` property changed"
                },
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                }
            }];
    }
    static get elementRef() { return "hostElement"; }
    static get watchers() {
        return [{
                "propName": "selected",
                "methodName": "onSelectedChange"
            }];
    }
    static get listeners() {
        return [{
                "name": "resize",
                "method": "onWindowResize",
                "target": "window",
                "capture": false,
                "passive": true
            }, {
                "name": "tabClick",
                "method": "onTabClick",
                "target": undefined,
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=tabs.js.map
