export default class Node {
    constructor(nodeType, nodeName, nodeValue) {
        this.nodeType = nodeType;
        this.nodeName = nodeName;
        this.nodeValue = nodeValue;
        this.parentNode = null;
        this.previousSibling = null;
        this.nextSibling = null;
        this.firstChild = null;
        this.lastChild = null;
        this._childNodes = undefined;
    }
    get childNodes() {
        let children = this._childNodes;
        if (children === undefined) {
            children = this._childNodes = new ChildNodes(this);
        }
        return children;
    }
    cloneNode(deep) {
        const node = this._cloneNode();
        if (deep === true) {
            let child = this.firstChild;
            let nextChild = child;
            while (child !== null) {
                nextChild = child.nextSibling;
                node.appendChild(child.cloneNode(true));
                child = nextChild;
            }
        }
        return node;
    }
    appendChild(newChild) {
        if (newChild.nodeType === 11 /* DOCUMENT_FRAGMENT_NODE */) {
            insertFragment(newChild, this, this.lastChild, null);
            return newChild;
        }
        if (newChild.parentNode) {
            newChild.parentNode.removeChild(newChild);
        }
        newChild.parentNode = this;
        const refNode = this.lastChild;
        if (refNode === null) {
            this.firstChild = newChild;
            this.lastChild = newChild;
        }
        else {
            newChild.previousSibling = refNode;
            refNode.nextSibling = newChild;
            this.lastChild = newChild;
        }
        return newChild;
    }
    insertBefore(newChild, refChild) {
        if (refChild == null) {
            return this.appendChild(newChild);
        }
        if (newChild.nodeType === 11 /* DOCUMENT_FRAGMENT_NODE */) {
            insertFragment(newChild, this, refChild ? refChild.previousSibling : null, refChild);
            return newChild;
        }
        if (newChild.parentNode) {
            newChild.parentNode.removeChild(newChild);
        }
        newChild.parentNode = this;
        const previousSibling = refChild.previousSibling;
        if (previousSibling) {
            previousSibling.nextSibling = newChild;
            newChild.previousSibling = previousSibling;
        }
        else {
            newChild.previousSibling = null;
        }
        refChild.previousSibling = newChild;
        newChild.nextSibling = refChild;
        if (this.firstChild === refChild) {
            this.firstChild = newChild;
        }
        return newChild;
    }
    removeChild(oldChild) {
        if (this.firstChild === oldChild) {
            this.firstChild = oldChild.nextSibling;
        }
        if (this.lastChild === oldChild) {
            this.lastChild = oldChild.previousSibling;
        }
        if (oldChild.previousSibling) {
            oldChild.previousSibling.nextSibling = oldChild.nextSibling;
        }
        if (oldChild.nextSibling) {
            oldChild.nextSibling.previousSibling = oldChild.previousSibling;
        }
        oldChild.parentNode = null;
        oldChild.nextSibling = null;
        oldChild.previousSibling = null;
        return oldChild;
    }
    _cloneNode() {
        return new Node(this.nodeType, this.nodeName, this.nodeValue);
    }
}
Node.ELEMENT_NODE = 1 /* ELEMENT_NODE */;
Node.TEXT_NODE = 3 /* TEXT_NODE */;
Node.COMMENT_NODE = 8 /* COMMENT_NODE */;
Node.DOCUMENT_NODE = 9 /* DOCUMENT_NODE */;
Node.DOCUMENT_FRAGMENT_NODE = 11 /* DOCUMENT_FRAGMENT_NODE */;
function insertFragment(fragment, newParent, before, after) {
    if (!fragment.firstChild) {
        return;
    }
    const firstChild = fragment.firstChild;
    let lastChild = firstChild;
    let node = firstChild;
    firstChild.previousSibling = before;
    if (before) {
        before.nextSibling = firstChild;
    }
    else {
        newParent.firstChild = firstChild;
    }
    while (node) {
        node.parentNode = newParent;
        lastChild = node;
        node = node.nextSibling;
    }
    lastChild.nextSibling = after;
    if (after) {
        after.previousSibling = lastChild;
    }
    else {
        newParent.lastChild = lastChild;
    }
}
class ChildNodes {
    constructor(node) {
        this.node = node;
    }
    item(index) {
        let child = this.node.firstChild;
        for (let i = 0; child && index !== i; i++) {
            child = child.nextSibling;
        }
        return child;
    }
}
//# sourceMappingURL=node.js.map