export interface ThreeDChartProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'.
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the rate of the animation. The default animation rate is 1
   * Default value: 1
   */
  animationSpeed?: number;
  /**
   * Sets whether the chart will rotate automatically.
   * Default value: false
   */
  autoRotate?: boolean;
  /**
   * Sets the speed of the automatic rotation.
   * Default value: 1
   */
  autoRotateSpeed?: number;
  /**
   * Sets the chart's background color. For example: '#DDFFE8'
   * Default value: #ffffff
   */
  backgroundColor?: string | null;
  /**
   * Sets the chart's background to a static linear gradient. The property must be set to an Array of Strings in the format: 'offset, color'
   * Default value: []
   */
  backgroundGradient?: any[];
  /**
   * Sets the chart's background to a static image. For example: 'https://www.htmlelements.com/demos/images/stars.jpg'
   * Default value: ""
   */
  backgroundImage?: string;
  /**
   * Sets the chart's background to a dynamic background image which rotates with the camera. The property must be set an Array of 6 images. All images must have aspect ratio 1:1
   * Default value: []
   */
  backgroundTexture?: any[];
  /**
   * Sets the camera's position. The property must be set to an {x, y, z} object.
   * Default value: [object Object]
   */
  cameraPosition?: ThreeDChartCameraPosition;
  /**
   * Sets the intial camera zoom. The default value is 1
   * Default value: 1
   */
  cameraZoom?: number;
  /**
   * Sets the caption (title) of the chart.
   * Default value: "Caption"
   */
  caption?: string;
  /**
   * Sets the chart's color pallete. jqxChart suppports 32 color schemes from 'scheme01' to 'scheme32'.
   * Default value: scheme01
   */
  colorScheme?: ThreeDChartColorScheme | string;
  /**
   * Sets the chart's controls settings.
   * Default value: [object Object]
   */
  controlsSettings?: any;
  /**
   * Allows substituting default items with custom 3D Objects. The property must be set to an Array of Objects in the format: { groupIndex, serieIndex, itemIndex, modelUrl }
   * Default value: []
   */
  customModels?: any[];
  /**
   * Sets the chart's data source.
   * Default value: 
   */
  dataSource?: any[];
  /**
   * Sets the description text of the chart.
   * Default value: "Description"
   */
  description?: string;
  /**
   * Enables or disables the chart.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets whether the chart's toolbar is enabled.
   * Default value: true
   */
  enableControlsToolbar?: boolean;
  /**
   * 
   * Default value: reset-camera,zoom-in,zoom-out,save-image,camera-control
   */
  controlsToolbarItems?: any;
  /**
   * Sets the chart's grid options.
   * Default value: [object Object]
   */
  gridOptions?: any;
  /**
   * Sets whether the legend will be created based on the chart's series or serie groups. "auto" - the legend index will change depending on the Chart type
   * Default value: auto
   */
  legendIndex?: ThreeDChartLegendIndex | string;
  /**
   * Sets the legend's layout.
   * Default value: [object Object]
   */
  legendLayout?: any;
  /**
   * Sets the light color of the 3D Scene.
   * Default value: "#ffffff"
   */
  lightColor?: string;
  /**
   * Sets or gets the locale. Used in conjunction with the property messages.
   * Default value: "en"
   */
  locale?: string;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale.
   * Default value:    * [object Object]
   */
  messages?: any;
  /**
   * Sets or gets a value indicating whether the Chart's layout is mirrored.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the selection mode.
   * Default value: multiple
   */
  selectionMode?: ThreeDChartSelectionMode | string;
  /**
   * The seriesGroups property is used to describe all series displayed on the chart. jqxChart supports multiple series of different types and series grouping. Each series group may have its own Value Axis (Y-axis) which allows you to have values with different scales displayed on the same chart at the same time. It also allows you to display multiple series types together on the same chart. For example, you can display all series in one group as lines and the series in a second group as columns. seriesGroups is an array of objects where each object represents one group.
   * Default value: 
   */
  seriesGroups?: ThreeDChartSeriesGroup[];
  /**
   * Determines whether to show grid connecting lines when a chart item is hovered over.
   * Default value: false
   */
  showConnectionLines?: boolean;
  /**
   * Determines whether to show or hide the chart series legend.
   * Default value: true
   */
  showLegend?: boolean;
  /**
   * Determines whether to show or hide the chart series legend table.
   * Default value: false
   */
  showLegendTable?: boolean;
  /**
   * Enables or disables the chart tooltips.
   * Default value: true
   */
  showToolTips?: boolean;
  /**
   * Sets the padding of the chart's title (caption).
   * Default value: [object Object]
   */
  titlePadding?: ThreeDChartPadding;
  /**
   * User defined tooltip text formatting callback function.
   * Default value: null
   */
  toolTipFormatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Tooltip data formatting settings for the values in the serie.
   * Default value: [object Object]
   */
  toolTipFormatSettings?: ThreeDChartFormatSettings;
  /**
   * Tooltip line color. By default it is set to the hovered item's color
   * Default value: null
   */
  toolTipLineColor?: string | null;
  /**
   * An object with settings about the Chart's y-axis (value axis).
   * Default value: [object Object]
   */
  valueAxis?: ThreeDChartValueAxis;
  /**
   * Sets the Chart's xAxis.
   * Default value: [object Object]
   */
  xAxis?: ThreeDChartXAxis;
  /**
   * Sets the Chart's zAxis.
   * Default value: [object Object]
   */
  zAxis?: ThreeDChartZAxis;
}
/**
 3D Chart is a feature-complete interactive WebGL 3D graph library that answers the data visualization needs of any modern web app.
*/
export interface ThreeDChart extends BaseElement, ThreeDChartProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * The event is raised when the user clicks on a chart element.
	* @param event. The custom event. Custom data event was created with: ev.detail(itemIndex, serieIndex, groupIndex)
   *  itemIndex - The item index of the item.
   *  serieIndex - The serie index of the item.
   *  groupIndex - The group index of the item.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when a chart element is shown.
	* @param event. The custom event. Custom data event was created with: ev.detail(itemIndex, serieIndex, groupIndex)
   *  itemIndex - The item index of the item.
   *  serieIndex - The serie index of the item.
   *  groupIndex - The group index of the item.
   */
  onShow?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when a chart element is hidden.
	* @param event. The custom event. Custom data event was created with: ev.detail(itemIndex, serieIndex, groupIndex)
   *  itemIndex - The item index of the item.
   *  serieIndex - The serie index of the item.
   *  groupIndex - The group index of the item.
   */
  onHide?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when a chart element is selected.
	* @param event. The custom event. Custom data event was created with: ev.detail(itemIndex, serieIndex, groupIndex)
   *  itemIndex - The item index of the item.
   *  serieIndex - The serie index of the item.
   *  groupIndex - The group index of the item.
   */
  onSelect?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when a chart element is unselected.
	* @param event. The custom event. Custom data event was created with: ev.detail(itemIndex, serieIndex, groupIndex)
   *  itemIndex - The item index of the item.
   *  serieIndex - The serie index of the item.
   *  groupIndex - The group index of the item.
   */
  onUnselect?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised after the chart's range selector position changes and after the chart ends rendering.
	* @param event. The custom event. Custom data event was created with: ev.detail(minValue, maxValue)
   *  minValue - The start value of the range selector.
   *  maxValue - The end value of the range selector.
   */
  onRangeSelectionChanged?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the chart's range selector position changes and before the chart starts rendering. The event can be default prevented to cancel the range selection change.
	* @param event. The custom event. Custom data event was created with: ev.detail(minValue, maxValue, oldMinValue, oldMaxValue)
   *  minValue - The start value of the range selector.
   *  maxValue - The end value of the range selector.
   *  oldMinValue - The previous start value of the range selector.
   *  oldMaxValue - The previous end value of the range selector.
   */
  onRangeSelectionChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the chart begins rendering.
	* @param event. The custom event.    */
  onRefreshBegin?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the chart finishes rendering.
	* @param event. The custom event.    */
  onRefreshEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the chart begins resizing.
	* @param event. The custom event.    */
  onResizeBegin?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the chart finishes resizing.
	* @param event. The custom event.    */
  onResizeEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Adds a new color sheme. If a scheme with the same name already exists, the method will update its colors.
   * @param {string} schemeName. The name of the custom color scheme.
   * @param {any[]} colorsArray. An array of color values.
   */
  addColorScheme(schemeName: string, colorsArray: any[]): void;
  /**
   * Begins an update of the chart. The chart will not be rendered until the endUpdate method is called.
   */
  beginUpdate(): void;
  /**
   * Ends an update of the chart. The chart will be rendered after the endUpdate method is called.
   * @param {boolean} refresh?. If set to true, the chart will complete a full refresh.
   */
  endUpdate(refresh?: boolean): void;
  /**
   * Returns the colors of a color scheme by name. If the scheme doesn't exist the method returns undefined.
   * @param {string} schemeName. The name of the color scheme.
   * @returns {any[]}
   */
  getColorScheme(schemeName: string): any[];
  /**
   * Gets the item with the specified indexes.
   * @param {number | null} groupIndex?. Series group index.
   * @param {number | null} serieIndex?. Series index.
   * @param {number | null} itemIndex?. Item (data point) index.
   * @returns {any}
   */
  getItemByIndexes(groupIndex?: number | null, serieIndex?: number | null, itemIndex?: number | null): any;
  /**
   * Gets an arrat of the items with the specified indexes. Leaving an index null will return all items that match the other indexes.
   * @param {number | null} groupIndex?. Series group index.
   * @param {number | null} serieIndex?. Series index.
   * @param {number | null} itemIndex?. Item (data point) index.
   * @returns {any}
   */
  getItemsByIndexes(groupIndex?: number | null, serieIndex?: number | null, itemIndex?: number | null): any;
  /**
   * Gets the indexes of the hidden series.
   * @returns {{ groupIndex: number, serieIndex: number, itemIndex: number }[]}
   */
  getHidden(): { groupIndex: number, serieIndex: number, itemIndex: number }[];
  /**
   * Gets the selected items.
   * @returns {any[]}
   */
  getSelection(): any[];
  /**
   * Gets the rendered values of the valueAxis labels.
   * @returns {any}
   */
  getValueAxisLabels(): any;
  /**
   * Gets the rendered values of the xAxis labels.
   * @returns {any}
   */
  getXAxisLabels(): any;
  /**
   * Gets the rendered values of the zAxis labels.
   * @returns {any}
   */
  getZAxisLabels(): any;
  /**
   * Hides all items of a chart group.
   * @param {number} groupIndex. Series group index.
   */
  hideGroup(groupIndex: number): void;
  /**
   * Hides a chart item.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   * @param {number} itemIndex?. Item (data point) index.
   */
  hideItem(groupIndex: number, serieIndex: number, itemIndex?: number): void;
  /**
   * Hides all items of a chart serie.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   */
  hideSerie(groupIndex: number, serieIndex: number): void;
  /**
   * Refreshes the content of the chart element after a property or data update.
   */
  refresh(): void;
  /**
   * Removes an existing color scheme. If the scheme does not exist, the method has no effect.
   * @param {string} schemeName. The name of the custom color scheme.
   */
  removeColorScheme(schemeName: string): void;
  /**
   * Exports the chart's content as JPEG image.
   * @param {string} fileName?. File name.
   * @param {boolean} includeLegend?. Sets whether the legend will be part of the saved file.
   * @param {boolean} includeCaption?. Sets whether the caption will be part of the saved file.
   */
  saveAsJPEG(fileName?: string, includeLegend?: boolean, includeCaption?: boolean): void;
  /**
   * Exports the chart's content as PNG image.
   * @param {string} fileName?. File name.
   * @param {boolean} includeLegend?. Sets whether the legend will be part of the saved file.
   * @param {boolean} includeCaption?. Sets whether the caption will be part of the saved file.
   */
  saveAsPNG(fileName?: string, includeLegend?: boolean, includeCaption?: boolean): void;
  /**
   * Exports the chart's content as PDF file.
   * @param {string} fileName?. File name.
   * @param {boolean} includeLegend?. Sets whether the legend will be part of the saved file.
   * @param {boolean} includeCaption?. Sets whether the caption will be part of the saved file.
   */
  saveAsPDF(fileName?: string, includeLegend?: boolean, includeCaption?: boolean): void;
  /**
   * Selects a chart item. If selectionMode is 'one', the previous item will be unselected.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   * @param {number} itemIndex?. Item (data point) index.
   */
  selectItem(groupIndex: number, serieIndex: number, itemIndex?: number): void;
  /**
   * Shows all items of a chart group.
   * @param {number} groupIndex. Series group index.
   */
  showGroup(groupIndex: number): void;
  /**
   * Shows a chart item.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   * @param {number} itemIndex?. Item (data point) index.
   */
  showItem(groupIndex: number, serieIndex: number, itemIndex?: number): void;
  /**
   * Shows all items of a chart serie.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   */
  showSerie(groupIndex: number, serieIndex: number): void;
  /**
   * Sets the camera position to its position during the initialization.
   */
  setDefaultPosition(): void;
  /**
   * Sets the camera mode. Different camera modes change the control actions of the mouse. Available modes are 'zoom', 'pan' and 'default'.
   * @param {string} mode. Camera mode.
   */
  setCameraMode(mode: string): void;
  /**
   * Sets the camera position.
   * @param {number} x. X coordinate.
   * @param {number} y. Y coordinate.
   * @param {number} z. Z coordinate.
   * @param {boolean} animation?. Animation Enabled
   */
  setCameraPosition(x: number, y: number, z: number, animation?: boolean): void;
  /**
   * Sets the camera zoom.
   * @param {number} level. Zoom level.
   * @param {boolean} animation?. Animation Enabled
   */
  setCameraZoom(level: number, animation?: boolean): void;
  /**
   * Unelects a chart item.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   * @param {number} itemIndex?. Item (data point) index.
   */
  unselectItem(groupIndex: number, serieIndex: number, itemIndex?: number): void;
  /**
   * Updates the values of the chart series without full refresh of the entire chart. The method should be used for animation of frequently changing values.
   */
  update(): void;
}

/**Sets the camera's position. The property must be set to an {x, y, z} object. */
export interface ThreeDChartCameraPosition {
  /**
   * Sets the camera's x position. This is the left-right axis
   * Default value: 5
   */
  x?: number;
  /**
   * Sets the camera's y position. This is the up-down axis
   * Default value: 20
   */
  y?: number;
  /**
   * Sets the camera's z position. This is the forward-backward axis
   * Default value: 55
   */
  z?: number;
}

export interface ThreeDChartSeriesGroup {
  /**
   * Optional color bands dislayed in the chart's plot area.
   * Default value: null
   */
  bands?: ThreeDChartBand[];
  /**
   * Determines the data source of the serie
   * Default value: null
   */
  dataSource?: any;
  /**
   * Sets the Z-axis width of the series group. Can be set as number or percetage of the slot's width.
   * Default value: 2
   */
  depth?: number | string;
  /**
   * Callback function used to format series labels.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Object describing the format settings of series labels.
   * Default value: [object Object]
   */
  formatSettings?: ThreeDChartFormatSettings;
  /**
   * Sets the opacity of the series group.
   * Default value: 1
   */
  opacity?: number;
  /**
   * An array of chart series.
   * Default value: 
   */
  series?: ThreeDChartSeriesGroupSerie[];
  /**
   * When showLabels is set to true, the chart will render pie labels.
   * Default value: true
   */
  showLabels?: boolean;
  /**
   * The start angle (in degrees) of the polar coordinate system. Applicable to polar and spider charts only.
   * Default value: 0
   */
  startAngle?: number;
  /**
   * Sets the chart type. jqxChart supports several common chart types. You can easily plot series of different types on a common chart. A type must be specified for each series group. Currently, jqxChart supports the following series types:'column' - simple column series'stackedcolumn' - stacked column series'stackedcolumn100' - percentage stacked columns'rangecolumn' - floating column between two values'waterfall' - waterfall series'stackedwaterfall' - stacked waterfall series'line' - simple straight lines connecting the value points'stackedline' - stacked lines'stackedline100' - percentage stacked lines'spline' - smooth lines connecting the value points'stackedspline' - smooth stacked lines'stackedspline100' - percentage stacked smooth lines'stepline' - step line'stackedstepline' - stacked step line'stackedstepline100' - percentage stacked step line'area' - area connecting the value points with straight lines'stackedarea' - stacked area with straight lines between the points'stackedarea100' - percentage stacked area with straight lines between the points'rangearea' - floating area between pairs of value points'splinearea' - smooth area connecting the value points'stackedsplinearea' - stacked smooth area connecting the value points'stackedsplinearea100' - percentage stacked smooth area'splinerangearea' - smooth floating area between pairs of value points'steprangearea' - step area between pairs of value points'stackedsplineara' - smooth stacked area'steparea' - step area connecting the value points'stackedsteparea' - step stacked area'stackedsteparea100' - percentage stacked step area'pie' - circular chart divided into sectors, illustrating proportion'donut' - chart divided into circular sectors with different inner and outer radius'scatter' - data is displayed as a collection of points'stackedscatter' - data is displayed as a collection of points and the values are stacked'stackedscatter100' - data is displayed as a collection of points and the values are percentage stacked'bubble' - data is displayed as a collection of bubbles'stackedbubble' - data is displayed as a collection of bubbles and the values are stacked'stackedbubble100' - data is displayed as a collection of bubbles and the values are percentage stacked'candlestick' - display candlestick series using open, high, low, close data points'ohlc' - display OHLC series using open, high, low, close data points
   * Default value: column
   */
  type?: ThreeDChartType | string;
  /**
   * Sets the Y-axis width of the series group. Used to set the thickness of the line series
   * Default value: 2
   */
  verticalWidth?: number;
  /**
   * Sets the X-axis width of the series group. Can be set as number or percetage of the slot's width.
   * Default value: 2
   */
  width?: number | string;
}

export interface ThreeDChartBand {
  /**
   * Color used to fill the area between the minValue and the maxValue.
   * Default value: null
   */
  color?: string | null;
  /**
   * Band line color.
   * Default value: null
   */
  lineColor?: string | null;
  /**
   * Band line width.
   * Default value: null
   */
  lineWidth?: string | number | null;
  /**
   * End value of the color band.
   * Default value: NaN
   */
  maxValue?: any;
  /**
   * Start value of the color band.
   * Default value: NaN
   */
  minValue?: any;
  /**
   * Fraction indicating the fill opacity.
   * Default value: 1
   */
  opacity?: number;
}

/**Object describing the format settings of series labels. */
export interface ThreeDChartFormatSettings {
  /**
   * Optional date format string. This property is applicable only when displaying Date objects.
   * Default value: null
   */
  dateFormat?: string | null;
  /**
   * Decimal places of numeric values.
   * Default value: null
   */
  decimalPlaces?: number | null;
  /**
   * A symbol used to mark the border between the integer and fractional parts of a number. The default value is inherited from localization.decimalSeparator.
   * Default value: null
   */
  decimalSeparator?: string | null;
  /**
   * A boolean value indicating whether to display negative numbers in brackets.
   * Default value: false
   */
  negativeWithBrackets?: boolean;
  /**
   * Text to prepend to the value.
   * Default value: ""
   */
  prefix?: string;
  /**
   * Text to append to the value
   * Default value: ""
   */
  sufix?: string;
  /**
   * A symbol used to mark the border between thousands, millions, billions, etc. The default value is inherited from localization.thousandsSeparator.
   * Default value: null
   */
  thousandsSeparator?: string | null;
}

export interface ThreeDChartSeriesGroupSerie {
  /**
   * A custom function that returns the color of a data point. The function will receive the following parameters: dataValue, itemIndex, serie, group. The implementation of the function can return a single color which will be used as a fillColor and the other colors will be derived or it can return an object containing fillColor, fillColorSelected, etc.
   * Default value: null
   */
  colorFunction?: any;
  /**
   * Name of the field in the data source.
   * Default value: "null"
   */
  dataField?: string;
  /**
   * Data field used in candlestcik and ohlc series.
   * Default value: "null"
   */
  dataFieldClose?: string;
  /**
   * Data field used in candlestcik and ohlc series.
   * Default value: "null"
   */
  dataFieldHigh?: string;
  /**
   * Data field used in candlestcik and ohlc series.
   * Default value: "null"
   */
  dataFieldLow?: string;
  /**
   * Data field used in candlestcik and ohlc series.
   * Default value: "null"
   */
  dataFieldOpen?: string;
  /**
   * Data field used in in range column series as a start data field.
   * Default value: "null"
   */
  dataFieldFrom?: string;
  /**
   * Data field used in range column series as an end data field.
   * Default value: "null"
   */
  dataFieldTo?: string;
  /**
   * Data field used in bubble series.
   * Default value: "null"
   */
  radiusDataField?: string;
  /**
   * Name to display for this serie.
   * Default value: "null"
   */
  displayText?: string;
  /**
   * Name to display for candlestick and ohlc series.
   * Default value: "null"
   */
  displayTextClose?: string;
  /**
   * Name to display for candlestick and ohlc series.
   * Default value: "null"
   */
  displayTextHigh?: string;
  /**
   * Name to display for candlestick and ohlc series.
   * Default value: "null"
   */
  displayTextLow?: string;
  /**
   * Name to display for candlestick and ohlc series.
   * Default value: "null"
   */
  displayTextOpen?: string;
  /**
   * Callback function used to format the labels.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Object describing the format settings of the labels.
   * Default value: [object Object]
   */
  formatSettings?: ThreeDChartFormatSettings;
  /**
   * Specifies the summary of the series group.
   * Default value: ""
   */
  summary?: string;
  /**
   * Maximum angle in a pie, donut, polar and spider series.
   * Default value: 360
   */
  endAngle?: number;
  /**
   * Initial angle in pie and donut series.
   * Default value: 0
   */
  initialAngle?: number;
  /**
   * Inner radius of donut series in pixels or percents.
   * Default value: 0
   */
  innerRadius?: number;
  /**
   * Object describing the labels properties of the axis.
   * Default value: undefined
   */
  labels?: ThreeDChartLabels;
  /**
   * Fill color of the legend box. The default value is inherited from the serie's color.
   * Default value: null
   */
  legendFillColor?: string | null;
  /**
   * Legend data formatting function for the values in the serie.
   * Default value: null
   */
  legendFormatFunction?: any;
  /**
   * Legend data formatting settings for the values in the serie.
   * Default value: [object Object]
   */
  legendFormatSettings?: ThreeDChartFormatSettings;
  /**
   * Line color of the legend box. The default value is inherited from the serie's color.
   * Default value: null
   */
  legendLineColor?: string | null;
  /**
   * Line color for the serie.
   * Default value: null
   */
  lineColor?: string | null;
  /**
   * Line color for the serie when selected.
   * Default value: null
   */
  lineColorSelected?: string | null;
  /**
   * Line color for the marker symbols in serie.
   * Default value: null
   */
  lineColorSymbol?: string | null;
  /**
   * Determines the line tickness of the items in this serie.
   * Default value: null
   */
  lineWidth?: string | null | number;
  /**
   * Determines the line tickness of the items in this serie when selected.
   * Default value: null
   */
  lineWidthSelected?: string | null | number;
  /**
   * Min radius of bubble series in pixels or percents.
   * Default value: null
   */
  minRadius?: string | null | number;
  /**
   * Max radius of bubble series in pixels or percents.
   * Default value: null
   */
  maxRadius?: string | null | number;
  /**
   * Determines the opacity of the items in this serie.
   * Default value: 1
   */
  opacity?: number;
  /**
   * Outer radius of pie and donut series in pixels or percents.
   * Default value: null
   */
  radius?: number | null;
  /**
   * Radius change on selection of pie and donut series in pixels or percents.
   * Default value: null
   */
  selectedRadiusChange?: number | null;
  /**
   * Minimum angle in a pie, donut, polar and spider series.
   * Default value: 0
   */
  startAngle?: number;
  /**
   * Determines the size of the symbol element.
   * Default value: null
   */
  symbolSize?: number;
  /**
   * Determines the size of the symbol element. This property is applicable to line and area series only.
   * Default value: null
   */
  symbolSizeSelected?: number;
  /**
   * Determines the symbol type displayed for the data points in the serie. This parameter is applicable to line, area, scatter and bubble series only.
   * Default value: none
   */
  symbolType?: ThreeDChartSymbolType | string;
  /**
   * Determines the tooltip's background.
   * Default value: null
   */
  toolTipBackground?: string | null;
  /**
   * Determines the tooltip's CSS class name.
   * Default value: null
   */
  toolTipClass?: string | null;
  /**
   * Tooltip data formatting function for the values in the serie.
   * Default value: null
   */
  toolTipFormatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Tooltip data formatting settings for the values in the serie.
   * Default value: [object Object]
   */
  toolTipFormatSettings?: ThreeDChartFormatSettings;
  /**
   * Determines the tooltip's border lines color.
   * Default value: null
   */
  toolTipLineColor?: string | null;
  /**
   * Determines whether to use color gradients.
   * Default value: false
   */
  useGradientColors?: boolean;
}

/**Object describing the labels properties of the axis. */
export interface ThreeDChartLabels {
  /**
   * Text rotation angle.
   * Default value: 0
   */
  angle?: number;
  /**
   * Callback function used to format the labels.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Object describing the format settings of the labels.
   * Default value: [object Object]
   */
  formatSettings?: ThreeDChartFormatSettings;
  /**
   * Labels offset.
   * Default value: [object Object]
   */
  offset?: ThreeDChartOffset;
  /**
   * Radius of the labels in pie/donut series.
   * Default value: null
   */
  radius?: number | null;
  /**
   * Interval steps between label placements (multiples of the axis unit interval).
   * Default value: null
   */
  step?: number | null;
  /**
   * Sets the interval between the labels.
   * Default value: null
   */
  unitInterval?: number;
  /**
   * Possible values: true, false, 'custom'.Determines the visibility of labels. When 'custom' is set, displays only custom values/offsets from the labels.custom array.
   * Default value: true
   */
  visible?: boolean | string;
}

/**Labels offset. */
export interface ThreeDChartOffset {
  /**
   * Horizontal offset.
   * Default value: 0
   */
  x?: number | null;
  /**
   * Vertical offset.
   * Default value: 0
   */
  y?: number | null;
  /**
   * Horizontal offset on Z-axis.
   * Default value: 0
   */
  z?: number | null;
}

/**Sets the padding of the chart's title (caption). */
export interface ThreeDChartPadding {
  /**
   * Bottom padding.
   * Default value: 10
   */
  bottom?: number;
  /**
   * Left padding.
   * Default value: 5
   */
  left?: number;
  /**
   * Right padding.
   * Default value: 5
   */
  right?: number;
  /**
   * Top padding.
   * Default value: 5
   */
  top?: number;
}

/**An object with settings about the Chart's y-axis (value axis). */
export interface ThreeDChartValueAxis {
  /**
   * Sets the baseline value for the axis.
   * Default value: 0
   */
  baselineValue?: any;
  /**
   * Sets the text displayed on the axis.
   * Default value: null
   */
  displayText?: string | null;
  /**
   * Custom function to format the displayed values along the axis.
   * Default value: null
   */
  formatFunction?: {(value?: any): string};
  /**
   * Settings used to format the displayed values along the axis.
   * Default value: [object Object]
   */
  formatSettings?: ThreeDChartFormatSettings;
  /**
   * Object describing the grid lines properties of the valueAxis.
   * Default value: [object Object]
   */
  gridLines?: ThreeDChartLines;
  /**
   * Object describing the labels properties of the axis.
   * Default value: [object Object]
   */
  labels?: ThreeDChartLabels;
  /**
   * Determines whether to use logarithmic scale.
   * Default value: false
   */
  logarithmicScale?: boolean;
  /**
   * Base for logarithmic scale.
   * Default value: 10
   */
  logarithmicScaleBase?: number;
  /**
   * Sets the maximum value of the valueAxis.
   * Default value: NaN
   */
  maxValue?: any;
  /**
   * Sets the minimum value of the valueAxis.
   * Default value: NaN
   */
  minValue?: any;
  /**
   * Sets the interval between the units.
   * Default value: null
   */
  unitInterval?: number | null;
  /**
   * Shows or hides the valueAxis.
   * Default value: true
   */
  visible?: boolean;
}

/**Object describing the grid lines properties of the valueAxis. */
export interface ThreeDChartLines {
  /**
   * Color of the grid lines.
   * Default value: ""
   */
  color?: string;
  /**
   * Interval steps between grid line placements (multiples of the axis unit interval).
   * Default value: null
   */
  step?: number | null;
  /**
   * Possible values: true, false, 'custom'.Determines the visibility of grid lines. When 'custom' is set, displays only custom values/offsets from the gridLines.custom array.
   * Default value: true
   */
  visible?: boolean | string;
}

/**Sets the Chart's xAxis. */
export interface ThreeDChartXAxis {
  /**
   * Points to a data field in the data source.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Optional custom xAxis display text.
   * Default value: null
   */
  displayText?: string | null;
  /**
   * Specifies whether the values are displayed in reverse order.
   * Default value: false
   */
  flip?: boolean;
  /**
   * Custom function to format the displayed values along the axis.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Settings used to format the displayed values along the axis.
   * Default value: [object Object]
   */
  formatSettings?: ThreeDChartFormatSettings;
  /**
   * Object describing the grid lines properties of the xAxis.
   * Default value: [object Object]
   */
  gridLines?: ThreeDChartLines;
  /**
   * Object describing the labels properties of the axis.
   * Default value: [object Object]
   */
  labels?: ThreeDChartLabels;
  /**
   * Determines whether to use logarithmic scale.
   * Default value: false
   */
  logarithmicScale?: boolean;
  /**
   * Base for logarithmic scale.
   * Default value: 10
   */
  logarithmicScaleBase?: number;
  /**
   * Sets the maximum value of the xAxis.
   * Default value: NaN
   */
  maxValue?: any;
  /**
   * Sets the minimum value of the xAxis.
   * Default value: NaN
   */
  minValue?: any;
  /**
   * Definition of a range selector on the xAxis. The range selector itself is also an instance of smart-chart.
   * Default value: [object Object]
   */
  rangeSelector?: ThreeDChartRangeSelector;
  /**
   * The type of the axis. 'auto' - automatically detects and switches to 'basic', 'linear' or 'date'.'date' - when displaying dates.'basic' - displays all data points sequentially.'linear' - linear arrangement by the value of the xAxis data field.
   * Default value: auto
   */
  type?: ThreeDChartXAxisType | string;
  /**
   * Sets the interval between the units.
   * Default value: null
   */
  unitInterval?: number | null;
  /**
   * Shows or hides the xAxis.
   * Default value: true
   */
  visible?: boolean;
}

/**Definition of a range selector on the xAxis. The range selector itself is also an instance of ${namespace.toLowerCase()}-chart. */
export interface ThreeDChartRangeSelector {
  /**
   * Sets the color of the range selector chart. If null, it will be set to the same color as the chart
   * Default value: null
   */
  color?: string | null;
  /**
   * Callback function used to format the values.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Chart Format Settings
   * Default value: null
   */
  formatSettings?: any;
  /**
   * Object describing the labels properties of the axis.
   * Default value: [object Object]
   */
  labels?: ThreeDChartLabels;
  /**
   * Sets the opacity of the range selector chart.
   * Default value: 1
   */
  opacity?: number;
  /**
   * Sets the range selector chart type
   * Default value: area
   */
  serieType?: ThreeDChartRangeSelectorSerieType | string;
  /**
   * Shows or hides the range selector.
   * Default value: false
   */
  visible?: boolean;
}

/**Sets the Chart's zAxis. */
export interface ThreeDChartZAxis {
  /**
   * Points to a data field in the data source.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Optional custom zAxis display text.
   * Default value: null
   */
  displayText?: string | null;
  /**
   * Specifies whether the values are displayed in reverse order.
   * Default value: false
   */
  flip?: boolean;
  /**
   * Custom function to format the displayed values along the axis.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Settings used to format the displayed values along the axis.
   * Default value: [object Object]
   */
  formatSettings?: ThreeDChartFormatSettings;
  /**
   * Object describing the grid lines properties of the zAxis.
   * Default value: [object Object]
   */
  gridLines?: ThreeDChartLines;
  /**
   * Object describing the labels properties of the axis.
   * Default value: [object Object]
   */
  labels?: ThreeDChartLabels;
  /**
   * Determines whether to use logarithmic scale.
   * Default value: false
   */
  logarithmicScale?: boolean;
  /**
   * Base for logarithmic scale.
   * Default value: 10
   */
  logarithmicScaleBase?: number;
  /**
   * Sets the maximum value of the zAxis.
   * Default value: NaN
   */
  maxValue?: any;
  /**
   * Sets the minimum value of the zAxis.
   * Default value: NaN
   */
  minValue?: any;
  /**
   * Sets the interval between the units.
   * Default value: null
   */
  unitInterval?: number | null;
  /**
   * Shows or hides the zAxis.
   * Default value: true
   */
  visible?: boolean;
}

declare global {
    interface Document {
        createElement(tagName: "smart-3d-chart"): ThreeDChart;
        querySelector(selectors: "smart-3d-chart"): ThreeDChart | null;
        querySelectorAll(selectors: "smart-3d-chart"): NodeListOf<ThreeDChart>;
        getElementsByTagName(qualifiedName: "smart-3d-chart"): HTMLCollectionOf<ThreeDChart>;
        getElementsByName(elementName: "smart-3d-chart"): NodeListOf<ThreeDChart>;
    }
}

/**Sets or gets the animation mode. Animation is disabled when the property is set to 'none'. */
export declare type Animation = 'none' | 'simple' | 'advanced';
/**Sets the chart's color pallete. jqxChart suppports 32 color schemes from 'scheme01' to 'scheme32'. */
export declare type ThreeDChartColorScheme = 'scheme01' | 'scheme02' | 'scheme03' | 'scheme04' | 'scheme05' | 'scheme06' | 'scheme07' | 'scheme08' | 'scheme09' | 'scheme10' | 'scheme11' | 'scheme12' | 'scheme13' | 'scheme14' | 'scheme15' | 'scheme16' | 'scheme17' | 'scheme18' | 'scheme19' | 'scheme20' | 'scheme21' | 'scheme22' | 'scheme23' | 'scheme24' | 'scheme25' | 'scheme26' | 'scheme27' | 'scheme28' | 'scheme29' | 'scheme30' | 'scheme31' | 'scheme32' | 'custom';
/**Sets whether the legend will be created based on the chart's series or serie groups. "auto" - the legend index will change depending on the Chart type */
export declare type ThreeDChartLegendIndex = 'auto' | 'serie' | 'group';
/**Determines the selection mode. */
export declare type ThreeDChartSelectionMode = 'none' | 'single' | 'multiple';
/**Determines the symbol type displayed for the data points in the serie. This parameter is applicable to line, area, scatter and bubble series only. */
export declare type ThreeDChartSymbolType = 'none' | 'circle' | 'square' | 'diamond' | 'triangle_up' | 'triangle_down' | 'triangle_left' | 'triangle_right';
/**Sets the chart type. jqxChart supports several common chart types. You can easily plot series of different types on a common chart. A type must be specified for each series group. Currently, jqxChart supports the following series types:'column' - simple column series
'stackedcolumn' - stacked column series
'stackedcolumn100' - percentage stacked columns
'rangecolumn' - floating column between two values
'waterfall' - waterfall series
'stackedwaterfall' - stacked waterfall series
'line' - simple straight lines connecting the value points
'stackedline' - stacked lines
'stackedline100' - percentage stacked lines
'spline' - smooth lines connecting the value points
'stackedspline' - smooth stacked lines
'stackedspline100' - percentage stacked smooth lines
'stepline' - step line
'stackedstepline' - stacked step line
'stackedstepline100' - percentage stacked step line
'area' - area connecting the value points with straight lines
'stackedarea' - stacked area with straight lines between the points
'stackedarea100' - percentage stacked area with straight lines between the points
'rangearea' - floating area between pairs of value points
'splinearea' - smooth area connecting the value points
'stackedsplinearea' - stacked smooth area connecting the value points
'stackedsplinearea100' - percentage stacked smooth area
'splinerangearea' - smooth floating area between pairs of value points
'steprangearea' - step area between pairs of value points
'stackedsplineara' - smooth stacked area
'steparea' - step area connecting the value points
'stackedsteparea' - step stacked area
'stackedsteparea100' - percentage stacked step area
'pie' - circular chart divided into sectors, illustrating proportion
'donut' - chart divided into circular sectors with different inner and outer radius
'scatter' - data is displayed as a collection of points
'stackedscatter' - data is displayed as a collection of points and the values are stacked
'stackedscatter100' - data is displayed as a collection of points and the values are percentage stacked
'bubble' - data is displayed as a collection of bubbles
'stackedbubble' - data is displayed as a collection of bubbles and the values are stacked
'stackedbubble100' - data is displayed as a collection of bubbles and the values are percentage stacked
'candlestick' - display candlestick series using open, high, low, close data points
'ohlc' - display OHLC series using open, high, low, close data points
 */
export declare type ThreeDChartType = 'column' | 'stackedcolumn' | 'stackedcolumn100' | 'rangecolumn' | 'waterfall' | 'stackedwaterfall' | 'line' | 'stackedline' | 'stackedline100' | 'spline' | 'stackedspline' | 'stackedspline100' | 'stepline' | 'stackedstepline' | 'stackedstepline100' | 'area' | 'stackedarea' | 'stackedarea100' | 'rangearea' | 'splinearea' | 'stackedsplinearea' | 'stackedsplinearea100' | 'splinerangearea' | 'steprangearea' | 'stackedsplineara' | 'steparea' | 'stackedsteparea' | 'stackedsteparea100' | 'pie' | 'donut' | 'scatter' | 'stackedscatter' | 'stackedscatter100' | 'bubble' | 'stackedbubble' | 'stackedbubble100' | 'candlestick' | 'ohlc';
/**Sets the range selector chart type */
export declare type ThreeDChartRangeSelectorSerieType = 'line' | 'area';
/**The type of the axis. 'auto' - automatically detects and switches to 'basic', 'linear' or 'date'.
'date' - when displaying dates.
'basic' - displays all data points sequentially.
'linear' - linear arrangement by the value of the xAxis data field.
 */
export declare type ThreeDChartXAxisType = 'auto' | 'date' | 'basic' | 'linear';
export interface AccordionProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the data source that will be loaded to the Accordion.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the accordion. Disabled elements can not be interacted with.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the expanded item indexes. Using this property items can be expanded by passing in their indexes. The number of expanded items is limited by the expandMode.
   * Default value: 
   */
  expandedIndexes?: number[];
  /**
   * Sets or gets the expand mode. Expand mode determines how the items will expand or collapse.
   * Default value: singleFitHeight
   */
  expandMode?: AccordionExpandMode | string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "accordionItemRequired": "' requires an item from type \"jqx-accordion-item\".",
   *     "indexOutOfBound": "' method.",
   *     "invalidSettings": "' method accepts a string or an object as it's second parameter.",
   *     "noItems": ": No child elements found.",
   *     "overridingProperties": "' property is used by default."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines if the element is readonly or not. If the element true, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Enables or disables accordion reordering.
   * Default value: false
   */
  reorder?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines whether the element can be focused or not.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 Accordion organizes content within collapsable items.
*/
export interface Accordion extends BaseElement, AccordionProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when an item is collapsed.
	* @param event. The custom event. Custom data event was created with: ev.detail(content, index, label)
   *  content - The content of the item.
   *  index - The index of the item.
   *  label - The label of the item
   */
  onCollapse?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an item is going to be collapsed.
	* @param event. The custom event. Custom data event was created with: ev.detail(content, index, label)
   *  content - The content of the item.
   *  index - The index of the item.
   *  label - The label of the item
   */
  onCollapsing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a reordering operation is completed.
	* @param event. The custom event. Custom data event was created with: ev.detail(position, target, content, index, label)
   *  position - The current top and left position of the item that was dragged.
   *  target - The item that was dragged.
   *  content - The content of the item.
   *  index - The index of the item.
   *  label - The label of the item.
   */
  onDragEnd: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a reordering operation is started.
	* @param event. The custom event. Custom data event was created with: ev.detail(position, target, content, index, label)
   *  position - The current top and left position of the item that is about to be dragged.
   *  target - The item that is about to be dragged.
   *  content - The content of the item.
   *  index - The index of the item.
   *  label - The label of the item.
   */
  onDragStart: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when an item is expanded.
	* @param event. The custom event. Custom data event was created with: ev.detail(position, target, content, index, label)
   *  position - The current top and left position of the item.
   *  target - The item that was dragged.
   *  content - The content of the item.
   *  index - The index of the item.
   *  label - The label of the item.
   */
  onExpand?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an item is going to be expanded.
	* @param event. The custom event. Custom data event was created with: ev.detail(content, index, label)
   *  content - The content of the item.
   *  index - The index of the item.
   *  label - The label of the item
   */
  onExpanding?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Collapses an item at a specified index.
   * @param {number} position. The index of the collapsed item.
   */
  collapse(position: number): void;
  /**
   * Expands an item at a specified index.
   * @param {number} position. The index of the expanded item.
   */
  expand(position: number): void;
  /**
   * Inserts a new item at a specified index.
   * @param {number} index. The index where the item must be inserted.
   * @param {any} item. An object containing the values for the properties of the new item to be inserted.
   */
  insert(index: number, item: any): void;
  /**
   * Removes an item at a specified index.
   * @param {number} position. The index of the item to be removed.
   */
  removeAt(position: number): void;
  /**
   * Updates an item from the element.
   * @param {number} index. The index of the item to be updated.
   * @param {any} settings. An object containing the values for the properties of the item that will be updated.
   */
  update(index: number, settings: any): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-accordion"): Accordion;
        querySelector(selectors: "smart-accordion"): Accordion | null;
        querySelectorAll(selectors: "smart-accordion"): NodeListOf<Accordion>;
        getElementsByTagName(qualifiedName: "smart-accordion"): HTMLCollectionOf<Accordion>;
        getElementsByName(elementName: "smart-accordion"): NodeListOf<Accordion>;
    }
}

/**Sets or gets the expand mode. Expand mode determines how the items will expand or collapse. */
export declare type AccordionExpandMode = 'single' | 'singleFitHeight' | 'multiple' | 'toggle' | 'none';
export interface AccordionItemProperties {
  /**
   * Sets or gets header's arrow position. If the value is 'none' the arrow is not shown.
   * Default value: left
   */
  arrow?: AccordionItemArrow | string;
  /**
   * Sets or gets the content if the item.
   * Default value: ''
   */
  content?: string | HTMLElement;
  /**
   * Sets or gets the expanded state.
   * Default value: false
   */
  expanded?: boolean;
  /**
   * Sets or gets the focus state.
   * Default value: false
   */
  focused?: boolean;
  /**
   * Sets or gets the label if the item.
   * Default value: ""
   */
  label?: string;
}
/**
 Single item in an Accordion view.
*/
export interface AccordionItem extends BaseElement, AccordionItemProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the item is collapsed.
	* @param event. The custom event.    */
  onCollapse?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the item is expanded.
	* @param event. The custom event.    */
  onExpand?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-accordion-item"): AccordionItem;
        querySelector(selectors: "smart-accordion-item"): AccordionItem | null;
        querySelectorAll(selectors: "smart-accordion-item"): NodeListOf<AccordionItem>;
        getElementsByTagName(qualifiedName: "smart-accordion-item"): HTMLCollectionOf<AccordionItem>;
        getElementsByName(elementName: "smart-accordion-item"): NodeListOf<AccordionItem>;
    }
}

/**Sets or gets header's arrow position. If the value is 'none' the arrow is not shown. */
export declare type AccordionItemArrow = 'left' | 'right' | 'none';
export interface ArrayProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Sets or gets the indexing mode of the Array.
   * Default value: LabVIEW
   */
  arrayIndexingMode?: ArrayArrayIndexingMode | string;
  /**
   * A callback function that is called when the width, height or disabled properties of an inner element need to be updated. Applicable only when type is 'custom'.
   * Default value: null
   */
  changeProperty?: any;
  /**
   * Sets or gets the number of visible columns in the Array.
   * Default value: 1
   */
  columns?: number;
  /**
   * Sets or gets the default value of inner elements when type is 'custom'.
   * Default value: null
   */
  customWidgetDefaultValue?: any;
  /**
   * Sets or gets the dimensions of the Array.
   * Default value: 1
   */
  dimensions?: number;
  /**
   * Sets or gets disabled state of the Array.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the height of Array elements (row height).
   * Default value: 25
   */
  elementHeight?: number;
  /**
   * A callback function that can be used for applying settings to element widgets. When type is 'custom', widgets have to be initialized in this callback function.
   * Default value: null
   */
  elementTemplate?: any;
  /**
   * Sets or gets the width of Array elements (column width).
   * Default value: 75
   */
  elementWidth?: number;
  /**
   * A callback function that can be used for getting the value of element widgets.
   * Default value: null
   */
  getElementValue?: any;
  /**
   * Sets or gets the height of indexers.
   * Default value: 25
   */
  indexerHeight?: number;
  /**
   * Sets or gets the width of indexers.
   * Default value: 50
   */
  indexerWidth?: number;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback, related to localization module. 
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "callbackFunctionRequired": "jqx-array: When \"type\" is 'custom', the  callback function has to be implemented."
   *   }
   * }
   */
  messages?: any;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the number of visible rows in the Array.
   * Default value: 1
   */
  rows?: number;
  /**
   * A callback function that can be used for setting the value of element widgets.
   * Default value: null
   */
  setElementValue?: any;
  /**
   * Sets or gets whether to display the horizontal scrollbar.
   * Default value: false
   */
  showHorizontalScrollbar?: boolean;
  /**
   * Sets or gets whether to display the array indexers.
   * Default value: false
   */
  showIndexDisplay?: boolean;
  /**
   * Sets or gets whether to highlight selected elements.
   * Default value: false
   */
  showSelection?: boolean;
  /**
   * Sets or gets whether to display the vertical scrollbar.
   * Default value: false
   */
  showVerticalScrollbar?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets or gets the data type and element widgets to be used in the Array.
   * Default value: none
   */
  type?: ArrayType | string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the Array.
   * Default value: 
   */
  value?: any;
}
/**
 Array is broadly used in Engineering applications and displays a Grid of values.
*/
export interface Array extends BaseElement, ArrayProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when a visible row or column has been added or removed.
	* @param event. The custom event.    */
  onArraySizeChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the value of the Array is changed.
	* @param event. The custom event.    */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a dimension has been added or removed.
	* @param event. The custom event.    */
  onDimensionChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an Array element has been clicked.
	* @param event. The custom event.    */
  onElementClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Array is scrolled with one of the scrollbars.
	* @param event. The custom event.    */
  onScroll: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the column width or the row height has been changed.
	* @param event. The custom event.    */
  onSizeChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Adds a dimension to the array.<br /> <em>Note:</em> when adding multiple dimensions simultaneously, it is recommended to do so by dynamically setting the <strong>dimensions</strong> property.
   */
  addDimension(): void;
  /**
   * Clears the selection.
   */
  clearSelection(): void;
  /**
   * Copies the value of an Array element to the clipboard.
   * @param {number} Rowvisibleindex. The visible index of the row (y coordinate) of the element.
   * @param {number} Columnvisibleindex. The visible index of the column (x coordinate) of the element.
   */
  copyElementValueToClipboard(Rowvisibleindex: number, Columnvisibleindex: number): void;
  /**
   * Deletes a column in the <strong>value</strong> array.
   * @param {number} Columnindex. Index of the column to be deleted.
   */
  deleteColumn(Columnindex: number): void;
  /**
   * Deletes a row in the <strong>value</strong> array.
   * @param {number} Rowindex. Index of the row to be deleted.
   */
  deleteRow(Rowindex: number): void;
  /**
   * Empties the <strong>value</strong> array.
   */
  emptyArray(): void;
  /**
   * Designates the end of a selection started with the method <strong>startSelection</strong>.
   * @param {number} Rowboundindex. The bound index of the row (y coordinate) to end the selection to.
   * @param {number} Columnboundindex. The bound index of the column (x coordinate) to end the selection to.
   */
  endSelection(Rowboundindex: number, Columnboundindex: number): void;
  /**
   * Returns the HTML element at the specified visible row and column coordinates of the Array.
   * @param {number} RowVisibleIndex. The visible index of the row (y coordinate) of the element.
   * @param {number} ColumnVisibleIndex. The visible index of the column (x coordinate) of the element.
   * @returns {HTMLElement}
   */
  getElement(RowVisibleIndex: number, ColumnVisibleIndex: number): HTMLElement;
  /**
   * Returns an object with the values of the Array element width and height.
   * @returns {any}
   */
  getElementSize(): any;
  /**
   * Gets an array with the values of all indexers.
   * @returns {any[]}
   */
  getIndexerValue(): any[];
  /**
   * Returns an HTML element from the Array at the specified page coordinates and other information about this element.
   * @param {number} Pagexcoordinate. 
   * @param {number} Pageycoordinate. 
   * @returns {any}
   */
  hitTest(Pagexcoordinate: number, Pageycoordinate: number): any;
  /**
   * Inserts a column in the <strong>value</strong> array before the specified column. The new column is filled with default values.
   * @param {number} Columnindex. Index of the column to add a new column before.
   */
  insertColumnBefore(Columnindex: number): void;
  /**
   * Inserts a row in the <strong>value</strong> array before the specified row. The new row is filled with default values.
   * @param {number} Rowindex. Index of the row to add a new row before.
   */
  insertRowBefore(Rowindex: number): void;
  /**
   * Sets all <strong>value</strong> array members to the default value.
   */
  reinitializeArray(): void;
  /**
   * Removes a dimension from the array.<br /> <em>Note:</em> when removing multiple dimensions simultaneously, it is recommended to do so by dynamically setting the <strong>dimensions</strong> property.
   */
  removeDimension(): void;
  /**
   * Sets the array's <strong>type</strong> to <em>'none'</em>.
   */
  reset(): void;
  /**
   * Resizes Array elements (changes both the column width and the row height).
   * @param {number} Elementwidth. The new element (column) width.
   * @param {number} Elementheight. The new element (row) height.
   */
  resizeElement(Elementwidth: number, Elementheight: number): void;
  /**
   * Selects all members of the array.
   */
  selectAll(): void;
  /**
   * Selects an element with the passed row and column bound indexes.
   * @param {number} Rowboundindex. 
   * @param {number} Columnboundindex. 
   */
  selectElement(Rowboundindex: number, Columnboundindex: number): void;
  /**
   * Sets the column (element) width.
   * @param {number} Columnwidth. The new column width.
   */
  setColumnWidth(Columnwidth: number): void;
  /**
   * Sets the default value of array members.
   * @param {any} Defaultvalue. The new default value. Its data type should correspond to the <strong>type</strong> of the Array.
   */
  setDefaultValue(Defaultvalue: any): void;
  /**
   * Sets the value of one or more Array indexers.
   * @param {any[]} Settings. An array of objects with the fields index and value.
   */
  setIndexerValue(Settings: any[]): void;
  /**
   * Sets the row (element) height.
   * @param {number} Rowheight. The new row height.
   */
  setRowHeight(Rowheight: number): void;
  /**
   * Makes the last array member visible.
   */
  showLastElement(): void;
  /**
   * Designates the start of a selection. To end a selection, call <strong>endSelection</strong>.
   * @param {number} Rowboundindex. The bound index of the row (y coordinate) to start the selection from.
   * @param {number} Columnboundindex. The bound index of the column (x coordinate) to start the selection from.
   */
  startSelection(Rowboundindex: number, Columnboundindex: number): void;
  /**
   * Increases or decreases the visual gap between Array elements.
   */
  toggleElementGap(): void;
  /**
   * Transposes the array. Applicable only when <strong>dimensions</strong> is <em>2</em> (2D array).
   */
  transposeArray(): void;
  /**
   * Sets or gets the value of the whole array or sets the value of a member of the array.
   * @param {any} Newvalue?. If the method is used for setting the value of the whole array, the expected value is an array. If it is used for setting the value of an array member, the value can be of any applicable type.
   * @param {number | number[]} Elementindexes?. If this parameter is passed, only the value of the array member with the provided dimension indexes is set. Dimension indexes that are not passed are considered to be 0.
   * @returns {any[]}
   */
  val(Newvalue?: any, Elementindexes?: number | number[]): any[];
}

declare global {
    interface Document {
        createElement(tagName: "smart-array"): Array;
        querySelector(selectors: "smart-array"): Array | null;
        querySelectorAll(selectors: "smart-array"): NodeListOf<Array>;
        getElementsByTagName(qualifiedName: "smart-array"): HTMLCollectionOf<Array>;
        getElementsByName(elementName: "smart-array"): NodeListOf<Array>;
    }
}

/**Sets or gets the indexing mode of the Array. */
export declare type ArrayArrayIndexingMode = 'LabVIEW' | 'JavaScript';
/**Sets or gets the data type and element widgets to be used in the Array. */
export declare type ArrayType = 'none' | 'boolean' | 'numeric' | 'string' | 'custom';
export interface BarcodeProperties {
  /**
   * Sets the background color of the barcode element.
   * Default value: "white"
   */
  backgroundColor?: string;
  /**
   * Sets whether the barcode label is visible.
   * Default value: true
   */
  displayLabel?: boolean;
  /**
   * Sets the color of the barcode label.
   * Default value: "black"
   */
  labelColor?: string;
  /**
   * Sets the font family of the barcode label.
   * Default value: "monospace"
   */
  labelFont?: string;
  /**
   * Sets the font size of the barcode label.
   * Default value: 14
   */
  labelFontSize?: number;
  /**
   * Sets the bottom margin of the barcode label.
   * Default value: 5
   */
  labelMarginBottom?: number;
  /**
   * Sets the top margin of the barcode label.
   * Default value: 5
   */
  labelMarginTop?: number;
  /**
   * Sets the position of the barcode label.
   * Default value: bottom
   */
  labelPosition?: BarcodeLabelPosition | string;
  /**
   * Sets the color of the barcode lines.
   * Default value: "black"
   */
  lineColor?: string;
  /**
   * Sets the height of the barcode line.
   * Default value: 50
   */
  lineHeight?: number;
  /**
   * Sets the width of the barcode line.
   * Default value: 4
   */
  lineWidth?: number;
  /**
   * Sets the rendering mode of the barcode.
   * Default value: svg
   */
  renderAs?: BarcodeRenderAs | string;
  /**
   * Sets the barcode type
   * Default value: codabar
   */
  type?: BarcodeType | string;
  /**
   * Sets or gets the value of the barcode.
   * Default value: ""
   */
  value?: string;
  /**
   * Sets or gets the width of the barcode. If the width is set to 0, the width of the barcode is calculated automatically.
   * Default value: 0
   */
  width?: number;
  /**
   * Sets or gets the height of the barcode. If the height is set to 0, the height of the barcode is calculated automatically.
   * Default value: 0
   */
  height?: number;
}
/**
 Barcodes encodes text value in a specific pattern.
*/
export interface Barcode extends BaseElement, BarcodeProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the value of the barcode is invalid. 
	* @param event. The custom event. Custom data event was created with: ev.detail(invalidCharacters, lengthValidity, patternValidity, value)
   *  invalidCharacters - An array indicating the invalid characters.
   *  lengthValidity - A boolean indicating the length validity.
   *  patternValidity - A boolean indicating the pattern validity.
   *  value - the invalid value of the barcode.
   */
  onInvalid?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Exports the barcode.
   * @param {string} format. The format of the exported file - svg, png, jpg
   * @param {string} fileName?. The name of the exported file
   */
  export(format: string, fileName?: string): void;
  /**
   * Gets the base64 string of the barcode
   * @param {string} format. The dataURL format of the string - svg, png, jpg
   * @returns {string}
   */
  getDataURL(format: string): string;
  /**
   * Gets the base64 string of the barcode
   * @param {string} format. The dataURL format of the string - svg, png, jpg
   * @returns {any}
   */
  getDataURLAsync(format: string): any;
  /**
   * Gets the validity of the barcode
   * @returns {boolean}
   */
  isValid(): boolean;
}

declare global {
    interface Document {
        createElement(tagName: "smart-barcode"): Barcode;
        querySelector(selectors: "smart-barcode"): Barcode | null;
        querySelectorAll(selectors: "smart-barcode"): NodeListOf<Barcode>;
        getElementsByTagName(qualifiedName: "smart-barcode"): HTMLCollectionOf<Barcode>;
        getElementsByName(elementName: "smart-barcode"): NodeListOf<Barcode>;
    }
}

/**Sets the position of the barcode label. */
export declare type BarcodeLabelPosition = 'top' | 'bottom';
/**Sets the rendering mode of the barcode. */
export declare type BarcodeRenderAs = 'svg' | 'canvas';
/**Sets the barcode type */
export declare type BarcodeType = 'pharmacode' | 'codabar' | 'code128a' | 'code128b' | 'code128c' | 'msi' | 'msi10' | 'msi11' | 'msi1010' | 'msi1110' | 'ean13' | 'ean8' | 'code39' | 'code93';
export interface BreadcrumbProperties {
  /**
   * Enables or disables the "Add new item" (+) button.
   * Default value: false
   */
  addNewItem?: boolean;
  /**
   * Enables or disables the dragging of breadcrumb items.
   * Default value: false
   */
  allowDrag?: boolean;
  /**
   * Enables or disables the dropping of dragged breadcrumb items.
   * Default value: false
   */
  allowDrop?: boolean;
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Show/Hide the close button of breadcrumb items.
   * Default value: false
   */
  closeButtons?: boolean;
  /**
   * Determines the data source to load breadcrumb items from. The Array should contain objects. Each object defines a single breadcrumb item.
   * Default value: []
   */
  dataSource?: {label: string, value: string}[];
  /**
   * Enables or disables the Breadcrumb.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the template of breadcrumb items. The value of this property can be the id of an HTMLTemplateElement or the HTMLTemplateElement itself. If set to null, no template is applied.
   * Default value: null
   */
  itemTemplate?: any;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the minimum width of the Breadcrumb at which it will switch from normal to minimized mode. If set to null, the Breadcrumb does not minimize automatically.
   * Default value: null
   */
  minimizeWidth?: number;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 Breadcrumbs allow users to make selections from a range of values.
*/
export interface Breadcrumb extends BaseElement, BreadcrumbProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when a Breadcrumb item is closed.
	* @param event. The custom event. Custom data event was created with: ev.detail(item)
   *  item - The item that has been closed.
   */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a Breadcrumb item is about to be closed. The closing operation can be canceled by calling <code>event.preventDefault()</code> in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(item)
   *  item - The item that is going to be closed.
   */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a Breadcrumb item is dropped.
	* @param event. The custom event.    */
  onDragEnd: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a Breadcrumb item is being dragged.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, originalEvent, target)
   *  item - The item that is being dragged.
   *  originalEvent - The original event that initiates the dragging operation.
   *  target - The original target.
   */
  onDragging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the "Add new item" (+) button is clicked.
	* @param event. The custom event.    */
  onAddNewItem?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Adds an item.
   * @param {any} itemDetails. An Object with the fields "index", "label", and "value".
   */
  addItem(itemDetails: any): void;
  /**
   * Restores the Breadcrumb from minimized state back to normal.
   */
  maximize(): void;
  /**
   * Minimizes the Breadcrumb.
   */
  minimize(): void;
  /**
   * Removes an item.
   * @param {HTMLElement} item. The item to remove.
   */
  removeItem(item: HTMLElement): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-breadcrumb"): Breadcrumb;
        querySelector(selectors: "smart-breadcrumb"): Breadcrumb | null;
        querySelectorAll(selectors: "smart-breadcrumb"): NodeListOf<Breadcrumb>;
        getElementsByTagName(qualifiedName: "smart-breadcrumb"): HTMLCollectionOf<Breadcrumb>;
        getElementsByName(elementName: "smart-breadcrumb"): NodeListOf<Breadcrumb>;
    }
}

export interface ButtonProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the click mode for the element.
   * Default value: release
   */
  clickMode?: ClickMode | string;
  /**
   * Sets the content of the element.
   * Default value: ""
   */
  content?: any;
  /**
   * Enables or disables the button. 
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets the inner HTML of the element.
   * Default value: """"
   */
  innerHTML: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: """"
   */
  name?: string;
  /**
   * If the custom element is readonly, it cannot be interacted with.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets or gets the type of the button.
   * Default value: "Reset"
   */
  type?: string;
  /**
   * Sets or gets the button's value. 
   * Default value: ""
   */
  value?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 Buttons allow users to take actions, and make choices, with a single tap. Buttons communicate actions that users can take.
*/
export interface Button extends BaseElement, ButtonProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * Click event is triggered regarding to the chosen clickMode.
	* @param event. The custom event.    */
  onClick: ((this: any, ev: Event) => any) | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-button"): Button;
        querySelector(selectors: "smart-button"): Button | null;
        querySelectorAll(selectors: "smart-button"): NodeListOf<Button>;
        getElementsByTagName(qualifiedName: "smart-button"): HTMLCollectionOf<Button>;
        getElementsByName(elementName: "smart-button"): NodeListOf<Button>;
    }
}

/**Determines the click mode for the element. */
export declare type ClickMode = 'hover' | 'press' | 'release' | 'pressAndRelease';
export interface ButtonGroupProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the buttons configuration. The dataSource can be an array of strings/numbers or objects where the attributes represent the properties of a List Item. For example label, value. It can also be a callback that returns an Array of items as previously described.
   * Default value: []
   */
  dataSource?: any;
  /**
   * Determines the selection mode for the element.
   * Default value: one
   */
  selectionMode?: ButtonGroupSelectionMode | string;
  /**
   * Enables or disables the element. 
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: """"
   */
  name?: string;
  /**
   * If the custom element is readonly, it cannot be interacted with.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets or gets the button group's selected values. Represents an array of strings.
   * Default value: 
   */
  selectedValues?: string[];
  /**
   * Sets or gets the button group's selected indexes. Represents an array of numbers
   * Default value: 
   */
  selectedIndexes?: number[];
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 ButtonGroup creates a set of buttons that can work as normal buttons, radio buttons or checkboxes.
*/
export interface ButtonGroup extends BaseElement, ButtonGroupProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * Change event is triggered when the selectedValues/selectedIndexes are changed.
	* @param event. The custom event.    */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Selects/Unselects an item inside the element.
   * @param {number | string} value. The index or the value of the item to be selected/unselected.
   */
  select(value: number | string): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-button-group"): ButtonGroup;
        querySelector(selectors: "smart-button-group"): ButtonGroup | null;
        querySelectorAll(selectors: "smart-button-group"): NodeListOf<ButtonGroup>;
        getElementsByTagName(qualifiedName: "smart-button-group"): HTMLCollectionOf<ButtonGroup>;
        getElementsByName(elementName: "smart-button-group"): NodeListOf<ButtonGroup>;
    }
}

/**Determines the selection mode for the element. */
export declare type ButtonGroupSelectionMode = 'none' | 'one' | 'zeroOrOne' | 'zeroOrMany';
export interface CalendarProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Applies new animation settings when it is enabled. Properties:startSpeed - Determines the initial speed of the animation.easeThreshold - Determines the point at which the animation starts to slow down - the "ease effect".step - Determines the step ( scrolling interval ) at which the animation will run. stepEaseSize - Coefficient that is used to calculated the new step once the threshold has been passed. resetThreshold - Determines the threshold for animation reset. When it's reached the animation will start over.
   * Default value: null
   */
  animationSettings?: any;
  /**
   * Determines the date controls inside the header of the Calendar.
   * Default value: default
   */
  calendarMode?: CalendarMode | string;
  /**
   * Determines the format of the day names located above the days inside the calendar.
   * Default value: firstTwoLetters
   */
  dayNameFormat?: DayFormat | string;
  /**
   *  A callback that can be used to customize the format of the month name when calendarMode is set to 'default'.
   * Default value: null
   */
  dateFormatFunction?: any;
  /**
   * Enables or disables the Calendar.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Disables auto navigation when the user clicks on a date that's not from the current month in view.
   * Default value: false
   */
  disableAutoNavigation?: boolean;
  /**
   * Determines the date view of the calendar when calendarMode is set to 'default'
   * Default value: month
   */
  displayMode?: CalendarDisplayMode | string;
  /**
   * Determines the type of the month/year view when calendarMode is set to Default.
   * Default value: table
   */
  displayModeView?: CalendarDisplayModeView | string;
  /**
   * Determines the height of the month's drop down inside the Calendar.
   * Default value: 200
   */
  dropDownHeight?: string | number;
  /**
   * Determines the width of the month's drop down inside the Calendar.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Determines the first day of the week. From 0(Sunday) to 6(Saturday)
   * Default value: 0
   */
  firstDayOfWeek?: number;
  /**
   * Sets a custom footer template. Accepts the id of an HTMLTemplateElement or a reference ot it.
   * Default value: null
   */
  footerTemplate?: any;
  /**
   * Sets custom header template. Accepts the id of an HTMLTemplateElement or a reference ot it.
   * Default value: null
   */
  headerTemplate?: any;
  /**
   * Hides the names of the weekdays.
   * Default value: false
   */
  hideDayNames?: boolean;
  /**
   * Hides the dates from other months.
   * Default value: false
   */
  hideOtherMonthDays?: boolean;
  /**
   * Hides the arrow of the tooltip.
   * Default value: false
   */
  hideTooltipArrow?: boolean;
  /**
   * Sets the dates that will be displayed as important.
   * Default value: 
   */
  importantDates?: string[] | Date[];
  /**
   * Sets a template for the important dates. Accepts the id of an HTMLTemplate element inside the DOM of or a reference to it.
   * Default value: null
   */
  importantDatesTemplate?: any;
  /**
   *  Determines the language of the Calendar. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Determines the max date for the Calendar. Accepts date objects and valid date string formats.
   * Default value: new Date(2100, 1, 1)
   */
  max?: string | Date;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the min date for the Calendar. Accepts date objects and valid date string formats.
   * Default value: new Date(1900, 1, 1)
   */
  min?: string | Date;
  /**
   * Determines the number of months to be displayed inside the calendar. The maximum amount of months that can be shown is 12. By default only 1 month is shown.
   * Default value: 1
   */
  months?: number;
  /**
   * Determines the format of the month names in the header when DisplayMode is set to Default or when Months property is greater than 1. 
   * Default value: long
   */
  monthNameFormat?: MonthFormat | string;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: ""
   */
  name?: string;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets restricted dates. Restricted dates are dates that cannot be selected/hovered/focused. They are visualy styled as restricted. The dates can be javascript date objects or strings representing a valid date.
   * Default value: 
   */
  restrictedDates?: string[] | Date[];
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   *  Determines the direction of the navigation buttons located in the header and the animation.
   * Default value: landscape
   */
  scrollButtonsNavigationMode?: ViewLayout | string;
  /**
   *  Determines the position of the navigation buttons located inside the header. 
   * Default value: both
   */
  scrollButtonsPosition?: LayoutPosition | string;
  /**
   * Sets the dates that will be selected. Selected dates are styled differently than the rest. The dates can be Date objects or strings in a valid date format.
   * Default value: 
   */
  selectedDates?: string[] | Date[];
  /**
   * Determines the date selection mode.
   * Default value: default
   */
  selectionMode?: CalendarSelectionMode | string;
  /**
   * Sets the delay between clicks of the date navigation buttons located in the header of the Calendar. 
   * Default value: 80
   */
  spinButtonsDelay?: number;
  /**
   * Determines the initial delay before the action of the date navigation buttons located in the header of the Calendar. 
   * Default value: 0
   */
  spinButtonsInitialDelay?: number;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets a template for the title section of the Calendar. Accepts the id of an HTMLTemplate element inside the DOM of or a reference it.
   * Default value: null
   */
  titleTemplate?: any;
  /**
   * Enables/Disabled the tooltip for the important dates. If enabled when an important date is hovered a tooltip is displayed.
   * Default value: false
   */
  tooltip?: boolean;
  /**
   * Show/Hide the arrow of the the tooltip for the important dates. By default the arrow is visible.
   * Default value: true
   */
  tooltipArrow?: boolean;
  /**
   * Sets the delay of the tooltip before it appears.
   * Default value: 100
   */
  tooltipDelay?: number;
  /**
   * Set's a custom offset to the tooltip's position. Accepts an array of two numbers: the left coordinate and the top coordinate.
   * Default value: 
   */
  tooltipOffset?: number[][];
  /**
   * Sets the position of the tooltip.
   * Default value: top
   */
  tooltipPosition?: TooltipPosition | string;
  /**
   * Sets a template for the tooltip's content. Accepts the id of an HTMLTEmplate element inside the DOM or it's reference.
   * Default value: null
   */
  tooltipTemplate?: any;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Determines the orientation of the Calendar.
   * Default value: portrait
   */
  view?: ViewLayout | string;
  /**
   * Determines the visible sections of the Calendar. The view sections are : title, header, footer. Multiple sections can be applied at the same time. By default only the 'header' section is visible.
   * Default value: header
   */
  viewSections?: string[];
  /**
   * Enables/Disabled week numbering. If enabled week numbers are displayed infront of each week inside the Calendar.
   * Default value: false
   */
  weekNumbers?: boolean;
  /**
   * Determines the number of visible weeks. The value of the property ranges from 1 to 6. Where 1 is one week and 6 is a full month ( 6 weeks ).
   * Default value: 6
   */
  weeks?: number;
  /**
   * Determines the year format in the header when DisplayMode is set to Default or when Months property is greater than 1.
   * Default value: numeric
   */
  yearFormat?: YearFormat | string;
}
/**
 Calendar allows user to easily select one or more dates. This control supports multi-calendar view, special dates, holidays, weekends, decade views.
*/
export interface Calendar extends BaseElement, CalendarProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when a new date has been selected/unselected.
	* @param event. The custom event. Custom data event was created with: ev.detail(value)
   *  value - An array of all currently selected dates.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the displayMode is about to change. This event allows to cancel the operation by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldDisplayMode, newDisplayMode)
   *  oldDisplayMode - The previous display mode.
   *  newDisplayMode - The new display mode.
   */
  onDisplayModeChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the display mode has changed.
	* @param event. The custom event.    */
  onDisplayModeChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the view is changing. This navigation can be cancelled by using the preventDefault method.
	* @param event. The custom event. Custom data event was created with: ev.detail(value, type)
   *  value - The view's date.
   *  type - The view type - 'month', 'decade' or 'year'.
   */
  onNavigationChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the view is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(value, type)
   *  value - The view's date.
   *  type - The view type - 'month', 'decade' or 'year'.
   */
  onNavigationChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the tooltip for the important date is opened.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, value)
   *  target - The event target - tooltip.
   *  value - The important date of the hovered cell.
   */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the tooltip for the important date is closed.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, value)
   *  target - The event target - tooltip.
   *  value - The important date of the hovered cell.
   */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * Clears the selection. Removes all seleceted dates.
   */
  clearSelection(): void;
  /**
   * Navigates forwards/backwards depending on the argument.
   * @param {number | Date | string} step. The argument can be the following: <ul><li> number -  representing the number of months to scroll. Can be negavtive. If negative it will scroll backwards.</li><li> Date - a date object representing the Date to navigate to.</li><li> string - a string representing a valid Date, e.g. "2020-10-1" </li></ul>
   * @returns {boolean}
   */
  navigate(step: number | Date | string): boolean;
  /**
   * Selects or Unselects a date.
   * @param {Date | string} date. The date to be selected or unselected. The date can be a Date object or a string in valid date format.
   */
  select(date: Date | string): void;
  /**
   * Selects today's date.
   * @returns {Date}
   */
  today(): Date;
}

declare global {
    interface Document {
        createElement(tagName: "smart-calendar"): Calendar;
        querySelector(selectors: "smart-calendar"): Calendar | null;
        querySelectorAll(selectors: "smart-calendar"): NodeListOf<Calendar>;
        getElementsByTagName(qualifiedName: "smart-calendar"): HTMLCollectionOf<Calendar>;
        getElementsByName(elementName: "smart-calendar"): NodeListOf<Calendar>;
    }
}

/**Determines the date controls inside the header of the Calendar. */
export declare type CalendarMode = 'default' | 'classic';
/**Determines the format of the day names located above the days inside the calendar. */
export declare type DayFormat = 'narrow' | 'firstTwoLetters' | 'long' | 'short';
/**Determines the date view of the calendar when calendarMode is set to 'default' */
export declare type CalendarDisplayMode = 'month' | 'year' | 'decade';
/**Determines the type of the month/year view when calendarMode is set to Default. */
export declare type CalendarDisplayModeView = 'table' | 'list';
/**Determines the format of the month names in the header when DisplayMode is set to Default or when Months property is greater than 1. <br/> */
export declare type MonthFormat = 'narrow' | 'firstTwoLetters' | 'long' | 'short' | '2-digit' | 'numeric';
/**Determines the orientation of the Calendar. */
export declare type ViewLayout = 'landscape' | 'portrait';
/** Determines the position of the navigation buttons located inside the header.  */
export declare type LayoutPosition = 'near' | 'far' | 'both';
/**Determines the date selection mode. */
export declare type CalendarSelectionMode = 'none' | 'default' | 'many' | 'one' | 'oneExtended' | 'oneOrMany' | 'range' | 'week' | 'zeroOrMany' | 'zeroOrOne';
/**Sets the position of the tooltip. */
export declare type TooltipPosition = 'auto' | 'absolute' | 'bottom' | 'top' | 'left' | 'right';
/**Determines the year format in the header when DisplayMode is set to Default or when Months property is greater than 1.<br/> */
export declare type YearFormat = '2-digit' | 'numeric';
export interface CardProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * A callback function, used to add event handlers and other custom logic related to the content inside the card element.
   * Default value: null
   */
  contentHandler?: any;
  /**
   * This object is used to populate card's template. Object keys represent the placeholders surrounded in braces ( e.g.  ) inside the template of an item and their values are used to replace the bindings.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Disables the interaction with the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets custom card template. The template can be the ID of an HTMLTemplate element inside the DOM or it's reference. The content of the template may hold one or many property placeholders in format . These placeholders will be replaced with the values of the corresponding properties defined in the dataSource object. When setting the property to template ID, the property type is 'string'.
   * Default value: null
   */
  itemTemplate?: any;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 Card component with header, footer and content sections.
*/
export interface Card extends BaseElement, CardProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the card is swiped bottom.
	* @param event. The custom event.    */
  onSwipebottom?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the card is swiped left.
	* @param event. The custom event.    */
  onSwipeleft?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the card is swiped right.
	* @param event. The custom event.    */
  onSwiperight?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the card is swiped top.
	* @param event. The custom event.    */
  onSwipetop?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-card"): Card;
        querySelector(selectors: "smart-card"): Card | null;
        querySelectorAll(selectors: "smart-card"): NodeListOf<Card>;
        getElementsByTagName(qualifiedName: "smart-card"): HTMLCollectionOf<Card>;
        getElementsByName(elementName: "smart-card"): NodeListOf<Card>;
    }
}

export interface CardViewProperties {
  /**
   * Toggles the button for adding new cards.
   * Default value: false
   */
  addNewButton?: boolean;
  /**
   * Allows reordering by dragging cards.
   * Default value: false
   */
  allowDrag?: boolean;
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Describes the height for each card.
   * Default value: null
   */
  cardHeight?: number | null;
  /**
   * Describes the orientation of the card cells.
   * Default value: vertical
   */
  cellOrientation?: Orientation | string;
  /**
   * Allows collapsing the card content.
   * Default value: false
   */
  collapsible?: boolean;
  /**
   * Describes the columns properties:label - Sets the column namedataField - Sets the dataField nameicon - Sets the icon for the columnformatSettings - Sets the settings about the format for the current columnformatFunction - Function for customizing the value
   * Default value: 
   */
  columns?: CardViewColumn[];
  /**
   * Describes which data field to be set as cover.
   * Default value: "''"
   */
  coverField?: string;
  /**
   * Describes the cover image fit property.
   * Default value: crop
   */
  coverMode?: CardViewCoverMode | string;
  /**
   * Determines the data source for the item that will be displayed inside the card.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Sets the grid's data source settings when the dataSource property is set to an Array or URL.
   * Default value: [object Object]
   */
  dataSourceSettings?: DataSourceSettings;
  /**
   * Allows the edit option for the cards.
   * Default value: false
   */
  editable?: boolean;
  /**
   * Sets or gets the header position. The header contains the Customize, Filter, Sort, and Search buttons.
   * Default value: none
   */
  headerPosition?: CardViewHeaderPosition | string;
  /**
   * Sets or gets the locale. Used in conjunction with the property messages.
   * Default value: "en"
   */
  locale?: string;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "addFilter": "+ Add filter",
   *     "addImage": "Add",
   *     "and": "And",
   *     "apply": "Apply",
   *     "booleanFirst": "☐",
   *     "booleanLast": "☑",
   *     "cancel": "Cancel",
   *     "CONTAINS": "contains",
   *     "CONTAINS_CASE_SENSITIVE": "contains (case sensitive)",
   *     "coverField": "Cover field",
   *     "crop": "Crop",
   *     "customizeCards": "Customize cards",
   *     "dateFirst": "1",
   *     "dateLast": "9",
   *     "dateTabLabel": "DATE",
   *     "DOES_NOT_CONTAIN": "does not contain",
   *     "DOES_NOT_CONTAIN_CASE_SENSITIVE": "does not contain (case sensitive)",
   *     "draggedRecord": "Record ",
   *     "EMPTY": "empty",
   *     "ENDS_WITH": "ends with",
   *     "ENDS_WITH_CASE_SENSITIVE": "ends with (case sensitive)",
   *     "EQUAL": "equal",
   *     "EQUAL_CASE_SENSITIVE": "equal (case sensitive)",
   *     "filter": "Filter",
   *     "filteredByMultiple": " filters",
   *     "filteredByOne": "1 filter",
   *     "find": "Find a field",
   *     "findInView": "Find in view",
   *     "firstBy": "Sort by",
   *     "fit": "Fit",
   *     "found": "",
   *     "from": "from",
   *     "GREATER_THAN": "greater than",
   *     "GREATER_THAN_OR_EQUAL": "greater than or equal",
   *     "imageUrl": "New image URL:",
   *     "incorrectStructure": "'dataSource' must be an instance of Smart.DataAdapter or an array of objects with key-value pairs.",
   *     "LESS_THAN": "less than",
   *     "LESS_THAN_OR_EQUAL": "less than or equal",
   *     "nextRecord": "Next record",
   *     "noCoverField": "No cover field",
   *     "noData": "No data to display",
   *     "noFilters": "No filters applied",
   *     "noMatches": "No matched records",
   *     "noSorting": "No sorting applied",
   *     "noResults": "No results",
   *     "NOT_EMPTY": "not empty",
   *     "NOT_EQUAL": "not equal",
   *     "NOT_NULL": "not null",
   *     "now": "Now",
   *     "NULL": "null",
   *     "numberFirst": "1",
   *     "numberLast": "9",
   *     "ok": "OK",
   *     "or": "Or",
   *     "pickAnother": "Pick another field to sort by",
   *     "previousRecord": "Previous record",
   *     "removeImage": "Remove",
   *     "sort": "Sort",
   *     "sortedByMultiple": "Sorted by  fields",
   *     "sortedByOne": "Sorted by 1 field",
   *     "STARTS_WITH": "starts with",
   *     "STARTS_WITH_CASE_SENSITIVE": "starts with (case sensitive)",
   *     "stringFirst": "A",
   *     "stringLast": "Z",
   *     "thenBy": "then by",
   *     "timeTabLabel": "TIME",
   *     "toggleVisibility": "Toggle field visibility",
   *     "where": "Where"
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Describes the scrolling behavior of the element.
   * Default value: physical
   */
  scrolling?: Scrolling | string;
  /**
   * Describes which data field to be set as title.
   * Default value: ""
   */
  titleField?: string;
}
/**
 CardView creates Card-based layout. Supports Filtering, Sorting, Grouping, Editing and UI Virtualization.
*/
export interface CardView extends BaseElement, CardViewProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when a filter has been applied.
	* @param event. The custom event.    */
  onFilter?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when sorting has been applied.
	* @param event. The custom event.    */
  onSort?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the window is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the window is about to be opened. The opening operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the window is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the window is about to be closed. The closing operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user starts dragging the card.
	* @param event. The custom event.    */
  onDragStart: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the user is dragging the card.
	* @param event. The custom event.    */
  onDragging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user dragged the card.
	* @param event. The custom event.    */
  onDragEnd: ((this: any, ev: Event) => any) | null;
  /**
   * Adds filtering
   * @param {string[]} filters. Filter information
   * @param {string} operator?. Logical operator between the filters of different fields
   */
  addFilter(filters: string[], operator?: string): void;
  /**
   * Adds a new record
   * @param {number | string} recordId?. The id of the record to add
   * @param {any} data?. The data of the record to add
   * @param {string} position?. The position to add the record to. Possible values: 'first' and 'last'.
   */
  addRecord(recordId?: number | string, data?: any, position?: string): void;
  /**
   * Adds sorting
   * @param {[] | string} dataFields. The data field(s) to sort by
   * @param {[] | string} orderBy. The sort direction(s) to sort the data field(s) by
   */
  addSort(dataFields: [] | string, orderBy: [] | string): void;
  /**
   * Begins an edit operation
   * @param {number | string} recordId. The id of the record to edit
   */
  beginEdit(recordId: number | string): void;
  /**
   * Ends the current edit operation and discards changes
   */
  cancelEdit(): void;
  /**
   * Closes any open header panel (drop down)
   */
  closePanel(): void;
  /**
   * Ends the current edit operation and saves changes
   */
  endEdit(): void;
  /**
   * Makes sure a record is visible by scrolling to it. If succcessful, the method returns the HTML element of the record's card.
   * @param {number | string} recordId. The id of the record to scroll to
   * @returns {HTMLElement}
   */
  ensureVisible(recordId: number | string): HTMLElement;
  /**
   * Opens the "Customize cards" header panel (drop down)
   */
  openCustomizePanel(): void;
  /**
   * Opens the "Filter" header panel (drop down)
   */
  openFilterPanel(): void;
  /**
   * Opens the "Sort" header panel (drop down)
   */
  openSortPanel(): void;
  /**
   * Removes filtering
   */
  removeFilter(): void;
  /**
   * Removes a record
   * @param {number | string} recordId. The id of the record to remove
   */
  removeRecord(recordId: number | string): void;
  /**
   * Removes sorting
   */
  removeSort(): void;
  /**
   * Shows a column
   * @param {string} dataField. The data field of the column
   */
  showColumn(dataField: string): void;
}

export interface CardViewColumn {
  /**
   * Sets or gets the column's data source bound field.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Sets or gets the column's data type.
   * Default value: string
   */
  dataType?: CardViewColumnDataType | string;
  /**
   * Sets or gets the column's icon. Expects CSS class name.
   * Default value: 
   */
  icon?: any;
  /**
   * Sets or gets the column's image visibility.
   * Default value: false
   */
  image?: boolean;
  /**
   * Sets or gets the text displayed in the column's header.
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets whether the column is visible. Set the property to 'false' to hide the column.
   * Default value: true
   */
  visible?: boolean;
  /**
   * Sets or gets the column's format function.
   * Default value: null
   */
  formatFunction?: { (settings: { template?: string, column?: any, record?: any, value?: any }): any };
  /**
   * Sets or gets the column's format settings. You can use any of the build in formatting options or to NumberFormat object like that: 'Intl: {  NumberFormat: {  style: \'currency\', currency: \'EUR\' }}' or DateTimeFormat object like that: 'Intl: {  DateTimeFormat: {  dateStyle: \'full\' }}''
   * Default value: [object Object]
   */
  formatSettings?: any;
}

/**Sets the grid's data source settings when the <em>dataSource</em> property is set to an Array or URL. */
export interface DataSourceSettings {
  /**
   * Sets or gets whether a column will be auto-generated.
   * Default value: false
   */
  autoGenerateColumns?: boolean;
  /**
   * Sets or gets a children data field like 'children', 'items' in the data source. When this property is set, the dataAdapter will look for this data field when looping through the items. If it is found a hierarchical data source would be created.
   * Default value: ""
   */
  childrenDataField?: string;
  /**
   * Sets or gets the XML binding root.
   * Default value: ""
   */
  root?: string;
  /**
   * Sets or gets the XML binding root.
   * Default value: blackList
   */
  sanitizeHTML?: DataSourceSettingsSanitizeHTML | string;
  /**
   * Sets or gets the XML binding record.
   * Default value: ""
   */
  record?: string;
  /**
   * Sets or gets the data fields to group by.
   * Default value: []
   */
  groupBy?: string[];
  /**
   * Sets or gets the data fields which decribe the loaded data and data type. Ex: ['id: number', 'firstName: string', 'lastName: string']
   * Default value: null
   */
  dataFields?: DataSourceSettingsDataField[];
  /**
   * Sets or gets whether the data source type.
   * Default value: array
   */
  dataSourceType?: DataSourceSettingsDataSourceType | string;
  /**
   * Sets or gets the dataAdapter's id
   * Default value: ""
   */
  id?: string;
  /**
   * Sets or gets the key data field to be used for building the hierarchy. It is used in combination with the parentDataField property. Usually the 'id' field is used as key data field and 'parentId' as parent data field'
   * Default value: ""
   */
  keyDataField?: string;
  /**
   * Sets or gets the parent data field to be used for building the hierarchy. It is used in combination with the keyDataField property. Usually the 'id' field is used as key data field and 'parentId' as parent data field'
   * Default value: ""
   */
  parentDataField?: string;
  /**
   * Sets the 'mapChar' data field of the record
   * Default value: "."
   */
  mapChar?: string;
  /**
   * Sets the virtual data source function which is called each time the Grid requests data. Demos using 'virtualDataSource' are available on the Grid demos page.
   * Default value: null
   */
  virtualDataSource?: any;
  /**
   * Sets the virtual data source on expand function. This function is called when we load data on demand in Tree or TreeGrid and virtualDataSource in these components is set, too
   * Default value: null
   */
  virtualDataSourceOnExpand?: any;
}

export interface DataSourceSettingsDataField {
  /**
   * Sets the dataField name.
   * Default value: ""
   */
  name?: string;
  /**
   * Sets the dataField mapping path. For nested mapping, use '.'. Example: 'name.firstName'.
   * Default value: ""
   */
  map?: string;
  /**
   * Sets the dataField type.
   * Default value: string
   */
  dataType?: DataSourceSettingsDataFieldDataType | string;
}

declare global {
    interface Document {
        createElement(tagName: "smart-card-view"): CardView;
        querySelector(selectors: "smart-card-view"): CardView | null;
        querySelectorAll(selectors: "smart-card-view"): NodeListOf<CardView>;
        getElementsByTagName(qualifiedName: "smart-card-view"): HTMLCollectionOf<CardView>;
        getElementsByName(elementName: "smart-card-view"): NodeListOf<CardView>;
    }
}

/**Describes the orientation of the card cells. */
export declare type Orientation = 'horizontal' | 'vertical';
/**Sets or gets the column's data type. */
export declare type CardViewColumnDataType = 'string' | 'date' | 'boolean' | 'number' | 'array' | 'any';
/**Describes the cover image fit property. */
export declare type CardViewCoverMode = 'fit' | 'crop';
/**Sets or gets the XML binding root. */
export declare type DataSourceSettingsSanitizeHTML = 'all' | 'blackList' | 'none';
/**Sets the dataField type. */
export declare type DataSourceSettingsDataFieldDataType = 'string' | 'date' | 'boolean' | 'number' | 'array' | 'any';
/**Sets or gets whether the data source type. */
export declare type DataSourceSettingsDataSourceType = 'array' | 'json' | 'xml' | 'csv' | 'tsv';
/**Sets or gets the header position. The header contains the Customize, Filter, Sort, and Search buttons. */
export declare type CardViewHeaderPosition = 'none' | 'top' | 'bottom';
/**Describes the scrolling behavior of the element. */
export declare type Scrolling = 'physical' | 'virtual' | 'infinite' | 'deferred';
export interface CarouselProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * The items switch automatically if set to true or to a custom number(representing the timeout in milliseconds). This property works if slideShow property is enabled.
   * Default value: false
   */
  autoPlay?: boolean;
  /**
   * An array of objects. Each object defines an item. The following object properties are available: label - a string representing the label of the item.content - a string representing the content of the itemimage - a string representing a url link to an image.HTMLcontent - a string representing some HTML structure taht will be generated inside the item.
   * Default value: 
   */
  dataSource?: any[];
  /**
   * Specifies the timeout before a slide changes when a navigation button is pressed. Navigation buttons are repeat buttons that will repeat the oepration after the delay is passed.
   * Default value: 200
   */
  delay?: number;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Disabled the possibility to navigated to an item by clicking on item in displayMode 3d. By default users can navigate to items that are not currently active by clicking on them.
   * Default value: false
   */
  disableItemClick?: boolean;
  /**
   * Determines the display mode.
   * Default value: default
   */
  displayMode?: CarouselDisplayMode | string;
  /**
   * Hides the navigation buttons.
   * Default value: false
   */
  hideArrows?: boolean;
  /**
   * Hides the slide indication panel that shows which item is currently in view (active item).
   * Default value: false
   */
  hideIndicators?: boolean;
  /**
   * Can be used to customize the slide indicator panel of the accordion. The property can be a string that represents the id of an HTMLTemplateElement in the DOM or it's direct reference.
   * Default value: null
   */
  indicatorTemplate?: any;
  /**
   * Determines the interval (in milliseconds) between a slide transitions when slideShow is enabled.
   * Default value: 5000
   */
  interval?: number;
  /**
   * Used to completely customize the content of an item. The property can be a string that represents the id of an HTMLTemplateElement in the DOM or it's direct reference. The content of the template can contain property bindings that refer to the dataSource.
   * Default value: null
   */
  itemTemplate?: any;
  /**
   * Activates/deactivates keyboard navigation. By default, items can not be navigated via keyboard
   * Default value: false
   */
  keyboard?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Determines whether the the items should start over when the first or last item is reached.
   * Default value: false
   */
  loop?: boolean;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ":  Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * When slideShow property is set to true, the carousel changes the active slide automatically with a delay set in interval property.
   * Default value: false
   */
  slideShow?: boolean;
  /**
   * Enables or disables switching to the previous/next slide via swiping left/right. By default swiping is disabled.
   * Default value: false
   */
  swipe?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Activates/deactivates slide navigation via mouse wheel. By default it's disabled.
   * Default value: false
   */
  wheel?: boolean;
}
/**
 Carousel is a slideshow component for cycling through elements—images or slides of text
*/
export interface Carousel extends BaseElement, CarouselProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * Triggered when the active ( in view ) slide is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(index, previousIndex)
   *  index - The index of the new active slide.
   *  previousIndex - The index of the previous slide that was active.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Triggered when the process of slide changing starts.
	* @param event. The custom event. Custom data event was created with: ev.detail(index, previousIndex)
   *  index - The index of the new active slide.
   *  previousIndex - The index of the previous slide that was active.
   */
  onChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user swipes to the left inside the Carousel.
	* @param event. The custom event.    */
  onSwipeleft?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user swipes to the right inside the Carousel.
	* @param event. The custom event.    */
  onSwiperight?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Navigates to the next slide.
   */
  next(): void;
  /**
   * Pauses the slide show if <strong>slideShow</strong> is enabled.
   */
  pause(): void;
  /**
   * Starts the slide show if <strong>slideShow</strong> is enabled.
   */
  play(): void;
  /**
   * Navigates to the previous slide.
   */
  prev(): void;
  /**
   * Navigates to a specific slide with the given index.
   * @param {number} index. The index of the target slide.
   */
  slideTo(index: number): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-carousel"): Carousel;
        querySelector(selectors: "smart-carousel"): Carousel | null;
        querySelectorAll(selectors: "smart-carousel"): NodeListOf<Carousel>;
        getElementsByTagName(qualifiedName: "smart-carousel"): HTMLCollectionOf<Carousel>;
        getElementsByName(elementName: "smart-carousel"): NodeListOf<Carousel>;
    }
}

/**Determines the display mode. */
export declare type CarouselDisplayMode = 'default' | 'multiple' | '3d';
export interface ChartProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'.
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the animation duration in milliseconds. The value must be between 0 and 5000. If it is outside of this range jqxChart will reset it to the default value.
   * Default value: 300
   */
  animationDuration?: number;
  /**
   * Sets the chart's background color. For example: '#DDFFE8'
   * Default value: null
   */
  backgroundColor?: string | null;
  /**
   * Sets the chart's background image. For example: 'https://www.htmlelements.com/demos/images/carousel-large-1.jpg'
   * Default value: ""
   */
  backgroundImage?: string;
  /**
   * Sets the chart's border color. For example: '#098700'
   * Default value: null
   */
  borderLineColor?: string | null;
  /**
   * Sets the chart's border line width.
   * Default value: 1
   */
  borderLineWidth?: number;
  /**
   * Sets the caption (title) of the chart.
   * Default value: "Caption"
   */
  caption?: string;
  /**
   * Determines whether to clip plotted elements that overflow the axis boundaries.
   * Default value: true
   */
  clip?: boolean;
  /**
   * Sets the chart's color pallete. jqxChart suppports 32 color schemes from 'scheme01' to 'scheme32'.
   * Default value: scheme01
   */
  colorScheme?: ChartColorScheme | string;
  /**
   * Enables or disables overlapping of the column series.
   * Default value: false
   */
  columnSeriesOverlap?: boolean;
  /**
   * Gets or sets the color of the crosshairs lines. The 'enableCrosshairs' property should be 'true'.
   * Default value: null
   */
  crosshairsColor?: string | null;
  /**
   * Gets or sets the dash style of the crosshairs lines. The style is a series of numbers indicating line length followed by space length. The 'enableCrosshairs' property should be 'true'. For example: '3,3'
   * Default value: "2,2"
   */
  crosshairsDashStyle?: string;
  /**
   * Gets or sets the width of the crosshairs lines. The 'enableCrosshairs' property should be 'true'
   * Default value: 1
   */
  crosshairsLineWidth?: number;
  /**
   * Sets the chart's data source.
   * Default value: 
   */
  dataSource?: any[];
  /**
   * Sets the description text of the chart.
   * Default value: "Description"
   */
  description?: string;
  /**
   * Enables or disables the chart.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines the drawing function of jqxChart. When the property is set, you can override the jqxChart's drawing.
   * Default value: null
   */
  draw?: any;
  /**
   * Function for custom drawing before the caption and other chart elements.
   * Default value: null
   */
  drawBefore?: any;
  /**
   * Determines if the animation of the axes text is enabled.
   * Default value: false
   */
  enableAxisTextAnimation?: boolean;
  /**
   * Enables or disables the crosshairs lines. The lines are displayed in line and area series when you move over a data point.
   * Default value: false
   */
  enableCrosshairs?: boolean;
  /**
   * Determines whether to display the chart using greyscale colors.
   * Default value: false
   */
  greyScale?: boolean;
  /**
   * Sets the legend's layout.
   * Default value: [object Object]
   */
  legendLayout?: any;
  /**
   * Sets or gets the locale. Used in conjunction with the property messages.
   * Default value: "en"
   */
  locale?: string;
  /**
   * Localization object containing culture-specific properties required for formatting currencies, numbers and dates.
   * Default value: [object Object]
   */
  localization?: ChartLocalization;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale.
   * Default value:    * [object Object]
   */
  messages?: any;
  /**
   * Sets the left, top, right and bottom padding of the Chart.
   * Default value: [object Object]
   */
  padding?: Padding;
  /**
   * Determines the rendering engine used to display the chart. When the property is set to an empty string, jqxChart will automatically select an optimal rendering engine depending on the browser capabilities.
   * Default value: 
   */
  renderEngine?: ChartRenderEngine | string;
  /**
   * Sets or gets a value indicating whether the Chart's layout is mirrored.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * The seriesGroups property is used to describe all series displayed on the chart. jqxChart supports multiple series of different types and series grouping. Each series group may have its own Value Axis (Y-axis) which allows you to have values with different scales displayed on the same chart at the same time. It also allows you to display multiple series types together on the same chart. For example, you can display all series in one group as lines and the series in a second group as columns. seriesGroups is an array of objects where each object represents one group.
   * Default value: 
   */
  seriesGroups?: ChartSeriesGroup[];
  /**
   * Determines whether to display the chart's border line.
   * Default value: true
   */
  showBorderLine?: boolean;
  /**
   * Determines whether to show or hide the chart series legend.
   * Default value: true
   */
  showLegend?: boolean;
  /**
   * Enables or disables the chart tooltips.
   * Default value: true
   */
  showToolTips?: boolean;
  /**
   * Determines whether to show a composite tooltip containing information for all series.
   * Default value: false
   */
  showToolTipsOnAllSeries?: boolean;
  /**
   * Determines the set of default background, line, text and band colors that will be used in the Chart.
   * Default value: "light"
   */
  theme?: string;
  /**
   * Sets the padding of the chart's title (caption).
   * Default value: [object Object]
   */
  titlePadding?: Padding;
  /**
   * Tooltip background color.
   * Default value: null
   */
  toolTipBackground?: string | null;
  /**
   * User defined tooltip text formatting callback function.
   * Default value: null
   */
  toolTipFormatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Determines the tooltip hide delay in milliseconds.
   * Default value: 4000
   */
  toolTipHideDelay?: number;
  /**
   * Tooltip line color.
   * Default value: null
   */
  toolTipLineColor?: string | null;
  /**
   * Determines the tooltip show delay in milliseconds. Values may range from 0 to 10000 [ms].
   * Default value: 300
   */
  toolTipShowDelay?: number;
  /**
   * An object with settings about the Chart's y-axis (value axis).
   * Default value: [object Object]
   */
  valueAxis?: ChartValueAxis;
  /**
   * Sets the Chart's xAxis.
   * Default value: [object Object]
   */
  xAxis?: ChartXAxis;
}
/**
 Chart is a feature-complete interactive graph library that answers the data visualization needs of any modern web app.
*/
export interface Chart extends BaseElement, ChartProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * The event is raised when the user clicks on a chart annotation.
	* @param event. The custom event.    */
  onAnnotationClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the user moves the cursor above a chart annotation.
	* @param event. The custom event.    */
  onAnnotationMouseenter?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the user moves the cursor out of a chart annotation.
	* @param event. The custom event.    */
  onAnnotationMouseleave?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the user clicks on series element.
	* @param event. The custom event.    */
  onClick: ((this: any, ev: Event) => any) | null;
  /**
   * The event is raised when the user moves the cursor out of a series element.
	* @param event. The custom event.    */
  onMouseout: ((this: any, ev: Event) => any) | null;
  /**
   * The event is raised when the user moves the cursor above a series element.
	* @param event. The custom event.    */
  onMouseover: ((this: any, ev: Event) => any) | null;
  /**
   * The event is raised after the chart's range selector position changes and after the chart ends rendering.
	* @param event. The custom event.    */
  onRangeSelectionChanged?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the chart's range selector position changes and before the chart starts rendering.
	* @param event. The custom event.    */
  onRangeSelectionChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the chart begins rendering.
	* @param event. The custom event.    */
  onRefreshBegin?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when the chart finishes rendering.
	* @param event. The custom event.    */
  onRefreshEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * The event is raised when a serie is toggled by a user click in the chart's legend or through an API call.
	* @param event. The custom event.    */
  onToggle: ((this: any, ev: Event) => any) | null;
  /**
   * Adds a new color sheme. If a scheme with the same name already exists, the method will update its colors.
   * @param {string} schemeName. The name of the custom color scheme.
   * @param {any[]} colorsArray. An array of color values.
   */
  addColorScheme(schemeName: string, colorsArray: any[]): void;
  /**
   * Returns the colors of a color scheme by name. If the scheme doesn't exist the method returns undefined.
   * @param {string} schemeName. The name of the color scheme.
   * @returns {any[]}
   */
  getColorScheme(schemeName: string): any[];
  /**
   * Gets the rendered coordinates of a data point element.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   * @param {number} itemIndex. Item (data point) index.
   * @returns 
   */
  getItemCoord(groupIndex: number, serieIndex: number, itemIndex: number): { x: number, y: number, width: number, height: number, center: number, centerOffset: number, innerRadius: number, outerRadius: number, selectedRadiusChange: number, fromAngle: number, toAngle: number, radius: number };
  /**
   * Gets the number of rendered items in a specific serie.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   * @returns {number}
   */
  getItemsCount(groupIndex: number, serieIndex: number): number;
  /**
   * Gets the rendered coordinates and values of the valueAxis labels.
   * @param {number} groupIndex. Series group index.
   * @returns {any}
   */
  getValueAxisLabels(groupIndex: number): any;
  /**
   * Gets the rendered rectangle coordinates of the valueAxis of specific serie group.
   * @param {number} groupIndex. Series group index.
   * @returns {DOMRect}
   */
  getValueAxisRect(groupIndex: number): DOMRect;
  /**
   * Gets the valueAxis (vertical axis)'s value.
   * @param {number} offset. Vertical offset.
   * @param {number} groupIndex. Series group index.
   * @returns {any}
   */
  getValueAxisValue(offset: number, groupIndex: number): any;
  /**
   * Gets the rendered coordinates and values of the xAxis labels.
   * @param {number} groupIndex. Series group index.
   * @returns {any}
   */
  getXAxisLabels(groupIndex: number): any;
  /**
   * Gets the rendered rectangle coordinates of the x-Axis of specific serie group.
   * @param {number} groupIndex. Series group index.
   * @returns {DOMRect}
   */
  getXAxisRect(groupIndex: number): DOMRect;
  /**
   * Gets the xAxis (horizontal axis)'s value.
   * @param {number} offset. Horizontal offset.
   * @param {number} groupIndex. Series group index.
   * @returns {any}
   */
  getXAxisValue(offset: number, groupIndex: number): any;
  /**
   * Hides a chart serie. The result of calling this function is same as the user unchecking the legend box of a chart serie.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   * @param {number} itemIndex?. Item (data point) index. Applicable to pie and donut series only.
   */
  hideSerie(groupIndex: number, serieIndex: number, itemIndex?: number): void;
  /**
   * Hides the current tooltip if visible.
   * @param {number} hideDelay?. Hide delay.
   */
  hideToolTip(hideDelay?: number): void;
  /**
   * Prints the chart.
   */
  print(): void;
  /**
   * Refreshes the content of the chart element after a property or data update.
   */
  refresh(): void;
  /**
   * Removes an existing color scheme. If the scheme does not exist, the method has no effect.
   * @param {string} schemeName. The name of the custom color scheme.
   */
  removeColorScheme(schemeName: string): void;
  /**
   * Exports the chart's content as JPEG image.
   * @param {string} fileName?. File name.
   */
  saveAsJPEG(fileName?: string): void;
  /**
   * Exports the chart's content as PNG image.
   * @param {string} fileName?. File name.
   */
  saveAsPNG(fileName?: string): void;
  /**
   * Exports the chart's content as PDF.
   * @param {string} fileName?. File name.
   * @param {string} pageOrientation?. PDF page orientation. <strong>Possible values:</strong> 'portrait' (default), 'landscape'.
   */
  saveAsPDF(fileName?: string, pageOrientation?: string): void;
  /**
   * Shows a hidden chart serie. The result of calling this function is same as the user checking the legend box of a chart serie.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   * @param {number} itemIndex?. Item (data point) index. Applicable to pie and donut series only.
   */
  showSerie(groupIndex: number, serieIndex: number, itemIndex?: number): void;
  /**
   * Shows a the tooltip for a particular data point.
   * @param {number} groupIndex. Series group index.
   * @param {number} serieIndex. Series index.
   * @param {number} itemIndex. Item (data point) index.
   * @param {number} showDelay?. Show delay.
   * @param {number} hideDelay?. Hide delay.
   */
  showToolTip(groupIndex: number, serieIndex: number, itemIndex: number, showDelay?: number, hideDelay?: number): void;
  /**
   * Updates the values of the chart series without full refresh of the entire chart. The method should be used for animation of frequently changing values.
   */
  update(): void;
}

/**Localization object containing culture-specific properties required for formatting currencies, numbers and dates. */
export interface ChartLocalization {
  /**
   * A symbol used to mark the border between the integer and fractional parts of a number.
   * Default value: "."
   */
  decimalSeparator?: string;
  /**
   * An object containing datetime formatting patterns.
   * Default value: null
   */
  patterns?: any;
  /**
   * A symbol used to mark the border between thousands, millions, billions, etc.
   * Default value: ""
   */
  thousandsSeparator?: string;
}

/**Sets the left, top, right and bottom padding of the Chart. */
export interface Padding {
  /**
   * Bottom padding of the Chart.
   * Default value: 5
   */
  bottom?: number;
  /**
   * Left padding of the Chart.
   * Default value: 5
   */
  left?: number;
  /**
   * Right padding of the Chart.
   * Default value: 5
   */
  right?: number;
  /**
   * Top padding of the Chart.
   * Default value: 5
   */
  top?: number;
}

export interface ChartSeriesGroup {
  /**
   * An array of chart annotation objects.
   * Default value: null
   */
  annotations?: ChartAnnotation[];
  /**
   * Optional color bands dislayed in the chart's plot area.
   * Default value: null
   */
  bands?: ChartBand[];
  /**
   * Percentage gap between columns within the same serie.
   * Default value: 25
   */
  columnsGapPercent?: number;
  /**
   * Maximum width of columns in column series.
   * Default value: null
   */
  columnsMaxWidth?: number | null;
  /**
   * Minimum width of columns in column series.
   * Default value: 1
   */
  columnsMinWidth?: number;
  /**
   * Columns bottom width (as percentage of the total width).
   * Default value: 100
   */
  columnsBottomWidthPercent?: number;
  /**
   * Columns top width (as percentage of the total width).
   * Default value: 100
   */
  columnsTopWidthPercent?: number;
  /**
   * Determines the drawing function of the series group. When the property is set, you can draw after the series group has been plotted.
   * Default value: null
   */
  draw?: any;
  /**
   * Determines the data source of the serie
   * Default value: null
   */
  dataSource?: any;
  /**
   * Function for custom drawing before the series group.
   * Default value: null
   */
  drawBefore?: any;
  /**
   * Determines whether series are selectable.
   * Default value: true
   */
  enableSelection?: boolean;
  /**
   * Determines whether to toggle series visibility after click on a legend's box.
   * Default value: true
   */
  enableSeriesToggle?: boolean;
  /**
   * The end angle (in degrees) of the polar coordinate system. Applicable to polar and spider charts only.
   * Default value: 360
   */
  endAngle?: number;
  /**
   * Callback function used to format series labels.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Object describing the format settings of series labels.
   * Default value: [object Object]
   */
  formatSettings?: ChartFormatSettings;
  /**
   * Determines how line and area series are unselected based on mouse and touch events. If the value is set to 'click', once a line or area serie is selected, it will remain selected until the user clicks or moves the cursor outside the chart. In default mode, the serie will be unselected immediatelly after the cursor moves over another serie or outside the chart.
   * Default value: default
   */
  linesUnselectMode?: ChartUnselectMode | string;
  /**
   * Horizontal position of the center of the polar coordinate system. Applicable to polar and spider charts only.
   * Default value: null
   */
  offsetX?: number;
  /**
   * Vertical position of the center of the polar coordinate system. Applicable to polar and spider charts only.
   * Default value: null
   */
  offsetY?: number;
  /**
   * Specifies the orientation of the series group.
   * Default value: vertical
   */
  orientation?: Orientation | string;
  /**
   * Specifies the summary of the series group.
   * Default value: ""
   */
  summary?: string;
  /**
   * When polar is set to true, the chart will render in polar coordinates.
   * Default value: false
   */
  polar?: boolean;
  /**
   * The radius of the polar coordinate system. Applicable to polar and spider charts only.
   * Default value: null
   */
  radius?: number;
  /**
   * An array of chart series.
   * Default value: 
   */
  series?: ChartSeriesGroupSerie[];
  /**
   * Percentage gap between columns belonging to different series.
   * Default value: 10
   */
  seriesGapPercent?: number;
  /**
   * Determines whether to display overlapping data points in column, ohlc and candlestick series.
   * Default value: true
   */
  skipOverlappingPoints?: boolean;
  /**
   * When showLabels is set to true, the chart will render pie labels.
   * Default value: true
   */
  showLabels?: boolean;
  /**
   * When spider is set to true, the chart will render in polar coordinates.
   * Default value: false
   */
  spider?: boolean;
  /**
   * The start angle (in degrees) of the polar coordinate system. Applicable to polar and spider charts only.
   * Default value: 0
   */
  startAngle?: number;
  /**
   * Sets the chart type. jqxChart supports several common chart types. You can easily plot series of different types on a common chart. A type must be specified for each series group. Currently, jqxChart supports the following series types:'column' - simple column series'stackedcolumn' - stacked column series'stackedcolumn100' - percentage stacked columns'rangecolumn' - floating column between two values'waterfall' - waterfall series'stackedwaterfall' - stacked waterfall series'line' - simple straight lines connecting the value points'stackedline' - stacked lines'stackedline100' - percentage stacked lines'spline' - smooth lines connecting the value points'stackedspline' - smooth stacked lines'stackedspline100' - percentage stacked smooth lines'stepline' - step line'stackedstepline' - stacked step line'stackedstepline100' - percentage stacked step line'area' - area connecting the value points with straight lines'stackedarea' - stacked area with straight lines between the points'stackedarea100' - percentage stacked area with straight lines between the points'rangearea' - floating area between pairs of value points'splinearea' - smooth area connecting the value points'stackedsplinearea' - stacked smooth area connecting the value points'stackedsplinearea100' - percentage stacked smooth area'splinerangearea' - smooth floating area between pairs of value points'steprangearea' - step area between pairs of value points'stackedsplineara' - smooth stacked area'steparea' - step area connecting the value points'stackedsteparea' - step stacked area'stackedsteparea100' - percentage stacked step area'pie' - circular chart divided into sectors, illustrating proportion'donut' - chart divided into circular sectors with different inner and outer radius'scatter' - data is displayed as a collection of points'stackedscatter' - data is displayed as a collection of points and the values are stacked'stackedscatter100' - data is displayed as a collection of points and the values are percentage stacked'bubble' - data is displayed as a collection of bubbles'stackedbubble' - data is displayed as a collection of bubbles and the values are stacked'stackedbubble100' - data is displayed as a collection of bubbles and the values are percentage stacked'candlestick' - display candlestick series using open, high, low, close data points'ohlc' - display OHLC series using open, high, low, close data points
   * Default value: column
   */
  type?: ChartType | string;
  /**
   * Object describing the format settings of series tooltips.
   * Default value: [object Object]
   */
  toolTipFormatSettings?: ChartFormatSettings;
  /**
   * Sets the tooltip format function. The function is used to format the tooltips of the Chart serie
   * Default value: null
   */
  toolTipFormatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Determines whether to use color gradients.
   * Default value: false
   */
  useGradientColors?: boolean;
  /**
   * Object describing the valueAxis for this group. jqxChart allows you to use a common valueAxis and/or multiple value axes per serie group.
   * Default value: [object Object]
   */
  valueAxis?: ChartValueAxis;
  /**
   * Object describing the xAxis for this group.
   * Default value: [object Object]
   */
  xAxis?: ChartXAxis;
}

export interface ChartAnnotation {
  /**
   * Fill/background color of the annotation.
   * Default value: null
   */
  fillColor?: string | null;
  /**
   * Height of the annotation.
   * Default value: null
   */
  height?: number;
  /**
   * Line color of the annotation.
   * Default value: null
   */
  lineColor?: string | null;
  /**
   * Custom offset of the annotation relative to xValue & yValue.
   * Default value: [object Object]
   */
  offset?: Offset;
  /**
   * Line path command in case the type is 'path', e.g. 'M 10,10 L 20,20 L 50,50'.
   * Default value: null
   */
  path?: string | null;
  /**
   * Radius of the annotation in case its type is 'circle'.
   * Default value: null
   */
  radius?: number;
  /**
   * Object describing the text of the annotation.
   * Default value: [object Object]
   */
  text?: ChartAnnotationText;
  /**
   * Shape type of the annotation.
   * Default value: null
   */
  type?: ChartAnnotationType | string;
  /**
   * Width of the annotation.
   * Default value: null
   */
  width?: number;
  /**
   * Value to determine the horizontal offset of the annotation.
   * Default value: null
   */
  xValue?: number;
  /**
   * Optional 2nd value to determine the horizontal offset of the annotation bottom-right point.
   * Default value: null
   */
  xValue2?: number;
  /**
   * Value to determine the vertical offset of the annotation.
   * Default value: null
   */
  yValue?: number;
  /**
   * Optional 2nd value to determine the vertical offset of the annotation's bottom-right point.
   * Default value: null
   */
  yValue2?: number;
}

/**Custom offset of the annotation relative to <strong>xValue</strong> & <strong>yValue</strong>. */
export interface Offset {
  /**
   * Horizontal offset.
   * Default value: null
   */
  x?: number | null;
  /**
   * Vertical offset.
   * Default value: null
   */
  y?: number | null;
}

/**Object describing the text of the annotation. */
export interface ChartAnnotationText {
  /**
   * Text rotation angle.
   * Default value: null
   */
  angle?: number;
  /**
   * CSS class of the annotation.
   * Default value: null
   */
  class?: string | null;
  /**
   * Inner text color (fill).
   * Default value: null
   */
  fillColor?: string | null;
  /**
   * Horizontal text alignment.
   * Default value: center
   */
  horizontalAlignment?: HorizontalAlignment | string;
  /**
   * Outer text color (stroke).
   * Default value: null
   */
  lineColor?: string | null;
  /**
   * Offset of the annotation text relative to the base annotation offset.
   * Default value: [object Object]
   */
  offset?: Offset;
  /**
   * Position to rotate the text around.
   * Default value: centermiddle
   */
  rotationPoint?: ChartRotationPoint | string;
  /**
   * Text of the annotation.
   * Default value: null
   */
  value?: string | null;
  /**
   * Vertical text alignment.
   * Default value: center
   */
  verticalAlignment?: VerticalAlignment | string;
}

export interface ChartBand {
  /**
   * Color used to fill the area between the minValue and the maxValue.
   * Default value: null
   */
  color?: string | null;
  /**
   * Dash style of the band lines. The style is a series of numbers indicating line length followed by space length.
   * Default value: null
   */
  dashStyle?: string | null;
  /**
   * Band line color.
   * Default value: null
   */
  lineColor?: string | null;
  /**
   * Band line width.
   * Default value: null
   */
  lineWidth?: string | number | null;
  /**
   * End value of the color band.
   * Default value: NaN
   */
  maxValue?: any;
  /**
   * Start value of the color band.
   * Default value: NaN
   */
  minValue?: any;
  /**
   * Fraction indicating the fill opacity.
   * Default value: 1
   */
  opacity?: number;
}

/**Object describing the format settings of series labels. */
export interface ChartFormatSettings {
  /**
   * Optional date format string. This property is applicable only when displaying Date objects.
   * Default value: null
   */
  dateFormat?: string | null;
  /**
   * Decimal places of numeric values.
   * Default value: null
   */
  decimalPlaces?: number | null;
  /**
   * A symbol used to mark the border between the integer and fractional parts of a number. The default value is inherited from localization.decimalSeparator.
   * Default value: null
   */
  decimalSeparator?: string | null;
  /**
   * A boolean value indicating whether to display negative numbers in brackets.
   * Default value: false
   */
  negativeWithBrackets?: boolean;
  /**
   * Text to prepend to the value.
   * Default value: ""
   */
  prefix?: string;
  /**
   * Text to append to the value
   * Default value: ""
   */
  sufix?: string;
  /**
   * A symbol used to mark the border between thousands, millions, billions, etc. The default value is inherited from localization.thousandsSeparator.
   * Default value: null
   */
  thousandsSeparator?: string | null;
}

export interface ChartSeriesGroupSerie {
  /**
   * Offset from the center point in a pie/donut series.
   * Default value: 0
   */
  centerOffset?: number;
  /**
   * A custom function that returns the color of a data point. The function will receive the following parameters: dataValue, itemIndex, serie, group. The implementation of the function can return a single color which will be used as a fillColor and the other colors will be derived or it can return an object containing fillColor, fillColorSelected, etc.
   * Default value: null
   */
  colorFunction?: any;
  /**
   * Color palette to use when rendering the serie.
   * Default value: scheme01
   */
  colorScheme?: ChartColorScheme | string;
  /**
   * Name of the field in the data source.
   * Default value: "null"
   */
  dataField?: string;
  /**
   * Data field used in candlestcik and ohlc series.
   * Default value: "null"
   */
  dataFieldClose?: string;
  /**
   * Data field used in candlestcik and ohlc series.
   * Default value: "null"
   */
  dataFieldHigh?: string;
  /**
   * Data field used in candlestcik and ohlc series.
   * Default value: "null"
   */
  dataFieldLow?: string;
  /**
   * Data field used in candlestcik and ohlc series.
   * Default value: "null"
   */
  dataFieldOpen?: string;
  /**
   * Data field used in in range column series as a start data field.
   * Default value: "null"
   */
  dataFieldFrom?: string;
  /**
   * Data field used in range column series as an end data field.
   * Default value: "null"
   */
  dataFieldTo?: string;
  /**
   * Data field used in bubble series.
   * Default value: "null"
   */
  radiusDataField?: string;
  /**
   * Name to display for this serie.
   * Default value: "null"
   */
  displayText?: string;
  /**
   * Name to display for candlestick and ohlc series.
   * Default value: "null"
   */
  displayTextClose?: string;
  /**
   * Name to display for candlestick and ohlc series.
   * Default value: "null"
   */
  displayTextHigh?: string;
  /**
   * Name to display for candlestick and ohlc series.
   * Default value: "null"
   */
  displayTextLow?: string;
  /**
   * Name to display for candlestick and ohlc series.
   * Default value: "null"
   */
  displayTextOpen?: string;
  /**
   * Determines how to display value gaps in line series.
   * Default value: skip
   */
  emptyPointsDisplay?: ChartSeriesGroupSerieEmptyPointsDisplay | string;
  /**
   * Determines whether the serie is selectable.
   * Default value: true
   */
  enableSelection?: boolean;
  /**
   * Determines whether to toggle the series visibility after click on a legend's box.
   * Default value: true
   */
  enableSeriesToggle?: boolean;
  /**
   * Determines the labels rotation radius when the Chart is 'pie' or 'donut'.
   * Default value: 0
   */
  labelRadius?: number;
  /**
   * Callback function used to format the labels.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Object describing the format settings of the labels.
   * Default value: [object Object]
   */
  formatSettings?: ChartFormatSettings;
  /**
   * Specifies the summary of the series group.
   * Default value: ""
   */
  summary?: string;
  /**
   * Maximum angle in a pie, donut, polar and spider series.
   * Default value: 360
   */
  endAngle?: number;
  /**
   * Fill color for the serie.
   * Default value: null
   */
  fillColor?: string | null;
  /**
   * Alternating fill color for the serie. Applicable to OHLC series only.
   * Default value: null
   */
  fillColorAlt?: string | null;
  /**
   * Alternating fill color for the serie when selected. Applicable to OHLC series only.
   * Default value: null
   */
  fillColorAltSelected?: string | null;
  /**
   * Fill color for the serie when selected.
   * Default value: null
   */
  fillColorSelected?: string | null;
  /**
   * Fill color for the marker symbols in the serie.
   * Default value: null
   */
  fillColorSymbol?: string | null;
  /**
   * Fill color for the the marker symbols in serie when selected.
   * Default value: null
   */
  fillColorSymbolSelected?: string | null;
  /**
   * Determines whether to display the serie in grey scale.
   * Default value: false
   */
  greyScale?: boolean | null;
  /**
   * Determines whether to hide slices of toggled points in pie and donut series.
   * Default value: false
   */
  hiddenPointsDisplay?: boolean;
  /**
   * Initial angle in pie and donut series.
   * Default value: 0
   */
  initialAngle?: number;
  /**
   * Inner radius of donut series in pixels or percents.
   * Default value: 0
   */
  innerRadius?: number;
  /**
   * Object describing the labels properties of the axis.
   * Default value: undefined
   */
  labels?: ChartLabels;
  /**
   * Fill color of the legend box. The default value is inherited from the serie's color.
   * Default value: null
   */
  legendFillColor?: string | null;
  /**
   * Legend data formatting function for the values in the serie.
   * Default value: null
   */
  legendFormatFunction?: any;
  /**
   * Legend data formatting settings for the values in the serie.
   * Default value: [object Object]
   */
  legendFormatSettings?: ChartFormatSettings;
  /**
   * Line color of the legend box. The default value is inherited from the serie's color.
   * Default value: null
   */
  legendLineColor?: string | null;
  /**
   * Line color for the serie.
   * Default value: null
   */
  lineColor?: string | null;
  /**
   * Line color for the serie when selected.
   * Default value: null
   */
  lineColorSelected?: string | null;
  /**
   * Line color for the marker symbols in serie.
   * Default value: null
   */
  lineColorSymbol?: string | null;
  /**
   * Line color for the marker symbols in the serie when selected.
   * Default value: null
   */
  lineColorSymbolSelected?: string | null;
  /**
   * A string sequence of numbers represening line and space lengths, e.g. '2,2'.
   * Default value: null
   */
  lineDashStyle?: string | null;
  /**
   * Determines how line and area series are unselected based on mouse and touch events. If the value is set to 'click', once a line or area serie is selected, it will remain selected until the user clicks or moves the cursor outside the chart. In default mode, the serie will be unselected immediatelly after the cursor moves over another serie or outside the chart.
   * Default value: default
   */
  linesUnselectMode?: ChartUnselectMode | string;
  /**
   * Determines the line tickness of the items in this serie.
   * Default value: null
   */
  lineWidth?: string | null | number;
  /**
   * Determines the line tickness of the items in this serie when selected.
   * Default value: null
   */
  lineWidthSelected?: string | null | number;
  /**
   * Min radius of bubble series in pixels or percents.
   * Default value: null
   */
  minRadius?: string | null | number;
  /**
   * Max radius of bubble series in pixels or percents.
   * Default value: null
   */
  maxRadius?: string | null | number;
  /**
   * Determines the opacity of the items in this serie.
   * Default value: 1
   */
  opacity?: number;
  /**
   * Outer radius of pie and donut series in pixels or percents.
   * Default value: null
   */
  radius?: number | null;
  /**
   * Radius change on selection of pie and donut series in pixels or percents.
   * Default value: null
   */
  selectedRadiusChange?: number | null;
  /**
   * Minimum angle in a pie, donut, polar and spider series.
   * Default value: 0
   */
  startAngle?: number;
  /**
   * Determines the size of the symbol element.
   * Default value: null
   */
  symbolSize?: number;
  /**
   * Determines the size of the symbol element. This property is applicable to line and area series only.
   * Default value: null
   */
  symbolSizeSelected?: number;
  /**
   * Determines the symbol type displayed for the data points in the serie. This parameter is applicable to line, area, scatter and bubble series only.
   * Default value: none
   */
  symbolType?: ChartSymbolType | string;
  /**
   * Determines the tooltip's background.
   * Default value: null
   */
  toolTipBackground?: string | null;
  /**
   * Determines the tooltip's CSS class name.
   * Default value: null
   */
  toolTipClass?: string | null;
  /**
   * Tooltip data formatting function for the values in the serie.
   * Default value: null
   */
  toolTipFormatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Tooltip data formatting settings for the values in the serie.
   * Default value: [object Object]
   */
  toolTipFormatSettings?: ChartFormatSettings;
  /**
   * Determines the tooltip's border lines color.
   * Default value: null
   */
  toolTipLineColor?: string | null;
  /**
   * Determines whether to use color gradients.
   * Default value: false
   */
  useGradientColors?: boolean;
}

/**Object describing the labels properties of the axis. */
export interface ChartLabels {
  /**
   * Text rotation angle.
   * Default value: 0
   */
  angle?: number;
  /**
   * boolean determining if auto rotation is enabled.
   * Default value: false
   */
  autoRotate?: boolean;
  /**
   * Labels background color.
   * Default value: null
   */
  backgroundColor?: string | null;
  /**
   * Labels background opacity.
   * Default value: 1
   */
  backgroundOpacity?: number | null;
  /**
   * Labels border line color.
   * Default value: null
   */
  borderColor?: string | null;
  /**
   * Labels border line opacity.
   * Default value: 1
   */
  borderOpacity?: number | null;
  /**
   * CSS class of the labels.
   * Default value: null
   */
  class?: string | null;
  /**
   * An array of custom values/offsets where a label will be displayed. Applicable only if labels.visible is set to 'custom'.
   * Default value: null
   */
  custom?: [] | null;
  /**
   * Callback function used to format the labels.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Object describing the format settings of the labels.
   * Default value: [object Object]
   */
  formatSettings?: ChartFormatSettings;
  /**
   * Horizontal labels alignment.
   * Default value: center
   */
  horizontalAlignment?: HorizontalAlignment | string;
  /**
   * Determines whether to use direct lines for the labels in pie/donut series.
   * Default value: true
   */
  linesAngles?: boolean;
  /**
   * Determines whether to use lines for the labels in pie/donut series.
   * Default value: true
   */
  linesEnabled?: boolean;
  /**
   * Labels offset.
   * Default value: [object Object]
   */
  offset?: Offset;
  /**
   * Object describing the padding of the labels.
   * Default value: [object Object]
   */
  padding?: Padding;
  /**
   * Radius of the labels in pie/donut series.
   * Default value: null
   */
  radius?: number | null;
  /**
   * Position to rotate the text around.
   * Default value: auto
   */
  rotationPoint?: ChartRotationPoint | string;
  /**
   * Interval steps between label placements (multiples of the axis unit interval).
   * Default value: null
   */
  step?: number | null;
  /**
   * Sets the interval between the labels.
   * Default value: null
   */
  unitInterval?: number;
  /**
   * Vertical labels alignment.
   * Default value: center
   */
  verticalAlignment?: VerticalAlignment | string;
  /**
   * Possible values: true, false, 'custom'.Determines the visibility of labels. When 'custom' is set, displays only custom values/offsets from the labels.custom array.
   * Default value: true
   */
  visible?: boolean | string;
}

/**Object describing the valueAxis for this group. jqxChart allows you to use a common valueAxis and/or multiple value axes per serie group. */
export interface ChartValueAxis {
  /**
   * Alternating background color between grid lines.
   * Default value: ""
   */
  alternatingBackgroundColor?: string;
  /**
   * Second alternating background color.
   * Default value: ""
   */
  alternatingBackgroundColor2?: string;
  /**
   * Opacity of the alternating background.
   * Default value: 1
   */
  alternatingBackgroundOpacity?: number;
  /**
   * Sets the size of the axis.
   * Default value: null
   */
  axisSize?: number | string | null;
  /**
   * Optional color bands dislayed in the chart's plot area.
   * Default value: null
   */
  bands?: ChartBand[];
  /**
   * Sets the baseline value for the axis.
   * Default value: 0
   */
  baselineValue?: any;
  /**
   * boolean determining whether to draw the axis or the user will use APIs to draw it.
   * Default value: false
   */
  customDraw?: boolean;
  /**
   * Sets the description of the value axis.
   * Default value: ""
   */
  description?: string;
  /**
   * Specifies whether the values axis is displayed.
   * Default value: true
   */
  displayValueAxis?: boolean;
  /**
   * Specifies whether the values are displayed in reverse order.
   * Default value: false
   */
  flip?: boolean;
  /**
   * Custom function to format the displayed values along the axis.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Settings used to format the displayed values along the axis.
   * Default value: [object Object]
   */
  formatSettings?: ChartFormatSettings;
  /**
   * Object describing the grid lines properties of the valueAxis.
   * Default value: [object Object]
   */
  gridLines?: ChartLines;
  /**
   * Object describing the labels properties of the axis.
   * Default value: [object Object]
   */
  labels?: ChartLabels;
  /**
   * Object describing the line properties of the axis.
   * Default value: [object Object]
   */
  line?: ChartLine;
  /**
   * Determines whether to use logarithmic scale.
   * Default value: false
   */
  logarithmicScale?: boolean;
  /**
   * Base for logarithmic scale.
   * Default value: 10
   */
  logarithmicScaleBase?: number;
  /**
   * Sets the maximum value of the valueAxis.
   * Default value: NaN
   */
  maxValue?: any;
  /**
   * Sets the minimum value of the valueAxis.
   * Default value: NaN
   */
  minValue?: any;
  /**
   * Object describing the padding of the axis.
   * Default value: [object Object]
   */
  padding?: Padding;
  /**
   * Sets the axis position. The values 'left' and 'right' are available in the default case. If the valueAxis is horizontal, only 'top' and 'bottom' are available.
   * Default value: left
   */
  position?: AxisPosition | string;
  /**
   * Text rotation angle.
   * Default value: null
   */
  textRotationAngle?: number | null;
  /**
   * Value Axis Type
   * Default value: ""
   */
  type?: string;
  /**
   * Position to rotate the text around.
   * Default value: auto
   */
  textRotationPoint?: ChartRotationPoint | string;
  /**
   * Object describing the tick marks properties of the valueAxis.
   * Default value: [object Object]
   */
  tickMarks?: ChartLines;
  /**
   * Object describing the title of the valueAxis.
   * Default value: [object Object]
   */
  title?: ChartTitle;
  /**
   * Sets the interval between the units.
   * Default value: null
   */
  unitInterval?: number | null;
  /**
   * Specifies whether the axis values will be aligned with the tick marks.
   * Default value: true
   */
  valuesOnTicks?: boolean;
  /**
   * Shows or hides the valueAxis.
   * Default value: true
   */
  visible?: boolean;
}

/**Object describing the grid lines properties of the valueAxis. */
export interface ChartLines {
  /**
   * Color of the grid lines.
   * Default value: ""
   */
  color?: string;
  /**
   * An array of custom values/offsets where a grid line will be displayed. Applicable only if gridLines.visible is set to 'custom'.
   * Default value: null
   */
  custom?: [] | null;
  /**
   * Grid lines dash style, e.g. '2,2'.
   * Default value: ""
   */
  dashStyle?: string;
  /**
   * Line width (in pixels) of the grid lines.
   * Default value: 1
   */
  lineWidth?: string | number | null;
  /**
   * Size (in pixels) of the tick marks.
   * Default value: 4
   */
  size?: string | number | null;
  /**
   * Interval steps between grid line placements (multiples of the axis unit interval).
   * Default value: null
   */
  step?: number | null;
  /**
   * Sets the interval between the grid lines.
   * Default value: null
   */
  unitInterval?: number;
  /**
   * Possible values: true, false, 'custom'.Determines the visibility of grid lines. When 'custom' is set, displays only custom values/offsets from the gridLines.custom array.
   * Default value: true
   */
  visible?: boolean | string;
}

/**Object describing the line properties of the axis. */
export interface ChartLine {
  /**
   * Color of axis line.
   * Default value: ""
   */
  color?: string;
  /**
   * Line dash style, e.g. '2,2'. The default is inherited from gridLines.dashStyle.
   * Default value: ""
   */
  dashStyle?: string;
  /**
   * Line width. The default is inherited from gridLines.lineWidth.
   * Default value: 1
   */
  lineWidth?: number;
  /**
   * boolean determining the visibility of the axis line.
   * Default value: true
   */
  visible?: boolean;
}

/**Object describing the title of the valueAxis. */
export interface ChartTitle {
  /**
   * CSS class of the title text.
   * Default value: null
   */
  class?: string | null;
  /**
   * Horizontal alignment.
   * Default value: center
   */
  horizontalAlignment?: HorizontalAlignment | string;
  /**
   * Text of the title.
   * Default value: ""
   */
  text?: string;
  /**
   * Vertical alignment.
   * Default value: center
   */
  verticalAlignment?: VerticalAlignment | string;
  /**
   * boolean determining the visibility of the title.
   * Default value: true
   */
  visible?: boolean;
}

/**Object describing the xAxis for this group. */
export interface ChartXAxis {
  /**
   * Alternating background color between grid lines.
   * Default value: ""
   */
  alternatingBackgroundColor?: string;
  /**
   * Second alternating background color.
   * Default value: ""
   */
  alternatingBackgroundColor2?: string;
  /**
   * Opacity of the alternating background.
   * Default value: 1
   */
  alternatingBackgroundOpacity?: number;
  /**
   * Sets the size of the xAxis.
   * Default value: null
   */
  axisSize?: number | string | null;
  /**
   * Optional color bands dislayed in the chart's plot area.
   * Default value: null
   */
  bands?: ChartBand[];
  /**
   * The base unit when used with 'date' axis.
   * Default value: null
   */
  baseUnit?: ChartBaseUnit | null | string;
  /**
   * boolean determining whether to draw the axis or the user will use APIs to draw it.
   * Default value: false
   */
  customDraw?: boolean;
  /**
   * Points to a data field in the data source.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Optional date format for parsing the data from the data source. Applicable when xAxis.type is set to 'date'.
   * Default value: null
   */
  dateFormat?: string | null;
  /**
   * Optional custom xAxis display text.
   * Default value: null
   */
  displayText?: string | null;
  /**
   * Specifies whether the values are displayed in reverse order.
   * Default value: false
   */
  flip?: boolean;
  /**
   * Custom function to format the displayed values along the axis.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Settings used to format the displayed values along the axis.
   * Default value: [object Object]
   */
  formatSettings?: ChartFormatSettings;
  /**
   * Object describing the grid lines properties of the xAxis.
   * Default value: [object Object]
   */
  gridLines?: ChartLines;
  /**
   * Object describing the labels properties of the axis.
   * Default value: [object Object]
   */
  labels?: ChartLabels;
  /**
   * Object describing the line properties of the axis.
   * Default value: [object Object]
   */
  line?: ChartLine;
  /**
   * Determines whether to use logarithmic scale.
   * Default value: false
   */
  logarithmicScale?: boolean;
  /**
   * Base for logarithmic scale.
   * Default value: 10
   */
  logarithmicScaleBase?: number;
  /**
   * Sets the maximum value of the xAxis.
   * Default value: NaN
   */
  maxValue?: any;
  /**
   * Sets the minimum value of the xAxis.
   * Default value: NaN
   */
  minValue?: any;
  /**
   * Object describing the padding of the axis.
   * Default value: [object Object]
   */
  padding?: Padding;
  /**
   * Sets the axis position. The values 'bottom' and 'top' are available in the default case. If the xAxis is vertical, only 'left' and 'right' are available.
   * Default value: bottom
   */
  position?: AxisPosition | string;
  /**
   * Definition of a range selector on the xAxis. The range selector itself is also an instance of smart-chart.
   * Default value: [object Object]
   */
  rangeSelector?: ChartRangeSelector;
  /**
   * xAxis's text value.
   * Default value: ""
   */
  text?: string;
  /**
   * Text rotation angle.
   * Default value: null
   */
  textRotationAngle?: number | null;
  /**
   * Position to rotate the text around.
   * Default value: auto
   */
  textRotationPoint?: ChartRotationPoint | string;
  /**
   * Object describing the tick marks properties of the xAxis.
   * Default value: [object Object]
   */
  tickMarks?: ChartLines;
  /**
   * Object describing the title of the xAxis.
   * Default value: [object Object]
   */
  title?: ChartTitle;
  /**
   * Custom function to format xAxis values in tooltips.
   * Default value: null
   */
  toolTipFormatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Settings used to format xAxis values in tooltips.
   * Default value: [object Object]
   */
  toolTipFormatSettings?: ChartFormatSettings;
  /**
   * The type of the axis. 'auto' - automatically detects and switches to 'basic', 'linear' or 'date'.'date' - when displaying dates.'basic' - displays all data points sequentially.'linear' - linear arrangement by the value of the xAxis data field.
   * Default value: auto
   */
  type?: ChartXAxisType | string;
  /**
   * Sets the interval between the units.
   * Default value: null
   */
  unitInterval?: number | null;
  /**
   * Specifies whether the axis values will be aligned with the tick marks.
   * Default value: true
   */
  valuesOnTicks?: boolean;
  /**
   * Shows or hides the xAxis.
   * Default value: true
   */
  visible?: boolean;
}

/**Definition of a range selector on the xAxis. The range selector itself is also an instance of ${namespace.toLowerCase()}-chart. */
export interface ChartRangeSelector {
  /**
   * Sets the range selector chart's background color.
   * Default value: null
   */
  backgroundColor?: string | null;
  /**
   * Sets the range selector chart's background image.
   * Default value: ""
   */
  backgroundImage?: string;
  /**
   * The base unit when used with 'date' axis.
   * Default value: null
   */
  baseUnit?: ChartBaseUnit | null | string;
  /**
   * Sets the range selector chart's border color.
   * Default value: null
   */
  borderLineColor?: string | null;
  /**
   * Sets the range selector chart's border line width.
   * Default value: null
   */
  borderLineWidth?: number | null;
  /**
   * Sets the caption (title) of the range selector chart.
   * Default value: ""
   */
  caption?: string;
  /**
   * Sets the range selector chart's color pallete. jqxChart suppports 32 color schemes from 'scheme01' to 'scheme32'.
   * Default value: scheme01
   */
  colorScheme?: ChartColorScheme | string;
  /**
   * Enables or disables overlapping of the column series.
   * Default value: false
   */
  columnSeriesOverlap?: boolean;
  /**
   * Percentage gap between columns within the same serie.
   * Default value: 25
   */
  columnsGapPercent?: number;
  /**
   * Points to a data field in the data source.
   * Default value: null
   */
  dataField?: string | null;
  /**
   * Sets the description text of the range selector chart.
   * Default value: ""
   */
  description?: string;
  /**
   * Determines whether to display the range selector chart using greyscale colors.
   * Default value: false
   */
  greyScale?: boolean | null;
  /**
   * Object describing the grid lines properties of the range selector chart's xAxis.
   * Default value: [object Object]
   */
  gridLines?: ChartLines;
  /**
   * Callback function used to format the values.
   * Default value: null
   */
  formatFunction?: {(value?: any, index?: number, series?: any): string};
  /**
   * Chart Format Settings
   * Default value: null
   */
  formatSettings?: any;
  /**
   * Object describing the labels properties of the axis.
   * Default value: [object Object]
   */
  labels?: ChartLabels;
  /**
   * Sets the maximum value of the range selector chart's xAxis.
   * Default value: NaN
   */
  maxValue?: any;
  /**
   * Sets the minimum value of the range selector chart's xAxis.
   * Default value: NaN
   */
  minValue?: any;
  /**
   * Object describing the padding of the range selector chart.
   * Default value: [object Object]
   */
  padding?: Padding;
  /**
   * Sets the range selector chart position.
   * Default value: left
   */
  position?: AxisPosition | string;
  /**
   * An HTML element outside the chart to render the range selector chart to.
   * Default value: null
   */
  renderTo?: HTMLElement | null;
  /**
   * Sets or gets a value indicating whether the range selector chart's layout is mirrored.
   * Default value: false
   */
  rightToLeft?: boolean | null;
  /**
   * Percentage gap between columns belonging to different series.
   * Default value: 10
   */
  seriesGapPercent?: number;
  /**
   * (Optional) The seriesGroups property is used to describe all series displayed on the range selector chart.Please refer to the main seriesGroups property entry for more information and a full subproperty list.
   * Default value: null
   */
  seriesGroups?: [] | null;
  /**
   * Sets the range selector chart type. jqxChart supports several common chart types. You can easily plot series of different types on a common chart. A type must be specified for each series group. Currently, jqxChart supports the following series types:'column' - simple column series'stackedcolumn' - stacked column series'stackedcolumn100' - percentage stacked columns'rangecolumn' - floating column between two values'waterfall' - waterfall series'stackedwaterfall' - stacked waterfall series'line' - simple straight lines connecting the value points'stackedline' - stacked lines'stackedline100' - percentage stacked lines'spline' - smooth lines connecting the value points'stackedspline' - smooth stacked lines'stackedspline100' - percentage stacked smooth lines'stepline' - step line'stackedstepline' - stacked step line'stackedstepline100' - percentage stacked step line'area' - area connecting the value points with straight lines'stackedarea' - stacked area with straight lines between the points'stackedarea100' - percentage stacked area with straight lines between the points'rangearea' - floating area between pairs of value points'splinearea' - smooth area connecting the value points'stackedsplinearea' - stacked smooth area connecting the value points'stackedsplinearea100' - percentage stacked smooth area'splinerangearea' - smooth floating area between pairs of value points'steprangearea' - step area between pairs of value points'stackedsplineara' - smooth stacked area'steparea' - step area connecting the value points'stackedsteparea' - step stacked area'stackedsteparea100' - percentage stacked step area'pie' - circular chart divided into sectors, illustrating proportion'donut' - chart divided into circular sectors with different inner and outer radius'scatter' - data is displayed as a collection of points'stackedscatter' - data is displayed as a collection of points and the values are stacked'stackedscatter100' - data is displayed as a collection of points and the values are percentage stacked'bubble' - data is displayed as a collection of bubbles'stackedbubble' - data is displayed as a collection of bubbles and the values are stacked'stackedbubble100' - data is displayed as a collection of bubbles and the values are percentage stacked'candlestick' - display candlestick series using open, high, low, close data points'ohlc' - display OHLC series using open, high, low, close data points
   * Default value: area
   */
  serieType?: ChartType | string;
  /**
   * Determines whether to display the range selector chart's border line.
   * Default value: false
   */
  showBorderLine?: boolean | null;
  /**
   * Sets the size of the range selector chart.
   * Default value: null
   */
  size?: number | null;
  /**
   * Determines whether to display overlapping data points in column, ohlc and candlestick series.
   * Default value: true
   */
  skipOverlappingPoints?: boolean;
  /**
   * Sets the padding of the chart's title (caption).
   * Default value: [object Object]
   */
  titlePadding?: Padding;
  /**
   * Sets the interval between the units.
   * Default value: null
   */
  unitInterval?: number | null;
  /**
   * An object with settings about the range selector Chart's y-axis (value axis).
   * Default value: [object Object]
   */
  valueAxis?: ChartValueAxis;
  /**
   * Shows or hides the range selector.
   * Default value: false
   */
  visible?: boolean;
}

declare global {
    interface Document {
        createElement(tagName: "smart-chart"): Chart;
        querySelector(selectors: "smart-chart"): Chart | null;
        querySelectorAll(selectors: "smart-chart"): NodeListOf<Chart>;
        getElementsByTagName(qualifiedName: "smart-chart"): HTMLCollectionOf<Chart>;
        getElementsByName(elementName: "smart-chart"): NodeListOf<Chart>;
    }
}

/**Sets the range selector chart's color pallete. jqxChart suppports 32 color schemes from 'scheme01' to 'scheme32'. */
export declare type ChartColorScheme = 'scheme01' | 'scheme02' | 'scheme03' | 'scheme04' | 'scheme05' | 'scheme06' | 'scheme07' | 'scheme08' | 'scheme09' | 'scheme10' | 'scheme11' | 'scheme12' | 'scheme13' | 'scheme14' | 'scheme15' | 'scheme16' | 'scheme17' | 'scheme18' | 'scheme19' | 'scheme20' | 'scheme21' | 'scheme22' | 'scheme23' | 'scheme24' | 'scheme25' | 'scheme26' | 'scheme27' | 'scheme28' | 'scheme29' | 'scheme30' | 'scheme31' | 'scheme32' | 'custom';
/**Determines the rendering engine used to display the chart. When the property is set to an empty string, jqxChart will automatically select an optimal rendering engine depending on the browser capabilities. */
export declare type ChartRenderEngine = null | 'SVG' | 'HTML5';
/**Horizontal alignment. */
export declare type HorizontalAlignment = 'left' | 'center' | 'right';
/**Position to rotate the text around. */
export declare type ChartRotationPoint = 'auto' | 'left' | 'center' | 'right' | 'topleft' | 'topcenter' | 'topright' | 'bottomleft' | 'bottomcenter' | 'bottomright' | 'centermiddle';
/**Vertical alignment. */
export declare type VerticalAlignment = 'top' | 'center' | 'bottom';
/**Shape type of the annotation. */
export declare type ChartAnnotationType = 'text' | 'rect' | 'circle' | 'line' | 'path' | 'null';
/**Determines how line and area series are unselected based on mouse and touch events. If the value is set to 'click', once a line or area serie is selected, it will remain selected until the user clicks or moves the cursor outside the chart. In default mode, the serie will be unselected immediatelly after the cursor moves over another serie or outside the chart. */
export declare type ChartUnselectMode = 'click' | 'default';
/**Determines how to display value gaps in line series. */
export declare type ChartSeriesGroupSerieEmptyPointsDisplay = 'connect' | 'skip' | 'zero';
/**Determines the symbol type displayed for the data points in the serie. This parameter is applicable to line, area, scatter and bubble series only. */
export declare type ChartSymbolType = 'none' | 'circle' | 'square' | 'diamond' | 'triangle_up' | 'triangle_down' | 'triangle_left' | 'triangle_right';
/**Sets the range selector chart type. jqxChart supports several common chart types. You can easily plot series of different types on a common chart. A type must be specified for each series group. Currently, jqxChart supports the following series types:'column' - simple column series
'stackedcolumn' - stacked column series
'stackedcolumn100' - percentage stacked columns
'rangecolumn' - floating column between two values
'waterfall' - waterfall series
'stackedwaterfall' - stacked waterfall series
'line' - simple straight lines connecting the value points
'stackedline' - stacked lines
'stackedline100' - percentage stacked lines
'spline' - smooth lines connecting the value points
'stackedspline' - smooth stacked lines
'stackedspline100' - percentage stacked smooth lines
'stepline' - step line
'stackedstepline' - stacked step line
'stackedstepline100' - percentage stacked step line
'area' - area connecting the value points with straight lines
'stackedarea' - stacked area with straight lines between the points
'stackedarea100' - percentage stacked area with straight lines between the points
'rangearea' - floating area between pairs of value points
'splinearea' - smooth area connecting the value points
'stackedsplinearea' - stacked smooth area connecting the value points
'stackedsplinearea100' - percentage stacked smooth area
'splinerangearea' - smooth floating area between pairs of value points
'steprangearea' - step area between pairs of value points
'stackedsplineara' - smooth stacked area
'steparea' - step area connecting the value points
'stackedsteparea' - step stacked area
'stackedsteparea100' - percentage stacked step area
'pie' - circular chart divided into sectors, illustrating proportion
'donut' - chart divided into circular sectors with different inner and outer radius
'scatter' - data is displayed as a collection of points
'stackedscatter' - data is displayed as a collection of points and the values are stacked
'stackedscatter100' - data is displayed as a collection of points and the values are percentage stacked
'bubble' - data is displayed as a collection of bubbles
'stackedbubble' - data is displayed as a collection of bubbles and the values are stacked
'stackedbubble100' - data is displayed as a collection of bubbles and the values are percentage stacked
'candlestick' - display candlestick series using open, high, low, close data points
'ohlc' - display OHLC series using open, high, low, close data points
 */
export declare type ChartType = 'column' | 'stackedcolumn' | 'stackedcolumn100' | 'rangecolumn' | 'waterfall' | 'stackedwaterfall' | 'line' | 'stackedline' | 'stackedline100' | 'spline' | 'stackedspline' | 'stackedspline100' | 'stepline' | 'stackedstepline' | 'stackedstepline100' | 'area' | 'stackedarea' | 'stackedarea100' | 'rangearea' | 'splinearea' | 'stackedsplinearea' | 'stackedsplinearea100' | 'splinerangearea' | 'steprangearea' | 'stackedsplineara' | 'steparea' | 'stackedsteparea' | 'stackedsteparea100' | 'pie' | 'donut' | 'scatter' | 'stackedscatter' | 'stackedscatter100' | 'bubble' | 'stackedbubble' | 'stackedbubble100' | 'candlestick' | 'ohlc';
/**Sets the range selector chart position. */
export declare type AxisPosition = 'bottom' | 'top' | 'left' | 'right';
/**The base unit when used with 'date' axis. */
export declare type ChartBaseUnit = null | 'year' | 'month' | 'day' | 'hour' | 'minute' | 'second' | 'millisecond';
/**The type of the axis. 'auto' - automatically detects and switches to 'basic', 'linear' or 'date'.
'date' - when displaying dates.
'basic' - displays all data points sequentially.
'linear' - linear arrangement by the value of the xAxis data field.
 */
export declare type ChartXAxisType = 'auto' | 'date' | 'basic' | 'linear';
export interface CheckBoxProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Sets or gets the checked state. 
   * Default value: false
   */
  checked?: boolean;
  /**
   * Determines which part of the element can be used to toggle it.
   * Default value: both
   */
  checkMode?: CheckMode | string;
  /**
   * Sets the click mode of the checkbox.
   * Default value: release
   */
  clickMode?: ClickMode | string;
  /**
   * Enables or disables the checkbox. 
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the elements's innerHTML.
   * Default value: """"
   */
  innerHTML: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ":  Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: """"
   */
  name?: string;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets or gets the value of the element.
   * Default value: """"
   */
  value?: string;
}
/**
 Checkbox is a component used for allowing a user to make a multiple choice. Broadly used in the forms and surveys.
*/
export interface CheckBox extends BaseElement, CheckBoxProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the widget is checked/unchecked.
	* @param event. The custom event. Custom data event was created with: ev.detail(value, oldValue, changeType)
   *  value - A boolean value indicating the new state of the button ( checked or not ).
   *  oldValue - A boolean value indicating the previous state of the button ( checked or not ).
   *  changeType - A string flag indicating whether the change event was triggered via API or an event.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the widget is checked.
	* @param event. The custom event. Custom data event was created with: ev.detail(changeType)
   *  changeType - A string flag indicating whether the change event was triggered via API or an event.
   */
  onCheckValue?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the widget is unchecked.
	* @param event. The custom event. Custom data event was created with: ev.detail(changeType)
   *  changeType - A string flag indicating whether the change event was triggered via API or an event.
   */
  onUncheckValue?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-check-box"): CheckBox;
        querySelector(selectors: "smart-check-box"): CheckBox | null;
        querySelectorAll(selectors: "smart-check-box"): NodeListOf<CheckBox>;
        getElementsByTagName(qualifiedName: "smart-check-box"): HTMLCollectionOf<CheckBox>;
        getElementsByName(elementName: "smart-check-box"): NodeListOf<CheckBox>;
    }
}

/**Determines which part of the element can be used to toggle it. */
export declare type CheckMode = 'both' | 'input' | 'label';
export interface CheckInputProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the delay before the drop down opens to show the matches from the auto complete operation. The delay is measured in miliseconds.
   * Default value: 100
   */
  autoCompleteDelay?: number;
  /**
   * Determines the data source that will be loaded to the Input. The dataSource can be an array of strings/numbers or objects where the attributes represent the properties of a List Item. For example label, value. It can also be a callback that returns an Array of items as previously described.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines the position of the drop down button.
   * Default value: right
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to an empty string. In this case the height of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Sets the purpose of the input and what, if any, permission the user agent has to provide automated assistance in filling out the element's input when in a form, as well as guidance to the browser as to the type of information expected in the element. This value corresponds to the standard HTML autocomplete attribute and can be set to values such as 'on', 'name', 'organization', 'street-address', etc.
   * Default value: "off"
   */
  inputPurpose?: string;
  /**
   * Determines the maximum number of matched items that should be visible inside the drop down as a result of a new autoComplete query. By default the maximum number of 8 items can be displayed inside the drop down.
   * Default value: 8
   */
  items?: number;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the minimum number of characters inside the input in order to trigger the autocomplete functionality that will open the drop down and show the matched items.
   * Default value: 1
   */
  minLength?: number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the drop down is opened or not.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Sets or gets the query that is used to filter the items. Query is used by the autoComplete operation. Empty string means that all items from the data source will be displayed and no filter query is applied.
   * Default value: 
   */
  query?: string | number;
  /**
   * Determines the auto complete query mode. This property also determines the matching algorithm for the autocomplete operation.
   * Default value: containsIgnoreCase
   */
  queryMode?: CheckInputQueryMode | string;
  /**
   * Determines whether ot not the user can enter text inside the input. if dropDownButtonPosition is set to 'left' or 'right' then readonly determines whether the element acts as a ComboBox or a DropDownList if a dataSource is provided.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the delimiter between the selected items in the input.
   * Default value: ","
   */
  separator?: string;
  /**
   * Determines whether an additional item is displayed as the first item in the options list, which allows to select/unselect all items.
   * Default value: false
   */
  selectAll?: boolean;
  /**
   * Gets or sets an array of selected values.
   * Default value: 
   */
  selectedValues?: any;
  /**
   * Determines whether the items are sorted alphabetically or not
   * Default value: false
   */
  sorted?: boolean;
  /**
   * Determines the sorting algorithm - ascending(asc) or descending(desc) if sort is enabled.
   * Default value: "asc"
   */
  sortDirection?: string;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines the input type. Input type determines what input can be entered.
   * Default value: ""
   */
  type?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element.
   * Default value: ""
   */
  value?: string;
}
/**
 CheckInput specifies an input field where the user can enter data. Auto-complete options with checkboxes are displayed for easier input. Checkboxes allow to select/unselect one or multiple items.
*/
export interface CheckInput extends BaseElement, CheckInputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, oldLabel, oldValue, value)
   *  label - The label of the new selected item.
   *  oldLabel - The label of the item that was previously selected before the event was triggered.
   *  oldValue - The value of the item that was previously selected before the event was triggered.
   *  value - The value of the new selected item.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Closes the drop down.
   */
  close(): void;
  /**
   * Ensures that the active ( selected ) item is always visible.
   */
  ensureVisible(): void;
  /**
   * Opens the drop down.
   */
  open(): void;
  /**
   * Selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-check-input"): CheckInput;
        querySelector(selectors: "smart-check-input"): CheckInput | null;
        querySelectorAll(selectors: "smart-check-input"): NodeListOf<CheckInput>;
        getElementsByTagName(qualifiedName: "smart-check-input"): HTMLCollectionOf<CheckInput>;
        getElementsByName(elementName: "smart-check-input"): NodeListOf<CheckInput>;
    }
}

/**Determines the position of the drop down button. */
export declare type DropDownButtonPosition = 'none' | 'left' | 'right' | 'top' | 'bottom';
/**Determines the auto complete query mode. This property also determines the matching algorithm for the autocomplete operation. */
export declare type CheckInputQueryMode = 'contains' | 'containsIgnoreCase' | 'doesNotContain' | 'doesNotContainIgnoreCase' | 'equals' | 'equalsIgnoreCase' | 'startsWith' | 'startsWithIgnoreCase' | 'endsWith' | 'endsWithIgnoreCase';
export interface ChipProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Sets a custom avatar that is positioned on the left side of the chip. The avatar can be an image(if the value is a url to an image), plain text or HTML.
   * Default value: null
   */
  avatar?: string | null;
  /**
   * Determines whether a close button is displayed on the right side of the element.
   * Default value: false
   */
  closeButton?: boolean;
  /**
   * Enables or disables the element. Disabled elements can not be interacted with.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets a custom template for the chip. The template can be a string that represents the id of an HTMLTemplateElement inside the DOM or it's direct reference.
   * Default value: null
   */
  itemTemplate?: any;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * [object Object]
   */
  messages?: any;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets the text content of the chip. The text inside the chip represents it's value. Value must be of type string. By default it's an empty string.
   * Default value: ""
   */
  value?: string;
}
/**
 Chip is a simple rounded box that displays a single value and optionally an icon.
*/
export interface Chip extends BaseElement, ChipProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the chip is closed.
	* @param event. The custom event. Custom data event was created with: ev.detail(value)
   *  value - A string representing the current value of the element.
   */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * Closes the chip and removes it from the DOM.
   */
  close(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-chip"): Chip;
        querySelector(selectors: "smart-chip"): Chip | null;
        querySelectorAll(selectors: "smart-chip"): NodeListOf<Chip>;
        getElementsByTagName(qualifiedName: "smart-chip"): HTMLCollectionOf<Chip>;
        getElementsByName(elementName: "smart-chip"): NodeListOf<Chip>;
    }
}

export interface CircularProgressBarProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Enables or disables the element. 
   * Default value: false
   */
  disabled?: boolean;
  /**
   * A callback function defining the new format for the label of the Progress Bar.
   * Default value: null
   */
  formatFunction?: {(value: number): string};
  /**
   * Sets the value of the Circular Progress bar to indeterminate state(null) and starts the animation.
   * Default value: false
   */
  indeterminate?: boolean;
  /**
   * Sets the filling direction of the Circular Progress Bar.
   * Default value: false
   */
  inverted?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback, related to localization module. 
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets progress bars maximum possible value.
   * Default value: 100
   */
  max?: number;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ":  Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets progress bars minimum possible value.
   * Default value: 0
   */
  min?: number;
  /**
   * Enables/Disabled the label for the Progress Bar.
   * Default value: false
   */
  showProgressValue?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the progress bar
   * Default value: 0
   */
  value?: number;
}
/**
 Progress Bar displayed as a circle.
*/
export interface CircularProgressBar extends BaseElement, CircularProgressBarProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the value is changed.
	* @param event. The custom event.    */
  onChange: ((this: any, ev: Event) => any) | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-circular-progress-bar"): CircularProgressBar;
        querySelector(selectors: "smart-circular-progress-bar"): CircularProgressBar | null;
        querySelectorAll(selectors: "smart-circular-progress-bar"): NodeListOf<CircularProgressBar>;
        getElementsByTagName(qualifiedName: "smart-circular-progress-bar"): HTMLCollectionOf<CircularProgressBar>;
        getElementsByName(elementName: "smart-circular-progress-bar"): NodeListOf<CircularProgressBar>;
    }
}

export interface ColorInputProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the delay before the drop down opens to show the matches from the auto complete operation. The delay is measured in miliseconds.
   * Default value: 100
   */
  autoCompleteDelay?: number;
  /**
   * Determines the data source ( that represent valid colors ) that will be loaded to the Input. The dataSource can be an array of strings or objects where the attributes represent the properties of a List Item. For example label, value. It can also be a callback that returns an Array of items as previously described.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines the colors that will be displayed and their layout.
   * Default value: default
   */
  displayMode?: ColorInputDisplayMode | string;
  /**
   * Determines the position of the drop down button.
   * Default value: none
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to an empty string. In this case the height of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Sets the purpose of the input and what, if any, permission the user agent has to provide automated assistance in filling out the element's input when in a form, as well as guidance to the browser as to the type of information expected in the element. This value corresponds to the standard HTML autocomplete attribute and can be set to values such as 'on', 'name', 'organization', 'street-address', etc.
   * Default value: "off"
   */
  inputPurpose?: string;
  /**
   * Determines the maximum number of matched items that should be visible inside the drop down as a result of a new autoComplete query. By default the maximum number of 8 items can be displayed inside the drop down.
   * Default value: 8
   */
  items?: number;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the minimum number of characters inside the input in order to trigger the autocomplete functionality that will open the drop down and show the matched items.
   * Default value: 1
   */
  minLength?: number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the drop down is opened or not.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Sets or gets the query that is used to filter the items. Query is used by the autoComplete operation. Empty string means that all items from the data source will be displayed and no filter query is applied.
   * Default value: 
   */
  query?: string | number;
  /**
   * Determines the auto complete query mode. This property also determines the matching algorithm for the autocomplete operation.
   * Default value: containsIgnoreCase
   */
  queryMode?: ColorQueryMode | string;
  /**
   * Determines whether the user can enter text inside the input or not. Determines whether the element acts as a ComboBox or a DropDownList if a dataSource is provided.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element.
   * Default value: ""
   */
  value?: string;
  /**
   * Determines what will be displayed inside the color picker's action section.
   * Default value: default
   */
  valueDisplayMode?: ColorValueDisplayMode | string;
  /**
   * Determines the format of the color. Whether it's in HEX, RGB or RGBA. By default it shows the color depending on the displayMode.
   * Default value: default
   */
  valueFormat?: ColorValueFormat | string;
}
/**
 ColorInput is an input field with colors displayed in a DropDown grid like in Excel.
*/
export interface ColorInput extends BaseElement, ColorInputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, oldLabel, oldValue, value)
   *  label - The label of the new selected color.
   *  oldLabel - The label of the color that was previously selected before the event was triggered.
   *  oldValue - The value of the color that was previously selected before the event was triggered.
   *  value - The value of the new selected color.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Closes the drop down.
   */
  close(): void;
  /**
   * Opens the drop down.
   */
  open(): void;
  /**
   * Selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-color-input"): ColorInput;
        querySelector(selectors: "smart-color-input"): ColorInput | null;
        querySelectorAll(selectors: "smart-color-input"): NodeListOf<ColorInput>;
        getElementsByTagName(qualifiedName: "smart-color-input"): HTMLCollectionOf<ColorInput>;
        getElementsByName(elementName: "smart-color-input"): NodeListOf<ColorInput>;
    }
}

/**Determines the colors that will be displayed and their layout. */
export declare type ColorInputDisplayMode = 'default' | 'grid';
/**Determines the auto complete query mode. This property also determines the matching algorithm for the autocomplete operation. */
export declare type ColorQueryMode = 'contains' | 'containsIgnoreCase' | 'doesNotContain' | 'doesNotContainIgnoreCase' | 'equals' | 'equalsIgnoreCase' | 'startsWith' | 'startsWithIgnoreCase' | 'endsWith' | 'endsWithIgnoreCase';
/**Determines what will be displayed inside the color picker's action section. */
export declare type ColorValueDisplayMode = 'default' | 'colorBox' | 'colorCode' | 'none';
/**Determines the format of the color. Whether it's in HEX, RGB or RGBA. By default it shows the color depending on the displayMode. */
export declare type ColorValueFormat = 'default' | 'rgb' | 'rgba' | 'hex';
export interface ColorPanelProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Specifies how the value is applied.
   * Default value: instantly
   */
  applyValueMode?: ColorApplyValueMode | string;
  /**
   * Defines the number of columns for the colors in displayModes 'grid', 'hexagonal' and 'spectrumGrid'.
   * Default value: 8
   */
  columnCount?: number;
  /**
   * Enables or disables the element. Disabled elements can not be interacted with.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines the colors that will be displayed and their layout.
   * Default value: default
   */
  displayMode?: ColorDisplayMode | string;
  /**
   * By default clicking on color panel's preview container returns the color value to it's previous state. 'disableUndo' prevents this functionality.
   * Default value: false
   */
  disableUndo?: boolean;
  /**
   * Allows to edit the alpha(transparency) of the colors via an editor/slider in the following displayModes: 'palette', 'radial', 'hexagonal'
   * Default value: false
   */
  editAlphaChannel?: boolean;
  /**
   * Allows to select a custom color via an editor popup. Custom color selection is available in modes that don't have this option by default, like: 'grid', 'default, 'spectrum grid'.
   * Default value: false
   */
  enableCustomColors?: boolean;
  /**
   * Defines an Array of colors that will be used as the Theme Colors in the corresponding section in displayMode: 'default'.
   * Default value: null
   */
  gridThemeColors?: string[] | null;
  /**
   * Defines an Array of colors that will be used as the Shade Colors in the corresponding section of displayMode: 'default'.
   * Default value: null
   */
  gridShadeColors?: string[] | null;
  /**
   * Defines an Array of colors that will be used as the Standart Colors in the corresponding section of displayMode: 'default'.
   * Default value: null
   */
  gridStandardColors?: [] | null;
  /**
   * Hides the alpha editor. Alpha editor is an input containing the value of the current color opacity. The input is available in the following modes: 'radial', 'palette', 'hexagonal'. The input is only visible if there's enough space. This editor is visible by default.
   * Default value: false
   */
  hideAlphaEditor?: boolean;
  /**
   * Determines which color editors will be hidden first when there's not enough space for all of them to be visible. By default the editors are only visible in 'palette', 'radial' and 'hexagonal' display modes. This property allows to prioritize the visibility of the editors.
   * Default value: RGB,HEX,alpha,previewContainer
   */
  hideContentToFit?: string[];
  /**
   * HEX editor is an input containing the hexadecimal representation of a color. This editor is visible by default. Setting 'hideRGBeditor' to true hides it.
   * Default value: false
   */
  hideHEXEditor?: boolean;
  /**
   * Hides the preview container. Preview container is used to show the currently selected value in 'palette', 'radial' and 'hexagonal' display modes.
   * Default value: false
   */
  hidePreviewContainer?: boolean;
  /**
   * Hides the RGB editor. This editor is a group of three separate inputs for the Red, Green and Blue values of the color.
   * Default value: false
   */
  hideRGBEditor?: boolean;
  /**
   * Inverts the colors in 'spectrumGrid', 'hexagonal', 'radial' modes.
   * Default value: false
   */
  inverted?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages.
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid ''!",
   *     "propertyInvalidValueType": "Invalid ''!",
   *     "methodInvalidValueType": "Invalid ''!",
   *     "methodInvalidArgumentsCount": "Invalid '' argument(s)!",
   *     "methodInvalidReturnType": "Invalid ''!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": "'.",
   *     "htmlTemplateNotSuported": ": Web Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": ".",
   *     "redPrefix": "R:",
   *     "greenPrefix": "G:",
   *     "bluePrefix": "B:",
   *     "hexPrefix": "#:",
   *     "alphaPrefix": "Alpha:",
   *     "ok": "OK",
   *     "cancel": "CANCEL",
   *     "customColor": "CUSTOM COLOR ..."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines what colors will be displayed in 'spectrumGrid', 'grid' and 'hexagonal' displayModes.
   * Default value: default
   */
  palette?: ColorPalette | string;
  /**
   * Defines an array of colors that form a custom palette. This palette can be used in displayModes 'grid' and 'spectrum grid' if the palette property is set to custom. The value of the property can be an array of strings or objects that contain valid colors ( HEX, RGBA, etc).
   * Default value: null
   */
  paletteColors?: any;
  /**
   * Defines an array of colors that represent a predefined list of custom colors. This palette can be used in displayModes 'grid', 'default' and 'spectrum grid'. Custom colors are displayed at the bottom of the color grid below the button for custom color selection. They are only visible if enableCustomColors property is true.
   * Default value: null
   */
  paletteCustomColors?: string[] | null;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines how the tooltip displays the value of the color that is being hovered.
   * Default value: hex
   */
  tooltipDisplayMode?: ColorTooltipDisplayMode | string;
  /**
   * Represents the value of the selected color.
   * Default value: "null"
   */
  value?: string;
  /**
   * Determines the format of the color. Whether it's in HEX, RGB or RGBA. By default it shows the color depending on the displayMode.
   * Default value: default
   */
  valueFormat?: ColorValueFormat | string;
  /**
   * Determines the value member for the color when the paletteColors consists of objects. Usefull in cases where the colors are loaded as objects in an array and the attribute that holds the color key is not named 'value'.
   * Default value: "null"
   */
  valueMember?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 ColorPanel is an advanced color chooser with Pallete, Spectrum Grid, Radial Palette and Excel-like options.
*/
export interface ColorPanel extends BaseElement, ColorPanelProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the color is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previously selected color.
   *  value - The new selected color.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the cancel button is clicked. 'Cancel' button is visible only when <strong>applyValueMode</strong> is set to <strong>useButtons</strong>.
	* @param event. The custom event.    */
  onCancelButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the custom color selection view is opened/closed. Custom color selection view is available when <strong>enableCustomColors</strong> property is true.
	* @param event. The custom event. Custom data event was created with: ev.detail(value)
   *  value - A boolean that indicates whether the custom color view is shown or not.
   */
  onCustomColorSelection?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the ok button is clicked. 'Ok' button is visible only when <strong>applyValueMode</strong> is set to <strong>useButtons</strong>.
	* @param event. The custom event.    */
  onOkButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-color-panel"): ColorPanel;
        querySelector(selectors: "smart-color-panel"): ColorPanel | null;
        querySelectorAll(selectors: "smart-color-panel"): NodeListOf<ColorPanel>;
        getElementsByTagName(qualifiedName: "smart-color-panel"): HTMLCollectionOf<ColorPanel>;
        getElementsByName(elementName: "smart-color-panel"): NodeListOf<ColorPanel>;
    }
}

/**Specifies how the value is applied. */
export declare type ColorApplyValueMode = 'instantly' | 'useButtons';
/**Determines the colors that will be displayed and their layout. */
export declare type ColorDisplayMode = 'default' | 'grid' | 'palette' | 'radial' | 'hexagonal' | 'spectrumGrid' | 'materialGrid';
/**Determines what colors will be displayed in 'spectrumGrid', 'grid' and 'hexagonal' displayModes. */
export declare type ColorPalette = 'default' | 'gray' | 'red' | 'green' | 'blue' | 'custom';
/**Determines how the tooltip displays the value of the color that is being hovered. */
export declare type ColorTooltipDisplayMode = 'none' | 'rgb' | 'rgba' | 'hex';
export interface ColorPickerProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * This property allows editting of colors via the input inside the element's action section. Accepts values in all supported types. This property works when 'valueDisplayMode' is set to default or colorCode.
   * Default value: false
   */
  editable?: boolean;
  /**
   * Determines the delay before the opened drop down closes when dropDownOpenMode is set to 'auto'.
   * Default value: 100
   */
  autoCloseDelay?: number;
  /**
   * Specifies how the user applies the selected value. In 'instantly' mode the value is applied immediately when color is selected. In 'useButtons' mode are shown 'Ok' and 'Cancel' buttons at the botom of the colorpicker's drop down. Only click on 'OK' button applies the value.
   * Default value: instantly
   */
  applyValueMode?: ColorApplyValueMode | string;
  /**
   * Defines the number of columns for the colors in displayModes 'grid', 'hexagonal' and 'spectrumGrid'.
   * Default value: 8
   */
  columnCount?: number;
  /**
   * Determines the colors that will be displayed and their layout.
   * Default value: default
   */
  displayMode?: ColorDisplayMode | string;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * By default clicking on color panel's preview container returns the color value to it's previous state. 'disableUndo' prevents this functionality.
   * Default value: false
   */
  disableUndo?: boolean;
  /**
   * Sets the parent container of the dropDown (the popup). The expected value is CSS Selector, ID or 'body'. Used when a CSS property of unknowned parent is interfering with the visibility of the dropDown. Example: 'body'.
   * Default value: "body"
   */
  dropDownAppendTo?: string;
  /**
   * Determines how the drop down is going to open.
   * Default value: default
   */
  dropDownOpenMode?: DropDownOpenMode | string;
  /**
   * Determines the position of the drop down button.
   * Default value: right
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Determines the vertical position of the dropDown. 'Auto' means its automatically determined depending on the viewport size.
   * Default value: auto
   */
  dropDownPosition?: DropDownPosition | string;
  /**
   * Sets the height of the drop down. Default value of null means that CSS variables are used. This property should be used when the browser doesn not support CSS variables.
   * Default value: "auto"
   */
  dropDownHeight?: string;
  /**
   * If this property is enabled, when the element's dropdown is opened, a transparent overlay is positioned between the dropdown and the rest of the document.
   * Default value: false
   */
  dropDownOverlay?: boolean;
  /**
   * Sets the width of the drop down. Default value of null means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: "auto"
   */
  dropDownWidth?: string;
  /**
   * Allows to edit the alpha(transparency) of the colors via an editor/slider in the following displayModes: 'palette', 'radial', 'hexagonal'
   * Default value: false
   */
  editAlphaChannel?: boolean;
  /**
   * Allows to select a custom color via an editor popup. Custom color selection is available in modes that don't have this option by default, like: 'grid', 'default, 'spectrum grid'.
   * Default value: false
   */
  enableCustomColors?: boolean;
  /**
   * Defines an Array of colors that will be used as the Theme Colors in the corresponding section of displayMode: 'grid' and 'default'.
   * Default value: null
   */
  gridThemeColors?: string[] | null;
  /**
   * Defines an Array of colors that will be used as the Shade Colors in the corresponding section of displayMode: 'grid' and 'default'.
   * Default value: null
   */
  gridShadeColors?: string[] | null;
  /**
   * Defines an Array of colors that will be used as the Standart Colors in the corresponding section of displayMode: 'grid' and 'default'.
   * Default value: null
   */
  gridStandardColors?: string[] | null;
  /**
   * Hides the alpha editor. Alpha editor is an input containing the value of the current color opacity. The input is available in the following modes: 'radial', 'palette', 'hexagonal'. The input is only visible if there's enough space. This editor is visible by default.
   * Default value: false
   */
  hideAlphaEditor?: boolean;
  /**
   * Determines which color editors will be hidden first when there's not enough space for all of them to be visible. By default the editors are only visible in 'palette', 'radial' and 'hexagonal' display modes. This property allows to prioritize the visibility of the editors.
   * Default value: RGB,HEX,alpha,previewContainer
   */
  hideContentToFit?: string[];
  /**
   * HEX editor is an input containing the hexadecimal representation of a color. This editor is visible by default. Setting 'hideRGBeditor' to true hides it.
   * Default value: false
   */
  hideHEXEditor?: boolean;
  /**
   * Hides the preview container. Preview container is used to show the currently selected value in 'palette', 'radial' and 'hexagonal' display modes.
   * Default value: false
   */
  hidePreviewContainer?: boolean;
  /**
   * Hides the RGB editor. This editor is a group of three separate inputs for the Red, Green and Blue values of the color.
   * Default value: false
   */
  hideRGBEditor?: boolean;
  /**
   * Sets additional helper text below the element that is only visible when the element is focused.
   * Default value: ""
   */
  hint?: string;
  /**
   * Inverts the colors in 'spectrumGrid', 'hexagonal', 'radial' modes.
   * Default value: false
   */
  inverted?: boolean;
  /**
   * Sets a label above the element.
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid ''!",
   *     "propertyInvalidValueType": "Invalid ''!",
   *     "methodInvalidValueType": "Invalid ''!",
   *     "methodInvalidArgumentsCount": "Invalid '' argument(s)!",
   *     "methodInvalidReturnType": "Invalid ''!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": "'.",
   *     "htmlTemplateNotSuported": ": Web Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": ".",
   *     "redPrefix": "R:",
   *     "greenPrefix": "G:",
   *     "bluePrefix": "B:",
   *     "hexPrefix": "#:",
   *     "alphaPrefix": "Alpha:",
   *     "ok": "OK",
   *     "cancel": "CANCEL",
   *     "customColor": "CUSTOM COLOR ..."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the popup is opened or closed
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines what colors will be displayed in 'spectrumGrid', 'grid' and 'hexagonal' displayModes.
   * Default value: default
   */
  palette?: ColorPalette | string;
  /**
   * Defines an array of colors that form a custom palette. This palette can be used in displayModes 'grid' and 'spectrum grid' if the palette property is set to custom. The value of the property can be an array of strings or objects that contain valid colors ( HEX, RGBA, etc).
   * Default value: null
   */
  paletteColors?: {name: string, value: string}[] | string[] | null;
  /**
   * Defines an array of colors that represent a predefined list of custom colors. This palette can be used in displayModes 'grid', 'default' and 'spectrum grid'. Custom colors are displayed at the bottom of the color grid below the button for custom color selection. They are only visible if enableCustomColor property is true.
   * Default value: null
   */
  paletteCustomColors?: string[] | null;
  /**
   * The placeholder is shown when the value is not set yet or is set to null.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Disables user interaction with the element.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Determines whether the resize indicator in the bottom right corner of the drop down is visible or not. This property is used in conjunction with resizeMode.
   * Default value: false
   */
  resizeIndicator?: boolean;
  /**
   * Determines whether the dropDown can be resized or not. When resizing is enabled, a resize bar appears on the top/bottom side of the drop down.
   * Default value: null
   */
  resizeMode?: ResizeMode | string;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines how the tooltip displays the value of the color that is being hovered.
   * Default value: hex
   */
  tooltipDisplayMode?: ColorTooltipDisplayMode | string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Represents the value of the selected color as the value of the element.
   * Default value: "null"
   */
  value?: string;
  /**
   * Determines the format of the color. Whether it's in HEX, RGB or RGBA. By default it shows the color depending on the displayMode.
   * Default value: default
   */
  valueFormat?: ColorValueFormat | string;
  /**
   * Determines which elements will be displayed in color picker's action section.
   * Default value: default
   */
  valueDisplayMode?: ColorValueDisplayMode | string;
}
/**
 ColorPicker is an advanced color picking component with Pallete, Spectrum Grid, Radial Palette and Excel-like options. Users can input colors either by a dropdown or input field.
*/
export interface ColorPicker extends BaseElement, ColorPickerProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when user clicks on the action button. 'Ok' button is visible only when <strong>applyValueMode</strong> is set to <strong>useButtons</strong>.
	* @param event. The custom event.    */
  onActionButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the cancel button is clicked. 'Cancel' button is visible only when <strong>applyValueMode</strong> is set to <strong>useButtons</strong>.
	* @param event. The custom event.    */
  onCancelButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the color value is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previously selected color.
   *  value - The new selected color.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the drop down is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the drop down is about to be closed. This event allows to cancel the closing operation calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the custom color selection view is opened/closed. Custom color selection view is available when <strong>enableCustomColors</strong> property is true.
	* @param event. The custom event. Custom data event was created with: ev.detail(value)
   *  value - A boolean that indicates whether the custom color view is shown or not.
   */
  onCustomColorSelection?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user clicks on the drop down button.
	* @param event. The custom event.    */
  onDropDownButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the ok button is clicked.
	* @param event. The custom event.    */
  onOkButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down is about to be opened. This event allows to cancel the opening operation calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user starts resizing the drop down.
	* @param event. The custom event. Custom data event was created with: ev.detail(position)
   *  position - An object containing the current left and top positions of the drop down.
   */
  onResizeStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the resizing of the drop down is finished.
	* @param event. The custom event. Custom data event was created with: ev.detail(position)
   *  position - An object containing the current left and top positions of the drop down.
   */
  onResizeEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Opens the drop down of the color picker.
   */
  open(): void;
  /**
   * Closes the drop down of the color picker.
   */
  close(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-color-picker"): ColorPicker;
        querySelector(selectors: "smart-color-picker"): ColorPicker | null;
        querySelectorAll(selectors: "smart-color-picker"): NodeListOf<ColorPicker>;
        getElementsByTagName(qualifiedName: "smart-color-picker"): HTMLCollectionOf<ColorPicker>;
        getElementsByName(elementName: "smart-color-picker"): NodeListOf<ColorPicker>;
    }
}

/**Determines how the drop down is going to open. */
export declare type DropDownOpenMode = 'none' | 'default' | 'dropDownButton' | 'auto';
/**Determines the vertical position of the dropDown. 'Auto' means its automatically determined depending on the viewport size. */
export declare type DropDownPosition = 'auto' | 'top' | 'bottom' | 'overlay-top' | 'overlay-center' | 'overlay-bottom' | 'center-bottom' | 'center-top';
/**Determines whether the dropDown can be resized or not. When resizing is enabled, a resize bar appears on the top/bottom side of the drop down. */
export declare type ResizeMode = 'none' | 'horizontal' | 'vertical' | 'both';
export interface ColumnPanelProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the data source that will be loaded to the column panel.
   * Default value: null
   */
  dataSource?: ColumnPanelDataSource[];
  /**
   * Enables or disables the column panel.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages.
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid ''!",
   *     "propertyInvalidValueType": "Invalid ''!",
   *     "methodInvalidValueType": "Invalid ''!",
   *     "methodInvalidArgumentsCount": "Invalid '' argument(s)!",
   *     "methodInvalidReturnType": "Invalid ''!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": "'.",
   *     "htmlTemplateNotSuported": ": Web Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "apply": "Apply",
   *     "cancel": "Cancel",
   *     "find": "Find a field",
   *     "noResults": "No results"
   *   }
   * }
   */
  messages?: any;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 Column Panel allows toggling the visibility and changing the order of columns.
*/
export interface ColumnPanel extends BaseElement, ColumnPanelProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the "Apply" button is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(value, positionChanged)
   *  value - The current configuration of columns (data source).
   *  positionChanged - A boolean detail that shows whether the columns have been reordered.
   */
  onApply?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the "Cancel" button is clicked.
	* @param event. The custom event.    */
  onCancel: ((this: any, ev: Event) => any) | null;
}

export interface ColumnPanelDataSource {
  /**
   * The column's data field.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Determines whether the column's visibility can be toggled.
   * Default value: false
   */
  disableToggle?: boolean;
  /**
   * A valid CSS class name applied to the column's icon.
   * Default value: ""
   */
  icon?: string;
  /**
   * The column's label.
   * Default value: ""
   */
  label?: string;
  /**
   * Determines whether the column is visible in the panel.
   * Default value: true
   */
  visible?: boolean;
}

declare global {
    interface Document {
        createElement(tagName: "smart-column-panel"): ColumnPanel;
        querySelector(selectors: "smart-column-panel"): ColumnPanel | null;
        querySelectorAll(selectors: "smart-column-panel"): NodeListOf<ColumnPanel>;
        getElementsByTagName(qualifiedName: "smart-column-panel"): HTMLCollectionOf<ColumnPanel>;
        getElementsByName(elementName: "smart-column-panel"): NodeListOf<ColumnPanel>;
    }
}

export interface ComboBoxProperties {
  /**
   * Used only when dropDownOpenMode is set to 'auto'. Determines the delay before the opened drop down closes if the pointer is not over the element.
   * Default value: 100
   */
  autoCloseDelay?: number;
  /**
   * Determines the autocomplete mode. Auto complete modes filter the items from the dataSource and show only those that match the input.
   * Default value: none
   */
  autoComplete?: AutoComplete | string;
  /**
   * Determines the delay before the drop down opens to show the matches from the auto complete operation. The delay is measured in miliseconds.
   * Default value: 100
   */
  autoCompleteDelay?: number;
  /**
   * Allows the user to define a custom key name ( or multiple key names) to open that popup with.
   * Default value: 
   */
  autoOpenShortcutKey?: string[];
  /**
   * Determines the data source that will be loaded to the ComboBox. The dataSource can be an array of strings/numbers or objects where the attributes represent the properties of a List Item. For example label, value, group. It can also be a callback that returns an Array of items as previously described.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the combo box.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines whether an indicator will appear during filtering and remote item loading.
   * Default value: false
   */
  displayLoadingIndicator?: boolean;
  /**
   * Sets or gets the displayMember. The displayMember specifies the name of a property to display. The name is contained in the collection specified by the 'dataSource' property.
   * Default value: """"
   */
  displayMember?: string;
  /**
   * Determines the drop down parent. The expected value is CSS Selector, ID or 'body'. The drop down can be removed from the body of the element and continue to work in another container. This is usefull when one of the parents of the element doesn't allow overflowing, by settings this property to 'body' the drop down will be appended to the DOM and the popup will open/close as usual. dropDownAppendTo can be a string representing the id of an HTML element on the page or a direct reference to that element. Reseting it back to null will take the drop down back to it's original place.
   * Default value: "null"
   */
  dropDownAppendTo?: string;
  /**
   * Determines the position of the drop down button.
   * Default value: right
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to an empty string. In this case the height of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the maximum Height of the drop down. By default it's set to an empty string. In this case the maxHeight of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownMaxHeight?: string | number;
  /**
   * Sets the maximum Width of the drop down. By default it's set to an empty string. In this case the maxWidth of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownMaxWidth?: string | number;
  /**
   * Sets the minimum Height of the drop down. By default it's set to an empty string. In this case the minHeight of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownMinHeight?: string | number;
  /**
   * Sets the minimum Width of the drop down. By default it's set to an empty string. In this case the minWidth of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownMinWidth?: string | number;
  /**
   * Determines how the drop down is going to open.
   * Default value: default
   */
  dropDownOpenMode?: DropDownOpenMode | string;
  /**
   * If this property is enabled, when the element's dropdown is opened, a transparent overlay is positioned between the dropdown and the rest of the document. The purpose of the overlay is to make sure that clicking anywhere outside the popup will will target the overlay and not the DOM.
   * Default value: false
   */
  dropDownOverlay?: boolean;
  /**
   * Determines the placeholder for the drop down, displayed when there are no items in it.
   * Default value: "No Items"
   */
  dropDownPlaceholder?: string;
  /**
   * Determines the position of the drop down when opened.
   * Default value: auto
   */
  dropDownPosition?: DropDownPosition | string;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Determines the behavior of the element when Escape key is pressed.
   * Default value: null
   */
  escKeyMode?: ComboBoxEscKeyMode | string;
  /**
   * Determines whether filtering is enabled.
   * Default value: false
   */
  filterable?: boolean;
  /**
   * Determines the placeholder for the drop down list filter input field.
   * Default value: ""
   */
  filterInputPlaceholder?: string;
  /**
   * Determines the filtering mode of the Combo box.
   * Default value: startsWithIgnoreCase
   */
  filterMode?: FilterMode | string;
  /**
   * If enabled, the items will be grouped by their first letter. Can't be applied if the dataSource already contains groups.
   * Default value: false
   */
  grouped?: boolean;
  /**
   * Determines which attribute from the dataSource object will be used as the group member for the items. If not set, by default 'group' property is used from the source object. groupMember is especially usefull when loading the data from a JSON file as a dataSource for the ListBox and there's a specific property that should be used to group the items.
   * Default value: ""
   */
  groupMember?: string;
  /**
   * Sets additional helper text below the element. The hint is visible only when the element is focused.
   * Default value: ""
   */
  hint?: string;
  /**
   * Determines the visibility of the horizontal Scroll bar inside the drop down.
   * Default value: auto
   */
  horizontalScrollBarVisibility?: HorizontalScrollBarVisibility | string;
  /**
   * Represents the property name of a List item. Determines the value of the input when a ListItem is selected. Usefull in cases where the user wants to display for example the value of an item instead of it's label. By default the label is displayed in the input.
   * Default value: ""
   */
  inputMember?: string;
  /**
   * Sets the purpose of the input and what, if any, permission the user agent has to provide automated assistance in filling out the element's input when in a form, as well as guidance to the browser as to the type of information expected in the element. This value corresponds to the standard HTML autocomplete attribute and can be set to values such as 'on', 'name', 'organization', 'street-address', etc.
   * Default value: "off"
   */
  inputPurpose?: string;
  /**
   * IncrementalSearchDelay property specifies the time-interval in milliseconds until the previous search query is cleared. The timer starts when the user stops typing. A new query can be started only when the delay has passed.
   * Default value: 700
   */
  incrementalSearchDelay?: number;
  /**
   * Sets ot gets the mode of the incremental search mode. Incremental search is enabled by default. Typing while the drop down is focused starts the incremental search.
   * Default value: startsWithIgnoreCase
   */
  incrementalSearchMode?: SearchMode | string;
  /**
   * Sets the height for all list items. Used only when virtualization is enabled.
   * Default value: null
   */
  itemHeight?: number;
  /**
   * Determines the item width measuring algorithm.
   * Default value: auto
   */
  itemMeasureMode?: ListItemMeasureMode | string;
  /**
   * A getter that returns an array of all List items inside the drop down.
   * Default value: 
   */
  items?: {label: string, value: string}[];
  /**
   * The itemTemplate property is a string that represents the id of an HTMLTemplateElement in the DOM.  It's used to set a customize the content of the list items.
   * Default value: null
   */
  itemTemplate?: any;
  /**
   * Sets a little text label above the element.
   * Default value: ""
   */
  label?: string;
  /**
   * Determines the text that will be displayed next to the loading indicator when the loader is visible and it's position is top or bottom.
   * Default value: "Loading..."
   */
  loadingIndicatorPlaceholder?: string;
  /**
   * Determines the position of the loading indicator.
   * Default value: center
   */
  loadingIndicatorPosition?: VerticalAlignment | string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the minimum number of characters inside the input in order to trigger the autocomplete functionality
   * Default value: 2
   */
  minLength?: number;
  /**
   * Determines the maximum number of characters inside the input.
   * Default value: -1
   */
  maxLength?: number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the popup is opened or closed
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the input's placeholder.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Disables user interaction with the element.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines whether the resize indicator in the bottom right corner of the drop down is visible or not. This property is used in conjunction with resizeMode.
   * Default value: false
   */
  resizeIndicator?: boolean;
  /**
   * Determines whether the dropDown can be resized or not. When resizing is enabled, a resize bar appears on the top/bottom side of the drop down.
   * Default value: null
   */
  resizeMode?: ResizeMode | string;
  /**
   * Determines what will be displayed in the input.
   * Default value: plain
   */
  selectionDisplayMode?: SelectionDisplayMode | string;
  /**
   * Sets or gets the selected indexes. Selected indexes represents an array of the indexes of the items that should be selected.
   * Default value: 
   */
  selectedIndexes?: number[];
  /**
   * Sets or gets elected indexes. Selected values represents the values of the items that should be selected.
   * Default value: 
   */
  selectedValues?: string[];
  /**
   * Determines how many items can be selected.
   * Default value: zeroAndOne
   */
  selectionMode?: ListSelectionMode | string;
  /**
   * Determines whether the items are sorted alphabetically or not
   * Default value: false
   */
  sorted?: boolean;
  /**
   * Determines sorting direction - ascending(asc) or descending(desc)
   * Default value: "asc"
   */
  sortDirection?: string;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets a custom content for the tokens when selectionDisplayMode is set to 'tokens'. Tokens are used only for multiple selection.
   * Default value: null
   */
  tokenTemplate?: any;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value.
   * Default value: ""
   */
  value?: string;
  /**
   * Determines the value member of an item. Stored as value in the item object. Similar to groupMember, valueMember is especially usefull when using data from a JSON file as a dataSource for the ListBox and there's a specific property that should be used for the value the items.
   * Default value: """"
   */
  valueMember?: string;
  /**
   * Determines the visibility of the vertical scroll bar.
   * Default value: auto
   */
  verticalScrollBarVisibility?: VerticalScrollBarVisibility | string;
  /**
   * Determines weather or not Virtualization is used for the Combo box. Virtualization allows a huge amount of items to be loaded to the List box while preserving the performance. For example a milion items can be loaded to the list box.
   * Default value: false
   */
  virtualized?: boolean;
}
/**
 ComboBox is the alternate for the HTML select tag with editable option. It supports data binding, auto-complete, filtering, grouping, cascading and more.
*/
export interface ComboBox extends BaseElement, ComboBoxProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(addedItems, disabled, index, label, removedItems, selected, value)
   *  addedItems - An array of List items that have been selected.
   *  disabled - A flag indicating whether or not the item that caused the change event is disabled.
   *  index - The index of the List item that triggered the event.
   *  label - The label of the List item that triggered the event.
   *  removedItems - An array of List items that have been unselected before the event was fired.
   *  selected - The selected state of the List item that triggered the event. If an item was selected the value will be true and vice versa.
   *  value - The value of the List item that triggered the event.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the drop down list is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the drop down list is about to be closed. This event allows to cancel the closing operation calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an item is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(disabled, index, label, selected, value)
   *  disabled - Indicates whether the List item that was clicked is disabled or not.
   *  index - Indicates the index of the List item that was clicked.
   *  label - The label of the List item that was clicked.
   *  selected - Indicates whether the List item that was clicked is selected or not.
   *  value - The value of the List item that was clicked.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down list is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down list is about to be opened. This event allows to cancel the opening operation calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user starts resizing the drop down.
	* @param event. The custom event. Custom data event was created with: ev.detail(position)
   *  position - An object containing the current left and top positions of the drop down.
   */
  onResizeStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the resizing of the drop down is finished.
	* @param event. The custom event. Custom data event was created with: ev.detail(position)
   *  position - An object containing the current left and top positions of the drop down.
   */
  onResizeEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user scrolls to the end of the dropDown list.
	* @param event. The custom event.    */
  onScrollBottomReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user scrolls to the start of the dropDown list.
	* @param event. The custom event.    */
  onScrollTopReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a token item(pill) has been clicked. This event allows to cancel the opening operation calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onTokenClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Appends a ListItem to the end of the list of items inside element.
   * @param {Node} node. A ListItem element that should be added to the rest of the items as the last item.
   * @returns {Node}
   */
  appendChild<T extends Node>(node: Node): T;
  /**
   * Removes all items from the drop down list.
   */
  clearItems(): void;
  /**
   * Unselects all items.
   */
  clearSelection(): void;
  /**
   * Closes the dropDown list.
   */
  close(): void;
  /**
   * Ensures the desired item is visible by scrolling to it.
   * @param {HTMLElement | string} item. A list item or value of the desired item to be visible.
   */
  ensureVisible(item: HTMLElement | string): void;
  /**
   * Returns an item instance from the dropDown list.
   * @param {string} value. The value of an item from the drop down list.
   * @returns {HTMLElement}
   */
  getItem(value: string): HTMLElement;
  /**
   * Inserts a new item at a specified position.
   * @param {number} position. The position where the item must be inserted.
   * @param {any} item. Describes the properties of the item that will be inserted.
   */
  insert(position: number, item: any): void;
  /**
   * Inserts a new ListItem before another in the list.
   * @param {Node} node. A ListItem element that should be added before the referenceItem in the list.
   * @param {Node | null} referenceNode. A ListItem element that acts as the reference item before which a new item is about to be inserted. The referenceNode must be in the same list as the node.
   * @returns {Node}
   */
  insertBefore<T extends Node>(node: Node, referenceNode: Node | null): T;
  /**
   * Opens the dropDown list.
   */
  open(): void;
  /**
   * Removes an item at a specified position.
   * @param {number} position. The position of the removed item.
   */
  removeAt(position: number): void;
  /**
   * Removes a ListItem from the list of items inside element.
   * @param {Node} node. A ListItem element that is part of the list of items inside the element.
   * @returns {Node}
   */
  removeChild<T extends Node>(node: Node): T;
  /**
   * Selects an item from the dropDown list. 
   * @param {string | HTMLElement} item. A string, representing the value of the item or an HTML Element referencing an item from the list
   */
  select(item: string | HTMLElement): void;
  /**
   * Unselects an item from the dropDown list. 
   * @param {string | HTMLElement} item. A string, representing the value of the item or an HTML Element referencing an item from the list
   */
  unselect(item: string | HTMLElement): void;
  /**
   * Updates an item from the dropDown list.
   * @param {number} position. The position where the item must be updated.
   * @param {any} value. The value of the updated item.
   */
  update(position: number, value: any): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-combo-box"): ComboBox;
        querySelector(selectors: "smart-combo-box"): ComboBox | null;
        querySelectorAll(selectors: "smart-combo-box"): NodeListOf<ComboBox>;
        getElementsByTagName(qualifiedName: "smart-combo-box"): HTMLCollectionOf<ComboBox>;
        getElementsByName(elementName: "smart-combo-box"): NodeListOf<ComboBox>;
    }
}

/**Determines the autocomplete mode. Auto complete modes filter the items from the dataSource and show only those that match the input. */
export declare type AutoComplete = 'none' | 'auto' | 'inline' | 'manual';
/**Determines the behavior of the element when Escape key is pressed. */
export declare type ComboBoxEscKeyMode = 'none' | 'previousValue' | 'clearValue';
/**Determines the filtering mode of the Combo box. */
export declare type FilterMode = 'contains' | 'containsIgnoreCase' | 'custom' | 'doesNotContain' | 'doesNotContainIgnoreCase' | 'equals' | 'equalsIgnoreCase' | 'startsWith' | 'startsWithIgnoreCase' | 'endsWith' | 'endsWithIgnoreCase';
/**Determines the visibility of the horizontal Scroll bar inside the drop down. */
export declare type HorizontalScrollBarVisibility = 'auto' | 'disabled' | 'hidden' | 'visible';
/**Sets ot gets the mode of the incremental search mode. Incremental search is enabled by default. Typing while the drop down is focused starts the incremental search. */
export declare type SearchMode = 'contains' | 'containsIgnoreCase' | 'doesNotContain' | 'doesNotContainIgnoreCase' | 'equals' | 'equalsIgnoreCase' | 'startsWith' | 'startsWithIgnoreCase' | 'endsWith' | 'endsWithIgnoreCase';
/**Determines the item width measuring algorithm. */
export declare type ListItemMeasureMode = 'auto' | 'precise';
/**Determines what will be displayed in the input. */
export declare type SelectionDisplayMode = 'plain' | 'placeholder' | 'tokens';
/**Determines how many items can be selected. */
export declare type ListSelectionMode = 'none' | 'oneOrManyExtended' | 'zeroOrMany' | 'oneOrMany' | 'zeroAndOne' | 'zeroOrOne' | 'one' | 'checkBox' | 'radioButton';
/**Determines the visibility of the vertical scroll bar. */
export declare type VerticalScrollBarVisibility = 'auto' | 'disabled' | 'hidden' | 'visible';
export interface CountryInputProperties {
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets additional class names to the Input drop down.
   * Default value: 
   */
  dropDownClassList?: any;
  /**
   * Determines the position of the drop down button.
   * Default value: none
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to an empty string. In this case the height of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the drop down is opened or not.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Sets or gets an array of country codes which will be used instead of the default one with all countries. The country code should be ISO 3166-1 alpha-2 codes(https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2).
   * Default value: []
   */
  onlyCountries?: any;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Sets or gets the selected country of the element. The country code should be ISO 3166-1 alpha-2 codes(https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2).
   * Default value: ""
   */
  selectedCountry?: string;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element.
   * Default value: ""
   */
  value?: string;
}
/**
 The Country Input specifies an input field where the user can select a country.
*/
export interface CountryInput extends BaseElement, CountryInputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, oldLabel, oldValue, value)
   *  label - The label of the new selected item.
   *  oldLabel - The label of the item that was previously selected before the event was triggered.
   *  oldValue - The value of the item that was previously selected before the event was triggered.
   *  value - The value of the new selected item.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered on each key up event of the Input, if the value is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value before it was changed.
   *  value - The new value.
   */
  onChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user clicks on an item from the popup list.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value)
   *  item - The item that was clicked.
   *  label - The label of the item that was clicked.
   *  value - The value of the item that was clicked.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Closes the drop down.
   */
  close(): void;
  /**
   * Ensures that the active ( selected ) item is always visible.
   */
  ensureVisible(): void;
  /**
   * Opens the drop down.
   */
  open(): void;
  /**
   * Selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-country-input"): CountryInput;
        querySelector(selectors: "smart-country-input"): CountryInput | null;
        querySelectorAll(selectors: "smart-country-input"): NodeListOf<CountryInput>;
        getElementsByTagName(qualifiedName: "smart-country-input"): HTMLCollectionOf<CountryInput>;
        getElementsByName(elementName: "smart-country-input"): NodeListOf<CountryInput>;
    }
}

export interface CustomizationDialogProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Array with filtered fields and their settings.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Sets or gets the displayMember. The displayMember specifies the name of an object property to display. The name is contained in the collection specified by the 'dataSource' property.
   * Default value: ""
   */
  displayMember?: string;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets whether is enabled/disabled filtering tab.
   * Default value: false
   */
  filtering?: boolean;
  /**
   * Sets whether is enabled/disabled grouping tab.
   * Default value: false
   */
  grouping?: boolean;
  /**
   * Set's the buttons that will be visible in the header section.
   * Default value: apply,close
   */
  headerButtons?: string[];
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback, related to localization module. 
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "ambiguousIndexes": "jqx-tabs: Initially set jqx-tab-item indexes are ambiguous and are ignored in favour of the HTML structure.",
   *     "detailsObjectRequired": "jqx-tabs: The method \"insert\" requires a details Object to be passed as a second argument.",
   *     "invalidIndex": "jqx-tabs: '' method accepts an index of type number.",
   *     "referenceNodeNotChild": "jqx-tabs: Passed  is not part of this jqx-tabs element.",
   *     "tabItemRequired": "jqx-tabs: The method '' requires a \"jqx-tab-item\" element to be passed as an argument.",
   *     "sorting": "SORTING",
   *     "grouping": "GROUPING",
   *     "filtering": "FILTERING",
   *     "columnChooser": "COLUMN CHOOSER",
   *     "applyButton": "Apply",
   *     "closeButton": "Close",
   *     "columnsToSort": "Columns to Sort",
   *     "columnsToGroup": "Columns to Group",
   *     "placeholderSorting": "No sorting applied",
   *     "placeholderGrouping": "No grouping applied",
   *     "and": "And",
   *     "notand": "Not And",
   *     "or": "Or",
   *     "notor": "Not Or",
   *     "=": "Equals",
   *     "<>": "Does not equal",
   *     ">": "Greater than",
   *     ">=": "Greater than or equal to",
   *     "<": "Less than",
   *     "<=": "Less than or equal to",
   *     "startswith": "Starts with",
   *     "endswith": "Ends with",
   *     "contains": "Contains",
   *     "notcontains": "Does not contain",
   *     "isblank": "Is blank",
   *     "isnotblank": "Is not blank"
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets whether the columns reordering is active.
   * Default value: false
   */
  reorder?: boolean;
  /**
   * Sets or gets the tab, wich is selected on initialization.
   * Default value: 0
   */
  selectedTab?: number;
  /**
   * Sets whether is enabled/disabled sorting tab.
   * Default value: false
   */
  sorting?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets element's value. It's represente by object, contained
   * Default value: null
   */
  value?: any;
  /**
   * Determines the value member of an item. Stored as value in the item object.
   * Default value: ""
   */
  valueMember?: string;
  /**
   * Sets or gets whether the columns of the element could be hidden.
   * Default value: false
   */
  visibility?: boolean;
}
/**
 Defines a dialog for customization of filtering, sorting.
*/
export interface CustomizationDialog extends BaseElement, CustomizationDialogProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the dialog is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the dialog is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the data in the value property is changed.
	* @param event. The custom event.    */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the dialog is closed via clicking the apply button.
	* @param event. The custom event.    */
  onApply?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Opens the dialog
   */
  open(): void;
  /**
   * Closes the dialog.
   */
  close(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-customization-dialog"): CustomizationDialog;
        querySelector(selectors: "smart-customization-dialog"): CustomizationDialog | null;
        querySelectorAll(selectors: "smart-customization-dialog"): NodeListOf<CustomizationDialog>;
        getElementsByTagName(qualifiedName: "smart-customization-dialog"): HTMLCollectionOf<CustomizationDialog>;
        getElementsByName(elementName: "smart-customization-dialog"): NodeListOf<CustomizationDialog>;
    }
}

export interface DataAdapter {
  /**
   * Sets or gets whether the binding is automatic after creating a data adapter instance.
   * Default value: true
   */
  autoBind?: boolean;
  /**
   * Sets or gets whether the binding is asynchronous.
   * Default value: false
   */
  async?: boolean;
  /**
   * Gets the bound data source after the binding is completed.
   * Default value: 
   */
  boundSource?: any[];
  /**
   * Gets the bound hierarchical data source after the binding is completed.
   * Default value: 
   */
  boundHierarchy?: any[];
  /**
   * Sets or gets a children data field like 'children', 'items' in the data source. When this property is set, the dataAdapter will look for this data field when looping through the items. If it is found a hierarchical data source would be created.
   * Default value: ""
   */
  childrenDataField?: string;
  /**
   * Sets or gets the data fields to group by.
   * Default value: []
   */
  groupBy?: string[];
  /**
   * Sets or gets the data fields which decribe the loaded data and data type. Ex: ['id: number', 'firstName: string', 'lastName: string']
   * Default value: string[]
   */
  dataFields?: string[];
  /**
   * Sets or gets the data source of the adapter.
   * Default value: []
   */
  dataSource?: any;
  /**
   * Sets or gets whether the data source type.
   * Default value: array
   */
  dataSourceType?: DataAdapterDataSourceType | string;
  /**
   * Sets or gets the dataAdapter's id
   * Default value: ""
   */
  id?: string;
  /**
   * Sets or gets the key data field to be used for building the hierarchy. It is used in combination with the parentDataField property. Usually the 'id' field is used as key data field and 'parentId' as parent data field'
   * Default value: ""
   */
  keyDataField?: string;
  /**
   * Gets the data source length.
   * Default value: 0
   */
  length?: number;
  /**
   * Sets or gets whether the request type.
   * Default value: GET
   */
  method?: DataAdapterMethod | string;
  /**
   * Sets or gets the parent data field to be used for building the hierarchy. It is used in combination with the keyDataField property. Usually the 'id' field is used as key data field and 'parentId' as parent data field'
   * Default value: ""
   */
  parentDataField?: string;
  /**
   * Sets or gets the virtual data source on expand function. This function is called when we load data on demand in Tree or TreeGrid and virtualDataSource in these components is set, too
   * Default value: null
   */
  virtualDataSourceOnExpand?: DataAdapterVirtualDataSourceOnExpand[];
  /**
   * Sets or gets the total length of items. If it is unknown, do not set this parameter.
   * Default value: 0
   */
  virtualDataSourceLength?: number;
  /**
   * Sets or gets a boolean param which determines whether to cache or not the new loaded data.
   * Default value: true
   */
  virtualDataSourceCache?: boolean;
  /**
   * Sets or gets the virtual data source function
   * Default value: null
   */
  virtualDataSource?: DataAdapterVirtualDataSource[];
  /**
   * Adds a new item.
   * @param {any} item. A new data source item.
   * @param {string | number} parentId?. id of a parent data item, when we want to add a child in a hierarchy
   */
  add(item: any, parentId?: string | number): void;
  /**
   * Clears the adapter.
   */
  clear(): void;
  /**
   * Clears the sorting.
   */
  clearSort(): void;
  /**
   * Clears the filters.
   */
  clearFilter(): void;
  /**
   * Clears the grouping.
   */
  clearGroup(): void;
  /**
   * Data binds the adapter to the data source. If the data source is remote, perform a new AJAX call.
   */
  dataBind(): void;
  /**
   * Notifies for data changes like adding, removing, updating items.
   * @param {any} callback. The callback function which is called when a change happened. { action: 'add' | 'bindingComplete' | 'removeLast' | 'remove' | 'move' | 'update' | 'insert', data: [], index?: number }
   */
  notify(callback: any): void;
  /**
   * The find method returns the value of the first element in the provided array that satisfies the provided testing function.
   * @param {any} callback. Function to execute on each value in the array, taking 2 arguments - element and index. The element is the current item in the array. The index is the index of the current element.
   * @returns {any}
   */
  find(callback: any): any;
  /**
   * Filters the data source.
   * @param {string[]} dataFields. The data fields.
   * @param {any} filterGroups. The filter group objects.
   */
  filter(dataFields: string[], filterGroups: any): void;
  /**
   * Filters the data source.
   * @param {string} dataField. The data field.
   * @param {string[]} filterExpessions. Filter expression like '= 5'. Allowed operators '=', '<','>','<>', '<=', '>=', 'starts with','contains','ends with', '', 'null'
   */
  filterBy(dataField: string, filterExpessions: string[]): void;
  /**
   * Gets the index of an item.
   * @param {number} index. The index of the item.
   * @returns {any}
   */
  indexOf(index: number): any;
  /**
   * Inserts an item.
   * @param {number} index. The index where to insert the item.
   * @param {any} item. An object containing the values for the properties of the item that will be updated.
   */
  insert(index: number, item: any): void;
  /**
   * Moves an item.
   * @param {number} from. The item's old index.
   * @param {number} to. The item's new index.
   */
  move(from: number, to: number): void;
  /**
   * Removes the last item from the data source.
   */
  removeLast(): void;
  /**
   * Removes an item at a specified index.
   * @param {number} position. The index of the item to be removed.
   */
  removeAt(position: number): void;
  /**
   * Sorts the data source.
   * @param {string[]} dataFields. The data fields.
   * @param {string[]} sortOrders. The sort orders array with 'asc' or 'desc' or null values.
   */
  sort(dataFields: string[], sortOrders: string[]): void;
  /**
   * Sorts the data source.
   * @param {string} dataField. The data field.
   * @param {string} dataType. The type of the data field - string, number, date, boolean.
   * @param {string} sortOrder. The sort order - 'asc' or 'desc' or null.
   */
  sortBy(dataField: string, dataType: string, sortOrder: string): void;
  /**
   * Aggregates data from the data source.
   * @param {any[]} summaryItems. The summary items. Each summary item represents a data field with summary function: 'avg' - Average aggregate, 'count' - Count aggregate, 'min' - Min aggregate, 'max' - Max aggregate, 'sum' - Sum aggregate, 'product' - Product aggregate, 'stdev' - Standard deviation on a sample., 'stdevp' - Standard deviation on an entire population., 'varp' - Variance on an entire population., 'var' - Variance on a sample.
   */
  summarize(summaryItems: any[]): void;
  /**
   * Returns an array of all items.
   * @returns {any[]}
   */
  toArray(): any[];
  /**
   * Updates an item from the element.
   * @param {number} index. The index of the item to be updated.
   * @param {any} item. An object containing the values for the properties of the item that will be updated.
   */
  update(index: number, item: any): void;
}

export interface DataAdapterVirtualDataSourceOnExpand {
  /**
   * Result callback function which is executed when the data loading is completed.
   * Default value: null
   */
  resultCallbackFunction?: DataAdapterVirtualDataSourceOnExpandResultCallbackFunction;
  /**
   * Details of the object, which made the request
   * Default value: null
   */
  details?: DataAdapterVirtualDataSourceOnExpandDetails;
}

/**Result callback function which is executed when the data loading is completed. */
export interface DataAdapterVirtualDataSourceOnExpandResultCallbackFunction {
(
  /**
   * result callback function settings
   * Default value: null
   */
  settings?: any

): void;
}

/**Details of the object, which made the request */
export interface DataAdapterVirtualDataSourceOnExpandDetails {
  /**
   * Data start index.
   * Default value: 0
   */
  first?: number;
  /**
   * Data end index.
   * Default value: 0
   */
  last?: number;
  /**
   * Sorting information.
   * Default value: null
   */
  sorting?: DataAdapterVirtualDataSourceOnExpandDetailsSorting;
  /**
   * Filtering information.
   * Default value: null
   */
  filtering?: any;
  /**
   * Grouping information
   * Default value: []
   */
  grouping?: string[];
  /**
   * Request action type
   * Default value: undefined
   */
  action?: DataAdapterVirtualDataSourceOnExpandDetailsAction | string;
}

/**Sorting information. */
export interface DataAdapterVirtualDataSourceOnExpandDetailsSorting {
  /**
   * Sort order.
   * Default value: asc
   */
  sortOrder?: DataAdapterVirtualDataSourceOnExpandDetailsSortingSortOrder | string;
  /**
   * Sort index.
   * Default value: -1
   */
  sortIndex?: number;
}

export interface DataAdapterVirtualDataSource {
  /**
   * Result callback function which is executed when the data loading is completed.
   * Default value: null
   */
  resultCallbackFunction?: DataAdapterVirtualDataSourceResultCallbackFunction;
  /**
   * Details of the object, which made the request
   * Default value: null
   */
  details?: DataAdapterVirtualDataSourceDetails;
}

/**Result callback function which is executed when the data loading is completed. */
export interface DataAdapterVirtualDataSourceResultCallbackFunction {
(
  /**
   * result callback function settings
   * Default value: null
   */
  settings?: any

): void;
}

/**Details of the object, which made the request */
export interface DataAdapterVirtualDataSourceDetails {
  /**
   * Data start index.
   * Default value: 0
   */
  first?: number;
  /**
   * Data end index.
   * Default value: 0
   */
  last?: number;
  /**
   * Sorting information.
   * Default value: null
   */
  sorting?: any;
  /**
   * Filtering information.
   * Default value: null
   */
  filtering?: any;
  /**
   * Grouping information
   * Default value: []
   */
  grouping?: string[];
  /**
   * Request action type
   * Default value: undefined
   */
  action?: DataAdapterVirtualDataSourceDetailsAction | string;
  /**
   * Expanded Row when data is loaded on demand in Tree Hierarchy scenarios.
   * Default value: undefined
   */
  row?: any;
}
/**Sets or gets whether the data source type. */
export declare type DataAdapterDataSourceType = 'array' | 'json' | 'xml' | 'csv' | 'tsv';
/**Sets or gets whether the request type. */
export declare type DataAdapterMethod = 'GET' | 'POST';
/**Sort order. */
export declare type DataAdapterVirtualDataSourceOnExpandDetailsSortingSortOrder = 'asc' | 'desc';
/**Request action type */
export declare type DataAdapterVirtualDataSourceOnExpandDetailsAction = 'sort' | 'filter' | 'dataBind' | 'scroll' | 'group' | 'expand' | 'pageIndexChange' | 'pageSizeChange';
/**Request action type */
export declare type DataAdapterVirtualDataSourceDetailsAction = 'sort' | 'filter' | 'dataBind' | 'scroll' | 'group' | 'expand' | 'pageIndexChange' | 'pageSizeChange';
export interface DateInputProperties {
  /**
   * Determines whether the calendar button pop-up will be closed automatically when date or time is selected through it.
   * Default value: false
   */
  autoClose?: boolean;
  /**
   * Determines the delay before the calendar pop-up is automatically closed. Applicable only when autoClose is set to true.
   * Default value: 500
   */
  autoCloseDelay?: number;
  /**
   * Determines the format of the dates displayed in the input. Accepts valid ECMAScript Internationalization API format. Intl.DateTimeFormat is used to format date strings in JavaScript. By default the date format is 'numeric'. The default value is: { day: 'numeric', month: 'numeric', year: 'numeric' }
   * Default value: { day: 'numeric', month: 'numeric', year: 'numeric' }
   */
  dateTimeFormat?: any;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines the position of the drop down button.
   * Default value: right
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to 'auto'.
   * Default value: auto
   */
  dropDownHeight?: string | number;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Sets the format string. When this property is set, the dateTimeFormat property will be disabled and the formatting will use the value of the formatString. Built-in Date formats:// short date pattern'd' - 'M/d/yyyy',// long date pattern'D' - 'dddd, MMMM dd, yyyy',// short time pattern't' - 'h:mm tt',// long time pattern'T' - 'h:mm:ss tt',// long date, short time pattern'f' - 'dddd, MMMM dd, yyyy h:mm tt',// long date, long time pattern'F' - 'dddd, MMMM dd, yyyy h:mm:ss tt',// month/day pattern'M' - 'MMMM dd',// month/year pattern'Y' - 'yyyy MMMM',// S is a sortable format that does not vary by culture'S' - 'yyyy'-'MM'-'dd'T'HH':'mm':'ss'Date format strings:'d'-the day of the month;'dd'-the day of the month'ddd'-the abbreviated name of the day of the week'dddd'- the full name of the day of the week'h'-the hour, using a 12-hour clock from 1 to 12'hh'-the hour, using a 12-hour clock from 01 to 12'H'-the hour, using a 24-hour clock from 0 to 23'HH'- the hour, using a 24-hour clock from 00 to 23'm'-the minute, from 0 through 59'mm'-the minutes,from 00 though59'M'- the month, from 1 through 12'MM'- the month, from 01 through 12'MMM'-the abbreviated name of the month'MMMM'-the full name of the month's'-the second, from 0 through 59'ss'-the second, from 00 through 59't'- the first character of the AM/PM designator'tt'-the AM/PM designator'y'- the year, from 0 to 99'yy'- the year, from 00 to 99'yyy'-the year, with a minimum of three digits'yyyy'-the year as a four-digit number;'yyyyy'-the year as a four-digit number.
   * Default value: ""
   */
  formatString?: string;
  /**
   * Sets the purpose of the input and what, if any, permission the user agent has to provide automated assistance in filling out the element's input when in a form, as well as guidance to the browser as to the type of information expected in the element. This value corresponds to the standard HTML autocomplete attribute and can be set to values such as 'on', 'name', 'organization', 'street-address', etc.
   * Default value: "off"
   */
  inputPurpose?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Determines the max date for the Calendar displayed inside the popup.
   * Default value: new Date(2100, 1, 1)
   */
  max?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the min date for the Calendar displayed inside the popup.
   * Default value: new Date(1900, 1, 1)
   */
  min?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the drop down is opened or not.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Determines whether ot not the user can enter text inside the input. if dropDownButtonPosition is set to 'left' or 'right' then readonly determines whether the element acts as a ComboBox or a DropDownList if a dataSource is provided.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element. Expected value is: Date string, Date object or null.
   * Default value: 
   */
  value?: any;
}
/**
 DateInput specifies an input field where the user can enter a date. It also has a popup with a Calendar that allows to pick a date.
*/
export interface DateInput extends BaseElement, DateInputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, oldLabel, oldValue, value)
   *  label - The label of the new selected item.
   *  oldLabel - The label of the item that was previously selected before the event was triggered.
   *  oldValue - The value of the item that was previously selected before the event was triggered.
   *  value - The value of the new selected item.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Closes the drop down.
   */
  close(): void;
  /**
   * Opens the drop down.
   */
  open(): void;
  /**
   * Selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
  /**
   * Returns the value in the desired format.
   * @param {string | Date} value. The value to be formatted by the method. 
   * @param {any} format?. The object that contains the formatting properties. The argument should contain Intl.DateTimeFormat valid properties.
   */
  getFormattedValue(value: string | Date, format?: any): void;
  /**
   * Returns the date of the input.
   */
  getValue(): void;
  /**
   * Sets the date of the input.
   * @param {string | Date} value. The value to be set. 
   */
  setValue(value: string | Date): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-date-input"): DateInput;
        querySelector(selectors: "smart-date-input"): DateInput | null;
        querySelectorAll(selectors: "smart-date-input"): NodeListOf<DateInput>;
        getElementsByTagName(qualifiedName: "smart-date-input"): HTMLCollectionOf<DateInput>;
        getElementsByName(elementName: "smart-date-input"): NodeListOf<DateInput>;
    }
}

export interface DateRangeInputProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the format of the dates displayed in the input. Accepts valid ECMAScript Internationalization API format. By default the date format is determined by the 'locale' property. Intl.DateTimeFormat is used to format date strings in JavaScript
   * Default value: {"day": "numeric", "month": "numeric", "year": "numeric" }
   */
  dateFormat?: DateRangeFormat;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines the position of the drop down button.
   * Default value: right
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to an empty string. In this case the height of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Determines whether the 'Today/Clear' icons will be visible or not.
   * Default value: false
   */
  icons?: boolean;
  /**
   * Sets the purpose of the input and what, if any, permission the user agent has to provide automated assistance in filling out the element's input when in a form, as well as guidance to the browser as to the type of information expected in the element. This value corresponds to the standard HTML autocomplete attribute and can be set to values such as 'on', 'name', 'organization', 'street-address', etc.
   * Default value: "off"
   */
  inputPurpose?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Determines the max date for the Calendar displayed inside the popup.
   * Default value: new Date(2100, 1, 1)
   */
  max?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the min date for the Calendar displayed inside the popup.
   * Default value: new Date(1900, 1, 1)
   */
  min?: any;
  /**
   * Determines the number of months that will be displayed inside the popup.
   * Default value: 1
   */
  months?: number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the drop down is opened or not.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Determines whether ot not the user can enter text inside the input. if dropDownButtonPosition is set to 'left' or 'right' then readonly determines whether the element acts as a ComboBox or a DropDownList if a dataSource is provided.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the delimiter between the selected dates in the input. This delimiter is ued to distinguish the starting date from the ending date.
   * Default value: " - "
   */
  separator?: string;
  /**
   * Determines the format of the dates displayed in the input. Accepts valid ECMAScript Internationalization API format. By default the date foramt is determined by the 'locale' property.
   * Default value: { hour: '2-digit', minute: '2-digit' }
   */
  timeFormat?: TimeRangeFormat;
  /**
   * Determines whether time selection is available or not.
   * Default value: false
   */
  timepicker?: boolean;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element. The type of the value depends on the valueType property.
   * Default value: 
   */
  value?: any;
  /**
   * Determines the value type returned from the `value` property.
   * Default value: string
   */
  valueType?: DateRangeInputValueType | string;
}
/**
 DateRangeInput specifies an input field where the user can enter a date range ( from date - to date ). It also has a popup with a Calendar that allows to select a date range. Time selection is an additional feature that can be enabled which allows to specify time for the date range.
*/
export interface DateRangeInput extends BaseElement, DateRangeInputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, oldLabel, oldValue, value)
   *  label - The label of the new selected item.
   *  oldLabel - The label of the item that was previously selected before the event was triggered.
   *  oldValue - The value of the item that was previously selected before the event was triggered.
   *  value - The value of the new selected item.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Closes the drop down.
   */
  close(): void;
  /**
   * Opens the drop down.
   */
  open(): void;
  /**
   * Selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
}

/**Determines the format of the dates displayed in the input. Accepts valid ECMAScript Internationalization API format. By default the date format is determined by the 'locale' property. Intl.DateTimeFormat is used to format date strings in JavaScript */
export interface DateRangeFormat {
  /**
   * Day format.
   * Default value: numeric
   */
  day?: DateRangeFormatDay | string;
  /**
   * Month format.
   * Default value: numeric
   */
  month?: DateRangeFormatMonth | string;
  /**
   * Year format.
   * Default value: numeric
   */
  year?: DateRangeFormatYear | string;
}

/**Determines the format of the dates displayed in the input. Accepts valid ECMAScript Internationalization API format. By default the date foramt is determined by the 'locale' property. */
export interface TimeRangeFormat {
  /**
   * Hour format.
   * Default value: 2-digit
   */
  hour?: TimeRangeFormatHour | string;
  /**
   * Minute format.
   * Default value: 2-digit
   */
  minute?: TimeRangeFormatMinute | string;
}

declare global {
    interface Document {
        createElement(tagName: "smart-date-range-input"): DateRangeInput;
        querySelector(selectors: "smart-date-range-input"): DateRangeInput | null;
        querySelectorAll(selectors: "smart-date-range-input"): NodeListOf<DateRangeInput>;
        getElementsByTagName(qualifiedName: "smart-date-range-input"): HTMLCollectionOf<DateRangeInput>;
        getElementsByName(elementName: "smart-date-range-input"): NodeListOf<DateRangeInput>;
    }
}

/**Day format. */
export declare type DateRangeFormatDay = 'numeric' | '2-digit';
/**Month format. */
export declare type DateRangeFormatMonth = 'numeric' | '2-digit' | 'narrow' | 'short' | 'long';
/**Year format. */
export declare type DateRangeFormatYear = 'numeric' | '2-digit';
/**Hour format. */
export declare type TimeRangeFormatHour = 'numeric' | '2-digit';
/**Minute format. */
export declare type TimeRangeFormatMinute = 'numeric' | '2-digit';
/**Determines the value type returned from the `value` property. */
export declare type DateRangeInputValueType = 'string' | 'object';
export interface DateTimePickerProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Applies new animation settings to the calendar pop-up when it is enabled. Properties:startSpeed - Determines the initial speed of the animation.easeThreshold - Determines the point at which the animation starts to slow down - the "ease effect".step - Determines the step ( scrolling interval ) at which the animation will run. stepEaseSize - Coefficient that is used to calculated the new step once the threshold has been passed. resetThreshold - Determines the threshold for animation reset. When it's reached the animation will start over.
   * Default value: null
   */
  animationSettings?: any;
  /**
   * Determines whether the calendar button pop-up will be closed automatically when date or time is selected through it.
   * Default value: false
   */
  autoClose?: boolean;
  /**
   * Determines the delay before the calendar pop-up is automatically closed. Applicable only when autoClose is set to true.
   * Default value: 500
   */
  autoCloseDelay?: number;
  /**
   * Determines whether the calendar button is visible or not. The calendar button is used to open the Calendar popup to select a date.
   * Default value: false
   */
  calendarButton?: boolean;
  /**
   * Determines the position of the calendar button.
   * Default value: right
   */
  calendarButtonPosition?: DropDownButtonPosition | string;
  /**
   * Determines the header mode of the calendar pop-up.
   * Default value: default
   */
  calendarMode?: CalendarMode | string;
  /**
   * Sets or gets the format of calendar pop-up's day names.
   * Default value: firstTwoLetters
   */
  dayNameFormat?: DayFormat | string;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Disables auto navigation of the calendar pop-up when the user clicks on a date that's not from month in view.
   * Default value: false
   */
  disableAutoNavigation?: boolean;
  /**
   * Determines the time zone to display the value in.
   * Default value: unspecified
   */
  displayKind?: DateTimePickerDisplayKind | string;
  /**
   * Determines the type of the month/year view in the calendar pop-up when calendarMode is set to Default.
   * Default value: table
   */
  displayModeView?: CalendarDisplayModeView | string;
  /**
   * Sets custom container to append the pop-up to. By default, it is in the DateTimePicker. The value of the property can be an HTML element or the id of an HTML element.
   * Default value: "null"
   */
  dropDownAppendTo?: string;
  /**
   * Sets or gets the pop-up display mode (what components appear in it, and its behaviour).
   * Default value: default
   */
  dropDownDisplayMode?: DateTimePickerDropDownDisplayMode | string;
  /**
   * If this property is enabled, when the element's dropdown is opened, a transparent overlay is positioned between the dropdown and the rest of the document.
   * Default value: false
   */
  dropDownOverlay?: boolean;
  /**
   * Determines the pop-up position when opened.
   * Default value: auto
   */
  dropDownPosition?: DropDownPosition | string;
  /**
   * Determines how the the value can be edited inside the input.
   * Default value: default
   */
  editMode?: DateTimePickerEditMode | string;
  /**
   * Determines whether the value can be incremented/decremented with the mouse wheel when the mouse is over the input.
   * Default value: false
   */
  enableMouseWheelAction?: boolean;
  /**
   * Determines the first day of the week of the calendar pop-up. From 0(Sunday) to 6(Saturday)
   * Default value: 0
   */
  firstDayOfWeek?: number;
  /**
   * Sets or gets the footer template of the calendar pop-up. The value of this property can be the id of an HTMLTemplateElement or the HTMLTemplateElement itself. If set to null, a default template is applied.
   * Default value: null
   */
  footerTemplate?: any;
  /**
   * Determines whether to display a footer.
   * Default value: false
   */
  footer?: boolean;
  /**
   * Determines the pattern that is used to display the value in. Built-in Date formats:// short date pattern'd' - 'M/d/yyyy',// long date pattern'D' - 'dddd, MMMM dd, yyyy',// short time pattern't' - 'h:mm tt',// long time pattern'T' - 'h:mm:ss tt',// long date, short time pattern'f' - 'dddd, MMMM dd, yyyy h:mm tt',// long date, long time pattern'F' - 'dddd, MMMM dd, yyyy h:mm:ss tt',// month/day pattern'M' - 'MMMM dd',// month/year pattern'Y' - 'yyyy MMMM',// S is a sortable format that does not vary by culture'S' - 'yyyy'-'MM'-'dd'T'HH':'mm':'ss'Date format strings:'d'-the day of the month;'dd'-the day of the month'ddd'-the abbreviated name of the day of the week'dddd'- the full name of the day of the week'h'-the hour, using a 12-hour clock from 1 to 12'hh'-the hour, using a 12-hour clock from 01 to 12'H'-the hour, using a 24-hour clock from 0 to 23'HH'- the hour, using a 24-hour clock from 00 to 23'm'-the minute, from 0 through 59'mm'-the minutes,from 00 though59'M'- the month, from 1 through 12'MM'- the month, from 01 through 12'MMM'-the abbreviated name of the month'MMMM'-the full name of the month's'-the second, from 0 through 59'ss'-the second, from 00 through 59't'- the first character of the AM/PM designator'tt'-the AM/PM designator'y'- the year, from 0 to 99'yy'- the year, from 00 to 99'yyy'-the year, with a minimum of three digits'yyyy'-the year as a four-digit number;'yyyyy'-the year as a four-digit number.
   * Default value: "dd-MMM-yy HH:mm:ss.fff"
   */
  formatString?: string;
  /**
   * Sets custom header template for the calendar pop-up. Accepts the id of an HTMLTemplateElement or a reference ot it.
   * Default value: null
   */
  headerTemplate?: any;
  /**
   * Hides the names of the weekdays in the calendar pop-up.
   * Default value: false
   */
  hideDayNames?: boolean;
  /**
   * Determines if dates from other months are visible or not in the calendar pop-up.
   * Default value: false
   */
  hideOtherMonthDays?: boolean;
  /**
   * Hides the arrow of the tooltip in the calendar pop-up.
   * Default value: false
   */
  hideTooltipArrow?: boolean;
  /**
   * Sets additional helper text below the element that appears only when the element is focused.
   * Default value: ""
   */
  hint?: string;
  /**
   * Sets dates, displayed as important in the calendar pop-up.
   * Default value: 
   */
  importantDates?: string[] | Date[];
  /**
   * Sets a template for the important dates of the calendar pop-up. Accepts the id of or a reference to an HTMLTemplateElement.
   * Default value: null
   */
  importantDatesTemplate?: string | HTMLTemplateElement;
  /**
   * Sets or gets the increment/decrement interval when a date/time segment inside the input has not been highlighted. The default interval is 1 second. If a numeric value is passed, it represents milliseconds.
   * Default value: new JQX.Utilities.TimeSpan(0, 0, 1)
   */
  interval?: any;
  /**
   * Sets a label above the element.
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets the locale that determines what language is used to localize the labels inside the DateTimePicker.
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets the maximum allowed value.
   * Default value: new JQX.Utilities.DateTime(3001, 1, 1)
   */
  max?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "now": "Now",
   *     "dateTabLabel": "DATE",
   *     "timeTabLabel": "TIME"
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the minimum allowed value.
   * Default value: new JQX.Utilities.DateTime(1600, 1, 1)
   */
  min?: any;
  /**
   * Sets or gets the element's name, which is used as a reference when the data is submitted in a form.
   * Default value: ""
   */
  name?: string;
  /**
   * Enables or disables the setting of the value property to be null or empty string. If the property is enabled and the value in the input is deleted, the value will be set to null and the placeholder will be displayed. Otherwise, the current date and time will be set.
   * Default value: false
   */
  nullable?: boolean;
  /**
   * Sets or gets whether the calendar pop-up is opened.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Sets or gets the placeholder text to be shown in the input when value is null.
   * Default value: "Enter date"
   */
  placeholder?: string;
  /**
   * Disables user interaction with the element.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets an array of dates (DateTime or Date objects) that cannot be selected.
   * Default value: 
   */
  restrictedDates?: Date[];
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Sets or gets whether the spin buttons are displayed. Spin buttons are used to increment/decrement the date/time.
   * Default value: false
   */
  spinButtons?: boolean;
  /**
   * Sets the delay between repeats of the spin buttons in miliseconds.
   * Default value: 75
   */
  spinButtonsDelay?: number;
  /**
   * Sets a delay before the first repeat iteration of spin buttons in miliseconds.
   * Default value: 0
   */
  spinButtonsInitialDelay?: number;
  /**
   * Sets or gets the position of the spin buttons.
   * Default value: right
   */
  spinButtonsPosition?: DateTimePickerSpinButtonsPosition | string;
  /**
   * Sets or gets the element's visual theme. 
   * Default value: ""
   */
  theme?: string;
  /**
   * Enables/Disabled the tooltip for the important dates in the calendar pop-up.
   * Default value: false
   */
  tooltip?: boolean;
  /**
   * Sets the delay of the calendar pop-up's tooltip before it appears.
   * Default value: 100
   */
  tooltipDelay?: number;
  /**
   * Sets the position of the tooltip in the calendar pop-up.
   * Default value: top
   */
  tooltipPosition?: TooltipPosition | string;
  /**
   * Sets a template for the content of the calendar pop-up's tooltip. Accepts the id of or a reference to an HTMLTemplateElement.
   * Default value: null
   */
  tooltipTemplate?: string | HTMLTemplateElement;
  /**
   * Sets or gets if the element can be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Determines the validation mechanism for the value by min/max.
   * Default value: strict
   */
  validation?: Validation | string;
  /**
   * Sets or gets the value. The value represents the current date/time that is set to the element as a DateTime object.
   * Default value: new JQX.Utilities.DateTime()
   */
  value?: any;
  /**
   * Enables/Disabled week numbering in the calendar pop-up. 
   * Default value: false
   */
  weekNumbers?: boolean;
  /**
   * Determines the number of visible weeks in the calendar pop-up.
   * Default value: 6
   */
  weeks?: number;
  /**
   * Determines the first year of a 100 year span within which 2-digit years fall. The default value, 1926, would interpret the 2-digit year values 26-99 as 1926 to 1999 and values 00-25 as being 2000 to 2025.
   * Default value: 1926
   */
  yearCutoff?: number;
}
/**
 Date & time selection component. It allows to select both date and time with the same control.
*/
export interface DateTimePicker extends BaseElement, DateTimePickerProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the value is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The old value before it was changed presented as a DateTime object.
   *  value - The new value presented as a DateTime object.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the calendar pop-up is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the calendar pop-up is about to be closed. The closing operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the calendar pop-up is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the calendar pop-up is about to be opened. The opening operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Closes the calendar pop-up.
   */
  close(): void;
  /**
   * Focuses the input.
   */
  focus(): void;
  /**
   * Opens the calendar pop-up.
   */
  open(): void;
  /**
   * Selects the text inside the input.
   */
  select(): void;
  /**
   * Gets a Date object.
   * @returns {Date}
   */
  getDate(): Date;
  /**
   * Sets the date of the DateTimePicker.
   * @param {Date} date. The date object to be set.
   */
  setDate(date: Date): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-date-time-picker"): DateTimePicker;
        querySelector(selectors: "smart-date-time-picker"): DateTimePicker | null;
        querySelectorAll(selectors: "smart-date-time-picker"): NodeListOf<DateTimePicker>;
        getElementsByTagName(qualifiedName: "smart-date-time-picker"): HTMLCollectionOf<DateTimePicker>;
        getElementsByName(elementName: "smart-date-time-picker"): NodeListOf<DateTimePicker>;
    }
}

/**Determines the time zone to display the value in. */
export declare type DateTimePickerDisplayKind = 'UTC' | 'local' | 'unspecified';
/**Sets or gets the pop-up display mode (what components appear in it, and its behaviour). */
export declare type DateTimePickerDropDownDisplayMode = 'auto' | 'default' | 'classic' | 'calendar' | 'timePicker';
/**Determines how the the value can be edited inside the input. */
export declare type DateTimePickerEditMode = 'default' | 'full' | 'partial';
/**Sets or gets the position of the spin buttons. */
export declare type DateTimePickerSpinButtonsPosition = 'left' | 'right';
/**Determines the validation mechanism for the value by min/max. */
export declare type Validation = 'strict' | 'interaction';
export interface DockingLayoutProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * A getter that returns an array of all DockingLayout items that are auto hidden inside the element.
   * Default value: 
   */
  autoHideItems?: any;
  /**
   * Enable/Disable the automatic state loading. There must be a previously saved state of the Layout in order to load it.
   * Default value: false
   */
  autoLoadState?: boolean;
  /**
   * Enable/Disable the automatic state saving. Note: In order to save the state of the element it must have an id.
   * Default value: false
   */
  autoSaveState?: boolean;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * If set to false it will disable the dragging of DockingLayout items. If set items can only be repositioned using the API methods.
   * Default value: true
   */
  draggable: boolean;
  /**
   * If set to false it will disable item floating. This means that if a Window is floated as a result of dragging it will be returned back ot it's original position instead of being floated outside the DockingLayout. Already floated LayoutPanel items will not be affected.
   * Default value: true
   */
  floatable?: boolean;
  /**
   * Hides all splitter bars inside the element.
   * Default value: false
   */
  hideSplitterBars?: boolean;
  /**
   * A getter that returns an array of all DockingLayout items that are docked inside the element.
   * Default value: 
   */
  items?: any;
  /**
   * A getter that returns an array of all DockingLayout items that have been closed.
   * Default value: 
   */
  closedItems?: any;
  /**
   * Determines the structure of the element. This property represents an array of objects that define the hierarchy of the items inside the element and their specific settings. Each object must have a type attribute that defines it's behavior.  Three types of objects are allowed:   LayoutGroup - Represents a group of items (Splitter). Used when the user wants more than one DockingLayout item in a single container.  Properties: orientation - A string value indicating the orientation of the Splitter group. Possible values: 'horizontal', 'vertical'. size - A string | number value indicating the size of the Splitter group. items - An array of LayoutPanel object definitions. resizeMode - A string indicating the resize mode. Possible values: 'none', 'adjacent', 'end', 'proportional'. resizeStep - A nummeric value that determines the step of resizing. liveResize - Determines if splitter resizing happens while dragging or not.   LayoutPanel - Represents a DockingLayout item (TabsWindow). LayoutPanels can have one or many items (TabItem).  Properties:   id - the ID of the LayoutPanel. autoHide - a boolean property that determines if the LayoutPanel is autoHidden. autoHidePosition - determines the autoHide position of the item if 'autoHide' property is set. Possible values: 'top', 'bottom', 'left', 'right'. dropPosition - Determines the possible positions for the item at which a new item can be dropped as a result of dragging. Possible values: 'top', 'bottom', 'left', 'right', 'center', 'header', 'layout-top', 'layout-bottom', 'layout-left', 'layout-right'. Positions with the 'layout' prefix reflect on LayoutPanelItems that are children of the LayoutPanel. label - the Label of the LayoutPanel window. tabPosition - Determines the position of the Tab labels inside the LayoutPanel. layout - determines the DockingLayout owner of the LayoutPanel. Accepts a string indicating the ID of a DockingLayout on the page or a direct reference to it. headerButtons - an Array of strings that define the buttons in the header section of the DockingLayout item. tabCloseButtons - a boolean property that Enables or disables the close buttons inside each Tab item label inside the DockingLayout item. tabOverflow - same as 'overflow' property of jqxTabs. It defines the overflow mode of the labels of the Tab items inside a DockingLayout item.selectionMode - the same as jqxTabs selection modes. Applies to Tab items inside a DockingLayout item. tabResize - the same as 'resize' property of jqxTabs. Allows resizing the Tab labels inside the DockingLayout item. locked - Locks the size of the item and does not allow resizing. max - sets the maximum size of the item. min - sets the minimum size of the item size - sets the size of the item. items - an array of objects. Each object defines the structure of a LayoutPanelItem.   LayoutPanelItem - Represents a LayoutPanel item (TabItem). Properties:  id - the ID of the Tab item. label - a string representing the label of the Tab item. content - represents the content of the Tab item. Can be anything. selected - determines if the item is selected. By default the first added item to the LayoutPanel is automatically selected. draggable - a boolean property that allows to disable the dragging of the Tab item.    
   * Default value: 
   */
  layout?: any;
  /**
   * When enabled the resizing operation happens live. By default this feature is not enabled and the user sees a hightlighted bar while dragging instead of the actual splitter bar.
   * Default value: false
   */
  liveResize?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback, related to localization module. 
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "' method accepts an instance of JQX.TabsWindow",
   *     "invalidNodeRemove": "' method accepts an instance of JQX.TabsWindow that is a child of the DockingLayout.",
   *     "invalidNodeType": "' requires a \"jqx-tabs-window\" element to be passed as an argument.",
   *     "invalidTargetNode": "' requires an index of an item that is not hidden/closed. Only visible items that are part of the Layout are valid.",
   *     "invalidIndex": "' method accepts an index of type number.",
   *     "noId": "jqxDockingLayout requires an id in order to save/load a state."
   *   }
   * }
   */
  messages?: any;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the resize step during reisizing
   * Default value: 5
   */
  resizeStep?: number;
  /**
   * Determines the snap mode. Two modes are available:   simple - allows dragging of a single tab item inside or outside the layout. A semi-transparent highlighter is used to indicate the possible locations where the dragged item can be dropped. The user has to drop the dragged item inside one of the possible drop zones indicated by the highlighter. advanced - allows dragging of a whole TabsWindow with items or a single tab item. Uses a Visual Studio style feedback that indicates the possible drop locations. The user has to drop the target over one of the icons inside the feedback.   The feedback/highlighter is displayed when the dragging of an item begins. 
   * Default value: advanced
   */
  snapMode?: DockingLayoutSnapMode | string;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * A getter that returns an array of all DockingLayout items that have been undocked. Undocked items are no more part of the Layout's interal structure but can be inserted by dragging them in.
   * Default value: 
   */
  undockedItems?: any;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 DockingLayout enables the creation of complex layouts consisting of panels that can be floated, docked, nested, resized, pinned.
*/
export interface DockingLayout extends BaseElement, DockingLayoutProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the tab selection is changed. <strong>Note:</strong> Change event may be thrown by other JQX Custom Elements nested inside the Tab items.
	* @param event. The custom event.    */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a Tab item or a whole Tabs Window item ( DockingLayout item ) is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a Tab item/Tabs Window is about to be closed. The closing operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an item's position inside the Layout or it's size has been changed. Indicates that a state change has occured.
	* @param event. The custom event.    */
  onStateChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when item resizing begins.
	* @param event. The custom event.    */
  onResizeStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when item resizing finishes.
	* @param event. The custom event.    */
  onResizeEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Makes a "smart-tabs-window" node, that is a child of the Layout, auto hidden by placing it at the Bottom position inside the element. This means that the item will be positioned near the bottom side of the layout and it's content will be hidden until the user selects one of it's labels.
   * @param {HTMLElement | number | string} node. The "smart-tabs-window" or "smart-tab-item" node to append
   */
  autoHideBottom(node: HTMLElement | number | string): void;
  /**
   * Makes a "smart-tabs-window" node, that is a child of the Layout, auto hidden by placing it at the Left position inside the layout. This means that the item will be positioned near the left side of the layout and it's content will be hidden until the user selects one of it's labels.
   * @param {HTMLElement | number | string} node. The "smart-tabs-window" or "smart-tab-item" node to append
   */
  autoHideLeft(node: HTMLElement | number | string): void;
  /**
   * Makes a "smart-tabs-window" node, that is a child of the Layout, auto hidden by placing it at the Right position inside the layout. This means that the item will be positioned near the right side of the layout and it's content will be hidden until the user selects one of it's labels.
   * @param {HTMLElement | number | string} node. The "smart-tabs-window" or "smart-tab-item" node to append
   */
  autoHideRight(node: HTMLElement | number | string): void;
  /**
   * Makes a "smart-tabs-window" node, that is a child of the Layout, auto hidden by placing it at the Top position inside the layout. This means that the item will be positioned near the top side of the layout and it's content will be hidden until the user selects one of it's labels.
   * @param {HTMLElement | number | string} node. The "smart-tabs-window" or "smart-tab-item" node to append
   */
  autoHideTop(node: HTMLElement | number | string): void;
  /**
   * Clears the localStorage of any previous cached state of the DockingLayout.
   */
  clearState(): void;
  /**
   * The method will reset an autohidden item to it's normal behavior and re-insert it at a specified position. It can also be used to insert items into the DockingLayout. <strong>Note:</strong> Items inserted via this method are added as a top level items.
   * @param {string | number | Node} node. An autohidden "smart-tabs-window" item instance or a new "smart-tabs-window" instance.
   * @returns {Node}
   */
  dock(node: string | number | Node): Node;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The new item is inserted before the target item which corresponds to the index passed as the first argument to the method.
   * @param {number | HTMLElement | string} index. The index to insert a new TabsWindow at.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertBeforeItem(index: number | HTMLElement | string, item: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The new item is inserted after the target item which corresponds to the index passed as the first argument to the method.
   * @param {number | HTMLElement | string} index. The index to insert a new TabsWindow at.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertAfterItem(index: number | HTMLElement | string, item: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The new item is inserted by splitting  the target item which corresponds to the index passed as the first argument to the method in two and placing the new item as it's left neighbour horizontally.
   * @param {number | HTMLElement | string} index. The index to insert a new TabsWindow at.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertIntoLeft(index: number | HTMLElement | string, item: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The new item is inserted by splitting  the target item which corresponds to the index passed as the first argument to the method in two and placing the new item as it's right neighbour horizontally.
   * @param {number | HTMLElement | string} index. The index to insert a new TabsWindow at.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertIntoRight(index: number | HTMLElement | string, item: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The new item is inserted by splitting  the target item which corresponds to the index passed as the first argument to the method in two and placing the new item as it's top neighbour vertically.
   * @param {number | HTMLElement | string} index. The index to insert a new TabsWindow at.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertIntoTop(index: number | HTMLElement | string, item: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The new item is inserted by splitting  the target item which corresponds to the index passed as the first argument to the method in two and placing the new item as it's bottom neighbour vertically.
   * @param {number | HTMLElement | string} index. The index to insert a new TabsWindow at.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertIntoBottom(index: number | HTMLElement | string, item: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The new item is inserted as a top level item positioned at the top side inside the Layout.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertLayoutTop(item: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The new item is inserted as a top level item positioned at the bottom side inside the Layout.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertLayoutBottom(item: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The new item is inserted as a top level item positioned at the left side inside the Layout.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertLayoutLeft(item: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The new item is inserted as a top level item positioned at the right inside the Layout.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertLayoutRight(item: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The target item and it's neighbour items are placed inside a new splitter item that acts as the new neghbour of the newly added item via this method. The new item is inserted above the target ( at position Top).
   * @param {number | HTMLElement | string} index. The index to insert a new TabsWindow at.
   * @param {any} tabsWindow. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertOutsideTargetGroupTop(index: number | HTMLElement | string, tabsWindow: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The target item and it's neighbour items are placed inside a new splitter item that acts as the new neghbour of the newly added item via this method. The new item is inserted bellow the target ( at position Bottom).
   * @param {number | HTMLElement | string} index. The index to insert a new TabsWindow at.
   * @param {any} tabsWindow. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertOutsideTargetGroupBottom(index: number | HTMLElement | string, tabsWindow: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The target item and it's neighbour items are placed inside a new splitter item that acts as the new neghbour of the newly added item via this method. The new item is inserted before the target ( at position Left).
   * @param {number | HTMLElement | string} index. The index to insert a new TabsWindow at.
   * @param {any} tabsWindow. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertOutsideTargetGroupLeft(index: number | HTMLElement | string, tabsWindow: any): void;
  /**
   * Inserts a new TabsWindow into the DockingLayout or creates a TabsWindow instance from an object passed as the second argument. The target item and it's neighbour items are placed inside a new splitter item that acts as the new neghbour of the newly added item via this method. The new item is inserted after the target ( at position Right).
   * @param {number | HTMLElement | string} index. The index to insert a new TabsWindow at.
   * @param {any} tabsWindow. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   */
  insertOutsideTargetGroupRight(index: number | HTMLElement | string, tabsWindow: any): void;
  /**
   * Inserts a new TabsWindow. The window is in floating mode and is undocked.
   * @param {any} item. An instance of a TabsWindow or an Object with the fields "label", "items" and other additional.
   * @param {number | string} left?. The left position of the new window. You can use number, px or %. For example: '10px'.
   * @param {number | string} top?. The top position of the new window. You can use number, px or %. For example: '10px'.
   */
  insertFloatingWindow(item: any, left?: number | string, top?: number | string): void;
  /**
   * The method returns an array of all autohidden items.
   * @param {string} orientation?. Determines which auto hidden items to return ( vertical or horizontal ). If not set the method will return all autohidden items. Possible values: 'vertical', 'horizontal'.
   * @returns {any[]}
   */
  getAutoHideItems(orientation?: string): any[];
  /**
   * The method returns the index of a target item.
   * @param {HTMLElement} node. Returns the index of the target item.
   * @returns {number}
   */
  getIndex(node: HTMLElement): number;
  /**
   * Returns an array of objects representing the current structure of the element. Each object represents a Layout item with it's settings and hierarchy.
   * @param {boolean} noInstances?. Determines if the returned array will contain HTML references or not. When saving to localStorage the resulted array should not contain any HTMLElement references.
   * @returns {any[]}
   */
  getState(noInstances?: boolean): any[];
  /**
   * Returns the Splitter parent of a Layout item
   * @param {HTMLElement} item?. DockingLayout item
   * @returns {HTMLElement}
   */
  getItemGroupElement(item?: HTMLElement): HTMLElement;
  /**
   * Returns a JSON array of objects representing the current structure of the element. Ready to be persisted to LocalStorage.
   * @returns {any[]}
   */
  getJSONStructure(): any[];
  /**
   * Loads a previously saved state of the element. If no state is provided as an argument the method will do a localStorage lookup.
   * @param {any[]} state?. An array of objects that represents a cached state of the DockingLayout. The result of calling the 'saveState' method.
   */
  loadState(state?: any[]): void;
  /**
   * Removes a DockingLayout item (TabsWindow) from the element.
   * @param {number | HTMLElement | string} index. The index of the TabsWindow to remove or a reference to it.
   */
  removeAt(index: number | HTMLElement | string): void;
  /**
   * Removes all items from the element.
   */
  removeAll(): void;
  /**
   * Removes a "smart-tabs-window" node that is a DockingLayout item.
   * @param {Node} node. The "smart-tabs-window" node to remove.
   * @returns {Node}
   */
  removeChild<T extends Node>(node: Node): T;
  /**
   * Saves the current state of the DockingLayout to LocalStorage. The state includes the hierarchy and size of the items.
   */
  saveState(): void;
  /**
   * The method undocks/removes an item from the Layout and places it at the same position but it's no more part of the layout. The item becomes an 'outher' item that can be reinserted at any time.
   * @param {string | number | Node} node. A "smart-tabs-window" instance that is part of the DockingLayout.
   */
  undock(node: string | number | Node): void;
  /**
   * Updates a TabsWindow and it's contents.
   * @param {number | HTMLElement | string} index. The index of the TabsWindow to update.
   * @param {any} settings. An object that contains the new settings for the TabsWindow item. Settings object is the same as defining a new TabsWindow with the exception of 'items' array where the items are defined. In order to change the label or content of a Tab item the user has to specify the index of the target tab item.
   */
  update(index: number | HTMLElement | string, settings: any): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-docking-layout"): DockingLayout;
        querySelector(selectors: "smart-docking-layout"): DockingLayout | null;
        querySelectorAll(selectors: "smart-docking-layout"): NodeListOf<DockingLayout>;
        getElementsByTagName(qualifiedName: "smart-docking-layout"): HTMLCollectionOf<DockingLayout>;
        getElementsByName(elementName: "smart-docking-layout"): NodeListOf<DockingLayout>;
    }
}

/**Determines the snap mode. Two modes are available:   simple - allows dragging of a single tab item inside or outside the layout. A semi-transparent highlighter is used to indicate the possible locations where the dragged item can be dropped. The user has to drop the dragged item inside one of the possible drop zones indicated by the highlighter.
 advanced - allows dragging of a whole TabsWindow with items or a single tab item. Uses a Visual Studio style feedback that indicates the possible drop locations. The user has to drop the target over one of the icons inside the feedback.
  <br/> The feedback/highlighter is displayed when the dragging of an item begins.  */
export declare type DockingLayoutSnapMode = 'simple' | 'advanced';
export interface DropDownButtonProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the delay before the opened drop down closes when dropDownOpenMode is set to 'auto'.
   * Default value: 100
   */
  autoCloseDelay?: number;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets the parent container of the dropDown (the popup). Used when a CSS property of unknowned parent is interfering with the visibility of the dropDown.
   * Default value: "null"
   */
  dropDownAppendTo?: string;
  /**
   * Determines the position of the drop down button.
   * Default value: right
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the max height of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownMaxHeight?: string | number;
  /**
   * Sets the max width of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownMaxWidth?: string | number;
  /**
   * Sets the min height of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownMinHeight?: string | number;
  /**
   * Sets the min width of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownMinWidth?: string | number;
  /**
   * Determines how the drop down is going to open.
   * Default value: default
   */
  dropDownOpenMode?: DropDownOpenMode | string;
  /**
   * If this property is enabled, when the element's dropdown is opened, a transparent overlay is positioned between the dropdown and the rest of the document.
   * Default value: false
   */
  dropDownOverlay?: boolean;
  /**
   * Sets a placeholder text to appear when there is no content inside the dropdown.
   * Default value: "No Items"
   */
  dropDownPlaceholder?: string;
  /**
   * Determines the vertical position of the dropDown. 'Auto' means its automatically determined depending on the viewport size.
   * Default value: auto
   */
  dropDownPosition?: DropDownPosition | string;
  /**
   * Sets the width of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Sets additional helper text below the element. The hint is visible only when the element is focused.
   * Default value: ""
   */
  hint?: string;
  /**
   * Determines the visibility of the horizontal Scroll bar inside the drop down.
   * Default value: auto
   */
  horizontalScrollBarVisibility?: HorizontalScrollBarVisibility | string;
  /**
   * Sets a label above the element. The label is always visible.
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines whether the popup is opened or closed
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the element's placeholder, displayed in the element's action button container.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Determines the element's placeholder template, displayed in the element's action button container. You can pass 'string', 'function' or HTMLTemplateElement as a value.
   * Default value: null
   */
  placeholderTemplate?: any;
  /**
   * Disables user interaction with the element.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Determines whether the resize indicator in the bottom right corner is visible or not.
   * Default value: false
   */
  resizeIndicator?: boolean;
  /**
   * Determines whether the dropDown can be resized or not. When resizing is enabled, a resize bar appears on the top/bottom side of the drop down.
   * Default value: null
   */
  resizeMode?: ResizeMode | string;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Determines the visibility of the vertical scroll bar.
   * Default value: auto
   */
  verticalScrollBarVisibility?: VerticalScrollBarVisibility | string;
}
/**
 DropDownButton displays any type of content like components, text, images, etc in a DropDown.
*/
export interface DropDownButton extends BaseElement, DropDownButtonProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when user clicks on the action button. The action button is visible when the <strong>placeholder</strong> is set.
	* @param event. The custom event.    */
  onActionButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the drop down list is about to be closed. This event allows to cancel the closing operation calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user clicks on the drop down button.
	* @param event. The custom event.    */
  onDropDownButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down is about to be opened. This event allows to cancel the opening operation calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user starts resizing the drop down.
	* @param event. The custom event. Custom data event was created with: ev.detail(position)
   *  position - An object containing the current left and top positions of the drop down.
   */
  onResizeStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user finishes resizing the drop down.
	* @param event. The custom event. Custom data event was created with: ev.detail(position)
   *  position - An object containing the current left and top positions of the drop down.
   */
  onResizeEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Appends a new HTML node to the drop down.
   * @param {Node} node. The node to be appended
   * @returns {Node}
   */
  appendChild<T extends Node>(node: Node): T;
  /**
   * Closes the dropDown.
   */
  close(): void;
  /**
   * Opens the dropDown.
   */
  open(): void;
  /**
   * Removes everything from the drop down.
   */
  removeAll(): void;
  /**
   * Removes a child node from the drop down.
   * @param {Node} node. The node to remove.
   * @returns {Node}
   */
  removeChild<T extends Node>(node: Node): T;
  /**
   * Scrolls the drop down to a specific position.
   * @param {number} top. Y axis coordinate
   * @param {number} left. X axis coordinate
   */
  scrollTo(top: number, left: number): void;
   scrollTo(options?: ScrollToOptions): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-drop-down-button"): DropDownButton;
        querySelector(selectors: "smart-drop-down-button"): DropDownButton | null;
        querySelectorAll(selectors: "smart-drop-down-button"): NodeListOf<DropDownButton>;
        getElementsByTagName(qualifiedName: "smart-drop-down-button"): HTMLCollectionOf<DropDownButton>;
        getElementsByName(elementName: "smart-drop-down-button"): NodeListOf<DropDownButton>;
    }
}

export interface DropDownListProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Used only when dropDownOpenMode is set to 'auto'. Determines the delay before the opened drop down closes if the pointer is not over the element.
   * Default value: 100
   */
  autoCloseDelay?: number;
  /**
   * Determines the data source that will be loaded to the DropDownList. The dataSource can be an array of strings/numbers or objects where the attributes represent the properties of a List Item. For example label, value, group. It can also be a callback that returns an Array of items as previously described.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines whether an indicator will appear during filtering and remote item loading.
   * Default value: false
   */
  displayLoadingIndicator?: boolean;
  /**
   * Sets or gets the displayMember. The displayMember specifies the name of an object property to display. The name is contained in the collection specified by the 'dataSource' property.
   * Default value: """"
   */
  displayMember?: string;
  /**
   * Determines the drop down parent. The expected value is CSS Selector, ID or 'body'. The drop down can be removed from the body of the element and continue to work in another container. This is usefull when one of the parents of the element doesn't allow overflowing, by settings this property to 'body' the drop down will be appended to the DOM and the popup will open/close as usual. dropDownAppendTo can be a string representing the id of an HTML element on the page or a direct reference to that element. Reseting it back to null will take the drop down back to it's original place.
   * Default value: "null"
   */
  dropDownAppendTo?: string;
  /**
   * Determines the position of the drop down button.
   * Default value: right
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to an empty string. In this case the height of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the maximum Height of the drop down. By default it's set to an empty string. In this case the maxHeight of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownMaxHeight?: string | number;
  /**
   * Sets the maximum Width of the drop down. By default it's set to an empty string. In this case the maxWidth of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownMaxWidth?: string | number;
  /**
   * Sets the minimum Height of the drop down. By default it's set to an empty string. In this case the minHeight of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownMinHeight?: string | number;
  /**
   * Sets the minimum Width of the drop down. By default it's set to an empty string. In this case the minWidth of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownMinWidth?: string | number;
  /**
   * Determines how the drop down is going to open.
   * Default value: default
   */
  dropDownOpenMode?: DropDownOpenMode | string;
  /**
   * If this property is enabled, when the element's dropdown is opened, a transparent overlay is positioned between the dropdown and the rest of the document. The purpose of the overlay is to make sure that clicking anywhere outside the popup will will target the overlay and not the DOM.
   * Default value: false
   */
  dropDownOverlay?: boolean;
  /**
   * Determines the placeholder for the drop down list when it's empty.
   * Default value: "No Items"
   */
  dropDownPlaceholder?: string;
  /**
   * Determines the position of the drop down when opened.
   * Default value: auto
   */
  dropDownPosition?: DropDownPosition | string;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Determines whether filtering is enabled.
   * Default value: false
   */
  filterable?: boolean;
  /**
   * Determines the placeholder for the filter input inside the drop down that is only visible when filterable is enabled.
   * Default value: ""
   */
  filterInputPlaceholder?: string;
  /**
   * Determines the filtering mode of the drop down list.
   * Default value: startsWithIgnoreCase
   */
  filterMode?: FilterMode | string;
  /**
   * If enabled, the items will be grouped by their first letter. Can't be applied if the dataSource already contains groups.
   * Default value: false
   */
  grouped?: boolean;
  /**
   * Determines which attribute from the dataSource object will be used as the group member for the items. If not set, by default 'group' property is used from the source object. groupMember is especially usefull when loading the data from a JSON file as a dataSource for the ListBox and there's a specific property that should be used to group the items.
   * Default value: null
   */
  groupMember?: string | null;
  /**
   * Sets additional helper text below the element. The hint is visible only when the element is focused.
   * Default value: ""
   */
  hint?: string;
  /**
   * Determines the visibility of the horizontal Scroll bar inside the drop down.
   * Default value: auto
   */
  horizontalScrollBarVisibility?: HorizontalScrollBarVisibility | string;
  /**
   * Represents the property name of a List item. Determines the value of the input when a ListItem is selected. Usefull in cases where the user wants to display for example the value of an item instead of it's label. By default the label is displayed in the input.
   * Default value: ""
   */
  inputMember?: string;
  /**
   * IncrementalSearchDelay property specifies the time-interval in milliseconds until the previous search query is cleared. The timer starts when the user stops typing. A new query can be started only when the delay has passed.
   * Default value: 700
   */
  incrementalSearchDelay?: number;
  /**
   * Sets ot gets the mode of the incremental search mode. Incremental search is enabled by default. Typing while the drop down is focused starts the incremental search.
   * Default value: startsWithIgnoreCase
   */
  incrementalSearchMode?: SearchMode | string;
  /**
   * Sets the height for all list items. Used only when virtualization is enabled.
   * Default value: null
   */
  itemHeight?: number | null;
  /**
   * Determines the item width measuring algorithm.
   * Default value: auto
   */
  itemMeasureMode?: ListItemMeasureMode | string;
  /**
   * A getter that returns an array of all List items inside the drop down.
   * Default value: 
   */
  items?: any;
  /**
   * The itemTemplate property is a string that represents the id of an HTMLTemplateElement in the DOM or it's direct reference. It's used to set a customize the content of the list items.
   * Default value: null
   */
  itemTemplate?: any;
  /**
   * Sets a little text label above the element.
   * Default value: ""
   */
  label?: string;
  /**
   * Determines the text that will be displayed next to the loading indicator when the loader is visible and it's position is top or bottom.
   * Default value: "Loading..."
   */
  loadingIndicatorPlaceholder?: string;
  /**
   * Determines the position of the loading indicator.
   * Default value: center
   */
  loadingIndicatorPosition?: VerticalAlignment | string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the popup is opened or not.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the element's placeholder, displayed in the element's selection field.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Disables user interaction with the element.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines whether the resize indicator in the bottom right corner of the drop down is visible or not. This property is used in conjunction with resizeMode.
   * Default value: false
   */
  resizeIndicator?: boolean;
  /**
   * Determines whether the dropDown can be resized or not. When resizing is enabled, a resize bar appears on the top/bottom side of the drop down.
   * Default value: none
   */
  resizeMode?: ResizeMode | string;
  /**
   * Determines what will be displayed in the dropDown selection field.
   * Default value: plain
   */
  selectionDisplayMode?: SelectionDisplayMode | string;
  /**
   * Sets or gets the selected indexes. Selected indexes represents an array of the indexes of the items that should be selected.
   * Default value: 
   */
  selectedIndexes?: number[];
  /**
   * Sets or gets elected indexes. Selected values represents the values of the items that should be selected.
   * Default value: 
   */
  selectedValues?: string[];
  /**
   * Determines how many items can be selected.
   * Default value: zeroAndOne
   */
  selectionMode?: ListSelectionMode | string;
  /**
   * Determines whether the items are sorted alphabetically or not
   * Default value: false
   */
  sorted?: boolean;
  /**
   * Determines sorting direction - ascending(asc) or descending(desc)
   * Default value: "asc"
   */
  sortDirection?: string;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets a custom template for the content of the tokens when selectionDisplayMode is set to 'tokens'.
   * Default value: null
   */
  tokenTemplate?: any;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value. Returns the selection. Return type: {label: string, value: any}[].
   * Default value: null
   */
  value?: any;
  /**
   * Determines the value member of an item. Stored as value in the item object. Similar to groupMember, valueMember is especially usefull when using data from a JSON file as a dataSource for the ListBox and there's a specific property that should be used for the value the items.
   * Default value: """"
   */
  valueMember?: string;
  /**
   * Determines the visibility of the vertical scroll bar.
   * Default value: auto
   */
  verticalScrollBarVisibility?: VerticalScrollBarVisibility | string;
  /**
   * Determines weather or not Virtualization is used. Virtualization allows a huge amount of items to be loaded to the drop down while preserving the performance. For example a milion items can be loaded.
   * Default value: false
   */
  virtualized?: boolean;
}
/**
 The DropDownList is a form component that lets you choose a single predefined value from a list. It is a more advanced version of the 'select' tag.
*/
export interface DropDownList extends BaseElement, DropDownListProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when user clicks on the action button. The action button is only visible when dropDownOpenMode is set to 'dropDownbutton'.
	* @param event. The custom event.    */
  onActionButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(addedItems, disabled, index, label, removedItems, selected, value)
   *  addedItems - An array of List items that have been selected.
   *  disabled - A flag indicating whether or not the item that caused the change event is disabled.
   *  index - The index of the List item that triggered the event.
   *  label - The label of the List item that triggered the event.
   *  removedItems - An array of List items that have been unselected before the event was fired.
   *  selected - The selected state of the List item that triggered the event. If an item was selected the value will be true and vice versa.
   *  value - The value of the List item that triggered the event.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the drop down list is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the drop down list is about to be closed. This event allows to cancel the closing operation calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user clicks on the drop down button.
	* @param event. The custom event.    */
  onDropDownButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an item is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(disabled, index, label, selected, value)
   *  disabled - Indicates whether the List item that was clicked is disabled or not.
   *  index - Indicates the index of the List item that was clicked.
   *  label - The label of the List item that was clicked.
   *  selected - Indicates whether the List item that was clicked is selected or not.
   *  value - The value of the List item that was clicked.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down list is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down list is about to be opened. This event allows to cancel the opening operation calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user starts resizing the drop down.
	* @param event. The custom event. Custom data event was created with: ev.detail(position)
   *  position - An object containing the current left and top positions of the drop down.
   */
  onResizeStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the resizing of the drop down is finished.
	* @param event. The custom event. Custom data event was created with: ev.detail(position)
   *  position - An object containing the current left and top positions of the drop down.
   */
  onResizeEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user scrolls to the end of the dropDown list.
	* @param event. The custom event.    */
  onScrollBottomReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user scrolls to the start of the dropDown list.
	* @param event. The custom event.    */
  onScrollTopReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Appends a ListItem to the end of the list of items inside element.
   * @param {Node} node. A ListItem element that should be added to the rest of the items as the last item.
   * @returns {Node}
   */
  appendChild<T extends Node>(node: Node): T;
  /**
   * Removes all items from the drop down list.
   */
  clearItems(): void;
  /**
   * Unselects all items.
   */
  clearSelection(): void;
  /**
   * Closes the dropDown list.
   */
  close(): void;
  /**
   * Ensures the desired item is visible by scrolling to it.
   * @param {HTMLElement | string} item. A list item or value of the desired item to be visible.
   */
  ensureVisible(item: HTMLElement | string): void;
  /**
   * Returns an item instance from the dropDown list.
   * @param {string} value. The value of an item from the drop down list.
   * @returns {HTMLElement}
   */
  getItem(value: string): HTMLElement;
  /**
   * Inserts a new item at a specified position.
   * @param {number} position. The position where the item must be inserted.
   * @param {any} value. The value of the new item.
   */
  insert(position: number, value: any): void;
  /**
   * Inserts a new ListItem before another in the list.
   * @param {Node} node. A ListItem element that should be added before the referenceItem in the list.
   * @param {Node | null} referenceNode. A ListItem element that acts as the reference item before which a new item is about to be inserted. The referenceNode must be in the same list as the node.
   * @returns {Node}
   */
  insertBefore<T extends Node>(node: Node, referenceNode: Node | null): T;
  /**
   * Opens the dropDown list.
   */
  open(): void;
  /**
   * Removes an item at a specified position.
   * @param {number} position. The position of the removed item.
   */
  removeAt(position: number): void;
  /**
   * Removes a ListItem from the list of items inside element.
   * @param {Node} node. A ListItem element that is part of the list of items inside the element.
   * @returns {Node}
   */
  removeChild<T extends Node>(node: Node): T;
  /**
   * Selects an item from the dropDown list. 
   * @param {string | HTMLElement} item. A string, representing the value of the item or an HTML Element referencing an item from the list
   */
  select(item: string | HTMLElement): void;
  /**
   * Unselects an item from the dropDown list. 
   * @param {string | HTMLElement} item. A string, representing the value of the item or an HTML Element referencing an item from the list
   */
  unselect(item: string | HTMLElement): void;
  /**
   * Updates an item from the dropDown list.
   * @param {number} position. The position where the item must be updated.
   * @param {any} value. The value of the updated item.
   */
  update(position: number, value: any): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-drop-down-list"): DropDownList;
        querySelector(selectors: "smart-drop-down-list"): DropDownList | null;
        querySelectorAll(selectors: "smart-drop-down-list"): NodeListOf<DropDownList>;
        getElementsByTagName(qualifiedName: "smart-drop-down-list"): HTMLCollectionOf<DropDownList>;
        getElementsByName(elementName: "smart-drop-down-list"): NodeListOf<DropDownList>;
    }
}

export interface EditorProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Automatically loads the last saved state of the editor (from local storage) on element initialization. An id must be provided in order to load a previously saved state.
   * Default value: false
   */
  autoLoad?: boolean;
  /**
   * Automatically saves the current content of the editor. Saving happens at time intervas determined by the autoSaveInterval property while the element on focus. An id must be provided to the element in order to store the state.
   * Default value: false
   */
  autoSave?: boolean;
  /**
   * The property that determines the interval to automatically save the state of the Editor when the autoSave property is set.
   * Default value: 1000
   */
  autoSaveInterval?: number;
  /**
   * A formatting function for the char counter. Takes two arguments: chars - the current number of characters inside the Editor.maxCharCount - the maximum number of characters inside the Editor.
   * Default value: null
   */
  charCountFormatFunction?: any;
  /**
   * Sets or gets whether files will be automatically uploaded after selection.
   * Default value: false
   */
  autoUpload?: boolean;
  /**
   * Determines the content filtering settings.
   * Default value: [object Object]
   */
  contentFiltering?: EditorContentFiltering;
  /**
   * Determines the context menu for the Editor. The context menu is triggered when the user right clicks on the content area of the Editor.
   * Default value: default
   */
  contextMenu?: EditorContextMenu | string;
  /**
   * Allows to customize default the context menu of the Editor. The property accepts an array of items which can be strings that represent the value of the item, or objects of the following format: { label: string, value: string }, where the label will be displayed and the value will be action value for the item. The property also accepts a function that must return an array of items with the following format function (target: HTMLElement, type: string, defaultItems: string[]) { return defaultItems } and the following arguments: target - the element that is the target of the context menu.type - the type of context menu ( whether it's a table, image, link or other)defaultItems - an array of strings which represent the default items for the context menu.
   * Default value: null
   */
  contextMenuDataSource?: string[] | { label: string, value: 'string' }[] | Function | null;
  /**
   * Sets the Editor's Data Export options.
   * Default value: [object Object]
   */
  dataExport?: EditorDataExport;
  /**
   * Enables or disables the Editor.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Disables content editing inside Editor.
   * Default value: false
   */
  disableEditing?: boolean;
  /**
   * Disables the Quick Search Bar.
   * Default value: false
   */
  disableSearchBar?: boolean;
  /**
   * Determines the edit mode for the Editor. By default the editor's content accepts and parses HTML. However if set to 'markdown' the Editor can be used as a full time Markdown Editor by parsing the makrdown to HTML in preview mode.
   * Default value: html
   */
  editMode?: EditMode | string;
  /**
   * Determines whether the value returned from getHTML method and Source Code view are encoded or not.
   * Default value: false
   */
  enableHtmlEncode?: boolean;
  /**
   * Determines whether the Tab key can insert tab chars inside the Editor or change focus (default)
   * Default value: false
   */
  enableTabKey?: boolean;
  /**
   * Determines the time interval between results for the find and replace and search bar features.
   * Default value: 50
   */
  findAndReplaceTimeout?: number;
  /**
   * Determines whether the Toolbar is hidden or not.
   * Default value: false
   */
  hideToolbar?: boolean;
  /**
   * Determines whether the Inline Toolbar is hidden or not.
   * Default value: false
   */
  hideInlineToolbar?: boolean;
  /**
   * Determines the file format of the image/video that are uploaded from local storage. By default images/videos are stroed as base64.
   * Default value: base64
   */
  imageFormat?: EditorImageFormat | string;
  /**
   * Sets the content of the Editor as HTML. Allows to insert text and HTML.
   * Default value: "en"
   */
  innerHTML: string;
  /**
   * Defines an offset(x,y) for the Inline Toolbar positioning on the page.
   * Default value: [0, -5]
   */
  inlineToolbarOffset?: number[];
  /**
   * Determines the iframe settings of the Editor. When enabled the contents of the Editor are placed inside an iframe, isolated in a separate dom. The element allows to insert external resources into the iframe if needed.
   * Default value: [object Object]
   */
  iframeSettings?: EditorIframeSettings;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Sets a limit on the number of chars inside the Editor. 
   * Default value: null
   */
  maxCharCount?: number;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property language. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownName": "Invalid property name: ''!",
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid ''!",
   *     "propertyInvalidValueType": "Invalid ''!",
   *     "methodInvalidValueType": "Invalid ''!",
   *     "methodInvalidArgumentsCount": "Invalid '' argument(s)!",
   *     "methodInvalidReturnType": "Invalid ''!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": "'.",
   *     "htmlTemplateNotSuported": ": Web Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidValue": ".",
   *     "incorrectArgument": ".",
   *     "permissionsRequired": ".",
   *     "timeout": ": The import request has timed out.",
   *     "importError": ".",
   *     "exportError": ".",
   *     "ok": "Ok",
   *     "cancel": "Cancel",
   *     "alignLeft": "Align Left",
   *     "alignCenter": "Align Center",
   *     "alignRight": "Align Right",
   *     "alignJustify": "Align Justify",
   *     "segoeUi": "Segoe UI",
   *     "arial": "Arial",
   *     "georgia": "Georgia",
   *     "impact": "Impact",
   *     "tahoma": "Tahoma",
   *     "timesNewRoman": "Times New Roman",
   *     "verdana": "Verdana",
   *     "p": "Paragraph",
   *     "pre": "Code",
   *     "code": "Code",
   *     "blockquote": "Quotation",
   *     "h1": "Heading 1",
   *     "h2": "Heading 2",
   *     "h3": "Heading 3",
   *     "h4": "Heading 4",
   *     "h5": "Heading 5",
   *     "h6": "Heading 6",
   *     "bold": "Bold",
   *     "italic": "Italic",
   *     "underline": "Underline",
   *     "strikethrough": "Strikethrough",
   *     "orderedlist": "Ordered List",
   *     "unorderedlist": "Unordered List",
   *     "subscript": "Subscript",
   *     "superscript": "Superscript",
   *     "alignment": "Alignments",
   *     "fontname": "Font Name",
   *     "fontsize": "Font Size",
   *     "formats": "Formats",
   *     "backgroundcolor": "Background Color",
   *     "fontcolor": "Font Color",
   *     "redo": "Redo",
   *     "undo": "Undo",
   *     "indent": "Indent",
   *     "outdent": "Outdent",
   *     "createlink": "Hyperlink",
   *     "hyperlink": "Hyperlink",
   *     "editlink": "Hyperlink",
   *     "removelink": "Remove link",
   *     "openlink": "Open link",
   *     "image": "Image",
   *     "video": "Video",
   *     "table": "Table",
   *     "lowercase": "Lower Case",
   *     "uppercase": "Upper Case",
   *     "print": " Print",
   *     "cut": " Cut",
   *     "copy": " Copy",
   *     "paste": " Paste",
   *     "clearformat": "Clear Format",
   *     "fullscreen": "Full Screen",
   *     "restore": "Restore Screen",
   *     "sourcecode": "Source Code",
   *     "preview": "Preview",
   *     "splitmode": "Split Editor",
   *     "address": "Web Address",
   *     "text": "Display Text",
   *     "addressPlaceholder": "http://example.com",
   *     "textPlaceholder": "Enter Text",
   *     "targetPlaceholder": "Select Target",
   *     "titlePlaceholder": "Enter a Title",
   *     "urlPlaceholder": "http://example.com/image.png",
   *     "srcPlaceholder": "https://www.youtube.com/embed/video_link",
   *     "thumbnail": "Or provide a URL as a video thumbnail",
   *     "thumbnailPlaceholder": "https://www.link-to-thumbnail.jpg",
   *     "videoUrl": "Video URL",
   *     "videoUrlPlaceholder": "https://www.youtube.com/video_link",
   *     "captionPlaceholder": "Caption",
   *     "altPlaceholder": "Alternative Text",
   *     "widthPlaceholder": "auto",
   *     "heightPlaceholder": "auto",
   *     "target": "Open Link in",
   *     "linkBlankDescr": "New Window",
   *     "linkSelfDescr": "Same Window",
   *     "linkParentDescr": "Parent Frame",
   *     "linkTopDescr": "Full Body of the Window",
   *     "linkCustomDescr": "Custom Frame Name",
   *     "title": "Title",
   *     "url": "Or provide the URL to an image",
   *     "src": "Or provide the URL to an embed video",
   *     "width": "Width",
   *     "height": "Height",
   *     "alt": "Alternative Text",
   *     "caption": "Caption",
   *     "display": "Display",
   *     "displayPlaceholder": "Display",
   *     "displayBlock": "Block",
   *     "displayInline": "Inline",
   *     "draggable": "Enable Dragging",
   *     "resizable": "Enable Resizing",
   *     "browse": "Browse",
   *     "connectionError": ": File Upload requires connection to the server.",
   *     "wrongItemIndex": ": There is no file with such an index in the list of uploaded files.",
   *     "tooLongFileName": ": File name is too long.",
   *     "totalFiles": "Total files: ",
   *     "cancelFile": "Cancel File",
   *     "dashedborders": "Dashed Borders",
   *     "altrows": "Alternate Rows",
   *     "insertRowBefore": "Insert Row Before",
   *     "insertRowAfter": "Insert Row After",
   *     "deleteRow": "Delete Row",
   *     "insertColumnLeft": "Insert Column Left",
   *     "insertColumnRight": "Insert Column Right",
   *     "deleteColumn": "Delete Column",
   *     "alignTop": "Align Top",
   *     "alignMiddle": "Align Middle",
   *     "alignBottom": "Align Bottom",
   *     "delete": "Delete",
   *     "tablerows": "Table Rows",
   *     "tablecolumns": "Table Columns",
   *     "tablevalign": "Table Cell Vertical Align",
   *     "tablestyle": "Table Style",
   *     "tableheader": "Table Header",
   *     "buttonLabel": "Custom Table",
   *     "pasteLabel": "Choose the paste format action:",
   *     "cols": "Columns",
   *     "rows": "Rows",
   *     "alphabet": "abcdefghijklmnopqrstuvwxyz",
   *     "header": "Header",
   *     "column": "Column",
   *     "plainText": "Plain Text",
   *     "keepFormat": "Keep Format",
   *     "cleanFormat": "Clean Format",
   *     "roleDescription": "Text Editor",
   *     "iframeTitle": "Editor Content is Encloused in it's own DOM inside an iFrame",
   *     "toolbarButtonAriaLabel": "Toolbar Toggle Button",
   *     "primaryToolbarAriaLabel": "Primary Toolbar",
   *     "secondaryToolbarAriaLabel": "Secondary Toolbar",
   *     "inputAriaLabel": "Enter Text",
   *     "homeTab": "Home",
   *     "viewTab": "View",
   *     "insertTab": "Insert",
   *     "layoutTab": "Layout",
   *     "undoGroup": "Undo",
   *     "clipboardGroup": "Clipboard",
   *     "fontGroup": "Font",
   *     "paragraphGroup": "Paragraph",
   *     "editingGroup": "Editing",
   *     "tableGroup": "Tables",
   *     "imageGroup": "Images",
   *     "videoGroup": "Videos",
   *     "linkGroup": "Links",
   *     "viewsGroup": "Views",
   *     "deleteGroup": "Delete",
   *     "findandreplace": "Find and Replace",
   *     "requiredMessage": "The content of the Editor cannot be empty",
   *     "tableProperties": "Table Properties",
   *     "imageProperties": "Image Properties",
   *     "videoProperties": "Video Properties",
   *     "linkProperties": "Link Properties",
   *     "selectAll": "Select All",
   *     "deleteTable": "Delete Table",
   *     "deleteImage": "Delete Image",
   *     "deleteVideo": "Delete Video",
   *     "createLink": "Create Link",
   *     "deleteLink": "Delete Link",
   *     "copyImage": "Copy",
   *     "cutImage": "Cut",
   *     "copyVideo": "Copy",
   *     "cutVideo": "Cut",
   *     "find": "Find",
   *     "findPlaceholder": "",
   *     "replace": "Replace",
   *     "replaceWith": "Replace With",
   *     "replaceAll": "Replace All",
   *     "replacePlaceholder": "",
   *     "results": "Results",
   *     "resultsPlaceholder": "No match",
   *     "matchCase": "Match Case",
   *     "searchPlaceholder": "Search..."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets a to the element which can be used to submit the value of the Editor via a form.
   * Default value: null
   */
  name?: string | null;
  /**
   * Determines the format of the content that will be pasted inside the Editor.
   * Default value: keepFormat
   */
  pasteFormat?: PasteFormat | string;
  /**
   * Determines the placeholder that will be shown when there's no content inside the Editor.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Determines whether the clearFormat toolbar action should also remove inline styles from the currently selected node.
   * Default value: false
   */
  removeStylesOnClearFormat?: boolean;
  /**
   * Determines whether Editor's content is required ot not. If set and the Editor's content is empty, a notification will appear to notify that the Editor cannot be empty.
   * Default value: false
   */
  required?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines whether the value is sanitized from XSS content or not. When enabled scripts and other XSS vulnerabilities are not allowed to exist inside the Editor's as HTML content.
   * Default value: false
   */
  sanitized?: boolean;
  /**
   * Determines whether the char counter is visible or not. When enabled it is displayed in the bottom right corner. If maxCharCount is set and the content characters are equal or more than 70% of the maximum char count the counter is colored in order to warn the user. If the char count is equal or more than 90% the counter is again colored with a different warning color to indicate that the counter is near maximum. When maximum is reached, text input is not allowed.
   * Default value: false
   */
  showCharCount?: boolean;
  /**
   * Determines whether the editor may be checked for spelling errors.
   * Default value: true
   */
  spellCheck?: boolean;
  /**
   * Determines the refresh interval for the Source Code/Preview Panel when Split Mode is enabled. 
   * Default value: 100
   */
  splitModeRefreshTimeout?: number;
  /**
   * Sets or gets the upload URL. This property corresponds to the upload form's action attribute. For example, the uploadUrl property can point to a PHP file, which handles the upload operation on the server-side.
   * Default value: ""
   */
  uploadUrl?: string;
  /**
   * Sets or gets the remove URL. This property corresponds to the form's action attribute. For example, the removeUrl property can point to a PHP file, which handles the remove operation on the server-side.
   * Default value: ""
   */
  removeUrl?: string;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines the Toolbar items list. Each item can be string pointing to the name of the item or an object that defines a custom item or adds aditional settings to an item. The name of the items are case insensitive. An object definition should contain a name attribute that refers to the name of the item when modifing an existing toolbar item. The name attribute determines the action of the item. If set to 'custom' it is possible to create a custom toolbar item. If name attribute is not set or not valid it is treated as a separator, no a toolbar item. The following items are supported by default by the Editor: SourceCode - shows the HTML/Preview Panel by hiding the input panel. Item type - 'Toggle button'.SplitMode - shows both input and HTML/Preview Panel by splitting the Editor content in two sections. Item type - 'Toggle button'FullScreen - fits the viewport with the Editor by expanding it over the page content. Item type - 'Toggle button'.Alignment - aligns the selected content. Item type - 'Drop down'.FontName - changes the font family of the selected content. Item type - 'drop-down'.FontSize - changes the font size of the selected content. Item type - 'drop-down'.Formats - changes the format of the current selection. Itme type - 'drop-down'.TableRows - allows to insert/remove a row into a selected table element. Item type - 'drop-down'.TableColumns - allows to insert/remove a column into a selected table element. Itme type - 'drop-down'.TableVAlign - sets the vertical alignment of a selected table cell. Item type - 'drop-down'.TableStyle - sets additional styling to a selected table inside the Editor. Item type - 'drop-down'.BackgroundColor - changes the background color of the current selection. Item type - 'color-input'.FontColor - changes the font color of the current selection. Item type = 'color-input'.Bold - sets the currently selected text as bold or not. Item type - 'button'.Italic - sets the currently selected text as italic. Item type - 'button'. Underline - sets the currently selected text as underlined. Itme type - 'button'.Strikethrough - set the currently selected text as strikethrough. Item type - 'button'.Delete - deletes the current selection. Item type - 'button'.Undo - undoes the last operation. Item type - 'button'.Redo - redoes the previous operation. Item type - 'button'.Indent - indents the current selection once. Item type - 'button'.Outdent - outdents the current selection once. Item type - 'button'.OpenLink - triggers a hyperlink. Item type - 'button'.EditLink - creates/edits the selected hyperlink. Item type - 'button'.CreateLink - creates/edits the selected hyperlink. Item type - 'button'.RemoveLink - removes the currently selected hyperlink. Item type - 'button'.Hyperlink - same as createLink, triggers a Dialog Window for link creation. Item type - 'button'.Cut - Cuts the currently selected text. Item type - 'button'.Copy - copies the currently selected text. Item type - 'button'Paste - pastes the currenly copied/cut text from the Clipboard. Item type = 'button' or 'drop-down' when advanced attribute is set to 'true'.Image - triggers a Dialog Window to insert/edit an image. Item type - 'button'.Video - triggers a Dialog Window to insert/edit a video. Item type - 'button'.LowerCase - changes the current selection to lower case. Item type - 'button'.UpperCase - changes the current selection to upper case. Item type - 'button'.Print - opens the browser print preview window. Item type - 'button'.Caption - insert/remove a caption when a table is selected. Item type - 'button'.ClearFormat - removes the formatting of the currntly selected text. Item type - 'button'.Table - triggers a Dialog Window to insert a table. Item type - 'button'.TableHeader - insert/remove a header row to the currently selected table. Item type - 'button'.OrderedList - insert/remove an order list. Item type = 'button'.UnorderedList - insert/remove an unordered list. Item type - 'button'.Subscript - changes the currently selected text to subscript. Item type - 'button'.Superscript - changes the currently selected text to superscript. Item type - 'button'.FindAndReplace - opens a dialog that allows to find and replace text inside the Editor's content section. Item type - 'button'.  The inlineToolbarItems attribute is applicable only to the following items: 'table', 'image', 'hyperlink'. It accepts the same type of value as toolbarItems property but the toolbar items will be placed insinde the Inline Toolbar instead.
   * Default value: bold,italic,underline,|,formats,alignment,orderedList,unorderedList,|,hyperlink,image,video,|,sourceCode,redo,undo
   */
  toolbarItems?: ToolbarItem[];
  /**
   * Determines the toolbar mode of the Editor. The main toolbar of the Editor can appear as a Ribbon or as a Menu.
   * Default value: menu
   */
  toolbarMode?: ToolbarMode | string;
  /**
   * Allows to configure the SingleLineRibbon appearance by changing the order and items of the groups.
   * Default value: [{"name":"homeTab","groups":[{"name":"undoGroup","items":["undo","redo"]},{"name":"clipboardGroup","items":["cut","copy","paste"]},{"name":"fontGroup","items":["fontName","fontSize","backgroundColor","fontColor","clearFormat","formats","bold","italic","underline","strikethrough","superscript","subscript"]},{"name":"paragraphGroup","items":["orderedList","unorderedList","indent","outdent","alignment"]},{"name":"editingGroup","items":["findAndReplace"]}]},{"name":"insertTab","groups":[{"name":"tableGroup","items":["table"]},{"name":"imageGroup","items":["image"]}{"name":"videoGroup","items":["video"]},{"name":"linkGroup","items":["createLink","removeLink"]}]},{"name":"viewTab","groups":[{"name":"viewsGroup","items":["fullScreen","sourceCode","splitMode"]}]},{"name":"layoutTab","hidden":true,"groups":[{"name":"deleteGroup","items":["delete"]},{"name":"tableGroup","items":["table","tableHeader","tableRows","tableColumns","tableVAlign","tableStyle",""]},{"name":"imageGroup","items":["image","caption"]},{"name":"videoGroup","items":["video","caption"]}]}]
   */
  toolbarRibbonConfig?: { name: string, groups: { name: string, items: string[] }[] }[];
  /**
   * Determines the format of the content that will be pasted inside the Editor.
   * Default value: toggle
   */
  toolbarViewMode?: ToolbarViewMode | string;
  /**
   * Sticks the Toolbar to the top of the window and stays there while scrolling.
   * Default value: false
   */
  toolbarSticky?: boolean;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the Editor.
   * Default value: """"
   */
  value?: string;
  /**
   * A function that can be used to completly customize the Editor dialog that is used to insert/edit tables/images/videos/hyperlinks. The function accepts two arguments: target - the target dialog that is about to be opened.item - the toolbar item object that trigger the dialog.
   * Default value: null
   */
  windowCustomizationFunction?: any;
}
/**
 jqxEditor represents an advanced HTML text editor which can simplify web content creation or be a replacement of your HTML/Markdown Text Areas.
*/
export interface Editor extends BaseElement, EditorProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered on blur if the content is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The old value before the change.
   *  value - The new value after the change.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered after user input to indicate that the content is changed via user interaction.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The old value before the input change.
   *  value - The new value after the input change.
   */
  onChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered before a Toolbar action is started. The event can be canceled via event.preventDefault().
	* @param event. The custom event. Custom data event was created with: ev.detail(name)
   *  name - The name of the action.
   */
  onActionStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a Toolbar action has ended.
	* @param event. The custom event. Custom data event was created with: ev.detail(name)
   *  name - The name of the action.
   */
  onActionEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a Context menu item has been clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(originalEvent, value)
   *  originalEvent - The original click event.
   *  value - The value of the item.
   */
  onContextMenuItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Context Menu is opened.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, owner)
   *  target - The toolbar that is the target of the operation.
   *  owner - The tooltip target (the owner of the tooltip).
   */
  onContextMenuOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Context Menu is opening. The opening operation can be canceled via event.preventDefault().
	* @param event. The custom event. Custom data event was created with: ev.detail(target)
   *  target - The toolbar that is the target of the operation.
   */
  onContextMenuOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Context Menu is closed.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, owner)
   *  target - The toolbar that is the target of the operation.
   *  owner - The tooltip target (the owner of the tooltip).
   */
  onContextMenuClose?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Context Menu is closing. The closing operation can be canceled via event.preventDefault().
	* @param event. The custom event. Custom data event was created with: ev.detail(target)
   *  target - The toolbar that is the target of the operation.
   */
  onContextMenuClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an image/table/video resizing has started.
	* @param event. The custom event.    */
  onResizeStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an image/table/video resizing has ended.
	* @param event. The custom event. Custom data event was created with: ev.detail(target)
   *  target - The element that is resized (image/table or video).
   */
  onResizeEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the inline Toolbar is opened.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, owner)
   *  target - The toolbar that is the target of the operation.
   *  owner - The tooltip target (the owner of the tooltip).
   */
  onInlineToolbarOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the inline Toolbar is opening. The opening operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(target)
   *  target - The toolbar that is the target of the operation.
   */
  onInlineToolbarOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the inline Toolbar is closed.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, owner)
   *  target - The toolbar that is the target of the operation.
   *  owner - The tooltip target (the owner of the tooltip).
   */
  onInlineToolbarClose?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the inline Toolbar is closing.
	* @param event. The custom event. Custom data event was created with: ev.detail(target)
   *  target - The toolbar that is the target of the operation. The closing operation can be canceled by calling event.preventDefault() in the event handler function.
   */
  onInlineToolbarClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Drop Down Toolbar is opened.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, owner)
   *  target - The toolbar that is the target of the operation.
   *  owner - The tooltip target (the owner of the tooltip).
   */
  onDropDownToolbarOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Drop Down Toolbar is opening. The opening operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(target)
   *  target - The toolbar that is the target of the operation.
   */
  onDropDownToolbarOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Drop Down Toolbar is closed.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, owner)
   *  target - The toolbar that is the target of the operation.
   *  owner - The tooltip target (the owner of the tooltip).
   */
  onDropDownToolbarClose?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Drop Down Toolbar is closing. The closing operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(target)
   *  target - The toolbar that is the target of the operation.
   */
  onDropDownToolbarClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered the Dialog Window is opened.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, item)
   *  target - The window that is the target of the operation.
   *  item - The toolbar item is the target of the operation.
   */
  onDialogOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered before the Dialog Window is opened. The event can be prevented via event.preventDefault().
	* @param event. The custom event. Custom data event was created with: ev.detail(target, item)
   *  target - The window that is the target of the operation.
   *  item - The toolbar item that is the target of the operation.
   */
  onDialogOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Dialog Window is closed.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, item)
   *  target - The window that is the target of the operation.
   *  item - The toolbar item that is the target of the operation.
   */
  onDialogClose?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered before the Dialog Window is closing. The event can be prevented via event.preventDefault().
	* @param event. The custom event. Custom data event was created with: ev.detail(target, item)
   *  target - The window that is the target of the operation.
   *  item - The toolbar item that is the target of the operation.
   */
  onDialogClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the uploading of an image/video is successful.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, item, filename, type, size, index, status, serverResponse)
   *  target - The file upload element that is the target of the operation.
   *  item - The toolbar item that is the target of the operation.
   *  filename - The name of the uploaded file.
   *  type - The type of the uploaded file.
   *  size - The size of the uploaded file.
   *  index - The index of the uploaded file.
   *  status - The status of the uploaded file. Whether there was an error or success.
   *  serverResponse - The response of the remote server.
   */
  onImageUploadSuccess?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the uploading of an image/video is unsuccessful.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, item, filename, type, size, index, status, serverResponse)
   *  target - The file upload element that is the target of the operation.
   *  item - The toolbar item that is the target of the operation.
   *  filename - The name of the canceled file.
   *  type - The type of the canceled file.
   *  size - The size of the canceled file.
   *  index - The index of the canceled file.
   *  status - The status of the uploaded file. Whether there was an error or success.
   *  serverResponse - The response of the remote server.
   */
  onImageUploadFailed?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a Toolbar item is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(originalEvent, value)
   *  originalEvent - The original click event.
   *  value - The name of the toolbar item that was clicked.
   */
  onToobarItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a message is closed.
	* @param event. The custom event. Custom data event was created with: ev.detail(instance)
   *  instance - The toast item that is the target of the operation.
   */
  onMessageClose?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a message is opened.
	* @param event. The custom event. Custom data event was created with: ev.detail(instance)
   *  instance - The toast item that is the target of the operation.
   */
  onMessageOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Blurs the content of the Editor.
   */
  blur(): void;
  /**
   * Clears the content of the Editor.
   */
  clearContent(): void;
  /**
   * Collapse the Toolbar if the <b>toolbarViewMode</b> is set to 'toggle'.
   */
  collapseToolbar(): void;
  /**
   * Disables a Toolbar item.
   * @param {string} itemName. The name of the toolbar item to disable.
   */
  disableToolbarItem(itemName: string): void;
  /**
   * Expand the Toolbar if the <b>toolbarViewMode</b> is set to 'toggle'.
   */
  expandToolbar(): void;
  /**
   * Enables a previously disabled Toolbar item.
   * @param {string} itemName. The name of the toolbar item to enable.
   */
  enableToolbarItem(itemName: string): void;
  /**
   * Executes a command via the native <b>execCommand</b> method. The method returns true or false depending on whether the execution was successful or not. The following list of commands can be eexecuted: <ul><li>bold - makes the currently selected content bold. Example: <b>editor.executeCommand('bold');</b></li><li>italic - makes the currently selected content italic. Example: <b>editor.executeCommand('italic');</b></li><li>undelined - makes the currently selected content underlined. Example: <b>editor.executeCommand('underline');</b></li><li>strikeThrough - applies a single line strike through formatting to the currently selected content. Example: <b>editor.executeCommand('strikeThrough');</b></li><li>superscript - sets the selected content as superscript. Example: <b>editor.executeCommand('superscript');</b></li><li>subscript - sets the selected content as superscript. Example: <b>editor.executeCommand('subscript');</b></li><li>uppercase - changes the case of the current selection to upper. Example: <b>editor.executeCommand('uppercase');</b></li><li>lowercase - changes the case of the current selection to lower. Example: <b>editor.executeCommand('lowercase');</b></li><li>foreColor - changes the font color of the current content selection. Example: <b>editor.executeCommand('foreColor', '#000000');</b></li><li>fontName - changes the font name for the selected content. Example: <b>editor.executeCommand('fontName', 'Arial');</b></li><li>fontSize - changes the font size of the currently selected content. Example: <b>editor.executeCommand('fontSize', '15px');</b></li><li>hiliteColor - changes the background color of current selection. Example: <b>editor.executeCommand('hiliteColor', '#000000');</b></li><li>justifyCenter - aligns the content to the center. Example: <b>editor.executeCommand('justifyCenter');</b></li><li>justifyFull - aligns the content to be fully justified. Example: <b>editor.executeCommand('justifyFull');</b></li><li>justifyLeft - aligns the content to the left. Example: <b>editor.executeCommand('justifyLeft');</b></li><li>justifyRight - aligns the content to the right. Example: <b>editor.executeCommand('justifyRight');</b></li><li>undo - allows to undo the previous action. Example: <b>editor.executeCommand('undo');</b></li><li>redo - allows to redo the previous actions. Example: <b>editor.executeCommand('redo');</b></li><li>createLink - creates a hyperlink in the content section of the Editor. Example: <b>editor.executeCommand('createLink', { text: 'Links', url: 'http://', title : 'Link' });</b></li><li>indent - indents the content with one level. Example: <b>editor.executeCommand('indent');</b></li><li>outdent - outdents the content with one level. Example: <b>editor.executeCommand('outdent');</b></li><li>insertHTML - insert an HTML content as string at the current cursor location. Example: <b>editor.executeCommand('insertHTML', '<p>Text</p>');</b></li><li>insertOrderedList - inserts a new numbered list item. Example: <b>editor.executeCommand('insertOrderedList');</b></li><li>insertUnorderedList - inserts a new bulleted list item. Example: <b>editor.executeCommand('insertUnorderedList');</b></li><li>removeFormat - removes the formatting styles from currently selected text. Example: <b>editor.executeCommand('removeFormat');</b></li><li>insertText - inserts a text at the current cursor location. Example: <b>editor.executeCommand('insertText', 'Some text to insert');</b></li><li>insertImage - inserts an image at the current cursor location. Example: <b>editor.executeCommand('insertImage', { url: 'https://www.htmlelements.com/demos/images/carousel-medium-2.jpg'});</b></li></ul>
   * @param {string} commandName. The name of the command to execute.
   * @param {string | number} value?. The value for the command. Some commands require a value to be passed, others do not.
   * @returns {boolean}
   */
  executeCommand(commandName: string, value?: string | number): boolean;
  /**
   * Focuses the content of the Editor.
   */
  focus(): void;
  /**
   * Returns the number of characters inside the Editor's content.
   * @returns {number}
   */
  getCharCount(): number;
  /**
   * Returns the current selection range. By default the result is an object of type Range, but if the <b>editMode</b> property is set to 'markdown' the returned value is an object indicating the start/end indexes of the current selection. 
   * @returns {any}
   */
  getSelectionRange(): any;
  /**
   * Returns the content of the Editor as HTML. When <b>editMode</b> is set to 'markdown' the markdown is parsed and returned as HTML.
   * @returns {string}
   */
  getHTML(): string;
  /**
   * Returns the content of the Editor as text.
   * @returns {string}
   */
  getText(): string;
  /**
   * Hides a specific message or all messages if no argument is provided.
   * @param {HTMLElement | string | number} item?. Hides a specific message. The argument can be a DOM reference to a specific item, it's index or it's id. If the argument is not provided then all messages will be closed.
   */
  hideMessage(item?: HTMLElement | string | number): void;
  /**
   * Hides the last shown message.
   */
  hideLastMessage(): void;
  /**
   * Shows a custom message inside the Editor.
   * @param {string} message. The text message to be displayed.
   * @param {any} settings?. Additional settings that can be applied to the Toast element that handles the messages. This parameter should contain only valid Toast properties and values.
   * @returns {HTMLElement | undefined}
   */
  showMessage(message: string, settings?: any): HTMLElement | undefined;
  /**
   * Selects the text inside Editor's content.
   */
  selectAll(): void;
  /**
   * Selects a range of text inside the Editor. The method will find the nodes containing the text from the start to the end indexes and will select them as ranges. However, currently only FireFox supports multiple range selection. The rest of the browsers will only select the first node. If the editor is in 'html' <b>editMode</b> then the expected text will be selected regardless of the browser because there's only one node inside the editor.
   * @param {number} startIndex. The start index to select from.
   * @param {number} endIndex. The end index to select to.
   */
  selectRange(startIndex: number, endIndex: number): void;
  /**
   * Clears the local storage from previously stored states of the Editor with the current id.
   */
  clearState(): void;
  /**
   * Saves the current state of the Editor to local storage. Requires an id to be set to the Editor.
   */
  saveState(): void;
  /**
   * Loads the last stored state of the Editor from local storage. Requires an id to be set to the Editor.
   */
  loadState(): void;
  /**
   * Sets Editor into Split Mode. In split mode the HTML/Markdown editor and SourceCode/Preview panels are visible.
   * @param {boolean} value?. Determines whether to enter or leave split mode. By default the argument is not passed and the mode is toggled.
   */
  splitMode(value?: boolean): void;
  /**
   * Sets Editor into SourceCode/Preview Mode. In this mode the HTML view panel is displayed.
   * @param {boolean} value?. Determines whether to enter or leave split mode. By default the argument is not passed and the mode is toggled.
   */
  previewMode(value?: boolean): void;
  /**
   * Sets Editor into Full Screen Mode. If enabled the Editor is positioned above the page content and fills the screen.
   * @param {boolean} value?. Determines whether to enter or leave split mode. By default the argument is not passed and the mode is toggled.
   */
  fullScreenMode(value?: boolean): void;
  /**
   * Exports the content of the Editor in the desired format. The currently supported formats are: HTML, Markdown and PDF.
   * @param {string} dataFormat. The expected file format.
   * @param {any} callback?. A callback that is executed before the data is exported. Allows to modify the output.
   */
  exportData(dataFormat: string, callback?: any): void;
  /**
   * Imports the content of a file to the Editor. The currently supported formats are: TXT or HTML.
   * @param {any} source. The url to the file or an object that defines settings for the Ajax request like url, timeput, etc. Object format: { url: string, type: string, data: object, timeout: number }
   * @param {any} settings?. Additional settings for the ajax request. Such as loadError function, contentType, etc. Format: { contentType: string, beforeSend: Function, loadError: Function, beforeLoadComplete: Function  }
   */
  importData(source: any, settings?: any): void;
  /**
   * Opens the Print Preview Panel of the Browser to print Editor's content.
   */
  print(): void;
  /**
   * Allows to update the settings of a single toolbar item. The method returns <b>true</b> if successful.
   * @param {string | number} name. The name of the toolbar item or it's index inside the <b>toolbarItems</b> array.
   * @param {any} settings. A settings object for the toolbar item. It should have the same definition as when defining a custom toolbar item. You can read more about it in the dedicated topic for the Editor Toolbar on the website.
   * @returns {boolean | undefined}
   */
  updateToolbarItem(name: string | number, settings: any): boolean | undefined;
}

/**Determines the content filtering settings. */
export interface EditorContentFiltering {
  /**
   * Determines which element attributes to filter.
   * Default value: null
   */
  attributes?: string[];
  /**
   * Determines whether to allow (whiteList) or remove (blackList) the specified attributes.
   * Default value: blackList
   */
  attributeFilterMode?: EditorContentFilteringAttributeFilterMode | string;
  /**
   * Determines which element tags to filter.
   * Default value: null
   */
  tags?: string[];
  /**
   * Determines whether to allow (whiteList) or remove (blackList) the specified tags.
   * Default value: blackList
   */
  tagFilterMode?: EditorContentFilteringTagFilterMode | string;
  /**
   * Determines which style attributes to filter.
   * Default value: null
   */
  styleAttributes?: string[];
  /**
   * Determines whether to allow (whiteList) or remove (blackList) the specified style attributes.
   * Default value: blackList
   */
  styleAttributeFilterMode?: EditorContentFilteringStyleAttributeFilterMode | string;
}

/**Sets the Editor's Data Export options. */
export interface EditorDataExport {
  /**
   * Sets a custom style object of the dataExport. 
   * Default value: null
   */
  style?: any;
  /**
   * Sets the exported file's name.
   * Default value: "jqxScheduler"
   */
  fileName?: string;
  /**
   * Sets the page orientation, when exporting to PDF.
   * Default value: "portrait"
   */
  pageOrientation?: string;
}

/**Determines the iframe settings of the Editor. When enabled the contents of the Editor are placed inside an iframe, isolated in a separate dom. The element allows to insert external resources into the iframe if needed. */
export interface EditorIframeSettings {
  /**
   * Determines the attributes and their values that will be set to the iframe. Here's how to define attributes: attributes: { height: 500 }
   * Default value: null
   */
  attributes?: any;
  /**
   * Enables/Disables the usage of an iframe for the content of the Editor.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Determines the resources like scripts/styles that will be imported into the iframe. Here's how to define resources: resources: { 'style': { href: 'styles.css' }, 'script': { src: 'index.js', type: 'module' }} 
   * Default value: "portrait"
   */
  resources?: string;
}

export interface ToolbarItem {
  /**
   * The unique name of the toolbar item. The list of all possible names is available in the description above.
   * Default value: 
   */
  name?: string | null;
  /**
   * Determines whethet the item is disabled or not.
   * Default value: false
   */
  disabled?: boolean | null;
  /**
   * Applicable only to item 'paste'. Transforms the type of the Paste toolbar item to drop down list with paste format options.
   * Default value: false
   */
  advanced?: boolean | null;
  /**
   * Allows to set a different dataSource for the toolbar items of type 'drop-down' or 'color-input.
   * Default value: 
   */
  dataSource?: any;
  /**
   * Defines the list of toolbar items for the Inline Toolbar. Accept the same values as toolbarItems property.
   * Default value: 
   */
  inlineToolbarItems?: any;
  /**
   * Allows to set predefined values for the Dialog Window of the following toolbar items: 'table', 'image', 'video', 'hyperlink'.
   * Default value: [object Object]
   */
  editor?: ToolbarItemEditor;
  /**
   * Allows to set a template for a custom Toolbar item when the name attribute is set to 'custom'.
   * Default value: null
   */
  template?: any;
  /**
   * Determines the width of the toolbar item.
   * Default value: 
   */
  width?: number | string | null;
}

/**Allows to set predefined values for the Dialog Window of the following toolbar items: 'table', 'image', 'video', 'hyperlink'. */
export interface ToolbarItemEditor {
  /**
   * Allows to preset the value for the hyperlink address field in the Dialog Window.
   * Default value: 
   */
  address?: string | null;
  /**
   * Allows to preset the value for the hyperlink target field in the Dialog Window.
   * Default value: 
   */
  target?: string | null;
  /**
   * Allows to preset the value for the hyperlink text field in the Dialog Window.
   * Default value: 
   */
  text?: string | null;
  /**
   * Allows to preset the value for the hyperlink/image title field in the Dialog Window.
   * Default value: 
   */
  title?: string | null;
  /**
   * Allows to preset the value for the image/video file uploader in the Dialog Window.
   * Default value: null
   */
  file?: any;
  /**
   * Allows to preset the value for the image/video alt field in the Dialog Window.
   * Default value: 
   */
  alt?: string | null;
  /**
   * Allows to preset the value for the image/video url field in the Dialog Window.
   * Default value: 
   */
  url?: string | null;
  /**
   * Allows to preset the value for the image/table/video width field in the Dialog Window.
   * Default value: 
   */
  width?: string | number;
  /**
   * Allows to preset the value for the image/table/video height field in the Dialog Window.
   * Default value: 
   */
  height?: string | number;
  /**
   * Allows to preset the value for the image/video caption field in the Dialog Window.
   * Default value: 
   */
  caption?: string | null;
  /**
   * Allows to preset the value for the image/video alignment field in the Dialog Window.
   * Default value: 
   */
  alignment?: string | null;
  /**
   * Allows to preset the value for the image/video display field in the Dialog Window.
   * Default value: 
   */
  display?: string | null;
  /**
   * Allows to preset the value for the image draggable field in the Dialog Window.
   * Default value: false
   */
  draggable?: boolean | null;
  /**
   * Allows to preset the value for the image/table/video resizable field in the Dialog Window.
   * Default value: false
   */
  resizable?: boolean | null;
  /**
   * Allows to preset the value for the table cols field in the Dialog Window.
   * Default value: 
   */
  cols?: number | string | null;
  /**
   * Allows to preset the value for the table rows field in the Dialog Window.
   * Default value: 
   */
  rows?: number | string | null;
  /**
   * Allows to preset the value for the table header field in the Dialog Window.
   * Default value: false
   */
  tableheader?: boolean | null;
  /**
   * Allows to preset the value for the table altrows field in the Dialog Window.
   * Default value: false
   */
  altrows?: boolean | null;
  /**
   * Allows to preset the value for the table dashedborders field in the Dialog Window.
   * Default value: false
   */
  dashedborders?: boolean | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-editor"): Editor;
        querySelector(selectors: "smart-editor"): Editor | null;
        querySelectorAll(selectors: "smart-editor"): NodeListOf<Editor>;
        getElementsByTagName(qualifiedName: "smart-editor"): HTMLCollectionOf<Editor>;
        getElementsByName(elementName: "smart-editor"): NodeListOf<Editor>;
    }
}

/**Determines whether to allow (whiteList) or remove (blackList) the specified attributes. */
export declare type EditorContentFilteringAttributeFilterMode = 'blackList' | 'whiteList';
/**Determines whether to allow (whiteList) or remove (blackList) the specified tags. */
export declare type EditorContentFilteringTagFilterMode = 'blackList' | 'whiteList';
/**Determines whether to allow (whiteList) or remove (blackList) the specified style attributes. */
export declare type EditorContentFilteringStyleAttributeFilterMode = 'blackList' | 'whiteList';
/**Determines the context menu for the Editor. The context menu is triggered when the user right clicks on the content area of the Editor. */
export declare type EditorContextMenu = 'default' | 'browser' | 'none';
/**Determines the edit mode for the Editor. By default the editor's content accepts and parses HTML. However if set to 'markdown' the Editor can be used as a full time Markdown Editor by parsing the makrdown to HTML in preview mode. */
export declare type EditMode = 'html' | 'markdown';
/**Determines the file format of the image/video that are uploaded from local storage. By default images/videos are stroed as base64. */
export declare type EditorImageFormat = 'base64' | 'blob';
/**Determines the format of the content that will be pasted inside the Editor. */
export declare type PasteFormat = 'prompt' | 'plainText' | 'keepFormat' | 'cleanFormat';
/**Determines the toolbar mode of the Editor. The main toolbar of the Editor can appear as a Ribbon or as a Menu. */
export declare type ToolbarMode = 'menu' | 'singleLineRibbon';
/**Determines the format of the content that will be pasted inside the Editor. */
export declare type ToolbarViewMode = 'toggle' | 'multiRow' | 'scroll';
export interface ElementProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Enables or disables the accordion. Disabled elements can not be interacted with.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Returns true when the rendering is completed. Otherwise returns false
   * Default value: false
   */
  isRendered?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "accordionItemRequired": "' requires an item from type \"jqx-accordion-item\".",
   *     "indexOutOfBound": "' method.",
   *     "invalidSettings": "' method accepts a string or an object as it's second parameter.",
   *     "noItems": ": No child elements found.",
   *     "overridingProperties": "' property is used by default."
   *   }
   * }
   */
  messages?: any;
  /**
   * Callback called when the element is attached to the DOM.
   * Default value: null
   */
  onAttached?: any;
  /**
   * Callback called when the element is detached from the DOM.
   * Default value: null
   */
  onDetached?: any;
  /**
   * Callback called when the element is created.
   * Default value: null
   */
  onCreated?: any;
  /**
   * Callback called when the element is attached to the DOM for first time. It can be used for property initialization. 
   * Default value: null
   */
  onReady?: any;
  /**
   * Callback called when the element is rendered. 
   * Default value: null
   */
  onRender?: any;
  /**
   * Determines if the element is readonly or not. If the element true, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Determines whether the element is automatically rendered or is rendered after calling the render method.
   * Default value: auto
   */
  renderMode?: ElementRenderMode | string;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines whether the element can be focused or not.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 BaseElement extending the HTMLElement with localization, on-demand rendering, typed properties, two-way bound properties, property change notifications, lifecycle callbacks, automatic events listen and unlisten.
*/

declare global {
	interface Window {
		Smart: any;
		NG: any;
	}
}

export interface BaseElement extends HTMLElement, ElementProperties {
  /**
   * This event is triggered when the element is resized.
	* @param event. The custom event.    */
  onResize: ((ev: Event) => any) | null;
  /**
   * This event is triggered when a style which affects the layout of the element is changed is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(styleProperties)
   *  styleProperties - Array with changed style properties.
   */
  onStyleChanged?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Unwatches the element.
   */
  unwatch(): void;
  /**
   * Renders the element.
   */
  render(): void;
  /**
   * Watch for property changes.
   * @param {any[]} properties. Properties array to watch for changes.
   * @param {any} propertyChangedCallback. function(propertyName: string, oldValue: any, newValue: any)
   */
  watch(properties: any[], propertyChangedCallback: any): void;
  /**
   * Notifies with a callback function, when the rendering is completed.
   * @param {any} renderCallback. Callback which is called, when the rendering is completed.
   */
  whenRendered(renderCallback: any): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-element"): Element;
        querySelector(selectors: "smart-element"): Element | null;
        querySelectorAll(selectors: "smart-element"): NodeListOf<Element>;
        getElementsByTagName(qualifiedName: "smart-element"): HTMLCollectionOf<Element>;
        getElementsByName(elementName: "smart-element"): NodeListOf<Element>;
    }
}

/**Determines whether the element is automatically rendered or is rendered after calling the render method. */
export declare type ElementRenderMode = 'auto' | 'manual';
export interface FileUploadProperties {
  /**
   * Sets or gets the file types that can be submitted to the server via the element. This property corresponds to the 'accept' attribute of the hidden file input which is submitted to the URL specified by the uploadUrl property.
   * Default value: null
   */
  accept?: string | null;
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Appends the list with selected files to a new custom container specified by the user. If the value of the property is a string it must represent a valid id of an HTML element inside the DOM that will be used as the new container for the uploaded files list.
   * Default value: "null"
   */
  appendTo?: string;
  /**
   * Sets or gets whether files will be automatically uploaded after selection.
   * Default value: false
   */
  autoUpload?: boolean;
  /**
   * Allows to upload a directory. Files in all subfolders will be uploaded also. This option is supported only in Firefox and Chrome.
   * Default value: false
   */
  directory?: boolean;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Defines a custom container that will be used as the new drop zone for file uploads. The dropped files will be added in the fileUpload's list. If 'dropZone' property set to true, the default drop zone inside the element will be used instead. If set to certain id of an HTML element inside the DOM then it will be used as the drop zone.
   * Default value: null
   */
  dropZone?: any;
  /**
   * Hides the footer element and it's contents (Upload All, Pause All and Close All buttons).
   * Default value: false
   */
  hideFooter?: boolean;
  /**
   * Applies a custom template to the file items that represent the uploaded files.
   * Default value: null
   */
  itemTemplate?: any;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets the various text values used in the widget. Useful for localization. The localization object has the following fields: browse, pauseFile, cancelFile, uploadFile, pauseAll, cancelAll, uploadAll. It's recommended these messages to be set before element's initialization.
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "browse": "Browse",
   *     "uploadFile": "Upload File",
   *     "cancelFile": "Cancel File",
   *     "pauseFile": "Pause File",
   *     "uploadAll": "Upload All",
   *     "cancelAll": "Cancel All",
   *     "pauseAll": "Pause All",
   *     "totalFiles": "Total files: ",
   *     "connectionError": ": File Upload requires connection to the server.",
   *     "wrongItemIndex": ": There is no file with such an index in the list of uploaded files.",
   *     "tooLongFileName": ": File name is too long."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets whether multiple item uploads are allowed.
   * Default value: false
   */
  multiple?: boolean;
  /**
   * Sets or gets the name attribute of the hidden file input which is submitted to the URL specified by the uploadUrl property.
   * Default value: ""
   */
  name?: string;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Callback that can used to handle various server responses and error codes.
   * Default value: null
   */
  responseHandler?: any;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Callback function, used to change the headers of the file upload's XHR request.
   * Default value: null
   */
  setHeaders?: any;
  /**
   * Displays a progress bar at the bottom of each uploaded item to show the progress of the uploading process.
   * Default value: false
   */
  showProgress?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets or gets the upload URL. This property corresponds to the upload form's action attribute. For example, the uploadUrl property can point to a PHP file, which handles the upload operation on the server-side.
   * Default value: ""
   */
  uploadUrl?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the remove URL. This property corresponds to the form's action attribute. For example, the removeUrl property can point to a PHP file, which handles the remove operation on the server-side.
   * Default value: ""
   */
  removeUrl?: string;
  /**
   * Gets the file upload value.
   * Default value: null
   */
  value?: any;
  /**
   * Callback used to validate the files immediatelly after their selection. Retuns a boolean value. If the returned value is false, the file is removed from list and a 'validationError is fired.
   * Default value: null
   */
  validateFile?: any;
}
/**
 FileUpload provides an easy and integrated way for users to upload multiple files.
*/
export interface FileUpload extends BaseElement, FileUploadProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when a file has been selected.
	* @param event. The custom event. Custom data event was created with: ev.detail(filename, type, size, index)
   *  filename - The name of the selected file.
   *  type - The type of the selected file.
   *  size - The size of the selected file.
   *  index - The index of the selected file.
   */
  onFileSelected?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a file upload operation is canceled.
	* @param event. The custom event. Custom data event was created with: ev.detail(filename, type, size, index)
   *  filename - The name of the canceled file.
   *  type - The type of the canceled file.
   *  size - The size of the canceled file.
   *  index - The index of the canceled file.
   */
  onUploadCanceled?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a file upload operation is completed.
	* @param event. The custom event. Custom data event was created with: ev.detail(filename, type, size, index, status, serverResponse)
   *  filename - The name of the canceled file.
   *  type - The type of the canceled file.
   *  size - The size of the canceled file.
   *  index - The index of the canceled file.
   *  status - The status of the uploaded file. Whether there was an error or success.
   *  serverResponse - The response of the remote server.
   */
  onUploadCompleted?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when during the file upload process something happens and upload fails.
	* @param event. The custom event. Custom data event was created with: ev.detail(filename, type, size, index, status)
   *  filename - The name of the canceled file.
   *  type - The type of the canceled file.
   *  size - The size of the canceled file.
   *  index - The index of the canceled file.
   *  status - The status of the uploaded file. Whether there was an error or success.
   */
  onUploadError?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a file upload operation is paused.
	* @param event. The custom event. Custom data event was created with: ev.detail(filename, type, size, index)
   *  filename - The name of the paused file.
   *  type - The type of the paused file.
   *  size - The size of the paused file.
   *  index - The index of the paused file.
   */
  onUploadPaused?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a file upload operation is started.
	* @param event. The custom event. Custom data event was created with: ev.detail(filename, type, size, index)
   *  filename - The name of the file that is being uploaded.
   *  type - The type of the file that is being uploaded.
   *  size - The size of the file that is being uploaded.
   *  index - The index of the file that is being uploaded.
   */
  onUploadStarted?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered if the validation of a user defined 'validateFile' callback fails.
	* @param event. The custom event. Custom data event was created with: ev.detail(filename, type, size)
   *  filename - The name of the file which validation has failed.
   *  type - The type of the file which validation has failed.
   *  size - The size of the file which validation has failed.
   */
  onValidationError?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Opens a popup to browse for a file.
   */
  browse(): void;
  /**
   * Cancels all selected files. The files are removed from the list and their uploading is prevented.
   */
  cancelAll(): void;
  /**
   * Cancels a selected file. The file is removed from the file list and it's uploading is prevented.
   * @param {number} index. Index of the file which will be canceled.
   */
  cancelFile(index: number): void;
  /**
   * Pauses the uploading of all files. File upload is prevented but the files remain in the file list.
   */
  pauseAll(): void;
  /**
   * Pauses upload of a file with particular index. File upload is prevented but file ramains in the file list.
   * @param {number} id. Index of the file which will be paused.
   */
  pauseFile(id: number): void;
  /**
   * Uploads all selected files.
   */
  uploadAll(): void;
  /**
   * Uploads a selected file.
   * @param {number} id. Index of the file which will be uploaded.
   */
  uploadFile(id: number): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-file-upload"): FileUpload;
        querySelector(selectors: "smart-file-upload"): FileUpload | null;
        querySelectorAll(selectors: "smart-file-upload"): NodeListOf<FileUpload>;
        getElementsByTagName(qualifiedName: "smart-file-upload"): HTMLCollectionOf<FileUpload>;
        getElementsByName(elementName: "smart-file-upload"): NodeListOf<FileUpload>;
    }
}

export interface FilterBuilderProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Adds more operations, that can be used in the filter bilder's conditions structure.
   * Default value: 
   */
  customOperations?: {name: string, caption: string, icon: string}[];
  /**
   * Enables or disables the context menu.
   * Default value: false
   */
  disableContextMenu?: boolean;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Array with filtered fields and their settings.
   * Default value: null
   */
  fields?: any;
  /**
   * Sets or gets the format string of the editor of fields with type 'date'.
   * Default value: "dd-MMM-yy"
   */
  formatStringDate?: string;
  /**
   * Sets or gets the format string of the editor of fields with type 'datetime'.
   * Default value: "dd-MMM-yy HH:mm:ss"
   */
  formatStringDateTime?: string;
  /**
   * Sets hint in form of popup message.
   * Default value: "null"
   */
  hint?: string;
  /**
   * Defines icon's representatino as characters.
   * Default value: { '=': '=', '<>': '<>', '>': '>', '>=': '>=', '<': '<', '<=': '<=', 'startswith': 'a|bc', 'endswith': 'ab|c', 'contains': 'abc', 'notcontains': '!abc', 'isblank': 'o', 'isnotblank': '*'}
   */
  icons?: FilterBuilderIcons;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback, related to localization module. 
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Defines maximum number of allowed conditions in the element.
   * Default value: null
   */
  maxConditions?: number | null;
  /**
   * Defines maximum number of allowed conditions in group.
   * Default value: null
   */
  maxConditionsPerGroup?: number | null;
  /**
   * Defines maximum level of grouping in the FilterBuilder.
   * Default value: null
   */
  maxLevel?: number | null;
  /**
   * Defines field names of the filtered element.
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "addCondition": "Add Condition",
   *     "addGroup": "Add Group",
   *     "and": "And",
   *     "notand": "Not And",
   *     "or": "Or",
   *     "notor": "Not Or",
   *     "=": "Equals",
   *     "<>": "Does not equal",
   *     ">": "Greater than",
   *     ">=": "Greater than or equal to",
   *     "<": "Less than",
   *     "<=": "Less than or equal to",
   *     "startswith": "Starts with",
   *     "endswith": "Ends with",
   *     "contains": "Contains",
   *     "notcontains": "Does not contain",
   *     "isblank": "Is blank",
   *     "isnotblank": "Is not blank",
   *     "wrongParentGroupIndex": "' method.",
   *     "missingFields": ": Fields are required for proper condition's adding. Set \"fields\" source and then conditions will be added as expected.",
   *     "wrongElementNode": "' method.",
   *     "invalidDataStructure": ": Used invalid data structure in updateCondition/updateGroup method."
   *   }
   * }
   */
  messages?: any;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * In restrictedMode set to true, adding of new groups and groups editing by user interaction are denied.
   * Default value: false
   */
  restrictedMode?: boolean;
  /**
   * Enables or disables the display of the icons.
   * Default value: false
   */
  showIcons?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * The value is represented by multidimensional array. The array contains group operators and conditions. Each group can contain nested structures at multiple levels.
   * Default value: or
   */
  value?: string[];
  /**
   * Callback used to format the content of the value fields.
   * Default value: null
   */
  valueFormatFunction?: any;
  /**
   * Sets the placeholder text used in the condition's value box in case the value is not set.
   * Default value: "&lt;enter a value&gt;"
   */
  valuePlaceholder?: string;
}
/**
 FilterBuilder allows you to dynamically build filters.
*/
export interface FilterBuilder extends BaseElement, FilterBuilderProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the element's value is changed.
	* @param event. The custom event.    */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when an editor is closed.
	* @param event. The custom event.    */
  onEditorClose?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an editor starts to close.
	* @param event. The custom event.    */
  onEditorClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an editor is opened.
	* @param event. The custom event.    */
  onEditorOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an editor starts to open.
	* @param event. The custom event.    */
  onEditorOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when some of the filterbuilder's building blocks is clicked.
	* @param event. The custom event.    */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the built-in menu is opened. If the <strong>disableContextMenu</strong> property is set to true this event is not fired.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the built-in menu starts to open. If the <strong>disableContextMenu</strong> property is set to true this event is not fired.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the built-in menu is closed. If the <strong>disableContextMenu</strong> property is set to true this event is not fired.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the built-in menu  starts to close. If the <strong>disableContextMenu</strong> property is set to true this event is not fired.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Adds new condition in particular group. 
   * @param {string | HTMLElement} parentGroup. A string, representing the id of the item or an HTML Element referencing this condition.
   * @param {any[]} data. An array, representing condition. Condition's element's role is related to their position in the condition's array. At index 0 - field name, index 1 - condition's opertor, index 2 - condition's value.
   */
  addCondition(parentGroup: string | HTMLElement, data: any[]): void;
  /**
   * Adds new group in particular parent group.
   * @param {string | HTMLElement} parentGroup. A string, representing the id of the item or an HTML Element referencing this group.
   * @param {string} data. A string, representing the group operator.
   */
  addGroup(parentGroup: string | HTMLElement, data: string): void;
  /**
   * Removes condition.
   * @param {string | HTMLElement} item. A string, representing the id of the item or an HTML Element referencing this condition.
   */
  removeCondition(item: string | HTMLElement): void;
  /**
   * Deletes group and all of  it's children
   * @param {string | HTMLElement} item. A string, representing the id of the item or an HTML Element referencing this group.
   */
  removeGroup(item: string | HTMLElement): void;
  /**
   * Returns a string representation of the filter builder's value.
   * @param {boolean} useLabels?. Controls the way of string representation. In mode without labels the value object is stringified only.
   * @returns {string}
   */
  toString(useLabels?: boolean): string;
  /**
   * Updates condition.
   * @param {string | HTMLElement} item. A string, representing the id of the item or an HTML Element referencing targeted condition.
   * @param {any[]} data. An array, representing condition. Condition's element's role is related to their position in the condition's array. At index 0 - field name, index 1 - condition's opertor, index 2 - condition's value.
   */
  updateCondition(item: string | HTMLElement, data: any[]): void;
  /**
   * Updates group
   * @param {string | HTMLElement} item. A string, representing the id of the item or an HTML Element referencing this group.
   * @param {string} data. A string, representing the group operator.
   */
  updateGroup(item: string | HTMLElement, data: string): void;
}

/**Defines icon's representatino as characters. */
export interface FilterBuilderIcons {
  /**
   * 
   * Default value: undefined
   */
  0?: any;
  /**
   * 
   * Default value: undefined
   */
  1?: any;
  /**
   * 
   * Default value: undefined
   */
  2?: any;
  /**
   * 
   * Default value: undefined
   */
  3?: any;
  /**
   * 
   * Default value: undefined
   */
  4?: any;
  /**
   * 
   * Default value: undefined
   */
  5?: any;
  /**
   * 
   * Default value: undefined
   */
  6?: any;
  /**
   * 
   * Default value: undefined
   */
  7?: any;
  /**
   * 
   * Default value: undefined
   */
  8?: any;
  /**
   * 
   * Default value: undefined
   */
  9?: any;
  /**
   * 
   * Default value: undefined
   */
  10?: any;
  /**
   * 
   * Default value: undefined
   */
  11?: any;
  /**
   * 
   * Default value: undefined
   */
  12?: any;
  /**
   * 
   * Default value: undefined
   */
  13?: any;
  /**
   * 
   * Default value: undefined
   */
  14?: any;
  /**
   * 
   * Default value: undefined
   */
  15?: any;
  /**
   * 
   * Default value: undefined
   */
  16?: any;
  /**
   * 
   * Default value: undefined
   */
  17?: any;
  /**
   * 
   * Default value: undefined
   */
  18?: any;
  /**
   * 
   * Default value: undefined
   */
  19?: any;
  /**
   * 
   * Default value: undefined
   */
  20?: any;
  /**
   * 
   * Default value: undefined
   */
  21?: any;
  /**
   * 
   * Default value: undefined
   */
  22?: any;
  /**
   * 
   * Default value: undefined
   */
  23?: any;
  /**
   * 
   * Default value: undefined
   */
  24?: any;
  /**
   * 
   * Default value: undefined
   */
  25?: any;
  /**
   * 
   * Default value: undefined
   */
  26?: any;
  /**
   * 
   * Default value: undefined
   */
  27?: any;
  /**
   * 
   * Default value: undefined
   */
  28?: any;
  /**
   * 
   * Default value: undefined
   */
  29?: any;
  /**
   * 
   * Default value: undefined
   */
  30?: any;
  /**
   * 
   * Default value: undefined
   */
  31?: any;
  /**
   * 
   * Default value: undefined
   */
  32?: any;
  /**
   * 
   * Default value: undefined
   */
  33?: any;
  /**
   * 
   * Default value: undefined
   */
  34?: any;
  /**
   * 
   * Default value: undefined
   */
  35?: any;
  /**
   * 
   * Default value: undefined
   */
  36?: any;
  /**
   * 
   * Default value: undefined
   */
  37?: any;
  /**
   * 
   * Default value: undefined
   */
  38?: any;
  /**
   * 
   * Default value: undefined
   */
  39?: any;
  /**
   * 
   * Default value: undefined
   */
  40?: any;
  /**
   * 
   * Default value: undefined
   */
  41?: any;
  /**
   * 
   * Default value: undefined
   */
  42?: any;
  /**
   * 
   * Default value: undefined
   */
  43?: any;
  /**
   * 
   * Default value: undefined
   */
  44?: any;
  /**
   * 
   * Default value: undefined
   */
  45?: any;
  /**
   * 
   * Default value: undefined
   */
  46?: any;
  /**
   * 
   * Default value: undefined
   */
  47?: any;
  /**
   * 
   * Default value: undefined
   */
  48?: any;
  /**
   * 
   * Default value: undefined
   */
  49?: any;
  /**
   * 
   * Default value: undefined
   */
  50?: any;
  /**
   * 
   * Default value: undefined
   */
  51?: any;
  /**
   * 
   * Default value: undefined
   */
  52?: any;
  /**
   * 
   * Default value: undefined
   */
  53?: any;
  /**
   * 
   * Default value: undefined
   */
  54?: any;
  /**
   * 
   * Default value: undefined
   */
  55?: any;
  /**
   * 
   * Default value: undefined
   */
  56?: any;
  /**
   * 
   * Default value: undefined
   */
  57?: any;
  /**
   * 
   * Default value: undefined
   */
  58?: any;
  /**
   * 
   * Default value: undefined
   */
  59?: any;
  /**
   * 
   * Default value: undefined
   */
  60?: any;
  /**
   * 
   * Default value: undefined
   */
  61?: any;
  /**
   * 
   * Default value: undefined
   */
  62?: any;
  /**
   * 
   * Default value: undefined
   */
  63?: any;
  /**
   * 
   * Default value: undefined
   */
  64?: any;
  /**
   * 
   * Default value: undefined
   */
  65?: any;
  /**
   * 
   * Default value: undefined
   */
  66?: any;
  /**
   * 
   * Default value: undefined
   */
  67?: any;
  /**
   * 
   * Default value: undefined
   */
  68?: any;
  /**
   * 
   * Default value: undefined
   */
  69?: any;
  /**
   * 
   * Default value: undefined
   */
  70?: any;
  /**
   * 
   * Default value: undefined
   */
  71?: any;
  /**
   * 
   * Default value: undefined
   */
  72?: any;
  /**
   * 
   * Default value: undefined
   */
  73?: any;
  /**
   * 
   * Default value: undefined
   */
  74?: any;
  /**
   * 
   * Default value: undefined
   */
  75?: any;
  /**
   * 
   * Default value: undefined
   */
  76?: any;
  /**
   * 
   * Default value: undefined
   */
  77?: any;
  /**
   * 
   * Default value: undefined
   */
  78?: any;
  /**
   * 
   * Default value: undefined
   */
  79?: any;
  /**
   * 
   * Default value: undefined
   */
  80?: any;
  /**
   * 
   * Default value: undefined
   */
  81?: any;
  /**
   * 
   * Default value: undefined
   */
  82?: any;
  /**
   * 
   * Default value: undefined
   */
  83?: any;
  /**
   * 
   * Default value: undefined
   */
  84?: any;
  /**
   * 
   * Default value: undefined
   */
  85?: any;
  /**
   * 
   * Default value: undefined
   */
  86?: any;
  /**
   * 
   * Default value: undefined
   */
  87?: any;
  /**
   * 
   * Default value: undefined
   */
  88?: any;
  /**
   * 
   * Default value: undefined
   */
  89?: any;
  /**
   * 
   * Default value: undefined
   */
  90?: any;
  /**
   * 
   * Default value: undefined
   */
  91?: any;
  /**
   * 
   * Default value: undefined
   */
  92?: any;
  /**
   * 
   * Default value: undefined
   */
  93?: any;
  /**
   * 
   * Default value: undefined
   */
  94?: any;
  /**
   * 
   * Default value: undefined
   */
  95?: any;
  /**
   * 
   * Default value: undefined
   */
  96?: any;
  /**
   * 
   * Default value: undefined
   */
  97?: any;
  /**
   * 
   * Default value: undefined
   */
  98?: any;
  /**
   * 
   * Default value: undefined
   */
  99?: any;
  /**
   * 
   * Default value: undefined
   */
  100?: any;
  /**
   * 
   * Default value: undefined
   */
  101?: any;
  /**
   * 
   * Default value: undefined
   */
  102?: any;
  /**
   * 
   * Default value: undefined
   */
  103?: any;
  /**
   * 
   * Default value: undefined
   */
  104?: any;
  /**
   * 
   * Default value: undefined
   */
  105?: any;
  /**
   * 
   * Default value: undefined
   */
  106?: any;
  /**
   * 
   * Default value: undefined
   */
  107?: any;
  /**
   * 
   * Default value: undefined
   */
  108?: any;
  /**
   * 
   * Default value: undefined
   */
  109?: any;
  /**
   * 
   * Default value: undefined
   */
  110?: any;
  /**
   * 
   * Default value: undefined
   */
  111?: any;
  /**
   * 
   * Default value: undefined
   */
  112?: any;
  /**
   * 
   * Default value: undefined
   */
  113?: any;
  /**
   * 
   * Default value: undefined
   */
  114?: any;
  /**
   * 
   * Default value: undefined
   */
  115?: any;
  /**
   * 
   * Default value: undefined
   */
  116?: any;
  /**
   * 
   * Default value: undefined
   */
  117?: any;
  /**
   * 
   * Default value: undefined
   */
  118?: any;
  /**
   * 
   * Default value: undefined
   */
  119?: any;
  /**
   * 
   * Default value: undefined
   */
  120?: any;
  /**
   * 
   * Default value: undefined
   */
  121?: any;
  /**
   * 
   * Default value: undefined
   */
  122?: any;
  /**
   * 
   * Default value: undefined
   */
  123?: any;
  /**
   * 
   * Default value: undefined
   */
  124?: any;
  /**
   * 
   * Default value: undefined
   */
  125?: any;
  /**
   * 
   * Default value: undefined
   */
  126?: any;
  /**
   * 
   * Default value: undefined
   */
  127?: any;
  /**
   * 
   * Default value: undefined
   */
  128?: any;
  /**
   * 
   * Default value: undefined
   */
  129?: any;
  /**
   * 
   * Default value: undefined
   */
  130?: any;
  /**
   * 
   * Default value: undefined
   */
  131?: any;
  /**
   * 
   * Default value: undefined
   */
  132?: any;
  /**
   * 
   * Default value: undefined
   */
  133?: any;
  /**
   * 
   * Default value: undefined
   */
  134?: any;
  /**
   * 
   * Default value: undefined
   */
  135?: any;
  /**
   * 
   * Default value: undefined
   */
  136?: any;
  /**
   * 
   * Default value: undefined
   */
  137?: any;
  /**
   * 
   * Default value: undefined
   */
  138?: any;
  /**
   * 
   * Default value: undefined
   */
  139?: any;
  /**
   * 
   * Default value: undefined
   */
  140?: any;
  /**
   * 
   * Default value: undefined
   */
  141?: any;
  /**
   * 
   * Default value: undefined
   */
  142?: any;
  /**
   * 
   * Default value: undefined
   */
  143?: any;
  /**
   * 
   * Default value: undefined
   */
  144?: any;
  /**
   * 
   * Default value: undefined
   */
  145?: any;
  /**
   * 
   * Default value: undefined
   */
  146?: any;
  /**
   * 
   * Default value: undefined
   */
  147?: any;
  /**
   * 
   * Default value: undefined
   */
  148?: any;
  /**
   * 
   * Default value: undefined
   */
  149?: any;
  /**
   * 
   * Default value: undefined
   */
  150?: any;
  /**
   * 
   * Default value: undefined
   */
  151?: any;
  /**
   * 
   * Default value: undefined
   */
  152?: any;
  /**
   * 
   * Default value: undefined
   */
  153?: any;
  /**
   * 
   * Default value: undefined
   */
  154?: any;
  /**
   * 
   * Default value: undefined
   */
  155?: any;
  /**
   * 
   * Default value: undefined
   */
  156?: any;
  /**
   * 
   * Default value: undefined
   */
  157?: any;
  /**
   * 
   * Default value: undefined
   */
  158?: any;
  /**
   * 
   * Default value: undefined
   */
  159?: any;
  /**
   * 
   * Default value: undefined
   */
  160?: any;
  /**
   * 
   * Default value: undefined
   */
  161?: any;
  /**
   * 
   * Default value: undefined
   */
  162?: any;
  /**
   * 
   * Default value: undefined
   */
  163?: any;
  /**
   * 
   * Default value: undefined
   */
  164?: any;
  /**
   * 
   * Default value: undefined
   */
  165?: any;
  /**
   * 
   * Default value: undefined
   */
  166?: any;
  /**
   * 
   * Default value: undefined
   */
  167?: any;
  /**
   * 
   * Default value: undefined
   */
  168?: any;
  /**
   * 
   * Default value: undefined
   */
  169?: any;
  /**
   * 
   * Default value: undefined
   */
  170?: any;
  /**
   * 
   * Default value: undefined
   */
  171?: any;
  /**
   * 
   * Default value: undefined
   */
  172?: any;
  /**
   * 
   * Default value: undefined
   */
  173?: any;
  /**
   * 
   * Default value: undefined
   */
  174?: any;
  /**
   * 
   * Default value: undefined
   */
  175?: any;
  /**
   * 
   * Default value: undefined
   */
  176?: any;
  /**
   * 
   * Default value: undefined
   */
  177?: any;
  /**
   * 
   * Default value: undefined
   */
  178?: any;
  /**
   * 
   * Default value: undefined
   */
  179?: any;
  /**
   * 
   * Default value: undefined
   */
  180?: any;
  /**
   * 
   * Default value: undefined
   */
  181?: any;
  /**
   * 
   * Default value: undefined
   */
  182?: any;
  /**
   * 
   * Default value: undefined
   */
  183?: any;
  /**
   * 
   * Default value: undefined
   */
  184?: any;
  /**
   * 
   * Default value: undefined
   */
  185?: any;
}

declare global {
    interface Document {
        createElement(tagName: "smart-filter-builder"): FilterBuilder;
        querySelector(selectors: "smart-filter-builder"): FilterBuilder | null;
        querySelectorAll(selectors: "smart-filter-builder"): NodeListOf<FilterBuilder>;
        getElementsByTagName(qualifiedName: "smart-filter-builder"): HTMLCollectionOf<FilterBuilder>;
        getElementsByName(elementName: "smart-filter-builder"): NodeListOf<FilterBuilder>;
    }
}

export interface FilterPanelProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Defines which operation buttons will be shown in the filter panel
   * Default value: cancel,clear,filter
   */
  buttons?: string[];
  /**
   * Array in mode: 'excel', determines the data to extract unique filter values from. The expected format of the data is an array of objects with key-value pairs (JSON array)
   * Default value: null
   */
  data?: number[];
  /**
   * Describes filtered data field.
   * Default value: "null"
   */
  dataField?: string;
  /**
   * If set to an array, determines a custom collection of filter values to be displayed. If set to a callback function, the user can add custom filter values dynamically to the default filter values (in this case, the evaluateFilterExpression function must also be implemented) .dataSource  is only available in mode: 'excel'.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables filter panel.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Callback function, used for custom evaluations in filter panel.
   * Default value: null
   */
  evaluateFilterExpression?: any;
  /**
   * Defines which filter type is used.
   * Default value: string
   */
  filterType?: FilterPanelFilterType | string;
  /**
   * Format string used in filterType 'Date'.
   * Default value: "d"
   */
  formatString?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Defines field names of the filtered element.
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid ''!",
   *     "propertyInvalidValueType": "Invalid ''!",
   *     "methodInvalidValueType": "Invalid ''!",
   *     "methodInvalidArgumentsCount": "Invalid '' argument(s)!",
   *     "methodInvalidReturnType": "Invalid ''!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": "'.",
   *     "htmlTemplateNotSuported": ": Web Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "addCondition": "Add Condition",
   *     "addGroup": "Add Group",
   *     "and": "and",
   *     "blanks": "(Blanks)",
   *     "cancel": "Cancel",
   *     "clear": "Clear",
   *     "contains": "contains",
   *     "containsCaseSensitive": "contains (case sensitive)",
   *     "dateTabLabel": "DATE",
   *     "doesNotContain": "does not contain",
   *     "doesNotContainCaseSensitive": "does not contain (case sensitive)",
   *     "empty": "empty",
   *     "endsWith": "ends with",
   *     "endsWithCaseSensitive": "ends with (case sensitive)",
   *     "equal": "equal",
   *     "equalCaseSensitive": "equal (case sensitive)",
   *     "filter": "Filter",
   *     "greaterThan": "greater than",
   *     "greaterThanOrEqual": "greater than or equal",
   *     "lessThan": "less than",
   *     "lessThanOrEqual": "less than or equal",
   *     "notEmpty": "not empty",
   *     "notEqual": "not equal",
   *     "notNull": "not null",
   *     "null": "null ",
   *     "or": "or",
   *     "placeholderDate": "Enter date",
   *     "placeholderTime": "Enter time",
   *     "selectAll": "(Select All)",
   *     "showRows": "Show rows where:",
   *     "startsWith": "starts with",
   *     "startsWithCaseSensitive": "starts with (case sensitive)",
   *     "timeTabLabel": "TIME"
   *   }
   * }
   */
  messages?: any;
  /**
   * Desfines filter panel's  mode
   * Default value: default
   */
  mode?: FilterPanelMode | string;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 The filter panel allows you to customize and display the applied filter expressions.
*/
export interface FilterPanel extends BaseElement, FilterPanelProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the current filtering is discarted.
	* @param event. The custom event.    */
  onCancel: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the current filtering is cleared.
	* @param event. The custom event.    */
  onClear?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when filter panel settings are applied.
	* @param event. The custom event.    */
  onFilter?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Discards current filtering.
   */
  cancel(): void;
  /**
   * Clears current filtering.
   */
  clear(): void;
  /**
   * Evaluates a filter.
   * @param {any} value. The evalueated element in filter panel.
   */
  evaluate(value: any): void;
  /**
   * Applies current filtering.
   */
  filter(): void;
  /**
   * Gets the current filter state.
   * @returns {any}
   */
  getState(): any;
  /**
   * Loads a previously saved filter state.
   * @param {any} state. An object returned by the method getState.
   */
  loadState(state: any): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-filter-panel"): FilterPanel;
        querySelector(selectors: "smart-filter-panel"): FilterPanel | null;
        querySelectorAll(selectors: "smart-filter-panel"): NodeListOf<FilterPanel>;
        getElementsByTagName(qualifiedName: "smart-filter-panel"): HTMLCollectionOf<FilterPanel>;
        getElementsByName(elementName: "smart-filter-panel"): NodeListOf<FilterPanel>;
    }
}

/**Defines which filter type is used. */
export declare type FilterPanelFilterType = 'numeric' | 'string' | 'date' | 'boolean';
/**Desfines filter panel's  mode */
export declare type FilterPanelMode = 'default' | 'excel';
export interface FormProperties {
  /**
   * Sets or gets the form columns.
   * Default value: 1
   */
  columns?: number;
  /**
   * Sets or gets the form controls.
   * Default value: null
   */
  controls?: Control[];
  /**
   * Callback function for handling status changes. The status could be  'disabled', 'invalid', 'pending', 'valid'.
   * Default value: null
   */
  onStatusChanges?: {(value: string): void};
  /**
   * Callback function for handling value changes. The function argument is the form's value as a JSON object.
   * Default value: null
   */
  onValueChanges?: {(value: any): void};
  /**
   * Sets or Gets the labels position.
   * Default value: left
   */
  labelPosition?: FormLabelPosition | string;
  /**
   * Makes the form readonly.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Shows / hides the colon after the labels.
   * Default value: false
   */
  showColonAfterLabel?: boolean;
  /**
   * Shows / hides validation summary.
   * Default value: true
   */
  showSummary?: boolean;
  /**
   * Gets the Form's state. Each member in the state has { dirty, untouched, disabled } properties.
   * Default value: null
   */
  state?: any;
  /**
   * Gets or Sets the Form value.
   * Default value: null
   */
  value?: any;
  /**
   * Automatically validates the form when it is created.
   * Default value: false
   */
  validateOnLoad?: boolean;
}
/**
 Reactive Form Component with Advanced Validation
*/
export interface Form extends BaseElement, FormProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * Adds a control to the Form.
   * @param {any} controlOptions. Control options. The control options description is available in the <em>controls</em> property.
   */
  addControl(controlOptions: any): void;
  /**
   * Gets a control by its name(dataField).
   * @param {string} dataField. dataField of a FormControl or FormGroup
   * @returns {Control}
   */
  getControl(dataField: string): Control;
  /**
   * Inserts a control to the Form.
   * @param {number} index. Control insert index
   * @param {any} controlOptions. Control options. The control options description is available in the <em>controls</em> property.
   */
  insertControl(index: number, controlOptions: any): void;
  /**
   * Remove a control from the Form.
   * @param {any} controlOptions. Control options. The control options description is available in the <em>controls</em> property.
   */
  removeControl(controlOptions: any): void;
  /**
   * Submits the form.
   * @param {any} submitOptions?. Sets the submit options object. The object may have the following properties: <em>async</em>, <em>action</em>, <em>target</em>, <em>method</em>. <em>async</em> determines whether the form will be submitted asynchronously. <em>action</em> determines the submit url, <em>method</em> sets whether the submit is through 'GET' or 'POST'. <em>target</em> determines the submit target.
   */
  submit(submitOptions?: any): void;
  /**
   * Clears the form.
   */
  reset(): void;
  /**
   * Validates the form.
   */
  validate(): void;
}

export interface Control {
  /**
   * HTML Content displayed after the Form Control
   * Default value: ""
   */
  append?: string;
  /**
   * JSON object with initialization properties of the UI component. Example: { dataSource: ['item 1', 'item 2', 'item 3'] } will set the dataSource property of the Form control.
   * Default value: null
   */
  controlOptions?: any;
  /**
   * The type of the control.
   * Default value: input
   */
  controlType?: ControlControlType | string;
  /**
   * Sets the Form Group columns.
   * Default value: 1
   */
  columns?: number;
  /**
   * Sets the Form control column span.
   * Default value: 1
   */
  columnSpan?: number;
  /**
   * 
   * Default value: null
   */
  controls?: Control[];
  /**
   * Sets the Form control data field. The control's inner input's name is set to the dataField value and in the FormGroup it is accessible through the dataField value.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Sets the Form control disabled mode.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Gets whether the Form control is 'dirty' i.e its value is changed by the user.
   * Default value: false
   */
  dirty?: boolean;
  /**
   * Gets or Sets the Form control's info icon's tooltip.
   * Default value: ""
   */
  info?: string;
  /**
   * Gets whether the Form control is invalid.
   * Default value: false
   */
  invalid?: boolean;
  /**
   * Gets or Sets the Form control's label.
   * Default value: ""
   */
  label?: string;
  /**
   * Gets or Sets the Form control's label position.
   * Default value: left
   */
  labelPosition?: ControlLabelPosition | string;
  /**
   * Gets or Sets the offset between the label and the control.
   * Default value: 10
   */
  labelOffset?: number;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets whether the navigation buttons are displayed. The property has effect when the viewMode property is set.
   * Default value: "left"
   */
  labelAlign?: string;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets the next button label.
   * Default value: "Next"
   */
  nextButtonLabel?: string;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets the back button label.
   * Default value: "Back"
   */
  backButtonLabel?: string;
  /**
   * HTML Content displayed before the Form Control
   * Default value: ""
   */
  prepend?: string;
  /**
   * Gets or Sets the Form control readonly mode.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Gets whether the Form control is not touched by the user. This flag is changed usually on blur, after the user interacted with the Form control
   * Default value: false
   */
  untouched?: boolean;
  /**
   * Gets or Sets the placeholder.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets whether the navigation buttons are displayed. The property has effect when the viewMode property is set.
   * Default value: false
   */
  showButtons?: boolean;
  /**
   * Sets or Gets the Form control or Form group value.
   * Default value: null
   */
  value?: any;
  /**
   * Gets whether the Form control is valid.
   * Default value: false
   */
  valid?: boolean;
  /**
   * Sets or gets the column's validation rules. The expected value is an Array of Objects. Each object should have a 'type' property that can be set to 'required', 'min', 'max', 'minLength', 'maxLength', 'email', 'null', 'requiredTrue', 'minData', 'maxDate', 'pattern'. The 'value' property should be set, too. For validation rule types 'required', 'requiredTrue' and 'null' you can skip the 'value' property. Optional property is 'message', which determines the error message.
   * Default value: null
   */
  validationRules?: [] | null;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets the form'group view mode.
   * Default value: 
   */
  viewMode?: ControlViewMode | string;
}

declare global {
    interface Document {
        createElement(tagName: "smart-form"): Form;
        querySelector(selectors: "smart-form"): Form | null;
        querySelectorAll(selectors: "smart-form"): NodeListOf<Form>;
        getElementsByTagName(qualifiedName: "smart-form"): HTMLCollectionOf<Form>;
        getElementsByName(elementName: "smart-form"): NodeListOf<Form>;
    }
}

/**The type of the control. */
export declare type ControlControlType = 'button' | 'boolean' | 'comboBox' | 'checkInput' | 'datetime' | 'dropDownList' | 'group' | 'input' | 'label' | 'multiInput' | 'multiComboInput' | 'mask' | 'number' | 'password' | 'radioButton' | 'submit' | 'textarea' | 'template';
/**Gets or Sets the Form control's label position. */
export declare type ControlLabelPosition = 'left' | 'top';
/**FormGroup only(when controlType is set to 'group'). Gets or Sets the form'group view mode. */
export declare type ControlViewMode = null | 'accordion' | 'tabs' | 'breadcrumb';
/**Sets or Gets the labels position. */
export declare type FormLabelPosition = 'left' | 'top';
export interface FormControlProperties {
  /**
   * Gets or Sets the FormControl Action. This property is used when the 'controlType' is 'button' or 'submit'
   * Default value: 
   */
  action?: FormControlAction | string;
  /**
   * Sets or Gets the alignment of the FormControl
   * Default value: left
   */
  align?: FormControlAlign | string;
  /**
   * HTML Content displayed after the Form Control
   * Default value: 
   */
  appendHTML?: any;
  /**
   * JSON object with initialization properties of the UI component. Example: { dataSource: ['item 1', 'item 2', 'item 3'] } will set the dataSource property of the Form control.
   * Default value: 
   */
  controlOptions?: any;
  /**
   * The type of the control.
   * Default value: input
   */
  controlType?: FormControlControlType | string;
  /**
   * Sets the Form Group columns.
   * Default value: 1
   */
  columns?: number;
  /**
   * Sets the Form control column span.
   * Default value: 1
   */
  columnSpan?: number;
  /**
   * Sets the Form control data field. The control's inner input's name is set to the dataField value and in the FormGroup it is accessible through the dataField value.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Sets the Form control disabled mode.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Gets whether the Form control is 'dirty' i.e its value is changed by the user.
   * Default value: false
   */
  dirty?: boolean;
  /**
   * Gets or Sets the Form control's info icon's tooltip.
   * Default value: ""
   */
  info?: string;
  /**
   * Gets whether the Form control is invalid.
   * Default value: false
   */
  invalid?: boolean;
  /**
   * Gets or Sets the Form control's label.
   * Default value: ""
   */
  label?: string;
  /**
   * Gets or Sets the Form control's label position.
   * Default value: left
   */
  labelPosition?: FormControlLabelPosition | string;
  /**
   * Gets or Sets the offset between the label and the control.
   * Default value: 10
   */
  labelOffset?: number;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets whether the navigation buttons are displayed. The property has effect when the viewMode property is set.
   * Default value: left
   */
  labelAlign?: FormControlAlign | string;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets the next button label.
   * Default value: "Next"
   */
  nextButtonLabel?: string;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets the back button label.
   * Default value: "Back"
   */
  backButtonLabel?: string;
  /**
   * Gets or Sets the FormControl placeholder.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * HTML Content displayed before the Form Control
   * Default value: 
   */
  prependHTML?: any;
  /**
   * Gets or Sets the Form control readonly mode.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Gets or Sets whether this field is required.
   * Default value: false
   */
  required?: boolean;
  /**
   * Gets whether the Form control is not touched by the user. This flag is changed usually on blur, after the user interacted with the Form control
   * Default value: false
   */
  untouched?: boolean;
  /**
   * Gets or Sets whether colon is displayed after the label.
   * Default value: false
   */
  showColonAfterLabel?: boolean;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets whether the navigation buttons are displayed. The property has effect when the viewMode property is set.
   * Default value: false
   */
  showButtons?: boolean;
  /**
   * Sets or Gets the Form control or Form group value.
   * Default value: null
   */
  value?: any;
  /**
   * Gets whether the Form control is valid.
   * Default value: false
   */
  valid?: boolean;
  /**
   * Sets or gets the column's validation rules. The expected value is an Array of Objects. Each object should have a 'type' property that can be set to 'required', 'min', 'max', 'minLength', 'maxLength', 'email', 'null', 'requiredTrue', 'minData', 'maxDate', 'pattern'. The 'value' property should be set, too. For validation rule types 'required', 'requiredTrue' and 'null' you can skip the 'value' property. Optional property is 'message', which determines the error message.
   * Default value: null
   */
  validationRules?: [] | null;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets the form'group view mode.
   * Default value: 
   */
  viewMode?: FormControlViewMode | string;
}
/**
 Form Control
*/
export interface FormControl extends BaseElement, FormControlProperties {

  /* Get a member by its name */
  [name: string]: any;
}

declare global {
    interface Document {
        createElement(tagName: "smart-form-control"): FormControl;
        querySelector(selectors: "smart-form-control"): FormControl | null;
        querySelectorAll(selectors: "smart-form-control"): NodeListOf<FormControl>;
        getElementsByTagName(qualifiedName: "smart-form-control"): HTMLCollectionOf<FormControl>;
        getElementsByName(elementName: "smart-form-control"): NodeListOf<FormControl>;
    }
}

/**Gets or Sets the FormControl Action. This property is used when the 'controlType' is 'button' or 'submit' */
export declare type FormControlAction = 'reset' | 'submit';
/**FormGroup only(when controlType is set to 'group'). Gets or Sets whether the navigation buttons are displayed. The property has effect when the viewMode property is set. */
export declare type FormControlAlign = 'left' | 'center' | 'right';
/**The type of the control. */
export declare type FormControlControlType = 'button' | 'boolean' | 'comboBox' | 'checkInput' | 'datetime' | 'dropDownList' | 'group' | 'input' | 'label' | 'multiInput' | 'multiComboInput' | 'mask' | 'number' | 'password' | 'radioButton' | 'submit' | 'textarea' | 'template';
/**Gets or Sets the Form control's label position. */
export declare type FormControlLabelPosition = 'left' | 'top';
/**FormGroup only(when controlType is set to 'group'). Gets or Sets the form'group view mode. */
export declare type FormControlViewMode = null | 'accordion' | 'tabs' | 'breadcrumb';
export interface FormGroupProperties {
  /**
   * Sets or gets the form columns.
   * Default value: 1
   */
  columns?: number;
  /**
   * Sets the Form control data field. The control's inner input's name is set to the dataField value and in the FormGroup it is accessible through the dataField value.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Gets or Sets the Form control's label.
   * Default value: ""
   */
  label?: string;
  /**
   * 
   * Default value: null
   */
  controls?: Control[];
  /**
   * Callback function for handling status changes. The function argument is a string which could be: 'valid', 'invalid', 'disabled', 'pending'
   * Default value: null
   */
  onStatusChanges?: {(value: string): void};
  /**
   * Callback function for handling value changes. The function argument is the form's value as a JSON object.
   * Default value: null
   */
  onValueChanges?: {(value: any): void};
  /**
   * Sets or Gets the labels position.
   * Default value: left
   */
  labelPosition?: FormGroupLabelPosition | string;
  /**
   * Makes the form readonly.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Shows / hides the colon after the labels.
   * Default value: false
   */
  showColonAfterLabel?: boolean;
  /**
   * Shows / hides validation summary.
   * Default value: true
   */
  showSummary?: boolean;
  /**
   * Gets or Sets the Form value.
   * Default value: null
   */
  value?: any;
}
/**
 Form Group
*/
export interface FormGroup extends BaseElement, FormGroupProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * Adds a control to the Form.
   * @param {any} controlOptions. Control options. The control options description is available in the <em>controls</em> property.
   */
  addControl(controlOptions: any): void;
  /**
   * Gets a control by its name(dataField).
   * @param {string} dataField. dataField of a FormControl or FormGroup
   * @returns {Control}
   */
  getControl(dataField: string): Control;
  /**
   * Inserts a control to the Form.
   * @param {number} index. Control insert index
   * @param {any} controlOptions. Control options. The control options description is available in the <em>controls</em> property.
   */
  insertControl(index: number, controlOptions: any): void;
  /**
   * Remove a control from the Form.
   * @param {any} controlOptions. Control options. The control options description is available in the <em>controls</em> property.
   */
  removeControl(controlOptions: any): void;
}

export interface Control {
  /**
   * HTML Content displayed after the Form Control
   * Default value: 
   */
  appendHTML?: any;
  /**
   * JSON object with initialization properties of the UI component. Example: { dataSource: ['item 1', 'item 2', 'item 3'] } will set the dataSource property of the Form control.
   * Default value: 
   */
  controlOptions?: any;
  /**
   * The type of the control.
   * Default value: input
   */
  controlType?: ControlControlType | string;
  /**
   * Sets the Form Group columns.
   * Default value: 1
   */
  columns?: number;
  /**
   * Sets the Form control column span.
   * Default value: 1
   */
  columnSpan?: number;
  /**
   * Sets the Form control data field. The control's inner input's name is set to the dataField value and in the FormGroup it is accessible through the dataField value.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Sets the Form control disabled mode.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Gets whether the Form control is 'dirty' i.e its value is changed by the user.
   * Default value: false
   */
  dirty?: boolean;
  /**
   * Gets or Sets the Form control's info icon's tooltip.
   * Default value: ""
   */
  info?: string;
  /**
   * Gets whether the Form control is invalid.
   * Default value: false
   */
  invalid?: boolean;
  /**
   * Gets or Sets the Form control's label.
   * Default value: ""
   */
  label?: string;
  /**
   * Gets or Sets the Form control's label position.
   * Default value: left
   */
  labelPosition?: ControlLabelPosition | string;
  /**
   * Gets or Sets the offset between the label and the control.
   * Default value: 10
   */
  labelOffset?: number;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets whether the navigation buttons are displayed. The property has effect when the viewMode property is set.
   * Default value: "left"
   */
  labelAlign?: string;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets the next button label.
   * Default value: "Next"
   */
  nextButtonLabel?: string;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets the back button label.
   * Default value: "Back"
   */
  backButtonLabel?: string;
  /**
   * HTML Content displayed before the Form Control
   * Default value: 
   */
  prependHTML?: any;
  /**
   * Gets or Sets the Form control readonly mode.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Gets whether the Form control is not touched by the user. This flag is changed usually on blur, after the user interacted with the Form control
   * Default value: false
   */
  untouched?: boolean;
  /**
   * Gets or Sets whether colon is displayed after the label.
   * Default value: false
   */
  showColonAfterLabel?: boolean;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets whether the navigation buttons are displayed. The property has effect when the viewMode property is set.
   * Default value: false
   */
  showButtons?: boolean;
  /**
   * Sets or Gets the Form control or Form group value.
   * Default value: null
   */
  value?: any;
  /**
   * Gets whether the Form control is valid.
   * Default value: false
   */
  valid?: boolean;
  /**
   * Sets or gets the column's validation rules. The expected value is an Array of Objects. Each object should have a 'type' property that can be set to 'required', 'min', 'max', 'minLength', 'maxLength', 'email', 'null', 'requiredTrue', 'minData', 'maxDate', 'pattern'. The 'value' property should be set, too. For validation rule types 'required', 'requiredTrue' and 'null' you can skip the 'value' property. Optional property is 'message', which determines the error message.
   * Default value: null
   */
  validationRules?: [] | null;
  /**
   * FormGroup only(when controlType is set to 'group'). Gets or Sets the form'group view mode.
   * Default value: 
   */
  viewMode?: ControlViewMode | string;
}

declare global {
    interface Document {
        createElement(tagName: "smart-form-group"): FormGroup;
        querySelector(selectors: "smart-form-group"): FormGroup | null;
        querySelectorAll(selectors: "smart-form-group"): NodeListOf<FormGroup>;
        getElementsByTagName(qualifiedName: "smart-form-group"): HTMLCollectionOf<FormGroup>;
        getElementsByName(elementName: "smart-form-group"): NodeListOf<FormGroup>;
    }
}

/**Sets or Gets the labels position. */
export declare type FormGroupLabelPosition = 'left' | 'top';
export interface GanttChartProperties {
  /**
   * Determines whether nonworkingDays/nonworkingHours are taken into considuration when determining the dateEnd of the tasks. When this property is enabled the dateEnd of the tasks is calculated to include only the actual working time. By default it's disabled and only calendar time is used.
   * Default value: false
   */
  adjustToNonworkingTime?: boolean;
  /**
   * Recalculates the tasks that are connected and re-schedules them according to their connections. If no connections are present, autoScheduling has no effect until a connection is created. Connection types determines the start/end date limits of the tasks.
   * Default value: false
   */
  autoSchedule?: boolean;
  /**
   * Affects the tasks only when autoSchedule is enabled. When set to true, the tasks are strictly scheduled ( according to their connections ) and dragging is not allowed.  Users can set lag to specific connections to determine a delay of overlap of between the two tasks ( interval of time in miliseconds ). Lag one of the attributes of a task connection and should be set in the dataSource where the task connections are defined.
   * Default value: false
   */
  autoScheduleStrictMode?: boolean;
  /**
   * Determines the scroll speed when dragging when autoScroll property is set. 
   * Default value: 5
   */
  autoScrollStep?: number;
  /**
   * Enabled/Disables the colummn header menu. When enabled and the user hovers a column header, a drop down button will appear that triggers a column menu for quick actions like sorting, filtering, etc. The actions depend on the enabled Gantt features, for example the filtering action will be available only if filtering is enabled for the element.
   * Default value: false
   */
  columnMenu?: boolean;
  /**
   * Determines whether the Table columns are resizable or not. When enabled it is possible to resize the columns from the header cells of the Table in both Task and Resource timelines.
   * Default value: false
   */
  columnResize?: boolean;
  /**
   * Determines resize feedback is used during column resizing. This property is applicable only when columnResize is enabled.
   * Default value: false
   */
  columnResizeFeedback?: boolean;
  /**
   * Enables/Disables the current time indicator. Current time indicator shows the current time in the appropriate view cells.
   * Default value: false
   */
  currentTimeIndicator?: boolean;
  /**
   * Determines the refresh interval in seconds for the currentTimeIndicator. 
   * Default value: 1
   */
  currentTimeIndicatorInterval?: number;
  /**
   * Sets the GanttChart's Data Export options.
   * Default value: [object Object]
   */
  dataExport?: GanttChartDataExport;
  /**
   * Determines the tasks that will be loaded inside the Timeline. Each item represents an object that should contain the following properties: label - the label of the TaskdateStart - the starting date of the Task. Should be a string representing a valid date.dateEnd - the ending date of the Task. Should be a string representing a valid date.type - determines the type of the task. Whether it's a simple task, a project or a milestone. Each type of task has specific behavior and additional attributes..  Additional properties: connections - an array of objects representing the connection between two tasks. Each connection (object) should have the following properties : target - a number representing the index of the target tasktype - a number representing the type of the connection. Four types of connections are available: 0 - is a connection of type Start-to-Start 1 - is a connection of type End-to-Start 2 - is a connection of type End-to-End3 - is a connection of type Start-to-End lag - a number that determines the delay between two connected auto scheduled tasks. Lag property can be a positive or a negative number. When negative it determines the overlap between two connected tasks. This property is used in conjuction with autoSchedule.duration - determines the duration of a Task in days, hours, minutes, seconds or miliseconds. Very usefull when the the dateEnd of a Task is unknown. The duration always shows the calendar time whether it is in days/hours or other.minDuration - sets the minimum duration of a task. maxDuration - sets the maximum duration of a task.minDateStart - determines the mininum date that a task can start from. Must be if type string and should represent a valid date.maxDateStart - determines the maximum date that a task can start from. Must be if type string and should represent a valid date.minDateEnd - determines the mininum date that a task can end. Must be if type string and should represent a valid date.maxDateEnd - determines the maximum date that a task can end. Must be if type string and should represent a valid date.progress - a number that determines the progress of a task ( from 0 to 100 ).overdue - a boolean that indicates whether the task's dateEnd has surpassed it's deadline date.disableDrag - a boolean property that disables the dragging of a task when set to true.disableResize - a boolean property that disables the resizing of a task when set to true.dragProject - a boolean that determines whether or not the whole project (along with the tasks) can be dragged while dragging the project task. Applicalbe only to Projects.segments - an array of objects that allows to devide a task into multiple segements. Each segment (except the first) can have a different starting date, duration and label.synchronized - a boolean that if set the project task's start/end dates are automatically calculated based on the tasks. By default a synchronized project task can't be dragged alone. Applicable only to Project tasks.expanded - a boolean that determines if a project is expanded or not. If not all of it's sub-tasks are not visible. Only the project task itself is visible. By default no projects are expanded. Applicable only to project tasks..GanttChart also accepts a DataAdapter instance as dataSource. You can read more about the dataAdapter here - https://www.htmlelements.com/docs/data-adapter/.
   * Default value: 
   */
  dataSource?: any;
  /**
   * Determines the format of the dates in the timeline header when they represent days.
   * Default value: short
   */
  dayFormat?: GanttDayFormat | string;
  /**
   * Determines a specific end date for the Timeline. Usefull when the user wants custom ending date for the Timeline. If no date is set the end date of the timeline is automatically determined from the tasks.
   * Default value: 
   */
  dateEnd?: string | Date;
  /**
   * Determines a specific starting date for the Timeline. Usefull when the user wants custom starting date for the Timeline. If no date is set the start date of the timeline is automatically determined from the tasks.
   * Default value: 
   */
  dateStart?: string | Date;
  /**
   * Determines the date markers that will be displayed inside the timeline. Date markers allow to mark and even label specific dates (including time) inside the GanttChart timeline.
   * Default value: null
   */
  dateMarkers?: GanttChartDateMarker[];
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Disables auto scrolling while dragging/resizing a task bar inside the Timeline.
   * Default value: false
   */
  disableAutoScroll?: boolean;
  /**
   * Disables dragging of tasks in the Timeline.
   * Default value: false
   */
  disableTaskDrag?: boolean;
  /**
   * Disables task progress changing in the Timeline.
   * Default value: false
   */
  disableTaskProgressChange?: boolean;
  /**
   * Disables resizing of tasks in the Timeline.
   * Default value: false
   */
  disableTaskResize?: boolean;
  /**
   * Disables the selection inside the GanttChart.
   * Default value: false
   */
  disableSelection?: boolean;
  /**
   * Disables the task segment dragging.
   * Default value: false
   */
  disableSegmentDrag?: boolean;
  /**
   * Disables the task segment resizing.
   * Default value: false
   */
  disableSegmentResize?: boolean;
  /**
   * Disables the window editor for the GanttChart.
   * Default value: false
   */
  disableWindowEditor?: boolean;
  /**
   * Determines in what unit is task duration property measured.
   * Default value: milisecond
   */
  durationUnit?: Duration | string;
  /**
   * Determines whether a dedicated filter row is used for Table filtering instead of the default filter input. This property has no effect if filtering is not enabled.
   * Default value: false
   */
  filterRow?: boolean;
  /**
   * Determines the view start day. Sunday is 0, Monday is 1, Saturday is 6. By default it's Sunday.
   * Default value: -1
   */
  firstDayOfWeek?: number;
  /**
   * Groups the tasks inside the Task timeline according to the resources they are assigned to. Unassigned tasks are placed in a default group labeled 'Unassigned'.
   * Default value: false
   */
  groupByResources?: boolean;
  /**
   * Allows to create a custom header content for the Task Panel. The attribute accepts an HTMLTemplate element, it's id or a function.
   * Default value: null
   */
  headerTemplate?: any;
  /**
   * Determines whether the dateMarkers are visible or not.
   * Default value: false
   */
  hideDateMarkers?: boolean;
  /**
   * By default the Timeline has a three level header - timeline details, timeline second details and timeline header. This property hides the header container( the bottom container ).
   * Default value: false
   */
  hideTimelineHeader?: boolean;
  /**
   * By default the Timeline has a three level header - timeline details, timeline second details and timeline header. This property hides the header details container( the top container ).
   * Default value: false
   */
  hideTimelineHeaderDetails?: boolean;
  /**
   * By default the Timeline has a three level header - timeline details and timeline header. This property hides the second header details container( the middle container ).
   * Default value: true
   */
  hideTimelineSecondHeaderDetails?: boolean;
  /**
   * Shows the selection column of the Task/Resource Table. When applied a checkbox column is displayed that allows to select tasks/resources.
   * Default value: false
   */
  showSelectionColumn?: boolean;
  /**
   * Hides the Resource panel regardless of the resources availability By default the Resource panel is visible if resources are added to the GanttChart. This property allows to hide the Resource panel permanently.
   * Default value: false
   */
  hideResourcePanel?: boolean;
  /**
   * Determines weather or not horizontal scrollbar is shown.
   * Default value: auto
   */
  horizontalScrollBarVisibility?: HorizontalScrollBarVisibility | string;
  /**
   * Determines the format of the dates inside the timeline header when they represent hours.
   * Default value: numeric
   */
  hourFormat?: HourFormat | string;
  /**
   * When enabled, scrolling to the end of the horizotal timeline, triggers the creation of additional to extend the time range. The number of cells to be added when the scrollbar reaches the end position is determined by the infiniteTimelineStep.
   * Default value: false
   */
  infiniteTimeline?: boolean;
  /**
   * Determines the number of cells to be added when the horizontal scroll bar of the Timeline reaches it's end position when infiniteTimeline is enabled.
   * Default value: 5
   */
  infiniteTimelineStep?: number;
  /**
   * When set the Timeline is positioned on the left side while the Task Tree is positioned on the right. By default it's vice versa.
   * Default value: false
   */
  inverted?: boolean;
  /**
   * Determines whether keyboard navigation inside the Table is enabled or not. Keyboard navigation affects both Task and Resource Tables. It allows to navigate between items. the following keyboard shortcut keys are available for focused tasks inside the Task Table: Enter - opens the Window editor to edit the currently focused task.Delete - opens a confirmation window regarding the deletion of the currently focused task.
   * Default value: false
   */
  keyboardNavigation?: boolean;
  /**
   *  Determines the language of the GanttChart. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Detetmines the maximum possible date of the Timeline.
   * Default value: 2100-1-1
   */
  max?: string | Date;
  /**
   * Detetmines the minimum possible date of the Timeline.
   * Default value: 1900-1-1
   */
  min?: string | Date;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * [object Object]
   */
  messages?: any;
  /**
   * Determines the format of the dates the timeline header when they represent months.
   * Default value: short
   */
  monthFormat?: MonthFormat | string;
  /**
   * Determines the scale in Month view.
   * Default value: week
   */
  monthScale?: MonthScale | string;
  /**
   * Determines the nonworking days of the week from 0 to 6, where 0 is the first day of the week and 6 is the last day. Nonworking days will be displayed with colored cells inside the timeline and will not affect the dateEnd of the tasks unless the adjustToNonworkingTime property is enabled.
   * Default value: 
   */
  nonworkingDays?: number[];
  /**
   * Determines the nonworking hours of a day. Hours are represented as numbers inside an array (e.g. [1,2,3] - means 1,2 and 3 AM) or number ranges represented as nested arrays(e.g. [[0,6]] - means from 0 to 6 AM). In the timline the cells that represent nonworking days are colored differently from the rest and will not affect the dateEnd of the tasks unless the adjustToNonworkingTime property is enabled.
   * Default value: 
   */
  nonworkingHours?: number[] | number[][];
  /**
   * A function that can be used to completly customize the popup Window that is used to interact width tasks by changing their properties. The function as three arguments: target - the target popup Window that is about to be opened.type - the type of the window. The type determines the purpose of the window. Three possible values: 'task' (task editing), 'confirm' ( confirmation window), 'connection' (used when deleting a connection between tasks). item - the connection/task object that is the target of the window.
   * Default value: null
   */
  popupWindowCustomizationFunction?: any;
  /**
   * Determines which Tab items are visible inside the popup window. Three possible values are allowed: general - the general tab with task properties determines by the taskColumns property.dependency - the dependency tab which shows the connections to the task and allows to create/delete connections.segments - the segments tab which shows the segments of the task and allows to created/delete segments..
   * Default value: ['general', 'dependency', 'segments']
   */
  popupWindowTabs?: string[];
  /**
   * A format function for the Timeline task progress label. The expected result from the function is a string. The label is hidden by default can be shown with the showProgressLabel property.
   * Default value: null
   */
  progressLabelFormatFunction?: any;
  /**
   * A getter that returns a flat structure as an array of all resources inside the element.
   * Default value: null
   */
  resources?: GanttChartResource[];
  /**
   * Deteremines the columns that will be visible in the Resource Tree. Each entry in the value of this property must be of type Object.  It should contain the label and value keys. The value of label determines the column header label inside the Task Tree. The value of value determines the content of the cells in the column. By default, one column with all resource labels is visible.  Additional properties: formatFunction - a function that allows to customize the content of each record in the column. The function accepts two arguments - the actual label as string that is going to be inserted and the index of the resource. The function must return a string as the new content. min - controls the min size of the column max - controls the max size of the columnsize - controls the actual size of the column
   * Default value: { "label": "resourceColumnLabel", "value": "label" }
   */
  resourceColumns?: GanttChartResourceColumn[];
  /**
   * Determines whether the Resource Table is filterable or not.
   * Default value: false
   */
  resourceFiltering?: boolean;
  /**
   * A format function that allows to re-format the group row labels when groupByResources is enabled.
   * Default value: null
   */
  resourceGroupFormatFunction?: any;
  /**
   * Allows to create a custom header content for the Resource Panel. The attribute accepts an HTMLTemplate element, it's id or a function.
   * Default value: null
   */
  resourcePanelHeaderTemplate?: any;
  /**
   * Determines the min size of the Resource Panel.
   * Default value: 100
   */
  resourcePanelMin?: number | string;
  /**
   * Determines the size of the Resource Panel.
   * Default value: 
   */
  resourcePanelSize?: number | string;
  /**
   * Determines the refresh rate of the Resource Panel when dragging/resizing/progress changing a Task from the Timeline. This property allows to customize the interval between resource Tree/Timeline refreshes. By default they refresh immediately after a change.
   * Default value: 0
   */
  resourcePanelRefreshRate?: number;
  /**
   * A callback that can be used to customize the content of the resource Timeline cells. The callback accepts three arguments: taskIndexes - an array of task indexes that are assigned to the resource for the current cell.resourceIndex - the index of the resource.cellDate - the date of the current cell. This property is used when resourceTimelineView is set to custom. Depending on the resourceTimelineMode, it should return: string - when the resourceTimelineMode is set to 'diagram'.object - when the resourceTimelineMode is set to 'histogram'. The object should have two attributes: capacity and maxCapacity, in order to be visualized as a histogram.. Another usage of this callback is to customize the timeline resource representation completely if resourceTimelineMode is set to custom.
   * Default value: null
   */
  resourceTimelineFormatFunction?: any;
  /**
   * Determines how the capacity of the resources will be visualized inside the resource timeline. By default, the capacity is measured in hours depending on the view property of the element.
   * Default value: diagram
   */
  resourceTimelineMode?: GanttChartResourceTimelineMode | string;
  /**
   * Determines how the resources will be displayed inside the resource Timeline.
   * Default value: hours
   */
  resourceTimelineView?: GanttChartResourceTimelineView | string;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Sets which tasks to select by their id or gets the currently selected task ids. If no id is provided for the task, an internal id is generated for each task according to it's index(tree path).
   * Default value: 
   */
  selectedTaskIds?: number[] | string[];
  /**
   * Sets which resources to select by their id or gets the currently selected resource ids. If no id is provided for the resource, an internal id is generated for each resource according to it's index(tree path).
   * Default value: 
   */
  selectedResourceIds?: number[] | string[];
  /**
   * Enables/Disables the current time shader. If enabled all cells that represent past time will be shaded.
   * Default value: false
   */
  shadeUntilCurrentTime?: boolean;
  /**
   * Determines whether the baselnes of the tasks are visible or not. Baselines are defined via the 'planned' attribute on the task objects of the dataSource property.
   * Default value: false
   */
  showBaseline?: boolean;
  /**
   * Shows the progress label inside the progress bars of the Timeline tasks.
   * Default value: false
   */
  showProgressLabel?: boolean;
  /**
   * If set the dateStart/dateEnd of the tasks will be coerced to the nearest timeline cell date ( according to the view ). Affects the dragging operation as well.
   * Default value: false
   */
  snapToNearest?: boolean;
  /**
   * Allows to set a custom sorting function to be executed when a column is sorted. Can be used to override the default sorting behavior. The function contains the following parameters: dataSource - the Table's data sourcesortColumns - an array of the data fields of columns to be sorted bydirections - an array of sort directions to be sorted by (corresponding to sortColumns)defaultCompareFunctions - an array of default compare functions to be sorted by (corresponding to sortColumns), useful if the sorting of some columns does not have to be overridden.
   * Default value: null
   */
  sortFunction?: { (dataSource: any, sortColumns: string[], directions: string[], defaultCompareFunctions: { (firstRecord: any, secondRecord: any): number }[]): void };
  /**
   * Determines whether the GanttChart can be sorted by one, more then one or no columns.
   * Default value: none
   */
  sortMode?: GanttChartSortMode | string;
  /**
   * A getter that returns a flat structure as an array of all tasks inside the element.
   * Default value: 
   */
  tasks?: GanttChartTask[];
  /**
   * Deteremines the columns that will be visible in the Task Tree. Each entry in the value of this property must be of type Object.  It should contain the label and value keys. The value of label determines the column header label inside the Task Tree. The value of value determines the content of the cells in the column. It should reference a task attribute from the dataSource. By default, one column with all task labels is visible.  Additional properties: formatFunction - a function that allows to customize the content of each record in the column. The function accepts one argument - the actual label as string that is going to be inserted and must return some content. min - controls the min size of the column max - controls the max size of the column size - controls the actual size of the columncustomEditor - a callback that can be used to set a custom editor for the column when editing via the window. It accepts two arguments label - the label of the columnvalue - the value of the column. The callback must return the editor.setCustomEditorValue - a callback that is used to set the value of the custom editor.getCustomEditorValue - a callback that is used to get the value of the custom editor.
   * Default value: { "label": "Task Name", "value": "label" }
   */
  taskColumns?: GanttChartTaskColumn[];
  /**
   * Determines whether the Task Table is filterable or not.
   * Default value: false
   */
  taskFiltering?: boolean;
  /**
   * Determines the min size of the Task Panel. Used when Resource Panel is visible.
   * Default value: 200
   */
  taskPanelMin?: string | number;
  /**
   * Determines the size of the Task Panel. Used when Resource Panel is visible.
   * Default value: 
   */
  taskPanelSize?: string | number;
  /**
   * Determines the min width of the timeline.
   * Default value: 200
   */
  timelineMin?: string | number;
  /**
   * Determines the min width of the task table.
   * Default value: 100
   */
  treeMin?: string | number;
  /**
   * Determines the size(width) of the task table.
   * Default value: 100
   */
  treeSize?: string | number;
  /**
   * A format function for the Header of the Timeline. The function provides the following arguments: date - a Date object that represets the date for the current cell.type - a string that represents the type of date that the cell is showing, e.g. 'month', 'week', 'day', etc.isHeaderDetails - a boolean that indicates whether the current cell is part of the Header Details Container or not.value - a string that represents the default value for the cell provided by the element.
   * Default value: null
   */
  timelineHeaderFormatFunction?: any;
  /**
   * Determines whether the tooltips are enabled or not. Tooltips are available for timeline tasks, resources, connections, indicators and segments.
   * Default value: [object Object]
   */
  tooltip?: GanttChartTooltip;
  /**
   * Determines weather or not vertical scrollbar is shown.
   * Default value: auto
   */
  verticalScrollBarVisibility?: VerticalScrollBarVisibility | string;
  /**
   * Determines the viewing date range of the timeline. Possible values: day - The timeline show the hours of the day.week - the timeline shows the days of the week.month - the timeline shows the days of the month.year - the timeline shows the months of the year.resource - displays the current tasks by grouping them according to the resources they have assigned. The unassigned tasks will be placed in a separate group called 'Unassigned'.  The timeline has a header section that contains the labels of each cell according to the date inside them. The header is splitted in two sections in order to give a more detailed information of the dates.
   * Default value: year
   */
  view?: GanttChartView | string;
  /**
   * Determines the format of the dates inside the timeline header when they represent years.
   * Default value: numeric
   */
  yearFormat?: YearFormat | string;
  /**
   * Determines the format of the dates inside the timeline header when they represent weeks. 
   * Default value: long
   */
  weekFormat?: WeekFormat | string;
  /**
   * Sets or gets the element's visual theme. 
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets or gets if the element can be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 Gantt charts are specialized bar charts that help clearly represent how tasks and resources are allocated over time in planning, project management, and scheduling applications.
*/
export interface GanttChart extends BaseElement, GanttChartProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when a batch update was started after executing the <b>beginUpdate</b> method.
	* @param event. The custom event.    */
  onBeginUpdate?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a batch update was ended from after executing the <b>endUpdate</b> method.
	* @param event. The custom event.    */
  onEndUpdate?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user starts connecting one task to another. This event allows to cancel the operation by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(startIndex)
   *  startIndex - The index of the task that a connection is started from.
   */
  onConnectionStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user completes a connection between two tasks.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, startTaskId, startIndex, endIndex, endTaskId, type)
   *  id - The id of the connection that was created.
   *  startTaskId - The id of the task that a connection is started from.
   *  startIndex - The index of the task that a connection is started from.
   *  endIndex - The index of the task that a connection ended to.
   *  endTaskId - The id of the task that a connection ended to.
   *  type - The type of connection. Fours types are available: <ul><li><b>0</b> - start-to-start</li><li><b>1</b> - end-to-start</li><li><b>2</b> - end-to-end</li><li><b>3</b> - start-to-end</li></ul>
   */
  onConnectionEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a Task is selected/unselected.
	* @param event. The custom event. Custom data event was created with: ev.detail(value, oldValue)
   *  value - The index of the new selected task.
   *  oldValue - The index of the previously selected task.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a Tree column is resized. Column resizing is controled by the <b>columnResize</b> property.
	* @param event. The custom event. Custom data event was created with: ev.detail(dataField, headerCellElement, widthInPercentages, width)
   *  dataField - The name of the column. Corresponds to the <b>value</b> attribute of a <b>taskColumns/resourceColumns</b> object.
   *  headerCellElement - The HTMLElement column cell element that was resized.
   *  widthInPercentages - The new width of the column in percentages.
   *  width - The new width of the column in pixels.
   */
  onColumnResize?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered just before the window for task editing or tooltip is closing. The closing operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(owner, item, target, type)
   *  owner - The HTMLElement that is the owner of the tooltip. This attribute is defined only when the event is related to the tooltip.
   *  item - The item object that is related to the tooltip owner. It can be a task/segment/resource/indicator object. This attribute is defined only when the event is related to the tooltip.
   *  target - The instance of the window/tooltip that is going to close.
   *  type - The type of window/tooltip that is going to close. There are three types of windows inside GanttChart: <ul><li><b>confirm</b> - a confirm window. This type of window is usually used to confirm the deletion of a task.</li><li><b>task</b> - a window used for task editing.</li><li><b>connection</b> - a window used to delete a connection.</li></ul>. If the event is a tooltip event, there are several tooltip types: <ul><li>indicator - when the tooltip owner is an indicator.</li><li>segment - when the tooltip owner is a task segment.</li><li>task - when the tooltip owner is a task.</li><li>resource - when the tooltip target is a resource.</li></ul>
   */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the window for task editing is closed( hidden )
	* @param event. The custom event. Custom data event was created with: ev.detail(owner, item, target, type)
   *  owner - The HTMLElement that is the owner of the tooltip. This attribute is defined only when the event is related to the tooltip
   *  item - The item object that is related to the tooltip owner. It can be a task/segment/resource/indicator object. This attribute is defined only when the event is related to the tooltip.
   *  target - The instance of the window/tooltip that is closed.
   *  type - The type of window/tooltip that is closed. There are three types of windows inside GanttChart: <ul><li><b>confirm</b> - a confirm window. This type of window is usually used to confirm the deletion of a task.</li><li><b>task</b> - a window used for task editing.</li><li><b>connection</b> - a window used to delete a connection.</li></ul>. If the event is a tooltip event, there are several tooltip types: <ul><li>indicator - when the tooltip owner is an indicator.</li><li>segment - when the tooltip owner is a task segment.</li><li>task - when the tooltip owner is a task.</li><li>resource - when the tooltip target is a resource.</li></ul>
   */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when an item is collapsed.
	* @param event. The custom event. Custom data event was created with: ev.detail(isGroup, item, index, label, value)
   *  isGroup - A boolean flag indicating whether the collapsed item is a resource group. This is the case when <b>groupByResoruces</b> is enabled.
   *  item - The object details of the collapsed item.
   *  index - The index of the collapsed item.
   *  label - The label of the collapsed item.
   *  value - The value of the collapsed item.
   */
  onCollapse?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when dragging of a task starts. This event allows to cancel the operation by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, item, dateStart, dateEnd, segment)
   *  id - The id of the task that is going to be dragged.
   *  item - The object of the task that is going to be dragged.
   *  dateStart - The start date of the task that is going to be dragged.
   *  dateEnd - The end date of the task that is going to be dragged.
   *  segment - The segment object that is going to be dragged. This attribute is undefined if a segment is not going to be dragged.
   */
  onDragStart: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when dragging of a task finishes.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, item, dateStart, dateEnd, segment)
   *  id - The id of the task that is was dragged.
   *  item - The object of the task that is was dragged.
   *  dateStart - The start date of the task that is was dragged.
   *  dateEnd - The end date of the task that is was dragged.
   *  segment - The segment object that was dragged. This attribute is undefined if a segment has not been dragged.
   */
  onDragEnd: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when an item is expanded.
	* @param event. The custom event. Custom data event was created with: ev.detail(isGroup, item, index, label, value)
   *  isGroup - A boolean flag indicating whether the collapsed item is a resource group. This is the case when <b>groupByResoruces</b> is enabled.
   *  item - The index of the expanded item.
   *  index - The index of the expanded item.
   *  label - The label of the expanded item.
   *  value - The value of the expanded item.
   */
  onExpand?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the GanttChart is filtered.
	* @param event. The custom event. Custom data event was created with: ev.detail(type, action, filters)
   *  type - The type of items that have been filtered ( task or resource ).
   *  action - The name of the filtering action (whether filtering is added or removed).
   *  filters - The filters that have been applied. Filters represent JQX.Utilities.FilterGroup objects.
   */
  onFilter?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a task, resource or connection is clicked inside the Timeline or the Tree columns.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, item, type, originalEvent)
   *  id - The id of the task.
   *  item - The item that was clicked. It can be a task, resource or connection.
   *  type - The type of item. Possible values are: 'task', 'project', 'resource', 'connection'.
   *  originalEvent - The original DOM event.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a Task/Resource/Connection is inserted.
	* @param event. The custom event. Custom data event was created with: ev.detail(type, item)
   *  type - The type of item that has been modified. The type could be: 'connection', 'task', 'project', 'resource'.
   *  item - An object that represents the actual item with it's attributes.
   */
  onItemInsert?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a Task/Resource/Connection is removed.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, type, item)
   *  id - The id of the task.
   *  type - The type of item that has been modified. The type could be: 'connection', 'task', 'project', 'resource'.
   *  item - An object that represents the actual item with it's attributes.
   */
  onItemRemove?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a Task/Resource/Connection is updated.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, type, item)
   *  id - The id of the task.
   *  type - The type of item that has been modified. The type could be: 'connection', 'task', 'project', 'resource'.
   *  item - An object that represents the actual item with it's attributes.
   */
  onItemUpdate?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered just before the window for task editing or tooltip is opening. The opening operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(owner, item, target, type)
   *  owner - The HTMLElement that is the owner of the tooltip. This attribute is defined only when the event is related to the tooltip
   *  item - The item object that is related to the tooltip owner. It can be a task/segment/resource/indicator object. This attribute is defined only when the event is related to the tooltip.
   *  target - The instance of the window/tooltip that is going to open.
   *  type - The type of window/tooltip that is going to open. There are three types of windows inside GanttChart: <ul><li><b>confirm</b> - a confirm window. This type of window is usually used to confirm the deletion of a task.</li><li><b>task</b> - a window used for task editing.</li><li><b>connection</b> - a window used to delete a connection.</li></ul>. If the event is a tooltip event, there are several tooltip types: <ul><li>indicator - when the tooltip owner is an indicator.</li><li>segment - when the tooltip owner is a task segment.</li><li>task - when the tooltip owner is a task.</li><li>resource - when the tooltip target is a resource.</li></ul>
   */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the window for task editing is opened( visible ) or when the tooltip is opened.
	* @param event. The custom event. Custom data event was created with: ev.detail(owner, item, target, type)
   *  owner - The HTMLElement that is the owner of the tooltip. This attribute is defined only when the event is related to the tooltip
   *  item - The item object that is related to the tooltip owner. It can be a task/segment/resource/indicator object. This attribute is defined only when the event is related to the tooltip.
   *  target - The instance of the window/tooltip that is opened.
   *  type - The type of window/tooltip that is opened. There are three types of windows inside GanttChart: <ul><li><b>confirm</b> - a confirm window. This type of window is usually used to confirm the deletion of a task.</li><li><b>task</b> - a window used for task editing.</li><li><b>connection</b> - a window used to delete a connection.</li></ul>. If the event is a tooltip event, there are several tooltip types: <ul><li>indicator - when the tooltip owner is an indicator.</li><li>segment - when the tooltip owner is a task segment.</li><li>task - when the tooltip owner is a task.</li><li>resource - when the tooltip target is a resource.</li></ul>
   */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the progress of a task bar starts to change as a result of user interaction. This event allows to cancel the operation by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, index, progress)
   *  id - The id of the task.
   *  index - The index of the task which progress is going to be changed.
   *  progress - The progress of the task before it is changed.
   */
  onProgressChangeStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the progress of a task is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, index, progress)
   *  id - The id of the task.
   *  index - The index of the task which progress is has been changed.
   *  progress - The progress of the task after it was changed.
   */
  onProgressChangeEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when resizing of a task starts. This event allows to cancel the operation by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, item, dateStart, dateEnd, segment)
   *  id - The id of the task that is going to be resized.
   *  item - The object of the task that is going to be resized.
   *  dateStart - The start date of the task that is going to be resized.
   *  dateEnd - The end date of the task that is going to be resized.
   *  segment - The segment object that is going to be resized. This attribute is undefined if a segment is not going to be resized.
   */
  onResizeStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the resizing of a task finishes.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, item, dateStart, dateEnd, segment)
   *  id - The id of the task that was resized.
   *  item - The object of the task that was resized.
   *  dateStart - The start date of the task that was resized.
   *  dateEnd - The end date of the task that was resized.
   *  segment - The segment object that was resized. This attribute is undefined if a segment has not been resized.
   */
  onResizeEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the GanttChart is sorted by some column.
	* @param event. The custom event. Custom data event was created with: ev.detail(type, columns, sortDataFields, sortOrders, sortDataTypes)
   *  type - The type of columns that have been sorted ( task or resource column ).
   *  columns - An array of objects that contains the currently sorted column objects.
   *  sortDataFields - The dataFields of the columns that have been sorted. The dataField corresponds to the <b>value</b> property of a <b>taskColumns/resourceColumns</b> object.
   *  sortOrders - The orders of the columns that have been sorted.
   *  sortDataTypes - The data types of the columns that have been sorted.
   */
  onSort?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Timeline has been scrolled to the bottom.
	* @param event. The custom event.    */
  onScrollBottomReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Timeline has been scrolled to the top.
	* @param event. The custom event.    */
  onScrollTopReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Timeline has been scrolled to the beginning (horizontally).
	* @param event. The custom event.    */
  onScrollLeftReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the Timeline has been scrolled to the end (horizontally).
	* @param event. The custom event.    */
  onScrollRightReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Adds a custom filter to a specific column (task or resource column).
   * @param {any} columns. An object or an array of objects with the following syntax: <ul><li><b>type</b> - indicates the type of column to filter. Possible values are 'task' or 'resource'.</li><li><b>value</b> - the value of the column that must match the value attribute of a taskColumns/resourceColumns object(e.g. 'label', 'dateStart', etc).</li></ul>.
   * @param {any} filterGroup. A Smart.Utilities.FilterGroup object. Here's an example for creating a FilterGroup object: <pre>const filterGroup = new window.Smart.Utilities.FilterGroup(), filterObject = filterGroup.createFilter('string', 'Task B', 'STARTS_WITH_CASE_SENSITIVE'); filterGroup.addFilter('or', filterObject); gantt.addFilter({ type: 'task', value: 'label' }, filterGroup);</pre>
   */
  addFilter(columns: any, filterGroup: any): void;
  /**
   * Clears the currently applied filters.
   */
  clearFilters(): void;
  /**
   * Clears the currently applied column sorting.
   */
  clearSort(): void;
  /**
   * Unselects all currently selected items inside the GanttChart including Tasks and Resources. It also clears the assignment highlgihters.
   */
  clearSelection(): void;
  /**
   * Removes a previously saved state of the element form LocalStorage according to it's id. <strong>Requires an id to be set to the element.</strong>
   */
  clearState(): void;
  /**
   * Removes all tasks. 
   */
  clearTasks(): void;
  /**
   * Removes all resources. 
   */
  clearResources(): void;
  /**
   * Creates a connection between two tasks. 
   * @param {number | string} startTaskIndex. The id of the start task or the connection string like '2-3-0'. <b>If the complete connections string is provided as the first argument, the rest of the method arguments are not necessary</b>
   * @param {number | string} taskEndIndex?. The id of the end task.
   * @param {number} connectionType?. The type of the connection. A numeric value from 0 to 3. The connection type can be: <ul><li><b>0</b> - Start-to-Start connection.</li><li><b>1</b> - End-to-Start connection.</li><li><b>2</b> - End-to-End connection.</li><li><b>3</b> - Start-to-End connection.</li></ul>
   * @param {number} lag?. The connection lag in miliseconds. Used by the Auto scheduling algorithm in order allow some slack time slack time before or after the next task begins/ends. Lag is measured in miliseconds. It can be a negative (lead) or a positive (lag) number.
   */
  createConnection(startTaskIndex: number | string, taskEndIndex?: number | string, connectionType?: number, lag?: number): void;
  /**
   * Collapses an expanded project.
   * @param {string | number} id. The id of a project item that should be collapsed.
   */
  collapse(id: string | number): void;
  /**
   * Starts an update operation. This is appropriate when calling multiple methods or set multiple properties at once.
   */
  beginUpdate(): void;
  /**
   * Ends the update operation. This method will resume the rendering and will refresh the GanttChart.
   */
  endUpdate(): void;
  /**
   * Refereshes the GanttChart after resizing by recalculating the Scrollbars. 
   * @param {boolean} fullRefresh?. If set the GanttChart will be re-rendered completely.
   */
  refresh(fullRefresh?: boolean): void;
  /**
   * Makes sure a Task is visible by scrolling to it.
   * @param {string | number} taskId. The id of the target Task.
   */
  ensureVisible(taskId: string | number): void;
  /**
   * Expands a collapsed project with tasks.
   * @param {string | number} id. The id of a project task that should be expanded.
   */
  expand(id: string | number): void;
  /**
   * Exports the data of Tree of the GanttChart.
   * @param {string} dataFormat. Determines the format of the exported file. Three possible values are available: <ul><li><b>pdf</b></li><li><b>xlsx</b></li><li><b>html</b></li><li><b>tsv</b></li><li><b>csv</b></li><li><b>xml</b></li></ul>
   * @param {any} callback?. A callback that allows to format the exported data based on a condition. For additional details, refer ro the Smart Export Documentation.
   */
  exportData(dataFormat: string, callback?: any): void;
  /**
   * Returns all existing connections. The connections are returned as objects that contain detailed information. Each object in the array has the following keys: 'id' - connection id, 'type' - connection type, 'startTaskId' - connection's start task id, 'endTaskId' - connection's end task id, 'startIndex' - connection's start task index, 'endIndex' - connection's end task index, 'lag' - lag time. 
   * @returns {any}
   */
  getConnections(): any;
  /**
   * Returns the connection details for the target connection which includes: startTask, endTask, startTaskId, endTaskId and type of the corresponding connection. Connection types are described in detail under the `connectionEnd` event description in this document and in a dedicated topic available on the website.
   * @param {string} connectionId. A connection id. Each connection has a unique id that is assigned when a connection is created.
   * @returns {any}
   */
  getConnectionDetails(connectionId: string): any;
  /**
   * Returns a JSON representation of all tasks inside the element along with their connections and settings.
   * @returns {any[]}
   */
  getState(): any[];
  /**
   * Returns the Tree path of a task/resource. The tree path is used as task/resource id if no id is provided by the user.
   * @param {any} item. A GattChartTask/GanttChartResource item object.
   * @returns {string}
   */
  getItemPath(item: any): string;
  /**
   * Returns the task object that corresponds to the id/path.
   * @param {string | number} itemId. The id/path of a task.
   * @returns {any}
   */
  getTask(itemId: string | number): any;
  /**
   * Returns an array of all GanttChart tasks.
   * @returns {any[]}
   */
  getTasks(): any[];
  /**
   * Returns the index of a task.
   * @param {any} task. A GattChartTask object.
   * @returns {number}
   */
  getTaskIndex(task: any): number;
  /**
   * Returns the connections definitions of a task.
   * @param {any} taskId. A GanttChartTask object or it's id.
   * @returns {any}
   */
  getTaskConnections(taskId: any): any;
  /**
   * Returns the Project of a task or undefined if it does not have one.
   * @param {any} task. A GantChartTask object.
   * @returns {any}
   */
  getTaskProject(task: any): any;
  /**
   * Returns the resource object that corresponds to the id/path.
   * @param {string | number} itemId. The id/path of a resource.
   * @returns {any}
   */
  getResource(itemId: string | number): any;
  /**
   * Returns an array of all GanttChart resources.
   * @returns {any[]}
   */
  getResources(): any[];
  /**
   * Returns the index of a resource.
   * @param {any} resource. A GanttChartResource object.
   * @returns {number}
   */
  getResourceIndex(resource: any): number;
  /**
   * Returns the tasks that are assigned to the resource.
   * @param {any} resource. A GanttChartResource object or it's id.
   * @returns {any}
   */
  getResourceTasks(resource: any): any;
  /**
   * Returns the currently selected tasks/resource ids. If selection is disabled or no items are selected returns null.
   * @returns {any}
   */
  getSelectedIds(): any;
  /**
   * Returns the currently selected tasks.
   * @returns {any}
   */
  getSelectedTasks(): any;
  /**
   * Returns the currently selected resources.
   * @returns {any}
   */
  getSelectedResources(): any;
  /**
   * Returns the working hours of the day as numbers.
   * @returns {any}
   */
  getWorkingHours(): any;
  /**
   * Hides the tooltip if it's visible.
   * @returns {any}
   */
  hideTooltip(): any;
  /**
   * Depending on the nonworkingDays property, returns true or false whether the target date is on a working day or not.
   * @param {Date} date. A javascript Date object or a string/number which represents a valid JS Date.
   */
  isWorkingDay(date: Date): void;
  /**
   * Loads a previously saved state of the element or checks LocalStorage for any saved states if no argument is passed to the method.
   * @param {any[]} state?. An Array containing a valid structure of Gantt Chart tasks.
   */
  loadState(state?: any[]): void;
  /**
   * Removes all connections between tasks. 
   */
  removeAllConnections(): void;
  /**
   * Removes a connection between tasks. The function accepts three arguments(task's start index, end index and connection type) or one connection string argument which describes the connection.
   * @param {number | string} startTaskIndex. The index of the start task or the connection string like '2-3-0.
   * @param {number} taskEndIndex?. The index of the end task.
   * @param {number} connectionType?. The type of the connection. A numeric value from 0 to 3.
   * @returns {any}
   */
  removeConnection(startTaskIndex: number | string, taskEndIndex?: number, connectionType?: number): any;
  /**
   * Removes all connections of a task or between two tasks if the second argument is provided and valid.
   * @param {any} taskStart. The start task object or it's id.
   * @param {any} taskEnd?. The end task object or it's id.
   */
  removeTaskConnection(taskStart: any, taskEnd?: any): void;
  /**
   * Shows the tooltip for a specific element.
   * @param {HTMLElement} target. The HTMLElement that will be the target of the tooltip.
   * @param {string} content?. Allows to set a custom content for the Tooltip.
   */
  showTooltip(target: HTMLElement, content?: string): void;
  /**
   * Saves the current settings of the element to LocalStorage. <strong>Requires an id to be set to the element.</strong>
   * @param {any[]} state?. An Array containing a valid structure of Gantt Chart tasks.
   */
  saveState(state?: any[]): void;
  /**
   * Inserts a new task in the timeline. The new task can be inserted as a sub task of a project by passing the appropriate argument for the project id or as a root level item.
   * @param {any} taskObject. An object describing a Gantt Chart task.
   * @param {any} project?. A number or string that represents the id of a project (e.g. '0') or a project object definition present in the GanttChart. This parameter determines the parent project of the task that will be inserted. If <b>null</b> is passed as an arguemnt the new task will be inserted at root level without a parent project.
   * @param {number} index?. The index where the new item should be inserted(e.g. 2). This index will determine the position of the newly inserted task.
   * @returns {string | number | undefined}
   */
  insertTask(taskObject: any, project?: any, index?: number): string | number | undefined;
  /**
   * Updates a task/project/milestone.
   * @param {any} taskId. A number or string that represents the id of a task/project(e.g. '0') or the object definition of the task/project.
   * @param {any} taskObject. An object describing a Gantt Chart task. The properties of this object will be applied to the desired task.
   */
  updateTask(taskId: any, taskObject: any): void;
  /**
   * Removes a task from the timeline.
   * @param {any} taskId. A number or string that represents the id of a task or the actual item object.
   */
  removeTask(taskId: any): void;
  /**
   * Inserts a new resource.
   * @param {string | number} resourceId. A string that represents the id of a resource or it's hierarchical position, e.g. '0' ( following SmartTree syntax), or a number that represents the index of a resource.
   * @param {any} resourceObject?. An object describing a Gantt Chart resource.
   */
  insertResource(resourceId: string | number, resourceObject?: any): void;
  /**
   * Updates an existing resource.
   * @param {any} resourceId. A string that represents the id of a resource or it's hierarchical position, e.g. '0' ( following SmartTree syntax), or a number that represents the index of a resource.
   * @param {any} taskObject. An object describing a Gantt Chart resource. The properties of this object will be applied to the target resource.
   */
  updateResource(resourceId: any, taskObject: any): void;
  /**
   * Removes a resource.
   * @param {any} resourceId. A string that represents the id of a resource or it's hierarchical position, e.g. '0' ( following SmartTree syntax), or a number that represents the index of a resource.
   */
  removeResource(resourceId: any): void;
  /**
   * Opens the popup Window for specific task to edit or to delete a connection if a connection string is passed.
   * @param {any} taskId. A string or number that represents the id of a task or the task object that is going to be edited or a connection string(e.g. '2-0-0').
   */
  openWindow(taskId: any): void;
  /**
   * Closes an opened popup Window. The method will close any opened popup window inside the element.
   */
  closeWindow(): void;
  /**
   * Prepares the GanttChart for printing by opening the browser's Print Preview.
   */
  print(): void;
  /**
   * Allows to sets the working days and hours at once.
   * @param  settings. An object definition that contains the days and hours that should be working. The days and hours can be defined as an array of numbers where each number is a day/hour, strings where each string represents a range of days/hours (e.g. '1-5' or '2:00-8:00') or nested array of numbers (e.g. [[1,5]] or [[2, 8]]) which means from 1 to 5 or 2 to 8.
   */
  setWorkTime(settings: { days: (number | string | number[])[], hours: (number | string | number[])[] }): void;
  /**
   * Allows to select a task based on it's id.
   * @param {string | number} id. The id of the task to select.
   */
  selectTask(id: string | number): void;
  /**
   * Allows to select a resource based on it's id.
   * @param {string | number} id. The id of the resource to select.
   */
  selectResource(id: string | number): void;
  /**
   * Allows to unselect a task based on it's id.
   * @param {string | number} id. The id of the task to unselect.
   */
  unselectTask(id: string | number): void;
  /**
   * Allows to unselect a resource based on it's id.
   * @param {string | number} id. The id of the resource to unselect.
   */
  unselectResource(id: string | number): void;
  /**
   * Allows to unset previously set working time. The opposte method for <b>setWorkingTime</b>.
   * @param  settings. An object definition that contains the days and hours that should not be working. The days and hours can be defined as an array of numbers where each number is a day/hour, strings where each string represents a range of days/hours (e.g. '1-5' or '2:00-8:00') or nested array of numbers (e.g. [[1,5]] or [[2, 8]]) which means from 1 to 5 or 2 to 8.
   */
  unsetWorkTime(settings: { days: (number | string | number[])[], hours: (number | string | number[])[] }): void;
  /**
   * Sorts the GanttChart tasks/resources if <b>sortable</b> is enabled.
   * @param {any} columns. An Array of objects which determine which columns to be sorted, the sort order and the type of item to sort: task or resource. If no arguments are provided sorting will be removed. <br /> An object should have the following properties: <ul><li><b>value</b> - a string that represents the value of a <b>taskColumn</b> to sort.</li><li><b>sortOrder</b> - a string that represents the sorting order for the column: 'asc' (asscending sorting) or 'desc' (descending) are possible values. </li><li><b>type</b> - a string that represents the type of item to sort. This property determines which panel will be sorted. Two possible values: 'task', 'resource'.</li></ul>
   */
  sort(columns: any): void;
}

/**Sets the GanttChart's Data Export options. */
export interface GanttChartDataExport {
  /**
   * Determines whether to export filtered items or not. By default filtered data is not exported.
   * Default value: false
   */
  exportFiltered?: boolean;
  /**
   * Sets the exported file's name.
   * Default value: "jqxGanttChart"
   */
  fileName?: string;
  /**
   * Determines the type of items that is going to be exported. 
   * Default value: task
   */
  itemType?: GanttChartDataExportItemType | string;
}

export interface GanttChartDateMarker {
  /**
   * Determines the marker label.
   * Default value: 
   */
  label?: string | null;
  /**
   * Determines the date for the marker. The date can include time as well.
   * Default value: null
   */
  date?: string | Date | number;
  /**
   * Allows to add a custom class name to the marker.
   * Default value: "null"
   */
  className?: string;
}

export interface GanttChartResource {
  /**
   * Resource class. Used to style the resource Timeline.
   * Default value: ""
   */
  class?: string;
  /**
   * The capacity of a resource. By default it is used to determine the working capacity ( in hours ) of the resource.
   * Default value: 8
   */
  capacity?: number;
  /**
   * Resource visibility.
   * Default value: false
   */
  hidden?: boolean | undefined;
  /**
   * Resource id. The unique id of the resource.
   * Default value: ""
   */
  id?: string;
  /**
   * Resource label.
   * Default value: 0
   */
  label?: string | null;
  /**
   * Resource min capacity
   * Default value: 0
   */
  minCapacity?: number;
  /**
   * Resource max capacity. By default this property is used for the resource timeline histogram where maxCapacity is the maximum working capacity in hours of the resource.
   * Default value: 0
   */
  maxCapacity?: number;
  /**
   * Resource progress. Progress is the total progress of the resource based on the tasks it is assigned to. This property is automatically calculated.
   * Default value: 0
   */
  progress?: number;
  /**
   * Resource type.
   * Default value: 
   */
  type?: any;
  /**
   * Resource value.
   * Default value: 
   */
  value?: any;
  /**
   * Resource workload. Workload is the total working time in hours of a resource based on the tasks it is assigned to. This property is automatically calculated.
   * Default value: 0
   */
  workload?: string | number;
}

export interface GanttChartResourceColumn {
  /**
   * Column's label.
   * Default value: 
   */
  label?: string | null;
  /**
   * Column's value. The value shold reference an actual resoruce attribute.
   * Default value: 
   */
  value?: string | null;
  /**
   * Column's min size.
   * Default value: 0
   */
  min?: string | number | null;
  /**
   * Column's size.
   * Default value: 0
   */
  size?: string | number | null;
  /**
   * Column's format function. You can use it to customize the column label's rendering.
   * Default value: null
   */
  formatFunction?: any;
}

export interface GanttChartTask {
  /**
   * Tasks connection.
   * Default value: undefined
   */
  connections?: any;
  /**
   * Project, Task or Milestone CSS class.
   * Default value: ""
   */
  class?: string;
  /**
   * Project, Task or Milestone start date.
   * Default value: 
   */
  dateStart?: string | Date;
  /**
   * Project, Task or Milestone end date.
   * Default value: 
   */
  dateEnd?: string | Date;
  /**
   * Determines the deadline for the Project, Task or Milestone.
   * Default value: 
   */
  deadline?: string | Date;
  /**
   * Project, Task or Milestone with disabled resources.
   * Default value: false
   */
  disableResources?: boolean;
  /**
   * Project, Task or Milestone dragging is disabled.
   * Default value: false
   */
  disableDrag?: boolean;
  /**
   * Project, Task or Milestone resizing is disabled.
   * Default value: false
   */
  disableResize?: boolean;
  /**
   * Project, Task or Milestone drag enabled in the view.
   * Default value: true
   */
  dragProject?: boolean;
  /**
   * The duration of the tasks in miliseconds. The duration unit can be changed via the durationUnit property.
   * Default value: 0
   */
  duration?: number | undefined;
  /**
   * Project, Task or Milestone expanded state in the view.
   * Default value: false
   */
  expanded?: boolean;
  /**
   * Task visibility.
   * Default value: false
   */
  hidden?: boolean | undefined;
  /**
   * Project, Task or Milestone id.
   * Default value: 
   */
  id?: string | null;
  /**
   * Determines the indicators for the task. Task indicators represent special dates that are displayed inside the task's row.
   * Default value: null
   */
  indicators?: GanttChartTaskIndicator[];
  /**
   * Project, Task or Milestone label.
   * Default value: 
   */
  label?: string | null;
  /**
   * Project, Task or Milestone format function.
   * Default value: null
   */
  formatFunction?: any;
  /**
   * Project, Task or Milestone max start date.
   * Default value: 
   */
  maxDateStart?: string | Date;
  /**
   * Project, Task or Milestone min start date.
   * Default value: 
   */
  minDateStart?: string | Date;
  /**
   * Project, Task or Milestone max end date.
   * Default value: 
   */
  maxDateEnd?: string | Date;
  /**
   * Project, Task or Milestone min end date.
   * Default value: 
   */
  minDateEnd?: string | Date;
  /**
   * The minimum duration of the Project, Task or Milestone in miliseconds. The units can be changed via the durationUnit property.
   * Default value: 0
   */
  minDuration?: number | undefined;
  /**
   * The maximum duration of the Project, Task or Milestone in miliseconds. The unit can be changed via the durationUnit property.
   * Default value: 0
   */
  maxDuration?: number | undefined;
  /**
   * Determines whether the task is overdue it's deadline date or not. The property acts only as a getter. By default it's false, unless there's a deadline defined for the task and the dateEnd has surpassed the deadline date.
   * Default value: false
   */
  overdue?: boolean;
  /**
   * Determines the planned dateStart/dateEnd for as the baseline for the task.
   * Default value: undefined
   */
  planned?: GanttChartTaskPlanned;
  /**
   * Project, Task or Milestone progress.
   * Default value: 0
   */
  progress?: number;
  /**
   * Project, Task or Milestone resources.
   * Default value: 
   */
  resources?: any;
  /**
   * Determines the segments of a task. GanttChart items of type 'task' can be segmented into smaller pieces. This property stores the segment definitions. At least two segments need to be defined in order to segment a task. The first segment should start on the same date as the task. The Last segment should end on the same date as the task.
   * Default value: null
   */
  segments?: GanttChartTaskSegment[];
  /**
   * Project, Task or Milestone synchronized in the view.
   * Default value: false
   */
  synchronized?: boolean;
  /**
   * Project's tasks. Only projects can have tasks.
   * Default value: 
   */
  tasks?: any;
  /**
   * Project, Task or Milestone type. Possible values are 'project', 'milestone' and 'task'
   * Default value: task
   */
  type?: GanttChartTaskType | string;
  /**
   * Project, Task or Milestone value.
   * Default value: 
   */
  value?: any;
}

export interface GanttChartTaskIndicator {
  /**
   * Indicator label.
   * Default value: ""
   */
  label?: string;
  /**
   * Indicator date(can include time).
   * Default value: 
   */
  date?: string | Date;
  /**
   * A custom class name that can be applied to the indicator's element in order to apply some styling via CSS.
   * Default value: "null"
   */
  className?: string;
  /**
   * Represents the code for an icon that will be displayed infront of the indicator label inside the timeline.
   * Default value: "null"
   */
  icon?: string;
  /**
   * Determines the tooltip content for the indicator. By default indicators do not show tooltips when hovered.
   * Default value: "null"
   */
  tooltip?: string;
}

/**Determines the planned dateStart/dateEnd for as the baseline for the task. */
export interface GanttChartTaskPlanned {
  /**
   * Determines the planned dateStart of the task.
   * Default value: 
   */
  dateStart?: string | Date;
  /**
   * Determines the planned dateEnd of the task.
   * Default value: 
   */
  dateEnd?: string | Date;
  /**
   * Determines the planned duration of the task.
   * Default value: 0
   */
  duration?: number | undefined;
}

export interface GanttChartTaskSegment {
  /**
   * Segment label.
   * Default value: 
   */
  label?: string | null;
  /**
   * Segment start date.
   * Default value: 
   */
  dateStart?: string | Date;
  /**
   * Segment end date.
   * Default value: 
   */
  dateEnd?: string | Date;
  /**
   * Determines whether segment dragging is disabled.
   * Default value: false
   */
  disableDrag?: boolean;
  /**
   * Determines whether segment resizing is disabled.
   * Default value: false
   */
  disableResize?: boolean;
  /**
   * The duration of a segment in miliseconds(unit). The duration unit can be changed via the durationUnit property.
   * Default value: 0
   */
  duration?: number | undefined;
  /**
   * Segment label format function.
   * Default value: null
   */
  formatFunction?: any;
}

export interface GanttChartTaskColumn {
  /**
   * Determines whether the task propery determined by column can be edited from the Window editor or not. By default editing is enabled.
   * Default value: false
   */
  disableEdit?: boolean;
  /**
   * Applies only to column's that display dates (e.g. dateStart/dateEnd, etc). This property allows to define a JS Intl.DateTimeFormat object in order to format the dates of the column. Here is an example value of the property: dateFormat: { year: '2-digit', month: 'long', day: 'numeric' }. Another option is to use a date format string. Built-in Date formats:// short date pattern'd' - 'M/d/yyyy',// long date pattern'D' - 'dddd, MMMM dd, yyyy',// short time pattern't' - 'h:mm tt',// long time pattern'T' - 'h:mm:ss tt',// long date, short time pattern'f' - 'dddd, MMMM dd, yyyy h:mm tt',// long date, long time pattern'F' - 'dddd, MMMM dd, yyyy h:mm:ss tt',// month/day pattern'M' - 'MMMM dd',// month/year pattern'Y' - 'yyyy MMMM',// S is a sortable format that does not vary by culture'S' - 'yyyy'-'MM'-'dd'T'HH':'mm':'ss'Date format strings:'d'-the day of the month;'dd'-the day of the month'ddd'-the abbreviated name of the day of the week'dddd'- the full name of the day of the week'h'-the hour, using a 12-hour clock from 1 to 12'hh'-the hour, using a 12-hour clock from 01 to 12'H'-the hour, using a 24-hour clock from 0 to 23'HH'- the hour, using a 24-hour clock from 00 to 23'm'-the minute, from 0 through 59'mm'-the minutes,from 00 though59'M'- the month, from 1 through 12'MM'- the month, from 01 through 12'MMM'-the abbreviated name of the month'MMMM'-the full name of the month's'-the second, from 0 through 59'ss'-the second, from 00 through 59't'- the first character of the AM/PM designator'tt'-the AM/PM designator'y'- the year, from 0 to 99'yy'- the year, from 00 to 99'yyy'-the year, with a minimum of three digits'yyyy'-the year as a four-digit number;'yyyyy'-the year as a four-digit number.
   * Default value: null
   */
  dateFormat?: any;
  /**
   * Applies only to column's that display numbers. This property allows to define a JS Intl.NumberFormat object in order to format the numbers of the column. Another option is to use a number format string. Number format strings: 'd' - decimal numbers.'f' - floating-point numbers.'n' - integer numbers.'c' - currency numbers.'p' - percentage numbers.For adding decimal places to the numbers, add a number after the formatting striFor example: 'c3' displays a number in this format $25.256
   * Default value: null
   */
  numberFormat?: any;
  /**
   * Column's label.
   * Default value: 
   */
  label?: string | null;
  /**
   * Column's value.
   * Default value: 
   */
  value?: string | null;
  /**
   * Column's size.
   * Default value: 0
   */
  size?: string | number | null;
  /**
   * Column's min width.
   * Default value: 0
   */
  minWidth?: string | number | null;
  /**
   * Column's format function. You can use it to customize the column label's rendering.
   * Default value: null
   */
  formatFunction?: any;
  /**
   * A function that allows to set a custom editor for the column's value in the Editor Window. The function must return an HTMLElement. The function has two arguments: name - the name of the task property that is going to be edited.value - the value of the task property that is going to be edited. It's also important to define a getCustomEditorValue to return the value from the custom editor.
   * Default value: null
   */
  customEditor?: any;
  /**
   * A function that is used in conjunction with customEditor allows to return the value of the custom editor. Since the editor is unknown by the element, this function allows to return the expected result from the editor. It has one argument - an HTMLElement that contains the custom editor that was previously defined by the customEditor function.
   * Default value: null
   */
  getCustomEditorValue?: any;
  /**
   * A function that is used in conjunction with customEditor allows to set the value of the custom editor. Since the editor is unknown by the element, this function allows to set the expected value to the editor. It has three arguments - editor - an HTMLElement that contains the custom editor that was previously defined by the customEditor function.columnValue - the value property of the column.value - the value of the task property that the column displays(the editor value).
   * Default value: null
   */
  setCustomEditorValue?: any;
}

/**Determines whether the tooltips are enabled or not. Tooltips are available for timeline tasks, resources, connections, indicators and segments. */
export interface GanttChartTooltip {
  /**
   * Determines whether the tooltip will have an arrow or not.
   * Default value: false
   */
  arrow?: boolean;
  /**
   * Determines the delay (in miliseconds) before the tooltip is opened.
   * Default value: 50
   */
  delay?: number;
  /**
   * Enabled or disables the tooltips.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Determines the [horizontal, vertical] offset (in pixels) for the tooltip position when opened.
   * Default value: 
   */
  offset?: number[];
}

declare global {
    interface Document {
        createElement(tagName: "smart-gantt-chart"): GanttChart;
        querySelector(selectors: "smart-gantt-chart"): GanttChart | null;
        querySelectorAll(selectors: "smart-gantt-chart"): NodeListOf<GanttChart>;
        getElementsByTagName(qualifiedName: "smart-gantt-chart"): HTMLCollectionOf<GanttChart>;
        getElementsByName(elementName: "smart-gantt-chart"): NodeListOf<GanttChart>;
    }
}

/**Determines the type of items that is going to be exported.  */
export declare type GanttChartDataExportItemType = 'task' | 'resource';
/**Determines the format of the dates in the timeline header when they represent days. */
export declare type GanttDayFormat = '2-digit' | 'numeric' | 'long' | 'short' | 'narrow';
/**Determines in what unit is task duration property measured. */
export declare type Duration = 'day' | 'hour' | 'minute' | 'second' | 'milisecond';
/**Determines the format of the dates inside the timeline header when they represent hours. */
export declare type HourFormat = 'default' | '2-digit' | 'numeric';
/**Determines the scale in Month view. */
export declare type MonthScale = 'day' | 'week';
/**Determines how the capacity of the resources will be visualized inside the resource timeline. By default, the capacity is measured in hours depending on the <b>view</b> property of the element. */
export declare type GanttChartResourceTimelineMode = 'diagram' | 'histogram' | 'custom';
/**Determines how the resources will be displayed inside the resource Timeline. */
export declare type GanttChartResourceTimelineView = 'hours' | 'tasks' | 'custom';
/**Determines whether the GanttChart can be sorted by one, more then one or no columns. */
export declare type GanttChartSortMode = 'none' | 'one' | 'many';
/**Project, Task or Milestone type. Possible values are 'project', 'milestone' and 'task' */
export declare type GanttChartTaskType = 'project' | 'milestone' | 'task';
/**Determines the viewing date range of the timeline. Possible values: day - The timeline show the hours of the day.
week - the timeline shows the days of the week.
month - the timeline shows the days of the month.
year - the timeline shows the months of the year.
resource - displays the current tasks by grouping them according to the resources they have assigned. The unassigned tasks will be placed in a separate group called 'Unassigned'.
 <br /> The timeline has a header section that contains the labels of each cell according to the date inside them. The header is splitted in two sections in order to give a more detailed information of the dates. */
export declare type GanttChartView = 'day' | 'week' | 'month' | 'quarter' | 'year';
/**Determines the format of the dates inside the timeline header when they represent weeks.  */
export declare type WeekFormat = 'long' | 'numeric';
export interface GaugeProperties {
  /**
   * Determines the type of gauge's indicator.
   * Default value: needle
   */
  analogDisplayType?: GaugeAnalogDisplayType | string;
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Sets or gets gauge's animation duration. Applicable only when animation is not 'none'.
   * Default value: 300
   */
  animationDuration?: number;
  /**
   * With the coerce property true, the value is set to the nearest value allowed by the interval property. 
   * Default value: false
   */
  coerce?: boolean;
  /**
   * Sets or gets whether custom ticks at (possibly) uneven interval will be plotted. The ticks to be plotted are defined with the property customTicks.
   * Default value: false
   */
  customInterval?: boolean;
  /**
   * If customInterval is enabled, sets a list of ticks to be plotted. If coerce is set to true, the value will snap to these ticks.
   * Default value: 0,50,100
   */
  customTicks?: number[];
  /**
   * Determines the date pattern of the labels when mode is 'date'.
   * Default value: "d"
   */
  dateLabelFormatString?: string;
  /**
   * Sets or gets the char to use as the decimal separator in numeric values. 
   * Default value: "".""
   */
  decimalSeparator?: string;
  /**
   * Enables or disables the digital display of the element.
   * Default value: false
   */
  digitalDisplay?: boolean;
  /**
   * Sets the position of the digital display inside the element.
   * Default value: bottom
   */
  digitalDisplayPosition?: GaugeDigitalDisplayPosition | string;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Callback function which allows rendering of a custom needle. Applicable only to analogDisplayType needle.
   * Default value: null
   */
  drawNeedle?: any;
  /**
   * Sets or gets Gauge's end angle. This property specifies the end of the gauge's scale and is measured in degrees.
   * Default value: 210
   */
  endAngle?: number;
  /**
   * When cooerce property is true, all values coerce to the interval's value.
   * Default value: 1
   */
  interval?: number;
  /**
   * Sets the direction of the gauge. If true - the positions of the gauge's start and end are switched.
   * Default value: false
   */
  inverted?: boolean;
  /**
   * A callback function that can be used to format the values displayed inside the gauge labels.
   * Default value: null
   */
  labelFormatFunction?: any;
  /**
   * Determines the visibility of the labels inside the element.
   * Default value: all
   */
  labelsVisibility?: LabelsVisibility | string;
  /**
   * Sets or gets the locale. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Enables or disables the usage of logarithmic scale in the element.
   * Default value: false
   */
  logarithmicScale?: boolean;
  /**
   * Determines the maximum value for the scale of the element.
   * Default value: 100
   */
  max?: number;
  /**
   * Determines when the value of the element is updated.
   * Default value: switchWhileDragging
   */
  mechanicalAction?: DragMechanicalAction | string;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "significantPrecisionDigits": ": the properties significantDigits and precisionDigits cannot be set at the same time.",
   *     "invalidMinOrMax": " value. Max cannot be lower than Min.",
   *     "noInteger": ": precisionDigits could be set only on \"floatingPoint\" scaleType."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the minimum value for the scale of the element. 
   * Default value: 0
   */
  min?: number;
  /**
   * Determines whether the element works with numbers or dates.
   * Default value: numeric
   */
  mode?: ScaleMode | string;
  /**
   * Sets or gets the element's name, which is used as a reference when the data is submitted.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines the position of the needle when analogDisplayType is 'needle'. 
   * Default value: center
   */
  needlePosition?: GaugeNeedlePosition | string;
  /**
   * Determines the number of digits after the decimal point. Applicable only when scaleType is 'floatingPoint'.
   * Default value: null
   */
  precisionDigits?: number;
  /**
   * This property represents an array of objects. Each object is a different range. The range is a colored area with specific size.
   * Default value: 
   */
  ranges?: {startValue: number, endValue: number, className: string}[];
  /**
   * When the element is read only the users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts. If enabled, the scale is inverted and the labels and digital display are oriented from right to left.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the position of the scale in the element. 
   * Default value: inside
   */
  scalePosition?: GaugeScalePosition | string;
  /**
   * Determines the type of the gauge's value and scale. 
   * Default value: floatingPoint
   */
  scaleType?: ScaleType | string;
  /**
   * Enables or disables scientific notation.
   * Default value: false
   */
  scientificNotation?: boolean;
  /**
   * This property indicates whether the gauge ranges are visible or not.
   * Default value: false
   */
  showRanges?: boolean;
  /**
   * Enables or disables displaying of units.
   * Default value: false
   */
  showUnit?: boolean;
  /**
   * Determining how many significant digits are in a number. Applicable only when scaleType is 'integer'.
   * Default value: null
   */
  significantDigits?: number | null;
  /**
   * Determines how the Gauge will size.
   * Default value: circle
   */
  sizeMode?: GaugeSizeMode | string;
  /**
   * Sets or gets gauge's start angle. This property specifies the beggining of the gauge's scale and is measured in degrees.
   * Default value: -30
   */
  startAngle?: number;
  /**
   * Sets or gets the element's visual theme.
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines the position of the ticks in the Gauge.
   * Default value: scale
   */
  ticksPosition?: TicksPosition | string;
  /**
   * Determines the visibility of the ticks.
   * Default value: minor
   */
  ticksVisibility?: TicksVisibility | string;
  /**
   * Sets or gets if the element can be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the name of unit used for the values on the scale of the element.
   * Default value: "kg"
   */
  unit?: string;
  /**
   * Sets the value's validation by min/max.
   * Default value: strict
   */
  validation?: Validation | string;
  /**
   * Sets or gets the value of the element. The value can be a date only when scaleType is 'date'.
   * Default value: 0
   */
  value?: string | number | Date;
  /**
   * Sets or gets the word length. Applicable only when scaleType is 'integer'.
   * Default value: int32
   */
  wordLength?: WordLength | string;
}
/**
 Gauge displays an indicator within a range of values.
*/
export interface Gauge extends BaseElement, GaugeProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the value of the element is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value of the element.
   *  value - The new value of the element.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Focuses the element.
   */
  focus(): void;
  /**
   * Gets the optimal size of the element (the current width and the height based on the plotted internal elements). 
   * @returns {any}
   */
  getOptimalSize(): any;
  /**
   * Get/set the value of the gauge.
   * @param {string | number | Date} value?. The value to be set. If no parameter is passed, returns the current value of the gauge. The value can be a date only when <b>scaleType</b> is 'date'.
   * @returns {string}
   */
  val(value?: string | number | Date): string;
}

declare global {
    interface Document {
        createElement(tagName: "smart-gauge"): Gauge;
        querySelector(selectors: "smart-gauge"): Gauge | null;
        querySelectorAll(selectors: "smart-gauge"): NodeListOf<Gauge>;
        getElementsByTagName(qualifiedName: "smart-gauge"): HTMLCollectionOf<Gauge>;
        getElementsByName(elementName: "smart-gauge"): NodeListOf<Gauge>;
    }
}

/**Determines the type of gauge's indicator. */
export declare type GaugeAnalogDisplayType = 'needle' | 'fill' | 'line';
/**Sets the position of the digital display inside the element. */
export declare type GaugeDigitalDisplayPosition = 'bottom' | 'center' | 'right' | 'top';
/**Determines the visibility of the labels inside the element. */
export declare type LabelsVisibility = 'all' | 'endPoints' | 'none';
/**Determines when the value of the element is updated. */
export declare type DragMechanicalAction = 'switchUntilReleased' | 'switchWhenReleased' | 'switchWhileDragging';
/**Determines whether the element works with numbers or dates. */
export declare type ScaleMode = 'date' | 'numeric';
/**Determines the position of the needle when analogDisplayType is 'needle'.  */
export declare type GaugeNeedlePosition = 'center' | 'edge';
/**Determines the position of the scale in the element.  */
export declare type GaugeScalePosition = 'inside' | 'outside' | 'none';
/**Determines the type of the gauge's value and scale.  */
export declare type ScaleType = 'floatingPoint' | 'integer';
/**Determines how the Gauge will size. */
export declare type GaugeSizeMode = 'auto' | 'circle' | 'none';
/**Determines the position of the ticks in the Gauge. */
export declare type TicksPosition = 'scale' | 'track';
/**Determines the visibility of the ticks. */
export declare type TicksVisibility = 'major' | 'minor' | 'none';
/**Sets or gets the word length. Applicable only when scaleType is 'integer'. */
export declare type WordLength = 'int8' | 'int16' | 'int32' | 'int64' | 'uint8' | 'uint16' | 'uint32' | 'uint64';
export interface GridProperties {
  /**
   * An object containing settings related to the grid's appearance.
   * Default value: [object Object]
   */
  appearance?: GridAppearance;
  /**
   * An object containing settings related to the grid's behavior.
   * Default value: [object Object]
   */
  behavior?: GridBehavior;
  /**
   * An object containing settings related to the grid's layout.
   * Default value: [object Object]
   */
  layout?: GridLayout;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * The clipboard property is used to enable/disable clipboard operations with Ctrl+C, Ctrl+X and Ctrl+V keyboard navigations..
   * Default value: [object Object]
   */
  clipboard?: GridClipboard;
  /**
   * The columns property is used to describe all columns displayed in the grid. 
   * Default value: []
   */
  columns?: {label: string, dataField: string}[] | string[] | number | GridColumn[];
  /**
   * Context Menu is the drop-down menu displayed after right-clicking a Grid row. It allows you to delete row, edit cell or row depending on the edit mode. The 'contextMenuItemCustom' dataSource option allows you to add custom menu item to the context menu. You can replace the context menu by using the 'selector' property and setting it to ID of a Smart.Menu component.
   * Default value: [object Object]
   */
  contextMenu?: GridContextMenu;
  /**
   * Column Menu is the drop-down menu displayed after clicking the column header's drop-down button, which is displayed when you hover the column header. It allows you to customize column settings. For example: Sort, Filter or Group the Grid by the current column.
   * Default value: [object Object]
   */
  columnMenu?: GridColumnMenu;
  /**
   * Describes the settings of the column groups.
   * Default value: 
   */
  columnGroups?: GridColumnGroup[];
  /**
   * Sets or gets details about conditional formatting to be applied to the Grid's cells.
   * Default value: null
   */
  conditionalFormatting?: GridConditionalFormatting[];
  /**
   * Sets the Grid Charting Data Visualization.
   * Default value: [object Object]
   */
  charting?: GridCharting;
  /**
   * Sets the TreeGrid checkboxes.
   * Default value: [object Object]
   */
  checkBoxes?: GridCheckBoxes;
  /**
   * Sets the Grid Data Export options.
   * Default value: [object Object]
   */
  dataExport?: GridDataExport;
  /**
   * Sets the grid's data source. The value of dataSource can be an instance of JQX.DataAdapter or an Array.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Sets the grid's data source settings when the dataSource property is set to an Array or URL.
   * Default value: [object Object]
   */
  dataSourceSettings?: GridDataSourceSettings;
  /**
   * Describes the grid's editing settings.
   * Default value: [object Object]
   */
  editing?: GridEditing;
  /**
   * Describes the grid's filtering settings.
   * Default value: [object Object]
   */
  filtering?: GridFiltering;
  /**
   * Describes the grid's grouping settings.
   * Default value: [object Object]
   */
  grouping?: GridGrouping;
  /**
   * Sets the messages values.
   * Default value:    * [object Object]
   */
  messages?: any;
  /**
   * Callback function, which is called when the Grid needs a cell value to render it. When you implement this function, you can override the default cell value rendering.
   * Default value: null
   */
  onCellValue?: {(cell: GridCell): void};
  /**
   * Callback function, which is called when a cell value will be updated. This function is useful if you want to make Ajax call to a server to validate the cell changes. If the changes are validated, invoke the confirm function.
   * Default value: null
   */
  onCellUpdate?: {(cells: GridCell[], oldValues: any[], values: any[], confirm: {(commit: boolean): void}): void};
  /**
   * Callback function, which is called when a cell is rendered. This function is useful if you want to customize GridCell properties, before the cell is rendered.
   * Default value: null
   */
  onCellRender?: {(cell: GridCell): void};
  /**
   * Callback function() called before the grid has been initialized and the Grid's Virtual DOM is not created.
   * Default value: null
   */
  onBeforeInit?: {(): void};
  /**
   * Callback function() called when the grid is initializing and the Grid's Virtual DOM is created.
   * Default value: null
   */
  onInit?: {(): void};
  /**
   * Callback function() called after the grid has been initialized and the Grid's Virtual DOM is created.
   * Default value: null
   */
  onAfterInit?: {(): void};
  /**
   * Callback function(chart: JQX.Chart) called when the chart has been initialized. You can use this function to customize the Chart element settings.
   * Default value: null
   */
  onChartInit?: any;
  /**
   * Callback function() called when the grid has been rendered.
   * Default value: null
   */
  onRender?: any;
  /**
   * Callback function() called when the grid has been rendered for first time and bindings are completed. The component is ready.
   * Default value: null
   */
  onLoad?: any;
  /**
   * Callback function(event: KeyboardEvent) called when the grid is on focus and a keyboard key is pressed.
   * Default value: null
   */
  onKey?: {(event: KeyboardEvent): void};
  /**
   * Callback function, which is called when a row is initializing. This function can be used to customize the row settings.
   * Default value: null
   */
  onRowInit?: {(index: number, row: GridRow): void};
  /**
   * Callback function, which is called when a row detail is initializing. Row details are displayed below the row's data or in a separate dialog.
   * Default value: null
   */
  onRowDetailInit?: {(index: number, row: GridRow, details: HTMLElement): void};
  /**
   * Callback function, which is called when a row detail is udpating.
   * Default value: null
   */
  onRowDetailUpdated?: {(index: number, row: GridRow, details: HTMLElement): void};
  /**
   * Callback function which is called when a row history is updated. The row history for edits is recorded when the 'storeHistory' property is enabled.
   * Default value: null
   */
  onRowHistory?: {(index: number, row: GridRow, history: any[]): void};
  /**
   * Callback function which is called when a row style is updated. The row style can be changed by using the row dialog or the 'setRowStyle' method.
   * Default value: null
   */
  onRowStyle?: {(index: number, row: GridRow, history: any[]): void};
  /**
   * Callback function which is called when a row has been inserted.
   * Default value: null
   */
  onRowInserted?: {(index: number[], row: GridRow[]): void};
  /**
   * Callback function, which is called when a row has been removed.
   * Default value: null
   */
  onRowRemoved?: {(indexes: number[], rows: GridRow[]): void};
  /**
   * Callback function, which is called when row's cell values will be updated. This function is useful if you want to make Ajax call to a server to validate the edit changes. If the changes are validated, invoke the confirm function.
   * Default value: null
   */
  onRowUpdate?: {(index: number[], row: GridRow[], oldValues: any[], values: any[], confirm: {(commit: boolean): void}): void};
  /**
   * Callback function, called when a row has been updated.
   * Default value: null
   */
  onRowUpdated?: {(index: number[], row: GridRow[]): void};
  /**
   * Callback function called by the Grid when defined. It is used to get the CSS class applied to a row.
   * Default value: null
   */
  onRowClass?: {(index: number, data: any, row: GridRow[]): void};
  /**
   * Callback function called by the Grid when defined. It is used to get the CSS class applied to a cell.
   * Default value: null
   */
  onCellClass?: {(index: number, dataField: string, cellValue: any, data: any, row: GridRow[]): void};
  /**
   * Callback function, which is called when a column has been initialized. This function can be used to customize the column settings.
   * Default value: null
   */
  onColumnInit?: {(index: number, column: GridColumn): void};
  /**
   * Callback function, which called when a column has been inserted.
   * Default value: null
   */
  onColumnInserted?: {(index: number, column: GridColumn): void};
  /**
   * Callback function, which is called when a column has been removed.
   * Default value: null
   */
  onColumnRemoved?: {(index: number, column: GridColumn): void};
  /**
   * Callback function, which is called when a column has been updated.
   * Default value: null
   */
  onColumnUpdated?: {(index: number, column: GridColumn): void};
  /**
   * Callback function, which is called when a column has been cloned.
   * Default value: null
   */
  onColumnClone?: {(dataField: string, cloneColumnDataField: string, index: number, duplicateCells: boolean): void};
  /**
   * Callback function, which is called when a command is executed. The name argument is the command's name. The command argument is the command's function. details are built in command arguments passed by the Grid. The handled parameter allows you to cancel built-in command, because when you set it to true the Grid will not execute the default command's behavior.
   * Default value: null
   */
  onCommand?: {(name: string, command: any, details: GridCell, event: Event | KeyboardEvent | PointerEvent, handled: boolean): void};
  /**
   * Sets or gets the rows  CSS class rules. Different CSS class names are conditionally applied. Example: rowCSSRules: { 'cell-class-1': settings =&gt; settings.data.quantity === 5, 'cell-class-2': settings =&gt; settings.data.quantity &lt; 5, 'cell-class-3': settings =&gt; settings.data.quantity &gt; 5 }.  The settings object contains the following properties: index, data, row, api.
   * Default value: null
   */
  rowCSSRules?: any;
  /**
   * Sets or gets the id of the current user. Has to correspond to the id of an item from the users property/array. Depending on the current user, different privileges are enabled. If no current user is set, privileges depend on the element's properties.
   * Default value: 
   */
  currentUser?: string | number;
  /**
   * Sets the grid users. Expects an array with 'id', 'name' and optionally 'color' and 'image' properties.
   * Default value: []
   */
  users?: any[];
  /**
   * Sets the grid's image and filter upload settings for the image and attachment columns.
   * Default value: [object Object]
   */
  uploadSettings?: GridUploadSettings;
  /**
   * Describes the paging settings.
   * Default value: [object Object]
   */
  paging?: GridPaging;
  /**
   * Describes the pager settings.
   * Default value: [object Object]
   */
  pager?: GridPager;
  /**
   * Sets the row details.
   * Default value: [object Object]
   */
  rowDetail?: GridRowDetail;
  /**
   * Sets the scroll mode settings.
   * Default value: physical
   */
  scrolling?: Scrolling | string;
  /**
   * Describes the column header settings.
   * Default value: [object Object]
   */
  columnHeader?: GridColumnHeader;
  /**
   * Describes the summary row settings.
   * Default value: [object Object]
   */
  summaryRow?: GridSummaryRow;
  /**
   * Sets the grid's state settings.
   * Default value: [object Object]
   */
  stateSettings?: GridStateSettings;
  /**
   * Describes the settings for the group header.
   * Default value: [object Object]
   */
  groupHeader?: GridGroupHeader;
  /**
   * Describes the header settings of the grid.
   * Default value: [object Object]
   */
  header?: GridHeader;
  /**
   * Describes the footer settings of the grid.
   * Default value: [object Object]
   */
  footer?: GridFooter;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * The rows property is used to describe all rows displayed in the grid.
   * Default value: 
   */
  rows?: GridRow[];
  /**
   * Describes the selection settings.
   * Default value: [object Object]
   */
  selection?: GridSelection;
  /**
   * Describes sorting settings.
   * Default value: [object Object]
   */
  sorting?: GridSorting;
}
/**
 Data Grid UI Component that covers everything from paging, sorting, grouping, filtering, and editing to row and column virtualization, right-to-left layout, export to Excel and PDF and Accessibility.
*/
export interface Grid extends BaseElement, GridProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered, when the edit begins. After the event occurs, editing starts. If you need to prevent the editing for specific cells, rows or columns, you can call event.preventDefault();.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, dataField, row, column, cell, data, value)
   *  id - The edited row id.
   *  dataField - The edited column data field.
   *  row - The edited row.
   *  column - The edited column.
   *  cell - The edited cell.
   *  data - The edited row's data.
   *  value - The edited cell's value.
   */
  onBeginEdit?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the Grid's header toolbar is displayed and the 'OK' button of a header dropdown is clicked. For example, when you open the columns customize panel, reorder columns and click the 'OK' button.
	* @param event. The custom event. Custom data event was created with: ev.detail(type)
   *  type - The type of dropdown. Possible values: 'filter', 'sort', 'search', 'group', 'format', 'customize'
   */
  onBatchChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the Grid's header toolbar is displayed and the 'Cancel' button of a header dropdown is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(type)
   *  type - The type of dropdown. Possible values: 'filter', 'sort', 'search', 'group', 'format', 'customize'
   */
  onBatchCancel?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the selection is changed. When you select with a drag, the event is triggered when the drag starts and ends. 
	* @param event. The custom event. Custom data event was created with: ev.detail(started, finished, originalEvent)
   *  started - The flag is <em>true</em>, when the selection starts. The flag is <em>false</em>, when the selection ends and when the user changes the selection by dragging.
   *  finished - The flag is <em>true</em>, when the selection ends. The flag is <em>false</em>, when the selection starts and when the user changes the selection by dragging.
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered, when the user clicks on the header of a column.
	* @param event. The custom event. Custom data event was created with: ev.detail(column, dataField, originalEvent)
   *  column - The clicked column.
   *  dataField - The column's data field.
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   */
  onColumnClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user double clicks on the header of a column.
	* @param event. The custom event. Custom data event was created with: ev.detail(column, dataField, originalEvent)
   *  column - The double-clicked column.
   *  dataField - The column's data field.
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   */
  onColumnDoubleClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user resized a column.
	* @param event. The custom event. Custom data event was created with: ev.detail(column, dataField, oldWidth, width)
   *  column - The resized column.
   *  dataField - The column's data field.
   *  oldWidth - The old width of the column.
   *  width - The new width of the column.
   */
  onColumnResize?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user starts a column drag.
	* @param event. The custom event. Custom data event was created with: ev.detail(column, dataField, index, originalEvent)
   *  column - The column.
   *  dataField - The column's data field.
   *  index - The column's index
   *  originalEvent - The origianl Event object.
   */
  onColumnDragStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user drags a column.
	* @param event. The custom event. Custom data event was created with: ev.detail(column, dataField, index, data, originalEvent)
   *  column - The column.
   *  dataField - The column's data field.
   *  index - The column's index
   *  data - The dragging object. data.feedback and data.feedbackLine are HTML Elements which are displayed while the user drags. The object has error(), success() and data() methods which you can call to set the feedback state.
   *  originalEvent - The origianl Event object.
   */
  onColumnDragging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user drops a column.
	* @param event. The custom event. Custom data event was created with: ev.detail(column, dataField, index, newIndex, data, originalEvent)
   *  column - The column.
   *  dataField - The column's data field.
   *  index - The column's index
   *  newIndex - The column's new index
   *  data - The dragging object. data.feedback and data.feedbackLine are HTML Elements which are displayed while the user drags. The object has error(), success() and data() methods which you can call to set the feedback state.
   *  originalEvent - The origianl Event object.
   */
  onColumnDragEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user reorders a column.
	* @param event. The custom event. Custom data event was created with: ev.detail(column, dataField, index, newIndex, data, originalEvent)
   *  column - The column.
   *  dataField - The column's data field.
   *  index - The column's index
   *  newIndex - The column's new index
   *  data - The dragging object. data.feedback and data.feedbackLine are HTML Elements which are displayed while the user drags. The object has error(), success() and data() methods which you can call to set the feedback state.
   *  originalEvent - The origianl Event object.
   */
  onColumnReorder?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user enters a comment in the row edit dialog.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, comment)
   *  id - The row's id.
   *  comment - The comment object. The comment object has 'text: string', 'id: string', 'userId: string | number', and 'time: date' fields. The 'text' is the comment's text. 'id' is the comment's unique id, 'userId' is the user's id who entered the comment and 'time' is a javascript date object.
   */
  onCommentAdd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user removes a comment in the row edit dialog.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, comment)
   *  id - The row's id.
   *  comment - The comment object. The comment object has 'text: string', 'id: string', 'userId: string | number', and 'time: date' fields. The 'text' is the comment's text. 'id' is the comment's unique id, 'userId' is the user's id who entered the comment and 'time' is a javascript date object.
   */
  onCommentRemove?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user clicks on a context menu item.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, dataField, command)
   *  id - The row's id.
   *  dataField - The column's data field.
   *  command - Command function.
   */
  onContextMenuItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user starts a row drag.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, id, index, originalEvent)
   *  row - The row.
   *  id - The row's id
   *  index - The row's index
   *  originalEvent - The origianl Event object.
   */
  onRowDragStart?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user drags a row.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, id, index, data, originalEvent)
   *  row - The row.
   *  id - The row's id
   *  index - The row's index
   *  data - The dragging object. data.feedback and data.feedbackLine are HTML Elements which are displayed while the user drags. The object has error(), success() and data() methods which you can call to set the feedback state.
   *  originalEvent - The origianl Event object.
   */
  onRowDragging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user drags a row.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, id, index, newIndex, data, originalEvent)
   *  row - The row.
   *  id - The row's id
   *  index - The row's index
   *  newIndex - The row's new index
   *  data - The dragging object. data.feedback and data.feedbackLine are HTML Elements which are displayed while the user drags. The object has error(), success() and data() methods which you can call to set the feedback state.
   *  originalEvent - The origianl Event object.
   */
  onRowDragEnd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user reorders a row.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, id, index, newIndex, data, originalEvent)
   *  row - The row.
   *  id - The row's id
   *  index - The row's index
   *  newIndex - The row's new index
   *  data - The dragging object. data.feedback and data.feedbackLine are HTML Elements which are displayed while the user drags. The object has error(), success() and data() methods which you can call to set the feedback state.
   *  originalEvent - The origianl Event object.
   */
  onRowReorder?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user expands a row of the grid. The Grid is in TreeGrid/Grouping mode.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, id, originalEvent)
   *  row - The expanded row.
   *  id - The row's id
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   */
  onRowExpand?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user collapsed a row of the grid. The Grid is in TreeGrid/Grouping mode.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, id, originalEvent)
   *  row - The collapsed row. 
   *  id - The row's id
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   */
  onRowCollapse?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user clicks on a row of the grid.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, originalEvent, id, isRightClick, pageX, pageY)
   *  row - The clicked row.
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   *  id - Gets the row id.
   *  isRightClick - Gets whether the pointing device's right button is clicked.
   *  pageX - Gets the click's X position.
   *  pageY - Gets the click's Y position.
   */
  onRowClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user double clicks on a row of the grid.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, originalEvent, id, isRightClick, pageX, pageY)
   *  row - The double-clicked row.
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   *  id - Gets the row id.
   *  isRightClick - Gets whether the pointing device's right button is clicked.
   *  pageX - Gets the click's X position.
   *  pageY - Gets the click's Y position.
   */
  onRowDoubleClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user resized a row.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, id, oldHeight, height)
   *  row - The resized row.
   *  id - Gets the row id.
   *  oldHeight - The old height of the row.
   *  height - The new height of the row.
   */
  onRowResize?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user clicks on the row header's star.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, originalEvent, id, value)
   *  row - The clicked row.
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   *  id - Gets the row id.
   *  value - Gets whether the row is starred or not.
   */
  onRowStarred?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user clicks on a cell of the grid.
	* @param event. The custom event. Custom data event was created with: ev.detail(cell, originalEvent, id, dataField, isRightClick, pageX, pageY)
   *  cell - The clicked cell.
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   *  id - Gets the row id.
   *  dataField - Gets the column dataField.
   *  isRightClick - Gets whether the pointing device's right button is clicked.
   *  pageX - Gets the click's X position.
   *  pageY - Gets the click's Y position.
   */
  onCellClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user double clicks on a cell of the grid.
	* @param event. The custom event. Custom data event was created with: ev.detail(cell, originalEvent, id, dataField, isRightClick, pageX, pageY)
   *  cell - The double-clicked cell. 
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   *  id - Gets the row id.
   *  dataField - Gets the column dataField.
   *  isRightClick - Gets whether the pointing device's right button is clicked.
   *  pageX - Gets the click's X position.
   *  pageY - Gets the click's Y position.
   */
  onCellDoubleClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the edit ends.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, dataField, row, column, cell, data, value)
   *  id - The edited row id.
   *  dataField - The edited column data field.
   *  row - The edited row.
   *  column - The edited column.
   *  cell - The edited cell.
   *  data - The edited row's data.
   *  value - The edited cell's value.
   */
  onEndEdit?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when a filter is added or removed.
	* @param event. The custom event. Custom data event was created with: ev.detail(columns, data, expressions)
   *  columns - Array of columns.
   *  data - Array of {dataField: string, filter: object}. <em>dataField</em> is the column's data field. <em>filter</em> is a FilterGroup object.
   *  expressions - Array of {dataField: string, filter: string}. <em>dataField</em> is the column's data field. <em>filter</em> is a filter expression like 'startsWith B'. In each array item, you will have an object with column's name and filter string. Example: [['firstName', 'contains Andrew or contains Nancy'], ['quantity', '&lt;= 3 and &gt;= 8']], [['firstName', 'EQUAL' 'Andrew' or 'EQUAL' 'Antoni' or 'EQUAL' 'Beate']], [['lastName','CONTAINS' 'burke' or 'CONTAINS' 'peterson']]. Filter conditions used in the filter expressions: '=', 'EQUAL','&lt;&gt;', 'NOT_EQUAL', '!=', '&lt;', 'LESS_THAN','&gt;', 'GREATER_THAN', '&lt;=', 'LESS_THAN_OR_EQUAL', '&gt;=', 'GREATER_THAN_OR_EQUAL','starts with', 'STARTS_WITH','ends with', 'ENDS_WITH', '', 'EMPTY', 'CONTAINS','DOES_NOT_CONTAIN', 'NULL','NOT_NULL'
   */
  onFilter?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the rows grouping is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(groups)
   *  groups - Array of column data fields.
   */
  onGroup?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the add new column dialog is opened.
	* @param event. The custom event. Custom data event was created with: ev.detail(dataField)
   *  dataField - The column data field.
   */
  onOpenColumnDialog?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the add new column dialog is closed.
	* @param event. The custom event. Custom data event was created with: ev.detail(dataField)
   *  dataField - The column data field.
   */
  onCloseColumnDialog?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the grid is resized.
	* @param event. The custom event.    */
  onResize: ((ev: Event) => any) | null;
  /**
   * This event is triggered when the user touches and holds on the row for at least 300ms.
	* @param event. The custom event. Custom data event was created with: ev.detail(row, originalEvent)
   *  row - The tapped row.
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   */
  onRowTap?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user touches and holds on the cell for at least 300ms.
	* @param event. The custom event. Custom data event was created with: ev.detail(cell, originalEvent)
   *  cell - The tapped row.
   *  originalEvent - The original event object, which is 'pointer', 'touch' or 'mouse' Event object, depending on the device type and web browser
   */
  onCellTap?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user changes the pages.
	* @param event. The custom event.    */
  onPage?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when a sorting column is added or removed.
	* @param event. The custom event. Custom data event was created with: ev.detail(columns, data, sortDataFields, sortDataTypes, sortOrders, sortIndexes)
   *  columns - Array of columns.
   *  data - Array of {dataField: string, sortOrder: string, sortIndex: number}. <em>dataField</em> is the column's data field. <em>sortOrder</em> is 'asc' or 'desc', <em>sortIndex</em> is the index of the column in multi column sorting.
   *  sortDataFields - Array of column data fields.
   *  sortDataTypes - Array of column data types. The values in the array would be 'string', 'date', 'boolean' or 'number'.
   *  sortOrders - Array of column orders. The values in the array would be 'asc' or 'desc'.
   *  sortIndexes - Array of column sort indexes. When multiple sorting is applied the sort index is an important parameter as it specifies the priority of sorting.
   */
  onSort?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user reaches the bottom of the grid.
	* @param event. The custom event.    */
  onScrollBottomReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered, when the user reaches the top of the grid.
	* @param event. The custom event.    */
  onScrollTopReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Adds a row. When batch editing is enabled, the row is not saved until the batch edit is saved.
   * @param {any} data. row data matching the data source
   * @param {boolean} insertAtBottom?. Determines whether to add the new row to the bottom or top of the collection. The default value is 'true'
   * @param  callback?. Sets a callback function, which is called after the new row is added. The callback's argument is the new row.
   */
  addRow(data: any, insertAtBottom?: boolean, callback?: {(row: GridRow): void}): void;
  /**
   * Adds a new row and puts it into edit mode. When batch editing is enabled, the row is not saved until the batch edit is saved.
   * @param {string} position?. 'near' or 'far'
   * @returns {boolean}
   */
  addNewRow(position?: string): boolean;
  /**
   * Adds a new column.
   * @param {any} column. A Grid column object. See 'columns' property.
   * @returns {boolean}
   */
  addNewColumn(column: any): boolean;
  /**
   * Adds a new unbound row to the top or bottom. Unbound rows are not part of the Grid's dataSource. They become part of the dataSource, after an unbound row is edited.
   * @param {number} count. The count of unbound rows.
   * @param {string} position?. 'near' or 'far'
   * @returns {boolean}
   */
  addUnboundRow(count: number, position?: string): boolean;
  /**
   * Adds a filter to a column. This method will apply a filter to the Grid data. Example for adding multiple filters to a column: grid.addFilter('lastName', ['CONTAINS "burke"', 'or', 'CONTAINS "peterson"']). Example for adding single filter to a column: grid.addFilter('lastName', 'CONTAINS "burke"'). Example for adding numeric filter:  grid.addFilter('quantity', '&lt;= 5')
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @param {string} filter. Filter expression like: 'startsWith B'. Example 2: ['contains Andrew or contains Nancy'], Example 3:  ['quantity', '&lt;= 3 and &gt;= 8'].  Filter conditions which you can use in the expressions: '=', 'EQUAL','&lt;&gt;', 'NOT_EQUAL', '!=', '&lt;', 'LESS_THAN','&gt;', 'GREATER_THAN', '&lt;=', 'LESS_THAN_OR_EQUAL', '&gt;=', 'GREATER_THAN_OR_EQUAL','starts with', 'STARTS_WITH','ends with', 'ENDS_WITH', '', 'EMPTY', 'CONTAINS','DOES_NOT_CONTAIN', 'NULL','NOT_NULL'
   * @param {boolean} refreshFilters?. Set this to false, if you will use multiple 'addFilter' calls. By doing this, you will avoid unnecessary renders.
   */
  addFilter(dataField: string, filter: string, refreshFilters?: boolean): void;
  /**
   * Groups the Grid by a data field. This method will add a group to the Grid when grouping is enabled.
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   */
  addGroup(dataField: string): void;
  /**
   * Sorts the Grid by a data field. This method will add a sorting to the Grid when sorting is enabled.
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @param {string} sortOrder. column's sort order. Use 'asc' or 'desc'.
   */
  addSort(dataField: string, sortOrder: string): void;
  /**
   * Auto-sizes grid rows. This method will update the <em>height</em> of all Grid rows.
   */
  autoSizeRows(): void;
  /**
   * Auto-sizes grid columns. This method will update the <em>width</em> of all Grid columns.
   */
  autoSizeColumns(): void;
  /**
   * Auto-sizes grid column. This method will update the <em>width</em> of a Grid column by measuring the cells and column header label width.
   * @param {string} dataField?. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   */
  autoSizeColumn(dataField?: string): void;
  /**
   * This method returns true, if all rows in the Grid are selected.
   * @returns {boolean}
   */
  areAllRowsSelected(): boolean;
  /**
   * Starts an update operation. This is appropriate when calling multiple methods or set multiple properties at once.
   */
  beginUpdate(): void;
  /**
   * Begins row, cell or column. This method allows you to programmatically start a cell, row or column editing. After calling it, an editor HTMLElement will be created and displayed in the Grid.
   * @param {string | number} rowId. row bound id
   * @param {string} dataField?. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   */
  beginEdit(rowId: string | number, dataField?: string): void;
  /**
   * Clears all filters. Refreshes the view and updates all filter input components.
   */
  clearFilter(): void;
  /**
   * Clears all data groups. Refreshes the view and updates the DataGrid component.
   */
  clearGroups(): void;
  /**
   * Clears all sorting. Refreshes the view and updates the DataGrid component.
   */
  clearSort(): void;
  /**
   * Clears the selection that user have made. All row, cell and column selection highlights will be removed.
   */
  clearSelection(): void;
  /**
   * Cancels the editing. This method closes the cell editor and cancels the changes.
   */
  cancelEdit(): void;
  /**
   * Checks a TreeGrid row. This method updates the row's check-box.
   * @param {string | number} rowId. row bound id
   */
  checkRow(rowId: string | number): void;
  /**
   * Checks all TreeGrid or Grouping rows. This method updates all check-boxes in the TreeGrid or Grouping rows.
   */
  checkAllRows(): void;
  /**
   * Clears the user selection and empties the data source. The Grid will display 'No Rows' in the view.
   */
  clearRows(): void;
  /**
   * Closes the column drop-down menu.
   */
  closeMenu(): void;
  /**
   * Collapses a TreeGrid or Grouping row.
   * @param {string | number} rowId. row bound id
   */
  collapseRow(rowId: string | number): void;
  /**
   * Collapses all TreeGrid or Grouping rows.
   */
  collapseAllRows(): void;
  /**
   * Creates a Chart, when charting is enabled.
   * @param {string} type. Chart's type
   * @param {any} dataSource?. Chart's data source
   */
  createChart(type: string, dataSource?: any): void;
  /**
   * Delete a row. When batch editing is enabled, the row is not saved until the batch edit is saved.
   * @param {string | number} rowId. row bound id
   * @param  callback?. Sets a callback function, which is called after the row is deleted. The callback's argument is the deleted row.
   */
  deleteRow(rowId: string | number, callback?: {(row: GridRow): void}): void;
  /**
   * Scrolls to a row or cell. This method scrolls to a row or cell, when scrolling is necessary. If pagination is enabled, it will automatically change the page.
   * @param {string | number} rowId. row bound id
   * @param {string} dataField?. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @returns {boolean}
   */
  ensureVisible(rowId: string | number, dataField?: string): boolean;
  /**
   * Ends the editing. This method confirms all changes and closes the opened cell editor(s).
   */
  endEdit(): void;
  /**
   * Ends the update operation. This method will resume the rendering and will refresh the Grid.
   * @param {boolean} refresh?. The flag that control the calls of the refresh method.
   */
  endUpdate(refresh?: boolean): void;
  /**
   * Expands a TreeGrid or Grouping row. For example, if you want to expand the first group, then its second sub grup, then the first sub sub group, you can use: grid.expandRow('0.1.0');
   * @param {string | number} rowId. row bound id
   */
  expandRow(rowId: string | number): void;
  /**
   * Expands rows to a given group level. For example 'grid.expandRowsToGroupLevel(1);' means that all groups at the root level will be expanded.
   * @param {number} level. row group level
   */
  expandRowsToGroupLevel(level: number): void;
  /**
   * Expands all TreeGrid or Grouping rows.
   */
  expandAllRows(): void;
  /**
   * Exports the Grid data to .XLSX, .PDF, .JSON, .XML, .CSV, .TSV, .HTML, .JPEG or .PNG. The method uses the options of the <em>dataExport</em> property.
   * @param {string} Dataformat. 'xlsx', 'pdf', 'json', 'xml', 'csv', 'tsv', 'html', 'png', 'jpeg'.
   */
  exportData(Dataformat: string): void;
  /**
   * Finds entries by using a query and returns an array of row ids. Example: const rows = grid.find('nancy'); returns all rows that have 'nancy' value. Example 2: const rows = grid.find('nancy, davolio'); returns all rows that have 'nancy' and 'davolio' values in the same row. Example 3: const rows = grid.find(5, 'quantity', '>'); returns all rows where the value of the 'quantity' field is > 5. 
   * @param {string} query. Search query
   * @param {string} dataField?. Column data field.
   * @param {string} condition?. Conditions which you can use in the expressions: '=', 'EQUAL','&lt;&gt;', 'NOT_EQUAL', '!=', '&lt;', 'LESS_THAN','&gt;', 'GREATER_THAN', '&lt;=', 'LESS_THAN_OR_EQUAL', '&gt;=', 'GREATER_THAN_OR_EQUAL','starts with', 'STARTS_WITH','ends with', 'ENDS_WITH', '', 'EMPTY', 'CONTAINS','DOES_NOT_CONTAIN', 'NULL','NOT_NULL'
   * @returns {any[]}
   */
  find(query: string, dataField?: string, condition?: string): any[];
  /**
   * Finds entries by using a query and returns an array of cells. Each cell in the array is also an array in this format: [id, dataField, value]. Example: const cells = grid.findCells('nancy'); returns all cells that have 'nancy' value. Example 2: const cells = grid.findCells('nancy, davolio'); returns all cells that have 'nancy' and 'davolio' values.
   * @param {string} query. Search query. You can enter multiple search strings, by using ','. Example: 'nancy, davolio'
   * @returns {any[]}
   */
  findCells(query: string): any[];
  /**
   * Navigates to a page, when paging is enabled.
   * @param {number} index. page index
   */
  goToPage(index: number): void;
  /**
   * Navigates to the next page, when grid paging is enabled.
   */
  nextPage(): void;
  /**
   * Navigates to the prev page, when grid paging is enabled.
   */
  prevPage(): void;
  /**
   * Navigates to the first page, when grid paging is enabled.
   */
  firstPage(): void;
  /**
   * Navigates to the last page, when grid paging is enabled.
   */
  lastPage(): void;
  /**
   * Focuses and selects a cell or row. The keyboard navigation starts from the focused cell or row. Any previously applied selection will be cleared after calling this method.
   * @param {string | number} rowId. row bound id
   * @param {string} dataField?. column bound data field
   */
  focusAndSelect(rowId: string | number, dataField?: string): void;
  /**
   * Iterates through each row in the grid and calls the callback for each row. This is similar to the forEach method on a JavaScript array. This is called for each row, ignoring grouping, filtering or sorting applied in the Grid.
   * @param {any} rowCallback. Callback function with a row object as parameter. Example: grid.forEachRow((row) => { console.log(row.id) });
   */
  forEachRow(rowCallback: any): void;
  /**
   * Similar to forEachRow. Iterates through each row in the grid and calls the callback for each row. This method takes into account filtering and sorting applied to the Grid.
   * @param {any} rowCallback. Callback function with a row object as parameter. Example: grid.forEachRow((row) => { console.log(row.id) });
   */
  forEachRowAfterFilterAndSort(rowCallback: any): void;
  /**
   * Gets the maximum position of the vertical scrollbar. You can use this method in combination with the setVerticalScrollValue to apply a new scroll position.
   * @returns {number}
   */
  getVerticalScrollMax(): number;
  /**
   * Gets the position of the vertical scrollbar.
   * @returns {number}
   */
  getVerticalScrollValue(): number;
  /**
   * Gets the maximum position of the horizontal scrollbar. You can use this method in combination with the setHorizontalScrollValue to apply a new scroll position.
   * @returns {number}
   */
  getHorizontalScrollMax(): number;
  /**
   * Gets the position of the horizontal scrollbar.
   * @returns {number}
   */
  getHorizontalScrollValue(): number;
  /**
   * Gets the columns array. Each item in the array contains the column properties which are dynamically set by the user interaction and the columns initialization data properties such as: 'label', 'dataField', 'dataType', 'visible'.
   * @returns {any}
   */
  getColumns(): any;
  /**
   * Gets the editing cell(s), when the grid is editing.
   * @returns {any[]}
   */
  getEditCells(): any[];
  /**
   * Gets the groups array.
   * @returns {any[]}
   */
  getGroups(): any[];
  /**
   * Gets an array of columns with applied sorting. Each member in the array is with column's data field used as a key and 'sortOrder' and 'sortIndex' as a value.
   * @returns 
   */
  getSortedColumns(): {[dataField: string]: { sortOrder: string, sortIndex: number }};
  /**
   * Gets the selection.
   * @returns {any}
   */
  getSelection(): any;
  /**
   * Gets an Array where each item is an Array of row id and row data. If the Grid is used in virtual mode, the row data parameter is empty object, because the data is loaded on demand.
   * @returns {any[]}
   */
  getSelectedRows(): any[];
  /**
   * Gets the selected row ids.
   * @returns {any[]}
   */
  getSelectedRowIds(): any[];
  /**
   * Gets the selected row indexes.
   * @returns {any[]}
   */
  getSelectedRowIndexes(): any[];
  /**
   * Gets the selected cells. The method returns an array of cell. Each cell is an array with row id, column data field and cell value.
   * @returns {any[]}
   */
  getSelectedCells(): any[];
  /**
   * Gets an array of columns with applied filters.
   * @returns {any}
   */
  getFilteredColumns(): any;
  /**
   * Gets an array of rows, which are visible and match the applied filter.
   * @returns {any}
   */
  getVisibleRows(): any;
  /**
   * Gets the result of the getVisibleRows or the rows hierarchy, when the Grid is in TreeGrid/Grouping mode.
   * @returns {any}
   */
  getViewRows(): any;
  /**
   * Gets a JSON object with the following fields: 'sort', 'columns', 'expandedRows', 'filter', 'groups', 'paging', 'selectedCells', 'selectedrows'. The 'sort' represents an object which contains the sorted columns. Each key in that json object is the column's dataField item which has sortOrder: string and sortIndex: int properties. The sortOrder could be either 'asc' or 'desc'. Similarly, the filter object contains the filtered columns. Each key in that object is a column data field and each value has 'filters' array property with the applied filters to the column. The 'columns' property contains an array of columns with saved properties such as visible, width and freeze. The 'expandedRows' property contains the indexes of the expanded rows. The 'groups' property contains the grouped column data fields and the selectedCells and selectedRows include information about the cells or rows selection. These depend on the selection mode used in the Grid. The 'paging' object includes the sub-properties 'count', 'index' and 'size' which determine the count of pages, the current page's index and the page size.
   * @returns {any}
   */
  getState(): any;
  /**
   * Saves the Grid state and returns a JSON object with the following fields: 'sort', 'columns', 'expandedRows', 'filter', 'groups', 'paging', 'selectedCells', 'selectedrows'. The 'sort' represents an object which contains the sorted columns. Each key in that json object is the column's dataField item which has sortOrder: string and sortIndex: int properties. The sortOrder could be either 'asc' or 'desc'. Similarly, the filter object contains the filtered columns. Each key in that object is a column data field and each value has 'filters' array property with the applied filters to the column. The 'columns' property contains an array of columns with saved properties such as visible, width and freeze. The 'expandedRows' property contains the indexes of the expanded rows. The 'groups' property contains the grouped column data fields and the selectedCells and selectedRows include information about the cells or rows selection. These depend on the selection mode used in the Grid. The 'paging' object includes the sub-properties 'count', 'index' and 'size' which determine the count of pages, the current page's index and the page size.
   * @param {string} name?. state name
   * @returns {any}
   */
  saveState(name?: string): any;
  /**
   * Loads a previously saved Grid state. You can pass a state name when there is a state which was previously saved with the saveState(stateName) method call or a state object returned by the saveState or getState method calls. The state object is required to be a JSON object with the following fields: 'sort', 'columns', 'expandedRows', 'filter', 'groups', 'paging', 'selectedCells', 'selectedrows'. The 'sort' represents an object which contains the sorted columns. Each key in that json object is the column's dataField item which has sortOrder: string and sortIndex: int properties. The sortOrder could be either 'asc' or 'desc'. Similarly, the filter object contains the filtered columns. Each key in that object is a column data field and each value has 'filters' array property with the applied filters to the column. The 'columns' property contains an array of columns with saved properties such as visible, width and freeze. The 'expandedRows' property contains the indexes of the expanded rows. The 'groups' property contains the grouped column data fields and the selectedCells and selectedRows include information about the cells or rows selection. These depend on the selection mode used in the Grid. The 'paging' object includes the sub-properties 'count', 'index' and 'size' which determine the count of pages, the current page's index and the page size.
   * @param {any} state. state name or state object
   * @returns {any}
   */
  loadState(state: any): any;
  /**
   * Resets the Grid state.
   */
  resetState(): void;
  /**
   * Gets the changes from the batch edit.
   * @returns 
   */
  getBatchEditChanges(): { upDated: [{ id: string, dataField: string, oldValue: Object, newValue: Object }], deleted: [{id: string, data: Object}], added: [{id: string, data: Object}] };
  /**
   * Gets a value of a cell.
   * @param {string | number} rowId. row bound id
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @returns {any}
   */
  getCellValue(rowId: string | number, dataField: string): any;
  /**
   * Gets a column. Returns a Grid column object.
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @returns {GridColumn}
   */
  getColumn(dataField: string): GridColumn;
  /**
   * Gets a value of a column.
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @param {string} propertyName. The property name.
   * @returns {any}
   */
  getColumnProperty(dataField: string, propertyName: string): any;
  /**
   * Gets a value of a row.
   * @param {string | number} rowId. row bound id
   * @param {string} propertyName. The property name.
   * @returns {any}
   */
  getRowProperty(rowId: string | number, propertyName: string): any;
  /**
   * Gets a row. Returns a Grid row object.
   * @param {string | number} rowId. row bound id
   * @returns {GridRow}
   */
  getRow(rowId: string | number): GridRow;
  /**
   * Gets a row by its index. Returns a Grid row object.
   * @param {number} rowIndex. row bound index
   * @returns {GridRow}
   */
  getRowByIndex(rowIndex: number): GridRow;
  /**
   * Gets the Data source data associated to the row.
   * @param {string | number} rowId. row bound id
   * @returns {any}
   */
  getRowData(rowId: string | number): any;
  /**
   * Gets the Row's id by a row index.
   * @param {number} rowIndex. row index
   * @returns {string | number}
   */
  getRowId(rowIndex: number): string | number;
  /**
   * Gets whether a column's drop-down menu is opened.
   * @returns {boolean}
   */
  hasMenu(): boolean;
  /**
   * This method returns true, if any rows in the Grid are selected.
   * @returns {boolean}
   */
  hasSelectedRows(): boolean;
  /**
   * Hides the Details of a Row, when row details are enabled.
   * @param {string | number} rowId. row bound id
   */
  hideDetail(rowId: string | number): void;
  /**
   * Highlights a column. Highlights a Grid column.
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   */
  highlightColumn(dataField: string): void;
  /**
   * Highlights a cell. Calling the method a second time toggle the highlight state.
   * @param {string | number} rowId. row bound id
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @param {string} className?. CSS Class Name
   */
  highlightCell(rowId: string | number, dataField: string, className?: string): void;
  /**
   * Highlights a row. Calling the method a second time toggle the highlight state.
   * @param {string | number} rowId. row bound id
   * @param {string} className?. CSS Class Name
   */
  highlightRow(rowId: string | number, className?: string): void;
  /**
   * Inserts a row. When batch editing is enabled, the row is not saved until the batch edit is saved.
   * @param {any} data. row data matching the data source
   * @param {number} index?. Determines the insert index. The default value is the last index.
   * @param  callback?. Sets a callback function, which is called after the new row is added. The callback's argument is the new row.
   */
  insertRow(data: any, index?: number, callback?: {(row: GridRow): void}): void;
  /**
   * Opens a column drop-down menu.
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   */
  openMenu(dataField: string): void;
  /**
   * Opens a context menu. Note that context menu should be enabled.
   * @param {number} left. Left Position.
   * @param {number} top. Top Position.
   */
  openContextMenu(left: number, top: number): void;
  /**
   * Prints the Grid data. The method uses the options of the <em>dataExport</em> property. When printed, the Grid will not display any scrollbars so all rows and columns will be displayed. The grid will auto resize width and height to fit all contents. To customize the printing options, you can use  the <em>dataExport</em> property.
   */
  print(): void;
  /**
   * Renders the grid. This method will make a full-refresh like in the initial Grid creation. It will create Rows, Columns and Cells HTML Elements and then refresh the Grid layout.
   */
  render(): void;
  /**
   * Refreshes the grid with the current property values. This method will refresh the Grid layout.
   */
  refresh(): void;
  /**
   * Refreshes the grid cells in view. The method is useful for live-updates of cell values.
   */
  refreshView(): void;
  /**
   * Removes a column filter. 
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @param {boolean} refreshFilters?. Set this to false, if you need to make multiple removeFilter calls.
   */
  removeFilter(dataField: string, refreshFilters?: boolean): void;
  /**
   * Removes a group by data field. This method will remove a group to the Grid when grouping is enabled.
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   */
  removeGroup(dataField: string): void;
  /**
   * Removes a sorting by data field. This method will remove a sorting from a Grid column.
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   */
  removeSort(dataField: string): void;
  /**
   * Re-sorts the Grid by using the already applied column sortings and re-renders the Grid.
   */
  refreshSort(): void;
  /**
   * Reverts the batch edit changes. This method cancels all changes made by the end-user.
   */
  revertBatchEdit(): void;
  /**
   * Reorders two DataGrid columns.
   * @param {string | number} dataField. The data field or column index of the first grid column.
   * @param {string | number} referenceDataField. The data field or column index of the second grid column.
   * @param {boolean} insertAfter?. Determines whether to insert the first column after the reference column.
   */
  reorderColumns(dataField: string | number, referenceDataField: string | number, insertAfter?: boolean): void;
  /**
   * Sorts the Grid by a data field. This method will add or remove sorting, when sorting is enabled. To remove the sorting, use 'null' for the sortOrder parameter.
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @param {string | null} sortOrder. column's sort order. Use 'asc', 'desc' or null.
   */
  sortBy(dataField: string, sortOrder: string | null): void;
  /**
   * Swaps two DataGrid columns.
   * @param {string | number} dataField. The data field or column index of the first grid column.
   * @param {string | number} referenceDataField. The data field or column index of the second grid column.
   */
  swapColumns(dataField: string | number, referenceDataField: string | number): void;
  /**
   * Saves the batch edit changes. This method confirms the editing changes made by the end-user.
   */
  saveBatchEdit(): void;
  /**
   * Selects a row, cell or column.
   * @param {string | number} rowId. row bound id
   * @param {string} dataField?. column bound data field
   */
  select(rowId: string | number, dataField?: string): void;
  /**
   * Selects a range of rows, cells or columns. The result of the method depends on the <em>selection</em> configuration of the Grid.
   * @param {string | number} rowId. row bound id
   * @param {string} dataField. column bound data field
   * @param {string | number} endRowId. row bound id
   * @param {string} endDataField. column bound data field
   */
  selectRange(rowId: string | number, dataField: string, endRowId: string | number, endDataField: string): void;
  /**
   * Selects a range of rows.
   * @param {string | number} rowId. row bound id
   * @param {string | number} endRowId. row bound id
   */
  selectRowsRange(rowId: string | number, endRowId: string | number): void;
  /**
   * Selects multiple rows by their ids.
   * @param {(string | number)[]} rowId. Array of row ids
   */
  selectRows(rowId: (string | number)[]): void;
  /**
   * Selects all rows.
   */
  selectAllRows(): void;
  /**
   * Selects multiple rows by their index.
   * @param {number[]} rowIndex. Array of row indexes
   */
  selectRowsByIndex(rowIndex: number[]): void;
  /**
   * Selects rows by using a query. Example: grid.selectRowsByQuery('nancy'); selects all rows that have 'nancy' value. Example 2: grid.selectRowsByQuery('nancy, davolio'); selects all rows that have 'nancy' and 'davolio' values in the same row. Example 3: grid.selectRowsByQuery(5, 'quantity', '>'); selects all rows where the value of the 'quantity' field is > 5. 
   * @param {string} query. Search query
   * @param {string} dataField?. Column data field.
   * @param {string} condition?. Conditions which you can use in the expressions: '=', 'EQUAL','&lt;&gt;', 'NOT_EQUAL', '!=', '&lt;', 'LESS_THAN','&gt;', 'GREATER_THAN', '&lt;=', 'LESS_THAN_OR_EQUAL', '&gt;=', 'GREATER_THAN_OR_EQUAL','starts with', 'STARTS_WITH','ends with', 'ENDS_WITH', '', 'EMPTY', 'CONTAINS','DOES_NOT_CONTAIN', 'NULL','NOT_NULL'
   */
  selectRowsByQuery(query: string, dataField?: string, condition?: string): void;
  /**
   * Selects multiple cells by their ids and dataFields. Example: grid.selectCells([0, 1, 2], ['firstName', 'quantity', 'date']); - selects the 'firstName', 'quantity' and 'date' cells from the first, second and third rows.
   * @param {(string | number)[]} rowIds. Array of row ids
   * @param {string[]} dataFields. Array of data fields.
   */
  selectCells(rowIds: (string | number)[], dataFields: string[]): void;
  /**
   * Selects cells by using a query. Example: grid.selectCellsByQuery('nancy'); selects all cells that have 'nancy' value. Example 2: grid.selectCellsByQuery('nancy, davolio'); selects all cells that have 'nancy' and 'davolio' values in the same row. 
   * @param {string} query. Search query
   */
  selectCellsByQuery(query: string): void;
  /**
   * Sets a new value to a cell.
   * @param {string | number} rowId. row bound id
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @param {string | number | Date | boolean} value. New Cell value.
   */
  setCellValue(rowId: string | number, dataField: string, value: string | number | Date | boolean): void;
  /**
   * Sets new columns to the Grid. The grid will redraw all the column headers, and then redraw all of the rows. By using 'setColumns', the grid will compare the new columns passed as argument to the method with existing columns. The Grid will automatically create new columns, keep old columns if they already exist and remove columns which are not in the 'setColumns' method argument. The benefit of that is that the state of the column like(sort, filter, width or other) will be kept, if the column exsits after the new columns are applied.
   * @param {GridColumn[]} columns. Columns array.
   */
  setColumns(columns: GridColumn[]): void;
  /**
   * Sets a property to a column.
   * @param {string} dataField. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   * @param {string} propertyName. The column property's name.
   * @param {any} value. The new property value.
   */
  setColumnProperty(dataField: string, propertyName: string, value: any): void;
  /**
   * Sets a property to a row.
   * @param {string | number} rowId. row bound id
   * @param {string} propertyName. The row property's name.
   * @param {any} value. The new property value.
   */
  setRowProperty(rowId: string | number, propertyName: string, value: any): void;
  /**
   * Sets a style to a row.
   * @param {string | number} rowId. row bound id
   * @param  rowStyle. The row style object. The object may have one or all of the following properties: 'background', 'color', 'fontSize', 'fontFamily', 'textDecoration', 'fontStyle', 'fontWeight'.
   */
  setRowStyle(rowId: string | number, rowStyle: {background?: string, color?: string, fontSize?: string, fontFamily?: string, textDecoration?: string, fontStyle?: string, fontWeight?: string}): void;
  /**
   * Sets a style to a row.
   * @param {string | number} rowId. row bound id
   * @param {string} dataField. Column bound field name.
   * @param  rowStyle. The cell style object. The object may have one or all of the following properties: 'background', 'color', 'fontSize', 'fontFamily', 'textDecoration', 'fontStyle', 'fontWeight'.
   */
  setCellStyle(rowId: string | number, dataField: string, rowStyle: {background?: string, color?: string, fontSize?: string, fontFamily?: string, textDecoration?: string, fontStyle?: string, fontWeight?: string}): void;
  /**
   * Sets the position of the vertical scrollbar. You can use this method in combination with the getVerticalScrollValue and getVerticalScrollMax.
   * @param {number} value. The new scroll position
   */
  setVerticalScrollValue(value: number): void;
  /**
   * Sets the position of the horizontal scrollbar. You can use this method in combination with the getHorizontalScrollValue and getHorizontalScrollMax.
   * @param {number} value. The new scroll position
   */
  setHorizontalScrollValue(value: number): void;
  /**
   * Shows the Details of a Row, when row details are enabled.
   * @param {string | number} rowId. row bound id
   */
  showDetail(rowId: string | number): void;
  /**
   * Updates a row. When batch editing is enabled, the row is not saved until the batch edit is saved.
   * @param {string | number} rowId. row bound id
   * @param {any} data. row data matching the data source
   * @param  callback?. Sets a callback function, which is called after the row is updated. The callback's argument is the updated row.
   */
  updateRow(rowId: string | number, data: any, callback?: {(row: GridRow): void}): void;
  /**
   * Unselects a row, cell or column.
   * @param {string | number} rowId. row bound id
   * @param {string} dataField?. column bound data field. For example, if you have a column with dataField: 'firstName', set 'firstName' here.
   */
  unselect(rowId: string | number, dataField?: string): void;
  /**
   * Unchecks a TreeGrid row. Sets its check-box to false.
   * @param {string | number} rowId. row bound id
   */
  uncheckRow(rowId: string | number): void;
  /**
   * Unchecks all TreeGrid or Grouping rows. Sets all check-boxes to false.
   */
  uncheckAllRows(): void;
  /**
   * Toggles a TreeGrid row. When this method is called, the row is expanded, if it's state is collapsed and collapsed if it's state is expanded.
   * @param {string | number} rowId. row bound id
   */
  toggleRow(rowId: string | number): void;
}

/**An object containing settings related to the grid's appearance. */
export interface GridAppearance {
  /**
   * Starting row index of alternating colors.
   * Default value: 0
   */
  alternationStart?: number;
  /**
   * Ending row index of alternating colors.
   * Default value: 0
   */
  alternationEnd?: number;
  /**
   * number of row color alternations.
   * Default value: 0
   */
  alternationCount?: number;
  /**
   * Enables sticky columns.
   * Default value: false
   */
  allowColumnStickyPosition?: boolean;
  /**
   * Enables row hover effect. Applies a hover style to the cells.
   * Default value: false
   */
  allowHover?: boolean;
  /**
   * Enables header hover effect. Applies a hover style to the headers.
   * Default value: true
   */
  allowHeaderHover?: boolean;
  /**
   * Enables row toggle animation. This animation starts when you expand/collapse a row in TreeGrid/Grouping mode.
   * Default value: false
   */
  allowRowToggleAnimation?: boolean;
  /**
   * Enables row detail toggle animation. This animation starts when you expand/collapse a row in TreeGrid/Grouping mode.
   * Default value: false
   */
  allowRowDetailToggleAnimation?: boolean;
  /**
   * Enables sorting animation. Data records are sorted with animation, when sorting is applied.
   * Default value: false
   */
  allowSortAnimation?: boolean;
  /**
   * Enables column label animation. The label is moved to 'left' with animation, when the column header's drop-down button is displayed on hover or sorting is applied.
   * Default value: true
   */
  allowColumnLabelAnimation?: boolean;
  /**
   * Enables column menu animation. When you click on the column header's drop-down button, the menu is animated.
   * Default value: true
   */
  allowColumnMenuAnimation?: boolean;
  /**
   * Enables column sort button animation. When you click on a sortable column to change the sort order, the sort button is animated.
   * Default value: true
   */
  allowColumnSortButtonAnimation?: boolean;
  /**
   * Enables column action button animation. The drop-down button displayed on column header hover is optionally animated.
   * Default value: true
   */
  allowColumnActionButtonAnimation?: boolean;
  /**
   * Enables column filter button animation.
   * Default value: true
   */
  allowColumnFilterButtonAnimation?: boolean;
  /**
   * If enabled, automatically shows column sort button. The sort button is displayed only when the column is sorted. If the property's value is false, sort button will be always displayed to indicate that sorting is possible.
   * Default value: true
   */
  autoShowColumnSortButton?: boolean;
  /**
   * If enabled, automatically shows column action button.
   * Default value: true
   */
  autoShowColumnActionButton?: boolean;
  /**
   * If enabled, automatically shows column filter button.
   * Default value: true
   */
  autoShowColumnFilterButton?: boolean;
  /**
   * Generates labels as 'numbers' or 'letters'. This property affects the rendering of the row header.
   * Default value: number
   */
  autoGenerateRowLabelMode?: GridAppearanceAutoGenerateRowLabelMode | string;
  /**
   * Generates labels as 'numbers' or 'letters.  This property affects the rendering of the column header.
   * Default value: letter
   */
  autoGenerateColumnLabelMode?: GridAppearanceAutoGenerateColumnLabelMode | string;
  /**
   * Sets the visibility of the loading indicator. This is the Loading... image displayed in the Grid while loading data.
   * Default value: false
   */
  displayLoadingIndicator?: boolean;
  /**
   * Sets the loading indicator label.  This is the Text displayed while loading data.
   * Default value: "Loading..."
   */
  loadingIndicatorPlaceholder?: string;
  /**
   * Sets the placeholder of the Grid. The placeholder is displayed when the Grid is empty.
   * Default value: "No Rows"
   */
  placeholder?: string;
  /**
   * Sets the duration of sorting animation. This property is related to the allowSortAnimation property.
   * Default value: 500
   */
  sortAnimationDuration?: number;
  /**
   * Shows or hides Row header.
   * Default value: false
   */
  showRowHeader?: boolean;
  /**
   * Shows or hides Row headers. In TreeGrid, the non-leaf tree items are displayed as normal rows. If this property is set to true, they are displayed as headers similar to the grouping rendering.
   * Default value: false
   */
  showTreeRowHeader?: boolean;
  /**
   * Shows row indexes in the row header. The showRowHeader property should be true
   * Default value: false
   */
  showRowHeaderNumber?: boolean;
  /**
   * Shows edit icon when a cell or row is in edit state.
   * Default value: true
   */
  showRowHeaderEditIcon?: boolean;
  /**
   * Shows select icon when the pointer is over the row header cell.
   * Default value: false
   */
  showRowHeaderSelectIcon?: boolean;
  /**
   * Shows focus icon on cell or row focus.
   * Default value: false
   */
  showRowHeaderFocusIcon?: boolean;
  /**
   * Shows drag icon on the row header. 
   * Default value: false
   */
  showRowHeaderDragIcon?: boolean;
  /**
   * Shows column header lines.
   * Default value: true
   */
  showColumnHeaderLines?: boolean;
  /**
   * Shows drag icon on the column header when drag is enabled. The icon is displayed when you move the mouse cursor to the column header's left edge.
   * Default value: false
   */
  showColumnHeaderDragIcon?: boolean;
  /**
   * Shows column lines.
   * Default value: true
   */
  showColumnLines?: boolean;
  /**
   * Shows row lines.
   * Default value: true
   */
  showRowLines?: boolean;
  /**
   * Shows column groups in the Hide columns panel. Column groups and columns are shown in a tree-like structure. When the property is set to false, the column groups are not displayed and the column labels contain the column group name.
   * Default value: false
   */
  showColumnGroupsInColumnPanel?: boolean;
  /**
   * Shows filtered column background, when filter is applied.
   * Default value: false
   */
  showFilterColumnBackground?: boolean;
  /**
   * Shows sorted column background, when sorting is applied.
   * Default value: false
   */
  showSortColumnBackground?: boolean;
  /**
   * Shows frozen column background, when the Grid has frozen columns.
   * Default value: true
   */
  showFrozenColumnBackground?: boolean;
  /**
   * Shows filtered row background, when the Grid has frozen rows.
   * Default value: true
   */
  showFrozenRowBackground?: boolean;
  /**
   * Shows column sort button.
   * Default value: true
   */
  showColumnSortButton?: boolean;
  /**
   * Shows column filter button.
   * Default value: true
   */
  showColumnFilterButton?: boolean;
  /**
   * Shows column description button. The button's style is customizable through the Grid CSS.
   * Default value: false
   */
  showColumnDescriptionButton?: boolean;
  /**
   * Shows column icon within the column's header.
   * Default value: false
   */
  showColumnIcon?: boolean;
  /**
   * Shows column custom button. User-defined button shown in the column header.
   * Default value: false
   */
  showColumnCustomButton?: boolean;
  /**
   * Shows column action button. This is the drop-down button displayed in the column header.
   * Default value: true
   */
  showColumnActionButton?: boolean;
  /**
   * Shows tooltips when user hovers columns or cells.
   * Default value: false
   */
  showTooltips?: boolean;
  /**
   * Shows horizontal scrollbar on fixed columns.
   * Default value: false
   */
  showHorizontalScrollBarOnFixedColumns?: boolean;
  /**
   * Shows vertical scrollbar on fixed columns.
   * Default value: false
   */
  showVerticalScrollBarOnFixedColumns?: boolean;
  /**
   * Shows the today's date as 'Today' vs '7/8/2022'. When the property is set to false, it will display the date.
   * Default value: true
   */
  showTodayDateAsString?: boolean;
}

/**An object containing settings related to the grid's behavior. */
export interface GridBehavior {
  /**
   * Auto-Resize on double-click of a column's right border.
   * Default value: true
   */
  allowColumnAutoSizeOnDoubleClick?: boolean;
  /**
   * Auto-Resize on double-click of a row's bottom border.
   * Default value: true
   */
  allowRowAutoSizeOnDoubleClick?: boolean;
  /**
   * Determines whether row reorder is enabled.
   * Default value: false
   */
  allowRowReorder?: boolean;
  /**
   * Determines whether column reorder is enabled.
   * Default value: false
   */
  allowColumnReorder?: boolean;
  /**
   * Determines whether column freeze with drag and drop is enabled. When other columns are frozen/pinned, drag the column to the existing frozen area. When no columns are pinned, drag the column to the edge of the grid and wait for approximately one second. The grid will then assume you want to freeze/pin and create a frozen/pinned area and place the column into it.
   * Default value: false
   */
  allowColumnFreeze?: boolean;
  /**
   * Sets the column resize mode. split resize mode 'grows' or 'shrinks' the resize element's size and 'shrinks' or 'grows' the next sibling element's size. growAndShrink resize mode 'grows' or 'shrinks' the resize element's size
   * Default value: none
   */
  columnResizeMode?: GridResizeMode | string;
  /**
   * Sets the row resize mode. split resize mode 'grows' or 'shrinks' the resize element's size and 'shrinks' or 'grows' the next sibling element's size. growAndShrink resize mode 'grows' or 'shrinks' the resize element's size
   * Default value: none
   */
  rowResizeMode?: GridResizeMode | string;
}

/**An object containing settings related to the grid's layout. */
export interface GridLayout {
  /**
   * Enables or disables the Cells Value wrapping. When the property is true, cell value can wrap in multiple lines.
   * Default value: false
   */
  allowCellsWrap?: boolean;
  /**
   * Automatically sets width to any new Column which does not have its 'width' property set.
   * Default value: false
   */
  autoSizeNewColumn?: boolean;
  /**
   * Sets the width of the auto-generated Grid columns.
   * Default value: null
   */
  autoGenerateColumnWidth?: string | number | null;
  /**
   * Sets the width of the Grid columns.
   * Default value: null
   */
  columnWidth?: string | number | null;
  /**
   * Sets the height of the Grid columns.
   * Default value: null
   */
  columnHeight?: string | number | null;
  /**
   * Sets the minimum height of the Grid columns.
   * Default value: 30
   */
  columnMinHeight?: number;
  /**
   * Sets the minimum height of the Grid rows.
   * Default value: 30
   */
  rowMinHeight?: number;
  /**
   * Sets the height of the Grid rows. The property can be set to null, auto or a number.
   * Default value: null
   */
  rowHeight?: string | number | null;
}

/**The <em>clipboard</em> property is used to enable/disable clipboard operations with Ctrl+C, Ctrl+X and Ctrl+V keyboard navigations.. */
export interface GridClipboard {
  /**
   * Sets or gets whether the property is enabled.
   * Default value: true
   */
  enabled?: boolean;
  /**
   * Sets or gets whether the copy-pasted values will be auto-filled by using automatic pattern detection. This is used in the Drag&Drop Multiple Cells selection. none does nothing. copy just copies the cells. 'fillSeries' detects and automatically fills the values. For example, if the selection has '1, 2' and the possible positions are more, the pasted values would be '1, 2, 3, 4, etc.
   * Default value: fillSeries
   */
  autoFillMode?: GridClipboardAutoFillMode | string;
  /**
   * Sets or gets a callback on paste.
   * Default value: null
   */
  onPasteValue?: {(args: {value: any, oldValue: any, dataField: string, id: string | number}): void};
}

export interface GridColumn {
  /**
   * Sets or gets the column's header alignment. Accepts: 'left', 'right', 'center'
   * Default value: left
   */
  align?: HorizontalAlignment | string;
  /**
   * Sets or gets whether the column can be exported.
   * Default value: true
   */
  allowExport?: boolean;
  /**
   * Sets or gets whether the column can be grouped.
   * Default value: true
   */
  allowGroup?: boolean;
  /**
   * Sets or gets whether the column can be hidden.
   * Default value: true
   */
  allowHide?: boolean;
  /**
   * Sets or gets whether the column can be selected.
   * Default value: true
   */
  allowSelect?: boolean | null;
  /**
   * Sets or gets whether the column can be edited.
   * Default value: true
   */
  allowEdit?: boolean;
  /**
   * Sets or gets whether the column can be sorted.
   * Default value: true
   */
  allowSort?: boolean;
  /**
   * Sets or gets whether the column can be edited, when header editing is enabled.
   * Default value: true
   */
  allowHeaderEdit?: boolean;
  /**
   * Sets or gets whether the column can be filtered.
   * Default value: true
   */
  allowFilter?: boolean;
  /**
   * Sets or gets whether the column can be reordered.
   * Default value: true
   */
  allowReorder?: boolean;
  /**
   * Sets or gets whether the column can be resized.
   * Default value: true
   */
  allowResize?: boolean;
  /**
   * Sets or gets whether the column can have 'null' values.
   * Default value: true
   */
  allowNull?: boolean;
  /**
   * Sets or gets the column's cells format. This property is used for applying a formatting to the cell values. Number format strings: 'd' - decimal numbers.'f' - floating-point numbers.'n' - integer numbers.'c' - currency numbers.'p' - percentage numbers.For adding decimal places to the numbers, add a number after the formatting striFor example: 'c3' displays a number in this format $25.256Built-in Date formats:// short date pattern'd' - 'M/d/yyyy',// long date pattern'D' - 'dddd, MMMM dd, yyyy',// short time pattern't' - 'h:mm tt',// long time pattern'T' - 'h:mm:ss tt',// long date, short time pattern'f' - 'dddd, MMMM dd, yyyy h:mm tt',// long date, long time pattern'F' - 'dddd, MMMM dd, yyyy h:mm:ss tt',// month/day pattern'M' - 'MMMM dd',// month/year pattern'Y' - 'yyyy MMMM',// S is a sortable format that does not vary by culture'S' - 'yyyy'-'MM'-'dd'T'HH':'mm':'ss'Date format strings:'d'-the day of the month;'dd'-the day of the month'ddd'-the abbreviated name of the day of the week'dddd'- the full name of the day of the week'h'-the hour, using a 12-hour clock from 1 to 12'hh'-the hour, using a 12-hour clock from 01 to 12'H'-the hour, using a 24-hour clock from 0 to 23'HH'- the hour, using a 24-hour clock from 00 to 23'm'-the minute, from 0 through 59'mm'-the minutes,from 00 though59'M'- the month, from 1 through 12'MM'- the month, from 01 through 12'MMM'-the abbreviated name of the month'MMMM'-the full name of the month's'-the second, from 0 through 59'ss'-the second, from 00 through 59't'- the first character of the AM/PM designator'tt'-the AM/PM designator'y'- the year, from 0 to 99'yy'- the year, from 00 to 99'yyy'-the year, with a minimum of three digits'yyyy'-the year as a four-digit number;'yyyyy'-the year as a four-digit number.
   * Default value: ""
   */
  cellsFormat?: string;
  /**
   * Sets or gets the column's cells alignment. Accepts: 'left', 'right' and 'center'
   * Default value: left
   */
  cellsAlign?: HorizontalAlignment | string;
  /**
   * Sets or gets the column's cells wrapping. Accepts: true or false.
   * Default value: false
   */
  cellsWrap?: boolean;
  /**
   * Sets or gets the column's cells vertical alignment. Accepts: 'top', 'bottom' and 'center'
   * Default value: center
   */
  cellsVerticalAlign?: VerticalAlignment | string;
  /**
   * Sets or gets the column's header CSS class name. You can apply multiple CSS class names by separating them with space.
   * Default value: ""
   */
  className?: string;
  /**
   * Sets or gets the column's cells CSS class name(s). The property can be used with string and function. You can apply multiple CSS class names by separating them with space or you can return a CSS class name(s) when you use it as a function. The function gets called with the following parameters: index - row's index, dataField - column's data field, cellValue - current cell's value, rowData - current row's data, row - GridRow object. Ex: cellsClassName: (index, dataField, value, rowData, row) => { if (index === 0) { return 'cell-class-1' } }
   * Default value: 
   */
  cellsClassName?: any;
  /**
   * Sets or gets the column's cells CSS class rules. Different CSS class names are conditionally applied. Example: label: 'Quantity', dataField: 'quantity', editor: 'numberInput', cellsClassRules: { 'one': settings => settings.value > 5, 'two': settings => settings.value &lt;5, 'three': settings => settings.value === 3 }.  The settings object contains the following properties: index, value, dataField, row, api.
   * Default value: null
   */
  cellsCSSRules?: any;
  /**
   * Sets the name of the column group.
   * Default value: ""
   */
  columnGroup?: string;
  /**
   * Sets or gets the column's data source bound field.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Sets or gets the column's data type. Any of the following value is valid: 'string', 'number', 'int', 'date', 'bool', 'object', 'any'
   * Default value: "string"
   */
  dataType?: string;
  /**
   * Sets or gets the column's data source bound field which will be displayed to the user. When the property is not set, it is equal to the 'dataField'.
   * Default value: ""
   */
  displayField?: string;
  /**
   * Sets or gets the column's description. The description of the column is displayed in the column's header, when the end-user moves the pointer over the description button. 'showDescriptionButton' property determines whether the description button is visible.
   * Default value: ""
   */
  description?: string;
  /**
   * Gets the HTML Element. The property returns null when the Column is not in the View.
   * Default value: null
   */
  element?: HTMLElement;
  /**
   * Sets or gets the column's editor. The property expects 'input', 'autoComplete', 'comboBox', 'dropDownList', 'image', 'numberInput', 'checkBox', 'multiInput', 'multiComboInput', 'checkInput', 'slider', 'dateTimePicker', 'timeInput', 'dateInput', 'dateRangeInput', 'maskedTextBox', 'textArea' or a custom object with 'template' property which defines the editor type, 'settings' property which defines the custom editor's properties, 'onInit(int row, string column, object editor, object rowData): void', 'onRender(int row, string column, object editor, object rowData): void', 'setValue(object value): void' and 'getValue(object value): any' callback functions.
   * Default value: null
   */
  editor?: any;
  /**
   * Sets or gets the Freeze mode. Accepts: 'near', 'far', true and false. Freezes/Pins the column to left(near) or right(far).
   * Default value: false
   */
  freeze?: Position | string;
  /**
   * Sets or gets the filter of the column. Example: ['contains Andrew or contains Nancy']. Example with numeric filter ['quantity', '&lt;= 3 and &gt;= 8']. Additional example with filter which we want to apply to a column with filterMenuMode='excel' - ['EQUAL' 'Andrew' or 'EQUAL' 'Antoni' or 'EQUAL' 'Beate']. Example with a string filter applied to a string column - ['CONTAINS' 'burke' or 'CONTAINS' 'peterson']. Filter conditions which you can use in the expressions: '=', 'EQUAL','&lt;&gt;', 'NOT_EQUAL', '!=', '&lt;', 'LESS_THAN','&gt;', 'GREATER_THAN', '&lt;=', 'LESS_THAN_OR_EQUAL', '&gt;=', 'GREATER_THAN_OR_EQUAL','starts with', 'STARTS_WITH','ends with', 'ENDS_WITH', '', 'EMPTY', 'CONTAINS','DOES_NOT_CONTAIN', 'NULL','NOT_NULL'
   * Default value: ""
   */
  filter?: string;
  /**
   * Sets or gets the filter menu mode of the column. In 'basic' mode, a single input is displayed in the filter menu. In 'default' mode, two input options are available for more precise filtering. In 'excel' mode, checked list with unique values is displayed.
   * Default value: default
   */
  filterMenuMode?: GridColumnFilterMenuMode | string;
  /**
   * Sets or gets the column's format function.
   * Default value: null
   */
  formatFunction?: {(formatObject: {row?: GridRow, column?: GridColumn, cell?: GridCell, oldValue?: any, value?: any, template?: any}): void};
  /**
   * Sets or gets the column's format settings. You can use any of the build in formatting options or to NumberFormat object like that: 'Intl: {  NumberFormat: {  style: \'currency\', currency: \'EUR\' }}' or DateTimeFormat object like that: 'Intl: {  DateTimeFormat: {  dateStyle: \'full\' }}''
   * Default value: [object Object]
   */
  formatSettings?: any;
  /**
   * Sets or gets the column's group.
   * Default value: ""
   */
  group?: string;
  /**
   * This function allows you to provide custom cell values, which will be displayed in the column's cells. The grid passes 3 arguments to the function - row id, column's dataField and row's data.
   * Default value: null
   */
  getCellValue?: any;
  /**
   * Gets the column's filter panel. The function should return HTMLElement which will represent the filter UI panel displayed in the filter menu.  The function works in combination with updateFilterPanel
   * Default value: null
   */
  getFilterPanel?: any;
  /**
   * Sets or gets the column's icon. Expects CSS class name.
   * Default value: ""
   */
  icon?: string;
  /**
   * Sets or gets the text displayed in the column's header.
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets the column header's template. The property expects the 'id' of HTMLTemplateElement, HTML string or function which returns html string.
   * Default value: 
   */
  labelTemplate?: string | HTMLTemplateElement | HTMLElement | {(label: string): string};
  /**
   * Sets or gets the minimum width.
   * Default value: 30
   */
  minWidth?: number;
  /**
   * Sets or gets the column's rowSpan function. Allows you to dynamically span cells.
   * Default value: null
   */
  rowSpan?: {(cellValue: any, rowIndex: number, data: any): number};
  /**
   * Sets or gets the sort order of the column. Accepts: 'asc', 'desc' and null.
   * Default value: null
   */
  sortOrder?: GridColumnSortOrder | null | string;
  /**
   * Sets or gets the sort index of the column. Accepts an integer value. This property can be used to get or set the column's sort index when sorting mode is 'many'.
   * Default value: null
   */
  sortIndex?: number;
  /**
   * Sets or gets a custom 'sortComparator' function. It can be used for implementing custom sorting. Ex: sortComparator: (value1, value2) =&gt; { if (value1 === value2) return 0; return value1 &lt;value2; }
   * Default value: null
   */
  sortComparator?: any;
  /**
   * Sets or gets whether the column's header action drop-down button is displayed. This button opens the column's menu.
   * Default value: true
   */
  showActionButton?: boolean;
  /**
   * Sets or gets whether the column's header icon is displayed.
   * Default value: false
   */
  showIcon?: boolean;
  /**
   * Sets or gets whether the column's header description button is displayed.
   * Default value: false
   */
  showDescriptionButton?: boolean;
  /**
   * Sets or gets the width. Accepts: 'number', 'px', 'em', 'auto', 'null' values.
   * Default value: 
   */
  width?: string | number;
  /**
   * Sets or gets the column's template. The property expects the 'id' of HTMLTemplateElement or HTML string which is displayed in the cells. Built-in string values are: 'checkBox', 'switchButton', 'radioButton', 'url', 'email', 'dropdownlist', 'list', 'progress', 'tags', 'autoNumber', 'modifiedBy', 'createdBy', 'createdTime', 'modifiedTime', 'images. For example, when you set the template to 'url', the cells will be render anchor tags. When you set the template property to HTMLTemplateElement you should consider that once a template is rendered, the formatObject.template property stores the rendered template component for further use.
   * Default value: 
   */
  template?: any;
  /**
   * Sets or gets the column's validation rules. The expected value is an Array of Objects. Each object should have a 'type' property that can be set to 'required', 'min', 'max', 'minLength', 'maxLength', 'email', 'null', 'requiredTrue', 'minData', 'maxDate', 'pattern'. The 'value' property should be set, too. For validation rule types 'required', 'requiredTrue' and 'null' you can skip the 'value' property. Optional property is 'message', which determines the error message.
   * Default value: null
   */
  validationRules?: [] | null;
  /**
   * Sets or gets the column's header vertical alignment. Accepts: 'top', 'bottom' and 'center'
   * Default value: center
   */
  verticalAlign?: VerticalAlignment | string;
  /**
   * Sets or gets the column summary. The property should be set to an array with the following possible values: 'sum', 'min', 'max', 'avg', 'count', 'median', 'stdev', 'stdevp', 'var', 'varp'.
   * Default value: 
   */
  summary?: string[];
  /**
   * Updates the column's filter panel. The function works in combination with getFilterPanel
   * Default value: null
   */
  updateFilterPanel?: any;
  /**
   * Sets or gets whether the column is visible. Set the property to 'false' to hide the column.
   * Default value: true
   */
  visible?: boolean;
}

/**Context Menu is the drop-down menu displayed after right-clicking a Grid row. It allows you to delete row, edit cell or row depending on the edit mode. The 'contextMenuItemCustom' dataSource option allows you to add custom menu item to the context menu. You can replace the context menu by using the 'selector' property and setting it to ID of a Smart.Menu component. */
export interface GridContextMenu {
  /**
   * Sets or gets whether the context menu is enabled. If the value is false, the context menu will not be displayed, when user right clicks on a row.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Sets the data sources to the context menu.
   * Default value: [object Object]
   */
  dataSource?: GridContextMenuDataSource;
  /**
   * Sets the ID or CSS Class of a Smart.Menu component to be used as a context menu for the Grid.
   * Default value: ""
   */
  selector?: string;
  /**
   * Sets the width of the context menu.
   * Default value: 250
   */
  width?: number;
  /**
   * Sets the height of the context menu.
   * Default value: null
   */
  height?: number | null;
}

/**Sets the data sources to the context menu. */
export interface GridContextMenuDataSource {
  /**
   * Describes the delete item.
   * Default value: [object Object]
   */
  contextMenuItemDelete?: GridCommand;
  /**
   * Describes the edit item.
   * Default value: [object Object]
   */
  contextMenuItemEdit?: GridCommand;
  /**
   * Describes the custom item.
   * Default value: [object Object]
   */
  contextMenuItemCustom?: GridCommand;
}

/**Describes the delete item. */
export interface GridCommand {
  /**
   * Sets the command of the context menu item.
   * Default value: "contextMenuItemDeleteCommand"
   */
  command?: string;
  /**
   * Enables the context menu item.
   * Default value: true
   */
  enabled?: boolean;
  /**
   * Sets the visibility of the context menu item.
   * Default value: true
   */
  visible?: boolean;
  /**
   * Sets the icon of the context menu item.
   * Default value: "jqx-icon-delete"
   */
  icon?: string;
  /**
   * Sets the label of the context menu item.
   * Default value: ""
   */
  label?: string;
}

/**Column Menu is the drop-down menu displayed after clicking the column header's drop-down button, which is displayed when you hover the column header. It allows you to customize column settings. For example: Sort, Filter or Group the Grid by the current column. */
export interface GridColumnMenu {
  /**
   * Automatically closes the column menu.
   * Default value: true
   */
  autoClose?: boolean;
  /**
   * Sets the data sources to the column menu.
   * Default value: [object Object]
   */
  dataSource?: GridColumnMenuDataSource;
  /**
   * Gets the visibility of the column menu. Returns true, when the column menu is visible.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Sets or gets whether the column menu is enabled. If the value is false, the column menu will not be displayed, when user hovers the column.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Sets the width of the column menu.
   * Default value: 250
   */
  width?: number;
  /**
   * Sets the height of the column menu.
   * Default value: null
   */
  height?: number | null;
}

/**Sets the data sources to the column menu. */
export interface GridColumnMenuDataSource {
  /**
   * Describes the settings of the column menu customize type
   * Default value: [object Object]
   */
  columnMenuCustomizeType?: GridCommand;
  /**
   * Describes the settings of the column menu item rename.
   * Default value: [object Object]
   */
  columnMenuItemRename?: GridCommand;
  /**
   * Describes the settings of the column menu item edit description.
   * Default value: [object Object]
   */
  columnMenuItemEditDescription?: GridCommand;
  /**
   * Describes the settings of the column menu item duplicate.
   * Default value: [object Object]
   */
  columnMenuItemClone?: GridCommand;
  /**
   * Describes the settings of the column menu item insert left.
   * Default value: [object Object]
   */
  columnMenuItemInsertLeft?: GridCommand;
  /**
   * Describes the settings of the of the column menu item insert right.
   * Default value: [object Object]
   */
  columnMenuItemInsertRight?: GridCommand;
  /**
   * Describes the settings of the column menu item sort ascending.
   * Default value: [object Object]
   */
  columnMenuItemSortAsc?: GridCommand;
  /**
   * Describes the settings of the column menu item sort descending.
   * Default value: [object Object]
   */
  columnMenuItemSortDesc?: GridCommand;
  /**
   * Describes the settings of the column menu item to add sorting.
   * Default value: [object Object]
   */
  columnMenuItemSort?: GridCommand;
  /**
   * Describes the settings of the column menu item remove sort.
   * Default value: [object Object]
   */
  columnMenuItemRemoveSort?: GridCommand;
  /**
   * Describes the settings of the column menu item filter.
   * Default value: [object Object]
   */
  columnMenuItemFilter?: GridCommand;
  /**
   * Describes the settings of the column menu item remove filter.
   * Default value: [object Object]
   */
  columnMenuItemRemoveFilter?: GridCommand;
  /**
   * Describes the settings of the column menu item group by.
   * Default value: [object Object]
   */
  columnMenuItemGroupBy?: GridCommand;
  /**
   * Describes the settings of the column menu item group by.
   * Default value: [object Object]
   */
  columnMenuItemRemoveGroupBy?: GridCommand;
  /**
   * Describes the settings of the column menu item hide.
   * Default value: [object Object]
   */
  columnMenuItemHide?: GridCommand;
  /**
   * Describes the settings of the column menu item delete.
   * Default value: [object Object]
   */
  columnMenuItemDelete?: GridCommand;
}

export interface GridColumnGroup {
  /**
   * Sets the label.
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets the column header's template. The property expects the 'id' of HTMLTemplateElement, HTML string or function which returns html string.
   * Default value: 
   */
  labelTemplate?: string | HTMLTemplateElement | HTMLElement | {(label: string): string};
  /**
   * Sets the align.
   * Default value: center
   */
  align?: HorizontalAlignment | string;
  /**
   * Sets the name of the column group.
   * Default value: ""
   */
  name?: string;
  /**
   * Sets the name of the parent column group.
   * Default value: ""
   */
  parentGroup?: string;
  /**
   * Sets the vertical align.
   * Default value: center
   */
  verticalAlign?: VerticalAlignment | string;
}

export interface GridConditionalFormatting {
  /**
   * The data field of a numeric column to format. Set 'all' to format all numeric columns.
   * Default value: "all"
   */
  column?: string;
  /**
   * The formatting condition.
   * Default value: lessThan
   */
  condition?: GridConditionalFormattingCondition | string;
  /**
   * The value to compare by. When condition is 'between', this is the start (from) value.
   * Default value: 0
   */
  firstValue?: number;
  /**
   * The fontFamily to apply to formatted cells.
   * Default value: ""
   */
  fontFamily?: string;
  /**
   * The fontSize to apply to formatted cells.
   * Default value: "The default fontSize as set in CSS"
   */
  fontSize?: string;
  /**
   * The background color to apply to formatted cells.
   * Default value: "The default backgroundColor as set in CSS"
   */
  highlight?: string;
  /**
   * When condition is 'between', this is the end (to) value. Otherwise, this value is not used.
   * Default value: 1
   */
  secondValue?: number;
  /**
   * The text color to apply to formatted cells.
   * Default value: "The default color as set in CSS"
   */
  text?: string;
}

/**Sets the Grid Charting Data Visualization. */
export interface GridCharting {
  /**
   * Sets or gets whether charting is enabled.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Sets or gets the charting colors. Accepts the 'colorScheme' values of our Chart component.
   * Default value: "scheme01"
   */
  colorScheme?: string;
  /**
   * Sets or gets the chart's container.
   * Default value: null
   */
  appendTo?: string | HTMLElement;
  /**
   * Sets or gets the charting dialog.
   * Default value: [object Object]
   */
  dialog?: Dialog;
}

/**Sets or gets the charting dialog. */
export interface Dialog {
  /**
   * Sets or gets the dialog header.
   * Default value: ""
   */
  header?: string;
  /**
   * Sets or gets the dialog height.
   * Default value: 400
   */
  height?: number;
  /**
   * Sets or gets the dialog width.
   * Default value: 400
   */
  width?: number;
  /**
   * Sets or gets the dialog Left position.
   * Default value: center
   */
  left?: string | number;
  /**
   * Sets or gets the dialog Top position.
   * Default value: center
   */
  top?: string | number;
  /**
   * Sets or gets whether the dialog is enabled.
   * Default value: true
   */
  enabled?: boolean;
  /**
   * Gets whether the dialog is displayed.
   * Default value: false
   */
  visible?: boolean;
}

/**Sets the TreeGrid checkboxes. */
export interface GridCheckBoxes {
  /**
   * Sets the visibility of the TreeGrid checkboxes.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Sets the three-state mode of the TreeGrid checkboxes. In that mode, checking a checkbox affects the parent-child rows checkboxes.
   * Default value: false
   */
  hasThreeStates?: boolean;
}

/**Sets the Grid Data Export options. */
export interface GridDataExport {
  /**
   * Sets whether the columns header will be exported.
   * Default value: true
   */
  header?: boolean;
  /**
   * Sets whether the export uses the applied filters.
   * Default value: null
   */
  filterBy?: any;
  /**
   * Sets the groups of the exported data.
   * Default value: null
   */
  groupBy?: any;
  /**
   * Sets a custom style object of the dataExport. 
   * Default value: null
   */
  style?: any;
  /**
   * Sets the exported file's name.
   * Default value: "jqxGrid"
   */
  fileName?: string;
  /**
   * Sets the page orientation, when exporting to PDF.
   * Default value: portrait
   */
  pageOrientation?: GridDataExportPageOrientation | string;
  /**
   * Sets the expand char displayed when the Grid with row hierarchy(TreeGrid / Grouped) is exported.
   * Default value: "+"
   */
  expandChar?: string;
  /**
   * Sets the collapse char displayed when the Grid with row hierarchy(TreeGrid / Grouped) is exported.
   * Default value: "-"
   */
  collapseChar?: string;
  /**
   * Exports only the visible data of the Grid.
   * Default value: false
   */
  view?: boolean;
  /**
   * Determines the start row index that will be exported or printed. 'view' should be set to true
   * Default value: null
   */
  viewStart?: number | null;
  /**
   * Determines the end row index that will be exported or printed. 'view' should be set to true
   * Default value: null
   */
  viewEnd?: number | null;
  /**
   * An array of row ids that denotes the rows to export.
   * Default value: null
   */
  rowIds?: {(): void};
}

/**Sets the grid's data source settings when the <em>dataSource</em> property is set to an Array or URL. */
export interface GridDataSourceSettings {
  /**
   * Sets or gets whether a column will be auto-generated.
   * Default value: false
   */
  autoGenerateColumns?: boolean;
  /**
   * Sets or gets a children data field like 'children', 'items' in the data source. When this property is set, the dataAdapter will look for this data field when looping through the items. If it is found a hierarchical data source would be created.
   * Default value: ""
   */
  childrenDataField?: string;
  /**
   * Sets or gets the XML binding root.
   * Default value: blackList
   */
  sanitizeHTML?: GridDataSourceSettingsSanitizeHTML | string;
  /**
   * Sets or gets the XML binding root.
   * Default value: ""
   */
  root?: string;
  /**
   * Sets or gets the XML binding record.
   * Default value: ""
   */
  record?: string;
  /**
   * Sets or gets the data fields to group by.
   * Default value: []
   */
  groupBy?: string[];
  /**
   * Sets or gets the data fields which decribe the loaded data and data type. Ex: ['id: number', 'firstName: string', 'lastName: string']
   * Default value: null
   */
  dataFields?: GridDataSourceSettingsDataField[];
  /**
   * Sets or gets whether the data source type.
   * Default value: array
   */
  dataSourceType?: GridDataSourceSettingsDataSourceType | string;
  /**
   * Sets or gets the dataAdapter's id
   * Default value: ""
   */
  id?: string;
  /**
   * Sets or gets the key data field to be used for building the hierarchy. It is used in combination with the parentDataField property. Usually the 'id' field is used as key data field and 'parentId' as parent data field'
   * Default value: ""
   */
  keyDataField?: string;
  /**
   * Sets or gets the parent data field to be used for building the hierarchy. It is used in combination with the keyDataField property. Usually the 'id' field is used as key data field and 'parentId' as parent data field'
   * Default value: ""
   */
  parentDataField?: string;
  /**
   * Sets the 'mapChar' data field of the record
   * Default value: "."
   */
  mapChar?: string;
  /**
   * Sets the virtual data source function which is called each time the Grid requests data. Example for calling the callback function with the new data set: resultCallbackFunction({dataSource: data}); Demos using 'virtualDataSource' are available on the Grid demos page. Example: https://www.htmlelements.com/demos/grid/virtualscroll/
   * Default value: null
   */
  virtualDataSource?: {(resultCallbackFunction: any, details: DataAdapterVirtualDataSourceDetails): void};
  /**
   * Sets the virtual data source on expand function. This function is called when we load data on demand in Tree or TreeGrid and virtualDataSource in these components is set, too. Example: https://www.htmlelements.com/demos/grid/virtual-tree-grid/
   * Default value: null
   */
  virtualDataSourceOnExpand?: any;
}

export interface GridDataSourceSettingsDataField {
  /**
   * Sets the dataField name.
   * Default value: ""
   */
  name?: string;
  /**
   * Sets the dataField mapping path. For nested mapping, use '.'. Example: 'name.firstName'.
   * Default value: ""
   */
  map?: string;
  /**
   * Sets the dataField type.
   * Default value: string
   */
  dataType?: GridDataSourceSettingsDataFieldDataType | string;
}

/**Describes the grid's editing settings. */
export interface GridEditing {
  /**
   * Enables row header cells editing.
   * Default value: false
   */
  allowRowHeaderEdit?: boolean;
  /**
   * Enables column headers editing.
   * Default value: false
   */
  allowColumnHeaderEdit?: boolean;
  /**
   * Automatically re-applies already applied column filters and sort orders, after editing.
   * Default value: true
   */
  autoUpdateFilterAndSort?: boolean;
  /**
   * Enables editing.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Enables batch editing. This allows editing multiple grid rows on the client side and sending them with a single server request.
   * Default value: false
   */
  batch?: boolean;
  /**
   * Determines the way editing is initiated.
   * Default value: click
   */
  action?: GridEditingAction | string;
  /**
   * Describes command keys.
   * Default value: [object Object]
   */
  commandKeys?: GridEditingCommandKeys;
  /**
   * Describes the grid's command bar settings. The command bar is a toolbar or statusbar with tools for saving and reverting edits.
   * Default value: [object Object]
   */
  commandBar?: GridEditingCommandBar;
  /**
   * Describes the grid's command column settings. The command column can be used to edit or delete a row.
   * Default value: [object Object]
   */
  commandColumn?: GridEditingCommandColumn;
  /**
   * Sets the grid's edit mode.
   * Default value: cell
   */
  mode?: GridEditingMode | string;
  /**
   * Describes the settings of the 'Add New Row' UI element which enables the quick adding of rows to the Grid with a single click.
   * Default value: [object Object]
   */
  addNewRow?: GridEditingAddNewRow;
  /**
   * Enables users to dynamically add new columns through the User Interface. When the add new column is visible, a '+' is displayed as a last column. Clicking it opens a dialog for adding new columns.
   * Default value: [object Object]
   */
  addNewColumn?: GridEditingAddNewColumn;
  /**
   * Describes dialog's editing settings.
   * Default value: [object Object]
   */
  dialog?: Dialog;
  /**
   * Describes add dialog's settings
   * Default value: [object Object]
   */
  addDialog?: Dialog;
  /**
   * Describes delete dialog's settings
   * Default value: [object Object]
   */
  deleteDialog?: Dialog;
}

/**Describes command keys. */
export interface GridEditingCommandKeys {
  /**
   * Describes the edit command key.
   * Default value: [object Object]
   */
  commandKeyEdit?: GridCommandKey;
  /**
   * Describes the cancel command key.
   * Default value: [object Object]
   */
  commandKeyCancel?: GridCommandKey;
  /**
   * Describes the update command key.
   * Default value: [object Object]
   */
  commandKeyUpdate?: GridCommandKey;
}

/**Describes the edit command key. */
export interface GridCommandKey {
  /**
   * Sets the name of the edit key command.
   * Default value: "commandKeyEditCommand"
   */
  command?: string;
  /**
   * Sets the key that invokes the edit command.
   * Default value: "F2"
   */
  key?: string;
}

/**Describes the grid's command bar settings. The command bar is a toolbar or statusbar with tools for saving and reverting edits. */
export interface GridEditingCommandBar {
  /**
   * Makes the command bar visible.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Sets the command bar's position.
   * Default value: near
   */
  position?: LayoutPosition | string;
  /**
   * Sets what is to be displayed in command bar buttons.
   * Default value: labelAndIcon
   */
  displayMode?: GridCommandDisplayMode | string;
  /**
   * Sets the command bar's data source.
   * Default value: [object Object]
   */
  dataSource?: GridEditingCommandBarDataSource;
}

/**Sets the command bar's data source. */
export interface GridEditingCommandBarDataSource {
  /**
   * Describes the settings of the command bar's button for adding rows.
   * Default value: [object Object]
   */
  commandBarAddRow?: GridCommand;
  /**
   * Describes the settings of the command bar's button for deleting rows.
   * Default value: [object Object]
   */
  commandBarDeleteRow?: GridCommand;
  /**
   * Describes the settings of the command bar's button for saving changes.
   * Default value: [object Object]
   */
  commandBarBatchSave?: GridCommand;
  /**
   * Describes the settings of the command bar's button for reverting changes.
   * Default value: [object Object]
   */
  commandBarBatchRevert?: GridCommand;
}

/**Describes the grid's command column settings. The command column can be used to edit or delete a row. */
export interface GridEditingCommandColumn {
  /**
   * Makes the command column visible.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Enables inline display of the command column.
   * Default value: false
   */
  inline?: boolean;
  /**
   * Sets the command column's position.
   * Default value: far
   */
  position?: Position | string;
  /**
   * Sets what is to be displayed in command column buttons.
   * Default value: icon
   */
  displayMode?: GridCommandDisplayMode | string;
  /**
   * Sets the command column's data source.
   * Default value: [object Object]
   */
  dataSource?: GridEditingCommandColumnDataSource;
  /**
   * Sets the width of the command column.
   * Default value: null
   */
  width?: number | null;
}

/**Sets the command column's data source. */
export interface GridEditingCommandColumnDataSource {
  /**
   * Describes the settings of the command column's button displayed in the column's header. By default, this Command opens a Menu with Column Chooser.
   * Default value: [object Object]
   */
  commandColumnMenu?: GridCommand;
  /**
   * Describes the settings of the command column's button for editing rows.
   * Default value: [object Object]
   */
  commandColumnEdit?: GridCommand;
  /**
   * Describes the settings of the command column's button for deleting rows.
   * Default value: [object Object]
   */
  commandColumnDelete?: GridCommand;
  /**
   * Describes the settings of the command column's button for updating rows.
   * Default value: [object Object]
   */
  commandColumnUpdate?: GridCommand;
  /**
   * Describes the settings of the command column's button for canceling edits.
   * Default value: [object Object]
   */
  commandColumnCancel?: GridCommand;
  /**
   * Describes the settings of the command column's row menu button.
   * Default value: [object Object]
   */
  commandColumnRowMenu?: GridCommand;
  /**
   * Describes the settings of the command column's custom button.
   * Default value: [object Object]
   */
  commandColumnCustom?: GridCommand;
}

/**Describes the settings of the 'Add New Row' UI element which enables the quick adding of rows to the Grid with a single click. */
export interface GridEditingAddNewRow {
  /**
   * Adds new row by clicking the Enter button, when the focused row is the last row.
   * Default value: false
   */
  autoCreate?: boolean;
  /**
   * Determines whether the newly added row enters automatically in edit mode, when added.
   * Default value: true
   */
  autoEdit?: boolean;
  /**
   * Sets the position of the 'Add New Row' UI element.
   * Default value: both
   */
  position?: LayoutPosition | string;
  /**
   * Sets or gets the display mode of the new row action. It could be either 'row' or 'button'.
   * Default value: row
   */
  displayMode?: GridEditingAddNewRowDisplayMode | string;
  /**
   * Makes the 'Add New Row' UI element visible.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Sets the label of the 'Add New Row' UI element.
   * Default value: ""
   */
  label?: string;
}

/**Enables users to dynamically add new columns through the User Interface. When the add new column is visible, a '+' is displayed as a last column. Clicking it opens a dialog for adding new columns. */
export interface GridEditingAddNewColumn {
  /**
   * Makes the 'Add New Row' UI element visible.
   * Default value: false
   */
  visible?: boolean;
}

/**Describes the grid's filtering settings. */
export interface GridFiltering {
  /**
   * Enables filtering.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Determines the filtering operator used in the Grid. By default filters are applied with 'and' operator i.e returns a set of rows matching the filter expressions of columnA AND columnB. The other option is to return a set of rows matching the filter expressions of columnA OR columnB. For example: grid.filtering.operator = 'or'; grid.addFilter('lastName', 'contains "davolio"') grid.addFilter('firstName', 'contains "Antoni"'); - that code will apply two filters to the Grid and will return all rows where firstName is 'Antoni' or the lastName is 'Davolio'
   * Default value: "and"
   */
  operator?: string;
  /**
   * An array of filtering conditions to apply to the DataGrid. Each member of the filter array is an array with two members. The first one is the column dataField to apply the filter to. The second one is the filtering condition. Example: [['firstName', 'contains Andrew or contains Nancy'], ['quantity', '&lt;= 3 and &gt;= 8']]. Additional example with filter which we want to apply to a column with filterMenuMode='excel' - [['firstName', 'EQUAL' 'Andrew' or 'EQUAL' 'Antoni' or 'EQUAL' 'Beate']]. Example with a string filter applied to a string column - [['lastName','CONTAINS' 'burke' or 'CONTAINS' 'peterson']]. Filter conditions which you can use in the expressions: '=', 'EQUAL','&lt;&gt;', 'NOT_EQUAL', '!=', '&lt;', 'LESS_THAN','&gt;', 'GREATER_THAN', '&lt;=', 'LESS_THAN_OR_EQUAL', '&gt;=', 'GREATER_THAN_OR_EQUAL','starts with', 'STARTS_WITH','ends with', 'ENDS_WITH', '', 'EMPTY', 'CONTAINS','DOES_NOT_CONTAIN', 'NULL','NOT_NULL'
   * Default value: 
   */
  filter?: any[];
  /**
   * Describes the filter row's settings.
   * Default value: [object Object]
   */
  filterRow?: GridFilteringFilterRow;
  /**
   * Describes the settings for the filter menu.
   * Default value: [object Object]
   */
  filterMenu?: GridFilteringFilterMenu;
  /**
   * (In Development)Describes the settings for the filter builder.
   * Default value: [object Object]
   */
  filterBuilder?: GridFilteringFilterBuilder;
}

/**Describes the filter row's settings. */
export interface GridFilteringFilterRow {
  /**
   * Makes the filter row visible.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Makes the filter row's menu visible.
   * Default value: false
   */
  menuVisible?: boolean;
  /**
   * Sets the way filtering through the filter row is applied.
   * Default value: auto
   */
  applyMode?: GridFilteringFilterRowApplyMode | string;
  /**
   * Sets the delay (in milliseconds) before applying filtering (when filtering.filterRow.applyMode is 'auto').
   * Default value: 300
   */
  autoApplyModeDelay?: number;
}

/**Describes the settings for the filter menu. */
export interface GridFilteringFilterMenu {
  /**
   * Sets the visibility of the filter menu.
   * Default value: true
   */
  visible?: boolean;
  /**
   * Sets the button text of the filter menu.
   * Default value: cancel,clear,filter
   */
  buttons?: string[];
  /**
   * Sets the filter menu mode.
   * Default value: default
   */
  mode?: GridFilteringFilterMenuMode | string;
  /**
   * Sets the filter menu datasource.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Sets the width of the filter menu.
   * Default value: 250
   */
  width?: number;
  /**
   * Sets the height of the filter menu.
   * Default value: 200
   */
  height?: number;
}

/**(In Development)Describes the settings for the filter builder. */
export interface GridFilteringFilterBuilder {
  /**
   * Sets the visibility of the filter builder.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Sets the height of the filter builder.
   * Default value: null
   */
  height?: number | null;
}

/**Describes the grid's grouping settings. */
export interface GridGrouping {
  /**
   * Enables grouping.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Enables collapsing of groups.
   * Default value: false
   */
  allowCollapse?: boolean;
  /**
   * Automatically expands all groups.
   * Default value: false
   */
  autoExpandAll?: boolean;
  /**
   * Automatically expands all groups to a given level.
   * Default value: 0
   */
  autoExpandToLevel?: number;
  /**
   * Automatically hides all grouped columns.
   * Default value: false
   */
  autoHideGroupColumn?: boolean;
  /**
   * Sets the group expand mode.
   * Default value: buttonClick
   */
  expandMode?: GridGroupingExpandMode | string;
  /**
   * Sets or gets the column's format function.
   * Default value: null
   */
  formatFunction?: {(formatObject: {row?: GridRow, column?: GridColumn, cell?: GridCell, value?: any, template?: any}): void};
  /**
   * Sets the group row height.
   * Default value: 50
   */
  groupRowHeight?: string | number;
  /**
   * Sets or gets the data fields to group by.
   * Default value: []
   */
  groupBy?: string[];
  /**
   * Sets the indent of the group.
   * Default value: 16
   */
  groupIndent?: number;
  /**
   * Describes the group bar's settings.
   * Default value: [object Object]
   */
  groupBar?: GridGroupingGroupBar;
  /**
   * Expands a group in the first grid render. Example: onGroupDefaultExpanded: (dataItem) =&gt;{ return dataItem.label === 'Peppermint Mocha Twist' }
   * Default value: null
   */
  onGroupDefaultExpanded?: any;
  /**
   * Sets the group render mode. 'basic' mode renders the group headers without taking into account the indent, groupRowHeight and column label properties. 'compact' mode is the same as basic, but also renders the column labels in the group headers. The default mode is 'advanced', which adds indents to groups that depend on the group level. In 'multipleColumns' mode, each group is displayed in its column.
   * Default value: advanced
   */
  renderMode?: GridGroupingRenderMode | string;
  /**
   * Sets the indent of the group toggle button.
   * Default value: 16
   */
  toggleButtonIndent?: number;
  /**
   * Describes the group summary row's settings.
   * Default value: [object Object]
   */
  summaryRow?: GridGroupingSummaryRow;
}

/**Describes the group bar's settings. */
export interface GridGroupingGroupBar {
  /**
   * Makes the group bar visible.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Enables column drag and drop.
   * Default value: false
   */
  allowColumnDragDrop?: boolean;
  /**
   * Enables column close buttons.
   * Default value: true
   */
  allowColumnCloseButtons?: boolean;
}

/**Describes the group summary row's settings. */
export interface GridGroupingSummaryRow {
  /**
   * Enables inline display of the group summary row.
   * Default value: true
   */
  inline?: boolean;
  /**
   * Makes the group summary row visible.
   * Default value: true
   */
  visible?: boolean;
}

/**Sets the grid's image and filter upload settings for the image and attachment columns. */
export interface GridUploadSettings {
  /**
   * Sets or file/image upload url.
   * Default value: ""
   */
  url?: string;
  /**
   * Sets or file/image remove url.
   * Default value: ""
   */
  removeUrl?: string;
  /**
   * Sets or gets the upload field name. In the backend, you can use this name to access the images data. For example in expressJS, you can use something like that: const images = req['files']['userfile[]'];
   * Default value: "userfile[]"
   */
  name?: string;
  /**
   * Additional data to pass to the server. The format should be a JSON string.
   * Default value: ""
   */
  data?: string;
  /**
   * Function called when the upload is completed. JSON object with 'files', 'status', 'fileURL' and 'serverResponse' are passed as parameters when the function is called by the Grid.
   * Default value: 
   */
  onUploadCompleted?: any;
  /**
   * Function called when the upload has failed. JSON object with 'files', 'status' and 'serverResponse' are passed as parameters when the function is called by the Grid.
   * Default value: 
   */
  onUploadError?: any;
}

/**Describes the paging settings. */
export interface GridPaging {
  /**
   * Enables pagination.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Describes the spinner pagination settings.
   * Default value: [object Object]
   */
  spinner?: GridPagingSpinner;
  /**
   * Sets the number of rows per page.
   * Default value: 10
   */
  pageSize?: number;
  /**
   * Sets the number of hierarchical rows per page. For example, displays 2 root groups per page, when grouping is enabled.
   * Default value: 2
   */
  pageHierarchySize?: number;
  /**
   * Sets the start page.
   * Default value: 0
   */
  pageIndex?: number;
}

/**Describes the spinner pagination settings. */
export interface GridPagingSpinner {
  /**
   * Enables spinner pagination.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Sets the step of page
   * Default value: 1
   */
  step?: number;
}

/**Describes the pager settings. */
export interface GridPager {
  /**
   * Sets the ellipsis display mode.
   * Default value: both
   */
  autoEllipsis?: GridPagerAutoEllipsis | string;
  /**
   * Sets the position of pager.
   * Default value: far
   */
  position?: LayoutPosition | string;
  /**
   * Sets a template for the pager.
   * Default value: 
   */
  template?: string | HTMLTemplateElement;
  /**
   * Describes the settings for the 'rows per page' option.
   * Default value: [object Object]
   */
  pageSizeSelector?: GridPagerPageSizeSelector;
  /**
   * Describes the summary settings.
   * Default value: [object Object]
   */
  summary?: GridPagerSummary;
  /**
   * Describes the navigation buttons settings.
   * Default value: [object Object]
   */
  navigationButtons?: GridPagerNavigationButtons;
  /**
   * Describes the settings about navigation input option.
   * Default value: [object Object]
   */
  navigationInput?: GridPagerNavigationInput;
  /**
   * Describes the settings for the numeric page buttons.
   * Default value: [object Object]
   */
  pageIndexSelectors?: GridPagerPageIndexSelectors;
  /**
   * Sets the visibility of pager.
   * Default value: false
   */
  visible?: boolean;
}

/**Describes the settings for the 'rows per page' option. */
export interface GridPagerPageSizeSelector {
  /**
   * Sets the visibility of the 'rows per page' option.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Sets the count of the 'rows per page' option.
   * Default value: 10,20,50
   */
  dataSource?: any;
  /**
   * Sets the position of the 'rows per page' option.
   * Default value: far
   */
  position?: Position | string;
}

/**Describes the summary settings. */
export interface GridPagerSummary {
  /**
   * Sets the position of the summary.
   * Default value: far
   */
  position?: Position | string;
  /**
   * Sets the visibility of the summary.
   * Default value: false
   */
  visible?: boolean;
}

/**Describes the navigation buttons settings. */
export interface GridPagerNavigationButtons {
  /**
   * Sets the navigation buttons position.
   * Default value: both
   */
  position?: LayoutPosition | string;
  /**
   * Describes the settings about buttons 'previous page' and 'next page'.
   * Default value: [object Object]
   */
  prevNextButtons?: GridPagerNavigationButtonsPrevNextButtons;
  /**
   * Describes the settings about buttons 'first page' and 'last page'.
   * Default value: [object Object]
   */
  firstLastButtons?: GridPagerNavigationButtonsFirstLastButtons;
  /**
   * Describes the labels settings for navigation buttons.
   * Default value: [object Object]
   */
  labels?: GridPagerNavigationButtonsLabels;
}

/**Describes the settings about buttons 'previous page' and 'next page'. */
export interface GridPagerNavigationButtonsPrevNextButtons {
  /**
   * Sets the visibility of 'previous page' and 'next page' buttons.
   * Default value: true
   */
  visible?: boolean;
}

/**Describes the settings about buttons 'first page' and 'last page'. */
export interface GridPagerNavigationButtonsFirstLastButtons {
  /**
   * Sets the visibility of 'first page' and 'last page' buttons.
   * Default value: true
   */
  visible?: boolean;
}

/**Describes the labels settings for navigation buttons. */
export interface GridPagerNavigationButtonsLabels {
  /**
   * Sets the visibility of labels for navigation buttons.
   * Default value: false
   */
  visible?: boolean;
}

/**Describes the settings about navigation input option. */
export interface GridPagerNavigationInput {
  /**
   * Sets the position of navigation input option.
   * Default value: far
   */
  position?: Position | string;
  /**
   * Sets the visibility of navigation input option.
   * Default value: false
   */
  visible?: boolean;
}

/**Describes the settings for the numeric page buttons. */
export interface GridPagerPageIndexSelectors {
  /**
   * Sets the visibility of numeric page buttons.
   * Default value: true
   */
  visible?: boolean;
  /**
   * Sets the number of visible page buttons.
   * Default value: 10
   */
  dataSource?: any;
}

/**Sets the row details. */
export interface GridRowDetail {
  /**
   * Enables the row details.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Sets the height of the row details.
   * Default value: 200
   */
  height?: number;
  /**
   * Sets the position of the Column which allows you to dynamically expand/collapse the row details.
   * Default value: near
   */
  position?: Position | string;
  /**
   * Sets the template of the row details.
   * Default value: 
   */
  template?: string | HTMLTemplateElement;
  /**
   * Sets the visibility of the Column which allows you to dynamically expand/collapse the row details.
   * Default value: true
   */
  visible?: boolean;
  /**
   * Sets the row details dialog. When enabled, row details will be displayed in a Dialog.
   * Default value: [object Object]
   */
  dialog?: Dialog;
}

/**Describes the column header settings. */
export interface GridColumnHeader {
  /**
   * Sets the column header visibility.
   * Default value: true
   */
  visible?: boolean;
}

/**Describes the summary row settings. */
export interface GridSummaryRow {
  /**
   * Sets the summary row visibility.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Sets the summary row editor. When you point over a summary row cell, an editor is displayed and you will be able to dynamically change the summary type.
   * Default value: false
   */
  editing?: boolean;
}

/**Sets the grid's state settings. */
export interface GridStateSettings {
  /**
   * Enables or disables auto-save of the Grid's state
   * Default value: false
   */
  autoSave?: boolean;
  /**
   * Enables or disables save/load of the grid state.
   * Default value: true
   */
  enabled?: boolean;
  /**
   * Sets or gets the current state.
   * Default value: ""
   */
  current?: string;
  /**
   * Container which is auto-filled with state objects when you call the saveState method or enable the autoSave of states. Each object has a key which is the state's name and value which is a json with the state's properties.
   * Default value: null
   */
  storage?: any;
  /**
   * Function called when the state is changed.
   * Default value: 
   */
  onStateChange?: any;
  /**
   * Array with state options such as 'sort', 'filter', 'expandedRows', 'paging', 'selectedCells', 'selectedRows', 'group', 'columns'. This property determines which parts of the state would be saved or loaded.
   * Default value: 
   */
  options?: string[];
}

/**Describes the settings for the group header. */
export interface GridGroupHeader {
  /**
   * Sets the visibility of the group header.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Sets a template for the group header.
   * Default value: 
   */
  template?: string | HTMLTemplateElement;
}

/**Describes the header settings of the grid. */
export interface GridHeader {
  /**
   * Sets the header visibility.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Sets a template for the header.
   * Default value: 
   */
  template?: string | HTMLTemplateElement | {(element: HTMLElement): void};
  /**
   * This callback function can be used for customization of the Header toolbar. The Toolbar HTML Element is passed as an argument.
   * Default value: null
   */
  onInit?: {(element: HTMLElement): void};
  /**
   * Determines the buttons displayed in the Grid header. 'columns' displays a button opening the columns chooser panel. 'filter'  displays a button opening the filtering panel.  'group' displays a button opening the grouping panel. 'sort'  displays a button opening the sorting panel. 'format'  displays a button opening the conditional formatting panel. 'search' displays a button opening the search panel.
   * Default value: [ "columns", "filter", "group", "sort", "format", "search" ]
   */
  buttons?: string[];
}

/**Describes the footer settings of the grid. */
export interface GridFooter {
  /**
   * Sets the footer visibility.
   * Default value: false
   */
  visible?: boolean;
  /**
   * Sets a template for the footer.
   * Default value: 
   */
  template?: string | HTMLTemplateElement | {(element: HTMLElement): void};
}

export interface GridRow {
  /**
   * Sets or gets the row can be expanded or collapsed.
   * Default value: true
   */
  allowToggle?: boolean;
  /**
   * Sets or gets the row can be resized.
   * Default value: true
   */
  allowResize?: boolean;
  /**
   * Sets or gets the row can be selected.
   * Default value: true
   */
  allowSelect?: boolean | null;
  /**
   * Sets or gets the row can be checked. This property is used when the Grid is in Tree Grid or Grouped mode.
   * Default value: true
   */
  allowCheck?: boolean;
  /**
   * Sets or gets the row's check state. This property is used when the Grid is in Tree Grid or Grouped mode.
   * Default value: true
   */
  checked?: boolean;
  /**
   * Gets the Row's Cells array.
   * Default value: []
   */
  cells?: GridCell[];
  /**
   * Gets the row's children array of GridRow. This property is associated to the TreeGrid and Groupng mode of the Grid.
   * Default value: null
   */
  children?: any;
  /**
   * Gets the row's bound data.
   * Default value: null
   */
  data?: any;
  /**
   * Sets or gets the row's detail height.
   * Default value: 200
   */
  detailHeight?: number;
  /**
   * Sets or gets the row's detail template.
   * Default value: null
   */
  detailTemplate?: string | HTMLTemplateElement | null;
  /**
   * Gets the HTML Element. The property returns null when the Row is not in the View.
   * Default value: null
   */
  element?: any;
  /**
   * Sets or gets the row is expanded. This property is used when the Grid is in Tree Grid or Grouped mode.
   * Default value: false
   */
  expanded?: boolean;
  /**
   * Gets the row's header element.
   * Default value: null
   */
  header?: HTMLElement;
  /**
   * Sets or gets the row's height.
   * Default value: 30
   */
  height?: number;
  /**
   * Gets the row's bound index.
   * Default value: -1
   */
  index?: number;
  /**
   * Gets the row's bound id.
   * Default value: 
   */
  id?: string | number;
  /**
   * Gets whether the row is leaf row in TreeGrid or Grouping mode.
   * Default value: false
   */
  leaf?: boolean;
  /**
   * Sets or gets the row's maximum height.
   * Default value: 100
   */
  maxHeight?: number;
  /**
   * Sets or gets the row's minimum height.
   * Default value: 30
   */
  minHeight?: number;
  /**
   * Sets or gets the Freeze mode. Accepts: 'near', 'far', true and false. Freezes/Pins the row to top(near) or bottom(far).
   * Default value: false
   */
  freeze?: Position | string;
  /**
   * Sets or gets whether the row is selected.
   * Default value: false
   */
  selected?: boolean;
  /**
   * Sets or gets whether the row detail is displayed.
   * Default value: false
   */
  showDetail?: boolean;
  /**
   * "Method which applies a style object to all cells. Expects a JSON object with the following allowed values: 'background', 'color', 'fontSize', 'fontFamily', 'fontWeight', 'fontStyle', 'textDecoration'
   * Default value: undefined
   */
  setStyle?: {(value: any): void};
  /**
   * Sets or gets whether the row is visible. Set the property to 'false' to hide the row.
   * Default value: true
   */
  visible?: boolean;
  /**
   * Gets the visible index of the row.
   * Default value: -1
   */
  visibleIndex?: number;
  /**
   * Methods which gets a cell, which is inside a row. A dataField string is a required argument, when you call this method.
   * Default value: -1
   */
  getCell?: any;
}

export interface GridCell {
  /**
   * "Sets or gets the horizontal alignment. Allowed values are: 'left', 'center' or 'right'".
   * Default value: "'left'"
   */
  align?: string;
  /**
   * Gets the column associated to the cell.
   * Default value: null
   */
  column?: any;
  /**
   * Sets or gets the cell's text color.
   * Default value: "''"
   */
  color?: string;
  /**
   * Sets or gets the cell's background.
   * Default value: "''"
   */
  background?: string;
  /**
   * Sets or gets the cell's borderColor.
   * Default value: "''"
   */
  borderColor?: string;
  /**
   * Sets or gets the cell's colSpan.
   * Default value: 1
   */
  colSpan?: number;
  /**
   * Sets or gets the cell's editor.
   * Default value: null
   */
  editor?: any;
  /**
   * Gets the HTMLElement associated to the cell.
   * Default value: null
   */
  element?: HTMLElement;
  /**
   * Gets a formatted number or Date.
   * Default value: null
   */
  getFormattedValue?: {(value: any, type: string): void};
  /**
   * Gets whether the cell is in edit mode.
   * Default value: false
   */
  isEditing?: boolean;
  /**
   * Gets the old value of the cell
   * Default value: null
   */
  oldValue?: any;
  /**
   * Sets or gets the cell's fontSize
   * Default value: "''"
   */
  fontSize?: string;
  /**
   * Sets or gets the cell's fontWeight
   * Default value: "''"
   */
  fontWeight?: string;
  /**
   * Sets or gets the cell's fontFamily
   * Default value: "''"
   */
  fontFamily?: string;
  /**
   * Sets or gets the cell's fontStyle
   * Default value: "''"
   */
  fontStyle?: string;
  /**
   * "Sets or gets whether the cell can be edited.".
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Gets the row object associated to the cell.
   * Default value: null
   */
  row?: any;
  /**
   * Sets or gets the row span.
   * Default value: 1
   */
  rowSpan?: number;
  /**
   * Sets or gets whether the cell is selected.
   * Default value: false
   */
  selected?: boolean;
  /**
   * "Each property is an object{name: string, value: any}. This function allows you to update multiple properties with single refresh.".
   * Default value: 
   */
  setProperties?: {(properties: {name: string, value: string}[]): void};
  /**
   * "Allowed values are: 'top', 'middle' or 'bottom'".
   * Default value: ""
   */
  tooltip?: string;
  /**
   * The cell's value
   * Default value: null
   */
  value?: any;
  /**
   * "Sets or gets the vertical alignment. Allowed values are: 'top', 'center' or 'bottom'".
   * Default value: "'center'"
   */
  verticalAlign?: string;
  /**
   * "Method which applies a cell style object. Expects a JSON object with the following possible values: 'background', 'color', 'fontSize', 'fontFamily', 'fontWeight', 'fontStyle', 'textDecoration'
   * Default value: undefined
   */
  setStyle?: {(value: any): void};
}

/**Describes the selection settings. */
export interface GridSelection {
  /**
   * Enables the selection option.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Sets or gets whether selection by clicking on a Row header is allowed.
   * Default value: false
   */
  allowRowHeaderSelection?: boolean;
  /**
   * Sets or gets whether selection by clicking on a Column header is allowed.
   * Default value: false
   */
  allowColumnHeaderSelection?: boolean;
  /**
   * Sets or gets whether selection by clicking on a Row is allowed.
   * Default value: true
   */
  allowRowSelection?: boolean;
  /**
   * Sets or gets whether selection by clicking on a cell is allowed.
   * Default value: false
   */
  allowCellSelection?: boolean;
  /**
   * Sets or gets whether selection by dragging(like in Excel) is allowed.
   * Default value: true
   */
  allowDragSelection?: boolean;
  /**
   * Sets or gets whether selection by dragging will automatically scroll the Grid view.
   * Default value: true
   */
  allowDragSelectionAutoScroll?: boolean;
  /**
   * Sets or gets whether the Cells selection bottom-right corner selection 'Drag Handle' (litte square like in Excel) is displayed. That handle allows you to resize the selection horizontally or vertically
   * Default value: true
   */
  allowCellDragSelectionHandle?: boolean;
  /**
   * Sets or gets whether the Cells selection can be dragged and dropped. Drag happens when the cursor is moved to the bottom of the cells selection. The cursor is changed to a drag cursor. Press the button and hold and move the selection. By default the dragged cell values are copied on drop.
   * Default value: true
   */
  allowCellDragDropSelectionHandle?: boolean;
  /**
   * Sets or gets whether the Cells selection will be auto-filled with values on drop when dragging through the 'Drag Handle'.
   * Default value: true
   */
  allowCellDragSelectionAutoFill?: boolean;
  /**
   * Sets or gets whether the default browser's text selection is enabled.
   * Default value: false
   */
  defaultSelection?: boolean;
  /**
   * Sets or gets whether the selection allows you to select 'one', 'many' or a variation of 'many' called 'extended'. 'one' allows you to have only single cell or row selected. 'many' 
   * Default value: many
   */
  mode?: GridSelectionMode | string;
  /**
   * Sets or gets whether the selection happens on 'click' or 'doubleClick'. 'none' means that selection can happen only through API.
   * Default value: click
   */
  action?: GridSelectionAction | string;
  /**
   * 
   * Default value: [object Object]
   */
  checkBoxes?: GridSelectionCheckBoxes;
  /**
   * 
   * Default value: ""
   */
  selected?: string;
}

export interface GridSelectionCheckBoxes {
  /**
   * Sets or gets whether the checkboxes are automatically displayed only when the mouse is over the Grid. When false, checkboses are always displayed
   * Default value: false
   */
  autoShow?: boolean;
  /**
   * Sets or gets whether the checkboxes selection is enabled. In that mode a new column with checkboxes is displayed.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Sets or gets whether the selection happens on 'click' or 'doubleClick'. 'none' means that selection can happen only through API.
   * Default value: click
   */
  action?: GridSelectionAction | string;
  /**
   * Sets or gets whether the checkbox selection selects all rows in the current page or all rows. The 'none' setting disables the header checkbox.
   * Default value: page
   */
  selectAllMode?: GridSelectionCheckBoxesSelectAllMode | string;
  /**
   * Sets or gets whether the position of the checkbox selection column.
   * Default value: near
   */
  position?: Position | string;
}

/**Describes sorting settings. */
export interface GridSorting {
  /**
   * Enables sorting.
   * Default value: false
   */
  enabled?: boolean;
  /**
   * Sets the sort columns to be sorted.
   * Default value: 
   */
  sort?: string[];
  /**
   * Maintains sorting when user edits data in the sorted column. The feature is useful when you want to apply sort just once and you set the property to false.
   * Default value: true
   */
  maintainSort?: boolean;
  /**
   * Sets the count of allowed sorting columns. When the property value is set to 'many', users can sort data by multiple columns.
   * Default value: one
   */
  mode?: GridSortingMode | string;
  /**
   * Sets the command key. The property is used in the multi-column sorting. If commandKey='Control', users will be able to sort by multiple columns only while holding the 'Control' key.
   * Default value: Default
   */
  commandKey?: GridSortingCommandKey | string;
  /**
   * Enables switching between the three sort states: ascending, descending and not sorted.
   * Default value: true
   */
  sortToggleThreeStates?: boolean;
  /**
   * Enables switching between the sort states on column click. This is the default behavior.
   * Default value: true
   */
  sortToggleOnClick?: boolean;
  /**
   * Enables switching between the sort states on column click and holding down the command key.
   * Default value: false
   */
  sortToggleOnClickAndCommandKey?: boolean;
}

declare global {
    interface Document {
        createElement(tagName: "smart-grid"): Grid;
        querySelector(selectors: "smart-grid"): Grid | null;
        querySelectorAll(selectors: "smart-grid"): NodeListOf<Grid>;
        getElementsByTagName(qualifiedName: "smart-grid"): HTMLCollectionOf<Grid>;
        getElementsByName(elementName: "smart-grid"): NodeListOf<Grid>;
    }
}

/**Generates labels as 'numbers' or 'letters'. This property affects the rendering of the row header. */
export declare type GridAppearanceAutoGenerateRowLabelMode = 'number' | 'letter';
/**Generates labels as 'numbers' or 'letters.  This property affects the rendering of the column header. */
export declare type GridAppearanceAutoGenerateColumnLabelMode = 'number' | 'letter';
/**Sets the row resize mode. split resize mode 'grows' or 'shrinks' the resize element's size and 'shrinks' or 'grows' the next sibling element's size. growAndShrink resize mode 'grows' or 'shrinks' the resize element's size */
export declare type GridResizeMode = 'none' | 'split' | 'growAndShrink';
/**Sets or gets whether the copy-pasted values will be auto-filled by using automatic pattern detection. This is used in the Drag&Drop Multiple Cells selection. none does nothing. copy just copies the cells. 'fillSeries' detects and automatically fills the values. For example, if the selection has '1, 2' and the possible positions are more, the pasted values would be '1, 2, 3, 4, etc. */
export declare type GridClipboardAutoFillMode = 'none' | 'copy' | 'fillSeries';
/**Sets or gets whether the position of the checkbox selection column. */
export declare type Position = 'near' | 'far';
/**Sets or gets the filter menu mode of the column. In 'basic' mode, a single input is displayed in the filter menu. In 'default' mode, two input options are available for more precise filtering. In 'excel' mode, checked list with unique values is displayed. */
export declare type GridColumnFilterMenuMode = 'basic' | 'default' | 'excel';
/**Sets or gets the sort order of the column. Accepts: 'asc', 'desc' and null. */
export declare type GridColumnSortOrder = 'asc' | 'desc' | null;
/**The formatting condition. */
export declare type GridConditionalFormattingCondition = 'between' | 'equal' | 'greaterThan' | 'lessThan' | 'notEqual';
/**Sets the page orientation, when exporting to PDF. */
export declare type GridDataExportPageOrientation = 'landscape' | 'portrait';
/**Sets or gets the XML binding root. */
export declare type GridDataSourceSettingsSanitizeHTML = 'all' | 'blackList' | 'none';
/**Sets the dataField type. */
export declare type GridDataSourceSettingsDataFieldDataType = 'string' | 'date' | 'boolean' | 'number' | 'array' | 'any';
/**Sets or gets whether the data source type. */
export declare type GridDataSourceSettingsDataSourceType = 'array' | 'json' | 'xml' | 'csv' | 'tsv';
/**Determines the way editing is initiated. */
export declare type GridEditingAction = 'none' | 'click' | 'doubleClick';
/**Sets what is to be displayed in command column buttons. */
export declare type GridCommandDisplayMode = 'label' | 'icon' | 'labelAndIcon';
/**Sets the grid's edit mode. */
export declare type GridEditingMode = 'cell' | 'row';
/**Sets or gets the display mode of the new row action. It could be either 'row' or 'button'. */
export declare type GridEditingAddNewRowDisplayMode = 'row' | 'button';
/**Sets the way filtering through the filter row is applied. */
export declare type GridFilteringFilterRowApplyMode = 'auto' | 'click';
/**Sets the filter menu mode. */
export declare type GridFilteringFilterMenuMode = 'default' | 'excel';
/**Sets the group expand mode. */
export declare type GridGroupingExpandMode = 'buttonClick' | 'rowClick';
/**Sets the group render mode. 'basic' mode renders the group headers without taking into account the indent, groupRowHeight and column label properties. 'compact' mode is the same as basic, but also renders the column labels in the group headers. The default mode is 'advanced', which adds indents to groups that depend on the group level. In 'multipleColumns' mode, each group is displayed in its column. */
export declare type GridGroupingRenderMode = 'basic' | 'compact' | 'advanced' | 'multipleColumns';
/**Sets the ellipsis display mode. */
export declare type GridPagerAutoEllipsis = 'none' | 'before' | 'after' | 'both';
/**Sets or gets whether the selection allows you to select 'one', 'many' or a variation of 'many' called 'extended'. 'one' allows you to have only single cell or row selected. 'many'  */
export declare type GridSelectionMode = 'one' | 'many' | 'extended';
/**Sets or gets whether the selection happens on 'click' or 'doubleClick'. 'none' means that selection can happen only through API. */
export declare type GridSelectionAction = 'none' | 'click' | 'doubleClick';
/**Sets or gets whether the checkbox selection selects all rows in the current page or all rows. The 'none' setting disables the header checkbox. */
export declare type GridSelectionCheckBoxesSelectAllMode = 'none' | 'page' | 'all';
/**Sets the count of allowed sorting columns. When the property value is set to 'many', users can sort data by multiple columns. */
export declare type GridSortingMode = 'one' | 'many';
/**Sets the command key. The property is used in the multi-column sorting. If commandKey='Control', users will be able to sort by multiple columns only while holding the 'Control' key. */
export declare type GridSortingCommandKey = 'Default' | 'Alt' | 'Control' | 'Shift';
export interface GroupPanelProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Sets or gets the the position of the close button of group panel items.
   * Default value: left
   */
  closeButtonPosition?: GroupPanelCloseButtonPosition | string;
  /**
   * Determines the data source that will be loaded to the group panel.Each member of the dataSource array is an object with the following fields:dataField - the dataField of the column to be grouped.dataType - the data type of the column to be grouped.groupIndex - the group order of columns. If this value is -1, the grouping will not be applied by this column initially.label - the column label to be displayed in the column selection input.icon - a specific class to be applied to the respective item in the column selection input.sortDirection - the sort direction to be applied when grouping. Possible values: 'ascending' and 'descending'.
   * Default value: null
   */
  dataSource?: {label: string, dataField: string, dataType: string, sortDirection: string, groupIndex: number}[];
  /**
   * Enables or disables the group panel.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages.
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets the maximum number of columns to group by. If set to null, there is no limit.
   * Default value: 8
   */
  maxLevel?: number;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid ''!",
   *     "propertyInvalidValueType": "Invalid ''!",
   *     "methodInvalidValueType": "Invalid ''!",
   *     "methodInvalidArgumentsCount": "Invalid '' argument(s)!",
   *     "methodInvalidReturnType": "Invalid ''!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": "'.",
   *     "htmlTemplateNotSuported": ": Web Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "apply": "Apply",
   *     "booleanFirst": "?",
   *     "booleanLast": "?",
   *     "cancel": "Cancel",
   *     "dateFirst": "1",
   *     "dateLast": "9",
   *     "from": "from",
   *     "numberFirst": "1",
   *     "numberLast": "9",
   *     "pickAnother": "Pick another field to group by",
   *     "firstBy": "Group by",
   *     "stringFirst": "A",
   *     "stringLast": "Z",
   *     "thenBy": "then by",
   *     "collapseAll": "Collapse all",
   *     "expandAll": "Expand all"
   *   }
   * }
   */
  messages?: any;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 Defines a group of grouped items in a panel.
*/
export interface GroupPanel extends BaseElement, GroupPanelProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the "Apply" button is clicked.
	* @param event. The custom event.    */
  onApply?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the "Cancel" button is clicked.
	* @param event. The custom event.    */
  onCancel: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the "Collapse all" button is clicked.
	* @param event. The custom event.    */
  onCollapseAll?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the "Expand all" button is clicked.
	* @param event. The custom event.    */
  onExpandAll?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-group-panel"): GroupPanel;
        querySelector(selectors: "smart-group-panel"): GroupPanel | null;
        querySelectorAll(selectors: "smart-group-panel"): NodeListOf<GroupPanel>;
        getElementsByTagName(qualifiedName: "smart-group-panel"): HTMLCollectionOf<GroupPanel>;
        getElementsByName(elementName: "smart-group-panel"): NodeListOf<GroupPanel>;
    }
}

/**Sets or gets the the position of the close button of group panel items. */
export declare type GroupPanelCloseButtonPosition = 'left' | 'right';
export interface InputProperties {
  /**
   * Determines the delay before the drop down opens to show the matches from the auto complete operation. The delay is measured in miliseconds.
   * Default value: 100
   */
  autoCompleteDelay?: number;
  /**
   * Determines the data source that will be loaded to the Input. The dataSource can be an array of strings/numbers or objects where the attributes represent the properties of a List Item. For example label, value. It can also be a callback that returns an Array of items as previously described. The data source item object may have the following fields: 'label' - string, 'value' - string or number, 'selected' - boolean, 'prefix' - string, 'suffix' - string, 'title' - string. The 'prefix' and 'suffix' add html before and after the label.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets additional class names to the Input drop down.
   * Default value: 
   */
  dropDownClassList?: any;
  /**
   * Determines the position of the drop down button.
   * Default value: none
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to an empty string. In this case the height of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Sets the purpose of the input and what, if any, permission the user agent has to provide automated assistance in filling out the element's input when in a form, as well as guidance to the browser as to the type of information expected in the element. This value corresponds to the standard HTML autocomplete attribute and can be set to values such as 'on', 'name', 'organization', 'street-address', etc.
   * Default value: "off"
   */
  inputPurpose?: string;
  /**
   * Determines the maximum number of matched items that should be visible inside the drop down as a result of a new autoComplete query. By default the maximum number of 8 items can be displayed inside the drop down.
   * Default value: 8
   */
  items?: number;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the minimum number of characters inside the input in order to trigger the autocomplete functionality that will open the drop down and show the matched items.
   * Default value: 1
   */
  minLength?: number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the drop down is opened or not.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Sets or gets the query that is used to filter the items. Query is used by the autoComplete operation. Empty string means that all items from the data source will be displayed and no filter query is applied.
   * Default value: 
   */
  query?: string | number;
  /**
   * Determines the auto complete query mode. This property also determines the matching algorithm for the autocomplete operation.
   * Default value: containsIgnoreCase
   */
  queryMode?: InputQueryMode | string;
  /**
   * Determines whether ot not the user can enter text inside the input. if dropDownButtonPosition is set to 'left' or 'right' then readonly determines whether the element acts as a ComboBox or a DropDownList if a dataSource is provided.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines whether the items are sorted alphabetically or not
   * Default value: false
   */
  sorted?: boolean;
  /**
   * Determines the sorting algorithm - ascending(asc) or descending(desc) if sort is enabled.
   * Default value: "asc"
   */
  sortDirection?: string;
  /**
   * Determines the selected index.
   * Default value: -1
   */
  selectedIndex?: number;
  /**
   * Determines the selected value.
   * Default value: 
   */
  selectedValue?: string | number;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines the input type. Input type determines what input can be entered.
   * Default value: ""
   */
  type?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element.
   * Default value: ""
   */
  value?: string;
}
/**
 Input specifies an input field where the user can enter data. Auto-complete options are displayed for easier input.
*/
export interface Input extends BaseElement, InputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, oldLabel, oldValue, value)
   *  label - The label of the new selected item.
   *  oldLabel - The label of the item that was previously selected before the event was triggered.
   *  oldValue - The value of the item that was previously selected before the event was triggered.
   *  value - The value of the new selected item.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered on each key up event of the Input, if the value is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value before it was changed.
   *  value - The new value.
   */
  onChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the popup is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the popup is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the user clicks on an item from the popup list.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value)
   *  item - The item that was clicked.
   *  label - The label of the item that was clicked.
   *  value - The value of the item that was clicked.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Closes the drop down.
   */
  close(): void;
  /**
   * Ensures that the active ( selected ) item is always visible.
   */
  ensureVisible(): void;
  /**
   * Opens the drop down.
   */
  open(): void;
  /**
   * Focuses and selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
  /**
   * Selects an item by value. For example, if your data source is ['Item 1', 'Item 2', 'Item 3'], you can use 'Item 1' as an argument. If your data source is an object with label and value, pass the value when you call selectItem.
   * @param {string | number} value. The item's value when the item is an object or string when the item is a string item.
   */
  selectItem(value: string | number): void;
  /**
   * Gets an item by value. For example, if your data source is ['Item 1', 'Item 2', 'Item 3'], you can use 'Item 1' as an argument. If your data source is an object with label and value, pass the value when you call selectItem.
   * @param {string | number} value. The item's value when the item is an object or string when the item is a string item.
   * @returns {any}
   */
  getItem(value: string | number): any;
  /**
   * Gets the selected item. For example, if your data source is ['Item 1', 'Item 2', 'Item 3'] and the user selected the second item, the method returns 'Item 2'. If your data source is an object with label and value, the returned value would be the 'value'.
   * @param {string | number} value. The item's value when the item is an object or string when the item is a string item.
   * @returns {any}
   */
  getSelectedItem(value: string | number): any;
}

declare global {
    interface Document {
        createElement(tagName: "smart-input"): Input;
        querySelector(selectors: "smart-input"): Input | null;
        querySelectorAll(selectors: "smart-input"): NodeListOf<Input>;
        getElementsByTagName(qualifiedName: "smart-input"): HTMLCollectionOf<Input>;
        getElementsByName(elementName: "smart-input"): NodeListOf<Input>;
    }
}

/**Determines the auto complete query mode. This property also determines the matching algorithm for the autocomplete operation. */
export declare type InputQueryMode = 'contains' | 'containsIgnoreCase' | 'doesNotContain' | 'doesNotContainIgnoreCase' | 'equals' | 'equalsIgnoreCase' | 'startsWith' | 'startsWithIgnoreCase' | 'endsWith' | 'endsWithIgnoreCase';
export interface KanbanProperties {
  /**
   * Enables or disables column reordering. When this property is set to true and allowDrag is enabled, users will be able to reoder columns through drag & drop. For example: Click and drag the first column's header and drop it over another column.
   * Default value: false
   */
  allowColumnReorder?: boolean;
  /**
   * Enables or disables column editing. When this property is set to true, users will be able to dynamically change the column's header label by double clicking on it.
   * Default value: false
   */
  allowColumnEdit?: boolean;
  /**
   * Enables or disables column removing. When this property is set to true, users will be able to dynamically remove a column through the column actions menu. the 'columnActions' property should be true.
   * Default value: false
   */
  allowColumnRemove?: boolean;
  /**
   * Enables or disables column hiding. When this property is set to true, users will be able to dynamically hide a column through the column actions menu. the 'columnActions' property should be true.
   * Default value: false
   */
  allowColumnHide?: boolean;
  /**
   * Toggles the visibility of the column buttons for adding tasks. A particular button can be disabled by setting addNewButton in the column's definition to false.
   * Default value: false
   */
  addNewButton?: boolean;
  /**
   * Determines whether the add button is visible in the column header and/or after the tasks in the column.
   * Default value: both
   */
  addNewButtonDisplayMode?: KanbanAddNewButtonDisplayMode | string;
  /**
   * Sets or gets whether a column with a button for adding new status columns to the Kanban will be displayed.
   * Default value: false
   */
  addNewColumn?: boolean;
  /**
   * Sets the width of the add new column. The property is used, if the 'columnWidth' property is set, too. It specifies the width of the dynamic new column.
   * Default value: null
   */
  addNewColumnWidth?: number | null;
  /**
   * Allows the dragging of tasks.
   * Default value: true
   */
  allowDrag?: boolean;
  /**
   * Allows the dropping of tasks.
   * Default value: true
   */
  allowDrop?: boolean;
  /**
   * This property changes the visual appeal of the Kanban columns and tasks. When set to true and the Kanban columns have their 'color' property set, the color is also applied to the tasks and edit dialog.
   * Default value: false
   */
  applyColumnColorToTasks?: boolean;
  /**
   * Enables or disables auto load state from the browser's localStorage. Information about tasks and their position and selected state, filtering, sorting, collapsed columns, as well as the values of the properties taskActions, taskComments, taskDue, taskPriority, taskProgress, taskTags, and taskUserIcon is loaded.
   * Default value: true
   */
  autoLoadState?: boolean;
  /**
   * Enables or disables auto save state to the browser's localStorage. Information about tasks and their position and selected state, filtering, sorting, collapsed columns, as well as the values of the properties taskActions, taskComments, taskDue, taskPriority, taskProgress, taskTags, and taskUserIcon is saved.
   * Default value: false
   */
  autoSaveState?: boolean;
  /**
   * Automatically updates the columns height depending on the tasks inside the column. The effect of this property is observed when 'columnColorEntireSurface' is true.
   * Default value: false
   */
  autoColumnHeight?: boolean;
  /**
   * Allows collapsing the card content.
   * Default value: false
   */
  collapsible?: boolean;
  /**
   * Displays colors in the column header, when the column's color property is set.
   * Default value: false
   */
  columnColors?: boolean;
  /**
   * Sets the Kanban columns width. When this property is set, the kanban columns width is set and a horizontal scrollbar may appear.
   * Default value: null
   */
  columnWidth?: number | null;
  /**
   * Displays background in the Kanban column.
   * Default value: false
   */
  columnColorEntireSurface?: boolean;
  /**
   * Displays a column footer which shows the summary of the column.
   * Default value: false
   */
  columnFooter?: boolean;
  /**
   * Describes the columns properties.
   * Default value: 
   */
  columns?: KanbanColumn[];
  /**
   * Toggles the visibility of the column actions icon.
   * Default value: false
   */
  columnActions?: boolean;
  /**
   * Determines whether task count information is displayed in column headers.
   * Default value: false
   */
  columnSummary?: boolean;
  /**
   * Determines whether a column header has a template. You can pass 'string', 'function' or HTMLTemplateElement as a value.
   * Default value: null
   */
  columnHeaderTemplate?: any;
  /**
   * Determines the column edit behavior. With the 'header' option, edit starts on double click on the column's label. In 'menu' mode, edit is allowed from the 'columnActions' menu. In 'headerAndMenu' option, column editing includes both options.
   * Default value: headerAndMenu
   */
  columnEditMode?: KanbanColumnEditMode | string;
  /**
   * Sets or gets the id of the current user. Has to correspond to the id of an item from the users property/array. Depending on the current user, different privileges are enabled. If no current user is set, privileges depend on the element's properties.
   * Default value: 
   */
  currentUser?: string | number;
  /**
   * Sets or gets whether the default dialog for adding/removing tasks or comments is disabled.
   * Default value: false
   */
  disableDialog?: boolean;
  /**
   * Sets or gets a customization function for the dialog. This function can be used to customize the dialog look or to replace it. The Kanban calls this function with 5 arguments - 'dialog', 'taskOrComment', 'editors', 'purpose' and 'type'. The dialog is the 'smart-window' instance used as a default Kanban dialog. 'taskOrComment' is object which could be Kanban task or comment. 'purpose' could be 'add' or 'edit' and 'type' could be 'task' or 'column' depending on the action.
   * Default value: null
   */
  dialogCustomizationFunction?: any;
  /**
   * Sets or gets a function called when the dialog is rendered. The Kanban calls this function with 6 arguments - 'dialog', 'editors', 'labels', 'tabs', 'layout', 'taskOrComment'. The dialog is the 'smart-window' instance used as a default Kanban dialog. 'taskOrComment' is object which could be Kanban task or comment. 'editors', 'labels', 'tabs' and 'layout' are JSON objects with key which describes the element type and value which is HTML Element.
   * Default value: null
   */
  dialogRendered?: any;
  /**
   * Determines the data source to be visualized in the kanban board.
   * Default value: null
   */
  dataSource?: KanbanDataSource[];
  /**
   * Determines the the relation (mapping) between the Kanban's default fields (keywords) and the data fields from the data source. Not necessary if both match. Only some of the default mapping can be overwritten.
   * Default value: { checklist: 'checklist', color: 'color', comments: 'comments', dueDate: 'dueDate', id: 'id', priority: 'priority', progress: 'progress', startDate: 'startDate', status: 'status', swimlane: 'swimlane', tags: 'tags', text: 'text', userId: 'userId'. createdUserId: 'createdUserId', createdDate: 'createdDate', updatedUserId: 'updatedUserId', updatedDate: 'updatedDate' }
   */
  dataSourceMap?: { checklist: string; color: string; comments: string; dueDate: string; id: string; priority: string; progress: string; startDate: string; status: string; swimlane: string; tags: string; text: string; userId: string; createdUserId: string; upDatedUserId: string; createdDate: Date; upDatedDate: Date;};
  /**
   * Determines the offset of the drag feedback element from the mouse cursor when dragging tasks. The first member of the array is the horizontal offset and the second one - the vertical offset. If set to 'auto', the offset is based on the mouse position when the dragging started.
   * Default value: auto
   */
  dragOffset?: boolean;
  /**
   * Sets or gets whether tasks can be edited (including the assigning of users to tasks).
   * Default value: false
   */
  editable?: boolean;
  /**
   * Sets or gets the format string of the "Due date" label and the "Start date" and "Due date" editors.
   * Default value: "d"
   */
  formatStringDate?: string;
  /**
   * Sets or gets the format string of comments time stamp.
   * Default value: "MMM d, HH:mm"
   */
  formatStringTime?: string;
  /**
   * Sets or gets the header position. The header contains the Customize, Filter, Sort, and Search buttons.
   * Default value: none
   */
  headerPosition?: KanbanHeaderPosition | string;
  /**
   * Sets or gets the way column hierarchy is represented.
   * Default value: columns
   */
  hierarchy?: KanbanHierarchy | string;
  /**
   * Sets or gets the locale. Used in conjunction with the property messages.
   * Default value: "en"
   */
  locale?: string;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * { 'en': { 'addFilter': '+ Add filter', 'and': 'And', 'apply': 'Apply', 'booleanFirst': '☐', 'booleanLast': '☑', 'cancel': 'Cancel', 'CONTAINS': 'contains', 'CONTAINS_CASE_SENSITIVE': 'contains (case sensitive)', 'dateFirst': '1', 'dateLast': '9', 'DOES_NOT_CONTAIN': 'does not contain', 'DOES_NOT_CONTAIN_CASE_SENSITIVE': 'does not contain (case sensitive)', 'EMPTY': 'empty', 'ENDS_WITH': 'ends with', 'ENDS_WITH_CASE_SENSITIVE': 'ends with (case sensitive)', 'EQUAL': 'equal', 'EQUAL_CASE_SENSITIVE': 'equal (case sensitive)', 'filter': 'Filter', 'filteredByMultiple': '%', 'removeComment': 'Remove comment', 'promptColumn': 'Are you sure you want to remove this column?'} }
   */
  messages?: any;
  /**
   * Callback function which can be used for customizing the tasks rendering. The Kanban calls it with 2 arguments - task html element and task data.
   * Default value: null
   */
  onTaskRender?: any;
  /**
   * Callback function which can be used for customizing the filter items. The function is called with 1 argument - Array of items which will be displayed in the filter drop down. You can modify that array to remove or update items to filter by.
   * Default value: null
   */
  onFilterPrepare?: any;
  /**
   * Callback function which can be used for customizing the sort items. The function is called with 1 argument - Array of items which will be displayed in the sort drop down. You can modify that array to remove or update items to sort by.
   * Default value: null
   */
  onSortPrepare?: any;
  /**
   * Callback function which can be used for customizing the column header rendering. The Kanban calls it with 3 arguments - column header html element and column data and column data field.
   * Default value: null
   */
  onColumnHeaderRender?: any;
  /**
   * Callback function which can be used for customizing the column footer rendering. The Kanban calls it with 3 arguments - column header html element and column data and column data field.
   * Default value: null
   */
  onColumnFooterRender?: any;
  /**
   * Determines selection mode.
   * Default value: zeroOrOne
   */
  selectionMode?: KanbanSelectionMode | string;
  /**
   * Sets or gets whether the tasks history will be stored and displayed in the task dialog.
   * Default value: false
   */
  storeHistory?: boolean;
  /**
   * Sets or gets the task history items that will be stored when storeHistory is enabled.
   * Default value: 20
   */
  storeHistoryItems?: number;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Sets or gets whether the edit dialog is displayed in readonly mode. In that mode it shows only the task details, but the editing is disabled. However, if comments are enabled, you will be able to add comments in the dialog.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Describes the swimlanes in the kanban board. Sub-columns are not applicable when swimlanes are present.
   * Default value: 
   */
  swimlanes?: KanbanSwimlane[];
  /**
   * Sets or gets the index of the column at which to start the swimlanes.
   * Default value: 0
   */
  swimlanesFrom?: number;
  /**
   * Sets or gets the index of the column at which to end the swimlanes. By default, swimlanes end at the last column.
   * Default value: null
   */
  swimlanesTo?: number;
  /**
   * Sets or gets the allowed tags. If no tags are listed, all tags from the data source are allowed.
   * Default value: 
   */
  tags?: any[];
  /**
   * Toggles the visibility of the task actions icon.
   * Default value: false
   */
  taskActions?: boolean;
  /**
   * Represents a callback function which is called when the task actions menu is created. The task actions element is passed as parameter and allows you to customize the menu. Example: (list) => { list.innerHTML = 'Custom Item'; list.onclick = () => { alert('clicked'); }}
   * Default value: null
   */
  taskActionsRendered?: any;
  /**
   * Toggles the visibility of the task comments icon.
   * Default value: false
   */
  taskComments?: boolean;
  /**
   * Toggles the visibility of the task due icon.
   * Default value: false
   */
  taskDue?: boolean;
  /**
   * Sets or gets whether tasks can be shown in all levels of column hierarchy or only on leaf columns.
   * Default value: false
   */
  taskPosition?: KanbanTaskPosition | string;
  /**
   * Toggles the visibility of the task priority icon (shown when priority is low or high).
   * Default value: true
   */
  taskPriority?: boolean;
  /**
   * Toggles the visibility of task progress bar and the completed sub-tasks label.
   * Default value: false
   */
  taskProgress?: boolean;
  /**
   * Sets the task custom fields displayed in the card. Each array item should have 'dataField', 'label' 'dataType' and optionally 'visible', 'image' and 'cover' properties. The 'dataField' determines the value, the label is displayed as title, 'dataType' is used for formatting and 'visible' determines whether the field will be displayed. If your string represents an image either URL or Base64, set image: true. If you want to display that image as a cover image, set cover:true, too.
   * Default value: 
   */
  taskCustomFields?: any;
  /**
   * The task's background color depends on the task's color property. By default the color is rendered within the task's left border.
   * Default value: false
   */
  taskColorEntireSurface?: boolean;
  /**
   * Displays an input in the task's card for adding dynamically a sub task. The 'taskSubTasks' property should be set to a value different than 'none'.
   * Default value: true
   */
  taskSubTasksInput?: boolean;
  /**
   * Sets the rendering mode of sub tasks. 'none' - default value. Sub tasks are displayed only in the edit dialog. 'onePerRow' - all sub tasks are displayed in the task's card. 'onlyUnfinished' - only tasks which are not completed are displayed in the task's card.
   * Default value: none
   */
  taskSubTasks?: KanbanTaskSubTasks | string;
  /**
   * Toggles the visibility of task tags.
   * Default value: true
   */
  taskTags?: boolean;
  /**
   * Toggles the visibility of the task user icon.
   * Default value: true
   */
  taskUserIcon?: boolean;
  /**
   * Sets or gets a template to be applied to task text. Can be a string beginning with # and referencing the id of a template element on the page. Can also be a function that modifies the task text or the template itself. Finally, it can also be a string that will be parsed.
   * Default value: null
   */
  textTemplate?: any;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines whether the priority list (as defined by the priority property) will be shown when clicking the priority icon. Only applicable if editable privileges are enabled.
   * Default value: false
   */
  priorityList?: boolean;
  /**
   * Determines the priority Kanban tasks can be assigned to. Example: [{label: 'low', value: 'low'}, {label: 'high', value: 'high'}]
   * Default value: 
   */
  priority?: KanbanPriority[];
  /**
   * Determines whether the user list (as defined by the users property) will be shown when clicking the user icon. Only applicable if editable privileges are enabled.
   * Default value: false
   */
  userList?: boolean;
  /**
   * Determines the users Kanban tasks can be assigned to and their characteristics and privileges.
   * Default value: 
   */
  users?: KanbanUser[];
}
/**
 Kanban represents a kanban board that visually depicts work at various stages of a process using cards to represent tasks and columns to represent each stage of the process.
*/
export interface Kanban extends BaseElement, KanbanProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when a task has been updated.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, task, value)
   *  oldValue - The old data of the task
   *  task - The HTML element of the task whose data has been changed
   *  value - The new data of the task
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the edit/prompt dialog is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the edit/prompt dialog is about to be closed. The closing operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column is added.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, dataField, collapsed)
   *  label - The column label.
   *  dataField - The column data field.
   *  collapsed - The column's collapsed state.
   */
  onColumnAdd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column is removed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, dataField, collapsed)
   *  label - The column label.
   *  dataField - The column data field.
   *  collapsed - The column's collapsed state.
   */
  onColumnRemove?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column is reordered.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldIndex, index, column)
   *  oldIndex - The column's old index.
   *  index - The column's new index.
   *  column - The column's data object with 'label', 'dataField' and 'collapsed' fields.
   */
  onColumnReorder?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column is updated.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value, column)
   *  oldValue - The column's old label.
   *  value - The column's new label.
   *  column - The column's data object with 'label', 'dataField' and 'collapsed' fields.
   */
  onColumnUpdate?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column header is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, dataField, collapsed)
   *  label - The column label.
   *  dataField - The column data field.
   *  collapsed - The column's collapsed state.
   */
  onColumnClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column header is double clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, dataField, collapsed)
   *  label - The column label.
   *  dataField - The column data field.
   *  collapsed - The column's collapsed state.
   */
  onColumnDoubleClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column is shown by using the column's action menu or the Kanban's 'show' method.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, dataField)
   *  label - The column label.
   *  dataField - The column data field.
   */
  onColumnShow?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column is hidden by using the column's action menu or the Kanban's 'hide' method.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, dataField)
   *  label - The column label.
   *  dataField - The column data field.
   */
  onColumnHide?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column is collapsed  by using the column's action menu or the Kanban's 'collapse' method.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, dataField)
   *  label - The column label.
   *  dataField - The column data field.
   */
  onColumnCollapse?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column is expanded by using the column's action menu or the Kanban's 'expand' method.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, dataField)
   *  label - The column label.
   *  dataField - The column data field.
   */
  onColumnExpand?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a comment is added to the Kanban Task.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, value)
   *  id - The task's id.
   *  value - The comment object. It has 'text: string, time: Date and userId:number' properties.
   */
  onCommentAdd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a comment is removed from the Kanban Task.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, value)
   *  id - The task's id.
   *  value - The comment object. It has 'text: string, time: Date and userId:number' properties.
   */
  onCommentRemove?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a comment is updated in the Kanban Task.
	* @param event. The custom event. Custom data event was created with: ev.detail(id, value)
   *  id - The task's id.
   *  value - The comment object. It has 'text: string, time: Date and userId:number' properties.
   */
  onCommentUpdate?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a task is dropped somewhere in the DOM. The dragging operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(container, data, item, items, originalEvent, previousContainer, target)
   *  container - the Kanban the dragged task(s) is dropped to
   *  data - an object with additional drag details
   *  item - the task that is dragged; if multiple tasks are dragged, this is the task that has been clicked when initiating the drag operation
   *  items - an array with all dragged tasks
   *  originalEvent - the original, browser, event that initiates the drag operation
   *  previousContainer - the Kanban the dragged item(s) is dragged from
   *  target - the element the dragged tasks are dropped to
   */
  onDragEnd: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the user is dragging a task.
	* @param event. The custom event. Custom data event was created with: ev.detail(data, item, items, originalEvent)
   *  data - an object with additional drag details
   *  item - the task that is dragged; if multiple tasks are dragged, this is the task that has been clicked when initiating the drag operation
   *  items - an array with all dragged tasks
   *  originalEvent - the original, browser, event that initiates the drag operation
   */
  onDragging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user starts dragging task(s). The dragging operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(container, data, item, items, originalEvent, previousContainer)
   *  container - the Kanban the dragged task(s) is dragged from
   *  data - an object with additional drag details
   *  item - the task that is dragged; if multiple tasks are dragged, this is the task that has been clicked when initiating the drag operation
   *  items - an array with all dragged tasks
   *  originalEvent - the original, browser, event that initiates the drag operation
   *  previousContainer - the Kanban the dragged item(s) is dragged from
   */
  onDragStart: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a filter has been applied.
	* @param event. The custom event.    */
  onFilter?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the edit/prompt dialog is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the edit/prompt dialog is about to be opened. The opening operation can be canceled by calling event.preventDefault() in the event handler function. If you want to cancel the default Kanban dialog, call event.preventDefault();
	* @param event. The custom event. Custom data event was created with: ev.detail(comment, purpose, task)
   *  comment - The comment that is about to be removed (if applicable).
   *  purpose - The purpose of the dialog to be opened - <em>'edit'</em> or <em>'prompt'</em>.
   *  task - The task that is about to be edited or removed (if applicable).
   */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when sorting has been applied.
	* @param event. The custom event.    */
  onSort?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered before a new task is added. You can use the event.detail.value and event.detail.id to customize the new Task before adding it to the Kanban. Example: kanban.onTaskBeforeAdd = (event) => { const data = event.detail.value; const id = event.detail.id; event.detail.id = 'BG12';}
	* @param event. The custom event. Custom data event was created with: ev.detail(value, id)
   *  value - The task data that is added to the Kanban.
   *  id - The task data id.
   */
  onTaskBeforeAdd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a new task is added. Example: kanban.onTaskAdd = (event) => { const data = event.detail.value; const id = event.detail.id; }
	* @param event. The custom event. Custom data event was created with: ev.detail(value, id)
   *  value - The task data that is added to the Kanban.
   *  id - The task data id.
   */
  onTaskAdd?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a task is removed.
	* @param event. The custom event. Custom data event was created with: ev.detail(value, id)
   *  value - The task data that is removed from the Kanban.
   *  id - The task data id.
   */
  onTaskRemove?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a task is updated.
	* @param event. The custom event. Custom data event was created with: ev.detail(value, oldValue, id)
   *  value - The task data that is updated.
   *  oldValue - The update task's old data.
   *  id - The task data id.
   */
  onTaskUpdate?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a task is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(value, id)
   *  value - The task data.
   *  id - The task data id.
   */
  onTaskClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a task is double clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(value, id)
   *  value - The task data.
   *  id - The task data id.
   */
  onTaskDoubleClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Adds filtering. Example: const filterGroup = new Smart.FilterGroup(); const filterObject = filterGroup.createFilter('string', 'Italy', 'contains'); filterGroup.addFilter('and', filterObject); kanban.addFilter([['Country', filterGroup]]);
   * @param {any} filters. Filter information. Example: kanban.addFilter([['Country', filterGroup]]);. Each array item is a sub array with two items - 'dataField' and 'filterGroup' object. The 'dataField' is any valid data field from the data source bound to the Kanban like 'dueDate', 'startDate' or custom fields like 'Country'. Filter conditions which you can use in the expressions: '=', 'EQUAL','&lt;&gt;', 'NOT_EQUAL', '!=', '&lt;', 'LESS_THAN','&gt;', 'GREATER_THAN', '&lt;=', 'LESS_THAN_OR_EQUAL', '&gt;=', 'GREATER_THAN_OR_EQUAL','starts with', 'STARTS_WITH','ends with', 'ENDS_WITH', '', 'EMPTY', 'CONTAINS','DOES_NOT_CONTAIN', 'NULL','NOT_NULL'
   * @param {string} operator?. Logical operator between the filters of different fields. Possible values are: 'and', 'or'. 
   */
  addFilter(filters: any, operator?: string): void;
  /**
   * Adds sorting. Example: kanban.addSort(['Country'], 'ascending');
   * @param {[] | string} dataFields. The data field(s) to sort by
   * @param {[] | string} orderBy. The sort direction(s) to sort the data field(s) by. Possible values are: 'ascending' and 'descending'.
   */
  addSort(dataFields: [] | string, orderBy: [] | string): void;
  /**
   * Adds a task to a Kanban column. If no data is specified, an empty task is added to the first column.
   * @param {any} data?. An object containing the new task's data
   */
  addTask(data?: any): void;
  /**
   * Adds a column to a Kanban. If no data is specified, an empty column is added.
   * @param {any} data?. An object containing the new column's data
   */
  addColumn(data?: any): void;
  /**
   * Begins an edit operation
   * @param {number | string | HTMLElement} task. The task's id or corresponding HTMLElement
   */
  beginEdit(task: number | string | HTMLElement): void;
  /**
   * Ends the current edit operation and discards changes
   */
  cancelEdit(): void;
  /**
   * Closes any open header panel (drop down)
   */
  closePanel(): void;
  /**
   * Collapses a Kanban column.
   * @param {number | string} column. The index or dataField of the column to collapse
   */
  collapse(column: number | string): void;
  /**
   * Clears the Kanban's selection.
   */
  clearSelection(): void;
  /**
   * Hides a Kanban column.
   * @param {number | string} column. The index or dataField of the column to hide
   */
  hide(column: number | string): void;
  /**
   * Creates a copy of a task in the same column.
   * @param {number | string | HTMLElement} task. The task's id or corresponding HTMLElement
   */
  copyTask(task: number | string | HTMLElement): void;
  /**
   * Ends the current edit operation and saves changes
   */
  endEdit(): void;
  /**
   * Makes sure a task is visible by scrolling to it. If succcessful, the method returns the HTML element of the task.
   * @param {number | string | HTMLElement} task. The task's id or corresponding HTMLElement
   * @returns {HTMLElement}
   */
  ensureVisible(task: number | string | HTMLElement): HTMLElement;
  /**
   * Expands a Kanban column.
   * @param {number | string} column. The index or dataField of the column to expand
   */
  expand(column: number | string): void;
  /**
   * Expands all Kanban columns.
   */
  expandAll(): void;
  /**
   * Exports the Kanban's data.
   * @param {string} dataFormat. The file format to export to. Supported formats: 'csv', 'html', 'json', 'pdf', 'tsv', 'xlsx', 'xml'.
   * @param {string} fileName. The name of the file to export to
   * @param {Function} callback?. A callback function to pass the exported data to (if fileName is not provided)
   * @returns {any}
   */
  exportData(dataFormat: string, fileName: string, callback?: Function): any;
  /**
   * Gets the data of a column. The returned value is a JSON object with the following fields: 'label', 'dataField', 'collapsed', 'collapsible', 'allowRemove', 'editable', 'reorder', 'orientation'.
   * @param {string} dataField. The column's data field
   * @returns {any}
   */
  getColumn(dataField: string): any;
  /**
   * Gets the data of a task. The returned value is a JSON object with the following fields: 'checklist', 'id', 'color', 'comments', 'history', 'dueDate', 'startDate', 'priority', 'progress', 'status', 'swimlane', 'tags', 'text', 'description', 'userId', 'createdUserId', 'createdDate', 'updatedUserId', 'updatedDate'
   * @param {number} id. The task's id
   * @returns {any}
   */
  getTask(id: number): any;
  /**
   * Gets the selected ids. The returned value is an array. Each item in the array is the 'id' of a selected task.
   * @param {number} id. The task's id
   * @returns {any}
   */
  getSelectedTasks(id: number): any;
  /**
   * Gets the Kanban's state.
   * @returns 
   */
  getState(): { collapsed: {}, dataSource: [], filtering: { filters: [], operator: string }, selection: { selected: [], selectionStart: number | string, selectionInColumn: string, swimlane: string }, sorting: { dataFields: [], dataTypes: [], orderBy: [] }, tabs: [], visibility: { taskActions: boolean, taskComments: boolean, taskDue: boolean, taskPriority: boolean, taskProgress: boolean, taskTags: boolean, taskUserIcon: boolean } };
  /**
   * Loads the Kanban's state.
   * @param  state?. An object returned by one of the methods getState or saveState. If not passed, gets saved state from the browser's localStorage.
   */
  loadState(state?: { collapsed: {}, dataSource: [], filtering: { filters: [], operator: string }, selection: { selected: [], selectionStart?: number | string, selectionInColumn: string, swimlane: string }, sorting: { dataFields: [], dataTypes: [], orderBy: [] }, tabs: [], visibility: { taskActions: boolean, taskComments: boolean, taskDue: boolean, taskPriority: boolean, taskProgress: boolean, taskTags: boolean, taskUserIcon: boolean } }): void;
  /**
   * Moves a task to a different column.
   * @param {number | string | HTMLElement} task. The task's id or corresponding HTMLElement
   * @param {string} newStatus. The new status of the task (its new column's dataField)
   */
  moveTask(task: number | string | HTMLElement, newStatus: string): void;
  /**
   * Opens the "Customize tasks" header panel (drop down)
   */
  openCustomizePanel(): void;
  /**
   * Opens the "Filter" header panel (drop down)
   */
  openFilterPanel(): void;
  /**
   * Opens the "Sort" header panel (drop down)
   */
  openSortPanel(): void;
  /**
   * Removes filtering
   */
  removeFilter(): void;
  /**
   * Removes sorting
   */
  removeSort(): void;
  /**
   * Removes a task.
   * @param {number | string | HTMLElement} task. The task's id or corresponding HTMLElement
   * @param {boolean} prompt?. Whether or not to prompt the user before removing the task
   */
  removeTask(task: number | string | HTMLElement, prompt?: boolean): void;
  /**
   * Removes a column.
   * @param {string} dataField. The column's data field
   */
  removeColumn(dataField: string): void;
  /**
   * Saves the Kanban's state to the browser's localStorage.
   */
  saveState(): void;
  /**
   * Shows a Kanban column.
   * @param {number | string} column. The index or dataField of the column to show
   */
  show(column: number | string): void;
  /**
   * Shows all Kanban columns.
   */
  showAllColumns(): void;
  /**
   * Selects a task.
   * @param {number | string} task. The task's id.
   */
  selectTask(task: number | string): void;
  /**
   * Unselects a task.
   * @param {number | string} task. The task's id.
   */
  unselectTask(task: number | string): void;
  /**
   * Updates a task.
   * @param {number | string | HTMLElement} task. The task's id or corresponding HTMLElement
   * @param {{}} newData. The new data to visualize in the task.
   */
  updateTask(task: number | string | HTMLElement, newData: {}): void;
  /**
   * Updates a column.
   * @param {string} dataField. The new column's data field
   * @param {{}} newData. The new data to visualize in the column.
   */
  updateColumn(dataField: string, newData: {}): void;
}

export interface KanbanColumn {
  /**
   * Sets or gets whether the column's button for adding tasks is visible. Works in conjunction with the Kanban property of the same name.
   * Default value: true
   */
  addNewButton?: boolean;
  /**
   * Sets or gets whether the column can be removed from the column menu.
   * Default value: true
   */
  allowRemove?: boolean;
  /**
   * Sets or gets whether the column is collapsed.
   * Default value: false
   */
  collapsed?: boolean;
  /**
   * Sets or gets whether the column is collapsible. Works in conjunction with the Kanban property of the same name.
   * Default value: true
   */
  collapsible?: boolean;
  /**
   * Sets or gets the column's sub-columns. Sub-columns has the same properties as top-level columns.
   * Default value: null
   */
  columns?: { addNewButton: boolean, collapsed: string, collapsible: string, columns: [], dataField: string, label: string, orientation: string, selected: string }[];
  /**
   * Sets or gets the column's data source bound field. Corresponds to the status field in the data source.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Sets or gets the text displayed in the column's header.
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets whether a column is editable.
   * Default value: true
   */
  editable?: boolean;
  /**
   * Sets or gets whether a column is visible.
   * Default value: true
   */
  visible?: boolean;
  /**
   * Sets or gets whether a column can be reordered.
   * Default value: true
   */
  reorder?: boolean;
  /**
   * Sets or gets whether the tasks in the column flow vertically or horizontally.
   * Default value: vertical
   */
  orientation?: KanbanColumnOrientation | string;
  /**
   * Sets or gets whether the column is selected. Only applicable to sub-columns when hierarchy is 'tabs'.
   * Default value: false
   */
  selected?: boolean;
  /**
   * Determines whether a column header has a template. You can pass 'string', 'function' or HTMLTemplateElement as a value.
   * Default value: null
   */
  headerTemplate?: any;
  /**
   * Sets the Kanban column width. When this property is set, the kanban column width is set and a horizontal scrollbar may appear.
   * Default value: null
   */
  width?: number | null;
}

export interface KanbanDataSource {
  /**
   * The task's unique ID.
   * Default value: 
   */
  id?: string | number;
  /**
   * A list of sub-tasks.
   * Default value: null
   */
  checklist?: { completed: boolean, text: string }[];
  /**
   * A color used to highlight the task's card visually.
   * Default value: "null"
   */
  color?: string;
  /**
   * Comments about a task.
   * Default value: null
   */
  comments?: { text: string, time: Date, userId: string | number }[];
  /**
   * The task's due date.
   * Default value: null
   */
  dueDate?: Date;
  /**
   * The task's priority.
   * Default value: "normal"
   */
  priority?: string;
  /**
   * The task's progress in percentages - a number from 0 to 100.
   * Default value: null
   */
  progress?: number;
  /**
   * The task's start date.
   * Default value: null
   */
  startDate?: Date;
  /**
   * The task's status. Has to correspond to the dataField of a column or sub-column from the columns property/array.
   * Default value: ""
   */
  status?: string;
  /**
   * The swimlane of the task. Has to correspond to the dataField of a swimlane from the swimlanes property/array, if it is defined.
   * Default value: "null"
   */
  swimlane?: string;
  /**
   * A comma-separated list of tags.
   * Default value: ""
   */
  tags?: string;
  /**
   * The text of the task.
   * Default value: ""
   */
  text?: string;
  /**
   * The ID of the user assigned to the task. Has to correspond to the id of an item from the users property/array.
   * Default value: null
   */
  userId?: string | number;
}

export interface KanbanSwimlane {
  /**
   * The swimlane's color.
   * Default value: "null"
   */
  color?: string;
  /**
   * Sets or gets the swimlane's data source bound field. Corresponds to the swimlane field in the data source.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Sets or gets the text displayed in the swimlane.
   * Default value: ""
   */
  label?: string;
}

export interface KanbanPriority {
  /**
   * The priority label displayed.
   * Default value: ""
   */
  label?: string;
  /**
   * The priority value.
   * Default value: ""
   */
  value?: string;
}

export interface KanbanUser {
  /**
   * Sets whether the user has a privilege to add or copy tasks.
   * Default value: true
   */
  allowAdd?: boolean;
  /**
   * Sets whether the user has a privilege to add comments to tasks.
   * Default value: true
   */
  allowComment?: boolean;
  /**
   * Sets whether the user has a privilege to drag tasks. Works along with the property allowDrag.
   * Default value: true
   */
  allowDrag?: boolean;
  /**
   * Sets whether the user has a privilege to edit tasks. Works along with the property editable.
   * Default value: true
   */
  allowEdit?: boolean;
  /**
   * Sets whether the user has a privilege to remove tasks.
   * Default value: true
   */
  allowRemove?: boolean;
  /**
   * The user's unique ID. Corresponds to the userId field in the data source and the property currentUser.
   * Default value: null
   */
  id?: string | number;
  /**
   * A url to an image representing the user.
   * Default value: ""
   */
  image?: string;
  /**
   * The user's name as it will appear in the Users list and "Assigned to" editor.
   * Default value: ""
   */
  name?: string;
}

declare global {
    interface Document {
        createElement(tagName: "smart-kanban"): Kanban;
        querySelector(selectors: "smart-kanban"): Kanban | null;
        querySelectorAll(selectors: "smart-kanban"): NodeListOf<Kanban>;
        getElementsByTagName(qualifiedName: "smart-kanban"): HTMLCollectionOf<Kanban>;
        getElementsByName(elementName: "smart-kanban"): NodeListOf<Kanban>;
    }
}

/**Determines whether the add button is visible in the column header and/or after the tasks in the column. */
export declare type KanbanAddNewButtonDisplayMode = 'top' | 'bottom' | 'both';
/**Sets or gets whether the tasks in the column flow vertically or horizontally. */
export declare type KanbanColumnOrientation = 'vertical' | 'horizontal';
/**Determines the column edit behavior. With the 'header' option, edit starts on double click on the column's label. In 'menu' mode, edit is allowed from the 'columnActions' menu. In 'headerAndMenu' option, column editing includes both options. */
export declare type KanbanColumnEditMode = 'header' | 'menu' | 'headerAndMenu';
/**Sets or gets the header position. The header contains the Customize, Filter, Sort, and Search buttons. */
export declare type KanbanHeaderPosition = 'none' | 'top' | 'bottom';
/**Sets or gets the way column hierarchy is represented. */
export declare type KanbanHierarchy = 'columns' | 'tabs';
/**Determines selection mode. */
export declare type KanbanSelectionMode = 'zeroOrOne' | 'zeroOrManyExtended';
/**Sets or gets whether tasks can be shown in all levels of column hierarchy or only on leaf columns. */
export declare type KanbanTaskPosition = 'all' | 'leaf';
/**Sets the rendering mode of sub tasks. 'none' - default value. Sub tasks are displayed only in the edit dialog. 'onePerRow' - all sub tasks are displayed in the task's card. 'onlyUnfinished' - only tasks which are not completed are displayed in the task's card. */
export declare type KanbanTaskSubTasks = 'none' | 'onePerRow' | 'onlyUnfinished';
export interface LayoutProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the options that will be available for selection inside the context menu.
   * Default value: delete
   */
  contextMenuDataSource?: any;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets Layout's data source.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Sets an object with string values, related to the different states of passwords strength.
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "invalidIndex": "' method accepts an index of type number.",
   *     "invalidNode": "' method accepts an object or an array of objects as it's second parameter.",
   *     "invalidSettings": "' method accepts a string or an object as it's second parameter.",
   *     "delete": "Delete",
   *     "noId": ": requires an id in order to save/load a state."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets Layout's main orientation. The orientation is applied to all Splitters inside the Layout unless they have their orientation explicitly set in the dataSource.
   * Default value: vertical
   */
  orientation?: Orientation | string;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Determines whether splitting is live or not.
   * Default value: false
   */
  allowLiveSplit?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the selected item.
   * Default value: null
   */
  selectedIndex?: any;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 Layout splits your content into resizable sections.
*/
export interface Layout extends BaseElement, LayoutProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered after resizing is completed.
	* @param event. The custom event.    */
  onResize: ((ev: Event) => any) | null;
  /**
   * This event is triggered when a change regarding the Layout's state has occured, such as inserting a new item, removing an item, etc.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, type)
   *  item - The Splitter item that was the target of a change.
   *  type - A description of the operation that has cause the change.
   */
  onStateChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldSelectedIndex, selectedIndex)
   *  oldSelectedIndex - The Splitter item that was previously selected.
   *  selectedIndex - The Splitter item that is currently selected.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a the context menu is about to be closed. The operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a the context menu is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a the context menu is about to be opened. The operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a the context menu is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an option from the context menu has been clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(target, item, label, value)
   *  target - The Splitter item that was the target of the context menu opening.
   *  item - The Context menu item that was clicked.
   *  label - The label of the context menu that was clicked.
   *  value - The value of the context menu that was clicked.
   */
  onMenuItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Returns a Layout item according to the index that is passed.
   * @param {number | string} index. The index of an item.
   */
  getItem(index: number | string): void;
  /**
   * Refreshes the Layout
   */
  refresh(): void;
  /**
   * Inserts a new item inside the Layout.
   * @param {any} type. The index of an item to be removed or an instance of Smart.SplitterItem.
   * @param {string | undefined} position?. A string that represents the position where the new item will be created.
   */
  createLayoutItem(type: any, position?: string | undefined): void;
  /**
   * Moves all children from one item to another.
   * @param {any} oldItem. The source item that will have it's content removed.
   * @param {any} newItem. The host item that will have it's content replaced.
   */
  moveChildren(oldItem: any, newItem: any): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-layout"): Layout;
        querySelector(selectors: "smart-layout"): Layout | null;
        querySelectorAll(selectors: "smart-layout"): NodeListOf<Layout>;
        getElementsByTagName(qualifiedName: "smart-layout"): HTMLCollectionOf<Layout>;
        getElementsByName(elementName: "smart-layout"): NodeListOf<Layout>;
    }
}

export interface LayoutGroupProperties {
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the modifiers of the Layout item.
   * Default value: resize,drag,close
   */
  modifiers?: any;
  /**
   * Determines the min size of the item.
   * Default value: 50
   */
  min?: number;
  /**
   * Determines the label of the item.
   * Default value: "Group"
   */
  label?: string;
  /**
   * Determines the group orientation.
   * Default value: vertical
   */
  orientation?: LayoutGroupOrientation | string;
  /**
   * Determines the size of the item.
   * Default value: null
   */
  size?: any;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 LayoutGroup represents a group of layout items inside the Layout.
*/
export interface LayoutGroup extends BaseElement, LayoutGroupProperties {

  /* Get a member by its name */
  [name: string]: any;
}

declare global {
    interface Document {
        createElement(tagName: "smart-layout-group"): LayoutGroup;
        querySelector(selectors: "smart-layout-group"): LayoutGroup | null;
        querySelectorAll(selectors: "smart-layout-group"): NodeListOf<LayoutGroup>;
        getElementsByTagName(qualifiedName: "smart-layout-group"): HTMLCollectionOf<LayoutGroup>;
        getElementsByName(elementName: "smart-layout-group"): NodeListOf<LayoutGroup>;
    }
}

/**Determines the group orientation. */
export declare type LayoutGroupOrientation = 'horizontal' | 'vertical';
export interface LayoutItemProperties {
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the modifiers of the Layout item.
   * Default value: resize,drag,close
   */
  modifiers?: any;
  /**
   * Determines the min size of the item.
   * Default value: 50
   */
  min?: number;
  /**
   * Determines the label of the item.
   * Default value: "Item"
   */
  label?: string;
  /**
   * Determines the size of the item.
   * Default value: null
   */
  size?: any;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
}
/**
 Layout item represents a single content section inside the Layout.
*/
export interface LayoutItem extends BaseElement, LayoutItemProperties {

  /* Get a member by its name */
  [name: string]: any;
}

declare global {
    interface Document {
        createElement(tagName: "smart-layout-item"): LayoutItem;
        querySelector(selectors: "smart-layout-item"): LayoutItem | null;
        querySelectorAll(selectors: "smart-layout-item"): NodeListOf<LayoutItem>;
        getElementsByTagName(qualifiedName: "smart-layout-item"): HTMLCollectionOf<LayoutItem>;
        getElementsByName(elementName: "smart-layout-item"): NodeListOf<LayoutItem>;
    }
}

export interface LedProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Sets or gets the check state.
   * Default value: false
   */
  checked?: boolean;
  /**
   * Determines when the element fires a click event.
   * Default value: release
   */
  clickMode?: ClickMode | string;
  /**
   * Enables or disables the LED.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets the content for the 'false' state.
   * Default value: """"
   */
  falseContent?: string;
  /**
   * Sets custom template for LED's false state.
   * Default value: null
   */
  falseTemplate?: any;
  /**
   * Sets the LED to indeterminate state.
   * Default value: false
   */
  indeterminate?: boolean;
  /**
   * Sets the content for the 'null' state.
   * Default value: """"
   */
  indeterminateContent?: string;
  /**
   * Sets a custom template for LED's indeterminate state.
   * Default value: null
   */
  indeterminateTemplate?: any;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: """"
   */
  name?: string;
  /**
   * If the widgets is readonly, the users cannot iteract with the element.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Sets the shape of LED.
   * Default value: round
   */
  shape?: LedShape | string;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets the content for the 'true' state.
   * Default value: """"
   */
  trueContent?: string;
  /**
   * Sets custom template for LED's true state.
   * Default value: null
   */
  trueTemplate?: any;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the element's value.
   * Default value: """"
   */
  value?: string;
}
/**
 LET Toggle button with Checked boolean value.
*/
export interface Led extends BaseElement, LedProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the widget is checked/unchecked.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value of the element before it was changed.
   *  value - The new value of the element.
   */
  onChange: ((this: any, ev: Event) => any) | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-led"): Led;
        querySelector(selectors: "smart-led"): Led | null;
        querySelectorAll(selectors: "smart-led"): NodeListOf<Led>;
        getElementsByTagName(qualifiedName: "smart-led"): HTMLCollectionOf<Led>;
        getElementsByName(elementName: "smart-led"): NodeListOf<Led>;
    }
}

/**Sets the shape of LED. */
export declare type LedShape = 'round' | 'square';
export interface ListBoxProperties {
  /**
   * Enables or disables the ability to drag list items out of the List box. Disabled items cannot be dragged.
   * Default value: false
   */
  allowDrag?: boolean;
  /**
   * Enables or disables the ability to drop list items inside the target List box.
   * Default value: false
   */
  allowDrop?: boolean;
  /**
   * Determines the number of color alternations in rows.
   * Default value: 0
   */
  alternationCount?: number;
  /**
   * Determines the ending index of color alternations in rows.
   * Default value: 0
   */
  alternationEnd?: number;
  /**
   * Determines the starting index of color alternations in rows
   * Default value: 0
   */
  alternationStart?: number;
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Enables or disables auto sorting. If sorted is enabled, but autoSort is false, the element will not be re-sorted automatically.
   * Default value: true
   */
  autoSort?: boolean;
  /**
   * Determines the data source that will be loaded to the ListBox. The dataSource can be an array of strings/numbers or objects where the attributes represent the properties of a List Item. For example label, value, group. It can also be a callback that returns an Array of items as previously described.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the list box.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines whether an indicator will appear during filtering and remote item loading.
   * Default value: false
   */
  displayLoadingIndicator?: boolean;
  /**
   * Sets or gets the displayMember. The displayMember specifies the name of an object property to display. The name is contained in the collection specified by the 'dataSource' property.
   * Default value: """"
   */
  displayMember?: string;
  /**
   * A callback function for customizing the HTML of the drag feedback. It receives one parameter - the currently dragged item.
   * Default value: null
   */
  dragFeedbackFormatFunction?: any;
  /**
   * Determines the offset of the drag feedback element from the mouse cursor when dragging an item. The first member of the array is the horizontal offset and the second one - the vertical offset.
   * Default value: 10,10
   */
  dragOffset?: number[];
  /**
   * Determines what happens when an item is dropped.
   * Default value: move
   */
  dropAction?: ListBoxDropAction | string;
  /**
   * Determines if list items can be edited or not. If enabled, items can be edited by double clicking on a target item ( that is not disabled ) or pressing the F2 key on the keyboard.
   * Default value: false
   */
  editable?: boolean;
  /**
   * Determines whether list items can be filtered or not. If enable a filter input appears at the top of the list box.
   * Default value: false
   */
  filterable?: boolean;
  /**
   * A callback that should return a condition that will be used for custom item filtering. Used in conjunction with filterMode 'custom'
   * Default value: null
   */
  filterCallback?: any;
  /**
   * Determines the filtering mode.
   * Default value: containsIgnoreCase
   */
  filterMode?: FilterMode | string;
  /**
   * Determines the placeholder for the filter input field.
   * Default value: ""
   */
  filterInputPlaceholder?: string;
  /**
   * If enabled, the items will be grouped by their first letter. Can't be applied if the dataSource already contains groups.
   * Default value: false
   */
  grouped?: boolean;
  /**
   * Determines which attribute from the dataSource object will be used as the group member for the items. If not set, by default 'group' property is used from the source object. groupMember is especially usefull when loading the data from a JSON file as a dataSource for the ListBox and there's a specific property that should be used to group the items.
   * Default value: null
   */
  groupMember?: string | null;
  /**
   * Determines the visibility of the horizontal Scroll bar.
   * Default value: auto
   */
  horizontalScrollBarVisibility?: HorizontalScrollBarVisibility | string;
  /**
   * IncrementalSearchDelay property specifies the time-interval in milliseconds until the previous search query is cleared. The timer starts when the user stops typing. A new query can be started only when the delay has passed.
   * Default value: 700
   */
  incrementalSearchDelay?: number;
  /**
   * Sets ot gets the mode of the incremental search mode. Incremental search is enabled by default. Typing while the List box is focused starts the incremental search.
   * Default value: startsWithIgnoreCase
   */
  incrementalSearchMode?: SearchMode | string;
  /**
   * Sets the height for all list box items. Used only when virtualization is enabled.
   * Default value: null
   */
  itemHeight?: number;
  /**
   * Determines the item width measuring algorithm.
   * Default value: auto
   */
  itemMeasureMode?: ListItemMeasureMode | string;
  /**
   * A getter that returns an array of all ListBox items.
   * Default value: 
   */
  items?: ListBoxItem[];
  /**
   * A string that represents the id of an HTMLTemplateElement inside the DOM or a reference to the template itself. It's used to set a custom template for the list items.
   * Default value: null
   */
  itemTemplate?: any;
  /**
   * Determines the text that will be displayed next to the loading indicator when the loader is visible and it's position is top or bottom.
   * Default value: "Loading..."
   */
  loadingIndicatorPlaceholder?: string;
  /**
   * Determines the position of the loading indicator.
   * Default value: center
   */
  loadingIndicatorPosition?: VerticalAlignment | string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property language. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "overridingProperties": "' property is used by default.",
   *     "invalidIndex": "' method accepts an index of type number or an Array of numbers.",
   *     "indexOutOfBound": "' method.",
   *     "invalidItem": "' method accepts an object or an array of objects as it's second parameter.",
   *     "invalidSettings": "' method accepts a string or an object as it's second parameter."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: """"
   */
  name?: string;
  /**
   * Determines the placeholder that will be shown when the List box is empty.
   * Default value: "No Items"
   */
  placeholder?: string;
  /**
   * Sets or gets the readonly property. If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Sets or gets the selected indexes. Selected indexes represents an array of the indexes of the items that should be selected.
   * Default value: 
   */
  selectedIndexes?: number[];
  /**
   * Sets or gets elected indexes. Selected values represents the values of the items that should be selected.
   * Default value: 
   */
  selectedValues?: string[];
  /**
   * Determines how many items can be selected depending on the selection mode.
   * Default value: oneOrManyExtended
   */
  selectionMode?: ListSelectionMode | string;
  /**
   * Determines when listbox selection is achieved - on 'press' or 'release'.
   * Default value: release
   */
  selectionChangeAction?: ListBoxSelectionChangeAction | string;
  /**
   * Determines whether the items are sorted alphabetically or not
   * Default value: false
   */
  sorted?: boolean;
  /**
   * Determines sorting direction - ascending(asc) or descending(desc)
   * Default value: "asc"
   */
  sortDirection?: string;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * Ensures the item with the target index is in view as the first (top) item in the list box.
   * Default value: -1
   */
  topVisibleIndex?: number;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value. Returns the selection. Return type: {label: string, value: any}[].
   * Default value: null
   */
  value?: any;
  /**
   * Determines the value member of an item. Stored as value in the item object. Similar to groupMember, valueMember is especially usefull when using data from a JSON file as a dataSource for the ListBox and there's a specific property that should be used for the value the items.
   * Default value: """"
   */
  valueMember?: string;
  /**
   * Determines the visibility of the vertical scroll bar.
   * Default value: auto
   */
  verticalScrollBarVisibility?: VerticalScrollBarVisibility | string;
  /**
   * Determines weather or not Virtualization is used for the ListBox. Virtualization allows a huge amount of items to be loaded to the List box while preserving the performance. For example a milion items can be loaded to the list box.
   * Default value: false
   */
  virtualized?: boolean;
}
/**
 ListBox allows the user to select one or more items from a list.
*/
export interface ListBox extends BaseElement, ListBoxProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when listbox binding is completed.
	* @param event. The custom event.    */
  onBindingComplete?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(addedItems, disabled, index, label, removedItems, selected, value)
   *  addedItems - An array of List items that have been selected.
   *  disabled - A flag indicating whether or not the item that caused the change event is disabled.
   *  index - The index of the List item that triggered the event.
   *  label - The label of the List item that triggered the event.
   *  removedItems - An array of List items that have been unselected before the event was fired.
   *  selected - The selected state of the List item that triggered the event. If an item was selected the value will be true and vice versa.
   *  value - The value of the List item that triggered the event.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when an item is dropped. The dragging operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(container, data, item, originalEvent, previousContainer, target)
   *  container - The List box that an item was dragged <strong>to.</strong>
   *  data - An object that contains data about the dragging operation like start position, start time, etc.
   *  item - The List item that was dragged.
   *  originalEvent - That original event that was fired.
   *  previousContainer - The List box that an item was dragged <strong>from</strong>.
   *  target - The event target.
   */
  onDragEnd: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a List item is being dragged.
	* @param event. The custom event. Custom data event was created with: ev.detail(data, item, originalEvent)
   *  data - An object that contains data about the dragging operation like start position, start time, etc.
   *  item - The List item that is being dragged. This is the item that has been clicked when initiating the drag operation
   *  originalEvent - The original event that initiates the dragging operation.
   */
  onDragging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an item is dragged. The dragging operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event. Custom data event was created with: ev.detail(container, data, item, originalEvent, previousContainer, target)
   *  container - The List box that an item was dragged <strong>to.</strong>
   *  data - An object that contains data about the dragging oepration like start position, start time, etc.
   *  item - The List item that was dragged.
   *  originalEvent - That original event that was fired.
   *  previousContainer - The List box that an item was dragged <strong>from</strong>.
   *  target - The event target.
   */
  onDragStart: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when an item is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(disabled, index, label, selected, value)
   *  disabled - Indicates whether the List item that was clicked is disabled or not.
   *  index - Indicates the index of the List item that was clicked.
   *  label - The label of the List item that was clicked.
   *  selected - Indicates whether the List item that was clicked is selected or not.
   *  value - The value of the List item that was clicked.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an item has been edited.
	* @param event. The custom event. Custom data event was created with: ev.detail(selected, disabled, index, label, value)
   *  selected - Indicates whether the List item is selected or not.
   *  disabled - Indicates whether the List item is disabled or not.
   *  index - The index of the List item that was edited.
   *  label - The label of the edited List item.
   *  value - The value of the List item that was edited.
   */
  onItemLabelChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user scrolls to the end of the list.
	* @param event. The custom event.    */
  onScrollBottomReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user scrolls to the beginning of the list.
	* @param event. The custom event.    */
  onScrollTopReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user swipes to the left, inside the listBox.
	* @param event. The custom event.    */
  onSwipeleft?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user swipes to the right, inside the listBox.
	* @param event. The custom event.    */
  onSwiperight?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Appends a ListItem to the end of the list of items inside element.
   * @param {Node} node. A ListItem element that should be added to the rest of the items as the last item.
   * @returns {Node}
   */
  appendChild<T extends Node>(node: Node): T;
  /**
   * Removes all items from the listBox.
   */
  clearItems(): void;
  /**
   * Unselects all items.
   */
  clearSelection(): void;
  /**
   * Ensures the target item is visible by scrolling to it.
   * @param {HTMLElement | string} item. A list item or value of the desired item to be visible.
   */
  ensureVisible(item: HTMLElement | string): void;
  /**
   * Returns an item instance from the listBox.
   * @param {string} value. The value of an item from the listBox.
   * @returns {HTMLElement}
   */
  getItem(value: string): HTMLElement;
  /**
   * Returns an array of ListBox items.
   * @returns {{label: string, value: string}[]}
   */
  getItems(): {label: string, value: string}[];
  /**
   * Inserts a new item at a specified index.
   * @param {number} index. The index where the item must be inserted.
   * @param {any} items. A single item/definition or an array of List Items/definitions of list items to be inserted. The format of the item definitions is the same as the format of the <strong>dataSource</strong> property.
   */
  insert(index: number, items: any): void;
  /**
   * Inserts a new ListItem before another in the list.
   * @param {Node} node. A ListItem element that should be added before the referenceItem in the list.
   * @param {Node | null} referenceNode. A ListItem element that acts as the reference item before which a new item is about to be inserted. The referenceNode must be in the same list as the node.
   * @returns {Node}
   */
  insertBefore<T extends Node>(node: Node, referenceNode: Node | null): T;
  /**
   * Removes an item at a specified index.
   * @param {number} index. The index of the removed item.
   */
  removeAt(index: number): void;
  /**
   * Removes a ListItem from the list of items inside element.
   * @param {Node} node. A ListItem element that is part of the list of items inside the element.
   * @returns {Node}
   */
  removeChild<T extends Node>(node: Node): T;
  /**
   * Selects an item from the listBox. 
   * @param {string | number | HTMLElement} item. A string, representing the value of the item or an HTML Element referencing an item from the list.
   */
  select(item: string | number | HTMLElement): void;
  /**
   * Unselects an item from the listBox. 
   * @param {string | HTMLElement} item. A string, representing the value of the item or an HTML Element referencing an item from the list
   */
  unselect(item: string | HTMLElement): void;
  /**
   * Updates an item from the listBox.
   * @param {number} index. The index of the item that is going to be updated.
   * @param {any} details. An object that contains the properties and their new values for the List item that should be updated. For example, label, value or selected attributes.
   */
  update(index: number, details: any): void;
}

export interface ListBoxItem {
  /**
   * The label of the list item.
   * Default value: 
   */
  label?: string | null;
  /**
   * The value of the list item.
   * Default value: 
   */
  value?: string | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-list-box"): ListBox;
        querySelector(selectors: "smart-list-box"): ListBox | null;
        querySelectorAll(selectors: "smart-list-box"): NodeListOf<ListBox>;
        getElementsByTagName(qualifiedName: "smart-list-box"): HTMLCollectionOf<ListBox>;
        getElementsByName(elementName: "smart-list-box"): NodeListOf<ListBox>;
    }
}

/**Determines what happens when an item is dropped. */
export declare type ListBoxDropAction = 'copy' | 'move' | 'none';
/**Determines when listbox selection is achieved - on 'press' or 'release'. */
export declare type ListBoxSelectionChangeAction = 'press' | 'release';
export interface ListItemProperties {
  /**
   * 
   * Default value: -1
   */
  alternationIndex?: number;
  /**
   * 
   * Default value: ""
   */
  color?: string;
  /**
   * 
   * Default value: plain
   */
  displayMode?: ListItemDisplayMode | string;
  /**
   * 
   * Default value: false
   */
  grouped?: boolean;
  /**
   * 
   * Default value: false
   */
  selected?: boolean;
  /**
   * 
   * Default value: ""
   */
  value?: string;
  /**
   * 
   * Default value: ""
   */
  label?: string;
  /**
   * 
   * Default value: ""
   */
  details?: string;
  /**
   * 
   * Default value: ""
   */
  group?: string;
  /**
   * 
   * Default value: false
   */
  hidden: boolean;
  /**
   * 
   * Default value: false
   */
  readonly?: boolean;
}
/**
 Defines a list item for ListBox, ComboBox, DropDownList.
*/
export interface ListItem extends BaseElement, ListItemProperties {

  /* Get a member by its name */
  [name: string]: any;
}

declare global {
    interface Document {
        createElement(tagName: "smart-list-item"): ListItem;
        querySelector(selectors: "smart-list-item"): ListItem | null;
        querySelectorAll(selectors: "smart-list-item"): NodeListOf<ListItem>;
        getElementsByTagName(qualifiedName: "smart-list-item"): HTMLCollectionOf<ListItem>;
        getElementsByName(elementName: "smart-list-item"): NodeListOf<ListItem>;
    }
}


export declare type ListItemDisplayMode = 'plain' | 'checkBox' | 'radioButton';
export interface ListItemsGroupProperties {
  /**
   * 
   * Default value: ""
   */
  label?: string;
}
/**
 Defines a group of list items.
*/
export interface ListItemsGroup extends BaseElement, ListItemsGroupProperties {

  /* Get a member by its name */
  [name: string]: any;
}

declare global {
    interface Document {
        createElement(tagName: "smart-list-items-group"): ListItemsGroup;
        querySelector(selectors: "smart-list-items-group"): ListItemsGroup | null;
        querySelectorAll(selectors: "smart-list-items-group"): NodeListOf<ListItemsGroup>;
        getElementsByTagName(qualifiedName: "smart-list-items-group"): HTMLCollectionOf<ListItemsGroup>;
        getElementsByName(elementName: "smart-list-items-group"): NodeListOf<ListItemsGroup>;
    }
}

export interface ListMenuProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines whether the element becomes focused on hover or not.
   * Default value: false
   */
  autoFocusOnMouseenter?: boolean;
  /**
   * Allows top-level ListMenu items to be checkable.
   * Default value: false
   */
  checkable?: boolean;
  /**
   * Sets or gets whether checkboxes and radio buttons can be displayed in the top level menu groups. This property is applicable only to the ListMenu itself, and not its smart-menu-item/smart-menu-items-group subitems. See also the property checkable.
   * Default value: false
   */
  checkboxes?: boolean;
  /**
   * Sets the check mode of top-level ListMenu items(groups).
   * Default value: checkbox
   */
  checkMode?: MenuCheckMode | string;
  /**
   * Determines the data source that will be loaded to the ListMenu. The data source represents an array of objects with the following properties: label - a string representing the text content of the item.value - the value of the item.shortcut - a string representing a shortuct for the item. It will be displayed inside the item.items - allows to define an array of sub menu items.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Displays or hides the loading indicator. Loading indicator is hidden by default.
   * Default value: false
   */
  displayLoadingIndicator?: boolean;
  /**
   * Determines the field in the data source that corresponds to an item's label.
   * Default value: "label"
   */
  displayMember?: string;
  /**
   * Sets custom outer container where the minimized dropdown will be appednded. By default it is in the ListMenu. The value of the property can be an HTML element or the id of an HTML element.
   * Default value: null
   */
  dropDownAppendTo?: string | HTMLElement;
  /**
   * If this property is enabled, when the element's minimized dropdown is opened, a transparent overlay is positioned between the dropdown and the rest of the document.
   * Default value: false
   */
  dropDownOverlay?: boolean;
  /**
   * Sets or gets the opening direction of the ListMenu minimized dropdown.
   * Default value: auto
   */
  dropDownPosition?: MenuDropDownPosition | string;
  /**
   * Enables or disables scrolling using the mouse wheel through overflowing menu items.
   * Default value: false
   */
  enableMouseWheelAction?: boolean;
  /**
   * Determines whether menu item filtering is enabled. When enabled a filter input is shown at the top of the element. Only items in the current view can be filtered.
   * Default value: false
   */
  filterable?: boolean;
  /**
   * Determines the placeholder for the filter input.
   * Default value: ""
   */
  filterInputPlaceholder?: string;
  /**
   * Determines the MenuItem property that will be used as a filtering criteria. By default the label property is used. It can be set to 'value' if the user wants to filter by the 'value' property or 'textContent' if the user wants to filter by text inside the MenuItem's content or any other property.
   * Default value: "label"
   */
  filterMember?: string;
  /**
   * Determines the filtering mode.
   * Default value: containsIgnoreCase
   */
  filterMode?: FilterMode | string;
  /**
   * If enabled, the items will be grouped by their first letter and sorted.
   * Default value: false
   */
  grouped?: boolean;
  /**
   * Determines the field in the data source that corresponds to an item group's subitems collection.
   * Default value: "items"
   */
  itemsMember?: string;
  /**
   * Determines the text that will be displayed next to the loading indicator when the loader is visible and it's position is top or bottom.
   * Default value: "Loading..."
   */
  loadingIndicatorPlaceholder?: string;
  /**
   * Determines the position of the loading indicator inside the element.
   * Default value: center
   */
  loadingIndicatorPosition?: VerticalAlignment | string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Allows to load a custom minimize icon from an HTMLTemplateElement.The property acceps the id of an HTMLTemplateElement or it's direct instance.
   * Default value: "null"
   */
  minimizeIconTemplate?: string;
  /**
   * Determines the minimum width of the ListMenu at which it will switch from normal to minimized mode. If set to null, the ListMenu does not minimize automatically.
   * Default value: null
   */
  minimizeWidth?: number | null;
  /**
   * Sets or gets the ListMenu's scroll buttons behavior.
   * Default value: auto
   */
  overflow?: Overflow | string;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines whether to use scrollbar or scrollButtons when content overflows an element's box.
   * Default value: scrollbar
   */
  scrollMode?: ListMenuScrollMode | string;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Determines the field in the data source that corresponds to an item's value.
   * Default value: "value"
   */
  valueMember?: string;
}
/**
 ListMenu allows you to present users a listing of options and sub options.
*/
export interface ListMenu extends BaseElement, ListMenuProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when a jqx-menu-items-group is expanded.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value, path, children)
   *  item - The menu item that was expanded.
   *  label - The label of the item that was expanded.
   *  value - The value of the item that was expanded.
   *  path - The path of the item that was expanded, e.g. '0.1', '1.1.2'.
   *  children - The children of the item that was expanded.
   */
  onExpand?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a menu item check state is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value, checked)
   *  item - The menu item which state was changed.
   *  label - The label of the item which state was changed.
   *  value - The value of the item which state was changed.
   *  checked - The checked state of the toggled item. If false the item is not toggled.
   */
  onItemCheckChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a list menu item is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value)
   *  item - The menu item that was clicked.
   *  label - The label of the clicked item.
   *  value - The value of the clicked item.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user scrolls to the bottom of the ListMenu.
	* @param event. The custom event.    */
  onScrollBottomReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user swipes to the left inside the ListMenu.
	* @param event. The custom event.    */
  onSwipeleft?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user swipes to the right inside the ListMenu.
	* @param event. The custom event.    */
  onSwiperight?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Adds an item to the list.
   * @param {HTMLElement} Item. A smart-menu-item to add to the List Menu.
   * @param {HTMLElement | string} Parent?. The smart-menu-items-group (or its id or numeric path) to add the item to.
   */
  addItem(Item: HTMLElement, Parent?: HTMLElement | string): void;
  /**
   * Navigates to the previous page (smart-menu-items-group).
   * @param {boolean} animation?. If set to false, disables collapse animation even if animation is enabled for the element
   */
  back(animation?: boolean): void;
  /**
   * Navigates to a particular page (smart-menu-items-group).
   * @param {string} id. The id or numeric path of a page (smart-menu-items-group).
   */
  changePage(id: string): void;
  /**
   * Checks an item.
   * @param {HTMLElement | string} item. smart-menu-item/smart-menu-items-group (or its id or numeric path).
   */
  checkItem(item: HTMLElement | string): void;
  /**
   * Gets an item by its id or numeric path.
   * @param {string} id. The id or numeric path of an item
   * @returns {HTMLElement}
   */
  getItem(id: string): HTMLElement;
  /**
   * Maximizes the List Menu.
   */
  maximize(): void;
  /**
   * Minimizes the List Menu. An icon is displayed and the menu is hidden when minimized.
   */
  minimize(): void;
  /**
   * Removes an item.
   * @param {HTMLElement | string} item. The smart-menu-item/smart-menu-items-group (or its id or numeric path) to remove.
   */
  removeItem(item: HTMLElement | string): void;
  /**
   * Unchecks an item.
   * @param {HTMLElement | string} item. smart-menu-item/smart-menu-items-group (or its id or numeric path).
   */
  uncheckItem(item: HTMLElement | string): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-list-menu"): ListMenu;
        querySelector(selectors: "smart-list-menu"): ListMenu | null;
        querySelectorAll(selectors: "smart-list-menu"): NodeListOf<ListMenu>;
        getElementsByTagName(qualifiedName: "smart-list-menu"): HTMLCollectionOf<ListMenu>;
        getElementsByName(elementName: "smart-list-menu"): NodeListOf<ListMenu>;
    }
}

/**Sets the check mode of top-level ListMenu items(groups). */
export declare type MenuCheckMode = 'checkbox' | 'radioButton';
/**Sets or gets the opening direction of the ListMenu minimized dropdown. */
export declare type MenuDropDownPosition = 'auto' | 'top-left' | 'top-right' | 'bottom-left' | 'bottom-right' | 'overlay-left' | 'overlay-right';
/**Sets or gets the ListMenu's scroll buttons behavior. */
export declare type Overflow = 'auto' | 'hidden' | 'scroll';
/**Determines whether to use scrollbar or scrollButtons when content overflows an element's box. */
export declare type ListMenuScrollMode = 'scrollbar' | 'scrollButtons';
export interface MapProperties {
  /**
   * Before you start using the maps component, you need a project with a billing account and the Maps JavaScript API enabled.To learn more, see https://developers.google.com/maps/gmp-get-started.The API key is a unique identifier that authenticates requests associated with your project for usage and billing purposes. You must have at least one API key associated with your project.
   * Default value: "null"
   */
  apiKey?: string;
  /**
   * The query parameter defines the location the user wants to display on the map.The user can input different values for continents, country, city or address.
   * Default value: "null"
   */
  query?: string;
  /**
   * The mode property handles which mode for the map to be used.
   * Default value: "null"
   */
  mode?: string;
  /**
   * The 'map-type' property gives the user a set of options to choose from  in order to display certain type of map such as: place,view, streetview.
   * Default value: "null"
   */
  mapType?: string;
  /**
   * This property is defined when used with 'stretview' map type and it defines the horizontal orientation of the streetview panorama.
   * Default value: 210
   */
  heading?: number;
  /**
   * The  'pitch' property set the value of the vertical orientation('up' and 'down') which by defaault from Google Maps is horizontal.
   * Default value: 10
   */
  pitch?: number;
  /**
   * The zoom level of the camera determines the scale of the map. At larger zoom levels more detail can be seen on the screen, while at smaller zoom levels more of the world can be seen on the component.For example the following values show the approximate level of detail you can expect to see at eah zoom level( 1: World, 5: Landmas,10: City,15: Streets,20: Buildings )
   * Default value: 12
   */
  zoom?: number;
}
/**
 Our map component makes it easy for displaying map with defined location by user requireing a small piece of code.
*/
export interface Map extends BaseElement, MapProperties {

  /* Get a member by its name */
  [name: string]: any;
}

declare global {
    interface Document {
        createElement(tagName: "smart-map"): Map;
        querySelector(selectors: "smart-map"): Map | null;
        querySelectorAll(selectors: "smart-map"): NodeListOf<Map>;
        getElementsByTagName(qualifiedName: "smart-map"): HTMLCollectionOf<Map>;
        getElementsByName(elementName: "smart-map"): NodeListOf<Map>;
    }
}

export interface MaskedTextBoxProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines whether promptChar can be entered as valid input by the user.
   * Default value: false
   */
  allowPromptAsInput?: boolean;
  /**
   * Determines whether the input accepts characters only from the ASCII character set.
   * Default value: false
   */
  asciiOnly?: boolean;
  /**
   * Specifies whether the input should be focused when the page is loaded.
   * Default value: false
   */
  autoFocus?: boolean;
  /**
   * Determines whether the mask is shown/hidden on focus/blur even if placeholder is not set.
   * Default value: false
   */
  autoShowMask?: boolean;
  /**
   * Determines whether literals and prompt characters are copied to the clipboard on cut/copy operations.
   * Default value: excludePromptAndLiterals
   */
  cutCopyMaskFormat?: MaskedTextBoxCutCopyMaskFormat | string;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Specifies the behavior on "Enter" key press. Default mode is "submit".
   * Default value: submit
   */
  enterKeyBehavior?: EnterKeyBehavior | string;
  /**
   * Determines whether the prompt character in the input mask is hidden when the masked text box isn't focused anymore.
   * Default value: false
   */
  hidePromptOnLeave?: boolean;
  /**
   * Sets additional helper text below the element. The hint is visible only when the element is focused.
   * Default value: ""
   */
  hint?: string;
  /**
   * Determines whether new user input overwrites the existing input value or not.
   * Default value: false
   */
  isOverwriteMode?: boolean;
  /**
   * Sets label above the element. The label is always visible.
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Defines the mask for the input.
   * Default value: "#####"
   */
  mask?: string;
  /**
   * Indicates whether all required fields of the mask have been populated or not.
   * Default value: false
   */
  maskCompleted?: boolean;
  /**
   * Indicates whether all required and optional fields of the mask have been populated or not.
   * Default value: false
   */
  maskFull?: boolean;
  /**
   * Determines the maximum number of characters that the user can enter.
   * Default value: 5
   */
  maxLength?: number;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: ""
   */
  name?: string;
  /**
   * A string that appears inside the input when there's no value and mask. 
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Determines the prompt char that is used for the mask of the element.
   * Default value: "_"
   */
  promptChar?: string;
  /**
   * If the element is readonly, the users cannot iteract with the element.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Determines whether the parsing of user input should stop after the first invalid character or not.
   * Default value: false
   */
  rejectInputOnFirstFailure?: boolean;
  /**
   * Specifies that the input must be filled in before submitting a form
   * Default value: false
   */
  required?: boolean;
  /**
   * Determines whether an input character that matches the prompt character should reset the current selected value in the input or not. Applicable only when allowPromptAsInput is enabled.
   * Default value: false
   */
  resetOnPrompt?: boolean;
  /**
   * Determines whether hitting space character resets the currently selected value from the input or not.
   * Default value: false
   */
  resetOnSpace?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Specifies whether the value of the input will be selected on focus or not.
   * Default value: false
   */
  selectAllOnFocus?: boolean;
  /**
   * Determines whether the value of the input should contain or not the prompt/literals of the mask.
   * Default value: excludePromptAndLiterals
   */
  textMaskFormat?: MaskedTextBoxTextMaskFormat | string;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element. 
   * Default value: ""
   */
  value?: string;
  /**
   * Callback function that allows to set custom validation on the value. If the function returns false then the value of the input is treated as not valid.
   * Default value: null
   */
  validation?: any;
}
/**
 MaskedTextBox uses a mask to control the input of the user.
*/
export interface MaskedTextBox extends BaseElement, MaskedTextBoxProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the value of the Text Box is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value before it was changed.
   *  value - The new value.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered on each key up event of the MaskedTextBox, if the value is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value before it was changed.
   *  value - The new value.
   */
  onChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered if the <b>validation</b> property is set. Indicates whether valiation has passed successfully or not.
	* @param event. The custom event. Custom data event was created with: ev.detail(success)
   *  success - A flag inidicating whether the validation was successfull or not.
   */
  onValidation?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Focuses the element. 
   */
  focus(): void;
  /**
   * Blurs the element. 
   */
  blur(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-masked-text-box"): MaskedTextBox;
        querySelector(selectors: "smart-masked-text-box"): MaskedTextBox | null;
        querySelectorAll(selectors: "smart-masked-text-box"): NodeListOf<MaskedTextBox>;
        getElementsByTagName(qualifiedName: "smart-masked-text-box"): HTMLCollectionOf<MaskedTextBox>;
        getElementsByName(elementName: "smart-masked-text-box"): NodeListOf<MaskedTextBox>;
    }
}

/**Determines whether literals and prompt characters are copied to the clipboard on cut/copy operations. */
export declare type MaskedTextBoxCutCopyMaskFormat = 'excludePromptAndLiterals' | 'includePrompt' | 'includeLiterals' | 'includePromptAndLiterals';
/**Specifies the behavior on "Enter" key press. Default mode is "submit". */
export declare type EnterKeyBehavior = 'clearOnSubmit' | 'submit';
/**Determines whether the value of the input should contain or not the prompt/literals of the mask. */
export declare type MaskedTextBoxTextMaskFormat = 'excludePromptAndLiterals' | 'includePrompt' | 'includeLiterals' | 'includePromptAndLiterals';
export interface MenuProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines delay (in milliseconds) before a Menu dropdown is closed when leaving the Menu with the mouse. Applicable only when selectionMode is 'mouseenter'.
   * Default value: 100
   */
  autoCloseDelay?: number;
  /**
   * If set to true, on mouseenter, the element receives focus automatically.
   * Default value: false
   */
  autoFocusOnMouseenter?: boolean;
  /**
   * Allows top-level Menu items (immediate children of the Menu) to be checkable. Sublevels are controlled by setting checkable to the respective smart-menu-items-group.
   * Default value: false
   */
  checkable?: boolean;
  /**
   * Sets or gets whether checkboxes and radio buttons can be displayed in the Menu. This property is applicable only to the Menu itself, and not its smart-menu-item/smart-menu-items-group subitems. See also the property checkable.
   * Default value: false
   */
  checkboxes?: boolean;
  /**
   * Sets the check mode of top-level Menu items (immediate children of the Menu). checkMode can be set to 'checkbox', 'radioButton', or a comma-separated list containing 'checkbox', 'radioButton', or 'none' (e.g. 'checkbox, radioButton, none, checkbox'). When set to a list, each value in the list is applied to groups of Menu items separated by an item with separator (item after the one with separator is the start of a new checkMode context). Sublevels are controlled by setting checkMode to the respective smart-menu-items-group.
   * Default value: checkbox
   */
  checkMode?: MenuCheckMode | string;
  /**
   * Sets the document event which closes any open Menu drop downs (or the Menu itself when mode is 'dropDown').
   * Default value: up
   */
  closeAction?: MenuCloseAction | string;
  /**
   * Determines the data source that will be loaded to the Menu. The data source represents an array of objects with the following properties: label - a string representing the text content of the item.value - the value of the item.shortcut - a string representing a shortuct for the item. It will be displayed inside the item.items - allows to define an array of sub menu items.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines the field in the data source that corresponds to an item's label.
   * Default value: "label"
   */
  displayMember?: string;
  /**
   * Sets custom outer container, where all dropdown containers must be appended. By default they are inside the menu. The value of the property can be an HTML element or the id of an HTML element. In mode 'dropDown', the property dropDownAppendTo also controls the parent element of the whole Menu. The open method works relatively to the original place of the Menu in the DOM.
   * Default value: null
   */
  dropDownAppendTo?: string | HTMLElement;
  /**
   * If this property is enabled, when an element's dropdown is opened, a transparent overlay is positioned between the dropdown and the rest of the document.
   * Default value: false
   */
  dropDownOverlay?: boolean;
  /**
   * Determines the opening direction of Menu dropdowns.
   * Default value: auto
   */
  dropDownPosition?: MenuDropDownPosition | string;
  /**
   * A getter that returns an array of all Menu items.
   * Default value: 
   */
  items?: any;
  /**
   * Determines the field in the data source that corresponds to an item group's subitems collection.
   * Default value: "items"
   */
  itemsMember?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Used to load a custom minimize icon from an HTMLTemplateElement object. The HTMLTemplateElement is selected by it's id.
   * Default value: "null"
   */
  minimizeIconTemplate?: string;
  /**
   * Determines the minimum width of the Menu at which it will switch from normal to minimized mode. If set to null, the Menu does not minimize automatically.
   * Default value: null
   */
  minimizeWidth?: number;
  /**
   * Determines the menu's display mode.
   * Default value: horizontal
   */
  mode?: MenuMode | string;
  /**
   * Opens or closes thte menu when it's in 'dropDown' mode.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Sets or gets the menu's scroll buttons behavior. Applicable only when dropDownAppendTo is not null.
   * Default value: auto
   */
  overflow?: Overflow | string;
  /**
   * If set to true, prevents the closing of the Menu or its dropdowns when Menu items are checked/unchecked.
   * Default value: false
   */
  preventCloseOnCheck?: boolean;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the menu's selection mode.
   * Default value: click
   */
  selectionMode?: MenuSelectionMode | string;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Determines the field in the data source that corresponds to an item's value.
   * Default value: "value"
   */
  valueMember?: string;
}
/**
 Horizontal, Vertical and Popup Menu. Popup Menus appear when a user taps an interactive UI element such as an icon, button, action, or content, such as selected items or text.
*/
export interface Menu extends BaseElement, MenuProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the menu is closed. The event is fired only in <em>'dropDown'</em> <strong>mode</strong>.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the menu is about to be closed. The closing operation can be canceled by calling event.preventDefault() in the event handler function. The event is fired only in <em>'dropDown'</em> <strong>mode</strong>.
	* @param event. The custom event. Custom data event was created with: ev.detail(trigger)
   *  trigger - Indicates whether the event was called from inside the element or programatically.
   */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a jqx-menu-items-group is collapsed.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value, path, children)
   *  item - The menu item that was collapsed.
   *  label - The label of the toggled item that was collapsed.
   *  value - The value of the toggled item that was collapsed.
   *  path - The path of the toggled item that was collapsed, e.g. '0.1', '1.1.2'.
   *  children - The children items of the toggled item that was collapsed.
   */
  onCollapse?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a jqx-menu-items-group is collapsing.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value, path, children)
   *  item - The menu item that is going to be collapsed.
   *  label - The label of the toggled item that is going to be collapsed.
   *  value - The value of the toggled item that is going to be collapsed.
   *  path - The path of the toggled item that is going to be collapsed, e.g. '0.1', '1.1.2'.
   *  children - The children items of the toggled item that is going to be collapsed.
   */
  onCollapsing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a jqx-menu-items-group is expanded.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value, path, children)
   *  item - The menu item that was expanded.
   *  label - The label of the toggled item that was expanded.
   *  value - The value of the toggled item that was expanded.
   *  path - The path of the toggled item that was expanded, e.g. '0.1', '1.1.2'.
   *  children - The children items of the toggled item that was expanded.
   */
  onExpand?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered before a jqx-menu-items-group is expanded.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value, path, children)
   *  item - The menu item that is going to be expanded.
   *  label - The label of the toggled item that is going to be expanded.
   *  value - The value of the toggled item that is going to be expanded.
   *  path - The path of the toggled item that is going to be expanded, e.g. '0.1', '1.1.2'.
   *  children - The children items of the toggled item that is going to be expanded.
   */
  onExpanding?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a menu item check state is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value, checked)
   *  item - The menu item which state was changed.
   *  label - The label of the item which state was changed.
   *  value - The value of the item which state was changed.
   *  checked - The checked state of the toggled item. If false the item is not toggled.
   */
  onItemCheckChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a menu item is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value)
   *  item - The menu item that is toggled.
   *  label - The label of the toggled item.
   *  value - The value of the toggled item.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the menu is opened. The event is fired only in <em>'dropDown'</em> <strong>mode</strong>.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the menu is about to be opened. The opening operation can be canceled by calling event.preventDefault() in the event handler function. The event is fired only in <em>'dropDown'</em> <strong>mode</strong>.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Adds an item to the menu.
   * @param {HTMLElement} Item. A smart-menu-item to add to the Menu.
   * @param {HTMLElement | string} Parent?. The smart-menu-items-group or its id or numeric path to add the item to.
   */
  addItem(Item: HTMLElement, Parent?: HTMLElement | string): void;
  /**
   * Checks an item.
   * @param {HTMLElement | string} item. smart-menu-item/smart-menu-items-group or its id or numeric path.
   */
  checkItem(item: HTMLElement | string): void;
  /**
   * Clears all Menu items.
   */
  clear(): void;
  /**
   * Clicks on an item programatically.
   * @param {HTMLElement | string} item. smart-menu-item/smart-menu-items-group or its id or numeric path.
   */
  clickItem(item: HTMLElement | string): void;
  /**
   * Closes the Menu when <strong>mode</strong> is <em>'dropDown'</em>.
   */
  close(): void;
  /**
   * Collapses an item.
   * @param {HTMLElement | string} item?. smart-menu-item/smart-menu-items-group or its id or numeric path. If no item is passed, all open items are collapsed.
   * @param {boolean} animation?. If set to false, disables collapse animation even if animation is enabled for the element.
   */
  collapseItem(item?: HTMLElement | string, animation?: boolean): void;
  /**
   * Expands an item.
   * @param {HTMLElement | string} item. smart-menu-item/smart-menu-items-group or its id or numeric path.
   * @param {boolean} animation?. If set to false, disables expand animation even if animation is enabled for the element.
   */
  expandItem(item: HTMLElement | string, animation?: boolean): void;
  /**
   * Gets an item by its id or numeric path.
   * @param {string} id. The id or numeric path of an item
   * @returns {HTMLElement}
   */
  getItem(id: string): HTMLElement;
  /**
   * Maximizes the Menu.
   */
  maximize(): void;
  /**
   * Minimizes the Menu.
   */
  minimize(): void;
  /**
   * Opens the Menu when <strong>mode</strong> is <em>'dropDown'</em>.
   * @param {number} left. Horizontal position
   * @param {number} top. Vertical position
   */
  open(left: number, top: number): void;
  /**
   * Removes an item from the menu.
   * @param {HTMLElement | string} item. The smart-menu-item/smart-menu-items-group or its id or numeric path to remove.
   */
  removeItem(item: HTMLElement | string): void;
  /**
   * Unchecks an item.
   * @param {HTMLElement | string} item. smart-menu-item/smart-menu-items-group (or its id or numeric path)
   */
  uncheckItem(item: HTMLElement | string): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-menu"): Menu;
        querySelector(selectors: "smart-menu"): Menu | null;
        querySelectorAll(selectors: "smart-menu"): NodeListOf<Menu>;
        getElementsByTagName(qualifiedName: "smart-menu"): HTMLCollectionOf<Menu>;
        getElementsByName(elementName: "smart-menu"): NodeListOf<Menu>;
    }
}

/**Sets the document event which closes any open Menu drop downs (or the Menu itself when mode is 'dropDown'). */
export declare type MenuCloseAction = 'down' | 'up' | 'none';
/**Determines the menu's display mode. */
export declare type MenuMode = 'horizontal' | 'vertical' | 'dropDown' | 'tree';
/**Determines the menu's selection mode. */
export declare type MenuSelectionMode = 'click' | 'mouseenter';
export interface MenuItemProperties {
  /**
   * 
   * Default value: true
   */
  checked?: boolean;
  /**
   * Enables or disables element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * 
   * Default value: 
   */
  label?: any;
  /**
   * 
   * Default value: null
   */
  level?: number;
  /**
   * 
   * Default value: true
   */
  separator?: boolean;
  /**
   * 
   * Default value: ""
   */
  shortcut?: string;
  /**
   * 
   * Default value: null
   */
  value?: any;
}
/**
 Defines a menu item.
*/
export interface MenuItem extends BaseElement, MenuItemProperties {

  /* Get a member by its name */
  [name: string]: any;
}

declare global {
    interface Document {
        createElement(tagName: "smart-menu-item"): MenuItem;
        querySelector(selectors: "smart-menu-item"): MenuItem | null;
        querySelectorAll(selectors: "smart-menu-item"): NodeListOf<MenuItem>;
        getElementsByTagName(qualifiedName: "smart-menu-item"): HTMLCollectionOf<MenuItem>;
        getElementsByName(elementName: "smart-menu-item"): NodeListOf<MenuItem>;
    }
}

export interface MenuItemsGroupProperties {
  /**
   * 
   * Default value: false
   */
  checkable?: boolean;
  /**
   * 
   * Default value: false
   */
  checked?: boolean;
  /**
   * 
   * Default value: checkbox
   */
  checkMode?: MenuCheckMode | string;
  /**
   * Enables or disables element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * 
   * Default value: null
   */
  dropDownHeight?: number;
  /**
   * 
   * Default value: false
   */
  expanded?: boolean;
  /**
   * 
   * Default value: 
   */
  label?: any;
  /**
   * 
   * Default value: null
   */
  level?: number;
  /**
   * 
   * Default value: false
   */
  separator?: boolean;
  /**
   * 
   * Default value: null
   */
  value?: any;
}
/**
 Defines a group of menu items.
*/
export interface MenuItemsGroup extends BaseElement, MenuItemsGroupProperties {

  /* Get a member by its name */
  [name: string]: any;
}

declare global {
    interface Document {
        createElement(tagName: "smart-menu-items-group"): MenuItemsGroup;
        querySelector(selectors: "smart-menu-items-group"): MenuItemsGroup | null;
        querySelectorAll(selectors: "smart-menu-items-group"): NodeListOf<MenuItemsGroup>;
        getElementsByTagName(qualifiedName: "smart-menu-items-group"): HTMLCollectionOf<MenuItemsGroup>;
        getElementsByName(elementName: "smart-menu-items-group"): NodeListOf<MenuItemsGroup>;
    }
}

export interface MultiColumnFilterPanelProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Sets or gets the position of the close button of multi column filter panel items.
   * Default value: left
   */
  closeButtonPosition?: MultiColumnFilterPanelCloseButtonPosition | string;
  /**
   * Determines the data source that will be loaded to the multi column filter panel.Each member of the dataSource array is an object with the following fields:dataField - the dataField of the column to be grouped.dataType - the data type of the column to be grouped.groupIndex - the group order of columns. If this value is -1, the grouping will not be applied by this column initially.label - the column label to be displayed in the column selection input.icon - a specific class to be applied to the respective item in the column selection input.sortDirection - the sort direction to be applied when grouping. Possible values: 'ascending' and 'descending'.
   * Default value: null
   */
  dataSource?: {label: string, dataField: string, dataType: string, sortDirection: string, groupIndex: number}[];
  /**
   * Enables or disables the multi column filter panel.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * A callback function that can be used to modify the settings of value editors.
   * Default value: null
   */
  editorCallback?: any;
  /**
   * Sets or gets the placeholder for the filter value editors.
   * Default value: "Value"
   */
  editorPlaceholder?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages.
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets the maximum number of columns to group by. If set to null, there is no limit.
   * Default value: 8
   */
  maxLevel?: number;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid ''!",
   *     "propertyInvalidValueType": "Invalid ''!",
   *     "methodInvalidValueType": "Invalid ''!",
   *     "methodInvalidArgumentsCount": "Invalid '' argument(s)!",
   *     "methodInvalidReturnType": "Invalid ''!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": "'.",
   *     "htmlTemplateNotSuported": ": Web Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "apply": "Apply",
   *     "booleanFirst": "?",
   *     "booleanLast": "?",
   *     "cancel": "Cancel",
   *     "dateFirst": "1",
   *     "dateLast": "9",
   *     "from": "from",
   *     "numberFirst": "1",
   *     "numberLast": "9",
   *     "pickAnother": "Pick another field to group by",
   *     "firstBy": "Group by",
   *     "stringFirst": "A",
   *     "stringLast": "Z",
   *     "thenBy": "then by",
   *     "collapseAll": "Collapse all",
   *     "expandAll": "Expand all"
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the logical operator between the items.
   * Default value: false
   */
  operator?: MultiColumnFilterPanelOperator | string;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets of gets the initial value of the element.
   * Default value: 
   */
  value?: any;
}
/**
 Defines an advanced filter panel used for Grid and CardView filtering.
*/
export interface MultiColumnFilterPanel extends BaseElement, MultiColumnFilterPanelProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the "Apply" button is clicked.
	* @param event. The custom event.    */
  onApply?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the "Cancel" button is clicked.
	* @param event. The custom event.    */
  onCancel: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the "Collapse all" button is clicked.
	* @param event. The custom event.    */
  onCollapseAll?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the "Expand all" button is clicked.
	* @param event. The custom event.    */
  onExpandAll?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-multi-column-filter-panel"): MultiColumnFilterPanel;
        querySelector(selectors: "smart-multi-column-filter-panel"): MultiColumnFilterPanel | null;
        querySelectorAll(selectors: "smart-multi-column-filter-panel"): NodeListOf<MultiColumnFilterPanel>;
        getElementsByTagName(qualifiedName: "smart-multi-column-filter-panel"): HTMLCollectionOf<MultiColumnFilterPanel>;
        getElementsByName(elementName: "smart-multi-column-filter-panel"): NodeListOf<MultiColumnFilterPanel>;
    }
}

/**Sets or gets the position of the close button of multi column filter panel items. */
export declare type MultiColumnFilterPanelCloseButtonPosition = 'left' | 'right';
/**Determines the logical operator between the items. */
export declare type MultiColumnFilterPanelOperator = 'and' | 'or';
export interface MultiComboInputProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the delay before the drop down opens to show the matches from the auto complete operation. The delay is measured in miliseconds.
   * Default value: 100
   */
  autoCompleteDelay?: number;
  /**
   * Enables or disables the close buttons when multiple items are selected.
   * Default value: false
   */
  hideInputTagsCloseButton?: boolean;
  /**
   * Enables or disables whether when an item has a 'color' property set, the tag of that item will set automatically the background to that color.
   * Default value: true
   */
  colorItems?: boolean;
  /**
   * Determines the data source that will be loaded to the MutliInput. The dataSource can be an array of strings/numbers or objects where the attributes represent the properties of a List Item. For example label, value. It can also be a callback that returns an Array of items as previously described.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines the position of the drop down button.
   * Default value: right
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to an empty string. In this case the height of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Sets the purpose of the input and what, if any, permission the user agent has to provide automated assistance in filling out the element's input when in a form, as well as guidance to the browser as to the type of information expected in the element. This value corresponds to the standard HTML autocomplete attribute and can be set to values such as 'on', 'name', 'organization', 'street-address', etc.
   * Default value: "off"
   */
  inputPurpose?: string;
  /**
   * Determines the maximum number of matched items that should be visible inside the drop down as a result of a new autoComplete query. By default the maximum number of 8 items can be displayed inside the drop down.
   * Default value: 8
   */
  items?: number;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the minimum number of characters inside the input in order to trigger the autocomplete functionality that will open the drop down and show the matched items.
   * Default value: 1
   */
  minLength?: number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the drop down is opened or not.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Enables or disables whether drop-down items are rendered as pills.
   * Default value: false
   */
  pills?: boolean;
  /**
   * Sets or gets the query that is used to filter the items. Query is used by the autoComplete operation. Empty string means that all items from the data source will be displayed and no filter query is applied.
   * Default value: 
   */
  query?: string | number;
  /**
   * Determines the auto complete query mode. This property also determines the matching algorithm for the autocomplete operation.
   * Default value: containsIgnoreCase
   */
  queryMode?: MultiComboInputQueryMode | string;
  /**
   * Determines whether ot not the user can enter text inside the input. if dropDownButtonPosition is set to 'left' or 'right' then readonly determines whether the element acts as a ComboBox or a DropDownList if a dataSource is provided.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the delimiter between the tags in the input field.
   * Default value: ""
   */
  separator?: string;
  /**
   * Determines whether only a single item can be selected.
   * Default value: false
   */
  singleSelect?: boolean;
  /**
   * Determines whether an additional item is displayed as the first item in the options list, which allows to select/unselect all items.
   * Default value: false
   */
  selectAll?: boolean;
  /**
   * Gets or sets an array of selected values.
   * Default value: 
   */
  selectedValues?: any;
  /**
   * Determines whether the items are sorted alphabetically or not
   * Default value: false
   */
  sorted?: boolean;
  /**
   * Determines the sorting algorithm - ascending(asc) or descending(desc) if sort is enabled.
   * Default value: "asc"
   */
  sortDirection?: string;
  /**
   * Determines whether the input field will contain tags for each selected item from the popup or just one that shows the number of selected items.
   * Default value: many
   */
  inputTagsMode?: MultiComboInputInputTagsMode | string;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines the input type. Input type determines what input can be entered.
   * Default value: ""
   */
  type?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element.
   * Default value: ""
   */
  value?: string;
}
/**
 MultiComboInput specifies an input field where the user can enter data. Auto-complete options with checkbxoes are displayed for easier input. Allows multiple selection. Selected items are added to the input field as tags.
*/
export interface MultiComboInput extends BaseElement, MultiComboInputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, oldLabel, oldValue, value)
   *  label - The label of the new selected item.
   *  oldLabel - The label of the item that was previously selected before the event was triggered.
   *  oldValue - The value of the item that was previously selected before the event was triggered.
   *  value - The value of the new selected item.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Closes the drop down.
   */
  close(): void;
  /**
   * Ensures that the active ( selected ) item is always visible.
   */
  ensureVisible(): void;
  /**
   * Opens the drop down.
   */
  open(): void;
  /**
   * Selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-multi-combo-input"): MultiComboInput;
        querySelector(selectors: "smart-multi-combo-input"): MultiComboInput | null;
        querySelectorAll(selectors: "smart-multi-combo-input"): NodeListOf<MultiComboInput>;
        getElementsByTagName(qualifiedName: "smart-multi-combo-input"): HTMLCollectionOf<MultiComboInput>;
        getElementsByName(elementName: "smart-multi-combo-input"): NodeListOf<MultiComboInput>;
    }
}

/**Determines the auto complete query mode. This property also determines the matching algorithm for the autocomplete operation. */
export declare type MultiComboInputQueryMode = 'contains' | 'containsIgnoreCase' | 'doesNotContain' | 'doesNotContainIgnoreCase' | 'equals' | 'equalsIgnoreCase' | 'startsWith' | 'startsWithIgnoreCase' | 'endsWith' | 'endsWithIgnoreCase';
/**Determines whether the input field will contain tags for each selected item from the popup or just one that shows the number of selected items. */
export declare type MultiComboInputInputTagsMode = 'one' | 'many';
export interface MultiInputProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the delay before the drop down opens to show the matches from the auto complete operation. The delay is measured in miliseconds.
   * Default value: 100
   */
  autoCompleteDelay?: number;
  /**
   * Determines the data source that will be loaded to the MutliInput. The dataSource can be an array of strings/numbers or objects where the attributes represent the properties of a List Item. For example label, value. It can also be a callback that returns an Array of items as previously described.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Determines the position of the drop down button.
   * Default value: right
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to an empty string. In this case the height of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Sets the purpose of the input and what, if any, permission the user agent has to provide automated assistance in filling out the element's input when in a form, as well as guidance to the browser as to the type of information expected in the element. This value corresponds to the standard HTML autocomplete attribute and can be set to values such as 'on', 'name', 'organization', 'street-address', etc.
   * Default value: "off"
   */
  inputPurpose?: string;
  /**
   * Determines the maximum number of matched items that should be visible inside the drop down as a result of a new autoComplete query. By default the maximum number of 8 items can be displayed inside the drop down.
   * Default value: 8
   */
  items?: number;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the minimum number of characters inside the input in order to trigger the autocomplete functionality that will open the drop down and show the matched items.
   * Default value: 1
   */
  minLength?: number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the drop down is opened or not.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Sets or gets the query that is used to filter the items. Query is used by the autoComplete operation. Empty string means that all items from the data source will be displayed and no filter query is applied.
   * Default value: 
   */
  query?: string | number;
  /**
   * Determines the auto complete query mode. This property also determines the matching algorithm for the autocomplete operation.
   * Default value: containsIgnoreCase
   */
  queryMode?: MultiInputQueryMode | string;
  /**
   * Determines whether ot not the user can enter text inside the input. if dropDownButtonPosition is set to 'left' or 'right' then readonly determines whether the element acts as a ComboBox or a DropDownList if a dataSource is provided.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the delimiter between the selected items in the input.
   * Default value: ","
   */
  separator?: string;
  /**
   * Gets or sets an array of selected values.
   * Default value: 
   */
  selectedValues?: any;
  /**
   * Determines whether an additional item is displayed as the first item in the options list, which allows to select/unselect all items.
   * Default value: false
   */
  selectAll?: boolean;
  /**
   * Determines whether the items are sorted alphabetically or not
   * Default value: false
   */
  sorted?: boolean;
  /**
   * Determines the sorting algorithm - ascending(asc) or descending(desc) if sort is enabled.
   * Default value: "asc"
   */
  sortDirection?: string;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines the input type. Input type determines what input can be entered.
   * Default value: ""
   */
  type?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element.
   * Default value: ""
   */
  value?: string;
}
/**
 MultiInput specifies an input field where the user can enter data. Auto-complete options are displayed for easier input. Allows multiple selection. Selected items are added to the input's value.
*/
export interface MultiInput extends BaseElement, MultiInputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, oldLabel, oldValue, value)
   *  label - The label of the new selected item.
   *  oldLabel - The label of the item that was previously selected before the event was triggered.
   *  oldValue - The value of the item that was previously selected before the event was triggered.
   *  value - The value of the new selected item.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Closes the drop down.
   */
  close(): void;
  /**
   * Ensures that the active ( selected ) item is always visible.
   */
  ensureVisible(): void;
  /**
   * Opens the drop down.
   */
  open(): void;
  /**
   * Selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-multi-input"): MultiInput;
        querySelector(selectors: "smart-multi-input"): MultiInput | null;
        querySelectorAll(selectors: "smart-multi-input"): NodeListOf<MultiInput>;
        getElementsByTagName(qualifiedName: "smart-multi-input"): HTMLCollectionOf<MultiInput>;
        getElementsByName(elementName: "smart-multi-input"): NodeListOf<MultiInput>;
    }
}

/**Determines the auto complete query mode. This property also determines the matching algorithm for the autocomplete operation. */
export declare type MultiInputQueryMode = 'contains' | 'containsIgnoreCase' | 'doesNotContain' | 'doesNotContainIgnoreCase' | 'equals' | 'equalsIgnoreCase' | 'startsWith' | 'startsWithIgnoreCase' | 'endsWith' | 'endsWithIgnoreCase';
export interface MultilineTextBoxProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines whether and how the value should be automatically capitalized as it is entered/edited by the user.
   * Default value: none
   */
  autoCapitalize?: MultilineTextBoxAutoCapitalize | string;
  /**
   * Determines whether the value of the control can be automatically completed by the browser.
   * Default value: off
   */
  autoComplete?: MultiLineTextBoxAutoComplete | string;
  /**
   * Determines whether element will auto expand when the input overflows vertically.
   * Default value: false
   */
  autoExpand?: boolean;
  /**
   * Determines whether the input should be focused when the page is loaded.
   * Default value: false
   */
  autoFocus?: boolean;
  /**
   * The cols attribute specifies the visible width of a input. If it is specified, it must be a positive integer. If it is not specified, the default value is 20.
   * Default value: 20
   */
  cols?: number;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Specifies how the characters are displayed inside the input.
   * Default value: default
   */
  displayMode?: TextBoxDisplayMode | string;
  /**
   * Determines the behavior on "Enter" key.
   * Default value: newLine
   */
  enterKeyBehavior?: MultilineTextBoxEnterKeyBehavior | string;
  /**
   * The form element that the element is associated with (its "form owner"). The value of the attribute must be the ID of a form element in the same document.
   * Default value: ""
   */
  form?: string;
  /**
   * Sets additional helper text below the element. Appears only when the element is focused.
   * Default value: null
   */
  hint?: any;
  /**
   * Controls horizontal scrollbar's visibility. 
   * Default value: auto
   */
  horizontalScrollBarVisibility?: HorizontalScrollBarVisibility | string;
  /**
   * Sets the purpose of the input and what, if any, permission the user agent has to provide automated assistance in filling out the element's input when in a form, as well as guidance to the browser as to the type of information expected in the element. This value corresponds to the standard HTML autocomplete attribute and can be set to values such as 'on', 'name', 'organization', 'street-address', etc.
   * Default value: "off"
   */
  inputPurpose?: string;
  /**
   * Sets label above the element. The label is displayed above the input and it's always visible.
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets the maximum number of characters that the user can enter.
   * Default value: null
   */
  maxLength?: number;
  /**
   * Sets or gets the minimum number of characters that the user can enter.
   * Default value: 0
   */
  minLength?: number;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: ""
   */
  name?: string;
  /**
   * The placeholder text that is displayed when no value is applied to the element. 
   * Default value: ""
   */
  placeholder?: string;
  /**
   * If enabled the users cannot iteract with the element.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Specifies that the user must fill in a value before submitting a form that contains the element.
   * Default value: false
   */
  required?: boolean;
  /**
   * Enables/ disables the resizing of the element. If enabled a resizing indicator appears in the bottom corner of the input area.
   * Default value: false
   */
  resizable?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * The number of visible text lines for the control.
   * Default value: 5
   */
  rows?: number;
  /**
   * Specifies whether the content of the input will be selected on focus.
   * Default value: false
   */
  selectAllOnFocus?: boolean;
  /**
   * Indicates the index of the last character in the current selection.
   * Default value: 0
   */
  selectionEnd?: number;
  /**
   * Indicates the index to the first character in the current selection.
   * Default value: 0
   */
  selectionStart?: number;
  /**
   * Specifies whether the element is to have its spelling and grammar checked or not.
   * Default value: false
   */
  spellCheck?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element.
   * Default value: ""
   */
  value?: string;
  /**
   * Controls vertical scrollbar's visibility. 
   * Default value: auto
   */
  verticalScrollBarVisibility?: VerticalScrollBarVisibility | string;
  /**
   * Indicates how the control wraps text.
   * Default value: soft
   */
  wrap?: MultilineTextBoxWrap | string;
}
/**
 Defines a multi-line text input control. MultilineTextBox can hold an unlimited number of characters, and the text renders in a fixed-width font
*/
export interface MultilineTextBox extends BaseElement, MultilineTextBoxProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the value of the text box is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value, type)
   *  oldValue - The previous value of the element before it was changed.
   *  value - The new value of the element.
   *  type - Indicates when the element was called, e.g. on blur or submit.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Focuses the element.
   */
  focus(): void;
  /**
   * The method is used to reset the value of the element box to it's initial state.
   */
  reset(): void;
  /**
   * Returns the currenctly selected text.
   * @param {string} displayMode. If <b>displayMode</b> is set to 'escaped', the value returned from the method contains escaped special characters.
   * @returns {string}
   */
  selection(displayMode: string): string;
  /**
   * Selects a certain part of the input text. If no arguments are specified the whole text will be selected.
   * @param {any} rangeFrom?. Determines the start index of the text selection.
   * @param {any} rangeTo?. Determines the end index of the text selection.
   */
  select(rangeFrom?: any, rangeTo?: any): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-multiline-text-box"): MultilineTextBox;
        querySelector(selectors: "smart-multiline-text-box"): MultilineTextBox | null;
        querySelectorAll(selectors: "smart-multiline-text-box"): NodeListOf<MultilineTextBox>;
        getElementsByTagName(qualifiedName: "smart-multiline-text-box"): HTMLCollectionOf<MultilineTextBox>;
        getElementsByName(elementName: "smart-multiline-text-box"): NodeListOf<MultilineTextBox>;
    }
}

/**Determines whether and how the value should be automatically capitalized as it is entered/edited by the user. */
export declare type MultilineTextBoxAutoCapitalize = 'none' | 'characters' | 'words';
/**Determines whether the value of the control can be automatically completed by the browser. */
export declare type MultiLineTextBoxAutoComplete = 'on' | 'off';
/**Specifies how the characters are displayed inside the input. */
export declare type TextBoxDisplayMode = 'default' | 'escaped';
/**Determines the behavior on "Enter" key. */
export declare type MultilineTextBoxEnterKeyBehavior = 'clearOnSubmit' | 'newLine' | 'submit';
/**Indicates how the control wraps text. */
export declare type MultilineTextBoxWrap = 'hard' | 'soft' | 'off';
export interface MultiSplitButtonProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines a data source used to generate element's permanently visible buttons.
   * Default value: 
   */
  buttonsDataSource?: string[];
  /**
   * Determines the data source of the multi split button's dropdown.
   * Default value: []
   */
  dataSource?: any;
  /**
   * Enables or disables jqxMultiSplitButton.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Displays or hides the loading indicator
   * Default value: false
   */
  displayLoadingIndicator?: boolean;
  /**
   * Sets or gets the displayMember. The displayMember specifies the name of an object property to display. The name is contained in the collection specified by the 'dataSource' property.
   * Default value: """"
   */
  displayMember?: string;
  /**
   * Sets the parent container of the button's dropDown list (the popup). The expected value is CSS Selector, ID or 'body'. Used when a CSS property of unknowned parent is interfering with the visibility of the dropDown list.
   * Default value: "null"
   */
  dropDownAppendTo?: string;
  /**
   * Determines position of the drop down button.
   * Default value: right
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Defines how element's drop down behaves. In 'none' mode drop down never opens. In 'dropDownButton' mode drop down is opened only via elelent's drop down button. In 'auto' mode drop down is opened on click on the whole top section.
   * Default value: dropDownButton
   */
  dropDownOpenMode?: DropDownOpenMode | string;
  /**
   * If this property is enabled, when the element's dropdown is opened, a transparent overlay is positioned between the dropdown and the rest of the document.
   * Default value: false
   */
  dropDownOverlay?: boolean;
  /**
   * Determines the vertical position of the dropDown list. 'Auto' means its automatically determined depending on the viewport size.
   * Default value: auto
   */
  dropDownPosition?: DropDownPosition | string;
  /**
   * Determines whether the Filtering is enabled.
   * Default value: false
   */
  filterable?: boolean;
  /**
   * Determines the filtering for the drop down list mode.
   * Default value: startsWithIgnoreCase
   */
  filterMode?: FilterMode | string;
  /**
   * Determines the placeholder for the drop down list filter input field.
   * Default value: ""
   */
  filterInputPlaceholder?: string;
  /**
   * If enabled, the items will be grouped by their first letter. Can't be applied if the dataSource already contains groups.
   * Default value: false
   */
  grouped?: boolean;
  /**
   * Sets or gets the groupMember. If it's not set, by default is used 'group' property of the source object.
   * Default value: "null"
   */
  groupMember?: string;
  /**
   * Sets ot gets the incrementalSearchDelay property. The incrementalSearchDelay specifies the time-interval in milliseconds after which the previous search string is deleted. The timer starts when you stop typing.
   * Default value: 700
   */
  incrementalSearchDelay?: number;
  /**
   * Sets ot gets the mode of the incremental search mode.
   * Default value: startsWithIgnoreCase
   */
  incrementalSearchMode?: SearchMode | string;
  /**
   * Determines the height of the items.
   * Default value: null
   */
  itemHeight?: number | null;
  /**
   * The itemTemplate property is a string that represents the id of an HTMLTemplateElement in the DOM. It's used to load list items from the HTMLTemplateElement.
   * Default value: null
   */
  itemTemplate?: any;
  /**
   * Determines the text that will be displayed next to the loading indicator when the loader is visible and it's position is top or bottom.
   * Default value: "Loading..."
   */
  loadingIndicatorPlaceholder?: string;
  /**
   * The position of the loading indicator.
   * Default value: center
   */
  loadingIndicatorPosition?: VerticalAlignment | string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback, related to localization module. 
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property language. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines whether the popup is opened or closed
   * Default value: false
   */
  opened?: boolean;
  /**
   * Disables user interaction with the element.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets selected indexes of buttons's dropDown.
   * Default value: 
   */
  selectedIndexes?: number[];
  /**
   * Sets or gets selected values of buttons's dropDown.
   * Default value: 
   */
  selectedValues?: string[];
  /**
   * Determines how many items can be selected.
   * Default value: one
   */
  selectionMode?: MultiSplitButtonSelectionMode | string;
  /**
   * Determines whether the items in the dropDown are sorted alphabetically or not
   * Default value: false
   */
  sorted?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Determines the value member of an item. Stored as value in the item object.
   * Default value: """"
   */
  valueMember?: string;
  /**
   * Determines weather or not Virtualization is used for the button's dropDownList.
   * Default value: false
   */
  virtualized?: boolean;
}
/**
 Buttons group with DropDown and multiple action buttons.
*/
export interface MultiSplitButton extends BaseElement, MultiSplitButtonProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when action button is clicked.
	* @param event. The custom event.    */
  onButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(addedItems, disabled, index, label, removedItems, selected, value)
   *  addedItems - An array of List items that have been selected.
   *  disabled - A flag indicating whether or not the item that caused the change event is disabled.
   *  index - The index of the List item that triggered the event.
   *  label - The label of the List item that triggered the event.
   *  removedItems - An array of List items that have been unselected before the event was fired.
   *  selected - The selected state of the List item that triggered the event. If an item was selected the value will be true and vice versa.
   *  value - The value of the List item that triggered the event.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when button's dropDown list is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when button's dropDown list is closing.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an item is clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(disabled, index, label, selected, value)
   *  disabled - Indicates whether the List item that was clicked is disabled or not.
   *  index - Indicates the index of the List item that was clicked.
   *  label - The label of the List item that was clicked.
   *  selected - Indicates whether the List item that was clicked is selected or not.
   *  value - The value of the List item that was clicked.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when button's dropDown list is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when button's dropDown list is opening.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user scrolls to the end of the dropDown list.
	* @param event. The custom event.    */
  onScrollBottomReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user scrolls to the start of the dropDown list.
	* @param event. The custom event.    */
  onScrollTopReached?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Closes button's dropDown list.
   */
  close(): void;
  /**
   * Returns an item instance occured in the element's drop down.
   * @param {string} value. The value of an item from the drop down list or a button.
   * @returns {HTMLElement}
   */
  getItem(value: string): HTMLElement;
  /**
   * Returns an array with the items from the split button's dropDown list.
   * @returns {any[]}
   */
  items(): any[];
  /**
   * Inserts a new item at a specified position in the drop down list.
   * @param {number} position. The position where the item must be inserted.
   * @param {any} value. The value of the new item.
   */
  insert(position: number, value: any): void;
  /**
   * Opens the splitButton's dropDown list.
   */
  open(): void;
  /**
   * Removes an item at a specified position in the drop down list.
   * @param {number} position. The position of the removed item.
   */
  removeAt(position: number): void;
  /**
   * Updates an item from the dropDown list.
   * @param {number} position. The position where the item must be updated.
   * @param {any} value. The value of the updated item.
   */
  update(position: number, value: any): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-multi-split-button"): MultiSplitButton;
        querySelector(selectors: "smart-multi-split-button"): MultiSplitButton | null;
        querySelectorAll(selectors: "smart-multi-split-button"): NodeListOf<MultiSplitButton>;
        getElementsByTagName(qualifiedName: "smart-multi-split-button"): HTMLCollectionOf<MultiSplitButton>;
        getElementsByName(elementName: "smart-multi-split-button"): NodeListOf<MultiSplitButton>;
    }
}

/**Determines how many items can be selected. */
export declare type MultiSplitButtonSelectionMode = 'none' | 'oneOrManyExtended' | 'zeroOrMany' | 'oneOrMany' | 'zeroOrOne' | 'one' | 'checkBox' | 'radioButton';
export interface NumberInputProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets the purpose of the input and what, if any, permission the user agent has to provide automated assistance in filling out the element's input when in a form, as well as guidance to the browser as to the type of information expected in the element. This value corresponds to the standard HTML autocomplete attribute and can be set to values such as 'on', 'name', 'organization', 'street-address', etc.
   * Default value: "off"
   */
  inputPurpose?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Determines the max number that can be displayed inside the input.
   * Default value: 9999999999
   */
  max?: number;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the min number that can be displayed inside the input.
   * Default value: -9999999999
   */
  min?: number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Sets or gets the value format of the element.
   * Default value: [object Object]
   */
  numberFormat?: any;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Determines whether ot not the user can enter text inside the input.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Sets or gets the incremental/decremental step for the value of the element.
   * Default value: 1
   */
  step?: number;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element.
   * Default value: 0
   */
  value?: any;
}
/**
 NumberInput specifies an input field where the user can enter a numbers.
*/
export interface NumberInput extends BaseElement, NumberInputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, oldLabel, oldValue, value)
   *  label - The label of the new selected item.
   *  oldLabel - The label of the item that was previously selected before the event was triggered.
   *  oldValue - The value of the item that was previously selected before the event was triggered.
   *  value - The value of the new selected item.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * Returns the value in the desired format.
   * @param {string | number} value. The value to be formatted by the method. 
   * @param {any} format?. The object that contains the formatting properties. The argument should contain Intl.NumberFormat valid properties. For example, { style: 'currency', currency: 'EUR' }
   * @returns {string}
   */
  getFormattedValue(value: string | number, format?: any): string;
  /**
   * Returns the number of the input.
   * @returns {number}
   */
  getValue(): number;
  /**
   * Selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
  /**
   * Sets the value of the input.
   * @param {string | number} value. The value to be set. 
   */
  setValue(value: string | number): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-number-input"): NumberInput;
        querySelector(selectors: "smart-number-input"): NumberInput | null;
        querySelectorAll(selectors: "smart-number-input"): NodeListOf<NumberInput>;
        getElementsByTagName(qualifiedName: "smart-number-input"): HTMLCollectionOf<NumberInput>;
        getElementsByName(elementName: "smart-number-input"): NodeListOf<NumberInput>;
    }
}

export interface NumericTextBoxProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Sets or gets the char to use as the decimal separator in numeric values. 
   * Default value: "."
   */
  decimalSeparator?: string;
  /**
   * Enables or disables the jqxNumericTextBox. 
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets the parent container of the radix dropdown.
   * Default value: null
   */
  dropDownAppendTo?: any;
  /**
   * Determines if a dropdown will be displayed when the radix display button is clicked. The dropdown shows options for changing to the binary, octal, decimal, and hexadecimal numeral systems.
   * Default value: false
   */
  dropDownEnabled?: boolean;
  /**
   * Enables or disables incrementing/decrementing the value using the mouse wheel in jqxNumericTextBox. 
   * Default value: false
   */
  enableMouseWheelAction?: boolean;
  /**
   * Sets additional helper text below the element. 
   * Default value: ""
   */
  hint?: string;
  /**
   * Sets or gets the input format of the widget. Setting this property dynamically can lead to precision loss. 
   * Default value: integer
   */
  inputFormat?: NumericTextBoxInputFormat | string;
  /**
   * Sets a label above the element. 
   * Default value: ""
   */
  label?: string;
  /**
   * If this property is enabled, leading zeros are added (if necessary) to the binary and hexadecimal representations of a number based on wordLength.
   * Default value: false
   */
  leadingZeros?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback, related to localization module. 
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets the maximum value of the widget. 
   * Default value: null
   */
  max?: number | string;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "binary": "BIN",
   *     "octal": "OCT",
   *     "decimal": "DEC",
   *     "hexadecimal": "HEX",
   *     "integerOnly": "jqxNumericTextBox: The property  can only be set when inputFormat is integer.",
   *     "noInteger": "jqxNumericTextBox: the property  cannot be set when inputFormat is integer.",
   *     "significantPrecisionDigits": "jqxNumericTextBox: the properties significantDigits and precisionDigits cannot be set at the same time."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the minimum value of the widget. 
   * Default value: null
   */
  min?: number | string;
  /**
   * The name of the control.
   * Default value: ""
   */
  name?: string;
  /**
   * Enables or disables the setting of the value property to null or empty string. 
   * Default value: false
   */
  nullable?: boolean;
  /**
   * Sets or gets whether the radix dropdown is opened. Applicable only when dropDownEnabled is true. 
   * Default value: false
   */
  opened?: boolean;
  /**
   * Sets or gets the pattern which the input value is displayed in when the element is not focused. All formats available to the NumberRenderer class can be applied as outputFormatString. 
   * Default value: "null"
   */
  outputFormatString?: string;
  /**
   * Determines the widget's place holder displayed when the widget's input is empty. 
   * Default value: """"
   */
  placeholder?: string;
  /**
   * Determines the number of digits after the decimal point. Applicable when inputFormat is either 'floatingPoint' or 'complex'. 
   * Default value: null
   */
  precisionDigits?: number;
  /**
   * Sets or gets the radix of the jqxNumericTextBox. The radix specifies the numeral system in which to display the widget's value. Applicable only when inputFormat is 'integer'. 
   * Default value: 10
   */
  radix?: NumericTextBoxRadix | string;
  /**
   * Enables or disables the radix display button of the jqxNumericTextBox. Applicable only when inputFormat is 'integer'. 
   * Default value: false
   */
  radixDisplay?: boolean;
  /**
   * Sets or gets the position of the radix display button of the jqxNumericTextBox. 
   * Default value: left
   */
  radixDisplayPosition?: NumericTextBoxDisplayPosition | string;
  /**
   * Sets or gets the readonly state of the jqxNumericTextBox. 
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Enables or disables outputting the value in scientific notation. Applicable only when inputFormat is 'integer'. 
   * Default value: false
   */
  scientificNotation?: boolean;
  /**
   * Determines whether to show the current value represented in all four numeral systems in the drop down.
   * Default value: false
   */
  showDropDownValues?: boolean;
  /**
   * Enables or disables the visibility of the units. 
   * Default value: false
   */
  showUnit?: boolean;
  /**
   * Determining how many significant digits are in a number. Applicable when inputFormat is either 'floatingPoint' or 'complex'. 
   * Default value: 8
   */
  significantDigits?: number;
  /**
   * Enables or disables the visibility of the spin buttons. 
   * Default value: false
   */
  spinButtons?: boolean;
  /**
   * Sets the delay between repeats of spin buttons in miliseconds. 
   * Default value: 75
   */
  spinButtonsDelay?: number;
  /**
   * Sets a delay before the first repeat iteration of spin buttons in miliseconds. 
   * Default value: 0
   */
  spinButtonsInitialDelay?: number;
  /**
   * Sets or gets the position of the spin buttons of the jqxNumericTextBox. 
   * Default value: right
   */
  spinButtonsPosition?: NumericTextBoxDisplayPosition | string;
  /**
   * Sets or gets the increase/decrease step. 
   * Default value: 1
   */
  spinButtonsStep?: number | string;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the name of unit used in jqxNumericTextBox widget. 
   * Default value: "kg"
   */
  unit?: string;
  /**
   * Sets the value's validation by min/max. If 'strict' is applied, the value is always validated by min and max. If 'interaction' is applied, programmatic value changes are not coerced to min/max and if min/max are changed, resulting in the current value being out of range, the value is not coerced, and no change event is fired.
   * Default value: strict
   */
  validation?: Validation | string;
  /**
   * Sets or gets the value of the jqxNumericTextBox widget. 
   * Default value: 0
   */
  value?: any;
  /**
   * Sets or gets the word length. Applicable only when inputFormat is 'integer'. If min and/or max are not set by default, they will be set automatically based on the specified word length. 
   * Default value: int32
   */
  wordLength?: WordLength | string;
}
/**
 input field for entering a number. Includes number formatting for Engineers and Scientists.
*/
export interface NumericTextBox extends BaseElement, NumericTextBoxProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the value is changed. 
	* @param event. The custom event.    */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the value in the input is being changed via keypress or paste. 
	* @param event. The custom event.    */
  onChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the dropdown is closed. 
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the dropdown is about to be closed. The closing operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the dropdown is opened. 
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the dropdown is about to be opened. The opening operation can be canceled by calling event.preventDefault() in the event handler function.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the radix is changed. 
	* @param event. The custom event.    */
  onRadixChange?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Focuses the NumericTextBox. 
   */
  focus(): void;
  /**
   * Get/set the value of the NumericTextBox. 
   * @param {string | number} value?. The value to be set. If no parameter is passed, returns the displayed value of the SmartNumericTextBox. 
   * @param {boolean} suppressValidation?. If <em>true</em> is passed, the passed value will be set to the SmartNumericTextBox without validation. 
   * @returns {string}
   */
  val(value?: string | number, suppressValidation?: boolean): string;
}

declare global {
    interface Document {
        createElement(tagName: "smart-numeric-text-box"): NumericTextBox;
        querySelector(selectors: "smart-numeric-text-box"): NumericTextBox | null;
        querySelectorAll(selectors: "smart-numeric-text-box"): NodeListOf<NumericTextBox>;
        getElementsByTagName(qualifiedName: "smart-numeric-text-box"): HTMLCollectionOf<NumericTextBox>;
        getElementsByName(elementName: "smart-numeric-text-box"): NodeListOf<NumericTextBox>;
    }
}

/**Sets or gets the input format of the widget. Setting this property dynamically can lead to precision loss.  */
export declare type NumericTextBoxInputFormat = 'integer' | 'floatingPoint' | 'complex';
/**Sets or gets the radix of the jqxNumericTextBox. The radix specifies the numeral system in which to display the widget's value. Applicable only when inputFormat is 'integer'.  */
export declare type NumericTextBoxRadix = '2' | '8' | '10' | '16' | 'binary' | 'octal' | 'decimal' | 'hexadecimal';
/**Sets or gets the position of the spin buttons of the jqxNumericTextBox.  */
export declare type NumericTextBoxDisplayPosition = 'left' | 'right';
export interface PagerProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Handles pager's elipsis. Ellipsis buttons are displayed as indicators and additional help to navigate between pages.
   * Default value: none
   */
  autoEllipsis?: PagerAutoEllipsis | string;
  /**
   * Enables or disables the pager.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback, related to localization module. 
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "firstButton": "First",
   *     "lastButton": "Last",
   *     "previousButton": "Previous",
   *     "nextButton": "Next",
   *     "navigateToLabel": "Go to:",
   *     "pageSizeLabel": "Show:",
   *     "navigateToInputPlaceholder": "",
   *     "ellipsis": "...",
   *     "summaryString": "of",
   *     "summaryPrefix": "of",
   *     "summarySuffix": ""
   *   }
   * }
   */
  messages?: any;
  /**
   * Handles the position of the navigation buttons.
   * Default value: near
   */
  navigationButtonsPosition?: LayoutPosition | string;
  /**
   * Gets/sets current page index.
   * Default value: 0
   */
  pageIndex?: number;
  /**
   * Defines the number of page index selectors.
   * Default value: 0
   */
  pageIndexSelectors?: number;
  /**
   * Gets/sets total number of items displayed on page.
   * Default value: 10
   */
  pageSize?: number;
  /**
   * Defines the data source of the element's page size selector drop down.
   * Default value: 10,25,50
   */
  pageSizeSelectorDataSource?: number[];
  /**
   * The number of pages in the element.
   * Default value: 100
   */
  pagesCount?: number;
  /**
   * If the element is readonly, users cannot interact with it.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Controlls displaying of the 'first' and 'last' navigation buttons.
   * Default value: false
   */
  showFirstLastNavigationButtons?: boolean;
  /**
   * Displays text content in navigation buttons instead default icons.
   * Default value: false
   */
  showNavigationButtonLabels?: boolean;
  /**
   * Determines whether the navigation input is displayed.
   * Default value: false
   */
  showNavigationInput?: boolean;
  /**
   * Determines whether the page index selectors are displayed.
   * Default value: false
   */
  showPageIndexSelectors?: boolean;
  /**
   * Determines whether the page size selector is displayed.
   * Default value: false
   */
  showPageSizeSelector?: boolean;
  /**
   * Controlls displaying of the 'previous' and 'next' navigation buttons.
   * Default value: false
   */
  showPrevNextNavigationButtons?: boolean;
  /**
   * Determines whether the page summary is displayed.
   * Default value: false
   */
  showSummary?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Gets/sets total number of records whose pagination the Pager controls. Useful when the Pager is part of a more complex element or application.
   * Default value: null
   */
  totalRecords?: number;
}
/**
 Pagination component that is used to navigate between a set of results.
*/
export interface Pager extends BaseElement, PagerProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when user selects a new item.
	* @param event. The custom event.    */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when page size is changed.
	* @param event. The custom event.    */
  onPageSizeChanged?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Selects first item.
   */
  first(): void;
  /**
   * Selects last item.
   */
  last(): void;
  /**
   * Navigates to particular item.
   * @param {any} pageIndex. 
   */
  navigateTo(pageIndex: any): void;
  /**
   * Selects next pager item.
   */
  next(): void;
  /**
   * Selects previous pager item.
   */
  prev(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-pager"): Pager;
        querySelector(selectors: "smart-pager"): Pager | null;
        querySelectorAll(selectors: "smart-pager"): NodeListOf<Pager>;
        getElementsByTagName(qualifiedName: "smart-pager"): HTMLCollectionOf<Pager>;
        getElementsByName(elementName: "smart-pager"): NodeListOf<Pager>;
    }
}

/**Handles pager's elipsis. Ellipsis buttons are displayed as indicators and additional help to navigate between pages. */
export declare type PagerAutoEllipsis = 'none' | 'before' | 'after' | 'both';
export interface PasswordInputProperties {
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * Determines the minimum number of characters inside the input in order to trigger the autocomplete functionality that will open the drop down and show the matched items.
   * Default value: 1
   */
  minLength?: number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element.
   * Default value: ""
   */
  value?: string;
}
/**
 PasswordInput specifies a password field where the user can enter data. It is similar to the password text box, but this component does not have additional functionality for tooltips and popups.
*/
export interface PasswordInput extends BaseElement, PasswordInputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the value is changed and the focus moved out of the element.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value.
   *  value - The new value.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered on each key up event of the PasswordInput, if the value is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value before it was changed.
   *  value - The new value.
   */
  onChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-password-input"): PasswordInput;
        querySelector(selectors: "smart-password-input"): PasswordInput | null;
        querySelectorAll(selectors: "smart-password-input"): NodeListOf<PasswordInput>;
        getElementsByTagName(qualifiedName: "smart-password-input"): HTMLCollectionOf<PasswordInput>;
        getElementsByName(elementName: "smart-password-input"): NodeListOf<PasswordInput>;
    }
}

export interface PasswordTextBoxProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Specifies that the element should be focused when the page is loaded.
   * Default value: false
   */
  autoFocus?: boolean;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Specifies the behavior on "Enter" key press. Default mode is "submit".
   * Default value: submit
   */
  enterKeyBehavior?: EnterKeyBehavior | string;
  /**
   * The form that the element is associated with (its "form owner"). The value of the attribute must be the ID of a form element in the same document.
   * Default value: ""
   */
  form?: string;
  /**
   * Sets additional helper text below the element. Appears only when the element is focused.
   * Default value: null
   */
  hint?: any;
  /**
   * Sets label above the element. The label is displayed above the input and it's always visible.
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback used to customize the format of the messages that are returned from the Localization Module.
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets the maximum number of characters that the user can enter.
   * Default value: null
   */
  maxLength?: number;
  /**
   * Sets an object with string values, related to the different states of passwords strength.
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM.",
   *     "invalidNode": ".",
   *     "passwordStrength": "Password strength",
   *     "short": "Short",
   *     "weak": "Weak",
   *     "far": "Far",
   *     "good": "Good",
   *     "strong": "Strong",
   *     "showPassword": "Show password"
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the minimum number of characters that the user can enter.
   * Default value: 2
   */
  minLength?: number;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting HTML forms.
   * Default value: ""
   */
  name?: string;
  /**
   * With this property the user can set a custom callback function that determines the password strength. The returned string from the function determines the how strong the current password is. The string should be one of the following: 'short', 'weak', 'far', 'good', 'strong'.
   * Default value: null
   */
  passwordStrength?: any;
  /**
   * The placeholder text that is displayed when no value is applied to the element. 
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Specifies that the user must fill in a value before submitting a form that contains the element.
   * Default value: false
   */
  required?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Specifies whether the content of the input will be selected on focus.
   * Default value: false
   */
  selectAllOnFocus?: boolean;
  /**
   * Determines whether the password icon is visible.
   * Default value: false
   */
  showPasswordIcon?: boolean;
  /**
   * Determines whether a tooltip which shows the password's strength will be shown.
   * Default value: false
   */
  showPasswordStrength?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Determines whether Tooltip's arrow will be shown or not.
   * Default value: false
   */
  tooltipArrow?: boolean;
  /**
   * Determines the delay before the tooltip appears in miliseconds.
   * Default value: 0
   */
  tooltipDelay?: number;
  /**
   * Determines the position of the tooltip.
   * Default value: top
   */
  tooltipPosition?: PasswordTextBoxTooltipPosition | string;
  /**
   * Sets a custom template for the content of the tooltip.
   * Default value: "null"
   */
  tooltipTemplate?: string;
  /**
   * If true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element. 
   * Default value: ""
   */
  value?: string;
}
/**
 PasswordTextBox lets the user enter a password with the text hidden.
*/
export interface PasswordTextBox extends BaseElement, PasswordTextBoxProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the value of the element is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value of the element before it was changed.
   *  value - The new value of the element.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered on each key up event of the TextBox, if the value is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value before it was changed.
   *  value - The new value.
   */
  onChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Focuses the element.
   */
  focus(): void;
  /**
   * The method is used to reset input to it's initial value. 
   */
  reset(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-password-text-box"): PasswordTextBox;
        querySelector(selectors: "smart-password-text-box"): PasswordTextBox | null;
        querySelectorAll(selectors: "smart-password-text-box"): NodeListOf<PasswordTextBox>;
        getElementsByTagName(qualifiedName: "smart-password-text-box"): HTMLCollectionOf<PasswordTextBox>;
        getElementsByName(elementName: "smart-password-text-box"): NodeListOf<PasswordTextBox>;
    }
}

/**Determines the position of the tooltip. */
export declare type PasswordTextBoxTooltipPosition = 'absolute' | 'bottom' | 'top' | 'left' | 'right';
export interface PathProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Determines the data source for the item that will be displayed inside the drop down.
   * Default value: null
   */
  dataSource?: any;
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the displayMember. The displayMember specifies the name of an object property to display. The name is contained in the collection specified by the 'dataSource' property.
   * Default value: """"
   */
  displayMember?: string;
  /**
   * Sets the parent container of the dropDown (the popup). Used when a CSS property of unknown parent is interfering with the visibility of the dropDown.
   * Default value: null
   */
  dropDownAppendTo?: any;
  /**
   * Sets the height of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the max height of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownMaxHeight?: string | number;
  /**
   * Sets the max width of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownMaxWidth?: string | number;
  /**
   * Sets the min height of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownMinHeight?: string | number;
  /**
   * Sets the min width of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownMinWidth?: string | number;
  /**
   * If this property is enabled, when the element's dropdown is opened, a transparent overlay is positioned between the dropdown and the rest of the document.
   * Default value: false
   */
  dropDownOverlay?: boolean;
  /**
   * Determines the vertical position of the dropDown. 'Auto' means its automatically determined depending on the viewport size.
   * Default value: auto
   */
  dropDownPosition?: DropDownPosition | string;
  /**
   * Sets the width of the drop down. Default value of empty string means that CSS variables are used. This property should be used when the browser doesn't support CSS variables.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Sets additional helper text below the element. 
   * Default value: ""
   */
  hint?: string;
  /**
   * Sets the element as an indicator.
   * Default value: false
   */
  indicator?: boolean;
  /**
   * A getter that returns an array of all Path items.
   * Default value: 
   */
  items?: any[];
  /**
   * Sets label above the element. 
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "invalidNode": "."
   *   }
   * }
   */
  messages?: any;
  /**
   * The name of the control.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the popup is opened or closed
   * Default value: false
   */
  opened?: boolean;
  /**
   * Determines the element's placeholder.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Determines the format of the path. Follows specific operation system criteria by changing the drive,folder separators. 
   * Default value: windows
   */
  pathFormat?: PathFormat | string;
  /**
   * Disables user interaction with the element.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * The value of the Path control.
   * Default value: ""
   */
  value?: string;
  /**
   * Determines the value member of an item. Stored as value in the item object.
   * Default value: """"
   */
  valueMember?: string;
  /**
   * Determines whether or not the element wraps to a new line if overflows. If set the Path can be wrapped on multiple lines. 
   * Default value: false
   */
  wrap?: boolean;
}
/**
 Path component is used to display the path to url.
*/
export interface Path extends BaseElement, PathProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when user clicks on the browse button.
	* @param event. The custom event.    */
  onBrowseButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the value is changed.
	* @param event. The custom event.    */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the drop down is closed.
	* @param event. The custom event.    */
  onClose: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when the drop down is closing.
	* @param event. The custom event.    */
  onClosing?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when user clicks on the drop down button.
	* @param event. The custom event.    */
  onDropDownButtonClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when an item from the popup is clicked.
	* @param event. The custom event.    */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down is opened.
	* @param event. The custom event.    */
  onOpen?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the drop down is opening.
	* @param event. The custom event.    */
  onOpening?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Closes the dropDown.
   */
  close(): void;
  /**
   * Opens the dropDown.
   */
  open(): void;
  /**
   * Set's the Path element to 'emptyPath' state and changes the value to '////'.
   */
  setToEmptyPath(): void;
  /**
   * Set's the Path element to 'notAPath' state and changes the value to '//'.
   */
  setToNotAPath(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-path"): Path;
        querySelector(selectors: "smart-path"): Path | null;
        querySelectorAll(selectors: "smart-path"): NodeListOf<Path>;
        getElementsByTagName(qualifiedName: "smart-path"): HTMLCollectionOf<Path>;
        getElementsByName(elementName: "smart-path"): NodeListOf<Path>;
    }
}

/**Determines the format of the path. Follows specific operation system criteria by changing the drive,folder separators.  */
export declare type PathFormat = 'windows' | 'unix';
export interface PhoneInputProperties {
  /**
   * Enables or disables the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets additional class names to the Input drop down.
   * Default value: 
   */
  dropDownClassList?: any;
  /**
   * Determines the position of the drop down button.
   * Default value: none
   */
  dropDownButtonPosition?: DropDownButtonPosition | string;
  /**
   * Sets the height of the drop down. By default it's set to an empty string. In this case the height of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownHeight?: string | number;
  /**
   * Sets the width of the drop down. By default it's set to an empty string. In this case the width of the drop down is controlled by a CSS variable.
   * Default value: 
   */
  dropDownWidth?: string | number;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * [object Object]
   */
  messages?: any;
  /**
   * Sets or gets the name attribute for the element. Name is used when submiting data inside an HTML form.
   * Default value: ""
   */
  name?: string;
  /**
   * Determines whether the input will be in international or national mode i.e whether the input will start with '+'.
   * Default value: false
   */
  nationalMode?: boolean;
  /**
   * Determines whether the drop down is opened or not.
   * Default value: false
   */
  opened?: boolean;
  /**
   * Sets or gets an array of country codes which will be used instead of the default one with all countries. The country code should be ISO 3166-1 alpha-2 codes(https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2).
   * Default value: []
   */
  onlyCountries?: any;
  /**
   * Determines the placeholder of the input.
   * Default value: ""
   */
  placeholder?: string;
  /**
   * Sets or gets the selected country of the element. The country code should be ISO 3166-1 alpha-2 codes(https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2).
   * Default value: ""
   */
  selectedCountry?: string;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Determines the theme for the element. Themes define the look of the elements.
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the value of the element.
   * Default value: ""
   */
  value?: string;
}
/**
 The Phone Input specifies an input field where the user can enter and validate a phone number.
*/
export interface PhoneInput extends BaseElement, PhoneInputProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(label, oldLabel, oldValue, value)
   *  label - The label of the new selected item.
   *  oldLabel - The label of the item that was previously selected before the event was triggered.
   *  oldValue - The value of the item that was previously selected before the event was triggered.
   *  value - The value of the new selected item.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered on each key up event of the Input, if the value is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(oldValue, value)
   *  oldValue - The previous value before it was changed.
   *  value - The new value.
   */
  onChanging?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the user clicks on an item from the popup list.
	* @param event. The custom event. Custom data event was created with: ev.detail(item, label, value)
   *  item - The item that was clicked.
   *  label - The label of the item that was clicked.
   *  value - The value of the item that was clicked.
   */
  onItemClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Closes the drop down.
   */
  close(): void;
  /**
   * Ensures that the active ( selected ) item is always visible.
   */
  ensureVisible(): void;
  /**
   * Returns the entered phone number with formatting.
   * @param {boolean} isInternational?. When you use 'false', the national phone number will be returned and the international phone number, when you use 'true' as parameter.
   * @returns {string}
   */
  getNumber(isInternational?: boolean): string;
  /**
   * Returns an item by its country dial code. The item is an object with 'label', 'value', 'iso2' and 'dialCode' properties.
   * @param {string} dialCode?. Returns the national or international phone number
   * @returns {any}
   */
  getItemByDialCode(dialCode?: string): any;
  /**
   * Returns the selected item. The item is an object with 'label', 'value', 'iso2' and 'dialCode' properties.
   * @returns {any}
   */
  getSelectedItem(): any;
  /**
   * Returns true or false depending on whether the entered phone number is valid.
   * @returns {boolean}
   */
  isValidNumber(): boolean;
  /**
   * Validates the entered phone number.
   */
  validate(): void;
  /**
   * Opens the drop down.
   */
  open(): void;
  /**
   * Selects the text inside the input or if it is <b>readonly</b> then the element is focused.
   */
  select(): void;
}

declare global {
    interface Document {
        createElement(tagName: "smart-phone-input"): PhoneInput;
        querySelector(selectors: "smart-phone-input"): PhoneInput | null;
        querySelectorAll(selectors: "smart-phone-input"): NodeListOf<PhoneInput>;
        getElementsByTagName(qualifiedName: "smart-phone-input"): HTMLCollectionOf<PhoneInput>;
        getElementsByName(elementName: "smart-phone-input"): NodeListOf<PhoneInput>;
    }
}

export interface PivotTableProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Sets or gets whether the reordering of columns is enabled.
   * Default value: false
   */
  columnReorder?: boolean;
  /**
   * Describes the columns of the PivotTable's original tabular data. Based on these settings and the data source, the actual columns of the PivotTable are dynamically generated.
   * Default value: null
   */
  columns?: PivotTableColumn[];
  /**
   * Sets or gets whether to show total columns for each pivot data point. When enabled, all summary columns must have the same summary function set by which total columns are calculated.
   * Default value: false
   */
  columnTotals?: boolean;
  /**
   * Sets or gets the position of total columns (shown when columnTotals is enabled).
   * Default value: near
   */
  columnTotalsPosition?: PivotTableColumnTotalsPosition | string;
  /**
   * Sets or gets details about conditional formatting to be applied to the PivotTable's cells.
   * Default value: null
   */
  conditionalFormatting?: PivotTableConditionalFormatting[];
  /**
   * Determines the original tabular data source of the PivotTable.
   * Default value: 
   */
  dataSource?: any;
  /**
   * Sets or gets whether the original tabular data sourse of the PivotTable will be pre-sorted based on columns with the rowGroup property (and their order).
   * Default value: false
   */
  defaultSortByRowGroups?: boolean;
  /**
   * Sets or gets whether to display the PivotTable's designer alongside the table itself. The designer allows for configuring column settings and applying filtering.
   * Default value: false
   */
  designer?: boolean;
  /**
   * Sets or gets the position of the PivotTable's designer (shown when designer is enabled).
   * Default value: far
   */
  designerPosition?: PivotTableDesignerPosition | string;
  /**
   * Disables the interaction with the element.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * If enabled, shows the original tabular data that has been aggregated in a PivotTable summary cell when the cell is double-clicked or F2 is pressed.
   * Default value: false
   */
  drillDown?: boolean;
  /**
   * If set, shows an export button in the drill down dialog.
   * Default value: 
   */
  drillDownDataExport?: PivotTableDrillDownDataExport | string;
  /**
   * Sets or gets the drill down table export file name.
   * Default value: ""
   */
  drillDownDataExportName?: string;
  /**
   * Sets or gets the drill down dialog callback function. The argument of the callback passed by the PivotTable is the drill-down Table component. You can use it to customize the table.
   * Default value: null
   */
  drillDownTableInit?: { (table: HTMLElement ): void };
  /**
   * Sets or gets the drill down custom action callback function. The argument of the callback passed by the PivotTable is the drill-down data source. You can use it to override the default drill-down UI i.e to replace our Dialog with Table.
   * Default value: null
   */
  drillDownCustomAction?: { (originalRecords: [] ): void };
  /**
   * Sets or gets whether sorting based on columns in classic row groups layout mode is enabled.
   * Default value: false
   */
  enableSortByRowGroups?: boolean;
  /**
   * Sets or gets whether the PivotTable's column header is sticky/frozen.
   * Default value: false
   */
  freezeHeader?: boolean;
  /**
   * A callback function that returns the default summary function of a summary column when it is dynamically assigned as such (e.g. by drag-drop in the designer).
   * Default value: null
   */
  getDefaultSummaryFunction?: { (column: PivotTableColumn): string };
  /**
   * Sets or gets whether to show a Grand total row aggregating the data of all rows.
   * Default value: false
   */
  grandTotal?: boolean;
  /**
   * Sets or gets the way row nesting (based on rowGroup columns) is displayed.
   * Default value: default
   */
  groupLayout?: PivotTableGroupLayout | string;
  /**
   * Sets or gets whether to hide the tooltip that displays details when multiple summary cells with non-null values are selected.
   * Default value: false
   */
  hideCellSelectionTooltip?: boolean;
  /**
   * Sets or gets whether to hide rows that contain only 0 or null values. Applicable only when there are rowGroup columns.
   * Default value: false
   */
  hideEmptyRows?: boolean;
  /**
   * Sets or gets whether navigation with the keyboard is enabled in the PivotTable.
   * Default value: false
   */
  keyboardNavigation?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Sets or gets an object specifying strings used in the element that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {"en":{"add":"Add condition","all":"All columns","apply":"Apply","average":"Average","between":"Between","calculation":"Calculation","cancel":"Cancel","center":"center","clear":"Clear","clearFilter":"Clear filter","close":"Close","column":"Column:","columns":"Columns","columnSettings":"Column settings","condition":"Condition:","conditionalFormatting":"Conditional Formatting","CONTAINS_CASE_SENSITIVE":"contains (case sensitive)","CONTAINS":"contains","count":"Count","decimalPlaces":"Decimal places","decimalSeparator":"Decimal separator","details":"Details","DOES_NOT_CONTAIN_CASE_SENSITIVE":"does not contain (case sensitive)","DOES_NOT_CONTAIN":"does not contain","dragHerePivots":"Drag here to set pivots","dragHereRowGroups":"Drag here to set row groups","dragHereSummaries":"Drag here to set summaries","EMPTY":"empty","ENDS_WITH_CASE_SENSITIVE":"ends with (case sensitive)","ENDS_WITH":"ends with","EQUAL_CASE_SENSITIVE":"equal (case sensitive)","equal":"Equal To","EQUAL":"equal","fields":"Fields","filter":"Filter","filterCondition":"Filter condition","filterPlaceholder":"Filter","filters":"Filters","firstButton":"First","fontFamily":"Font family:","fontSize":"Font size:","format":"Format:","formatColumn":"Format Column","grandTotal":"Grand Total","GREATER_THAN_OR_EQUAL":"greater than or equal","GREATER_THAN":"greater than","greaterThan":"Greater Than","groupHeader":"Group","highlight":"Highlight","invalidValue":"Invalid value","itemsPerPage":"Items per page:","lastButton":"Last","left":"left","LESS_THAN_OR_EQUAL":"less than or equal","LESS_THAN":"less than","lessThan":"Less Than","moveTo":"Move to","negativesInBrackets":"Negatives in brackets","nextButton":"Next","NOT_EMPTY":"not empty","NOT_EQUAL":"not equal","NOT_NULL":"not null","notApplicable":"N/A","notEqual":"Not Equal To","NULL":"null","numberAlignment":"Number alignment","numberFormat":"Number format","numberPrefix":"Number prefix","ok":"OK","pivots":"Pivots","previousButton":"Previous","remove":"Remove condition","right":"right","row":"Row","rowGroups":"Row Groups","sameSummaryFunctionRequired":"jqxPivotTable: When "columnTotals" is enabled, all summary columns must have the same "summary" function set (e.g. '
   */
  messages?: any;
  /**
   * Sets or gets what value is shown in cells that do not have aggregated data to display. By default (null), such cells are empty.
   * Default value: null
   */
  nullDefaultValue?: number;
  /**
   * A callback function executed each time a PivotTable cell is rendered.
   * Default value: null
   */
  onCellRender?: { (data: any, dynamicColumn: any, value: any, cell: HTMLTableCellElement): void };
  /**
   * A callback function executed each time a PivotTable column header cell is rendered.
   * Default value: null
   */
  onColumnRender?: { (settings: { text: string, cell: HTMLTableCellElement, column: PivotTableColumn, fullDefinition: any }): void };
  /**
   * A callback function executed when the PivotTable is being initialized.
   * Default value: null
   */
  onInit?: { (): void };
  /**
   * Sets or gets the page size (when paging is enabled).
   * Default value: 10
   */
  pageSize?: PivotTablePageSize | string;
  /**
   * Sets or gets the current (zero-based) page index (when paging is enabled).
   * Default value: 0
   */
  pageIndex?: number;
  /**
   * Sets or gets whether paging is enabled.
   * Default value: false
   */
  paging?: boolean;
  /**
   * Sets or gets the value indicating whether the element is aligned to support locales using right-to-left fonts.
   * Default value: false
   */
  rightToLeft?: boolean;
  /**
   * Sets or gets whether sorting by row (when a row group cell is clicked) is enabled. When columnTotals is also enabled, sorting is applied per "column group"; otherwise - for all columns.
   * Default value: false
   */
  rowSort?: boolean;
  /**
   * Sets or gets whether row summaries are displayed in the row headers. Example: Peterson(40) vs Peterson, when rowSummary is set to false.
   * Default value: true
   */
  rowSummary?: boolean;
  /**
   * Sets or gets whether to show row total columns for each summary column.
   * Default value: false
   */
  rowTotals?: boolean;
  /**
   * Sets or gets the position of row total columns (shown when rowTotals is enabled).
   * Default value: near
   */
  rowTotalsPosition?: PivotTableRowTotalsPosition | string;
  /**
   * Sets or gets whether row selection (via checkboxes) is enabled.
   * Default value: false
   */
  selection?: boolean;
  /**
   * Sets or gets the selection mode. Only applicable when selection is enabled.
   * Default value: many
   */
  selectionMode?: PivotTableSelectionMode | string;
  /**
   * Determines the sorting mode of the PivotTable.
   * Default value: none
   */
  sortMode?: PivotTableSortMode | string;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * Sets or gets whether the PivotTable's toolbar is shown. It contains two breadcrumb components that allow the modification of the row group and pivot columns, as well as the "Conditional Formatting" and "Fields" buttons that open a dialog with additional settings.
   * Default value: false
   */
  toolbar?: boolean;
  /**
   * Sets or gets whether when hovering a cell with truncated content, a tooltip with the full content will be shown.
   * Default value: false
   */
  tooltip?: boolean;
}
/**
 PivotTable is a table of statistics that summarizes the data of a more extensive table.
*/
export interface PivotTable extends BaseElement, PivotTableProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when a cell has been clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(dataField, row)
   *  dataField - The data field of the cell's dynamic column.
   *  row - The data of the cell's row.
   */
  onCellClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when the selection is changed.
	* @param event. The custom event. Custom data event was created with: ev.detail(type)
   *  type - The type of action that initiated the selection change. Possible types: 'programmatic', 'interaction', 'remove'.
   */
  onChange: ((this: any, ev: Event) => any) | null;
  /**
   * This event is triggered when a summary column header cell has been clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(columnDefinition, dataField)
   *  columnDefinition - An object detailing the clicked dynamic column.
   *  dataField - The data field of the cell's original column.
   */
  onColumnClick?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a row has been collapsed.
	* @param event. The custom event. Custom data event was created with: ev.detail(record)
   *  record - The (aggregated) data of the collapsed row.
   */
  onCollapse?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a total column has been collapsed.
	* @param event. The custom event. Custom data event was created with: ev.detail(columnDefinition)
   *  columnDefinition - The definition of the collapsed total column.
   */
  onCollapseTotalColumn?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a row has been expanded.
	* @param event. The custom event. Custom data event was created with: ev.detail(record)
   *  record - The (aggregated) data of the expanded row.
   */
  onExpand?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a total column has been expanded.
	* @param event. The custom event. Custom data event was created with: ev.detail(columnDefinition)
   *  columnDefinition - The definition of the expanded total column.
   */
  onExpandTotalColumn?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a filtering-related action is made.
	* @param event. The custom event. Custom data event was created with: ev.detail(action, filters)
   *  action - The filtering action. Possible actions: 'add', 'remove'.
   *  filters - The added filters. Only when action is 'add'.
   */
  onFilter?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * This event is triggered when a column header cell has been clicked.
	* @param event. The custom event. Custom data event was created with: ev.detail(columns)
   *  columns - An array with information about the dynamic columns the PivotTable has been sorted by.
   */
  onSort?: ((this: any, ev: Event) => any) | ((this: any, ev: CustomEvent<any>) => any) | null;
  /**
   * Adds a filter to a specific column.
   * @param {string} dataField. The column's data field.
   * @param {any} filter. FilterGroup object.
   */
  addFilter(dataField: string, filter: any): void;
  /**
   * Clears applied filters.
   */
  clearFilters(): void;
  /**
   * Clears selection.
   */
  clearSelection(): void;
  /**
   * Clears the PivotTable sorting.
   */
  clearSort(): void;
  /**
   * Collapses all rows (when multiple row groups are applied).
   */
  collapseAllRows(): void;
  /**
   * Collapses a row (when multiple row groups are applied).
   * @param {string | number} rowId. The id of the row to collapse. Can be retrieved from the <strong>rows</strong> collection.
   */
  collapseRow(rowId: string | number): void;
  /**
   * Expands all rows (when multiple row groups are applied).
   */
  expandAllRows(): void;
  /**
   * Expands a row (when multiple row groups are applied).
   * @param {string | number} rowId. The id of the row to expand. Can be retrieved from the <strong>rows</strong> collection.
   */
  expandRow(rowId: string | number): void;
  /**
   * Exports the PivotTable's data.
   * @param {string} dataFormat. The file format to export to. Supported formats: 'csv', 'html', 'json', 'pdf', 'tsv', 'xlsx', 'xml'.
   * @param {string} fileName. The name of the file to export to
   * @param {Function} callback?. A callback function to pass the exported data to (if fileName is not provided)
   * @returns {any}
   */
  exportData(dataFormat: string, fileName: string, callback?: Function): any;
  /**
   * Returns the current dynamic pivot columns.
   * @returns {any}
   */
  getDynamicColumns(): any;
  /**
   * Returns an array of selected row ids (when <strong>selectionMode</strong> is <em>'many'</em> or <em>'extended'</em>) or an array of selected cell details (when <strong>selectionMode</strong> is <em>'cell'</em>).
   * @returns {(string | number)[] | { dataField: string, rowId: string | number }[]}
   */
  getSelection(): (string | number)[] | { dataField: string, rowId: string | number }[];
  /**
   * Refreshes the PivotTable.
   */
  refresh(): void;
  /**
   * Removes filters applied to a specific column.
   * @param {string} dataField. The column's data field.
   */
  removeFilter(dataField: string): void;
  /**
   * Selects one or more rows (when <strong>selectionMode</strong> is <em>'many'</em> or <em>'extended'</em>) or a single cell (when <strong>selectionMode</strong> is <em>'cell'</em> and the second argument is passed).
   * @param {string | number | (string | number)[]} rowId. The id of the row (or an array of row ids) to select (or of the cell's parent row when <strong>selectionMode</strong> is <em>'cell'</em>). Can be retrieved from the <strong>rows</strong> collection.
   * @param {string} dataField?. The dataField of the dynamic column (can be retrieved by calling <strong>getDynamicColumns</strong>) of the cell to select (only applicable when <strong>selectionMode</strong> is <em>'cell'</em>).
   */
  select(rowId: string | number | (string | number)[], dataField?: string): void;
  /**
   * Sorts by a summary or group column.
   * @param {any} columnDefinition. The dynamic column's definition. Can be retrieved from the method <strong>getDynamicColumns</strong>.
   * @param {string} sortOrder?. Sort order. Possible values: 'asc' (ascending), 'desc' (descending), and null (removes sorting by column). If not provided, toggles the sorting.
   */
  sortBy(columnDefinition: any, sortOrder?: string): void;
  /**
   * Unselects one or more rows (when <strong>selectionMode</strong> is <em>'many'</em> or <em>'extended'</em>) or a single cell (when <strong>selectionMode</strong> is <em>'cell'</em> and the second argument is passed).
   * @param {string | number | (string | number)[]} rowId. The id of the row (or an array of row ids) to select (or of the cell's parent row when <strong>selectionMode</strong> is <em>'cell'</em>). Can be retrieved from the <strong>rows</strong> collection.
   * @param {string} dataField?. The dataField of the dynamic column (can be retrieved by calling <strong>getDynamicColumns</strong>) of the cell to select (only applicable when <strong>selectionMode</strong> is <em>'cell'</em>).
   */
  unselect(rowId: string | number | (string | number)[], dataField?: string): void;
}

export interface PivotTableColumn {
  /**
   * Sets or gets the header cell alignment for pivot and summary columns and cell alignment for row group columns.
   * Default value: left
   */
  align?: PivotTableColumnAlign | string;
  /**
   * Sets or gets whether the column can be filtered.
   * Default value: true
   */
  allowFilter?: boolean;
  /**
   * Sets or gets whether the column can be a pivot column.
   * Default value: false
   */
  allowPivot?: boolean;
  /**
   * Sets or gets whether the column can be a row group column.
   * Default value: false
   */
  allowRowGroup?: boolean;
  /**
   * Sets or gets whether summary columns based on the column can be sorted.
   * Default value: true
   */
  allowSort?: boolean;
  /**
   * Sets or gets the column's data source-bound field.
   * Default value: ""
   */
  dataField?: string;
  /**
   * Sets or gets the data type of the column's cells.
   * Default value: string
   */
  dataType?: PivotTableColumnDataType | string;
  /**
   * A callback function that can be used to modify the contents of a cell and the cell itself.
   * Default value: null
   */
  formatFunction?: { (settings: { value: any, row: string | number, column: string, cell: HTMLTableCellElement, template?: any }): void };
  /**
   * Sets or gets the column's displayed text (for example in summary column headers).
   * Default value: ""
   */
  label?: string;
  /**
   * Sets or gets whether the column is a pivot column. Data from pivot columns is represented as column hierarchy in the PivotTable.
   * Default value: false
   */
  pivot?: boolean;
  /**
   * Sets or gets whether the column is a row group column. Data from row group columns is represented as rows in the PivotTable. If multiple row groups are set, row hierarchy is displayed based on the order of the row group columns in the columns array.
   * Default value: false
   */
  rowGroup?: boolean;
  /**
   * Sets or gets the summary function to aggregate the column's data by and produce dynamic summary columns for each unique pivot data point. There must always be at least one summary column for the PivotTable to make sense. When columnTotals is enabled, all summary columns must have the same summary function set.
   * Default value: sum
   */
  summary?: PivotTableColumnSummary | string;
  /**
   * Sets or gets an object with settings for cells in summary columns. These settings are not applied if column formatFunction is also implemented.
   * Default value: [object Object]
   */
  summarySettings?: { align: string, prefix: string, decimalPlaces: number, thousandsSeparator: string, decimalSeparator: string, negativesInBrackets: boolean };
}

export interface PivotTableConditionalFormatting {
  /**
   * The data field of a numeric column to format. Set 'all' to format all numeric columns.
   * Default value: "all"
   */
  column?: string;
  /**
   * The formatting condition.
   * Default value: lessThan
   */
  condition?: PivotTableConditionalFormattingCondition | string;
  /**
   * The value to compare by. When condition is 'between', this is the start (from) value.
   * Default value: 0
   */
  firstValue?: number;
  /**
   * The fontFamily to apply to formatted cells.
   * Default value: The default fontFamily as set in CSS
   */
  fontFamily?: PivotTableConditionalFormattingFontFamily | string;
  /**
   * The fontSize to apply to formatted cells. The fontSize as set in CSS is used by default.
   * Default value: 14px
   */
  fontSize?: PivotTableConditionalFormattingFontSize | string;
  /**
   * The background color to apply to formatted cells.
   * Default value: "The default backgroundColor as set in CSS"
   */
  highlight?: string;
  /**
   * When condition is 'between', this is the end (to) value. Otherwise, this value is not used.
   * Default value: 1
   */
  secondValue?: number;
  /**
   * The text color to apply to formatted cells.
   * Default value: "The default color as set in CSS"
   */
  text?: string;
}

declare global {
    interface Document {
        createElement(tagName: "smart-pivot-table"): PivotTable;
        querySelector(selectors: "smart-pivot-table"): PivotTable | null;
        querySelectorAll(selectors: "smart-pivot-table"): NodeListOf<PivotTable>;
        getElementsByTagName(qualifiedName: "smart-pivot-table"): HTMLCollectionOf<PivotTable>;
        getElementsByName(elementName: "smart-pivot-table"): NodeListOf<PivotTable>;
    }
}

/**Sets or gets the header cell alignment for pivot and summary columns and cell alignment for row group columns. */
export declare type PivotTableColumnAlign = 'center' | 'left' | 'right';
/**Sets or gets the data type of the column's cells. */
export declare type PivotTableColumnDataType = 'boolean' | 'date' | 'number' | 'string';
/**Sets or gets the summary function to aggregate the column's data by and produce dynamic summary columns for each unique pivot data point. There must always be at least one summary column for the PivotTable to make sense. When columnTotals is enabled, all summary columns must have the same summary function set. */
export declare type PivotTableColumnSummary = 'avg' | 'count' | 'max' | 'median' | 'min' | 'product' | 'stdev' | 'stdevp' | 'sum' | 'var' | 'varp';
/**Sets or gets the position of total columns (shown when columnTotals is enabled). */
export declare type PivotTableColumnTotalsPosition = 'near' | 'far';
/**The formatting condition. */
export declare type PivotTableConditionalFormattingCondition = 'between' | 'equal' | 'greaterThan' | 'lessThan' | 'notEqual';
/**The fontFamily to apply to formatted cells. */
export declare type PivotTableConditionalFormattingFontFamily = 'The default fontFamily as set in CSS' | 'Arial' | 'Courier New' | 'Georgia' | 'Times New Roman' | 'Verdana';
/**The fontSize to apply to formatted cells. The fontSize as set in CSS is used by default. */
export declare type PivotTableConditionalFormattingFontSize = '8px' | '9px' | '10px' | '11px' | '12px' | '13px' | '14px' | '15px' | '16px';
/**Sets or gets the position of the PivotTable's designer (shown when designer is enabled). */
export declare type PivotTableDesignerPosition = 'near' | 'far';
/**If set, shows an export button in the drill down dialog. */
export declare type PivotTableDrillDownDataExport = null | 'xlsx' | 'pdf' | 'html' | 'json' | 'csv' | 'tsv' | 'xml';
/**Sets or gets the way row nesting (based on rowGroup columns) is displayed. */
export declare type PivotTableGroupLayout = 'classic' | 'default';
/**Sets or gets the page size (when paging is enabled). */
export declare type PivotTablePageSize = '10' | '25' | '50';
/**Sets or gets the position of row total columns (shown when rowTotals is enabled). */
export declare type PivotTableRowTotalsPosition = 'near' | 'far';
/**Sets or gets the selection mode. Only applicable when selection is enabled. */
export declare type PivotTableSelectionMode = 'many' | 'extended' | 'cell';
/**Determines the sorting mode of the PivotTable. */
export declare type PivotTableSortMode = 'none' | 'one' | 'many';
export interface PowerButtonProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Sets or gets the check state.
   * Default value: false
   */
  checked?: boolean;
  /**
   * Sets the click mode of the button.
   * Default value: release
   */
  clickMode?: ClickMode | string;
  /**
   * Enables or disables the power button.
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback, related to localization module. 
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets or gets an object specifying strings used in the widget that can be localized. Used in conjunction with the property locale. 
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any;
  /**
   * Sets or gets the widget's name.
   * Default value: """"
   */
  name?: string;
  /**
   * If the custom element is readonly, it cannot be interacted with.
   * Default value: false
   */
  readonly?: boolean;
  /**
   * Determines the theme. Theme defines the look of the element
   * Default value: ""
   */
  theme?: string;
  /**
   * If is set to true, the element cannot be focused.
   * Default value: false
   */
  unfocusable?: boolean;
  /**
   * Sets or gets the widget's value.
   * Default value: """"
   */
  value?: string;
}
/**
 PowerButton is On/Off rounded button.
*/
export interface PowerButton extends BaseElement, PowerButtonProperties {

  /* Get a member by its name */
  [name: string]: any;
  /**
   * This event is triggered when the widget is checked/unchecked.
	* @param event. The custom event.    */
  onChange: ((this: any, ev: Event) => any) | null;
}

declare global {
    interface Document {
        createElement(tagName: "smart-power-button"): PowerButton;
        querySelector(selectors: "smart-power-button"): PowerButton | null;
        querySelectorAll(selectors: "smart-power-button"): NodeListOf<PowerButton>;
        getElementsByTagName(qualifiedName: "smart-power-button"): HTMLCollectionOf<PowerButton>;
        getElementsByName(elementName: "smart-power-button"): NodeListOf<PowerButton>;
    }
}

export interface ProgressBarProperties {
  /**
   * Sets or gets the animation mode. Animation is disabled when the property is set to 'none'
   * Default value: advanced
   */
  animation?: Animation | string;
  /**
   * Enables or disables the element. 
   * Default value: false
   */
  disabled?: boolean;
  /**
   * Sets the value of the Progress bar to indeterminate state(null) and starts the animation.
   * Default value: false
   */
  indeterminate?: boolean;
  /**
   * Sets the filling direction of the Progress Bar.
   * Default value: false
   */
  inverted?: boolean;
  /**
   * A callback function defining the new format for the label of the Progress Bar.
   * Default value: null
   */
  formatFunction?: {(value: number): string};
  /**
   * Sets or gets the language. Used in conjunction with the property messages. 
   * Default value: "en"
   */
  locale?: string;
  /**
   * Callback, related to localization module. 
   * Default value: null
   */
  localizeFormatFunction?: any;
  /**
   * Sets progress bars maximum possible value.
   * Default value: 100
   */
  max?: number;
  /**
   * Sets an object with string values, related to the different states of passwords strength.
   * Default value:    * {
   *   "en": {
   *     "propertyUnknownType": "'' property is with undefined 'type' member!",
   *     "propertyInvalidValue": "Invalid '!",
   *     "propertyInvalidValueType": "Invalid '!",
   *     "elementNotInDOM": "Element does not exist in DOM! Please, add the element to the DOM, before invoking a method.",
   *     "moduleUndefined": "Module is undefined.",
   *     "missingReference": ".",
   *     "htmlTemplateNotSuported": ": Browser doesn't support HTMLTemplate elements.",
   *     "invalidTemplate": "' property accepts a string that must match the id of an HTMLTemplate element from the DOM."
   *   }
   * }
   */
  messages?: any