import {
  scrollIntoView
} from "./chunk.WDWV53JP.js";
import {
  LocalizeController
} from "./chunk.YKM42GKF.js";
import {
  SolidElement,
  customElement
} from "./chunk.HFHUY7YD.js";
import {
  __decorateClass,
  __spreadValues
} from "./chunk.S3NI7NKU.js";

// src/components/tab-group/tab-group.ts
import { css, html } from "lit";
import { property, query, state } from "lit/decorators.js";
import cx from "classix";
var SdTabGroup = class extends SolidElement {
  constructor() {
    super(...arguments);
    this.localize = new LocalizeController(this);
    this.tabs = [];
    this.panels = [];
    this.hasScrollControls = false;
    this.variant = "default";
    this.activation = "auto";
  }
  connectedCallback() {
    const whenAllDefined = Promise.all([
      customElements.whenDefined("sd-tab"),
      customElements.whenDefined("sd-tab-panel")
    ]);
    super.connectedCallback();
    this.resizeObserver = new ResizeObserver(() => {
      this.updateScrollControls();
    });
    this.mutationObserver = new MutationObserver((mutations) => {
      if (mutations.some((m) => !["aria-labelledby", "aria-controls"].includes(m.attributeName))) {
        setTimeout(() => this.setAriaLabels());
      }
      if (mutations.some((m) => m.attributeName === "disabled")) {
        this.syncTabsAndPanels();
      }
    });
    this.updateComplete.then(() => {
      this.syncTabsAndPanels();
      this.mutationObserver.observe(this, { attributes: true, childList: true, subtree: true });
      this.resizeObserver.observe(this.nav);
      whenAllDefined.then(() => {
        const intersectionObserver = new IntersectionObserver((entries, observer) => {
          var _a;
          if (entries[0].intersectionRatio > 0) {
            this.setAriaLabels();
            this.setActiveTab((_a = this.activeTab) != null ? _a : this.tabs[0], { emitEvents: false });
            observer.unobserve(entries[0].target);
          }
        });
        intersectionObserver.observe(this.tabGroup);
      });
    });
  }
  disconnectedCallback() {
    var _a;
    this.mutationObserver.disconnect();
    if (this.nav) {
      (_a = this.resizeObserver) == null ? void 0 : _a.unobserve(this.nav);
    }
  }
  getAllTabs(options = { includeDisabled: true }) {
    const slot = this.shadowRoot.querySelector('slot[name="nav"]');
    return [...slot.assignedElements()].filter((el) => {
      return options.includeDisabled ? el.tagName.toLowerCase() === "sd-tab" : el.tagName.toLowerCase() === "sd-tab" && !el.disabled;
    });
  }
  getAllPanels() {
    return [...this.body.assignedElements()].filter((el) => el.tagName.toLowerCase() === "sd-tab-panel");
  }
  getActiveTab() {
    return this.tabs.find((t) => t.matches(":focus"));
  }
  handleClick(event) {
    const target = event.target;
    const tab = target.closest("sd-tab");
    const tabGroup = tab == null ? void 0 : tab.closest("sd-tab-group");
    if (tabGroup !== this) {
      return;
    }
    if (tab !== null) {
      this.setActiveTab(tab, { scrollBehavior: "smooth" });
    }
  }
  handleKeyDown(event) {
    const target = event.target;
    const tab = target.closest("sd-tab");
    const tabGroup = tab == null ? void 0 : tab.closest("sd-tab-group");
    if (tabGroup !== this) {
      return;
    }
    if (["Enter", " "].includes(event.key)) {
      if (tab !== null) {
        this.setActiveTab(tab, { scrollBehavior: "smooth" });
        event.preventDefault();
      }
    }
    if (["Tab"].includes(event.key)) {
      const index = this.tabs.indexOf(this.getActiveTab());
      if (tab !== null) {
        scrollIntoView(this.tabs[index + 1], this.nav, "horizontal");
      }
    }
    if (["Shift", "Tab"].includes(event.key)) {
      const index = this.tabs.indexOf(this.getActiveTab());
      if (tab !== null) {
        scrollIntoView(this.tabs[index - 1], this.nav, "horizontal");
      }
    }
    if (["ArrowLeft", "ArrowRight", "ArrowUp", "ArrowDown", "Home", "End"].includes(event.key)) {
      const activeEl = this.tabs.find((t) => t.matches(":focus"));
      const isRtl = this.localize.dir() === "rtl";
      if ((activeEl == null ? void 0 : activeEl.tagName.toLowerCase()) === "sd-tab") {
        let index = this.tabs.indexOf(activeEl);
        if (event.key === "Home") {
          index = 0;
        } else if (event.key === "End") {
          index = this.tabs.length - 1;
        } else if (event.key === (isRtl ? "ArrowRight" : "ArrowLeft") || event.key === "ArrowUp") {
          index--;
        } else if (event.key === (isRtl ? "ArrowLeft" : "ArrowRight") || event.key === "ArrowDown") {
          index++;
        }
        if (index < 0) {
          index = this.tabs.length - 1;
        }
        if (index > this.tabs.length - 1) {
          index = 0;
        }
        this.tabs[index].focus({ preventScroll: true });
        if (this.activation === "auto") {
          this.setActiveTab(this.tabs[index], { scrollBehavior: "smooth" });
        }
        scrollIntoView(this.tabs[index], this.nav, "horizontal");
        event.preventDefault();
      }
    }
  }
  handleScrollToStart() {
    this.nav.scroll({
      left: this.localize.dir() === "rtl" ? this.nav.scrollLeft + this.nav.clientWidth : this.nav.scrollLeft - this.nav.clientWidth,
      behavior: "smooth"
    });
  }
  handleScrollToEnd() {
    this.nav.scroll({
      left: this.localize.dir() === "rtl" ? this.nav.scrollLeft - this.nav.clientWidth : this.nav.scrollLeft + this.nav.clientWidth,
      behavior: "smooth"
    });
  }
  /** Sets the active tab and panel. */
  setActiveTab(tab, options) {
    options = __spreadValues({
      emitEvents: true,
      scrollBehavior: "auto"
    }, options);
    if (tab !== this.activeTab && !tab.disabled) {
      const previousTab = this.activeTab;
      this.activeTab = tab;
      this.tabs.map((el) => el.active = el === this.activeTab);
      this.panels.map((el) => {
        var _a;
        return el.active = el.name === ((_a = this.activeTab) == null ? void 0 : _a.panel);
      });
      scrollIntoView(this.activeTab, this.nav, "horizontal", options.scrollBehavior);
      if (options.emitEvents) {
        if (previousTab) {
          this.emit("sd-tab-hide", { detail: { name: previousTab.panel } });
        }
        this.emit("sd-tab-show", { detail: { name: this.activeTab.panel } });
      }
    }
  }
  setAriaLabels() {
    this.tabs.forEach((tab) => {
      const panel = this.panels.find((el) => el.name === tab.panel);
      if (panel) {
        tab.setAttribute("aria-controls", panel.getAttribute("id"));
      }
    });
  }
  // This stores tabs and panels so we can refer to a cache instead of calling querySelectorAll() multiple times.
  syncTabsAndPanels() {
    this.tabs = this.getAllTabs({ includeDisabled: false });
    this.panels = this.getAllPanels();
    this.panels.forEach((panel) => {
      panel.tabIndex = 0;
    });
    this.updateComplete.then(() => this.updateScrollControls());
    if (this.tabs.length !== 0 && this.tabs[0].variant === "container") {
      this.variant = "container";
    }
  }
  updateScrollControls() {
    this.hasScrollControls = this.nav.scrollWidth > this.nav.clientWidth;
  }
  /** Shows the specified tab panel. */
  show(panel) {
    const tab = this.tabs.find((el) => el.panel === panel);
    if (tab) {
      this.setActiveTab(tab, { scrollBehavior: "smooth" });
    }
  }
  render() {
    const isRtl = this.localize.dir() === "rtl";
    return html`<div part="base" class="${cx("flex flex-col rounded-none")}" @click="${this.handleClick}" @keydown="${this.handleKeyDown}"><div part="nav" class="${cx(this.hasScrollControls && "relative py-0 px-12")}">${this.hasScrollControls ? html`<button part="scroll-button--start" exportparts="base:scroll-button__base" tabindex="-1" aria-hidden="true" class="${cx(
      "sd-interactive flex items-center justify-center absolute top-0 bottom-0 left-0 !outline-offset-0 border-b border-neutral-400 z-10",
      this.localize.dir() === "rtl" && "left-auto right-0"
    )}" @click="${this.handleScrollToStart}"><sd-icon library="system" name="${isRtl ? "chevron-up" : "chevron-down"}" label="previous" class="${cx("h-6 w-12 rotate-90 grid place-items-center")}"></sd-icon></button>` : ""}<div part="scroll-container" class="flex overflow-x-auto focus-visible:focus-outline !outline-offset-0"><div part="tabs" class="${cx("flex flex-auto relative flex-row")}" role="tablist"><div part="separation" class="w-full border-[0.25px] border-neutral-400 absolute bottom-0"></div><slot name="nav" @slotchange="${this.syncTabsAndPanels}"></slot></div></div>${this.hasScrollControls ? html`<button part="scroll-button--end" exportparts="base:scroll-button__base" tabindex="-1" aria-hidden="true" class="${cx(
      "sd-interactive flex items-center justify-center absolute top-0 bottom-0 right-0 !outline-offset-0 border-b border-neutral-400 z-10",
      this.localize.dir() === "rtl" && "right-auto left-0"
    )}" @click="${this.handleScrollToEnd}"><sd-icon library="system" name="${isRtl ? "chevron-down" : "chevron-up"}" label="next" class="${cx("h-6 w-12 rotate-90 grid place-items-center")}"></sd-icon></button>` : ""}</div><slot part="body" class="${cx("block auto py-8 px-6", this.variant === "container" && "border border-neutral-400 border-t-0")}" @slotchange="${this.syncTabsAndPanels}"></slot></div>`;
  }
};
SdTabGroup.styles = [
  ...SolidElement.styles,
  css`:host{box-sizing:border-box;display:block}[part=scroll-container]{scrollbar-width:none}[part=scroll-container]::-webkit-scrollbar{height:var(--sd-spacing-0,0);width:var(--sd-spacing-0,0)}:focus-within::slotted(sd-tab-panel){outline-color:rgb(var(--sd-color-primary,0 53 142)/1);outline-offset:2px;outline-style:solid;outline-width:2px}::slotted(sd-tab-panel){--padding:1rem 0}`
];
__decorateClass([
  query("[part=base]")
], SdTabGroup.prototype, "tabGroup", 2);
__decorateClass([
  query("[part=body]")
], SdTabGroup.prototype, "body", 2);
__decorateClass([
  query("[part=scroll-container]")
], SdTabGroup.prototype, "nav", 2);
__decorateClass([
  state()
], SdTabGroup.prototype, "hasScrollControls", 2);
__decorateClass([
  state()
], SdTabGroup.prototype, "variant", 2);
__decorateClass([
  property()
], SdTabGroup.prototype, "activation", 2);
SdTabGroup = __decorateClass([
  customElement("sd-tab-group")
], SdTabGroup);

export {
  SdTabGroup
};
