import {
  AutoplayController
} from "./chunk.F4VGSDIW.js";
import {
  ScrollController
} from "./chunk.2LMWBUBT.js";
import {
  SdCarouselItem
} from "./chunk.NUIZDE7R.js";
import {
  prefersReducedMotion
} from "./chunk.EY4JPJJQ.js";
import {
  watch
} from "./chunk.X2OQMWGY.js";
import {
  LocalizeController
} from "./chunk.YKM42GKF.js";
import {
  SolidElement,
  customElement
} from "./chunk.HFHUY7YD.js";
import {
  __decorateClass
} from "./chunk.S3NI7NKU.js";

// src/internal/math.ts
function clamp(value, min, max) {
  const noNegativeZero = (n) => Object.is(n, -0) ? 0 : n;
  if (value < min) {
    return noNegativeZero(min);
  }
  if (value > max) {
    return noNegativeZero(max);
  }
  return noNegativeZero(value);
}

// src/components/carousel/carousel.ts
import { css, html } from "lit";
import { map } from "lit/directives/map.js";
import { property, query, queryAll, state } from "lit/decorators.js";
import { range } from "lit/directives/range.js";
import cx from "classix";
var SdCarousel = class extends SolidElement {
  constructor() {
    super(...arguments);
    this.variant = "number";
    this.inverted = false;
    this.loop = false;
    this.autoplay = false;
    this.slidesPerPage = 1;
    this.slidesPerMove = 1;
    this.activeSlide = 0;
    this.currentPage = 1;
    this.pausedAutoplay = false;
    this.autoplayController = new AutoplayController(this, () => this.next());
    this.scrollController = new ScrollController(this);
    this.slides = this.getElementsByTagName("sd-carousel-item");
    // determines which slide is displayed
    // A map containing the state of all the slides
    this.intersectionObserverEntries = /* @__PURE__ */ new Map();
    this.localize = new LocalizeController(this);
    this.userInteracted = false;
    this.handleUserInteraction = () => {
      this.userInteracted = true;
    };
    this.handleSlotChange = (mutations) => {
      const needsInitialization = mutations.some(
        (mutation) => [...mutation.addedNodes, ...mutation.removedNodes].some(
          (node) => SdCarouselItem.isCarouselItem(node) && !node.hasAttribute("data-clone")
        )
      );
      if (needsInitialization) {
        this.initializeSlides();
      }
      this.requestUpdate();
    };
    this.unblockAutoplay = (e, button) => {
      if (e.detail) {
        button.blur();
      }
    };
  }
  connectedCallback() {
    super.connectedCallback();
    this.setAttribute("role", "region");
    this.setAttribute("aria-label", this.localize.term("carousel"));
    ["click", "keydown"].forEach((event) => this.addEventListener(event, this.handleUserInteraction));
    const intersectionObserver = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          this.intersectionObserverEntries.set(entry.target, entry);
          const slide = entry.target;
          slide.toggleAttribute("inert", !entry.isIntersecting);
          slide.classList.toggle("--in-view", entry.isIntersecting);
          slide.setAttribute("aria-hidden", entry.isIntersecting ? "false" : "true");
        });
      },
      {
        root: this,
        threshold: 0.6
      }
    );
    this.intersectionObserver = intersectionObserver;
    intersectionObserver.takeRecords().forEach((entry) => {
      this.intersectionObserverEntries.set(entry.target, entry);
    });
  }
  disconnectedCallback() {
    super.disconnectedCallback();
    this.intersectionObserver.disconnect();
    this.mutationObserver.disconnect();
    ["click", "keydown"].forEach((event) => this.removeEventListener(event, this.handleUserInteraction));
  }
  firstUpdated() {
    this.initializeSlides();
    this.mutationObserver = new MutationObserver(this.handleSlotChange);
    this.mutationObserver.observe(this, { childList: true, subtree: false });
  }
  getPageCount(totalSlides, slidesPerPage, slidesPerMove) {
    return Math.ceil((totalSlides - slidesPerPage) / slidesPerMove) + 1 > 0 ? Math.ceil((totalSlides - slidesPerPage) / slidesPerMove) + 1 : (
      // Returns 1 if the total number of slides is less than the number of slides per page
      1
    );
  }
  getCurrentPage(totalSlides, activeSlide, slidesPerPage, slidesPerMove) {
    return Math.ceil((totalSlides - slidesPerPage) / slidesPerMove) - Math.ceil((totalSlides - slidesPerPage - activeSlide) / slidesPerMove) + 1;
  }
  getSlides({ excludeClones = true } = {}) {
    return [...this.slides].filter((slide) => !excludeClones || !slide.hasAttribute("data-clone"));
  }
  handleKeyDown(event) {
    if (["ArrowLeft", "ArrowRight", "ArrowUp", "ArrowDown", "Home", "End"].includes(event.key)) {
      const target = event.target;
      const isRtl = this.localize.dir() === "rtl";
      const isFocusInPagination = target.closest('[part~="pagination-item"]') !== null;
      const isNext = event.key === "ArrowDown" || !isRtl && event.key === "ArrowRight" || isRtl && event.key === "ArrowLeft";
      const isPrevious = event.key === "ArrowUp" || !isRtl && event.key === "ArrowLeft" || isRtl && event.key === "ArrowRight";
      event.preventDefault();
      if (isPrevious) {
        this.previous();
      }
      if (isNext) {
        this.next();
      }
      if (event.key === "Home") {
        this.goToSlide(0);
      }
      if (event.key === "End") {
        this.goToSlide(this.getSlides().length - 1);
      }
      if (isFocusInPagination) {
        this.updateComplete.then(() => {
          var _a;
          const activePaginationItem = (_a = this.shadowRoot) == null ? void 0 : _a.querySelector(
            '[part~="pagination-item--active"]'
          );
          if (activePaginationItem) {
            activePaginationItem.focus();
          }
        });
      }
    }
  }
  handleScrollEnd() {
    const slides = this.getSlides();
    const entries = [...this.intersectionObserverEntries.values()];
    const firstIntersecting = entries.find((entry) => entry.isIntersecting);
    if (this.loop && (firstIntersecting == null ? void 0 : firstIntersecting.target.hasAttribute("data-clone"))) {
      const clonePosition = Number(firstIntersecting.target.getAttribute("data-clone"));
      this.goToSlide(clonePosition, "auto");
      return;
    }
    if (firstIntersecting) {
      this.activeSlide = slides.indexOf(firstIntersecting.target);
    }
  }
  /**
   * Pause the autoplay.
   */
  pause() {
    this.pausedAutoplay = true;
  }
  /**
   * Resume the autoplay
   */
  resume() {
    this.pausedAutoplay = false;
  }
  handlePausedAutoplay() {
    var _a, _b;
    if (this.pausedAutoplay) {
      this.autoplayController.stop();
      (_a = this.autoplayControls) == null ? void 0 : _a.setAttribute("aria-pressed", "false");
    } else if (this.autoplay) {
      this.autoplayController.start(3e3);
      (_b = this.autoplayControls) == null ? void 0 : _b.setAttribute("aria-pressed", "true");
    }
  }
  initializeSlides() {
    const slides = this.getSlides();
    const intersectionObserver = this.intersectionObserver;
    this.intersectionObserverEntries.clear();
    this.getSlides({ excludeClones: false }).forEach((slide, index) => {
      intersectionObserver.unobserve(slide);
      slide.classList.remove("--in-view");
      slide.classList.remove("--is-active");
      slide.setAttribute("aria-label", this.localize.term("slideNum", index + 1, slides.length));
      if (slide.hasAttribute("data-clone")) {
        slide.remove();
      }
    });
    if (this.loop) {
      const slidesPerPage = this.slidesPerPage;
      const lastSlides = slides.slice(-slidesPerPage);
      const firstSlides = slides.slice(0, slidesPerPage);
      lastSlides.reverse().forEach((slide, i) => {
        const clone = slide.cloneNode(true);
        clone.setAttribute("data-clone", String(slides.length - i - 1));
        this.prepend(clone);
      });
      firstSlides.forEach((slide, i) => {
        const clone = slide.cloneNode(true);
        clone.setAttribute("data-clone", String(i));
        this.append(clone);
      });
    }
    this.getSlides({ excludeClones: false }).forEach((slide) => {
      intersectionObserver.observe(slide);
    });
    this.goToSlide(this.activeSlide, "auto");
  }
  handleSlideChange() {
    const slides = this.getSlides();
    slides.forEach((slide, i) => {
      slide.classList.toggle("--is-active", i === this.activeSlide);
    });
    const newCurrentPage = this.getCurrentPage(slides.length, this.activeSlide, this.slidesPerPage, this.slidesPerMove);
    if (this.currentPage !== newCurrentPage) {
      this.currentPage = newCurrentPage;
    }
    if (this.hasUpdated) {
      this.emit("sd-slide-change", {
        detail: {
          index: this.activeSlide,
          slide: slides[this.activeSlide]
        }
      });
    }
    const pageCount = this.getPageCount(slides.length, this.slidesPerPage, this.slidesPerMove);
    if (this.currentPage > pageCount) {
      requestAnimationFrame(() => {
        this.nextTillFirst();
      });
    }
  }
  handleSlidesPerMoveChange() {
    const slides = this.getSlides({ excludeClones: false });
    const slidesPerMove = this.slidesPerMove;
    slides.forEach((slide, i) => {
      const shouldSnap = Math.abs(i - slidesPerMove) % slidesPerMove === 0;
      if (shouldSnap) {
        slide.style.removeProperty("scroll-snap-align");
      } else {
        slide.style.setProperty("scroll-snap-align", "none");
      }
    });
  }
  handleAutoplayChange() {
    this.autoplayController.stop();
    if (this.autoplay && !this.pausedAutoplay) {
      this.autoplayController.start(3e3);
    }
  }
  /**
   * Move the carousel backward by `slides-per-move` slides.
   *
   * @param behavior - The behavior used for scrolling.
   */
  previous(behavior = "smooth") {
    let previousIndex = this.activeSlide || this.activeSlide - this.slidesPerMove;
    let canSnap = false;
    while (!canSnap && previousIndex > 0) {
      previousIndex -= 1;
      canSnap = Math.abs(previousIndex - this.slidesPerMove) % this.slidesPerMove === 0;
    }
    if (this.currentPage - 1 === 0 && this.loop) {
      this.goToSlide(this.activeSlide - this.slidesPerPage, behavior);
    } else {
      this.goToSlide(previousIndex, behavior);
    }
  }
  /**
   * Move the carousel forward by `slides-per-move` slides.
   *
   * @param behavior - The behavior used for scrolling.
   */
  next(behavior = "smooth") {
    if (this.currentPage + 1 <= this.getPageCount(this.getSlides().length, this.slidesPerPage, this.slidesPerMove)) {
      this.goToSlide(this.activeSlide + this.slidesPerMove, behavior);
    } else {
      if (this.loop) {
        this.nextTillFirst(behavior);
      }
    }
  }
  nextTillFirst(behavior = "smooth") {
    while (this.activeSlide !== 0) {
      this.goToSlide(this.activeSlide + 1, behavior);
    }
    this.currentPage = this.getCurrentPage(
      this.getSlides().length,
      this.activeSlide,
      this.slidesPerPage,
      this.slidesPerMove
    );
  }
  /**
   * Scrolls the carousel to the slide specified by `index`.
   *
   * @param index - The slide index.
   * @param behavior - The behavior used for scrolling.
   */
  goToSlide(index, behavior = "smooth") {
    const { slidesPerPage, loop, scrollContainer } = this;
    const slides = this.getSlides();
    const slidesWithClones = this.getSlides({ excludeClones: false });
    const newActiveSlide = (index + slides.length) % slides.length;
    this.activeSlide = newActiveSlide;
    const nextSlideIndex = clamp(index + (loop ? slidesPerPage : 0), 0, slidesWithClones.length + 1);
    const nextSlide = slidesWithClones[nextSlideIndex];
    const scrollContainerRect = scrollContainer.getBoundingClientRect();
    const nextSlideRect = nextSlide.getBoundingClientRect();
    scrollContainer.scrollTo({
      left: nextSlideRect.left - scrollContainerRect.left + scrollContainer.scrollLeft,
      top: nextSlideRect.top - scrollContainerRect.top + scrollContainer.scrollTop,
      behavior: prefersReducedMotion() ? "auto" : behavior
    });
    if (this.userInteracted && !this.loop) {
      const isLastSlide = this.activeSlide === slides.length - 1 || this.activeSlide === this.slides.length - this.slidesPerPage;
      const isFirstSlide = this.activeSlide === 0;
      if (isLastSlide) {
        this.previousButton.focus({ preventScroll: true });
      } else if (isFirstSlide) {
        this.nextButton.focus({ preventScroll: true });
      }
    }
  }
  render() {
    const { scrollController, slidesPerMove } = this;
    const pagesCount = this.getPageCount(this.getSlides().length, this.slidesPerPage, this.slidesPerMove);
    const currentPage = this.getCurrentPage(
      this.getSlides().length,
      this.activeSlide,
      this.slidesPerPage,
      this.slidesPerMove
    );
    const prevEnabled = this.loop || currentPage > 1;
    const nextEnabled = this.loop || currentPage < pagesCount;
    const isLtr = this.localize.dir() === "ltr";
    return html`<div part="base" class="${cx(`carousel h-full w-full`)}"><div id="scroll-container" part="scroll-container" class="${cx(
      `carousel__slides mb-6
            grid max-h-full w-full items-center justify-items-center overflow-auto`,
      this.inverted ? "focus-visible:focus-outline-inverted" : "focus-visible:focus-outline",
      `overscroll-x-contain grid-flow-col auto-rows-[100%]
            snap-x snap-mandatory overflow-y-hidden`
    )}" style="--slides-per-page:${this.slidesPerPage}" aria-busy="${scrollController.scrolling ? "true" : "false"}" role="status" tabindex="0" @keydown="${this.handleKeyDown}" @scrollend="${this.handleScrollEnd}"><slot></slot></div><div part="controls" class="${cx("w-full flex items-center justify-center relative")}"><div part="navigation" class="${cx("carousel__navigation flex items-center justify-center")}"><button part="navigation-button navigation-button--previous" id="carousel__navigation-button--previous" ?disabled="${!prevEnabled ? true : false}" class="${cx(
      "!mr-6 !rounded-sm sd-interactive",
      !prevEnabled && "sd-interactive--disabled",
      this.inverted ? "sd-interactive--inverted" : "sd-interactive--reset"
    )}" aria-label="${this.localize.term("previousSlide")}" aria-controls="scroll-container" aria-disabled="${prevEnabled ? "false" : "true"}" @click="${prevEnabled ? (e) => {
      this.previous();
      this.unblockAutoplay(e, this.previousButton);
    } : null}"><slot name="previous-icon"><sd-icon class="${cx("h-6 w-6 rotate-90 grid place-items-center")}" library="system" name="${isLtr ? "chevron-down" : "chevron-up"}"></sd-icon></slot></button> ${this.variant === "dot" ? html`<div part="pagination-dot" role="tablist" class="${cx("carousel__pagination dot flex wrap items-center gap-2")}" aria-controls="scroll-container">${map(range(pagesCount), (index) => {
      const isActive = index + 1 === currentPage;
      return html`<button part="pagination-item ${isActive ? "pagination-item--active" : ""}" class="${cx(
        "carousel__pagination-item",
        "block cursor-pointer bg-none border-0 rounded-full",
        isActive ? "bg-accent" : "",
        this.inverted ? "focus-within:focus-outline-inverted" : "focus-within:focus-outline"
      )}" role="tab" tabindex="0" aria-selected="${isActive ? "true" : "false"}" aria-label="${this.localize.term("goToSlide", index + 1, pagesCount)}" @click="${(e) => {
        this.goToSlide(index * slidesPerMove);
        this.unblockAutoplay(e, this.paginationItems[index]);
      }}" @keydown="${this.handleKeyDown}"><span class="${cx(
        "h-4 w-4 block border hover:border-primary-500 rounded-full",
        this.inverted ? "border-white hover:border-primary-500" : "border-primary",
        isActive && "bg-accent border-none",
        isActive ? this.inverted ? "hover:bg-accent-300" : "hover:bg-accent-550" : ""
      )}"></span></button>`;
    })}</div>` : html`<span part="pagination-number" class="carousel__pagination number flex gap-0.5 cursor-default select-none" aria-controls="scroll-container"><span part="pagination-item" class="${cx("w-5 text-center border-b-2 border-accent", this.inverted ? "text-white" : "text-black")}">${currentPage}</span> <span part="pagination-divider" class="${cx("scale-y-[1.5]", "text-center", this.inverted ? "text-white" : "text-black")}">/</span> <span part="pagination-item" class="${cx("w-5 text-center", this.inverted ? "text-white" : "text-black")}">${pagesCount}</span></span>`} <button part="navigation-button navigation-button--next" id="carousel__navigation-button--next" ?disabled="${!nextEnabled ? true : false}" class="${cx(
      "!ml-6 !rounded-sm sd-interactive ",
      !nextEnabled && "sd-interactive--disabled",
      this.inverted ? "sd-interactive--inverted" : "sd-interactive--reset"
    )}" aria-label="${this.localize.term("nextSlide")}" aria-controls="scroll-container" aria-disabled="${nextEnabled ? "false" : "true"}" @click="${nextEnabled ? (e) => {
      this.next();
      this.unblockAutoplay(e, this.nextButton);
    } : null}"><slot name="next-icon"><sd-icon class="${cx("h-6 w-6 rotate-90 grid place-items-center")}" library="system" name="${isLtr ? "chevron-up" : "chevron-down"}"></sd-icon></slot></button></div><button class="${cx(
      "ml-6 !rounded-sm",
      "!absolute !right-0 sd-interactive",
      this.inverted ? "sd-interactive--inverted" : "sd-interactive--reset",
      !this.autoplay && "!hidden"
    )}" part="autoplay-controls" aria-label="${this.localize.term("autoplay")}" aria-pressed="true" @click="${(e) => {
      this.pausedAutoplay = !this.pausedAutoplay;
      if (e.detail) {
        this.autoplayControls.blur();
      }
    }}"><slot name="autoplay-start" class="${cx(!this.pausedAutoplay ? "hidden" : "")}"><sd-icon class="h-6 w-6 grid place-items-center" library="system" name="start"></sd-icon></slot><slot name="autoplay-pause" class="${cx(this.pausedAutoplay ? "hidden" : "")}"><sd-icon class="h-6 w-6 grid place-items-center" library="system" name="pause"></sd-icon></slot></button></div></div>`;
  }
};
SdCarousel.styles = [
  ...SolidElement.styles,
  css`:host{--slide-gap:var(--sl-spacing-medium,1rem);--scroll-hint:0px;display:flex}.carousel{grid-template-areas:". slides ." ". pagination ."}.carousel__pagination{grid-area:pagination}.carousel__slides{grid-area:slides;scrollbar-width:none;--slide-size:calc((100% - (var(--slides-per-page) - 1)*var(--slide-gap))/var(--slides-per-page));-moz-column-gap:var(--slide-gap);column-gap:var(--slide-gap);grid-auto-columns:var(--slide-size);padding-inline:var(--scroll-hint);scroll-padding-inline:var(--scroll-hint)}.carousel__slides::-webkit-scrollbar{display:none}@media (prefers-reduced-motion){:where(.carousel__slides){scroll-behavior:auto}}.carousel__navigation{grid-area:navigation}sd-button::part(label){align-items:center;display:flex;flex:1 1 auto;pointer-events:none}`
];
__decorateClass([
  query('[part~="autoplay-controls"]')
], SdCarousel.prototype, "autoplayControls", 2);
__decorateClass([
  query('[part~="navigation-button--previous"]')
], SdCarousel.prototype, "previousButton", 2);
__decorateClass([
  query('[part~="navigation-button--next"]')
], SdCarousel.prototype, "nextButton", 2);
__decorateClass([
  queryAll('[part~="pagination-item"]')
], SdCarousel.prototype, "paginationItems", 2);
__decorateClass([
  property({ type: String, reflect: true })
], SdCarousel.prototype, "variant", 2);
__decorateClass([
  property({ type: Boolean, reflect: true })
], SdCarousel.prototype, "inverted", 2);
__decorateClass([
  property({ type: Boolean, reflect: true })
], SdCarousel.prototype, "loop", 2);
__decorateClass([
  property({ type: Boolean, reflect: true })
], SdCarousel.prototype, "autoplay", 2);
__decorateClass([
  property({ type: Number, attribute: "slides-per-page" })
], SdCarousel.prototype, "slidesPerPage", 2);
__decorateClass([
  property({ type: Number, attribute: "slides-per-move" })
], SdCarousel.prototype, "slidesPerMove", 2);
__decorateClass([
  query("slot:not([name])")
], SdCarousel.prototype, "defaultSlot", 2);
__decorateClass([
  query(".carousel__slides")
], SdCarousel.prototype, "scrollContainer", 2);
__decorateClass([
  query(".carousel__pagination")
], SdCarousel.prototype, "paginationContainer", 2);
__decorateClass([
  state()
], SdCarousel.prototype, "activeSlide", 2);
__decorateClass([
  state()
], SdCarousel.prototype, "currentPage", 2);
__decorateClass([
  state()
], SdCarousel.prototype, "pausedAutoplay", 2);
__decorateClass([
  watch("pausedAutoplay")
], SdCarousel.prototype, "handlePausedAutoplay", 1);
__decorateClass([
  watch("loop", { waitUntilFirstUpdate: true }),
  watch("slidesPerPage", { waitUntilFirstUpdate: true })
], SdCarousel.prototype, "initializeSlides", 1);
__decorateClass([
  watch("activeSlide")
], SdCarousel.prototype, "handleSlideChange", 1);
__decorateClass([
  watch("slidesPerMove")
], SdCarousel.prototype, "handleSlidesPerMoveChange", 1);
__decorateClass([
  watch("autoplay")
], SdCarousel.prototype, "handleAutoplayChange", 1);
SdCarousel = __decorateClass([
  customElement("sd-carousel")
], SdCarousel);

export {
  SdCarousel
};
