"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Algebra = require("./algebra");
const factory_1 = require("./factory");
const SparqlGenerator = require('sparqljs').Generator;
const types = Algebra.types;
const eTypes = Algebra.expressionTypes;
let context;
const factory = new factory_1.default();
function toSparql(op) {
    let generator = new SparqlGenerator();
    return generator.stringify(toSparqlJs(op));
}
exports.toSparql = toSparql;
function toSparqlJs(op) {
    resetContext();
    op = removeQuads(op);
    let result = translateOperation(op);
    if (result.type === 'group')
        return result.patterns[0];
    return result;
}
exports.toSparqlJs = toSparqlJs;
function flatten(s) {
    return Array.prototype.concat(...s).filter(x => x);
}
function resetContext() {
    context = { project: false, extend: [], group: [], aggregates: [], order: [] };
}
function translateOperation(op) {
    // this allows us to differentiate between BIND and SELECT when translating EXTEND
    if (op.type !== types.EXTEND && op.type !== types.ORDER_BY)
        context.project = false;
    switch (op.type) {
        case types.EXPRESSION: return translateExpression(op);
        case types.ASK: return translateProject(op, types.ASK);
        case types.BGP: return translateBgp(op);
        case types.CONSTRUCT: return translateConstruct(op);
        case types.DESCRIBE: return translateProject(op, types.DESCRIBE);
        case types.DISTINCT: return translateDistinct(op);
        case types.EXTEND: return translateExtend(op);
        case types.FROM: return translateFrom(op);
        case types.FILTER: return translateFilter(op);
        case types.GRAPH: return translateGraph(op);
        case types.GROUP: return translateGroup(op);
        case types.JOIN: return translateJoin(op);
        case types.LEFT_JOIN: return translateLeftJoin(op);
        case types.MINUS: return translateMinus(op);
        case types.ORDER_BY: return translateOrderBy(op);
        case types.PATH: return translatePath(op);
        case types.PATTERN: return translatePattern(op);
        case types.PROJECT: return translateProject(op, types.PROJECT);
        case types.REDUCED: return translateReduced(op);
        case types.SERVICE: return translateService(op);
        case types.SLICE: return translateSlice(op);
        case types.UNION: return translateUnion(op);
        case types.VALUES: return translateValues(op);
    }
    throw new Error('Unknown Operation type ' + op.type);
}
function translateExpression(expr) {
    switch (expr.expressionType) {
        case eTypes.AGGREGATE: return translateAggregateExpression(expr);
        case eTypes.EXISTENCE: return translateExistenceExpression(expr);
        case eTypes.NAMED: return translateNamedExpression(expr);
        case eTypes.OPERATOR: return translateOperatorExpression(expr);
        case eTypes.TERM: return translateTermExpression(expr);
    }
    throw new Error('Unknown Expression Operation type ' + expr.expressionType);
}
function translatePathComponent(path) {
    switch (path.type) {
        case types.ALT: return translateAlt(path);
        case types.INV: return translateInv(path);
        case types.LINK: return translateLink(path);
        case types.NPS: return translateNps(path);
        case types.ONE_OR_MORE_PATH: return translateOneOrMorePath(path);
        case types.SEQ: return translateSeq(path);
        case types.ZERO_OR_MORE_PATH: return translateZeroOrMorePath(path);
        case types.ZERO_OR_ONE_PATH: return translateZeroOrOnePath(path);
    }
    throw new Error('Unknown Path type ' + path.type);
}
function translateTerm(term) {
    if (term.termType === 'BlankNode')
        return '_:' + term.value;
    if (term.termType === 'Literal') {
        // TODO: should check which safety checks are required
        let lit = term;
        let result = `"${term.value}"`;
        if (lit.language)
            result += '@' + lit.language;
        else if (lit.datatype && lit.datatype.value !== 'http://www.w3.org/2001/XMLSchema#string')
            result += '^^' + lit.datatype.value;
        return result;
    }
    if (term.termType === 'NamedNode')
        return term.value;
    if (term.termType === 'Variable')
        return '?' + term.value;
    throw new Error('Unknown Term type ' + term.termType);
}
// ------------------------- EXPRESSIONS -------------------------
function translateAggregateExpression(expr) {
    let result = {
        expression: translateExpression(expr.expression),
        type: 'aggregate',
        aggregation: expr.aggregator,
        distinct: expr.distinct
    };
    if (expr.separator)
        result.separator = expr.separator;
    return result;
}
function translateExistenceExpression(expr) {
    return {
        type: 'operation',
        operator: expr.not ? 'notexists' : 'exists',
        args: flatten([
            translateOperation(expr.input)
        ])
    };
}
function translateNamedExpression(expr) {
    return {
        type: 'functionCall',
        function: translateTerm(expr.name),
        args: expr.args.map(translateExpression)
    };
}
function translateOperatorExpression(expr) {
    if (expr.operator === 'desc') {
        let result = { expression: translateExpression(expr.args[0]) };
        result.descending = true;
        return result;
    }
    let result = {
        type: 'operation',
        operator: expr.operator,
        args: expr.args.map(translateExpression)
    };
    if (result.operator === 'in' || result.operator === 'notin')
        result.args = [result.args[0]].concat([result.args.slice(1)]);
    return result;
}
function translateTermExpression(expr) {
    return translateTerm(expr.term);
}
// ------------------------- OPERATIONS -------------------------
// these get translated in the project function
function translateBoundAggregate(op) {
    return op;
}
function translateBgp(op) {
    let patterns = op.patterns.map(translatePattern);
    if (patterns.length === 0)
        return null;
    return {
        type: 'bgp',
        triples: patterns
    };
}
function translateConstruct(op) {
    return {
        type: 'query',
        prefixes: {},
        queryType: "CONSTRUCT",
        template: op.template.map(translatePattern),
        where: flatten([
            translateOperation(op.input)
        ])
    };
}
function translateDistinct(op) {
    let result = translateOperation(op.input);
    // project is nested in group object
    result.patterns[0].distinct = true;
    return result;
}
function translateExtend(op) {
    if (context.project) {
        context.extend.push(op);
        return translateOperation(op.input);
    }
    return flatten([
        translateOperation(op.input),
        {
            type: 'bind',
            variable: translateTerm(op.variable),
            expression: translateExpression(op.expression)
        }
    ]);
}
function translateFrom(op) {
    let result = translateOperation(op.input);
    // project is nested in group object
    let obj = result.patterns[0];
    obj.from = {
        default: op.default.map(translateTerm),
        named: op.named.map(translateTerm)
    };
    return result;
}
function translateFilter(op) {
    return {
        type: 'group',
        patterns: flatten([
            translateOperation(op.input),
            { type: 'filter', expression: translateExpression(op.expression) }
        ])
    };
}
function translateGraph(op) {
    return {
        type: 'graph',
        patterns: flatten([translateOperation(op.input)]),
        name: translateTerm(op.name)
    };
}
function translateGroup(op) {
    let input = translateOperation(op.input);
    let aggs = op.aggregates.map(translateBoundAggregate);
    context.aggregates.push(...aggs);
    // TODO: apply possible extends
    context.group.push(...op.variables);
    return input;
}
function translateJoin(op) {
    return flatten([
        translateOperation(op.left),
        translateOperation(op.right)
    ]);
}
function translateLeftJoin(op) {
    let leftjoin = {
        type: 'optional',
        patterns: [
            translateOperation(op.right)
        ]
    };
    if (op.expression) {
        leftjoin.patterns.push({
            type: 'filter',
            expression: translateExpression(op.expression)
        });
    }
    leftjoin.patterns = flatten(leftjoin.patterns);
    return flatten([
        translateOperation(op.left),
        leftjoin
    ]);
}
function translateMinus(op) {
    let patterns = translateOperation(op.right);
    if (patterns.type === 'group')
        patterns = patterns.patterns;
    return flatten([
        translateOperation(op.left),
        {
            type: 'minus',
            patterns: patterns
        }
    ]);
}
function translateOrderBy(op) {
    context.order.push(...op.expressions);
    return translateOperation(op.input);
}
function translatePath(op) {
    // TODO: quads back to graph statement
    return {
        type: 'bgp',
        triples: [{
                subject: translateTerm(op.subject),
                predicate: translatePathComponent(op.predicate),
                object: translateTerm(op.object)
            }]
    };
}
function translatePattern(op) {
    // TODO: quads back to graph statement
    return {
        subject: translateTerm(op.subject),
        predicate: translateTerm(op.predicate),
        object: translateTerm(op.object)
    };
}
function replaceAggregatorVariables(s, map) {
    if (typeof s === 'string') {
        if (map[s])
            return map[s];
    }
    else if (Array.isArray(s)) {
        s = s.map(e => replaceAggregatorVariables(e, map));
    }
    else {
        for (let key of Object.keys(s))
            s[key] = replaceAggregatorVariables(s[key], map);
    }
    return s;
}
function translateProject(op, type) {
    let result = {
        type: 'query',
        prefixes: {}
    };
    if (type === types.PROJECT) {
        result.queryType = 'SELECT';
        result.variables = op.variables.map(translateTerm);
    }
    else if (type === types.ASK) {
        result.queryType = 'ASK';
    }
    else if (type === types.DESCRIBE) {
        result.queryType = 'DESCRIBE';
        result.variables = op.terms.map(translateTerm);
    }
    // backup values in case of nested queries
    // everything in extend, group, etc. is irrelevant for this project call
    let extend = context.extend;
    let group = context.group;
    let aggregates = context.aggregates;
    let order = context.order;
    resetContext();
    context.project = true;
    let input = flatten([translateOperation(op.input)]);
    if (input.length === 1 && input[0].type === 'group')
        input = input[0].patterns;
    result.where = input;
    let aggregators = {};
    // these can not reference each other
    for (let agg of context.aggregates)
        aggregators[translateTerm(agg.variable)] = translateExpression(agg);
    // do these in reverse order since variables in one extend might apply to an expression in an other extend
    let extensions = {};
    for (let i = context.extend.length - 1; i >= 0; --i) {
        let e = context.extend[i];
        extensions[translateTerm(e.variable)] = replaceAggregatorVariables(translateExpression(e.expression), aggregators);
    }
    if (context.group.length > 0)
        result.group = context.group.map(translateTerm).map(v => {
            if (extensions[v]) {
                let result = extensions[v];
                delete extensions[v]; // make sure there is only 1 'AS' statement
                return {
                    variable: v,
                    expression: result
                };
            }
            return { expression: v };
        });
    // descending expressions will already be in the correct format due to the structure of those
    if (context.order.length > 0)
        result.order = context.order.map(translateOperation).map(o => o.descending ? o : ({ expression: o }));
    // this needs to happen after the group because it might depend on variables generated there
    if (result.variables) {
        result.variables = result.variables.map((v) => {
            if (extensions[v])
                return {
                    variable: v,
                    expression: extensions[v]
                };
            return v;
        });
        // if the * didn't match any variables this would be empty
        if (result.variables.length === 0)
            result.variables = ['*'];
    }
    // convert filter to 'having' if it contains an aggregator variable
    // could always convert, but is nicer to use filter when possible
    if (result.where.length > 0 && result.where[result.where.length - 1].type === 'filter') {
        let filter = result.where[result.where.length - 1];
        if (objectContainsValues(filter, Object.keys(aggregators))) {
            result.having = flatten([replaceAggregatorVariables(filter.expression, aggregators)]);
            result.where.splice(-1);
        }
    }
    context.extend = extend;
    context.group = group;
    context.aggregates = aggregates;
    context.order = order;
    // subqueries need to be in a group
    result = { type: 'group', patterns: [result] };
    return result;
}
function objectContainsValues(o, vals) {
    if (Array.isArray(o))
        return o.some(e => objectContainsValues(e, vals));
    if (o === Object(o))
        return Object.keys(o).some(key => objectContainsValues(o[key], vals));
    return vals.indexOf(o) >= 0;
}
function translateReduced(op) {
    let result = translateOperation(op.input);
    // project is nested in group object
    result.patterns[0].reduced = true;
    return result;
}
function translateService(op) {
    let patterns = translateOperation(op.input);
    if (patterns.type === 'group')
        patterns = patterns.patterns;
    if (!Array.isArray(patterns))
        patterns = [patterns];
    return {
        type: 'service',
        name: translateTerm(op.name),
        silent: op.silent,
        patterns
    };
}
function translateSlice(op) {
    let result = translateOperation(op.input);
    // project is nested in group object
    let obj = result.patterns[0];
    if (op.start !== 0)
        obj.offset = op.start;
    if (op.length !== undefined)
        obj.limit = op.length;
    return result;
}
function translateUnion(op) {
    return {
        type: 'union',
        patterns: flatten([
            translateOperation(op.left),
            translateOperation(op.right)
        ])
    };
}
function translateValues(op) {
    // TODO: check if handled correctly when outside of select block
    return {
        type: 'values',
        values: op.bindings.map(binding => {
            let result = {};
            for (let v of op.variables) {
                let s = '?' + v.value;
                if (binding[s])
                    result[s] = translateTerm(binding[s]);
                else
                    result[s] = undefined;
            }
            return result;
        })
    };
}
// PATH COMPONENTS
function translateAlt(path) {
    let left = translatePathComponent(path.left);
    let right = translatePathComponent(path.right);
    if (left.pathType === '!' && right.pathType === '!') {
        return {
            type: 'path',
            pathType: '!',
            items: [{
                    type: 'path',
                    pathType: '|',
                    items: [].concat(left.items, right.items)
                }]
        };
    }
    return {
        type: 'path',
        pathType: '|',
        items: [left, right]
    };
}
function translateInv(path) {
    if (path.path.type === types.NPS) {
        let npsPath = path.path;
        let inv = npsPath.iris.map((iri) => {
            return {
                type: 'path',
                pathType: '^',
                items: [translateTerm(iri)]
            };
        });
        if (inv.length <= 1)
            return {
                type: 'path',
                pathType: '!',
                items: inv
            };
        return {
            type: 'path',
            pathType: '!',
            items: [{
                    type: 'path',
                    pathType: '|',
                    items: inv
                }]
        };
    }
    return {
        type: 'path',
        pathType: '^',
        items: [translatePathComponent(path.path)]
    };
}
function translateLink(path) {
    return translateTerm(path.iri);
}
function translateNps(path) {
    if (path.iris.length <= 1)
        return {
            type: 'path',
            pathType: '!',
            items: path.iris.map(translateTerm)
        };
    return {
        type: 'path',
        pathType: '!',
        items: [{
                type: 'path',
                pathType: '|',
                items: path.iris.map(translateTerm)
            }]
    };
}
function translateOneOrMorePath(path) {
    return {
        type: 'path',
        pathType: '+',
        items: [translatePathComponent(path.path)]
    };
}
function translateSeq(path) {
    return {
        type: 'path',
        pathType: '/',
        items: [translatePathComponent(path.left), translatePathComponent(path.right)]
    };
}
function translateZeroOrMorePath(path) {
    return {
        type: 'path',
        pathType: '*',
        items: [translatePathComponent(path.path)]
    };
}
function translateZeroOrOnePath(path) {
    return {
        type: 'path',
        pathType: '?',
        items: [translatePathComponent(path.path)]
    };
}
function removeQuads(op) {
    return removeQuadsRecursive(op, {});
}
// remove quads
function removeQuadsRecursive(op, graphs) {
    if (Array.isArray(op))
        return op.map(sub => removeQuadsRecursive(sub, graphs));
    if (!op.type)
        return op;
    if ((op.type === types.PATTERN || op.type === types.PATH) && op.graph) {
        if (!graphs[op.graph.value])
            graphs[op.graph.value] = { graph: op.graph, values: [] };
        graphs[op.graph.value].values.push(op);
        return op;
    }
    const result = {};
    const keyGraphs = {}; // unique graph per key
    const globalNames = {}; // track all the unique graph names for the entire Operation
    for (let key of Object.keys(op)) {
        const newGraphs = {};
        result[key] = removeQuadsRecursive(op[key], newGraphs);
        const graphNames = Object.keys(newGraphs);
        // create graph statements if multiple graphs are found
        if (graphNames.length > 1) {
            // nest joins
            let left = potentialGraphFromPatterns(newGraphs[graphNames[0]].values);
            for (let i = 1; i < graphNames.length; ++i) {
                const right = potentialGraphFromPatterns(newGraphs[graphNames[i]].values);
                left = factory.createJoin(left, right);
            }
            graphNames.map(name => delete newGraphs[name]);
            // this ignores the result object that is being generated, but should not be a problem
            // is only an issue for objects that have 2 keys where this can occur, which is none
            return left;
        }
        else if (graphNames.length === 1) {
            const graph = newGraphs[graphNames[0]].graph;
            keyGraphs[key] = graph;
            globalNames[graph.value] = graph;
        }
    }
    const graphNameSet = Object.keys(globalNames);
    if (graphNameSet.length > 0) {
        // also need to create graph statement if we are at the edge of the query
        if (graphNameSet.length === 1 && op.type !== types.PROJECT)
            graphs[graphNameSet[0]] = { graph: globalNames[graphNameSet[0]], values: [result] };
        else {
            // multiple graphs (or project), need to create graph objects for them
            for (let key of Object.keys(keyGraphs))
                if (keyGraphs[key].value.length > 0)
                    result[key] = factory.createGraph(result[key], keyGraphs[key]);
        }
    }
    return result;
}
function potentialGraphFromPatterns(patterns) {
    const bgp = factory.createBgp(patterns);
    const name = patterns[0].graph;
    if (name.value.length === 0)
        return bgp;
    return factory.createGraph(bgp, name);
}
//# sourceMappingURL=sparql.js.map