import * as A from "./algebra";
import Factory from "./factory";
import { Variable } from "rdf-js";
import * as RDF from 'rdf-js';
export default class Util {
    /**
     * Detects all in-scope variables.
     * In practice this means iterating through the entire algebra tree, finding all variables,
     * and stopping when a project function is found.
     * @param {Operation} op - Input algebra tree.
     * @returns {Variable[]} - List of unique in-scope variables.
     */
    static inScopeVariables(op: A.Operation): Variable[];
    /**
     * Recurses through the given algebra tree
     * A map of callback functions can be provided for individual Operation types to gather data.
     * The return value of those callbacks should indicate whether recursion should be applied or not.
     * Making modifications will change the original input object.
     * @param {Operation} op - The Operation to recurse on.
     * @param { [type: string]: (op: Operation) => boolean } callbacks - A map of required callback Operations.
     */
    static recurseOperation(op: A.Operation, callbacks: {
        [type: string]: (op: A.Operation) => boolean;
    }): void;
    /**
     * Creates a deep copy of the given Operation.
     * Creates shallow copies of the non-Operation values.
     * A map of callback functions can be provided for individual Operation types
     * to specifically modify the given objects before triggering recursion.
     * The return value of those callbacks should indicate whether recursion should be applied to this returned object or not.
     * @param {Operation} op - The Operation to recurse on.
     * @param { [type: string]: (op: Operation, factory: Factory) => RecurseResult } callbacks - A map of required callback Operations.
     * @param {Factory} factory - Factory used to create new Operations. Will use default factory if none is provided.
     * @returns {Operation} - The copied result.
     */
    static mapOperation(op: A.Operation, callbacks: {
        [type: string]: (op: A.Operation, factory: Factory) => RecurseResult;
    }, factory?: Factory): A.Operation;
    /**
     * Similar to the {@link mapOperation} function but specifically for expressions.
     * Both functions call each other while copying.
     * Should not be called directly since it does not execute the callbacks, these happen in {@link mapOperation}.
     * @param {Expression} expr - The Operation to recurse on.
     * @param { [type: string]: (op: Operation, factory: Factory) => RecurseResult } callbacks - A map of required callback Operations.
     * @param {Factory} factory - Factory used to create new Operations. Will use default factory if none is provided.
     * @returns {Operation} - The copied result.
     */
    private static mapExpression;
    static createUniqueVariable(label: string, variables: {
        [vLabel: string]: boolean;
    }, dataFactory: RDF.DataFactory): RDF.Variable;
}
/**
 * @interface RecurseResult
 * @property {Operation} result - The resulting A.Operation.
 * @property {boolean} recurse - Whether to continue with recursion.
 */
export interface RecurseResult {
    result: A.Operation;
    recurse: boolean;
}
