"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const algebra_1 = require("./algebra");
const factory_1 = require("./factory");
class Util {
    /**
     * Detects all in-scope variables.
     * In practice this means iterating through the entire algebra tree, finding all variables,
     * and stopping when a project function is found.
     * @param {Operation} op - Input algebra tree.
     * @returns {Variable[]} - List of unique in-scope variables.
     */
    static inScopeVariables(op) {
        const variables = [];
        function addVariable(v) {
            if (!variables.find(v2 => v.value === v2.value))
                variables.push(v);
        }
        // https://www.w3.org/TR/sparql11-query/#variableScope
        Util.recurseOperation(op, {
            [algebra_1.types.EXPRESSION]: (op) => {
                let expr = op;
                if (expr.expressionType === 'aggregate' && expr.variable) {
                    let agg = expr;
                    addVariable(agg.variable);
                }
                return true;
            },
            [algebra_1.types.EXTEND]: (op) => {
                let extend = op;
                addVariable(extend.variable);
                return true;
            },
            [algebra_1.types.GRAPH]: (op) => {
                let graph = op;
                if (graph.name.termType === 'Variable')
                    addVariable(graph.name);
                return true;
            },
            [algebra_1.types.GROUP]: (op) => {
                let group = op;
                group.variables.forEach(addVariable);
                return true;
            },
            [algebra_1.types.PATH]: (op) => {
                let path = op;
                if (path.subject.termType === 'Variable')
                    addVariable(path.subject);
                if (path.object.termType === 'Variable')
                    addVariable(path.object);
                if (path.graph.termType === 'Variable')
                    addVariable(path.graph);
                return true;
            },
            [algebra_1.types.PATTERN]: (op) => {
                let pattern = op;
                if (pattern.subject.termType === 'Variable')
                    addVariable(pattern.subject);
                if (pattern.predicate.termType === 'Variable')
                    addVariable(pattern.predicate);
                if (pattern.object.termType === 'Variable')
                    addVariable(pattern.object);
                if (pattern.graph.termType === 'Variable')
                    addVariable(pattern.graph);
                return true;
            },
            [algebra_1.types.PROJECT]: (op) => {
                let project = op;
                project.variables.forEach(addVariable);
                return false;
            },
            [algebra_1.types.SERVICE]: (op) => {
                let service = op;
                if (service.name.termType === 'Variable')
                    addVariable(service.name);
                return true;
            },
            [algebra_1.types.VALUES]: (op) => {
                let values = op;
                values.variables.forEach(addVariable);
                return true;
            },
        });
        return variables;
    }
    /**
     * Recurses through the given algebra tree
     * A map of callback functions can be provided for individual Operation types to gather data.
     * The return value of those callbacks should indicate whether recursion should be applied or not.
     * Making modifications will change the original input object.
     * @param {Operation} op - The Operation to recurse on.
     * @param { [type: string]: (op: Operation) => boolean } callbacks - A map of required callback Operations.
     */
    static recurseOperation(op, callbacks) {
        let result = op;
        let doRecursion = true;
        if (callbacks[op.type])
            doRecursion = callbacks[op.type](op);
        if (!doRecursion)
            return;
        let recurseOp = (op) => Util.recurseOperation(op, callbacks);
        switch (result.type) {
            case algebra_1.types.ALT:
                const alt = result;
                recurseOp(alt.left);
                recurseOp(alt.right);
                break;
            case algebra_1.types.ASK:
                const ask = result;
                recurseOp(ask.input);
                break;
            case algebra_1.types.BGP:
                const bgp = result;
                bgp.patterns.forEach(recurseOp);
                break;
            case algebra_1.types.CONSTRUCT:
                const construct = result;
                recurseOp(construct.input);
                construct.template.map(recurseOp);
                break;
            case algebra_1.types.DESCRIBE:
                const describe = result;
                recurseOp(describe.input);
                break;
            case algebra_1.types.DISTINCT:
                const distinct = result;
                recurseOp(distinct.input);
                break;
            case algebra_1.types.EXPRESSION:
                const expr = result;
                if (expr.expressionType === algebra_1.expressionTypes.EXISTENCE) {
                    const exist = expr;
                    recurseOp(exist.input);
                }
                break;
            case algebra_1.types.EXTEND:
                const extend = result;
                recurseOp(extend.input);
                recurseOp(extend.expression);
                break;
            case algebra_1.types.FILTER:
                const filter = result;
                recurseOp(filter.input);
                recurseOp(filter.expression);
                break;
            case algebra_1.types.FROM:
                const from = result;
                recurseOp(from.input);
                break;
            case algebra_1.types.GRAPH:
                const graph = result;
                recurseOp(graph.input);
                break;
            case algebra_1.types.GROUP:
                const group = result;
                recurseOp(group.input);
                group.aggregates.forEach(recurseOp);
                break;
            case algebra_1.types.INV:
                const inv = result;
                recurseOp(inv.path);
                break;
            case algebra_1.types.JOIN:
                const join = result;
                recurseOp(join.left);
                recurseOp(join.right);
                break;
            case algebra_1.types.LEFT_JOIN:
                const leftJoin = result;
                recurseOp(leftJoin.left);
                recurseOp(leftJoin.right);
                if (leftJoin.expression)
                    recurseOp(leftJoin.expression);
                break;
            case algebra_1.types.LINK:
                break;
            case algebra_1.types.MINUS:
                const minus = result;
                recurseOp(minus.left);
                recurseOp(minus.right);
                break;
            case algebra_1.types.NPS:
                break;
            case algebra_1.types.ONE_OR_MORE_PATH:
                const oom = result;
                recurseOp(oom.path);
                break;
            case algebra_1.types.ORDER_BY:
                const order = result;
                recurseOp(order.input);
                order.expressions.forEach(recurseOp);
                break;
            case algebra_1.types.PATH:
                const path = result;
                recurseOp(path.predicate);
                break;
            case algebra_1.types.PATTERN:
                break;
            case algebra_1.types.PROJECT:
                const project = result;
                recurseOp(project.input);
                break;
            case algebra_1.types.REDUCED:
                const reduced = result;
                recurseOp(reduced.input);
                break;
            case algebra_1.types.SEQ:
                const seq = result;
                recurseOp(seq.left);
                recurseOp(seq.right);
                break;
            case algebra_1.types.SERVICE:
                const service = result;
                recurseOp(service.input);
                break;
            case algebra_1.types.SLICE:
                const slice = result;
                recurseOp(slice.input);
                break;
            case algebra_1.types.UNION:
                const union = result;
                recurseOp(union.left);
                recurseOp(union.right);
                break;
            case algebra_1.types.VALUES:
                break;
            case algebra_1.types.ZERO_OR_MORE_PATH:
                const zom = result;
                recurseOp(zom.path);
                break;
            case algebra_1.types.ZERO_OR_ONE_PATH:
                const zoo = result;
                recurseOp(zoo.path);
                break;
            default: throw new Error('Unknown Operation type ' + result.type);
        }
    }
    /**
     * Creates a deep copy of the given Operation.
     * Creates shallow copies of the non-Operation values.
     * A map of callback functions can be provided for individual Operation types
     * to specifically modify the given objects before triggering recursion.
     * The return value of those callbacks should indicate whether recursion should be applied to this returned object or not.
     * @param {Operation} op - The Operation to recurse on.
     * @param { [type: string]: (op: Operation, factory: Factory) => RecurseResult } callbacks - A map of required callback Operations.
     * @param {Factory} factory - Factory used to create new Operations. Will use default factory if none is provided.
     * @returns {Operation} - The copied result.
     */
    static mapOperation(op, callbacks, factory) {
        let result = op;
        let doRecursion = true;
        factory = factory || new factory_1.default();
        if (callbacks[op.type])
            ({ result, recurse: doRecursion } = callbacks[op.type](op, factory));
        if (!doRecursion)
            return result;
        let mapOp = (op) => Util.mapOperation(op, callbacks, factory);
        switch (result.type) {
            case algebra_1.types.ALT:
                const alt = result;
                return factory.createAlt(mapOp(alt.left), mapOp(alt.right));
            case algebra_1.types.ASK:
                const ask = result;
                return factory.createAsk(mapOp(ask.input));
            case algebra_1.types.BGP:
                const bgp = result;
                return factory.createBgp(bgp.patterns.map(mapOp));
            case algebra_1.types.CONSTRUCT:
                const construct = result;
                return factory.createConstruct(mapOp(construct.input), construct.template.map(mapOp));
            case algebra_1.types.DESCRIBE:
                const describe = result;
                return factory.createDescribe(mapOp(describe.input), describe.terms);
            case algebra_1.types.DISTINCT:
                const distinct = result;
                return factory.createDistinct(mapOp(distinct.input));
            case algebra_1.types.EXPRESSION:
                const expr = result;
                return Util.mapExpression(expr, callbacks, factory);
            case algebra_1.types.EXTEND:
                const extend = result;
                return factory.createExtend(mapOp(extend.input), extend.variable, mapOp(extend.expression));
            case algebra_1.types.FILTER:
                const filter = result;
                return factory.createFilter(mapOp(filter.input), mapOp(filter.expression));
            case algebra_1.types.FROM:
                const from = result;
                return factory.createFrom(mapOp(from.input), [].concat(from.default), [].concat(from.named));
            case algebra_1.types.GRAPH:
                const graph = result;
                return factory.createGraph(mapOp(graph.input), graph.name);
            case algebra_1.types.GROUP:
                const group = result;
                return factory.createGroup(mapOp(group.input), [].concat(group.variables), group.aggregates.map(mapOp));
            case algebra_1.types.INV:
                const inv = result;
                return factory.createInv(mapOp(inv.path));
            case algebra_1.types.JOIN:
                const join = result;
                return factory.createJoin(mapOp(join.left), mapOp(join.right));
            case algebra_1.types.LEFT_JOIN:
                const leftJoin = result;
                return factory.createLeftJoin(mapOp(leftJoin.left), mapOp(leftJoin.right), leftJoin.expression ? mapOp(leftJoin.expression) : undefined);
            case algebra_1.types.LINK:
                const link = result;
                return factory.createLink(link.iri);
            case algebra_1.types.MINUS:
                const minus = result;
                return factory.createMinus(mapOp(minus.left), mapOp(minus.right));
            case algebra_1.types.NPS:
                const nps = result;
                return factory.createNps([].concat(nps.iris));
            case algebra_1.types.ONE_OR_MORE_PATH:
                const oom = result;
                return factory.createOneOrMorePath(mapOp(oom.path));
            case algebra_1.types.ORDER_BY:
                const order = result;
                return factory.createOrderBy(mapOp(order.input), order.expressions.map(mapOp));
            case algebra_1.types.PATH:
                const path = result;
                return factory.createPath(path.subject, mapOp(path.predicate), path.object, path.graph);
            case algebra_1.types.PATTERN:
                const pattern = result;
                return factory.createPattern(pattern.subject, pattern.predicate, pattern.object, pattern.graph);
            case algebra_1.types.PROJECT:
                const project = result;
                return factory.createProject(mapOp(project.input), [].concat(project.variables));
            case algebra_1.types.REDUCED:
                const reduced = result;
                return factory.createReduced(mapOp(reduced.input));
            case algebra_1.types.SEQ:
                const seq = result;
                return factory.createSeq(mapOp(seq.left), mapOp(seq.right));
            case algebra_1.types.SERVICE:
                const service = result;
                return factory.createService(mapOp(service.input), service.name, service.silent);
            case algebra_1.types.SLICE:
                const slice = result;
                return factory.createSlice(mapOp(slice.input), slice.start, slice.length);
            case algebra_1.types.UNION:
                const union = result;
                return factory.createUnion(mapOp(union.left), mapOp(union.right));
            case algebra_1.types.VALUES:
                const values = result;
                return factory.createValues([].concat(values.variables), values.bindings.map(b => Object.assign({}, b)));
            case algebra_1.types.ZERO_OR_MORE_PATH:
                const zom = result;
                return factory.createZeroOrMorePath(mapOp(zom.path));
            case algebra_1.types.ZERO_OR_ONE_PATH:
                const zoo = result;
                return factory.createZeroOrOnePath(mapOp(zoo.path));
            default: throw new Error('Unknown Operation type ' + result.type);
        }
    }
    /**
     * Similar to the {@link mapOperation} function but specifically for expressions.
     * Both functions call each other while copying.
     * Should not be called directly since it does not execute the callbacks, these happen in {@link mapOperation}.
     * @param {Expression} expr - The Operation to recurse on.
     * @param { [type: string]: (op: Operation, factory: Factory) => RecurseResult } callbacks - A map of required callback Operations.
     * @param {Factory} factory - Factory used to create new Operations. Will use default factory if none is provided.
     * @returns {Operation} - The copied result.
     */
    static mapExpression(expr, callbacks, factory) {
        let recurse = (op) => Util.mapOperation(op, callbacks, factory);
        switch (expr.expressionType) {
            case algebra_1.expressionTypes.AGGREGATE:
                if (expr.variable) {
                    const bound = expr;
                    return factory.createBoundAggregate(bound.variable, bound.aggregator, recurse(bound.expression), bound.distinct, bound.separator);
                }
                const aggregate = expr;
                return factory.createAggregateExpression(aggregate.aggregator, recurse(aggregate.expression), aggregate.distinct, aggregate.separator);
            case algebra_1.expressionTypes.EXISTENCE:
                const exist = expr;
                return factory.createExistenceExpression(exist.not, recurse(exist.input));
            case algebra_1.expressionTypes.NAMED:
                const named = expr;
                return factory.createNamedExpression(named.name, named.args.map(recurse));
            case algebra_1.expressionTypes.OPERATOR:
                const op = expr;
                return factory.createOperatorExpression(op.operator, op.args.map(recurse));
            case algebra_1.expressionTypes.TERM:
                const term = expr;
                return factory.createTermExpression(term.term);
            default: throw new Error('Unknown Expression type ' + expr.expressionType);
        }
    }
    static createUniqueVariable(label, variables, dataFactory) {
        let counter = 0;
        let labelLoop = label;
        while (variables[labelLoop]) {
            labelLoop = label + counter++;
        }
        return dataFactory.variable(labelLoop);
    }
}
exports.default = Util;
//# sourceMappingURL=util.js.map