"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const RDFString = require("rdf-string");
const sparqlalgebrajs_1 = require("sparqlalgebrajs");
const E = require("./expressions");
const C = require("./util/Consts");
const Err = require("./util/Errors");
const P = require("./util/Parsing");
const functions_1 = require("./functions");
const Consts_1 = require("./util/Consts");
function transformAlgebra(expr) {
    if (!expr) {
        throw new Err.InvalidExpression(expr);
    }
    const types = sparqlalgebrajs_1.Algebra.expressionTypes;
    switch (expr.expressionType) {
        case types.TERM:
            return transformTerm(expr);
        case types.OPERATOR:
            return transformOperator(expr);
        case types.NAMED:
            return transformNamed(expr);
        case types.EXISTENCE:
            return transformExistence(expr);
        case types.AGGREGATE:
            return transformAggregate(expr);
        default: throw new Err.InvalidExpressionType(expr);
    }
}
exports.transformAlgebra = transformAlgebra;
/**
 * Transforms an RDF term to the internal representation of a term,
 * assuming it is not a variable, which would be an expression (internally).
 *
 * @param term RDF term to transform into internal representation of a term
 */
function transformRDFTermUnsafe(term) {
    return transformTerm({
        term,
        type: 'expression',
        expressionType: 'term',
    });
}
exports.transformRDFTermUnsafe = transformRDFTermUnsafe;
function transformTerm(term) {
    if (!term.term) {
        throw new Err.InvalidExpression(term);
    }
    switch (term.term.termType) {
        case 'Variable': return new E.Variable(RDFString.termToString(term.term));
        case 'Literal': return transformLiteral(term.term);
        case 'NamedNode': return new E.NamedNode(term.term.value);
        case 'BlankNode': return new E.BlankNode(term.term.value);
        default: throw new Err.InvalidTermType(term);
    }
}
// TODO: Maybe do this with a map?
// tslint:disable-next-line:no-any
function transformLiteral(lit) {
    if (!lit.datatype) {
        return (lit.language)
            ? new E.LangStringLiteral(lit.value, lit.language)
            : new E.StringLiteral(lit.value);
    }
    switch (lit.datatype.value) {
        case null:
        case undefined:
        case '': {
            return (lit.language)
                ? new E.LangStringLiteral(lit.value, lit.language)
                : new E.StringLiteral(lit.value);
        }
        case Consts_1.TypeURL.XSD_STRING:
            return new E.StringLiteral(lit.value);
        case Consts_1.TypeURL.RDF_LANG_STRING:
            return new E.LangStringLiteral(lit.value, lit.language);
        case Consts_1.TypeURL.XSD_DATE_TIME: {
            const val = new Date(lit.value);
            if (isNaN(val.getTime())) {
                return new E.NonLexicalLiteral(undefined, lit.datatype, lit.value);
            }
            return new E.DateTimeLiteral(new Date(lit.value), lit.value);
        }
        case Consts_1.TypeURL.XSD_BOOLEAN: {
            if (lit.value !== 'true' && lit.value !== 'false') {
                return new E.NonLexicalLiteral(undefined, lit.datatype, lit.value);
            }
            return new E.BooleanLiteral(lit.value === 'true', lit.value);
        }
        case Consts_1.TypeURL.XSD_INTEGER:
        case Consts_1.TypeURL.XSD_DECIMAL:
        case Consts_1.TypeURL.XSD_NEGATIVE_INTEGER:
        case Consts_1.TypeURL.XSD_NON_NEGATIVE_INTEGER:
        case Consts_1.TypeURL.XSD_NON_POSITIVE_INTEGER:
        case Consts_1.TypeURL.XSD_POSITIVE_INTEGER:
        case Consts_1.TypeURL.XSD_LONG:
        case Consts_1.TypeURL.XSD_INT:
        case Consts_1.TypeURL.XSD_SHORT:
        case Consts_1.TypeURL.XSD_BYTE:
        case Consts_1.TypeURL.XSD_UNSIGNED_LONG:
        case Consts_1.TypeURL.XSD_UNSIGNED_INT:
        case Consts_1.TypeURL.XSD_UNSIGNED_SHORT:
        case Consts_1.TypeURL.XSD_UNSIGNED_BYTE:
        case Consts_1.TypeURL.XSD_INT: {
            const val = P.parseXSDDecimal(lit.value);
            if (val === undefined) {
                return new E.NonLexicalLiteral(undefined, lit.datatype, lit.value);
            }
            return new E.NumericLiteral(val, lit.datatype, lit.value);
        }
        case Consts_1.TypeURL.XSD_FLOAT:
        case Consts_1.TypeURL.XSD_DOUBLE: {
            const val = P.parseXSDFloat(lit.value);
            if (val === undefined) {
                return new E.NonLexicalLiteral(undefined, lit.datatype, lit.value);
            }
            return new E.NumericLiteral(val, lit.datatype, lit.value);
        }
        default: return new E.Literal(lit.value, lit.datatype, lit.value);
    }
}
exports.transformLiteral = transformLiteral;
function transformOperator(expr) {
    if (C.SpecialOperators.contains(expr.operator)) {
        const op = expr.operator;
        const args = expr.args.map((a) => transformAlgebra(a));
        const func = functions_1.specialFunctions.get(op);
        if (!func.checkArity(args)) {
            throw new Err.InvalidArity(args, op);
        }
        return new E.SpecialOperator(args, func.applyAsync, func.applySync);
    }
    else {
        if (!C.Operators.contains(expr.operator)) {
            throw new Err.UnknownOperator(expr.operator);
        }
        const op = expr.operator;
        const args = expr.args.map((a) => transformAlgebra(a));
        const func = functions_1.regularFunctions.get(op);
        if (!hasCorrectArity(args, func.arity)) {
            throw new Err.InvalidArity(args, op);
        }
        return new E.Operator(args, func.apply);
    }
}
// TODO: Support passing functions to override default behaviour;
function transformNamed(expr) {
    const funcName = expr.name.value;
    if (!C.NamedOperators.contains(funcName)) {
        throw new Err.UnknownNamedOperator(expr.name.value);
    }
    // tslint:disable-next-line:no-any
    const op = expr.name.value;
    const args = expr.args.map((a) => transformAlgebra(a));
    const func = functions_1.namedFunctions.get(op);
    return new E.Named(expr.name, args, func.apply);
}
exports.transformNamed = transformNamed;
function hasCorrectArity(args, arity) {
    // Infinity is used to represent var-args, so it's always correct.
    if (arity === Infinity) {
        return true;
    }
    // If the function has overloaded arity, the actual arity needs to be present.
    if (Array.isArray(arity)) {
        return arity.indexOf(args.length) >= 0;
    }
    return args.length === arity;
}
function transformAggregate(expr) {
    const name = expr.aggregator;
    return new E.Aggregate(name, expr);
}
exports.transformAggregate = transformAggregate;
function transformExistence(expr) {
    return new E.Existence(expr);
}
exports.transformExistence = transformExistence;
//# sourceMappingURL=Transformation.js.map