"use strict";
// tslint:disable:object-literal-sort-keys
// tslint:disable:max-classes-per-file
Object.defineProperty(exports, "__esModule", { value: true });
const RDFDM = require("@rdfjs/data-model");
const E = require("../expressions");
const C = require("../util/Consts");
const Err = require("../util/Errors");
const functions_1 = require("../functions");
const Helpers_1 = require("../functions/Helpers");
const Parsing_1 = require("../util/Parsing");
const Consts_1 = require("./../util/Consts");
const SyncEvaluator_1 = require("./SyncEvaluator");
// TODO: Support hooks
class AggregateEvaluator {
    constructor(expr, config, throwError) {
        this.throwError = false;
        this.expression = expr;
        this.evaluator = new SyncEvaluator_1.SyncEvaluator(expr.expression, config);
        this.aggregator = new exports.aggregators[expr.aggregator](expr);
        this.throwError = throwError;
    }
    /**
     * The spec says to throw an error when a set function is called on an empty
     * set (unless explicitly mentioned otherwise like COUNT).
     * However, aggregate error handling says to not bind the result in case of an
     * error. So to simplify logic in the caller, we return undefined by default.
     *
     * @param throwError wether this function should respect the spec and throw an error if no empty value is defined
     */
    static emptyValue(expr, throwError = false) {
        const val = exports.aggregators[expr.aggregator].emptyValue();
        if (val === undefined && throwError) {
            throw new Err.EmptyAggregateError();
        }
        return val;
    }
    /**
     * Put a binding from the result stream in the aggregate state.
     *
     * If any binding evaluation errors, the corresponding aggregate variable should be unbound.
     * If this happens, calling @see result() will return @constant undefined
     *
     * @param bindings the bindings to pass to the expression
     */
    put(bindings) {
        this.init(bindings);
        if (this.state) {
            this.put = this.__put;
            this.result = this.__result;
        }
    }
    result() {
        return this.aggregator.constructor.emptyValue();
    }
    /**
     * The actual put method. When the first binding has been given, and the state
     * of the evaluators initialised. The .put API function will be replaced with this
     * function, which implements the behaviour we want.
     *
     * @param bindings the bindings to pass to the expression
     */
    __put(bindings) {
        try {
            const term = this.evaluator.evaluate(bindings);
            this.state = this.aggregator.put(this.state, term);
        }
        catch (err) {
            this.safeThrow(err);
        }
    }
    /**
     * The actual result method. When the first binding has been given, and the state
     * of the evaluators initialised. The .result API function will be replaced with this
     * function, which implements the behaviour we want.
     *
     * @param bindings the bindings to pass to the expression
     */
    __result() {
        return this.aggregator.result(this.state);
    }
    init(start) {
        try {
            const startTerm = this.evaluator.evaluate(start);
            this.state = this.aggregator.init(startTerm);
        }
        catch (err) {
            this.safeThrow(err);
        }
    }
    safeThrow(err) {
        if (this.throwError) {
            throw err;
        }
        else {
            this.put = () => { return; };
            this.result = () => undefined;
        }
    }
}
exports.AggregateEvaluator = AggregateEvaluator;
class BaseAggregator {
    constructor(expr) {
        this.distinct = expr.distinct;
        this.separator = expr.separator || ' ';
    }
    static emptyValue() {
        return undefined;
    }
}
class Count extends BaseAggregator {
    static emptyValue() {
        return Helpers_1.number(0, Consts_1.TypeURL.XSD_INTEGER).toRDF();
    }
    init(start) {
        return 1;
    }
    put(state, term) {
        return state + 1;
    }
    result(state) {
        return Helpers_1.number(state, Consts_1.TypeURL.XSD_INTEGER).toRDF();
    }
}
class Sum extends BaseAggregator {
    constructor() {
        super(...arguments);
        this.summer = functions_1.regularFunctions.get(C.RegularOperator.ADDITION);
    }
    static emptyValue() {
        return Helpers_1.number(0, Consts_1.TypeURL.XSD_INTEGER).toRDF();
    }
    init(start) {
        const { value, type } = extractNumericValueAndTypeOrError(start);
        return new E.NumericLiteral(value, RDFDM.namedNode(type));
    }
    put(state, term) {
        const { value, type } = extractNumericValueAndTypeOrError(term);
        const internalTerm = new E.NumericLiteral(value, RDFDM.namedNode(type));
        const sum = this.summer.apply([state, internalTerm]);
        return sum;
    }
    result(state) {
        return state.toRDF();
    }
}
class Min extends BaseAggregator {
    init(start) {
        const { value } = extractNumericValueAndTypeOrError(start);
        return { minNum: value, minTerm: start };
    }
    put(state, term) {
        const { value } = extractNumericValueAndTypeOrError(term);
        if (value < state.minNum) {
            return {
                minNum: value,
                minTerm: term,
            };
        }
        return state;
    }
    result(state) {
        return state.minTerm;
    }
}
class Max extends BaseAggregator {
    init(start) {
        const { value } = extractNumericValueAndTypeOrError(start);
        return { maxNum: value, maxTerm: start };
    }
    put(state, term) {
        const { value } = extractNumericValueAndTypeOrError(term);
        if (value >= state.maxNum) {
            return {
                maxNum: value,
                maxTerm: term,
            };
        }
        return state;
    }
    result(state) {
        return state.maxTerm;
    }
}
class Average extends BaseAggregator {
    constructor() {
        super(...arguments);
        this.summer = functions_1.regularFunctions.get(C.RegularOperator.ADDITION);
        this.divider = functions_1.regularFunctions.get(C.RegularOperator.DIVISION);
    }
    static emptyValue() {
        return Helpers_1.number(0, Consts_1.TypeURL.XSD_INTEGER).toRDF();
    }
    init(start) {
        const { value, type } = extractNumericValueAndTypeOrError(start);
        const sum = new E.NumericLiteral(value, RDFDM.namedNode(type));
        return { sum, count: 1 };
    }
    put(state, term) {
        const { value, type } = extractNumericValueAndTypeOrError(term);
        const internalTerm = new E.NumericLiteral(value, RDFDM.namedNode(type));
        const sum = this.summer.apply([state.sum, internalTerm]);
        return {
            sum,
            count: state.count + 1,
        };
    }
    result(state) {
        const count = new E.NumericLiteral(state.count, RDFDM.namedNode(C.TypeURL.XSD_INTEGER));
        const result = this.divider.apply([state.sum, count]);
        return result.toRDF();
    }
}
class GroupConcat extends BaseAggregator {
    static emptyValue() {
        return Helpers_1.string('').toRDF();
    }
    init(start) {
        return start.value;
    }
    put(state, term) {
        return state + this.separator + term.value;
    }
    result(state) {
        return Helpers_1.string(state).toRDF();
    }
}
class Sample extends BaseAggregator {
    init(start) {
        return start;
    }
    put(state, term) {
        return state; // First value is our sample
    }
    result(state) {
        return state;
    }
}
exports.aggregators = {
    count: Count,
    sum: Sum,
    min: Min,
    max: Max,
    avg: Average,
    group_concat: GroupConcat,
    sample: Sample,
};
function extractNumericValueAndTypeOrError(term) {
    // TODO: Check behaviour
    if (term.termType !== 'Literal' || !C.NumericTypeURLs.contains(term.datatype.value)) {
        throw new Error('Term is not numeric');
    }
    const type = term.datatype.value;
    const value = Parsing_1.parseXSDFloat(term.value);
    return { type, value };
}
//# sourceMappingURL=AggregateEvaluator.js.map