"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const E = require("../expressions");
const Err = require("../util/Errors");
const Transformation_1 = require("../Transformation");
const sharedEvaluators = {
    term(expr, mapping) {
        return expr;
    },
    variable(expr, mapping) {
        const term = mapping.get(expr.name);
        if (!term) {
            throw new Err.UnboundVariableError(expr.name, mapping);
        }
        return Transformation_1.transformRDFTermUnsafe(term);
    },
};
class AsyncRecursiveEvaluator {
    constructor(context) {
        this.context = context;
        this.subEvaluators = {
            // Shared
            [E.ExpressionType.Term]: sharedEvaluators.term,
            [E.ExpressionType.Variable]: sharedEvaluators.variable,
            // Async
            [E.ExpressionType.Operator]: this.evalOperator,
            [E.ExpressionType.SpecialOperator]: this.evalSpecialOperator,
            [E.ExpressionType.Named]: this.evalNamed,
            [E.ExpressionType.Existence]: this.evalExistence,
            [E.ExpressionType.Aggregate]: this.evalAggregate,
        };
    }
    async evaluate(expr, mapping) {
        const evaluator = this.subEvaluators[expr.expressionType];
        if (!evaluator) {
            throw new Err.InvalidExpressionType(expr);
        }
        return evaluator.bind(this)(expr, mapping);
    }
    async evalOperator(expr, mapping) {
        const argPromises = expr.args.map((arg) => this.evaluate(arg, mapping));
        const argResults = await Promise.all(argPromises);
        return expr.apply(argResults);
    }
    async evalSpecialOperator(expr, mapping) {
        const evaluate = this.evaluate.bind(this);
        const context = {
            args: expr.args,
            mapping,
            evaluate,
            context: {
                now: this.context.now,
                baseIRI: this.context.baseIRI,
                bnode: this.context.bnode,
            },
        };
        return expr.applyAsync(context);
    }
    async evalNamed(expr, mapping) {
        const argPromises = expr.args.map((arg) => this.evaluate(arg, mapping));
        const argResults = await Promise.all(argPromises);
        return expr.apply(argResults);
    }
    async evalExistence(expr, mapping) {
        if (!this.context.exists) {
            throw new Err.NoExistenceHook();
        }
        return new E.BooleanLiteral(await this
            .context
            .exists(expr.expression, mapping));
    }
    // TODO: Remove?
    async evalAggregate(expr, _mapping) {
        if (!this.context.aggregate) {
            throw new Err.NoExistenceHook();
        }
        return Transformation_1.transformRDFTermUnsafe(await this
            .context
            .aggregate(expr.expression));
    }
}
exports.AsyncRecursiveEvaluator = AsyncRecursiveEvaluator;
class SyncRecursiveEvaluator {
    constructor(context) {
        this.context = context;
        this.subEvaluators = {
            // Shared
            [E.ExpressionType.Term]: sharedEvaluators.term,
            [E.ExpressionType.Variable]: sharedEvaluators.variable,
            // Sync
            [E.ExpressionType.Operator]: this.evalOperator,
            [E.ExpressionType.SpecialOperator]: this.evalSpecialOperator,
            [E.ExpressionType.Named]: this.evalNamed,
            [E.ExpressionType.Existence]: this.evalExistence,
            [E.ExpressionType.Aggregate]: this.evalAggregate,
        };
    }
    evaluate(expr, mapping) {
        const evaluator = this.subEvaluators[expr.expressionType];
        if (!evaluator) {
            throw new Err.InvalidExpressionType(expr);
        }
        return evaluator.bind(this)(expr, mapping);
    }
    evalOperator(expr, mapping) {
        const args = expr.args.map((arg) => this.evaluate(arg, mapping));
        return expr.apply(args);
    }
    evalSpecialOperator(expr, mapping) {
        const evaluate = this.evaluate.bind(this);
        const context = {
            args: expr.args,
            mapping,
            evaluate,
            context: {
                now: this.context.now,
                baseIRI: this.context.baseIRI,
                bnode: this.context.bnode,
            },
        };
        return expr.applySync(context);
    }
    evalNamed(expr, mapping) {
        const args = expr.args.map((arg) => this.evaluate(arg, mapping));
        return expr.apply(args);
    }
    evalExistence(expr, mapping) {
        if (!this.context.exists) {
            throw new Err.NoExistenceHook();
        }
        return new E.BooleanLiteral(this
            .context
            .exists(expr.expression, mapping));
    }
    evalAggregate(expr, mapping) {
        if (!this.context.aggregate) {
            throw new Err.NoAggregator();
        }
        return Transformation_1.transformRDFTermUnsafe(this
            .context
            .aggregate(expr.expression));
    }
}
exports.SyncRecursiveEvaluator = SyncRecursiveEvaluator;
class UnsupportedOperation extends Error {
    constructor(operation) {
        super(`Operation '${operation}' is unsupported in SimpleEvaluator`);
    }
}
exports.UnsupportedOperation = UnsupportedOperation;
//# sourceMappingURL=RecursiveExpressionEvaluator.js.map